########################################################################
# metaf2xml/parser.pm -- parse a METAR or TAF message
#
# copyright (c) metaf2xml 2006
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
########################################################################

package metaf2xml::parser;

########################################################################
# some things strictly perl
########################################################################
use strict;
use warnings;
use vars qw(@ISA @EXPORT @EXPORT_OK);
use POSIX qw(floor);

=head1 NAME

metaf2xml::parser - parse a METAR or TAF message

=head1 SYNOPSIS

  use metaf2xml::parser;

  my %metar = parseReport($msg, $is_taf);

=head1 DESCRIPTION

This module contains functions to analyze a string (per default as a METAR
message). Its main function returns a hash with all its components.

=head1 FUNCTIONS

=over

=cut

########################################################################
# export the functions provided by this module
########################################################################
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(
    version_parser
    $INHG2HPA
    rnd
    parseReport
);

########################################################################
# small stuff to be exported
########################################################################
our $INHG2HPA = 33.86388640341;

sub rnd {
    my ($val, $prec) = @_;
    return $prec * floor(($val / $prec) + 0.5);
}

########################################################################
# define lots of regular expressions
########################################################################
my $re_day = '(?:[0-2][0-9]|3[01])';
my $re_hour = '(?:[01][0-9]|2[0-3])';
my $re_min = '[0-5][0-9]';
my $re_bbb = '(?:(?:RR|CC|AA)[A-Z]|P[A-Z]{2})';
my $re_rwy = '(?:0[1-9]|[12][0-9]|3[0-6])';
my $re_rwy_des = "$re_rwy(?:[LCR])?";
my $re_rwy_des2 = '(?:[05][1-9]|[1267][0-9]|[38][0-6])'; # add 50 for rrR
my $re_wind_speed = 'P?[1-9]?[0-9]{2}';
my $re_wind_speed_unit = '(?:KT|TK|K T| KT|MPS|KMH)';
my $re_wind_dir = '(?:[0-2][0-9]|3[0-6])';
# EXTENSION: allow wind direction not rounded to 10 degrees
my $re_wind = "(VRB|///|${re_wind_dir}[0-9])(//|$re_wind_speed)(?:G($re_wind_speed))?($re_wind_speed_unit)";
my $re_vis_m = '[0-9]{4}';
my $re_vis_km = '[1-9][0-9]?KM';
# EXTENSION: allow P6000 (UERP ULOO UMKK UMOO URSS UTAK)
my $re_vis_m_km = "(?:P?$re_vis_m(?:NDV)?|$re_vis_km)";
my $re_vis_m_km_remark = "(?:$re_vis_m|[1-9][0-9]{2,3}M|$re_vis_km)";
my $re_frac16 = '[135]/16';
my $re_frac8 =  '[1357]/8';
my $re_frac4 =  '[13]/4';
my $re_frac2 =  '1/2';
my $re_vis_frac_sm0 = "(?:$re_frac16|$re_frac8|$re_frac4|$re_frac2)";
my $re_vis_frac_sm1 = "1(?: (?:$re_frac8|$re_frac4|$re_frac2))?";
# EXTENSION: allow /8 for 2 miles
my $re_vis_frac_sm2 = "2(?: (?:$re_frac8|$re_frac4|$re_frac2))?";
# EXTENSION: allow /2 for 3 miles
my $re_vis_frac_sm3 = "3(?: (?:$re_frac2))?";
my $re_vis_whole_sm = "(?:[04-9]|1[0-5]|[2-9][05]|[1-9][0-9]0)";
my $re_vis_sm =   "(?:M1/4"
                 . "|$re_vis_whole_sm"
                 . "|$re_vis_frac_sm0"
                 . "|$re_vis_frac_sm1"
                 . "|$re_vis_frac_sm2"
                 . "|$re_vis_frac_sm3)";
my $re_gs_size = "(?:M1/4|[1-9][0-9]*(?: $re_frac4| $re_frac2)?|$re_frac4|$re_frac2)";
my $re_rwy_vis = '[PM]?[0-9]{4}';
my $re_rwy_wind = "(?:WIND )?(?:RWY($re_rwy_des) ?|($re_rwy_des)/)((?:///|VRB|${re_wind_dir}0)(?://|$re_wind_speed(?:G$re_wind_speed)?)$re_wind_speed_unit)(?: (${re_wind_dir}[0-9])V(${re_wind_dir}[0-9]))?";
my $re_rwy_wind2 = "((?:///|VRB|${re_wind_dir}0)(?://|$re_wind_speed(?:G$re_wind_speed)?)$re_wind_speed_unit)/RWY($re_rwy_des)";

my $re_compass_dir = '(?:[NS]?[EW]|[NS])';
my $re_compass_dir16 = "(?:[NE]NE|[ES]SE|[SW]SW|[NW]NW|$re_compass_dir)";

my $re_phen_locdir_comp = "(?:(?:GRID )?(?:[1-9][0-9]*(?:KM)? ?)?$re_compass_dir16(?: QUAD)?)";
my $re_phen_locdir_comp_thru = "(?:O?MTNS|OMTS|(?:OVR|ALG) (?:LK|RIVER|MT(?:N?S)?|VLYS?)|(?:(?:OV?HD|(?:OVE?R |AT )?A[PD]|IN VLY|STNRY|(?:DISTANT |DSNT |VC )?(?:OBSCG MTNS )?(?:$re_phen_locdir_comp|ALQDS|V/D))(?:(?: ?- ?| THRU )(?:$re_phen_locdir_comp|OV?HD))*))";
my $re_phen_locdir_comp_and = "(?:$re_phen_locdir_comp_thru(?:(?:[/ ]| AND )$re_phen_locdir_comp_thru)*)";
my $re_phen_locdirs = "(?: $re_phen_locdir_comp_and)";
my $re_wx_mov_d2 = "(?: (MOVD?) ($re_compass_dir16(?:-$re_compass_dir16)*|OV?HD|STNRY|UNKN))";
my $re_num_quadrant1 = '(?:1(?:ST)?|2(?:ND)?|3(?:RD)?|4(?:TH)?)';
my $re_num_quadrant = "(?: $re_num_quadrant1(?:[- ]$re_num_quadrant1)* QUAD)";
my $re_weather_desc = '(?:MI|BC|PR|DR|BL|SH|TS|FZ)';
my $re_weather_ra = 'RA(?:DZ|SN|SG|PL|IC|GR|GS)*';
my $re_weather_dz = 'DZ(?:RA|SN|SG|PL|IC|GR|GS)*';
my $re_weather_sn = 'SN(?:RA|DZ|SG|PL|IC|GR|GS)*';
my $re_weather_ic = 'IC(?:RA|DZ|SN|SG|PL|GR|GS)*';
my $re_weather_prec = '(?:DZ|RA|SN|SG|PL|IC|GR|GS)+';
# EXTENSION: allow SHRADZ (ENHK)
my $re_weather_ts_sh = '(?:RA|SN|PL|GR|GS|DZ)+|UP';
my $re_weather_sg_pl_gr_gs = '(?:SG|PL|GR|GS)+';
# EXTENSION: allow SN, BC, PR, BR, PL with FZ
my $re_weather_fz = '(?:RA|DZ|FG|SN|BC|PR|BR|PL)+';
# EXTENSION: allow +/- with BLSN, BR, TS, FC
# EXTENSION: allow JP (adjacent precipitation in old METAR)
my $re_weather_w_i = "(?:$re_weather_prec|DS|SS"
                      . "|(?:TS|SH)(?:$re_weather_ts_sh)"
                      . "|FZ(?:SH)?(?:$re_weather_fz)"
                      . "|UP|BLSN|BR|FC|TS|JP)";
my $re_weather_wo_i = "(?:$re_weather_ic|FG|BR|SA|DU|HZ|FU|VA|SQ|PO|FC|TS"
                       . "|(?:FZ|MI|BC|PR)*FG"
                       . "|BL(?:$re_weather_sn|SA|DU)"
                       . "|DR(?:$re_weather_sn|SA|DU))";
# EXTENSION: allow VCTS..
my $re_weather_vc = "(?:FG|PO|FC|DS|SS|TS(?:$re_weather_ts_sh)?|SH|BL(?:$re_weather_sn|SA|DU)|VA)";
# annex 3, form:
# FZDZ FZRA DZ   RA   SHRA SN SHSN SG SHGR SHGS BLSN SS DS
# TSRA TSSN TSPL TSGR TSGS FC VA   PL UP
# annex 3, text:
# FZDZ FZRA DZ   RA   SHRA SN SHSN SG SHGR SHGS BLSN SS DS
#                          FC VA   PL    TS
# EXTENSION: allow FZFG
my $re_weather_re = "(?:(?:FZ|SH)?$re_weather_prec"
                     . "|TS(?:$re_weather_ts_sh)"
                     . "|FZFG|BLSN|DS|SS|TS|FC|VA|UP)";
# EXTENSION: allow [+-]VC...
my $re_weather = "(?:[+-]?$re_weather_w_i|$re_weather_wo_i|[+-]?VC$re_weather_vc)";

my $re_cloud_cov = '(?:FEW|SCT|BKN|OVC)';
my $re_cloud_base = '(?:///|[0-9]{3})';
my $re_cloud_type = '(?:AC|ACC|ACSL|AS|CB|CBMAM|CC|CCSL|CI|CS|CU|CUFRA|CF'
                     . '|NS|SAC|SC|SCSL|ST|STFRA|SF|TCU|///)';

my $re_colour1 = 'BLU\+?|WHT|GRN|YLO[12]?|AMB|RED';
my $re_colour = "(?:BLACK|(?:BLACK)?(?:$re_colour1)(?: ?(?:$re_colour1|FCST CANCEL))?)";

my $re_be_prec_be   = "(?:[BE]$re_hour?$re_min)";
my $re_be_prec = "(?:(?:FZ|SH)?(?:TS|RA|SN|UP|PL|DZ|GS|FC)+$re_be_prec_be+)";

my $re_rsc_cond = '(?:DRY|WET|SANDED)';
my $re_rsc_deposit = '(?:LSR|SLR|PSR|IR|WR)+(?://|[0-9]{2})?';
my $re_rsc = "(?:$re_rsc_deposit(?:P(?:[ /]$re_rsc_cond)?)?|(?:(?:P[ /])?$re_rsc_cond))";

my $re_snw_cvr_title = '(?:SNO?W? ?(?:CVR|COV(?:ER)?)?[/ ])';
my $re_snw_cvr_state =
       '(?:(?:ONE |MU?CH |TR(?:ACE)? ?)LOOSE|(?:MED(?:IUM)?|HARD) PACK(?:ED)?)';
my $re_snw_cvr = "(?:$re_snw_cvr_title?($re_snw_cvr_state)|${re_snw_cvr_title}NIL)";

my $re_temp = '(?:-?[0-9]+\.[0-9])';
my $re_precip = '(?:[0-9]{1,2}(?:\.[0-9])?(?: ?[MC]M)?)';

my $re_phenom_opac =
        '(?:(?:BL)?SN|FG|AS|AC|CC|CF|CI|CS|CU|TCU|NS|ST|SF|SC|ACC|CUFRA|CB|IC)';
my $re_trace_cloud = '(?:AC|AS|CF|CI|CS|CU|CUFRA|SC|SF|TCU)';

my $re_phen_desc_when = '(?:OCNL|FRQ|INTMT|CON(?:TU)?S|PAST HR)';
my $re_phen_desc_how  = '(?:LOW?|LWR|(?:VRY |V|PR )?(?:THN|THIN|THK|THICK)|ISOL|CVCTV|DSIPTD|FREEZING|PTCHY)';
my $re_phen_desc_strength  = '(?:(?:VRY |V|PR )?(?:LGT|FBL)|MDT|MOD)';
my $re_phen_desc_where = '(?:ALOFT|ARND)';
my $re_phen_desc = "(?:$re_phen_desc_when|$re_phen_desc_how|$re_phen_desc_strength|$re_phen_desc_where)";
my $re_phen_desc_other = "(?:$re_phen_desc_when|$re_phen_desc_where)";

my $re_ltg_types = '(?:CA|CC|CG|CW|IC)';

my $re_wind_shear_lvl = "WS([0-9]{3})/(${re_wind_dir}0${re_wind_speed}KT)";

my $re_phenomenon_other = "(?:LTG|VIRGA|AURBO|AURORA|FG BNK|FULYR|HZY|BINOVC|ICG|SHS?|SHWR|DEW|(?:VIS|CIG|SC) (?:HYR|HIER|LWR|RDCD|REDUCED)|CLDS?(?: EMBD)?|(?:GRASS )?FIRES?)";
my $re_phenomenon4 = "(?:($re_phenomenon_other|PCPN|VLY FG|HIR CLDS)|($re_cloud_type(?:[/-]$re_cloud_type)*)|($re_weather|SMOKE|HAZE)|($re_cloud_cov))";

my $re_loc_quadr3 = "(?:(?: TO)?($re_phen_locdirs)|(?: (DSNT))?($re_num_quadrant))";

my $re_estmd = '(?:EST(?:MD)?|ESTM?D|ESMTD|ESTIMATED)';
my $re_data_estmd = '(?:WI?NDS?(?: DATA)?|(?:CIG )?BLN|CIG|CEILING|SLP|ALSTG|CLD HGTS)';

########################################################################
# version_parser
########################################################################

=item version_parser()

It returns the CVS (RCS) id of the module file.

No arguments are expected.

=cut

sub version_parser {
    return 'parser: $Id: parser.pm,v 1.31 2006/11/09 19:44:34 metaf2xml Exp $';
}

########################################################################
# helper functions
########################################################################

# $cy: [ 'ABCD', 'AB', 'A' ]
sub _isValidCountry {
    my ($re, $cy) = @_;

    my %country = (
        cloudMaxCover =>
            ' LO ',
        NEFO_PLAYA =>
            ' SCSE ',
        relHumid =>
            ' OOSA ',
        rwyWind =>
            ' LG ',
        remarkCloudMaxCov =>
            ' LI OA ',
        remarkColour =>
            ' BKPR EG ET FH L OA PGUA ',
        remarkCloudTypeFamily =>
            ' K MM MU NS P RJ RO UT ',
        remarkCIGRAG =>
            ' CY P K ',
        remarkCIGVRBL =>
            ' CY ',
        remarkSLPdPa =>
         ' BGTL CM CW CY CZ ET EGVA K LQ MHSC MM MU NS NZ P RJ RK RO TI TJ TX ',
        remarkSPinHg =>
            ' ROTM ',
        remarkSLPinHg =>
            ' ',
        remarkGRIDWind =>
            ' NZ ',
        remarkRwyWind =>
            ' BG LC LH LT SP UL ',
        remarkRwyWind2 =>
            ' RC ',
        remarkRsc =>
            ' BG EGUN ET K LQTZ NZ PA PG PH RJ RK RO ',
        remarkPhenomOpacity =>
            ' CQ CW CY CZ ',
        remarkPhenomOpacityRev =>
            ' BG KTTS ',
        remarkCloudOpacityLvl =>
            ' RJ RO YB YM YP ',
        remarkCloudCoverLvl =>
            ' C EK K LI LL P NS TI TJ ',
        remarkCloudTypeLvl =>
            ' U ',
        remarkCloudTraces =>
            ' CW CY CZ ',
        remarkReWeather =>
            ' CY FQ TR ',
        remarkSfcVisM =>
            ' KQ ',
        remarkVisSfcTwr =>
            ' K PA ',
        remarkVisVarK =>
            ' K PA ',
        remarkVisVarCY =>
            ' CY ',
        remarkVisVarCYCZ =>
            ' CY CZ ',
        remarkPCPNPastHour =>
            ' CW CY CZ ',
        remarkVisLocM =>
            ' E KQ LQ NZ RJ RO ',
        remarkQNH =>
            ' RJ RO RP ',
        remarkQNHMB =>
            ' KQ ',
        remarkQFF =>
            ' MM ',
        remarkCorrected =>
            ' ET K LQ PA ',
        remarkSnowIncr =>
            ' K PA ',
        remarkHourly =>
            ' BG C ET K LQ MM MU NS P RJ RKSO RO TI TJ ',
        remarkPeakWind =>
            ' C ET K LQ NZ PA PH TI TJ ',
        remarkHourlyTemp =>
            ' C K P NS MU RJ RO TI TJ ',
        remarkSunshine =>
            ' K P ',
        remarkQFEhPa =>
            ' UTTP UTSS ',
        remarkDA_PA =>
            ' K CY ',
        remarkKN =>
            ' KN ',
        remarkKNIP =>
            ' KNIP ',
        remarkQFEmmHg =>
            ' U ZM ',
        remarkPhenCloud =>
            ' K PA PH ',
        remarkRainfall =>
            ' YA YB ',
        remarkRSNK =>
            ' KHMS ',
        remarkLAGPK =>
            ' KNTD ',
        remarkPrecipHourly3 =>
            ' KAEG KAUN KBVS KUKT KW22 ',
        remarkCloudCoverVar =>
            ' K MH PA PH PM PP ',
        remarkC_K_P_NS_RJ_RO_TI_TJ_ET =>
            ' C K P NS RJ RO TI TJ ET ',
        remarkSNWCVR =>
            ' C ',
        remarkObsTimeOffset =>
            ' C ',
        remarkBalloon =>
            ' CY CZ ',
        remarkRVR =>
            ' CY ',
        remarkTX =>
            ' FQ ',
        remarkTAFNxtFcstBy =>
            ' CB CY CZ ',
        remarkTAFNxtFcstAt =>
            ' CW CY ',
        remarkTAFAmdAt =>
            ' LT ',
        remarkTAFFcstAutoObs =>
            ' CB CW CY CZ ',
        remarkTAFFcstAutometar =>
            ' EF ',
    );
    return    index($country{$re}, ' ' . $cy->[0] . ' ') > -1
           || index($country{$re}, ' ' . $cy->[1] . ' ') > -1
           || index($country{$re}, ' ' . $cy->[2] . ' ') > -1;
}

sub _parseFraction {
    my ($r, $vis, $unitLength) = @_;

    $r->{unitLength} = $unitLength if $unitLength;
    if ($vis =~ /^M/) {
        $r->{distance} = 0.25;
        $r->{isLess}   = undef;
    } elsif ($vis =~ /^[0-9.]+$/o) {
        $r->{distance} = $vis;
    } elsif ($vis =~ m@^([1357])/([1-8]+)$@o) {
        $r->{distance} = $1 / $2;
    } else {
        $vis =~ m@(.)(?: ([0-9]*)/([0-9]*))?@;
        $r->{distance} = $1;
        $r->{distance} += $2 / $3 if $2;
    }
}

sub _parseWeather {
    my ($weather, $is_recent) = @_;
    my ($w, $int, $vc, $desc, $phen, $intensity, $delim, $weather_str);

    $weather_str = ($is_recent ? 'RE' : '') . $weather;
    return { notAvailable => undef, s => $weather_str } if $weather eq '//';
    return { NSW => undef, s => $weather_str } if $weather eq 'NSW';

    return { tornado => undef, s => $weather_str } if $weather eq '+FC';

    ($int, $vc, $desc, $phen) =
                      $weather =~ m@([+-])?(VC)?($re_weather_desc+)?([A-Z/]+)@o;
    $w->{s}          = $weather_str;
    $w->{intensity}  = ($int eq '-' ? 'LIGHT' : 'HEAVY') if defined $int;
    $w->{inVicinity} = undef if defined $vc;
    @{$w->{descriptor}} = $desc =~ /(..)+?/g if defined $desc;
    @{$w->{phenomenon}} = $phen =~ /(..)+?/g if defined $phen;
    return $w;
}

sub _parsePhenomOpacity {
    my ($r, $clds) = @_;

    for ($clds =~ /((?:[A-Z]+[0-9])+?)/g) {
        /(.*)(.)/;
        if (index($1, 'SN') >-1 || index($1, 'FG') > -1 || index($1, 'IC') > -1)
        {
            push @{$r->{phenomOpacity}}, {
                eights  => $2,
                weather => _parseWeather $1
            };
        } else {
            push @{$r->{phenomOpacity}}, {
                eights    => $2,
                cloudType => $1
            };
        }
    }
}

sub _parseRwyVis {
    my ($metar, $msg) = @_;
    my $r;

    if ($$msg =~ m@^(R($re_rwy_des)/(?:(////)|($re_rwy_vis)(?:V($re_rwy_vis))?(FT)?)(?:/)?([UDN])?) (.*)@o)
    {
        my $v;

        $v->{s}        = $1;
        $v->{rwyDesig} = $2;
        $v->{visTrend} = $7 if defined $7;
        if ($3) {
            $v->{RVR}{notAvailable} = undef;
        } else {
            $v->{RVR}{distance}   = $4;
            $v->{RVR}{unitLength} = defined $6 ? $6 : 'M';
            if (defined $5) {
                $v->{RVRVariations}{distance}   = $5;
                $v->{RVRVariations}{unitLength} = defined $6 ? $6 : 'M';
                $v->{RVRVariations}{isLess}     = undef
                    if $v->{RVRVariations}{distance} =~ s/^M//;
                $v->{RVRVariations}{isEqualGreater} = undef
                    if $v->{RVRVariations}{distance} =~ s/^P//;
                $v->{RVRVariations}{distance} += 0;
            }
            $v->{RVR}{isLess}         = undef
                if $v->{RVR}{distance} =~ s/^M//;
            $v->{RVR}{isEqualGreater} = undef
                if $v->{RVR}{distance} =~ s/^P//;
            $v->{RVR}{distance} += 0;
        }
        push @{$metar->{visRwy}}, $v;
        $$msg = $8;
        return 1;
    }
    return 0;
}

sub _parseRwyState {
    my ($metar, $msg) = @_;
    my $r;

    if ($$msg =~ m@^((SNOCLO)|(88|99|$re_rwy_des2)(?:(CLRD)|([0-9/])([0-9/])([0-9]{2}|//))([0-9]{2}|//)) (.*)@o)
    {
        $$msg = $9;
        $r->{s} = $1;
        if (defined $2) {
            $r->{SNOCLO} = undef;
        } else {
            if ($3 == 88) {
                $r->{rwyDesigAll} = undef;
            } elsif ($3 == 99) {
                $r->{rwyDesigRep} = undef;
            } elsif ($3 > 50) {
                $r->{rwyDesig} = sprintf '%02dR', $3 - 50;
            } else {
                $r->{rwyDesig} = $3;
            }
            if (defined $4) {
                $r->{cleared} = undef;
            } else {
                if ($5 eq '/') {
                    $r->{depositType}{notAvailable} = undef;
                } else {
                    $r->{depositType}{depositTypeVal} = $5;
                }
                if ($6 eq '/') {
                    $r->{depositExtent}{notAvailable} = undef;
                } elsif (   $6 eq '0' || $6 eq '1' || $6 eq '2'
                         || $6 eq '5' || $6 eq '9')
                {
                    $r->{depositExtent}{depositExtentVal} = $6;
                } else {
                    $r->{depositExtent}{invalidFormat} = $6;
                }
                if ($7 eq '//') {
                    $r->{depositDepth}{notAvailable} = undef;
                } elsif ($7 != 91) {
                    $r->{depositDepth}{depositDepthVal} = $7 + 0;
                } else {
                    $r->{depositDepth}{invalidFormat} = $7;
                }
            }
            if ($8 eq '//') {
                $r->{friction}{notAvailable} = undef;
            } elsif ($8 >= 1 && $8 <= 90) {
                $r->{friction}{coefficient} = $8 + 0;
            } elsif ($8 == 99) {
                $r->{friction}{unreliable} = undef;
            } elsif ($8 >= 91 && $8 <= 95) {
                $r->{friction}{brakingAction} = $8;
            } else {
                $r->{friction}{invalidFormat} = $8;
            }
        }
        push @{$metar->{rwyState}}, $r;
        return 1;
    }
    return 0;
}

sub _parseWind {
    my $wind = shift;
    my ($w, $dir, $speed, $gustSpeed, $unitSpeed);

    return { notAvailable => undef } if $wind eq '/////';
    ($dir, $speed, $gustSpeed, $unitSpeed) = $wind =~ m@^$re_wind$@o;
    if ($dir eq '///' && $speed eq '//') {
        $w->{notAvailable} = undef;
    } else {
        if ($dir eq '///') {
            $w->{dirNotAvailable} = undef;
        } elsif ($dir eq 'VRB') {
            $w->{dirVariable} = undef;
        } else {
            $w->{dir} = $dir + 0; # true, not magnetic
        }
        if ($speed eq '//') {
            $w->{speedNotAvailable} = undef;
        } else {
            $w->{speedGreater} = undef if $speed =~ s/^P//;
            $w->{speed} = $speed + 0;
            $w->{unitSpeed} = $unitSpeed;
            $w->{unitSpeed} =~ s/TK/KT/;
            $w->{unitSpeed} =~ s/ //;
        }
        if (defined $gustSpeed) {
            $w->{gustSpeedGreater} = undef if $gustSpeed =~ s/^P//;
            $w->{gustSpeed} = $gustSpeed + 0;
        }
    }
    return $w;
}

sub _parseCloud {
    my $cloud = shift;
    my $c;

    $c->{s} = $cloud;
    if ($cloud eq '//////') {
        $c->{notAvailable} = undef;
    } elsif ($cloud =~ m@^$re_cloud_cov$re_cloud_base(?: ?$re_cloud_type)?@o){
        $cloud =~ '(...)(?:(...) ?(.+)?)?';
        $c->{cloudCover} = $1;
        if (defined $2) {
            if ($2 eq '///') {
                $c->{baseBelowStation} = undef;
            } else {
                $c->{cloudBase} = $2 + 0; # AGL
            }
        }
        if (defined $3) {
            if ($3 eq '///') {
                $c->{cloudTypeNotAvailable} = undef;
            } else {
                $c->{cloudType} = $3;
            }
        }
    } else {
        $cloud =~ '(...)(.+)?';
        $c->{cloudCover} = $1;
        $c->{cloudType}  = $2 if defined $2;
    }
    return $c;
}

sub _parseColourCode {
    my $colour = shift;
    my $c;

    $colour =~ "^(BLACK)?($re_colour1)? ?(.+)?";
    $c->{s} = $colour;
    $c->{BLACK} = undef if defined $1;
    $c->{currentColour} =
            $2 eq 'BLU+' ? 'BLUplus' : ($2 eq 'FCST CANCEL' ? 'FCSTCANCEL' : $2)
        if defined $2;
    $c->{predictedColour} =
            $3 eq 'BLU+' ? 'BLUplus' : ($3 eq 'FCST CANCEL' ? 'FCSTCANCEL' : $3)
        if defined $3;
    return $c;
}

sub _parseLocations {
    my $loc_str = shift;
    my $loc_and;

    for ($loc_str =~ m@(?:[/ ]| AND )?($re_phen_locdir_comp_thru|UNKN)@og) {
        my ($loc_thru, $is_grid);
        for ($_ =~ m@(?: ?[/-] ?| THRU )?(OV?HD|(?:OVE?R |AT )?A[PD]|IN VLY|O?MTNS|OMTS|(?:OVE?R|ALG) (?:LK|RIVER|MT(?:N?S)?|VLYS?)|STNRY|UNKN|(?:DISTANT |DSNT |VC )?(?:OBSCG MTNS )?(?:ALQDS|V/D|$re_phen_locdir_comp))@og)
        {
            my $l;

            if ($_ =~ m@^(?:OV?HD|(?:OVE?R |AT )?A[PD]|IN VLY|O?MTNS|OMTS|(?:OVR|ALG) (?:LK|RIVER|MT(?:N?S)?|VLYS?)|STNRY|UNKN)$@) {
                s/^OVR MT(?:N?S)?/OMTNS/;
                s/O?MTNS/OMTNS/;
                s/OMTS/OMTNS/;
                s/OV?HD/OHD/;
                s/OVER/OVR/;
                s/AD$/AP/;
                s/^AP$/AT AP/;
                s/ /_/;
                $l->{locationSpec} = $_;
            } else {
                $_ =~ m@(?:(DISTANT|DSNT|VC) )?(OBSCG MTNS )?(?:TO ?)?(?:(?:(GRID )?(?:([1-9][0-9]*)(KM)? ?)?($re_compass_dir16)(?: (QUAD))?)|(ALQDS|V/D))@o;
                $l->{isDistant}  = undef
                    if defined $1 and ($1 eq 'DISTANT' or $1 eq 'DSNT');
                $l->{inVicinity} = undef if defined $1 and $1 eq 'VC';
                $l->{obscgMtns}  = undef if defined $2;
                if (defined $3 || $is_grid) {
                    $is_grid = 1;
                    $l->{isGrid} = undef;
                }
                if (defined $4) {
                    $l->{distance}   = $4;
                    $l->{unitLength} = defined $5 ? 'KM' : 'SM';
                }
                $l->{compassDir} = $6 if defined $6;
                $l->{isQuadrant} = undef if defined $7;
                ($l->{locationSpec} = $8) =~ tr /\/ /_/ if defined $8;
            }
            push @{$loc_thru}, $l;
        }
        push @{$loc_and}, $loc_thru;
    }
    return $loc_and;
}

sub _determineCeiling {
    my $cloud = shift;
    my ($ceil, $idx, $ii);
    $ceil = 1000;
    $idx = -1;
    $ii = -1;
    for (@$cloud) {
        $ii++;
        if (   exists $_->{cloudBase}
            && $_->{cloudBase} < $ceil
            && $_->{cloudBase} < 200 # TODO: actually MSL!
            && ($_->{cloudCover} eq 'BKN' || $_->{cloudCover} eq 'OVC'))
        {
            $ceil = $_->{cloudBase};
            $idx = $ii;
        }
    }
    $cloud->[$idx]{isCeiling} = undef if $idx > -1;
}

sub _parseQuadrants {
    my ($q, $isDistant) = @_;
    my $l;

    $l->{isDistant} = undef if $isDistant;
    @{$l->{quadrant}} = $q =~ /([1-4])/g;

    return [[$l]];
}

sub _parseUSTemp {
    my $temp = shift;

    $temp =~ /(.)(.*)/;
    return sprintf "%.1f", ($1 eq '1' ? -1 : 1) * $2 / 10;
}

sub _parsePhenomDescr {
    my ($r, $tag, $phen_descr) = @_;

    for ($phen_descr =~ /$re_phen_desc|BBLO/g) {
        s/CONTUS/CONS/;
        s/V(LGT|THN|THIN|FBL|THK|THICK)/VRY $1/;
        s/THIN/THN/;
        s/THICK/THK/;
        s/^LOW?/LOW/;
        s/^MOD/MDT/;

        push @{$r->{$tag}},
            { FRQ      => 'isFrequent',
              OCNL     => 'isOccasional',
              INTMT    => 'isIntermittent',
              CONS     => 'isContinuous',
              THK      => 'isThick',
             'PR THK'  => 'isPrettyThick',
             'VRY THK' => 'isVeryThick',
              THN      => 'isThin',
             'PR THN'  => 'isPrettyThin',
             'VRY THN' => 'isVeryThin',
              LGT      => 'isLight',
             'PR LGT'  => 'isPrettyLight',
             'VRY LGT' => 'isVeryLight',
              FBL      => 'isFeeble',
             'PR FBL'  => 'isPrettyFeeble',
             'VRY FBL' => 'isVeryFeeble',
              MDT      => 'isModerate',
              LOW      => 'isLow',
              LWR      => 'isLower',
              ISOL     => 'isIsolated',
              CVCTV    => 'isConvective',
              DSIPTD   => 'isDissipated',
             'PAST HR' => 'inPastHour',
              BBLO     => 'baseBelowStation',
              ALOFT    => 'isAloft',
              ARND     => 'isAround',
              FREEZING => 'isFreezing',
              PTCHY    => 'isPatchy',
            }->{$_};
    }
}

# "supplementary" section of TAFs
sub _parseTAFsuppl {
    my ($metar, $msg, $base_metar) = @_;
    my $r;

    if ($$msg =~ m@^(([56])([0-9])([0-9]{3})([0-9])) (.*)@o) {
        my ($ti, $type);

        $$msg = $6;
        $type = $2 eq '5' ? 'turbulence' : 'icing';
        $r->{$type} = {
            s               => $1,
            $type . 'Descr' => $3,
            layerBase       => $4 + 0,
        };
        $ti = $5;

        # if item repeated: level is layer top
        if ($$msg =~ m@^($2$3([0-9]{3})$5) (.*)@) {
            $$msg = $3;
            $r->{$type}{s} .= ' ' . $1;
            $r->{$type}{layerTop} = $2 + 0;
        } else {
            $r->{$type}{layerThickness} = $ti;
        }
        push @{$metar->{TAFsuppl}}, $r;
        return 1;
    }

    if ($$msg =~ m@^($re_wind_shear_lvl) (.*)@o) {
        $$msg = $4;
        $r->{windShearLvl} = {
            s     => $1,
            level => $2 + 0,
            wind  => _parseWind $3
        };
        push @{$metar->{TAFsuppl}}, $r;
        return 1;
    }

    if ($$msg =~ m@^(T(M)?([0-9]{2})/($re_hour)Z) (.*)@o) {
        $$msg = $5;
        $r->{s} = $1;
        $r->{temp} = $3 + 0;
        $r->{temp} *= -1 if defined $2;
        $r->{unitTemp} = 'C';
        $r->{hour} = $4;
        push @{$base_metar->{TAFsuppl}}, { tempAt => $r };
        return 1;
    }

    return 0;
}

########################################################################
# parseReport
########################################################################

=item parseReport()

This is the main function of the module.

The following arguments are expected:

=over

=item B<msg>

string that contains the message

=item B<is_taf>

boolean value, indicating whether it is a TAF message or not

=back

=back

=head1 RETURN VALUE

The return value is a hash with all the parsed components of the message. It
may also have values for the following keys:

=over

=item C<ERROR>

If a message could not be parsed C<ERROR> is a hash with the the keys I<descr>
(for where the error occured) and I<pos> - the original message with the
position of the error marked as C<< <@> >>.

=item C<WARNING>

The hash value is a string describing problems which where encountered during
parsing but which didn't prevent complete parsing.

=back

The message string is parsed in the following steps:

=cut

sub parseReport {
    my ($msg, $is_taf) = @_;

    my (%metar, $msg_hdr, $corr_msg);
    my ($day, $hour, $minute);
    my ($curr_day, $curr_hour, $curr_minute);
    my ($ii, $td, @cy, $delim, $tt);

=pod

Leading and trailing spaces are removed, multiple spaces are replaced by a
single one. Characters that are invalid in HTML or XML are also removed.

=cut

    $msg =~ s/ +/ /g;
    $msg =~ s/ $//;
    $msg =~ s/^ //;
    $msg =~ s/[^ -~]/?/gos; # avoid invalid XML and HTML

    $metar{msg} = $msg;

=pod

If the message starts with C<METAR>, C<SPECI>, or C<TAF> the argument
specifying the message type is overwritten.

=cut

    $msg_hdr = '';
    if ($msg =~ s/^(METAR )//) {
        $is_taf = 0;
        $msg_hdr = $1;
    } elsif ($msg =~ s/^(SPECI )//) {
        $is_taf = 0;
        $msg_hdr = $1;
        $metar{SPECI} = undef;
    } elsif ($msg =~ s/^(TAF )//) {
        $is_taf = 1;
        $msg_hdr = $1;
    }

    $metar{WARNING} = '';

    $msg .= ' '; # this makes parsing much easier

=pod

The message is checked for typical errors and corrected. Errors can be:

=over 2

=item

invalid characters (everything except capital letters, digits, dot, slash,
dollar sign, space, plus sign, minus sign)

=item

QNH with spaces

=item

temperature or dew point with spaces

=item

misspelled key words, or with missing or additional spaces

=item

missing key words

=item

wrong order of keywords

=item

removal of slashes before and after some components

=back

If the message is changed there will be a C<WARNING>.

=cut

    # EXTENSION: preprocessing
    # remove trailing =
    $msg =~ s/= $/ /os;

    # replace invalid characters
    $msg =~ s/[^A-Z0-9.,\/\$ +-]/?/gos;

    # QNH with spaces
    $msg =~ s/(?<= A) (?=[23][0-9]{3} )//o;
    $msg =~ s/(?<= Q) (?=[01][0-9]{3} )//o;
    $msg =~ s/(?<= (?:A[23]|Q[01])) (?=[0-9]{3} )//o;
    $msg =~ s/(?<= (?:A[23]|Q[01])[0-9]) (?=[0-9]{2} )//o;
    $msg =~ s/(?<= (?:A[23]|Q[01])[0-9]{2}) (?=[0-9] )//o;
    $msg =~ s/(?<= )(?:QNH)([23][0-9]{3})(?:INS)(?= )/A$1/o;

    # misspelled key words, or with missing or additional spaces
    $msg =~ s/ (?:CAMOK|CC?VOK) / CAVOK /o;
    $msg =~ s/ (?:NO(?: S)?IG|NOSI(?: G)?|N[L ]OSIG|NSI?G|(?:MO|N0)SIG|NOS I ?G) / NOSIG /o;
    $msg =~ s/(?<! )(?=(?:TEMPO|BECMG) )/ /o; # BLU+BLU+(TEMPO|BECMG)
    $msg =~ s/ (?:R MK|RMKS|RRMK) / RMK /o;
    $msg =~ s@(?<= RMK)(?=[^ /])@ @o;
    $msg =~ s@ 0VC(?=$re_cloud_base$re_cloud_type?|$re_cloud_type )@ OVC@og;
    $msg =~ s@ BNK(?=$re_cloud_base$re_cloud_type?|$re_cloud_type )@ BKN@og;
    $msg =~ s@( $re_cloud_type| FULYR)[/-](?=$re_compass_dir16 )@$1 @og;
    $msg =~ s@(?<= VR) (?=B$re_wind_speed$re_wind_speed_unit )@@og;
    $msg =~ s@(?<=[0-9]KT)S(?= )@@og;
    $msg =~ s@(?<= R) (?=WY)@@og;
    $msg =~ s@(?<= RW) (?=Y)@@og;
    $msg =~ s@(?<= RWY$re_rwy) (?=[LCR] )@@og;
    $msg =~ s@ RNW @ RWY @og;
    $msg =~ s@/(?=AURBO )@ @o;
    $msg =~ s@(?<= A)0(?=[12] )@O@o;
    $msg =~ s@(?<= ALQ)(?=S )@D@o;
    $msg =~ s@(?<= O)VH(?= )@HD@o;
    $msg =~ s@(?<= OCNL)Y(?= )@@o;
    $msg =~ s@(?<= PK)(?=WND )@ @o;
    $msg =~ s@( $re_phen_desc L)GT(?=$re_ltg_types+)@$1TG@o;
    $msg =~ s@( $re_phen_desc LTG) (?=$re_ltg_types+)@$1@o;
    $msg =~ s@ LIGHTNING @ LTG @og;
    $msg =~ s@ LTNG @ LTG @og;
    $msg =~ s@(?<= I)N(?=OVC )@@o;
    $msg =~ s@(?<= SP)O(?=TS )@@o;
    $msg =~ s@(?<= W) (?=IND )@@o;
    $msg =~ s@(?<= WIND RWY) (?=$re_rwy_des )@@o;
    $msg =~ s@(?<= (?:RWY|THR))($re_rwy_des ${re_wind_dir}0$re_wind_speed) (?=$re_wind_speed_unit )@$1@og;
    $msg =~ s@(?<= N)ORTH(?= )@@o;
    $msg =~ s@(?<= S)OUTH(?= )@@o;
    $msg =~ s@(?<= E)AST(?= )@@o;
    $msg =~ s@(?<= W)EST(?= )@@o;
    $msg =~ s@(?<= B)A(?=NK )@@o;
    $msg =~ s@(?<= F)(?= BNK )@G@o;
    $msg =~ s@( [+-]?)(RA|SN)SH @$1SH$2 @go;
    $msg =~ s@(?<= ISOL)D(?= )@@o;
    $msg =~ s@(?<= H)A(?=ZY )@@o;
    $msg =~ s@(?<= DS)(?=T )@N@o;
    $msg =~ s@(?<= D)IS(?=T )@SN@o;
    $msg =~ s@(?<= PL)(?=ME )@U@o;
    $msg =~ s@(?<= BL)K @ACK@o;
    $msg =~ s@(?<= INVIS)T(?= )@@o;
    $msg =~ s@(?<= FROIN)/ ?@ @o;
    $msg =~ s@(?<= AS)(?=CTD )@O@o;
    $msg =~ s@(?<=^[A-Z][A-Z0-9]{3} $re_day$re_hour$re_min)(?= )@Z@o
        unless $is_taf;
    $msg =~ s@(?<=^[A-Z][A-Z0-9]{3} $re_day$re_hour$re_min)KT(?= )@Z@o;
    $msg =~ s@(?<=^[A-Z][A-Z0-9]{3} $re_day$re_hour${re_min}Z )(${re_wind_dir}0)/(?=${re_wind_speed}$re_wind_speed_unit )@$1@o;
    $msg =~ s@(?<= [0-2]) (?=[0-9]0${re_wind_speed}$re_wind_speed_unit )@@go;
    $msg =~ s@(?<= 3) (?=[0-6]0${re_wind_speed}$re_wind_speed_unit )@@go;
    $msg =~ s@(?<=M)/(?=S )@P@o;
    $msg =~ s@(?<=[+]) (?=$re_weather_w_i )@@o;
    $msg =~ s@(?<=$re_cloud_base )(M?[0-9]{2}/)/(?=M?[0-9]{2} )@$1@o;
    $msg =~ s@(?<= BECMG| TEMPO)(?=$re_hour(?:$re_hour|24) )@ @go;
    $msg =~ s@ PRECIP @ PCPN @o;
    $msg =~ s@( T[XN]M?[0-9]{2}/(?:$re_hour|24)Z)(?=T[XN]M?[0-9]{2}/(?:$re_hour|24)Z )@$1 @o;
    $msg =~ s@/($re_snw_cvr_title)@ $1@;
    $msg =~ s@(?<= RWY0) (?=[1-9] )@@og;
    $msg =~ s@(?<= RWY[12]) (?=[0-9] )@@og;
    $msg =~ s@(?<= RWY3) (?=[0-6] )@@og;
    $msg =~ s@ DRFTG SNW @ DRSN @og;
    $msg =~ s@(?<= T)O(?=[0-9]{3}[01][0-9]{3} )@0@og;

    # missing key words
    $msg =~ s@ (?:TEMPO|INTER) ($re_hour$re_min)/($re_hour$re_min)(?= )@ TEMPO FM$1 TL$2@og;
    $msg =~ s@(?<! (?:BECMG|TEMPO) )(?=FM$re_hour$re_min )@BECMG @og
        unless $is_taf;

    # wrong order of keywords
    $msg =~ s@( DSNT| DISTANT)( (?:$re_weather|SMOKE|HAZE|$re_cloud_type|LTG)(?: TO)?)(?=$re_phen_locdirs )@$2$1@og;
    $msg =~ s@( $re_weather|SMOKE|HAZE| $re_cloud_type| LTG)((?: TO)?$re_phen_locdirs|$re_num_quadrant) (DSNT|DISTANT) @$1 $3$2 @og;

    # komma removed
    $msg =~ s@^((?:U|ZM).* RMK.*QFE[0-9][0-9][0-9]) (?=[0-9] )@$1,@;
    $msg =~ s@^(FQ.* RMK.*TX/[0-9][0-9]) (?=[0-9] )@$1,@;

    $corr_msg = $msg;
    if ($metar{msg} . ' ' ne $msg_hdr . $msg) {
        # we appended a blank, don't show it in warning
        my $msg_mod;
        $msg_mod = "$msg_hdr$msg";
        $msg_mod =~ s/ $//;
        $metar{WARNING} .= "WARNING: message modified\n to: $msg_mod\n"
    }

=pod

Then the components of the message are checked starting with the observation
station and the observation time (or issue time for TAF).

=cut

    if ($msg !~ /^([A-Z][A-Z0-9]{3}) (.*)/o) {
        $metar{ERROR}{descr} = 'obsStation';
        $metar{ERROR}{pos}   = substr($corr_msg, 0,
                                      (length $corr_msg) - length $msg)
                               . '@> ' . $msg;
        return %metar;
    }
    $metar{obsStation}{id} = $1;
    $metar{obsStation}{s} = $1;
    $msg = $2;

    @cy = $metar{obsStation}{id} =~ /(((.).)..)/;

    $tt = $is_taf ? 'issueTime' : 'obsTime';
    if ($msg =~ /^($re_day)($re_hour)($re_min)Z (.*)/o) {
        $msg = $4;
        $metar{$tt}{s} = "$1$2$3Z";
        @{$metar{$tt}}{'day', 'hour', 'minute'} = ($1, $2, $3);
    } elsif ($msg =~ /^($re_day$re_hour${re_min}KT) (.*)/o) {
        $msg = $2;
        $metar{$tt}{s} = $1;
        $metar{$tt}{invalidFormat} = $1;
    } elsif ($is_taf && $msg =~ /^$re_day$re_hour$re_min /o) {
        $metar{issueTime}{s} = '';
        $metar{issueTime}{invalidFormat} = '';
    } elsif ($msg =~ /^(NIL) $/o) {
        # EXTENSION: NIL instead of issueTime
        $metar{reportModifier}{modifierType} = $metar{reportModifier}{s} = $1;
        return %metar;
    } else {
        $metar{ERROR}{descr} = 'obsTime';
        $metar{ERROR}{pos}   = substr($corr_msg, 0,
                                      (length $corr_msg) - length $msg)
                               . '@> ' . $msg;
        return %metar;
    }

=pod

For METAR messages the next component may optionally be the report modifier to
indicate a missing message (C<NIL>), a message created by an automated station
(C<AUTO>), or an corrected (C<COR>) or retarded (C<RTD>) message. For Canada,
the component BBB may appear to indicate that the report has been retarded
(C<RR?>), corrected (C<CC?>), amended (C<AA?>), or segmented (C<P??>).

=cut

    # EXTENSION: BBB (for Canada)
    if (!$is_taf && $msg =~ /^(NIL|AUTO|COR|RTD|$re_bbb) (.*)/o) {
        $msg = $2;
        $metar{reportModifier}{s} = $1;
        if ($metar{reportModifier}{s} =~ 'NIL|AUTO|COR|RTD') {
            $metar{reportModifier}{modifierType} = $metar{reportModifier}{s};
        } else {
            $metar{reportModifier}{s} =~ '(.)(.)(.)';
            if ($1 eq 'P') {
                $metar{reportModifier}{modifierType} = $1;
                $metar{reportModifier}{segment} = "$2$3";
                $metar{reportModifier}{isLastSegment} = undef if $2 eq 'Z';
            } else {
                $metar{reportModifier}{modifierType} = "$1$2";
                if ($3 eq 'Z') {
                    $metar{reportModifier}{over24hLate}  = undef;
                } elsif ($3 eq 'Y') {
                    $metar{reportModifier}{sequenceLost} = undef;
                } else {
                    $metar{reportModifier}{bulletinSeq} = $3;
                }
            }
        }
        return %metar if $metar{reportModifier}{modifierType} eq 'NIL';
    }

=pod

For TAF messages there may follow the modifier C<NIL> (which terminates the
message), C<COR> (corrected), or C<AMD> (amended). Then follows the forecast
period and optionally C<CNL> to indicate a cancelled forecast.

=cut

    if ($is_taf) {
        if ($msg =~ /^(NIL|COR|AMD) (.*)/o) {
            $msg = $2;
            $metar{reportModifier}{s} = $1;
            $metar{reportModifier}{modifierType} = $1;
            return %metar if $1 eq 'NIL';
        }

        if ($msg =~ m/^($re_day)($re_hour)($re_hour|24) (.*)/o) {
            $msg = $4;
            $metar{fcstPeriod}{s} = "$1$2$3";
            @{$metar{fcstPeriod}}{'day', 'hourFrom', 'hourTill'} = ($1, $2, $3);
        } else {
            $metar{ERROR}{descr} = 'fcstPeriod';
            $metar{ERROR}{pos}   = substr($corr_msg, 0,
                                          (length $corr_msg) - length $msg)
                                   . '@> ' . $msg;
            return %metar;
        }

        if ($msg =~ m/^(CNL) (.*)/o) {
            $msg = $2;
            $metar{fcstCancelled}{s} = "$1";
            return %metar;
        }

        # EXTENSION: NIL after fcstPeriod
        if ($msg =~ m/^(NIL) (.*)/o) {
            $msg = $2;
            $metar{reportModifier}{s} = $1;
            $metar{reportModifier}{modifierType} = $1;
            return %metar;
        }

        # EXTENSION: NOT AVBL after fcstPeriod. Remarks may follow
        if ($msg =~ m/^(FCST NOT AVBL DUE NO OBS) (.*)/o) {
            $msg = $2;
            $metar{fcstNotAvbl}{s} = $1;
            $metar{fcstNotAvbl}{fcstNotAvblReason} = 'NOOBS';
        }
        if ($msg =~ m/^(FCST NOT AVBL DUE INSUFFICIENT OBS) (.*)/o) {
            $msg = $2;
            $metar{fcstNotAvbl}{s} = $1;
            $metar{fcstNotAvbl}{fcstNotAvblReason} = 'INSUFFICIENTOBS';
        }
    }

=pod

The first weather group is the surface wind. It may include a gust speed; the
wind speed and/or the wind direction may also be missing. There may be a second
group to report a varying wind direction.

=cut

    # EXTENSION: allow ///// (wind not available)
    # EXTENSION: allow /// for wind direction (not available)
    # EXTENSION: allow // for wind speed (not available)
    if ($msg =~ m@^(/////|(?:$re_wind)) (.*)@o)
    {
        $msg = $6;
        $metar{sfcWind}{s} = $1;
        $metar{sfcWind}{wind} = _parseWind $1;
    # EXTENSION: allow invalid formats
    } elsif ($msg =~ m@^(${re_wind_dir}0$re_wind_speed|[0-9]{6}$re_wind_speed_unit) (.*)@o) {
        $metar{sfcWind}{wind}{invalidFormat} = $1;
        $metar{sfcWind}{s} = $1;
        $msg = $2;
    # EXTENSION: allow missing wind
    } else {
        $metar{WARNING} .= "WARNING: No wind\n";
    }

    # EXTENSION:
    # Annex 3 Appendix 3 4.1.4.2.b.1 requires wind speed >=3 kt and
    #   variation 60-180 for this group: not checked
    if ($msg =~ /^(${re_wind_dir}[0-9])V(${re_wind_dir}[0-9]) (.*)/o) {
        if (exists $metar{sfcWind}) {
            $metar{sfcWind}{s} .= " ";
        } else {
            # TODO?: actually not "not available" but missing
            $metar{sfcWind}{wind}{dirNotAvailable} = undef;
            $metar{sfcWind}{wind}{speedNotAvailable} = undef;
            $metar{sfcWind}{s} = '';
        }
        $metar{sfcWind}{wind}{windVarLeft} = $1 + 0;
        $metar{sfcWind}{wind}{windVarRight} = $2 + 0;
        $metar{sfcWind}{s} .= "$1V$2";
        $msg = $3;
    }

=pod

If the weather conditions allow (visbility more than 10 km, no relevant weather,
no clouds below 5000 ft) this may be indicated by the keyword C<CAVOK>. The next
component after that should be the temperature (see below).

=cut

    # Annex 3 Appendix 3 2.2:
    #   vis >10km, no relevant weather, no cloud below 5000 ft
    if ($msg =~ '^CAVOK (.*)') {
        $metar{CAVOK} = undef; # cancels visibility, weather and cloud
        $msg = $1;
    }

    if (!exists $metar{CAVOK}) {

=pod

Otherwise follow the prevailing visibilty. It can have a compass direction
attached or C<NDV> if no directional variations can be given. There may be an
additional group for the minimum visibilty.

=cut

        # EXTENSION: allow 16 compass directions
        # EXTENSION: allow 'M' after re_vis_m
        if ($msg =~ /^((?:(P)?($re_vis_m)M?(NDV)?|($re_vis_km))($re_compass_dir16)?)(?: ($re_vis_m)($re_compass_dir))? (.*)/o)
        {
            $msg = $9;
            $metar{visPrev}{s} = $1;
            $metar{visPrev}{compassDir} = $6 if defined $6;
            if ($7) {
                $metar{visMin}{s}          = $7 . $8;
                $metar{visMin}{distance}   = $7 + 0;
                $metar{visMin}{unitLength} = 'M';
                $metar{visMin}{compassDir} = $8;
            }
            if (defined $3) {
                $metar{visPrev}{distance}       = $3 + 0;
                $metar{visPrev}{unitLength}     = 'M';
                $metar{visPrev}{isEqualGreater} = undef if defined $2;
                $metar{visPrev}{NDV}            = undef if defined $4;
            } else {
                $metar{visPrev}{distance}   = $5;
                $metar{visPrev}{distance}   =~ s/KM//;
                $metar{visPrev}{unitLength} = 'KM';
            }
        } elsif ($msg =~ m@^(${re_vis_sm})SM (.*)@o) {
            $msg = $2;
            $metar{visPrev}{s} = "$1SM";
            _parseFraction $metar{visPrev}, $1, 'SM';
        } elsif ($is_taf && $msg =~ m@^P([1-9][0-9]*)SM (.*)@o) {
            $msg = $2;
            $metar{visPrev}{s} = "P$1SM";
            _parseFraction $metar{visPrev}, $1, 'SM';
            $metar{visPrev}{isEqualGreater} = undef;
        # EXTENSION: allow //// and misformatted entry
        } elsif ($msg =~ m@^//// (.*)@o) {
            $metar{visPrev}{notAvailable} = undef;
            $metar{visPrev}{s} = '////';
            $msg = $1;
        } elsif ($msg =~ m@^([0-9]{3})(?: ($re_vis_m)($re_compass_dir))? (.*)@o)
        {
            $metar{visPrev}{invalidFormat} = $1;
            $metar{visPrev}{s} = $1;
            if ($2) {
                $metar{visMin}{s}          = $2 . $3;
                $metar{visMin}{distance}   = $2 + 0;
                $metar{visMin}{unitLength} = 'M';
                $metar{visMin}{compassDir} = $3;
            }
            $msg = $4;
        # EXTENSION: allow invalid formats
        } elsif ($msg =~ m@^(${re_vis_m}U) (.*)@o) {
            $metar{visPrev}{invalidFormat} = $1;
            $metar{visPrev}{s} = $1;
            $msg = $2;
        } else {
            # EXTENSION: allow missing visibility even without CAVOK
            $metar{WARNING} .= "WARNING: No visibility\n";
        }

=pod

After that may follow one or more runway visibilty ranges or C<RVRNO> if they
are not available.

=cut

        while (_parseRwyVis \%metar, \$msg) {};

        # EXTENSION: allow RVRNO (not available) (KADW, RKSG, PAED)
        while ($msg =~ m@^RVRNO (.*)@o) {
            $metar{RVRNO} = undef;
            $msg = $1;
        }

=pod

Then there can be up to 3 groups to describe the present weather.

=cut

        # EXTENSION: allow // (not available)
        while ($msg =~ m@^($re_weather|//|([+-]?(?:RATS|BRHZ|RESH|RETS|VC|VCRA|RABR))) (.*)@o) {
            if (defined $2) {
                push @{$metar{weather}}, { s => $2, invalidFormat => $2 };
            } else {
                push @{$metar{weather}}, _parseWeather $1;
            }
            $msg = $3;
        }

=pod

Then follow up to 3 groups for the sky condition (cloud cover and base or
vertical visibilty) optionally with cloud type. The keywords C<CLR>, C<SKC>, or
C<NSC> may indicate different sky conditions if no cloud cover is given.

=cut

        # EXTENSION: allow ////// (not available) without CB, TCU
        # WMO 306 Vol II
        #   CLR: no clouds below 10000 (FMH-1: 12000) ft detected by autom. st.
        # Annex 3 Appendix 3 4.5.4.1.e, f
        #   SKC: no clouds + VV not restricted but not CAVOK
        #   NSC: no significant clouds + no CB + VV not restr. but not CAVOK,SKC
        # AMOSSG/5-SoD Appendix C 4.9.1.4.b
        #   NCD: no clouds + CB, TCU detected by automatic observation system
        while ($msg =~ m@^((SKC|NSC|CLR|NCD)|VV($re_cloud_base)|(/{6}|$re_cloud_cov(?:$re_cloud_base(?: ?$re_cloud_type)?|$re_cloud_type))|($re_cloud_cov[0-9]{1,2}|$re_cloud_cov(?: [0-9]{1,3})?)) (.*)@o)
        {
            $msg = $6;
            if (defined $5) {
                push @{$metar{cloud}}, { s => $1, invalidFormat => $1 };
            } elsif (defined $2) {
                push @{$metar{cloud}}, { noClouds => $2, s => $1 };
            } elsif (defined $3) {
                $metar{visVert}{s} = $1;
                if ($3 eq '///') {
                    $metar{visVert}{notAvailable} = undef;
                } else {
                    $metar{visVert}{distance}   = $3 + 0;
                    $metar{visVert}{unitHeight} = 'FL';
                }
            } else {
                push @{$metar{cloud}}, _parseCloud $4;
            }
        }
        _determineCeiling $metar{cloud} if exists $metar{cloud};
    }

=pod

The following group contains the current air temperature and optionally the dew
point. If both are given the humidity can be determined.

=cut

    # EXTENSION: FMH-1: dew point is optional
    # EXTENSION: allow // for temperature and dew point
    # EXTENSION: allow XX for temperature and dew point
    if ($msg =~ '^((?:(M)?([0-9] ?[0-9])|(?://|XX))/((M)? ?([0-9] ?[0-9])|(?://|XX))?) (.*)')
    {
        $msg = $7;
        $metar{temperature}{s} = $1;
        if (defined $3) {
            ($metar{temperature}{air}{temp} = $3) =~ tr / //d;
            $metar{temperature}{air}{temp} += 0;
            $metar{temperature}{air}{temp} *= -1 if defined $2;
            $metar{temperature}{air}{unitTemp} = 'C';
        } else {
            $metar{temperature}{air}{notAvailable} = undef;
        }
        if (defined $4) {
            if (defined $6) {
                ($metar{temperature}{dewpoint}{temp} = $6) =~ tr / //d;
                $metar{temperature}{dewpoint}{temp} += 0;
                $metar{temperature}{dewpoint}{temp} *= -1 if defined $5;
                $metar{temperature}{dewpoint}{unitTemp} = 'C';
            } else {
                $metar{temperature}{dewpoint}{notAvailable} = undef;
            }
        }
        if (   exists $metar{temperature}{air}
            && exists $metar{temperature}{air}{temp}
            && exists $metar{temperature}{dewpoint}
            && exists $metar{temperature}{dewpoint}{temp})
        {
            my $t = $metar{temperature}{air}{temp};
            my $d = $metar{temperature}{dewpoint}{temp};

            # FGFS metar
            $metar{temperature}{relHumid1} =
                    100 * 10 ** (7.5 * ($d / ($d + 237.7) - $t / ($t + 237.7)));

            # http://www.bragg.army.mil/www-wx/wxcalc.htm
            # http://www.srh.noaa.gov/bmx/tables/rh.html
            $metar{temperature}{relHumid2} =
                      100 * ((112 - (0.1 * $t) + $d) / (112 + (0.9 * $t))) ** 8;

            # http://www.mattsscripts.co.uk/mweather.htm
            # http://ingrid.ldeo.columbia.edu/dochelp/QA/Basic/dewpoint.html
            $metar{temperature}{relHumid3} =
           100 * (6.11 * exp(5417.118093 * (1 / 273.15 - (1 / ($d + 273.15)))))
               / (6.11 * exp(5417.118093 * (1 / 273.15 - (1 / ($t + 273.15)))));

            # http://de.wikipedia.org/wiki/Taupunkt
            $metar{temperature}{relHumid4} =
                             100 * (6.11213 * exp(17.5043 * $d / (241.2 + $d)))
                                 / (6.11213 * exp(17.5043 * $t / (241.2 + $t)));
        }
    } else {
        $metar{WARNING} .= "WARNING: No temperature\n" unless $is_taf;
    }

=pod

The last regular item is the QNH given in hectopascal or in. Hg.

=cut

    # EXTENSION: allow // for last 2 digits
    # EXTENSION: allow tenths of hPa as .[0-9] (FLLS, HUEN, OPRN)
    # EXTENSION: allow 2 QNHs (OIxx), use first
    $ii = 0;
    while (   $ii < 2
        && $msg =~ '^((?:Q[01]|A[23])[0-9](?:[0-9]{2}|//)|[AQ]////|Q[01][0-9]{3}\.[0-9]) (.*)')
    {
        $msg = $2;
        next if $ii++;

        $metar{QNH}{s} = $1;
        my ($unit, $dig12, $dig34) = $1 =~ '(.)(..)(.*)';
        if ("$dig12$dig34" eq '////') {
            $metar{QNH}{notAvailable} = undef;
        } else {
            $dig34 = '00' if $dig34 eq '//';
            if ($unit eq 'Q') {
                $metar{QNH}{hPa} = "$dig12$dig34" + 0;
            } else {
                $metar{QNH}{inHg} = "$dig12$dig34" / 100;
            }
        }
    }
    $metar{WARNING} .= "WARNING: No QNH\n"
        unless $is_taf || exists $metar{QNH};

=pod

After that may follow country or station specific information: pressure, worst
cloud cover. There may also be groups indicating recent weather, wind shear,
runway conditions and runway winds.

=cut

    # EXTENSION: allow some (sea level?) pressure (OIxx)
    if ($msg =~ m@^([789][0-9]{2}[./][0-9]) (.*)@o) {
        $msg = $2;
        $metar{somePressure}{s} = $1;
        $metar{somePressure}{hPa} = $1;
        $metar{somePressure}{hPa} =~ s@/@.@;
    }

    # EXTENSION: allow worst cloud cover
    if (   _isValidCountry('cloudMaxCover', \@cy)
        && $msg =~ m@^($re_cloud_cov|SKC) (.*)@o)
    {
        $msg = $2;
        $metar{cloudMaxCover}{s} = $1;
        if ($1 eq 'SKC') {
            $metar{cloudMaxCover}{SKC} = undef;
        } else {
            $metar{cloudMaxCover}{cloudCover} = $1;
        }
    }

    while ($msg =~ m@^RE($re_weather_re) (.*)@o) {
        $msg = $2;
        push @{$metar{recWeather}}, _parseWeather $1, 1;
    }

    if ($msg =~ '^(WS ALL RWY) (.*)') {
        $msg = $2;
        push @{$metar{windShear}}, { s => $1, rwyDesigAll => undef };
    }
    while ($msg =~ m@^(WS RWY($re_rwy_des)) (.*)@o) {
        $msg = $3;
        push @{$metar{windShear}}, { s => $1, rwyDesig => $2 };
    }

    while (_parseRwyState \%metar, \$msg) {};

    # EXTENSION: allow colour code
    if ($msg =~ m@^($re_colour) (.*)@o) {
        $msg = $2;
        $metar{colourCode} = _parseColourCode $1;
    }

    # EXTENSION: country specific things
    if (   _isValidCountry('NEFO_PLAYA', \@cy)
        && $msg =~ m@^(NEFO PLAYA (?:([1-9][0-9]+0)FT|(SKC))) (.*)@o)
    {
        $msg = $4;
        $metar{NEFO_PLAYA}{s} = $1;
        if (defined $2) {
            $metar{NEFO_PLAYA}{cloudBaseFT} = $2 + 0;
        } else {
            $metar{NEFO_PLAYA}{SKC} = undef;
        }
    }

    # EXTENSION:
    while (   _isValidCountry('rwyWind', \@cy)
           && $msg =~ m@^($re_rwy_wind) (.*)@o)
    {
        my $r;
        $msg = $7;
        $r->{s} = $1;
        $r->{wind} = _parseWind $4;
        $r->{wind}{windVarLeft} = $5 + 0 if defined $5;
        $r->{wind}{windVarRight} = $6 + 0 if defined $6;
        $r->{rwyDesig} = defined $2 ? $2 : $3;
        push @{$metar{rwyWind}}, $r;
    }

=pod

Finally, there may be trends and remarks.

=cut

    if ($msg =~ '^(NOSIG) (.*)') {
        my $td;

        $td->{s} = $1;
        $td->{trendType} = $1;
        push @{$metar{trend}}, $td;
        $msg = $2;

        # EXTENSION: allow rwyState after NOSIG. Not considered trend data!
        while (_parseRwyState \%metar, \$msg) {};
    }

    # EXTENSION: allow RH after NOSIG. Not considered trend data!
    if (_isValidCountry('relHumid', \@cy) && $msg =~ m@^(RH([0-9]{2})) (.*)@o)
    {
        $msg = $3;
        $metar{RH}{s} = $1;
        $metar{RH}{relHumid} = $2 + 0;
    }

    # "supplementary" section of TAFs
    while ($is_taf && _parseTAFsuppl \%metar, \$msg, \%metar) {};

    # changes at midnight: 0000 with FM/AT, 2400 with TL
    # EXTENSION: allow missing period
    # EXTENSION: FM: allow time with Z
    # EXTENSION: FM: allow time without minutes
    # EXTENSION: FM: allow 24Z? or 2400Z?
    while (   (!$is_taf && $msg =~ '^(BECMG|TEMPO) (.*)')
           || ($is_taf && $msg =~ m@^(FM(?:($re_hour)($re_min)?|(24)(?:00)?)Z?|(?:(BECMG|TEMPO)|(?:PROB([34]0)( TEMPO)?))(?: ($re_hour)($re_hour|24))?) (.*)@o))
    {
        my $td;

        if ($is_taf) {
            $msg = $10;
            $td->{s} = $1;
            if (defined $2) {
                $td->{trendType} = 'BECMG';
                @{$td->{trendTime1}}{'timeSpec', 'hour'} = ('FM', $2);
                $td->{trendTime1}{minute} = defined $3 ? $3 : '00';
            } elsif (defined $4) {
                $td->{trendType} = 'BECMG';
                @{$td->{trendTime1}}{'timeSpec', 'hour'} = ('FM', $4);
                $td->{trendTime1}{minute} = '00';
            } else {
                if (defined $5) {
                    $td->{trendType} = $5;
                } else {
                    $td->{probability} = $6;
                    $td->{trendType} = defined $7 ? 'TEMPO' : 'PROB';
                }
                if (defined $8) {
                    @{$td->{trendTime1}}{'timeSpec', 'hour', 'minute'}
                                                             = ('FM', $8, '00');
                    @{$td->{trendTime2}}{'timeSpec', 'hour', 'minute'}
                                                             = ('TL', $9, '00');
                } else {
                    $metar{WARNING} .= "WARNING: period of occurrence missing for $1\n";
                }
            }
        } else {
            $msg = $2;
            $td->{s} = $1;
            $td->{trendType} = $1;

            if ($msg =~ m@^((?:TL2400|(?:FM|TL|AT)$re_hour$re_min)Z?) (.*)@o) {
                $msg = $2;
                $td->{s} .= " $1";
                @{$td->{trendTime1}}{'timeSpec', 'hour', 'minute'} =
                                                           $1 =~ /(..)(..)(..)/;

                if ($msg =~ m@^TL(2400Z?|$re_hour${re_min}Z?) (.*)@o) {
                    $td->{s} .= " TL$1";
                    $msg = $2;
                    $td->{trendTime2}{timeSpec} = 'TL';
                    @{$td->{trendTime2}}{'hour', 'minute'} = $1 =~ /(..)(..)/;
                }
            }
        }

        if ($msg =~ m@^($re_wind) (.*)@o)
        {
            $msg = $6;
            $td->{sfcWind}{s} = $1;
            $td->{sfcWind}{wind} = _parseWind $1;
        }

        if ($msg =~ '^CAVOK (.*)') {
            $td->{CAVOK} = undef;
            $msg = $1;
        }

        if (!exists $td->{CAVOK}) {
            # EXTENSION: allow 'M' after re_vis_m
            if ($msg =~ m@^(($re_vis_m)M?|($re_vis_km)|(${re_vis_sm})SM) (.*)@o)
            {
                $msg = $5;
                $td->{visPrev}{s} = $1;
                if (defined $2) {
                    $td->{visPrev}{distance}   = $2 + 0;
                    $td->{visPrev}{unitLength} = 'M';
                } elsif (defined $3) {
                    $td->{visPrev}{distance}   = $3;
                    $td->{visPrev}{distance}   =~ s/KM//;
                    $td->{visPrev}{unitLength} = 'KM';
                } else {
                    _parseFraction $td->{visPrev}, $4, 'SM';
                }
            } elsif ($is_taf && $msg =~ m@^P([1-9][0-9]*)SM (.*)@o) {
                $msg = $2;
                $td->{visPrev}{s} = "P$1SM";
                _parseFraction $td->{visPrev}, $1, 'SM';
                $td->{visPrev}{isEqualGreater} = undef;
            } elsif ($msg =~ m@^([0-9]{3}) (.*)@o) {
                $td->{visPrev}{s} = $1;
                $td->{visPrev}{invalidFormat} = $1;
                $msg = $2;
            }

            # EXTENSION: allow VC..
            while ($msg =~ m@^($re_weather|NSW) (.*)@o) {
                push @{$td->{weather}}, _parseWeather $1;
                $msg = $2;
            }

            $ii = 0;
            while ($msg =~ m@^((SKC|NSC)|VV($re_cloud_base)|($re_cloud_cov$re_cloud_base$re_cloud_type?|$re_cloud_cov$re_cloud_type)) (.*)@o)
            {
                $msg = $5;
                if (defined $2) {
                    push @{$td->{cloud}}, { noClouds => $2, s => $1 };
                } elsif (defined $3) {
                    $td->{visVert}{s} = $1;
                    if ($3 eq '///') {
                        $td->{visVert}{notAvailable} = undef;
                    } else {
                        $td->{visVert}{distance}   = $3 + 0;
                        $td->{visVert}{unitHeight} = 'FL';
                    }
                } else {
                    push @{$td->{cloud}}, _parseCloud $4;
                }
            }
            _determineCeiling $td->{cloud} if exists $td->{cloud};

            # EXTENSION: allow colour code
            if ($msg =~ m@^($re_colour) (.*)@o) {
                $msg = $2;
                $td->{colourCode} = _parseColourCode $1;
            }
        }

        # EXTENSION: allow rwyState after trend BECMG / TEMPO
        while (_parseRwyState $td, \$msg) {};

        # "supplementary" section of TAFs
        while ($is_taf && _parseTAFsuppl $td, \$msg, \%metar) {};

        push @{$metar{trend}}, $td;
    }

    while ($is_taf && $msg =~ m@^(T([XN])(M)?([0-9]{2})/($re_hour|24)Z) (.*)@o){
        my $r;

        $msg = $6;
        $metar{tempMaxMin} = () unless exists $metar{tempMaxMin};
        $r->{s} = $1;
        $r->{temp} = $4 + 0;
        $r->{temp} *= -1 if defined $3;
        $r->{unitTemp} = 'C';
        $r->{hour} = $5;
        push @{$metar{tempMaxMin}},
            { ($2 eq 'N' ? 'tempMin' : 'tempMax') => $r };
    }

    if ($msg =~ '^RMK[ /] ?(.*)') {
        my ($notRecognised, $parsed);

        @{$metar{remark}} = ();
        $msg = $1;

        $notRecognised = '';
        while ($msg ne '') {
            my $r;

            $parsed = 1;
            $r = {};
            if (   $notRecognised =~ /DUE(?: TO)?(?: $re_phen_desc| GROUND)?$/
                || $notRecognised =~ /BY(?: $re_phen_desc| GROUND)?$/)
            {
                $parsed = 0;
            } elsif (_parseRwyState $r, \$msg) {
                push @{$metar{remark}}, { rwyState => $r->{rwyState}[0] };
            } elsif (_parseRwyVis $r, \$msg) {
                push @{$metar{remark}}, { visRwy => $r->{visRwy}[0] };
            } elsif (   _isValidCountry('remarkCloudMaxCov', \@cy)
                && $msg =~ m@^($re_cloud_cov) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, {
                    cloudMaxCover => { s => $1, cloudCover => $1 },
                };
            } elsif (   _isValidCountry('remarkColour', \@cy)
                && $msg =~ m@^($re_colour) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { colourCode => _parseColourCode $1 };
            } elsif (   _isValidCountry('remarkCloudTypeFamily', \@cy)
                     && $msg =~ m@^(8/([0-9/])([0-9/])([0-9/])) (.*)@o)
            {
                $msg = $5;
                push @{$metar{remark}}, { cloudTypeFamily => {
                    s => $1,
                    $2 eq '/' ? ('cloudTypeLowAboveOvercast', undef)
                              : ('cloudTypeLow', $2),
                    $3 eq '/' ? ('cloudTypeMiddleAboveOvercast', undef)
                              : ('cloudTypeMiddle', $3),
                    $4 eq '/' ? ('cloudTypeHighAboveOvercast', undef)
                              : ('cloudTypeHigh', $4),
                }};
            } elsif ($msg =~ m@^(PWINO|FZRANO|TSNO|PNO|RVRNO|NO ?SPECI|VIA PHONE|RCRNR|HALO(?: VISBL)?|(?:LGT |HVY )?FROIN|SD[FGP]/HD[FGP]|VRBL CONDS|ACFT MSHP)[/ ](.*)@o)
            {
                $msg = $2;
                $r->{s} = $1;
                ($r->{keyword} = $1) =~ tr/ /_/;
                $r->{keyword} =~ s/NO_?SPECI/NOSPECI/;
                $r->{keyword} =~ s/_VISBL$//;
                push @{$metar{remark}}, { keyword => $r };
            } elsif ($msg =~ m@^(?:/) (.*)@o){
                $msg = $1;
                push @{$metar{remark}},
                                  { keyword => { s => '/', keyword => 'slash'}};
            } elsif ($msg =~ m@^(?:\$) (.*)@o){
                $msg = $1;
                push @{$metar{remark}}, { needMaint => { s => '$' }};
            } elsif (   _isValidCountry('remarkCIGRAG', \@cy)
                     && $msg =~ m@^(CIG (?:RAG|RGD)) (.*)@o)
            {
                $msg = $2;
                $r->{s} = $1;
                ($r->{keyword} = $1) =~ tr/ /_/;
                $r->{keyword} =~ s/RGD/RAG/;
                push @{$metar{remark}}, { keyword => $r };
            } elsif (   _isValidCountry('remarkCIGVRBL', \@cy)
                     && $msg =~ m@^(CIG VRBL? ($re_vis_sm)-($re_vis_sm)) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                $r->{visibilityFrom} = {};
                $r->{visibilityTo} = {};
                _parseFraction $r->{visibilityFrom}, $2, 'SM';
                _parseFraction $r->{visibilityTo}, $3, 'SM';
                push @{$metar{remark}}, { ceilVisVariable => $r };
            } elsif (   _isValidCountry('remarkSLPdPa', \@cy)
                     && $msg =~ m@^(SLP ?(?:([0-9]{2})([0-9])|NO|///)) (.*)@o)
            {
                $msg = $4;

                if (!defined $2) {
                    push @{$metar{remark}}, { SLPdPa => {
                        s => $1,
                        notAvailable => undef
                    }};
                # TODO: really need QNH? but values are REALLY close:
                # KLWM 161454Z AUTO 31012G21KT 10SM CLR M10/M18 A2808 RMK SLP511
                #   -> 951.1 hPa
                # KQHT 021355Z 26003KT 8000 IC M15/M16 A3070 RMK SLP484
                #   -> 1048.4 hPa
                # higher QNHs possible:
                # ETNS 161520Z 11006KT 7000 FEW020 M02/M05 Q1078 WHT WHT
                #   -> SLP78?
                # UNWW 021400Z 00000MPS CAVOK M25/M29 Q1056 NOSIG RMK QFE759
                #   -> SLP56?
                } elsif (!exists $metar{QNH} || !exists $metar{QNH}{inHg}) {
                    if ($2 > 80 || $2 < 20) { # only if within sensible range
                        my $slp;
                        $slp = "$2.$3";
                        $slp += $slp < 50 ? 1000 : 900;
                        push @{$metar{remark}}, { SLPdPa => {
                            s => $1,
                            hPa => sprintf '%.1f', $slp
                        }};
                    } else {
                        push @{$metar{remark}}, { SLPdPa => {
                            s => $1,
                            invalidFormat => "no QNH, x$2.$3 hPa"
                        }};
                    }
                } else {
                    my ($slp, @slp, @qnh, $qnhHPa);

                    @slp = ($1, $2, $3);
                    $qnhHPa = rnd($metar{QNH}{inHg} * $INHG2HPA, 1);

                    $slp = $qnhHPa;                       # start with given QNH
                    $slp =~ s/..$//;                  # remove 2 trailing digits
                    $slp .= "$slp[1].$slp[2]";              # append given value
                    $slp += 100 if $slp + 50 < $qnhHPa;     # ensure SLP > QNH
                    push @{$metar{remark}}, { SLPdPa => {
                        s => $slp[0],
                        hPa => sprintf '%.1f', $slp
                    }};
                }
            } elsif (   _isValidCountry('remarkSPinHg', \@cy)
                     && $msg =~ m@^(SP([23][0-9]\.[0-9]{3})) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { SLPinHg => {
                    s    => $1,
                    inHg => $2
                }};
            } elsif (   _isValidCountry('remarkSLPinHg', \@cy)
                     && $msg =~ m@^(SLP(?:([0-9]{3})|NO)) (.*)@o)
            {
                $msg = $3;

                if (!defined $2) {
                    push @{$metar{remark}}, { SLPinHg => {
                        s => $1,
                        notAvailable => undef
                    }};
                } elsif (!exists $metar{QNH} || !exists $metar{QNH}{inHg}) {
                    push @{$metar{remark}}, { SLPinHg => {
                        s => $1,
                        invalidFormat => "no QNH, x$2 in. Hg"
                    }};
                } else {
                    my ($qnhInHg, @slp);

                    @slp = ($1, $2);
                    $qnhInHg = $metar{QNH}{inHg} * 100;
                    $qnhInHg =~ s/...$//;
                    $qnhInHg .= "$slp[1]";
                    $qnhInHg /= 100;
                    $qnhInHg += 10 if $qnhInHg + 5 < $metar{QNH}{inHg};
                    push @{$metar{remark}}, { SLPinHg => {
                        s => $slp[0],
                        inHg => $qnhInHg
                    }};
                }
            } elsif (   _isValidCountry('remarkGRIDWind', \@cy)
                     && $msg =~ m@^GRID(${re_wind_dir}0${re_wind_speed}KT) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { gridWind => {
                    s    => "GRID$1",
                    wind => _parseWind $1
                }};
            } elsif (   _isValidCountry('remarkRwyWind', \@cy)
                     && $msg =~ m@^($re_rwy_wind) (.*)@o)
            {
                $msg = $7;
                $r->{rwyWind}{s} = $1;
                $r->{rwyWind}{rwyDesig} = defined $2 ? $2 : $3;
                $r->{rwyWind}{wind} = _parseWind $4;
                $r->{rwyWind}{wind}{windVarLeft} = $5 + 0 if defined $5;
                $r->{rwyWind}{wind}{windVarRight} = $6 + 0 if defined $6;
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkRwyWind2', \@cy)
                     && $msg =~ m@^($re_rwy_wind2) (.*)@o)
            {
                $msg = $4;
                $r->{rwyWind}{s} = $1;
                $r->{rwyWind}{wind} = _parseWind $2;
                $r->{rwyWind}{rwyDesig} = $3;
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkRsc', \@cy)
                     && $msg =~ m@^($re_rsc) (.*)@o)
            {
                $msg = $2;
                $r->{s} = $1;
                for ($1 =~ /(SLR|LSR|PSR|P|SANDED|WET|DRY|IR|WR|\/\/|[0-9]{2})+?/g)
                {
                    if ($_ eq '//') {
                        push @{$r->{arr}}, { notAvailable => undef };
                    } elsif ($_ =~ '[0-9]') {
                        push @{$r->{arr}}, { decelerometer => $_ };
                    } else {
                        push @{$r->{arr}}, { rwySfc => $_ };
                    }
                }
                push @{$metar{remark}}, { rwySfcCondition => $r };
            } elsif (   _isValidCountry('remarkPhenomOpacity', \@cy)
                     && $msg =~ m@^(((?:$re_phenom_opac[0-8])+)(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $6;
                $r->{s} = $1;
                $r->{phenomOpacity} = ();
                _parsePhenomOpacity $r, $2;
                $r->{cloudTypeAsoctd} = $3 if defined $4;
                $r->{cloudTypeEmbd}   = $3 if defined $5;
                push @{$metar{remark}}, { phenomOpacityList => $r };
            } elsif (   _isValidCountry('remarkPhenomOpacityRev', \@cy)
                     && $msg =~ m@^([0-8])($re_phenom_opac) (.*)@o)
            {
                $msg = $3;
                $r->{s} = "$1$2";
                $r->{phenomOpacity} = ();
                _parsePhenomOpacity $r, "$2$1";
                push @{$metar{remark}}, { phenomOpacityList => $r };
            } elsif (   _isValidCountry('remarkCloudOpacityLvl', \@cy)
                     && $msg =~ m@^(([0-8])(?:((?:BL)?SN|FG)|($re_cloud_type))([0-9]{3})(?: TO)?($re_phen_locdirs)?) (.*)@o)
            {
                $msg = $7;
                $r->{s} = $1;
                $r->{eights} = $2;
                $r->{weather} = _parseWeather $3 if defined $3;
                $r->{cloudType} = $4 if defined $4;
                $r->{cloudBase} = $5 + 0;
                $r->{locationAndList} = _parseLocations $6 if defined $6;
                push @{$metar{remark}}, { cloudOpacityLvl => $r };
            } elsif (   _isValidCountry('remarkCloudCoverVar', \@cy)
                     && $msg =~ m@^(($re_cloud_cov$re_cloud_base?) V ($re_cloud_cov)) (.*)@o)
            {
                $msg = $4;
                $r->{cloudCoverVar} = _parseCloud $2;
                $r->{cloudCoverVar}{s} = $1;
                $r->{cloudCoverVar}{cloudCover2} = $3;
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkCloudCoverLvl', \@cy)
                     && $msg =~ m@^($re_cloud_cov$re_cloud_base) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { cloud => _parseCloud $1 };
            } elsif (   _isValidCountry('remarkCloudTypeLvl', \@cy)
                     && $msg =~ m@^($re_cloud_type)($re_cloud_base) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { cloudTypeLvl => {
                    s         => "$1$2",
                    cloudType => $1,
                    cloudBase => $2 + 0
                }};
            } elsif (   _isValidCountry('remarkCloudTraces', \@cy)
                     && $msg =~ m@^(TR( LWR)?(?: CLD|((?:[/ ]$re_trace_cloud)+))) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                $r->{isLower} = undef if defined $2;
                if (defined $3) {
                    for ($3 =~ m@[/ ]($re_trace_cloud)@g) {
                        push @{$r->{cloudType}}, $_;
                    }
                } else {
                    $r->{cloudTypeNotAvailable} = undef;
                }
                push @{$metar{remark}}, { cloudTrace => $r };
            } elsif (   _isValidCountry('remarkCloudTraces', \@cy)
                     && $msg =~ m@^(((?:$re_trace_cloud[/ ])+)TR) (.*)@o)
            {
                $msg = $3;
                $r->{s} = $1;
                for ($2 =~ m@($re_trace_cloud)[/ ]@g) {
                    push @{$r->{cloudType}}, $_;
                }
                push @{$metar{remark}}, { cloudTrace => $r };
            } elsif (   _isValidCountry('remarkReWeather', \@cy)
                     && $msg =~ m@^RE($re_weather_re) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}},
                                     { recWeather => [ _parseWeather($1, 1) ] };
            } elsif (   _isValidCountry('remarkSfcVisM', \@cy)
                     && $msg =~ m@^SFC VIS (($re_vis_m)|($re_vis_km)) (.*)@o)
            {
                $msg = $4;
                $r->{s} = "SFC VIS $1";
                $r->{locationAt} = 'SFC';
                if (defined $2) {
                    $r->{visibility}{distance}   = $2 + 0;
                    $r->{visibility}{unitLength} = 'M';
                } else {
                    $r->{visibility}{distance}   = $3;
                    $r->{visibility}{distance}   =~ s/KM//;
                    $r->{visibility}{unitLength} = 'KM';
                }
                push @{$metar{remark}}, { visibilityAtLoc => $r };
            } elsif (   _isValidCountry('remarkVisSfcTwr', \@cy)
                     && $msg =~ m@^((SFC|TWR) VIS ($re_vis_sm)(?:SM)?) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                $r->{locationAt} = $2;
                $r->{visibility} = {};
                _parseFraction $r->{visibility}, $3, 'SM';
                push @{$metar{remark}}, { visibilityAtLoc => $r };
            } elsif (   _isValidCountry('remarkVisVarK', \@cy)
                     && $msg =~ m@^(VIS ($re_vis_sm)V($re_vis_sm)) (.*)@o)
            {
                $msg = $4;
                $r->{visVar1}{s} = $1;
                $r->{visVar2} = {};
                _parseFraction $r->{visVar1}, $2, 'SM';
                _parseFraction $r->{visVar2}, $3, 'SM';
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkVisVarCY', \@cy)
                     && $msg =~ m@^(VIS VRB ($re_vis_sm)-($re_vis_sm)) (.*)@o)
            {
                $msg = $4;
                $r->{visVar1}{s} = $1;
                $r->{visVar2} = {};
                _parseFraction $r->{visVar1}, $2, 'SM';
                _parseFraction $r->{visVar2}, $3, 'SM';
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkVisVarCYCZ', \@cy)
                     && $msg =~ m@^(VSBY VRBL ([0-9]\.[0-9])V([0-9]\.[0-9])) (.*)@o)
            {
                $msg = $4;
                $r->{visVar1}{s} = $1;
                $r->{visVar2} = {};
                _parseFraction $r->{visVar1}, $2, 'SM';
                _parseFraction $r->{visVar2}, $3, 'SM';
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkPCPNPastHour', \@cy)
                     && $msg =~ m@^(PCPN ([0-9]+\.[0-9])MM PAST HR) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { precipPastHour => {
                    s              => $1,
                    precipAmountMM => $2 + 0
                }};
            } elsif (   _isValidCountry('remarkVisLocM', \@cy)
                     && $msg =~ m@^VIS((?:$re_phen_locdirs $re_vis_m_km_remark)(?:(?: AND)?$re_phen_locdirs $re_vis_m_km_remark)*) (.*)@o)
            {
                my @arr;

                $msg = $2;
                $r->{visListLoc}{s} = "VIS$1";
                $r->{visListLoc}{arr} = \@arr;
                for ($1 =~ m@(?:AND)?($re_phen_locdirs $re_vis_m_km_remark)@og) {
                    my $v;

                    m@($re_phen_locdirs) (?:($re_vis_m)|([1-9][0-9]{2,3})M|($re_vis_km))@o;
                    $v->{locationAndList} = _parseLocations $1;
                    if (defined $2) {
                        $v->{visibility}{distance}   = $2 + 0;
                        $v->{visibility}{unitLength} = 'M';
                    } elsif (defined $3) {
                        $v->{visibility}{distance}   = $3  + 0;
                        $v->{visibility}{unitLength} = 'M';
                    } else {
                        $v->{visibility}{distance} = $4;
                        $v->{visibility}{distance} =~ s/KM//;
                        $v->{visibility}{unitLength} = 'KM';
                    }
                    push @arr, $v;
                }
                push @{$metar{remark}}, $r;
            } elsif ($msg =~ m@^(VIS($re_phen_locdirs) LWR) (.*)@o) {
                $msg = $3;
                push @{$metar{remark}}, { phenomenonAtLoc => {
                    s               => $1,
                    otherPhenom     => 'VIS_LWR',
                    locationAndList => _parseLocations $2
                }};
            } elsif (   _isValidCountry('remarkQNH', \@cy)
                     && $msg =~ m@^(A)([23][0-9]{3}) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { QNH => {
                    s => "$1$2",
                    inHg => $2 / 100
                }};
            } elsif (   _isValidCountry('remarkQNHMB', \@cy)
                     && $msg =~ m@^(Q([01][0-9]{3})MB) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { QNH => {
                    s => $1,
                    hPa => $2
                }};
            } elsif (   _isValidCountry('remarkQFF', \@cy)
                     && $msg =~ m@^([789][0-9]{2}) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { QFF => {
                    s => $1,
                    hPa => $1
                }};
            } elsif ($msg =~ m@^(QNH([01][0-9]{3}\.[0-9])) (.*)@o) {
                $msg = $3;
                push @{$metar{remark}}, { QNH => {
                    s => $1,
                    hPa => $2
                }};
            } elsif (   _isValidCountry('remarkCorrected', \@cy)
                     && $msg =~ m@^(COR ($re_hour)($re_min)) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { correctedAt => {
                    s      => $1,
                    hour   => $2,
                    minute => $3
                }};
            } elsif (   _isValidCountry('remarkSnowIncr', \@cy)
                     && $msg =~ m@^(SNINCR ([0-9]+)/([0-9]+)) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { snowIncr => {
                    s        => $1,
                    pastHour => $2,
                    onGround => $3
                }};
            } elsif (   _isValidCountry('remarkHourly', \@cy)
                     && $msg =~ m@^1([01][0-9]{3}) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { temp6hMax => {
                    s        => "1$1",
                    temp     => _parseUSTemp($1),
                    unitTemp => 'C'
                }};
            } elsif (   _isValidCountry('remarkHourly', \@cy)
                     && $msg =~ m@^2([01][0-9]{3}) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { temp6hMin => {
                    s        => "2$1",
                    temp     => _parseUSTemp($1),
                    unitTemp => 'C'
                }};
            } elsif (   _isValidCountry('remarkHourly', \@cy)
                     && $msg =~ m@^(5(?:([0-8])([0-9]{3})|////)) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { pressureTendency3h => {
                    s => $1,
                    defined $2 ? ('pressureTendency', $2,
                                  'pressureChange', $3 / 10)
                               : ('notAvailable', undef)
                }};
            } elsif (   _isValidCountry('remarkPeakWind', \@cy)
                     && $msg =~ m@^(PK WND (${re_wind_dir}[0-9]$re_wind_speed)/(?:($re_hour)?($re_min))) (.*)@o)
            {
                $msg = $5;
                push @{$metar{remark}}, { peakWind => {
                    s      => $1,
                    wind   => _parseWind("$2KT"),
                    defined $3 ? ('hour', $3) : (),
                    minute => $4
                }};
            } elsif (   $cy[0] eq 'KEHA'
                     && $msg =~ m@^(PK WND ($re_wind_speed) 000) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { peakWind => {
                    s    => $1,
                    wind => _parseWind("///$2KT")
                }};
            } elsif (   _isValidCountry('remarkHourlyTemp', \@cy)
                     && $msg =~ m@^(T([01][0-9]{3})([01][0-9]{3})?) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                $r->{air}{temp}          = _parseUSTemp($2);
                $r->{air}{unitTemp}      = 'C';
                if (defined $3) {
                    $r->{dewpoint}{temp}     = _parseUSTemp($3);
                    $r->{dewpoint}{unitTemp} = 'C';
                }
                push @{$metar{remark}}, { tempHourly => $r };
            } elsif (   _isValidCountry('remarkSunshine', \@cy)
                     && $msg =~ m@^(98(?:([0-9]{3})|///)) (.*)@o)
            {
                $msg = $3;
                $r->{s} = $1;
                if (defined $2) {
                    $r->{durationMinutes} = $2 + 0;
                } else {
                    $r->{notAvailable} = undef;
                }
                push @{$metar{remark}}, { durationOfSunshine => $r };
            } elsif (   _isValidCountry('remarkQFEhPa', \@cy)
                     && $msg =~ m@^(QFE([01]?[0-9]{3})) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { QFE => {
                    s => $1,
                    hPa => $2 + 0
                }};
            } elsif (   _isValidCountry('remarkDA_PA', \@cy)
                     && $msg =~ m@^(/?(DA|PA)[ /]?([+-]?[0-9]+)(?:FT)?/?) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { ({ DA => 'densityAlt',
                                             PA => 'pressureAlt'
                                           }->{$2})
                                          => {
                    s        => $1,
                    altitude => $3 + 0
                }};
            } elsif (   _isValidCountry('remarkKNIP', \@cy)
                     && $msg =~ m@^((RH|SST|AI|OAT)/([0-9]{2})F?) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                if ($2 eq 'SST' || $2 eq 'OAT') {
                    $r->{temp} = $3 + 0;
                    $r->{unitTemp} = 'F';
                } elsif ($2 eq 'RH') {
                    $r->{relHumid} = $3 + 0;
                } else {
                    $r->{"$2Val"} = $3 + 0;
                }
                push @{$metar{remark}}, { $2 => $r };
            } elsif (   _isValidCountry('remarkKN', \@cy)
                     && $msg =~ m@^((RH|SST)/([0-9]+)) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                if ($2 eq 'SST') {
                    $r->{temp} = $3 + 0;
                    $r->{unitTemp} = 'C';
                } else {
                    $r->{relHumid} = $3 + 0;
                }
                push @{$metar{remark}}, { $2 => $r };
            } elsif (   _isValidCountry('remarkQFEmmHg', \@cy)
                     && $msg =~ m@^(QFE ?([0-9]{3})(?:,([0-9]))?(?:/([0-9]{3,4}))?) (.*)@o)
            {
                my $qfe;
                $msg = $5;
                $qfe->{s} = $1;
                $qfe->{mmHg} = $2 + 0;
                $qfe->{mmHg} += $3 / 10 if defined $3;
                $qfe->{hPa} = $4 + 0 if defined $4;
                push @{$metar{remark}}, { QFE => $qfe };
            } elsif ($msg =~ m@^(((?:VIS|CHI)NO) RWY ?($re_rwy_des)) (.*)@o) {
                $msg = $4;
                push @{$metar{remark}}, { $2 => {
                    s => $1,
                    rwyDesig => $3
                }};
            } elsif (   _isValidCountry('remarkSNWCVR', \@cy)
                     && $msg =~ m@^($re_snw_cvr) (.*)@o)
            {
                $msg = $3;
                $r->{s} = $1;
                $r->{snowCoverType} = defined $2 ? $2 : 'NIL';
                $r->{snowCoverType} =~ s/ONE /ONE_/;
                $r->{snowCoverType} =~ s/MU?CH /MUCH_/;
                $r->{snowCoverType} =~ s/TR(?:ACE ?)? ?/TRACE_/;
                $r->{snowCoverType} =~ s/MED(?:IUM)?/MEDIUM/;
                $r->{snowCoverType} =~ s/ PACK(?:ED)?/_PACKED/;
                push @{$metar{remark}}, { snowCover => $r };
            } elsif (   _isValidCountry('remarkObsTimeOffset', \@cy)
                     && $msg =~ m@^(OBS TAKEN [+]([0-9]+)) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { obsTimeOffset => {
                    s       => $1,
                    minutes => $2
                }};
            } elsif (   _isValidCountry('remarkBalloon', \@cy)
                     && $msg =~ m@^((?:BLN (?:DSA?PRD ([0-9]+) ?FT[.]?|VI?SBL(?: TO)? ([0-9]+) ?FT))) (.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                if (defined $2) {
                    $r->{disappearedAt} = { distance => $2, unitHeight => 'FT'};
                } else {
                    $r->{visibleTo} = { distance => $3, unitHeight => 'FT' };
                }
                push @{$metar{remark}}, { balloon => $r };
            } elsif (   _isValidCountry('remarkRVR', \@cy)
                     && $msg =~ m@^RVR((?: RWY ?$re_rwy_des $re_rwy_vis ?FT[.]?)+) (.*)@o)
            {
                my $hdr = 'RVR ';
                $msg = $2;
                for ($1 =~ / (RWY ?$re_rwy_des $re_rwy_vis ?FT[.]?)/g) {
                    my $v;

                    $v->{s} = "$hdr$_";
                    $hdr    = '';

                    /RWY ?($re_rwy_des) ($re_rwy_vis) ?(FT)/;

                    $v->{rwyDesig}        = $1;
                    $v->{RVR}{distance}   = $2;
                    $v->{RVR}{unitLength} = $3;
                    $v->{RVR}{isLess}         = undef
                        if $v->{RVR}{distance} =~ s/^M//;
                    $v->{RVR}{isEqualGreater} = undef
                        if $v->{RVR}{distance} =~ s/^P//;
                    push @{$metar{remark}}, { visRwy => $v };
                }
            } elsif (   _isValidCountry('remarkTX', \@cy)
                     && $msg =~ m@^(TX/([0-9][0-9][,.][0-9])) (.*)@o)
            {
                $msg = $3;
                $r->{tempMaxFQ}{s} = $1;
                ($r->{tempMaxFQ}{temp} = $2) =~ s/,/./;
                $r->{tempMaxFQ}{unitTemp} = 'C';
                push @{$metar{remark}}, $r;

            } elsif (   _isValidCountry('remarkTAFNxtFcstBy', \@cy)
                     && $msg =~ m@^(NXT FCST BY ($re_hour)Z) (.*)@o)
            {
                $msg = $3;
                push @{$metar{remark}}, { nextFcstBy => {
                    s      => $1,
                    hour   => $2,
                    minute => '00'
                }};
            } elsif (   _isValidCountry('remarkTAFNxtFcstAt', \@cy)
                     && $msg =~ m@^(NXT FCST WILL BE ISSUED AT ($re_day)($re_hour)($re_min)Z) (.*)@o)
            {
                $msg = $5;
                push @{$metar{remark}}, { nextFcstAt => {
                    s      => $1,
                    day    => $2,
                    hour   => $3,
                    minute => $4
                }};
            } elsif (   _isValidCountry('remarkTAFAmdAt', \@cy)
                     && $msg =~ m@^(AMD AT ($re_day)($re_hour)($re_min)Z) (.*)@o)
            {
                $msg = $5;
                push @{$metar{remark}}, { amdAt => {
                    s      => $1,
                    day    => $2,
                    hour   => $3,
                    minute => $4
                }};
            } elsif (   _isValidCountry('remarkTAFFcstAutoObs', \@cy)
                     && $msg =~ m@^(FCST BASED ON AUTO OBS(?: ($re_hour)-($re_hour)Z)?[.]?) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { fcstAutoObs => {
                    s => $1,
                    defined $2 ? ('hourFrom', $2,
                                  'hourTill', $3)
                               : ()
                }};
            } elsif (   _isValidCountry('remarkTAFFcstAutometar', \@cy)
                     && $msg =~ m@^(BASED ON AUTOMETAR) (.*)@o)
            {
                $msg = $2;
                push @{$metar{remark}}, { fcstAutoMETAR => { s => $1 }};
            } elsif ($msg =~ m@^(((?:$re_data_estmd[ /])+)(?:VISUALLY )?$re_estmd) (.*)@o)
            {
                $msg = $3;
                $r->{s} = $1;
                for ($2 =~ /$re_data_estmd/og) {
                    s/WI?NDS?(?: DATA)?/WND/;
                    s/CEILING/CIG/;
                    s/(?:CIG )?BLN/CIG BLN/;
                    s/ /_/g;
                    push @{$r->{estimatedItem}}, $_;
                }
                push @{$metar{remark}}, { estimated => $r };
            } elsif ($msg =~ m@^($re_estmd((?:[ /]$re_data_estmd)+)) (.*)@o) {
                $msg = $3;
                $r->{s} = $1;
                for ($2 =~ /$re_data_estmd/og) {
                    s/WI?NDS?(?: DATA)?/WND/;
                    s/CEILING/CIG/;
                    s/(?:CIG )?BLN/CIG BLN/;
                    s/ /_/g;
                    push @{$r->{estimatedItem}}, $_;
                }
                push @{$metar{remark}}, { estimated => $r };
            } elsif ($msg =~ m@^((?:(THN) )?($re_cloud_cov) ABV ($re_cloud_base)) (.*)@o)
            {
                $msg = $5;
                $r->{cloudAbove}{s} = $1;
                $r->{cloudAbove}{isThin} = undef if defined $2;
                $r->{cloudAbove}{cloudCover} = $3;
                $r->{cloudAbove}{cloudBase} = $4 + 0;
                push @{$metar{remark}}, $r;
            } elsif (   _isValidCountry('remarkPhenCloud', \@cy)
                     && $msg =~ m@^((FU|(?:FZ)?FG|BR|PWR PLNT(?: PLUME)?) ($re_cloud_cov$re_cloud_base)) (.*)@o)
            {
                my ($cl, $phen);

                $msg = $4;
                $r->{s} = $1;
                $r->{cloud} = _parseCloud $3;
                $phen = $2;
                if ($phen =~ m@^$re_weather$@) {
                    $r->{weather} = _parseWeather $phen;
                } else {
                    ($r->{cloudPhenom} = $phen) =~ s/ /_/g;
                }
                push @{$metar{remark}}, { obscuration => $r };
            } elsif (   _isValidCountry('remarkRainfall', \@cy)
                     && $msg =~ m@^(RF([0-9]{2}\.[0-9])/([0-9]{3}\.[0-9])) (.*)@o)
            {
                $msg = $4;
                push @{$metar{remark}}, { rainfall => {
                    s             => $1,
                    rainfall10min => $2 + 0,
                    rainfall0900  => $3 + 0
                }};
            } elsif (   _isValidCountry('remarkRSNK', \@cy)
                     && $msg =~ m@^(RSNK (M)?([0-9]{2}(?:\.[0-9])?)/(${re_wind_dir}0$re_wind_speed(?:G$re_wind_speed)?)($re_wind_speed_unit)?) (.*)@o)
            {
                my ($temp, $unit);

                $msg = $6;
                $temp = $3 + 0;
                $temp *= -1 if defined $2;
                $unit = defined $5 ? $5 : 'KT';
                push @{$metar{remark}}, { RSNK => {
                    s    => $1,
                    air  => { temp => $temp, unitTemp => 'C' },
                    wind => _parseWind "$4$unit"
                }};
            } elsif (   _isValidCountry('remarkLAGPK', \@cy)
                     && $msg =~ m@^(LAG ?PK (M)?([0-9]{2})/(M)?([0-9]{2})/(${re_wind_dir}0$re_wind_speed(?:G$re_wind_speed)?)) (.*)@o)
            {
                my ($temp, $dewpoint);

                $msg = $7;
                $temp = $3 + 0;
                $temp *= -1 if defined $2;
                $dewpoint = $5 + 0;
                $dewpoint *= -1 if defined $4;
                push @{$metar{remark}}, { LAG_PK => {
                    s        => $1,
                    air      => { temp => $temp, unitTemp => 'C' },
                    dewpoint => { temp => $dewpoint, unitTemp => 'C' },
                    wind     => _parseWind "$6KT"
                }};
            } elsif (   _isValidCountry('remarkPrecipHourly3', \@cy)
                     && $msg =~ m@^(P([0-9]{3})) (.*)@o)
            {
                $msg = $3;
                $r->{s} = $1;
                $r->{precipAmountInch} = $2 / 100;
                $r->{precipHours} = 1;
                push @{$metar{remark}}, { precipHourly => $r };

# nearly the same pattern twice except for the position of $re_phen_desc
# (do not match too much but don't miss anything, either)

            } elsif ($msg =~ m@^(((?:$re_phen_desc[/ ])+)$re_phenomenon4( BBLO)?(?:$re_loc_quadr3$re_wx_mov_d2?|$re_loc_quadr3?$re_wx_mov_d2)(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $21;
                $r->{s} = $1;
                _parsePhenomDescr $r, 'phenomDescrPre', $2;
                if (defined $3) {
                    ($r->{otherPhenom} = $3) =~ tr/ /_/;
                    $r->{otherPhenom} =~ s/REDUCED/RDCD/;
                    $r->{otherPhenom} =~ s/AURORA/AURBO/;
                    $r->{otherPhenom} =~ s/SHWR/SH/;
                } elsif (defined $4) {
                    @{$r->{cloudType}} = split m@[/-]@, $4;
                } elsif (defined $5) {
                    $r->{weather} = _parseWeather $5 eq 'SMOKE' ? 'FU'
                                                  : ($5 eq 'HAZE' ? 'HZ' : $5);
                } else {
                    $r->{cloudCover} = $6;
                }
                _parsePhenomDescr $r, 'phenomDescrPost', $7 if defined $7;
                $r->{locationAndList} = _parseLocations $8 if defined $8;
                $r->{locationAndList} = _parseQuadrants $10, $9 if defined $10;
                $r->{$11} = _parseLocations $12 if defined $12;
                $r->{locationAndList} = _parseLocations $13 if defined $13;
                $r->{locationAndList} = _parseQuadrants $15, $14 if defined $15;
                $r->{$16} = _parseLocations $17 if defined $17;
                $r->{cloudTypeAsoctd} = $18 if defined $19;
                $r->{cloudTypeEmbd}   = $18 if defined $20;
                push @{$metar{remark}}, { phenomenonAtLoc => $r };
            } elsif ($msg =~ m@^($re_phenomenon4(?: IS)?((?:[/ ](?:$re_phen_desc|BBLO))*)(?:$re_loc_quadr3$re_wx_mov_d2?|$re_loc_quadr3?$re_wx_mov_d2)(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $20;
                $r->{s} = $1;
                if (defined $2) {
                    if ($2 eq 'VIS HIER') {
                        $r->{otherPhenom} = 'VIS_HYR';
                    } else {
                        ($r->{otherPhenom} = $2) =~ tr/ /_/;
                        $r->{otherPhenom} =~ s/REDUCED/RDCD/;
                        $r->{otherPhenom} =~ s/AURORA/AURBO/;
                        $r->{otherPhenom} =~ s/SHWR/SH/;
                    }
                } elsif (defined $3) {
                    @{$r->{cloudType}} = split m@[/-]@, $3;
                } elsif (defined $4) {
                    $r->{weather} = _parseWeather $4 eq 'SMOKE' ? 'FU'
                                                  : ($4 eq 'HAZE' ? 'HZ' : $4);
                } else {
                    $r->{cloudCover} = $5;
                }
                _parsePhenomDescr $r, 'phenomDescrPost', $6 if defined $6;
                $r->{locationAndList} = _parseLocations $7 if defined $7;
                $r->{locationAndList} = _parseQuadrants $9, $8 if defined $9;
                $r->{$10} = _parseLocations $11 if defined $11;
                $r->{locationAndList} = _parseLocations $12 if defined $12;
                $r->{locationAndList} = _parseQuadrants $14, $13 if defined $14;
                $r->{$15} = _parseLocations $16 if defined $16;
                $r->{cloudTypeAsoctd} = $17 if defined $18;
                $r->{cloudTypeEmbd}   = $17 if defined $19;
                push @{$metar{remark}}, { phenomenonAtLoc => $r };
            } elsif ($msg =~ m@^((?:PR(?:ESENT )?WX )?((?:$re_phen_desc[/ ])+)$re_phenomenon4( BBLO)?(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $11;
                $r->{s} = $1;
                _parsePhenomDescr $r, 'phenomDescrPre', $2 if defined $2;
                if (defined $3) {
                    ($r->{otherPhenom} = $3) =~ tr/ /_/;
                    $r->{otherPhenom} =~ s/REDUCED/RDCD/;
                    $r->{otherPhenom} =~ s/AURORA/AURBO/;
                    $r->{otherPhenom} =~ s/SHWR/SH/;
                } elsif (defined $4) {
                    @{$r->{cloudType}} = split m@[/-]@, $4;
                } elsif (defined $5) {
                    $r->{weather} = _parseWeather $5 eq 'SMOKE' ? 'FU'
                                                  : ($5 eq 'HAZE' ? 'HZ' : $5);
                } else {
                    $r->{cloudCover} = $6;
                }
                _parsePhenomDescr $r, 'phenomDescrPost', $7 if defined $7;
                $r->{cloudTypeAsoctd} = $8 if defined $9;
                $r->{cloudTypeEmbd}   = $8 if defined $10;
                push @{$metar{remark}}, { phenomenonOnly => $r };

            # first check a restricted set of phenomenon and descriptions!

            } elsif ($msg =~ m@^(GR ($re_gs_size)) (.*)@o) {
                $msg = $3;
                _parseFraction $r, $2;
                push @{$metar{remark}}, { hailStones => {
                    s => $1,
                    hailStoneSize => $r->{distance},
                    exists $r->{isLess} ? (isLess => undef) : ()
                }};
            } elsif ($msg =~ m@^((PRES[FR]R)( PAST HR)?) (.*)@o) {
                $msg = $4;
                $r->{s} = $1;
                $r->{otherPhenom} = $2;
                _parsePhenomDescr $r, 'phenomDescrPost', $3 if defined $3;
                push @{$metar{remark}}, { phenomenonOnly => $r };
            } elsif ($msg =~ m@^(($re_phenomenon_other)(?: IS)?((?:[/ ](?:$re_phen_desc_other|BBLO))*)(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $7;
                $r->{s} = $1;
                ($r->{otherPhenom} = $2) =~ tr/ /_/;
                $r->{otherPhenom} =~ s/REDUCED/RDCD/;
                $r->{otherPhenom} =~ s/AURORA/AURBO/;
                $r->{otherPhenom} =~ s/SHWR/SH/;
                _parsePhenomDescr $r, 'phenomDescrPost', $3 if defined $3;
                $r->{cloudTypeAsoctd} = $4 if defined $5;
                $r->{cloudTypeEmbd}   = $4 if defined $6;
                push @{$metar{remark}}, { phenomenonOnly => $r };
            } elsif ($msg =~ m@^((?:PR(?:ESENT )?WX )?$re_phenomenon4((?: IS)?(?:[/ ](?:$re_phen_desc|BBLO))*)(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $10;
                $r->{s} = $1;
                if (defined $2) {
                    ($r->{otherPhenom} = $2) =~ tr/ /_/;
                    $r->{otherPhenom} =~ s/REDUCED/RDCD/;
                    $r->{otherPhenom} =~ s/AURORA/AURBO/;
                    $r->{otherPhenom} =~ s/SHWR/SH/;
                } elsif (defined $3) {
                    @{$r->{cloudType}} = split m@[/-]@, $3;
                } elsif (defined $4) {
                    $r->{weather} = _parseWeather $4 eq 'SMOKE' ? 'FU'
                                                  : ($4 eq 'HAZE' ? 'HZ' : $4);
                } else {
                    $r->{cloudCover} = $5;
                }
                _parsePhenomDescr $r, 'phenomDescrPost', $6 if defined $6;
                $r->{cloudTypeAsoctd} = $7 if defined $8;
                $r->{cloudTypeEmbd}   = $7 if defined $9;
                push @{$metar{remark}}, { phenomenonOnly => $r };
            } elsif ($msg =~ m@^(((?:$re_phen_desc[/ ])*)LTG($re_ltg_types+)(?: TO)?($re_phen_locdirs)?$re_wx_mov_d2?(?: ($re_cloud_type) (?:(ASOCTD?)|(EMBD)))?) (.*)@o)
            {
                $msg = $10;
                $r->{s} = $1;
                _parsePhenomDescr $r, 'phenomDescrPre', $2 if defined $2;
                $r->{locationAndList} = _parseLocations $4 if defined $4;
                $r->{$5} = _parseLocations $6 if defined $6;
                $r->{cloudTypeAsoctd} = $7 if defined $8;
                $r->{cloudTypeEmbd}   = $7 if defined $9;
                $r->{lightningType} = ();
                for ($3 =~ /$re_ltg_types/og) {
                    push @{$r->{lightningType}}, $_;
                }
                push @{$metar{remark}}, { phenomenonAtLoc => $r };
            } elsif ($msg =~ m@^(AO(?:1|2A?)) (.*)@o) {
                $msg = $2;
                push @{$metar{remark}}, { obsStationType => {
                    s           => $1,
                    stationType => $1
                }};
            } elsif ($msg =~ m@^(CIG ($re_cloud_base)(?:(?:(?: (APCH))?(?: RWY ?| R?)?($re_rwy_des))(?: TO)?($re_phen_locdirs)?|(?: TO)?($re_phen_locdirs))) (.*)@o)
            {
                $msg = $7;
                $r->{s} = $1;
                $r->{cloudBase} = $2 + 0;
                $r->{isApproach} = undef if defined $3;
                $r->{rwyDesig} = $4 if defined $4;
                $r->{locationAndList} =_parseLocations $5 if defined $5;
                $r->{locationAndList} =_parseLocations $6 if defined $6;
                push @{$metar{remark}}, { ceilingAtLoc => $r };
            } elsif ($msg =~ m@^(VIS ($re_vis_sm)(?: (APCH))?(?: RWY ?| R?)?($re_rwy_des)) (.*)@o)
            {
                $msg = $5;
                $r->{s} = $1;
                $r->{visibility} = {};
                _parseFraction $r->{visibility}, $2, 'SM';
                $r->{isApproach} = undef if defined $3;
                $r->{rwyDesig} = $4;
                push @{$metar{remark}}, { visibilityAtLoc => $r };
            } elsif ($msg =~ m@^(NOSIG|CAVU|$re_estmd PASS (?:OPEN|CLOSED|CLSD|MARGINAL|MRGL)|PASS $re_estmd CLSD|EPC|EPO|EPM|RTS) (.*)@o) {
                $msg = $2;
                $r->{s} = $1;
                $r->{keyword} = $1;
                if ($r->{keyword} =~ m@PASS OP@o) {
                    $r->{keyword} = 'EPO';
                } elsif ($r->{keyword} =~ m@PASS .*CL@o) {
                    $r->{keyword} = 'EPC';
                } elsif ($r->{keyword} =~ m@PASS M@o) {
                    $r->{keyword} = 'EPM';
                }
                push @{$metar{remark}}, { keyword => $r };
            } elsif ($msg =~ m@^((?:CLIMAT ?)?($re_temp)/($re_temp)(?:/(TR|$re_precip)(?:/(NIL|$re_precip))?)?) (.*)@o)
            {
                my ($precip1, $precip2);

                $msg = $6;
                $r->{s} = $1;
                $r->{temp1}{temp} = $2 + 0;
                $r->{temp1}{unitTemp} = 'C';
                $r->{temp2}{temp} = $3 + 0;
                $r->{temp2}{unitTemp} = 'C';
                if (defined $4) {
                    if ($4 eq 'TR') {
                        $r->{precip1Traces} = undef;
                    } else {
                        $precip1 = $4;
                    }
                    if (defined $5) {
                        if ($5 eq 'NIL') {
                            $r->{precipAmount2MM} = 0;
                        } else {
                            $precip2 = $5;
                            if ($precip2 =~ s/ ?([MC]M)//) {
                                $r->{precipAmount2MM} = $precip2 + 0;
                                $r->{precipAmount2MM} *= 10 if $1 eq 'CM';
                            } else {
                                $r->{precipAmount2Inch} = $precip2 + 0;
                            }
                        }
                    }
                    if (defined $precip1) {
                        if ($precip1 =~ s/ ?([MC]M)//) {
                            $r->{precipAmount1MM} = $precip1 + 0;
                            $r->{precipAmount1MM} *= 10 if $1 eq 'CM';
                        } else {
                            $r->{precipAmount1Inch} = $precip1 + 0;
                        }
                    }
                }
                push @{$metar{remark}}, { climate => $r };
            } elsif ($msg =~ m@^((TORNADO|FUNNEL CLOUDS?|WATERSPOUT) ($re_be_prec_be+)(?: TO)?($re_phen_locdirs)$re_wx_mov_d2?) (.*)@o) {
                $msg = $7;
                $r->{s} = $1;
                $r->{tornadicActivityType} = lc $2;
                $r->{locationAndList} = _parseLocations $4;
                $r->{$5} = _parseLocations $6 if defined $6;
                $r->{start_end} = ();
                for ($3 =~ /$re_be_prec_be+?/og) {
                    my %s_e;
                    /(.)(..)?(..)/;
                    $s_e{hour} = $2 if defined $2;
                    $s_e{minute} = $3;
                    if ($1 eq 'B') {
                        push @{$r->{start_end}}, { startTime => \%s_e };
                    } else {
                        push @{$r->{start_end}}, { endTime => \%s_e };
                    }
                }
                $r->{tornadicActivityType} =~ s/(CLOUD)S?/$1/;
                $r->{tornadicActivityType} =~ s/ /_/;
                push @{$metar{remark}}, { tornadicActivity => $r };
            } elsif ($msg =~ m@^((?:FIRST|FST)(?:( STAFFED| STFD)|( MANNED))?(?: OBS?)?)[ /](.*)@o)
            {
                $msg = $4;
                $r->{s} = $1;
                $r->{isStaffed} = undef if defined $2;
                $r->{isManned} = undef if defined $3;
                push @{$metar{remark}}, { firstObs => $r };
            } elsif ($msg =~ m@^(NEXT ($re_day)($re_hour)($re_min)(?: ?UTC| ?Z)?) (.*)@o)
            {
                $msg = $5;
                $r->{s} = $1;
                @{$r->{obsAt}}{'day', 'hour', 'minute'} = ($2, $3, $4)
                    if defined $2;
                push @{$metar{remark}}, { nextObs => $r };
            } elsif ($msg =~ m@^(LAST(?:( STAFFED)|( MANNED))?(?: OBS?)?(?: ($re_day)($re_hour)($re_min)(?: ?UTC| ?Z)?)?)[ /](.*)@o)
            {
                $msg = $7;
                $r->{s} = $1;
                $r->{isStaffed} = undef if defined $2;
                $r->{isManned} = undef if defined $3;
                @{$r->{obsAt}}{'day', 'hour', 'minute'} = ($4, $5, $6)
                    if defined $4;
                push @{$metar{remark}}, { lastObs => $r };
            } elsif ($msg =~ m@^(RADAT (?:([0-9]{2})([0-9]{3})|MISG)) (.*)@o) {
                $msg = $4;
                $r->{s} = $1;
                if (defined $2) {
                    $r->{relHumid} = $2 + 0;
                    $r->{distance} = $3 + 0;
                    $r->{unitHeight} = 'FL';
                } else {
                    $r->{missing} = undef;
                }
                push @{$metar{remark}}, { RADAT => $r };
            } elsif ($cy[1] eq 'LF' && $msg =~ m@^([MB])([0-9]) (.*)@o) {
                $msg = $3;
                push @{$metar{remark}}, { reportConcerns => {
                    s       => "$1$2",
                    change  => $1,
                    subject => $2
                }};
            } elsif ($cy[1] eq 'LI') {
                my $re_cond_moun =
                    '(?:LIB|CLD SCT|VERS INC|CNS POST|CLD CIME'
                    . '|CIME INC|GEN INC|INC|INVIS)';
                my $re_chg_moun =
                    '(?:NC|CUF'
                     . '|ELEV (?:SLW|RAPID|STF)'
                     . '|ABB (?:SLW|RAPID)'
                     . '|STF(?: ABB)?'
                     . '|VAR RAPID)';
                my $re_cond_vall =
                    '(?:NIL|FOSCHIA(?: SKC SUP)?|NEBBIA(?: SCT)?'
                    . '|CLD SCT(?: NEBBIA INF)?|MAR CLD|INVIS)';
                my $re_chg_vall =
                    '(?:NC|DIM(?: ELEV| ABB)?|AUM(?: ELEV| ABB)?|ABB'
                    . '|NEBBIA INTER)';
                my $re_moun =
                            "$re_phen_locdirs? $re_cond_moun(?: $re_chg_moun)?";
                my $re_vall =
                            "$re_phen_locdirs? $re_cond_vall(?: $re_chg_vall)?";
                if ($msg =~ m@^(MON((?:$re_moun)+)) (.*)@o) {
                    $msg = $3;
                    $r->{s} = $1;
                    $r->{condMoun} = ();
                    for ($2 =~ m@($re_moun)@og) {
                        my $m;
                        m@($re_phen_locdirs)? ($re_cond_moun)(?: ($re_chg_moun))?@o;

                        $m->{locationAndList} =_parseLocations $1 if defined $1;
                        ($m->{condMounType}   = $2) =~ tr/ /_/;
                        ($m->{condMounChange} = $3) =~ tr/ /_/ if defined $3;
                        push @{$r->{condMoun}}, $m;
                    }
                    push @{$metar{remark}}, { conditionMountain => $r };
                } elsif ($msg =~ m@^(VAL((?:$re_vall)+)) (.*)@o) {
                    $msg = $3;
                    $r->{s} = $1;
                    $r->{condVall} = ();
                    for ($2 =~ m@($re_vall)@og) {
                        my $m;
                        m@($re_phen_locdirs)? ($re_cond_vall)(?: ($re_chg_vall))?@o;

                        $m->{locationAndList} =_parseLocations $1 if defined $1;
                        ($m->{condVallType}   = $2) =~ tr/ /_/;
                        ($m->{condVallChange} = $3) =~ tr/ /_/ if defined $3;
                        push @{$r->{condVall}}, $m;
                    }
                    push @{$metar{remark}}, { conditionValley => $r };
                } elsif ($msg =~ m@^(QU(L|K) ?([0-9/])(?: ?($re_compass_dir16))?) (.*)@o)
                {
                    my $key = $2 eq 'K' ? 'sea' : 'swell';

                    $msg = $5;
                    $r->{s} = $1;
                    if ($3 eq '/') {
                        $r->{notAvailable} = undef;
                    } else {
                        $r->{"${key}CondVal"} = $3;
                    }
                    $r->{locationAndList} = _parseLocations $4 if defined $4;
                    push @{$metar{remark}}, { "${key}Condition" => $r };
                } elsif ($msg =~ m@^(VIS (MAR) ([0-9]+) KM) (.*)@o) {
                    $msg = $4;
                    $r->{s} = $1;
                    $r->{locationAt} = $2;
                    $r->{visibility}{distance} = $3;
                    $r->{visibility}{unitLength} = 'KM';
                    push @{$metar{remark}}, { visibilityAtLoc => $r };
                } elsif ($msg =~ m@^((?:WIND THR ?|WT)($re_rwy_des) ($re_wind)) (.*)@o)
                {
                    $msg = $8;
                    push @{$metar{remark}}, { thrWind =>
                                              { s        => $1,
                                                rwyDesig => $2,
                                                wind     => _parseWind $3
                                              }
                                            };
                } elsif ($msg =~ m@^(VIS MIN ($re_vis_m)($re_compass_dir)?) (.*)@o) {
                    $msg = $4;
                    $r->{s}          = $1;
                    $r->{distance}   = $2 + 0;
                    $r->{unitLength} = 'M';
                    $r->{compassDir} = $3 if defined $3;
                    push @{$metar{remark}}, { visMin => $r };
                } else {
                    $parsed = 0;
                }
            } elsif($cy[1] eq 'LK' && $msg =~ m@^(REG QNH ([01][0-9]{3})) (.*)@)
            {
                $msg = $3;
                push @{$metar{remark}}, { regQNH => {
                    s   => $1,
                    hPa => $2
                }};
            } elsif ($cy[0] eq 'ZMUB' && $msg =~ m@^([0-9]{2}) (.*)@) {
                $msg = $2;
                push @{$metar{remark}}, { RH => {
                    s        => $1,
                    relHumid => $1
                }};
            } else {
                $parsed = 0;
            }
            if (!$parsed && _isValidCountry('remarkC_K_P_NS_RJ_RO_TI_TJ_ET', \@cy))
            {
                $parsed = 1;
                if ($msg =~ m@^(4([01][0-9]{3})([01][0-9]{3})?) (.*)@o) {
                    $msg = $4;
                    $r->{s} = $1;
                    $r->{temp24hMax}{temp} = _parseUSTemp $2;
                    $r->{temp24hMax}{unitTemp} = 'C';
                    if (defined $3) {
                        $r->{temp24hMin}{temp} = _parseUSTemp $3;
                        $r->{temp24hMin}{unitTemp} = 'C';
                    }
                    push @{$metar{remark}}, { temp24h => $r };
                } elsif ($msg =~ m@^(4/([0-9]{3})) (.*)@o) {
                    $msg = $3;
                    push @{$metar{remark}}, { snowOnGround => {
                        s                => $1,
                        precipAmountInch => $2 + 0
                    }};
                } elsif (   !_isValidCountry('remarkPrecipHourly3', \@cy)
                         && $msg =~ m@^((P|6|7)(?:([0-9]{4})|////)) (.*)@o)
                {
                    # Air Force Manual 15-111, Dec. '03, 2.8.1
                    $msg = $4;
                    $r->{s} = $1;
                    if (defined $3) {
                        $r->{precipAmountInch} = $3 / 100;
                    } else {
                        $r->{notAvailable} = undef;
                    }
                    if ($2 eq 'P') {
                        $r->{precipHours} = 1;
                    } elsif ($2 eq '7') {
                        $r->{precipHours} = 24;
                    } else {
                        # EXTENSION: allow 20 minutes
                        if (exists $metar{obsTime}{hour}) {
                            if ((  $metar{obsTime}{hour}
                                 . $metar{obsTime}{minute}) =~
                             '(?:(?:23|05|11|17)[45]|(?:00|06|12|18)[01])[0-9]')
                            {
                                $r->{precipHours} = 6;
                            } elsif (  ($metar{obsTime}{hour}
                                     . $metar{obsTime}{minute}) =~
                             '(?:(?:02|08|14|20)[45]|(?:03|09|15|21)[01])[0-9]')
                            {
                                $r->{precipHours} = 3;
                            } else {
                                $r->{precipHoursNotAvailable} = undef;
                            }
                        } else {
                            $r->{precipHoursNotAvailable} = undef;
                        }
                    }
                    push @{$metar{remark}}, { precipHourly => $r };
                } elsif ($msg =~ m@^(933([0-9]{3})) (.*)@o) {
                    $msg = $3;
                    push @{$metar{remark}}, { waterEquivOfSnow => {
                        s                => $1,
                        precipAmountInch => $2 / 10
                    }};
                } elsif ($msg =~ m@^(CIG ([0-9]{3})V([0-9]{3})) (.*)@o) {
                    $msg = $4;
                    push @{$metar{remark}}, { variableCeiling => {
                        s => $1,
                        cloudBaseFrom => $2 + 0,
                        cloudBaseTo   => $3 + 0
                    }};
                } elsif ($msg =~ m@^(CIG ([0-9]{4})V([0-9]{4})) (.*)@o) {
                    $msg = $4;
                    push @{$metar{remark}}, { variableCeiling => {
                        s => $1,
                        cloudBaseFrom => $2 / 100,
                        cloudBaseTo   => $3 / 100
                    }};
                } elsif ($msg =~ m@^(($re_be_prec+)($re_phen_locdirs)?$re_wx_mov_d2?) (.*)@o)
                {
                    $msg = $6;

                    $r->{beginEndPrecip}{s} = $1;
                    $r->{beginEndPrecip}{locationAndList} = _parseLocations $3
                        if defined $3;
                    $r->{beginEndPrecip}{$4} = _parseLocations $5 if defined $5;
                    $r->{beginEndPrecip}{precip} = ();
                    for ($2 =~ /($re_be_prec)+?/g) {
                        my @start_end;
                        my ($weather, $times) = /(.*?)($re_be_prec_be+)/g;
                        for ($times =~ /$re_be_prec_be+?/g) {
                            my %s_e;
                            /(.)(..)?(..)/;
                            $s_e{hour} = $2 if defined $2;
                            $s_e{minute} = $3;
                            if ($1 eq 'B') {
                                push @start_end, { startTime => \%s_e };
                            } else {
                                push @start_end, { endTime => \%s_e };
                            }
                        }
                        push @{$r->{beginEndPrecip}{precip}},
                             { weather => _parseWeather($weather),
                               start_end => \@start_end
                             }
                    }
                    push @{$metar{remark}}, $r;
                } elsif ($msg =~ m@^WSHFT (($re_hour)?($re_min)( FROPA)?) (.*)@o)
                {
                    $msg = $5;
                    $r->{windShift}{s} = "WSHFT $1";
                    $r->{windShift}{hour} = $2 if defined $2;
                    $r->{windShift}{minute} = $3;
                    $r->{windShift}{FROPA} = undef if defined $4;
                    push @{$metar{remark}}, $r;
                } elsif ($msg =~ m@^(VIS ($re_vis_sm) ($re_compass_dir16)) (.*)@o)
                {
                    my (@arr, $v);

                    $msg = $4;
                    $r->{visListLoc}{s} = $1;
                    $v->{visibility} = {};
                    _parseFraction $v->{visibility}, $2, 'SM';
                    $r->{visListLoc}{arr} = \@arr;
                    $v->{locationAndList} = _parseLocations $3;
                    push @arr, $v;
                    push @{$metar{remark}}, $r;
                } elsif ($msg =~ m@^VIS($re_phen_locdirs $re_vis_sm(?: SM)?(?:(?: AND)?$re_phen_locdirs $re_vis_sm(?: SM)?)*) (.*)@o)
                {
                    my @arr;

                    $msg = $2;
                    $r->{visListLoc}{s} = "VIS$1";
                    $r->{visListLoc}{arr} = \@arr;
                    for ($1 =~ m@(?:AND)?($re_phen_locdirs $re_vis_sm)@og) {
                        my ($v, $is_less, $vis);

                        m@($re_phen_locdirs) ($re_vis_sm)@o;
                        $v->{locationAndList} = _parseLocations $1;
                        $v->{visibility} = {};
                        _parseFraction $v->{visibility}, $2, 'SM';
                        push @arr, $v;
                    }
                    push @{$metar{remark}}, $r;
                } else {
                    $parsed = 0;
                }
            }
            if (!$parsed && $msg =~ m@^(BA (?:GOOD|POOR)|THN SPTS IOVC|FUOCTY|ALTM MISG|ALL WNDS GRID|CONTRAILS?|(?:SKY|MT(?:N?S)?) OBSC(?:URED)?|FOGGY|TWLGT) (.*)@o)
            {
                $parsed = 1;
                $msg = $2;
                $r->{s} = $1;
                ($r->{keyword} = $1) =~ tr/\/ /_/;
                $r->{keyword} =~ s/CONTRAILS?/CONTRAILS/;
                $r->{keyword} =~ s/OBSCURED/OBSC/;
                $r->{keyword} =~ s/MT(?:N?S)?/MTNS/;
                push @{$metar{remark}}, { keyword => $r };
            }
            if (!$parsed) {
                $msg =~ '(\S+) (.*)';
                $notRecognised .= ' ' unless $notRecognised eq '';
                $notRecognised .= $1;
                $msg = $2;
            }
            if ($parsed && $notRecognised ne '') {
                my $top = pop @{$metar{remark}};
                push @{$metar{remark}},
                                    { notRecognised => { s => $notRecognised }};
                push @{$metar{remark}}, $top;
                $notRecognised = '';
            }
        }
        push @{$metar{remark}}, { notRecognised => { s => $notRecognised }}
            if $notRecognised ne '';
    }

    if (   $is_taf
        && $msg =~ m@^(AMD (?:LTD TO CLD VIS AND WIND(?: (?:TIL ($re_hour|24)Z|(${re_hour})Z-($re_hour|24)Z))?|(NOT SKED))) (.*)@o)
    {
        $msg = $6;
        $metar{amendment}{s} = $1;
        if (defined $5) {
            $metar{amendment}{isNotScheduled} = undef;
        } else {
            $metar{amendment}{isLtdToCldVisWind} = undef;
            $metar{amendment}{hourTill} = $2 if defined $2;
            $metar{amendment}{hourFrom} = $3 if defined $3;
            $metar{amendment}{hourTill} = $4 if defined $4;
        }
    }

    if ($msg ne '') {
        $metar{ERROR}{descr} = 'other';
        $metar{ERROR}{pos}   = substr($corr_msg, 0,
                                      (length $corr_msg) - length $msg)
                               . '<@> ' . $msg;
    }
    return %metar;
}

1;
__END__
