//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  TMEXLnk.C - Link module to call on TMEX low-level functions to
//              excersize the MicroLAN (MLan) functions.
//
//  Version: 1.03
//
//  History: 1.00 -> 1.01  Return values in MLanLevel corrected.
//                         Added function msDelay.
//           1.02 -> 1.03  Add msGettick, always return MLanLevel success
//                         to hide adapters (DS9097E) that do not have
//                         power delivery capabilities.

#include "mlan.h"
#include <windows.h>

// external TMEX variables
extern long SessionHandle;
extern short far pascal TMTouchByte(long, short);
extern short far pascal TMTouchReset(long);
extern short far pascal TMTouchBit(long, short);
extern short far pascal TMProgramPulse(long);
extern short far pascal TMOneWireCom(long, short, short);
extern short far pascal TMOneWireLevel(long, short, short, short);

// exportable low-level functions
int MLanTouchReset(void);
int MLanTouchBit(int);
int MLanTouchByte(int);
int MLanWriteByte(int sendbyte);
int MLanReadByte(void);
int MLanSpeed(int);
int MLanLevel(int);
int MLanProgramPulse(void);
void msDelay(int);
long msGettick(void);

//--------------------------------------------------------------------------
// Reset all of the devices on the MicroLAN and return the result.
//
// Returns: TRUE(1):  presense pulse(s) detected, device(s) reset
//          FALSE(0): no presense pulses detected
//
int MLanTouchReset(void)
{
   int result;

   // Assume valid Session
   result = TMTouchReset(SessionHandle);

   // success if the normal or alarm presence
   if ((result == 1) || (result == 2))
      return TRUE;
   else
      return FALSE;
}


//--------------------------------------------------------------------------
// Send 1 bit of communication to the MicroLAN and return the
// result 1 bit read from the MicroLAN.  The parameter 'sendbit'
// least significant bit is used and the least significant bit
// of the result is the return bit.
//
// 'sendbit' - the least significant bit is the bit to send
//
// Returns: 0:   0 bit read from sendbit
//          1:   1 bit read from sendbit
//
int MLanTouchBit(int sendbit)
{
   // Assume valid Session
   return TMTouchBit(SessionHandle,(short)sendbit);
}


//--------------------------------------------------------------------------
// Send 8 bits of communication to the MicroLAN and return the
// result 8 bits read from the MicroLAN.  The parameter 'sendbyte'
// least significant 8 bits are used and the least significant 8 bits
// of the result is the return byte.
//
// 'sendbyte' - 8 bits to send (least significant byte)
//
// Returns:  8 bytes read from sendbyte
//
int MLanTouchByte(int sendbyte)
{
   // Assume valid Session
   return TMTouchByte(SessionHandle,(short)sendbyte);
}


//--------------------------------------------------------------------------
// Send 8 bits of communication to the MicroLAN and verify that the
// 8 bits read from the MicroLAN is the same (write operation).  
// The parameter 'sendbyte' least significant 8 bits are used.
//
// 'sendbyte' - 8 bits to send (least significant byte)
//
// Returns:  TRUE: bytes written and echo was the same
//           FALSE: echo was not the same 
//
int MLanWriteByte(int sendbyte)
{
   return (MLanTouchByte(sendbyte) == sendbyte) ? TRUE : FALSE;
}


//--------------------------------------------------------------------------
// Send 8 bits of read communication to the MicroLAN and and return the
// result 8 bits read from the MicroLAN.   
//
// Returns:  8 bytes read from MicroLAN
//
int MLanReadByte(void)
{
   return MLanTouchByte(0xFF);
}

//--------------------------------------------------------------------------
// Set the MicroLAN communucation speed.  
//
// 'NewSpeed' - new speed defined as
//                MODE_NORMAL     0x00
//                MODE_OVERDRIVE  0x01
//
// Returns:  current MicroLAN speed 
//
int MLanSpeed(int NewSpeed)
{
   return TMOneWireCom(SessionHandle,0,(short)NewSpeed);
}


//--------------------------------------------------------------------------
// Set the MicroLAN line level.  The values for NewLevel are
// as follows:
//
// 'NewLevel' - new level defined as
//                MODE_NORMAL     0x00
//                MODE_STRONG5    0x02
//                MODE_PROGRAM    0x04
//                MODE_BREAK      0x08 (not supported)
//
// Returns:  current MicroLAN level  
//
int MLanLevel(int NewLevel)
{
   int rslt;

   switch (NewLevel)
   {
      case MODE_NORMAL:
         rslt = TMOneWireLevel(SessionHandle,0,0,0);
         break;
      case MODE_STRONG5:
         rslt = TMOneWireLevel(SessionHandle,0,1,0);
         break;
      case MODE_PROGRAM:
         rslt = TMOneWireLevel(SessionHandle,0,3,0);
         break;
      case MODE_BREAK:
         rslt = TMOneWireLevel(SessionHandle,0,2,0);
         break;
      default:
         rslt = 0;
   }   

   // Assume TMEX can do it so always return NewLevel
   return NewLevel;
}


//--------------------------------------------------------------------------
// This procedure creates a fixed 480 microseconds 12 volt pulse 
// on the MicroLAN for programming EPROM iButtons.
//
// Returns:  TRUE  successful
//           FALSE program voltage not available  
//
int MLanProgramPulse(void)
{
   return TMProgramPulse(SessionHandle);
}


//--------------------------------------------------------------------------
//  Description:
//     Delay for at least 'len' ms
// 
void msDelay(int len)
{
   Sleep(len);
}


//--------------------------------------------------------------------------
// Get the current millisecond tick count.  Does not have to represent
// an actual time, it just needs to be an incrementing timer.
//
long msGettick(void)
{
   return GetTickCount();
}

