//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  thermoms.c - This utility mission a DS1921 
//               Thermochron iButton.
//
//  Version:   1.03
//

#define TMEXUTIL

#ifdef WIN32
   #include <conio.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include "mlan.h"   
#include "thermo.h"

// external function prototypes 
extern int  RunThermoScript(ThermoState script[],FILE *fp);
extern void MissionStatusToString(MissionStatus *, int, char *);
extern void InterpretStatus(MissionStatus *);
extern void InterpretHistogram(Histogram *);
extern void HistogramToString(Histogram *, int, char *);
extern void InterpretAlarms(TempAlarmEvents *, MissionStatus *);
extern void AlarmsToString(TempAlarmEvents *, char *);
extern void InterpretLog(Log *, MissionStatus *);
extern void LogToString(Log *, int, char *);
extern int  Aquire1WireNet(char *, char *);
extern void Release1WireNet(char *);

// local function prototypes
int InputMissionType(int);
int EnterNum(char *msg, int, long *, long, long);
void PrintResults(FILE *,int);


//----------------------------------------------------------------------
//  This is the Main routine for thermodl.
//
void main(short argc, char **argv)
{
   int Fahrenheit=FALSE;
   char str[800];
   char return_msg[128];

   // check arguments to see if request instruction with '?' or too many
   if ((argc < 2) || (argc > 3) || ((argc > 1) && (argv[1][0] == '?' || argv[1][1] == '?')))
   {
       printf("\nusage: thermoms 1wire_net_name </Fahrenheit>\n"
              "  - Thermochron configuration on the 1-Wire Net port\n"
              "  - 1wire_net_port required port name\n"
              "    example: \"COM1\" (Win32 DS2480),\"/dev/cua0\" \n"
              "    (Linux DS2480),\"1\" (Win32 TMEX)\n"
              "  - </Fahrenheit> optional Fahrenheit mode (default Celsius)\n"
              "  - version 1.03\n");
       exit(1);
   }

   // attempt to aquire the 1-Wire Net
   if (!Aquire1WireNet(argv[1], return_msg))
   {  
      printf("%s",return_msg);
      exit(1);
   }

   // success
   printf("%s",return_msg);

   //----------------------------------------
   // Introduction
   printf("\n/---------------------------------------------\n");
   printf("  Find and mission a DS1921 Thermochron iButton\n" 
          "  Version 1.03\n\n");

   // check arguments 
   Fahrenheit = FALSE;
   if (argc >= 3)
   {
      if ((argv[2][0] == '/') && 
           ((argv[2][1] == 'F') || (argv[2][1] == 'f')))
         Fahrenheit = TRUE;
   }

   // run the script to find the thermochron
   if (!RunThermoScript(StatusThermo,stdout))
   { 
      printf("Thermochon not found, end program\n");
      // release the 1-Wire Net
      Release1WireNet(return_msg);
      printf("%s",return_msg);
      exit(1);
   }

   // display mission status
   InterpretStatus(&MissStat);
   MissionStatusToString(&MissStat, Fahrenheit, &str[0]);
   printf("\n%s\n",str);

   // ask user mission questions
   if (!InputMissionType(Fahrenheit))
   {      
      printf("Aborting program\n");
      // release the 1-Wire Net
      Release1WireNet(return_msg);
      printf("%s",return_msg);
      exit(1);
   }

   // run the script to display the thermochron
   if (!RunThermoScript(MissionThermo,stdout))
   { 
      printf("Thermochon missioning not complete, end program\n");
      // release the 1-Wire Net
      Release1WireNet(return_msg);
      printf("%s",return_msg);
      exit(1);
   }

   // display the new mission status
   InterpretStatus(&MissStat);
   MissionStatusToString(&MissStat, Fahrenheit, &str[0]);
   printf("\n%s\n",str);

   // release the 1-Wire Net
   Release1WireNet(return_msg);
   printf("%s",return_msg);
   printf("End program normally\n");
   exit(0);
}


//--------------------------------------------------------------------------
//  Prints a message, and the current driver versions.
//
int InputMissionType(int ConvertToF)
{
   long num;
   float temp;
   char return_msg[128];
   
   // prompt to erase current mission
   if (!EnterNum("Erase current mission\n  (0) yes\n  (1) no\nAnswer:",1, &num, 0, 1))
      return FALSE;   

   // check for no erase 
   if (num == 1)
   {      
      printf("Current mission left undisturbed, end program\n");
      // release the 1-Wire Net
      Release1WireNet(return_msg);
      printf("%s",return_msg);
      exit(0);
   }

   // prompt for start delay
   if (!EnterNum("\nEnter start delay in minutes\n"
                 "Answer (0 to 65535):",5, &num, 0, 65535))
      return FALSE;   

   // set in mission status structure
   MissStat.start_delay = (ushort)num;
   
   // prompt for sample rate   
   if (!EnterNum("\nEnter sample rate in minutes\n"
                 "Answer (1 to 255):",3, &num, 1, 255))
      return FALSE;   

   // set in mission status structure
   MissStat.sample_rate = (uchar)num;

   // prompt to erase current mission
   if (!EnterNum("Enable roll-over\n  (0) yes\n  (1) no\nAnswer:",1, &num, 0, 1))
      return FALSE;   

   // rollover enabled?
   MissStat.rollover_enable = (num == 0);

   // prompt for high trip   
   if (ConvertToF)
   {
      if (!EnterNum("\nEnter high temperature threshold in Fahrenheit\n"
                    "Answer (-40 to 158):",3, &num, -40, 158))
         return FALSE; 
      temp = (float)((num - 32.0) * 5.0 / 9.0);  
   }
   else
   {
      if (!EnterNum("\nEnter high temperature threshold in Celsius\n"
                    "Answer (-40 to 70):",3, &num, -40, 70))
         return FALSE;   
      temp = (float)num;
   }

   // set in mission status structure
   MissStat.high_threshold = (uchar)(2 * (temp + 40));

   // prompt for low trip   
   if (ConvertToF)
   {
      if (!EnterNum("\nEnter low temperature threshold in Fahrenheit\n"
                    "Answer (-40 to 158):",3, &num, -40, 158))
         return FALSE; 
      temp = (float)((num - 32.0) * 5.0 / 9.0);  
   }
   else
   {
      if (!EnterNum("\nEnter low temperature threshold in Celsius\n"
                    "Answer (-40 to 70):",3, &num, -40, 70))
         return FALSE;   
      temp = (float)num;
   }

   // set in mission status structure
   MissStat.low_threshold = (uchar)(2 * (temp + 40));

   return TRUE;
}


//----------------------------------------------------------------------
// Enter a decimal string and convert it to an unsigned long
// Prompt again if not within min/max inclusive.
//                                                          
int EnterNum(char *msg, int numchars, long *value, long min, long max)
{
#ifdef WIN32 
   short tmp,cnt,isneg=FALSE; 
   char ch;

   // loop while not in correct range
   do
   {
      printf("%s",msg);
      *value = 0;

      // loop for each character read
      cnt = 0;
      do
      {
         ch = getch();

         // negative flag
         if (ch == '-')
         {
            if (!isneg)
            {
               isneg = TRUE;
               printf("-");
               cnt++;
            }
         }
         // backspace 
         if (ch == 0x08)
         {
            if (cnt)
            {
               if (isneg && (cnt == 1))
                  isneg = FALSE;
               else
                  *value /= 10;
               printf("%c %c",ch,ch);
               cnt--;
            } 
         }
         // escape 
         if (ch == 0x1B)
         {
            printf("  Aborted\n\n");
            return FALSE;
         }
         // enter 
         if (ch == 0x0D)
         {
            printf("\n");
            break;
         }
         // number 
         else if ((ch >= '0') && (ch <= '9'))
         {
            printf("%c",ch);
            tmp = ch - 0x30;
            *value *= 10; 
            *value += tmp;
            cnt++;
         }

      }
      while (cnt < numchars);

      printf("\n\n");

      if (isneg)
         *value = -*value;
   }
   while ((*value < min) || (*value > max));

   return TRUE;
#else
   char line[256];

   // loop while not in correct range
   do
   {
      printf("%s",msg);
      *value = 0;

      // read a line
      gets(line);

      // attempt to convert to a integer
      *value = atoi(line);
   }
   while ((*value < min) || (*value > max));

   return TRUE;
#endif
}




