/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#if (HAVE_PGSQL)
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"

#include "udm_xmalloc.h"
#ifdef WIN32
#include <process.h>
#endif

static int UdmPgSQLInitDB(UDM_DB *db)
{
  char port[8];
  const char* DBUser= UdmVarListFindStr(&db->Vars, "DBUser", NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars, "DBPass", NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", NULL);
  const char* DBSock= UdmVarListFindStr(&db->Vars, "socket", NULL);
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);

  /*
    If no host given then check if an alternative
    Unix domain socket is specified.
  */
  DBHost= (DBHost && DBHost[0]) ? DBHost : DBSock;
  
  sprintf(port,"%d",DBPort);
  db->specific= (void*) PQsetdbLogin(DBHost, DBPort ? port : 0, 0, 0,
                                     db->DBName, DBUser, DBPass);
  if (PQstatus((PGconn*)db->specific) == CONNECTION_BAD)
  {
    db->errcode=1;
    sprintf(db->errstr, "%s", PQerrorMessage((PGconn*)db->specific));
    return UDM_ERROR;
  }
  db->connected=1;
  return UDM_OK;
}


#define udm_oct2int(ch) ((ch) - '0')
#define udm_isoct(ch)   ((ch) >= '0' && (ch) <= '7')

static size_t PgUnescape(char *dst, const char *src, size_t length)
{
  char *dst0= dst;
  const char *srcend= src + length;
  for ( ; src < srcend; dst++, src++)
  {
    if (*src == '\\' && src + 4 <= srcend &&
        udm_isoct(src[1]) && udm_isoct(src[2]) && udm_isoct(src[3]))
    {
      *dst= udm_oct2int(src[1]) * 64 +
            udm_oct2int(src[2]) * 8 +
            udm_oct2int(src[3]);
      src+= 3;
    }
    else if (*src == '\\' && src[1] == '\\')
    {
      *dst= '\\';
      src++;
    }
    else
      *dst= *src;
  }
  return dst - dst0;
}



static int UdmPgSQLResultUnescape(UDM_SQLRES *res)
{
  size_t row;
  res->Items= (UDM_PSTR*)UdmMalloc(res->nRows * res->nCols * sizeof(UDM_PSTR));
  
  for (row= 0; row < res->nRows; row++)
  {
    size_t col, offs= res->nCols * row;
    for(col= 0; col < res->nCols; col++)
    {
      UDM_PSTR *I= &res->Items[offs + col];
      const char *ptr= PQgetvalue(res->pgsqlres,(int) row, (int) col);
      size_t len= PQgetlength(res->pgsqlres, (int) row, (int )col);
      I->val= (char*) UdmMalloc(len + 1);
      if (PQftype(res->pgsqlres, (int) col) == 17)
      {
        I->len= PgUnescape(I->val, ptr, len);
        I->val[I->len]= '\0';
      }
      else
      {
        memcpy(I->val, ptr, len);
        I->val[len]= '\0';
        I->len= len;
      }
    }
  }
  return UDM_OK;
}


static int UdmPgSQLQueryInternal(UDM_DB *db, UDM_SQLRES *res,
                                 const char *q, int unescape)
{
  size_t i;
  PGresult *PGres;
  PGconn *pgsql;
  int have_bytea= 0;
  
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }

  db->errcode=0;
  if(!db->connected)
  {
    UdmPgSQLInitDB(db);
    if(db->errcode)
      return UDM_ERROR;
  }
  
  pgsql= (PGconn*) db->specific;
  
  if(!(PGres= PQexec(pgsql,q)))
  {
    sprintf(db->errstr, "%s", PQerrorMessage(pgsql));
    db->errcode=1;
    return UDM_ERROR;
  }
  if(PQresultStatus(PGres)==PGRES_COMMAND_OK)
  {
    /* Free non-SELECT query */
    PQclear(PGres);
    return UDM_OK;
  }
  
  if(PQresultStatus(PGres) != PGRES_TUPLES_OK)
  {
    PQclear(PGres);
    sprintf(db->errstr, "%s", PQerrorMessage(pgsql));
    if(strstr(db->errstr,"uplicate") ||
       strstr(db->errstr, "") ||
       strstr(db->errstr, "повторный") ||
       strcasestr(db->errstr,"Duplizierter"))
    {
      return UDM_OK;
    }
    else
    {
      db->errcode = 1;
      return UDM_ERROR;
    }
  }
  
  if (!res)
  {
    /* 
     Don't allow to call UdmPgSQLQuery
     returning data with NULL res pointer
    */
    sprintf(db->errstr, 
            "UdmPgSQLQuery executed with res=NULL returned result %d, %s",
             PQresultStatus(PGres),PQerrorMessage(pgsql));
    db->errcode=1;
    return UDM_ERROR;
  }
  
  res->pgsqlres= PGres;
  res->nCols=(size_t)PQnfields(res->pgsqlres);
  res->nRows=(size_t)PQntuples(res->pgsqlres);
  res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
  for(i=0;i<res->nCols;i++)
  {
    res->Fields[i].sqlname = (char*)UdmStrdup(PQfname(res->pgsqlres,(int)i));
    if (PQftype(res->pgsqlres, (int) i) == 17)
      have_bytea++;
  }

  if (have_bytea && unescape)
    UdmPgSQLResultUnescape(res);

  return UDM_OK;
}

static int UdmPgSQLQuery(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  return UdmPgSQLQueryInternal(db, res, q, 1);
}

static int UdmPgSQLExecDirect(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  return UdmPgSQLQueryInternal(db, res, q, 0);
}


static void UdmPgSQLClose(UDM_DB *db)
{
  PQfinish((PGconn*)db->specific);
  db->specific = NULL;
}


static int UdmPgSQLBegin(UDM_DB *db)
{
  return UdmPgSQLQuery(db,NULL,"BEGIN WORK");
}


static int UdmPgSQLCommit(UDM_DB *db)
{
  return UdmPgSQLQuery(db,NULL,"END WORK");
}

static int
UdmPgSQLFetchRow(UDM_DB *db, UDM_SQLRES *res, UDM_PSTR *buf)
{
  size_t j;

  if (res->curRow >= res->nRows)
    return UDM_ERROR;
  
  for (j = 0; j < res->nCols; j++)
  {
    buf[j].val= PQgetvalue(res->pgsqlres,(int)res->curRow,(int)(j));
    buf[j].len= PQgetlength(res->pgsqlres, res->curRow, j);
    if (PQftype(res->pgsqlres, (int) j) == 17)
    {
      buf[j].len= PgUnescape(buf[j].val, buf[j].val, buf[j].len);
      buf[j].val[buf[j].len]= '\0';
    }
  }
  res->curRow++;
  return(UDM_OK);
}


UDM_SQLDB_HANDLER udm_sqldb_pgsql_handler =
{
  NULL,
  UdmPgSQLQuery,
  UdmPgSQLClose,
  UdmPgSQLBegin,
  UdmPgSQLCommit,
  NULL,
  NULL,
  NULL,
  UdmPgSQLFetchRow,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmPgSQLExecDirect
};

#endif
