/*
  Benchmark structure handling functions.
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "inc/mob.h"
#include "inc/benchmark.h"
#include "inc/cmdLine.h"


/* global variables externally defined */
extern struct globalSystem sys;


/* module-level variable definitions */
static struct bundle bundles[] = {{ "all", MAX_BENCHMARKS, {  BM_CACHE_SIZE, 
							      BM_CACHE_SHARE,
							      BM_CACHE_LINE,
							      BM_CACHE_ASSOC,
							      BM_CACHE_REPLACE,
							      BM_CACHE_WRITE,
							      BM_CACHE_INDEX,
							      BM_TLB_PAGE,
							      BM_TLB_ENTRIES,
							      BM_TLB_ASSOC }},
				   { "caches", 7, { BM_CACHE_SIZE, 
						    BM_CACHE_SHARE,
						    BM_CACHE_LINE,
						    BM_CACHE_ASSOC,
						    BM_CACHE_REPLACE,
						    BM_CACHE_WRITE,
						    BM_CACHE_INDEX }},
				  { "tlbs", 3, { BM_TLB_PAGE,
						 BM_TLB_ENTRIES,
						 BM_TLB_ASSOC }}};



/*
  Validate the data for a benchmark to determine whether 
  the test would need to be run again if requested.
*/
void validateBenchmarks( void ) {

  unsigned int m;

  /* loop through each benchmark and run the associated validation procedure */
  for( m = 0; m < MAX_BENCHMARKS; m++ )
    /* only check validity if not already marked for run */
    if( sys.benchmarks[m].state != STATE_RUN && sys.benchmarks[m].valid() )
      sys.benchmarks[m].state = STATE_VALID;
}


/*
  Mark a benchmark for running.

  id - benchmark id to set
  return - 0 if no error, -1 otherwise
*/  
int runBenchmark( unsigned int id ) {

  /* ensure the number is in range */
  if( id >= 0 && id < MAX_BENCHMARKS ) {
    /* mark the benchmark state as run */
    sys.benchmarks[id].state = STATE_RUN;
    return( 0 );
  }
  else
    return( -1 );
}


/*
  Mark a benchmark or possibly a set of benchmarks for running.

  mneumonic - mneumonic indicating which benchmark(s) to run
  return - 0 if no error, -1 otherwise
*/  
int runBenchmarks( char *mneumonic ) {

  unsigned int m, b;

  /* check each benchmark for a matching mneumonic */
  for( m = 0; m < MAX_BENCHMARKS; m++ )
    if( strcmp( mneumonic, sys.benchmarks[m].mneumonic ) == 0 )
      /* mark this benchmark */
      return( runBenchmark( m ) );

  /* check each benchmark bundle for a matching mneumonic */
  for( b = 0; b < MAX_BUNDLES; b++ )
    if( strcmp( mneumonic, bundles[b].mneumonic ) == 0 ) {
      /* mark each benchmark in the bundle */
      for( m = 0; m < bundles[b].numBenchmarks; m++ )
	runBenchmark( m );
      return( 0 );
    }

  /* must not have found mneumonic in benchmarks or bundles */
  return( -1 );
}


/*
  Mark all benchmark dependencies for running unless data is already valid.
*/
void runDependencies( void ) {

  unsigned int m, d, again;

  /* keep checking as long as new benchmarks are marked for running */
  do {
    again = 0;

    for( m = 0; m < MAX_BENCHMARKS; m++ )
      if( sys.benchmarks[m].state == STATE_RUN )
	/* check each dependency to determine whether it should be run as well */
	for( d = 0; d < sys.benchmarks[m].numDepends; d++ )
	  if( sys.benchmarks[sys.benchmarks[m].depends[d]].state < STATE_VALID ) {
	    runBenchmark( sys.benchmarks[m].depends[d] );
	    again++;
	  }
  } while( again );    
}

/*
  If no benchmarks are marked for run, mark everything not already marked valid.
*/
void finalizeBenchmarks( void ) {

  unsigned int m, run = 0;

  /* check if any benchmarks are marked for running */
  for( m = 0; m < MAX_BENCHMARKS; m++ )
    if( sys.benchmarks[m].state == STATE_RUN ) {
      run++;
      break;
    }

  /* if nothing marked for run, run everything not already valid */
  if( ! run ) 
    for( m = 0; m < MAX_BENCHMARKS; m++ )
      if( sys.benchmarks[m].state < STATE_VALID ) 
	sys.benchmarks[m].state = STATE_RUN;
}
