/*
  Command line parsing functions.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include "inc/mob.h"
#include "inc/cmdLine.h"
#include "inc/benchmark.h"


/* global variables externally defined */
extern struct globalSystem sys;


/* local function prototypes */
static void parseBenchmarks( char *arg );


/*
  Parse all command line arguments.

  argc - number of arguments
  argv - list of argument strings
 */
void parseCmdLine( int argc, char *argv[] ) {

  int c;

  /* set default values */
  sys.args.program = argv[0];

  /* parse each option and its arguments */
  while( (c = getopt( argc, argv, "b:c:o:r:t:v:ih?" )) != EOF )
    switch (c) {
    case 'b' : 
      parseBenchmarks( optarg );
      break;
    case 'c' :
      sys.args.config = optarg;
      break;
    case 'o' :
      sys.args.output = optarg;
      break;
    case 't' :
      sys.args.trials = atoi( optarg );
      if( sys.args.trials <= 0 )
	printUsage( "trials must be positive", 2 );
      break;
    case 'r' :
      sys.args.runTime = atoi( optarg )*1000;/* convert to usec */
      if( sys.args.runTime <= 0 )
	printUsage( "time must be positive", 2 );
      break;
    case 'v' :
      sys.args.verbosity = atoi( optarg );
      if( sys.args.verbosity < 0 )
	printUsage( "verbosity level must be zero or greater", 2 );
      break;
    case 'i' :
      sys.args.interactive++;
      break;
    default :
      printUsage( NULL, 0 );
    }
  
  /* check that no arguments were left on the command line */
  if( optind < argc )
    printUsage( NULL, 2 );
}


/*
  Parse a possible list of benchmark numbers and/or mneumonics.

  arg - benchmark option argument to parse
*/
static void parseBenchmarks( char *arg ) {

  char *mark, *sep = ARG_SEPARATOR;

  /* get pointer to first value */
  mark = strtok( arg, sep );
  /* continue to parse while available */
  while( mark ) {
    /* check whether mark is a number id or mneumonic */
    if( isalpha( (int)mark[0] ) ) {
      /* find mneumonic for individual or bundled benchmarks */
      if( runBenchmarks( mark ) )
      	printUsage( "benchmark not found", 2 );
    } 
    /* ensure number is within range and set to run state */
    else if( runBenchmark( atoi( mark ) ) )
      printUsage( "benchmark out of range", 2 );
    
    /* get next benchmark */
    mark = strtok( NULL, sep );
  }
}


/*
  Print the usage syntax with an optional error message.

  msg - informational message to print in addition to basic usage syntax
  ecode - exit code to return
*/
void printUsage( char *msg, int ecode ) {

  char *base;
  int m;

  /* find basename of current program */
  if( (base = strrchr( sys.args.program, '/' )) )
    base++;
  else
    base = sys.args.program;

  /* print an informational message if passed */
  if( msg )
    fprintf( stderr, "Error: %s\n", msg );
  
  /* print command-line usage and exit with error code */
  fprintf( stderr, "Usage: %s [-b <bm1>[,<bm2>,...]] [-c <config>] [-o output] [-t <trials>] [-r <runtime>] [-v [<verbosity>]] [-i] [-h|?]\n", base );

  /* list each available benchmark */
  fprintf( stderr, "Benchmarks:\n" );
  for( m = 0; m < MAX_BENCHMARKS; m++ )
    fprintf( stderr, "%d. %s\t- %s\n", m, sys.benchmarks[m].mneumonic, sys.benchmarks[m].name );
  fprintf( stderr, "\n" );

  exit( ecode );
}
