/**
 * Benchmark structure definition and hendling function prototypes.
 **/

#ifndef __BENCHMARK_H__
#define __BENCHMARK_H__


/* sizes of pre-defined lists */
#define MAX_BENCHMARKS 10 /* number of defined benchmarks */
#define MAX_BUNDLES    3 /* number of defined benchmark bundles */

/* benchmark ids */
#define BM_CACHE_SIZE    0
#define BM_CACHE_SHARE   1
#define BM_CACHE_LINE    2
#define BM_CACHE_ASSOC   3
#define BM_CACHE_REPLACE 4
#define BM_CACHE_WRITE   5
#define BM_CACHE_INDEX   6
#define BM_TLB_PAGE      7
#define BM_TLB_ENTRIES     8
#define BM_TLB_ASSOC     9


/* benchmark array initializer */
#define INIT_BENCHMARKS { \
 { "Cache Size/Levels", "cacheSize", STATE_NONE, cacheSize, cacheSizeValid, 0, {} }, \
 { "Cache Share", "cacheShare", STATE_NONE, cacheShare, cacheShareValid, 1, { BM_CACHE_SIZE } }, \
 { "Cache Line Size", "cacheLine", STATE_NONE, cacheLine, cacheLineValid, 1, { BM_CACHE_SIZE } }, \
 { "Cache Associativity", "cacheAssoc", STATE_NONE, cacheAssoc, cacheAssocValid, 1, { BM_CACHE_SIZE } }, \
 { "Cache Replacement Policy", "cacheReplace", STATE_NONE, cacheReplace, cacheReplaceValid, 1, { BM_CACHE_ASSOC } }, \
 { "Cache Write Policy", "cacheWrite", STATE_NONE, cacheWrite, cacheWriteValid, 1, { BM_CACHE_ASSOC } }, \
 { "Cache Indexing (Virtual/Physical)", "cacheAddress", STATE_NONE, cacheIndex, cacheIndexValid, 1, { BM_CACHE_ASSOC } }, \
 { "TLB Page Size", "tlbPage", STATE_NONE, tlbPage, tlbPageValid, 0, {} }, \
 { "TLB Entry Count", "tlbEntries", STATE_NONE, tlbEntries, tlbEntriesValid, 2, { BM_TLB_PAGE,BM_CACHE_ASSOC } }, \
 { "TLB Associativity", "tlbAssoc", STATE_NONE, tlbAssoc, tlbAssocValid, 1, { BM_TLB_PAGE} } }



/* enumeration types */
typedef enum state { STATE_NONE, STATE_VALID, STATE_DONE, STATE_RUN } state_t;


/* structure representing a single benchmark test */
struct benchmark {
  char *name;                           /* display name of benchmark */
  char *mneumonic;                      /* mneumonic used to select benchmark */
  state_t state;                        /* current state of the benchmark */
  int (*perform) ();                    /* function used to perform benchmark for all possible entities */
  int (*valid) ();                      /* function used to validate data read from config file */
  unsigned int numDepends;              /* number of dependencies in list */
  unsigned int depends[MAX_BENCHMARKS]; /* list of benchmark indexes that this test is dependent upon */
};


/* structure representing a collection of benchmarks bundled together for ease of operation */
struct bundle {
  char *mneumonic;                         /* mneumonic used to select benchmark bundle */
  unsigned int numBenchmarks;              /* number of benchmarks in bundle list */
  unsigned int benchmarks[MAX_BENCHMARKS]; /* list of indexes of benchmarks to use */
};



/* general function prototypes */


void validateBenchmarks( void );

int runBenchmark( unsigned int num );

int runBenchmarks( char *mneumonic );

void runDependencies( void );

void finalizeBenchmarks( void );



/* specific benchmark entry-point function prototypes */


int cacheSize( void );

int cacheSizeValid( void );

int cacheShare( void );

int cacheShareValid( void );

int cacheLine( void );

int cacheLineValid( void );

int cacheAssoc( void );

int cacheAssocValid( void );

int cacheReplace( void );

int cacheReplaceValid( void );

int cacheWrite( void );

int cacheWriteValid( void );

int cacheIndex( void );

int cacheIndexValid( void );

int tlbPage( void );

int tlbPageValid( void );

int tlbEntries( void );

int tlbEntriesValid( void );

int tlbAssoc( void );

int tlbAssocValid( void );



/* benchmark access functions used across benchmarks. */

double dataRead( char *array, unsigned int size, unsigned int stride, unsigned int sets );

double dataReadWrap( char *array, unsigned int size, unsigned int stride, unsigned int sets );

double dataWrite( char *array, unsigned int size, unsigned int stride, unsigned int sets );

double instExecute( char *code, unsigned int size, unsigned int stride, unsigned int sets );

double shareCheck( char *array, char *code, unsigned int size, unsigned int stride, unsigned int sets );

double shareCheckWrap( char *array, char *code, unsigned int size, unsigned int stride, unsigned int sets );

double ctrlRead( char *array, unsigned int size, unsigned int stride, unsigned int sets );

double ctrlReadWrap( char *array, unsigned int size, unsigned int stride, unsigned int sets );

#endif
