/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED `AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

#include "mod_log_data.h"

int alloc_log_data_config(apr_pool_t * pool, log_data_config_t * cfg)
{
    log_data_config_t new;

    if ((new =
	 (log_data_config_t) apr_pcalloc(pool,
					 sizeof(struct log_data_config))) ==
	NULL)
	return -1;

    *cfg = new;
    return 0;

}

static const char *log_data_set_display_hdr(cmd_parms *cmd, void *dummy,int flag)
{

    log_data_config_t cfg =
	ap_get_module_config(cmd->server->module_config,
			     &mod_log_data_module);


    cfg->display_hdr = flag;
    return NULL;

}

static const char *log_data_set_line_len(cmd_parms *cmd, void *dummy,
					 const char *arg)
{

    log_data_config_t cfg =
	ap_get_module_config(cmd->server->module_config,
			     &mod_log_data_module);
    int line_len = 0;

    line_len = atoi(arg);

    if (line_len == 0)
	return NULL;

    if (line_len < 0)
	return "LogDataLineLen Must be > 0";

    /* considering last caracter is O */
    cfg->line_len = line_len + 1;
    return NULL;

}

static const char *log_data_set_max_brigade(cmd_parms *cmd, void *dummy,
					    const char *arg)
{

    log_data_config_t cfg =
	ap_get_module_config(cmd->server->module_config,
			     &mod_log_data_module);
    int max_brigade = 0;

    max_brigade = atoi(arg);

    if (max_brigade == 0)
	return NULL;

    if (max_brigade < 0)
	return "LogDataMaxBrigades Must be > 0";

    cfg->max_brigade = max_brigade;
    return NULL;

}

static const char *log_data_set_max_bucket(cmd_parms *cmd, void *dummy,
					   const char *arg)
{

    log_data_config_t cfg =
	ap_get_module_config(cmd->server->module_config,
			     &mod_log_data_module);
    int max_buckets = 0;

    max_buckets = atoi(arg);

    if (max_buckets == 0)
	return NULL;

    if (max_buckets < 0)
	return "LogDataMaxBuckets Must be > 0";

    cfg->max_bucket = max_buckets;
    return NULL;

}

static void *create_log_data_server_config(apr_pool_t * p, server_rec *s)
{
    log_data_config_t cfg;

    alloc_log_data_config(p, &cfg);
    cfg->line_len = LOG_LINE_LEN;
    cfg->max_brigade = LOG_MAX_BRIGADE;
    cfg->max_bucket = LOG_MAX_BUCKET;

    return cfg;
}

int alloc_data_log(apr_pool_t * pool, log_data_t * data)
{
    log_data_t new;

    if ((new =
	 (log_data_t) apr_pcalloc(pool, sizeof(struct log_data))) == NULL)
	return -1;

    *data = new;
    return 0;
}

int dump_headers_from_table (request_rec *r, apr_table_t *hdrs)
{
    const apr_array_header_t *headers_in_array;
    const apr_table_entry_t *headers;
    int counter = 0;
	            

    headers_in_array = apr_table_elts(hdrs);
    headers = (const apr_table_entry_t *) headers_in_array->elts;
        

    for (counter = 0; counter < headers_in_array->nelts; counter++) {

    	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_NOTICE, 0, r->server,
    	                     "HEADER[%i]: %s = %s", counter, headers[counter].key,
                             headers[counter].val);
    }

    return 0;

}

int dump_buffer(request_rec *r, const char *buffer, int len)
{
    log_data_config_t cfg =
	ap_get_module_config(r->server->module_config, &mod_log_data_module);
    int i = 0;
    int charinline = 0;
    char *logline;

    logline = calloc(1, cfg->line_len);
    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		 "line len is %i", cfg->line_len);

    for (i = 0; i < len; i++) {

	if (isgraph(buffer[i])) {

	    if (charinline < cfg->line_len) {
		logline[charinline] = buffer[i];
	    }
	    charinline++;

	}
	else {
	    if (charinline < cfg->line_len) {
		logline[charinline] = '.';
		charinline++;
	    }
	}

	if (charinline == cfg->line_len - 1) {
	    logline[cfg->line_len] = '\0';
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0,
			 r->server, "%s", logline);
	    memset(logline, 0, cfg->line_len);
	    charinline = 0;
	}
    }

    if (charinline > 1) {
	logline[charinline] = 0;
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "%s", logline);
	memset(logline, 0, cfg->line_len);
    }

    free(logline);

    return 0;
}

static apr_status_t
dump_incoming_data_filter(ap_filter_t * f, apr_bucket_brigade * bb,
			  ap_input_mode_t mode, apr_read_type_e block,
			  apr_off_t readbytes)
{
    request_rec *r = f->r;
    apr_bucket *e;
    int seen_eos = 0;
    int config_stop = 0;
    log_data_t data_log;
    apr_status_t rv;
    log_data_config_t cfg =
	ap_get_module_config(r->server->module_config, &mod_log_data_module);


    apr_pool_userdata_get((void **) &data_log, "DATA_LOG", r->pool);

    if (!data_log) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "Seems to be the first brigade - creating data log");
	alloc_data_log(r->pool, &data_log);
	data_log->num_bucket = 0;
	data_log->num_brigade = 0;
	data_log->total_read = 0;
	apr_pool_userdata_set(data_log, "DATA_LOG", NULL, r->pool);
    }


	

    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		 "Logging Incoming data for %s", r->the_request);

    if (mode != AP_MODE_READBYTES) {
	return ap_get_brigade(f->next, bb, mode, block, readbytes);
    }

    rv = ap_get_brigade(f->next, bb, mode, block, readbytes);
    if (rv != APR_SUCCESS) {
	return rv;
    }
    data_log->num_brigade++;

    if (cfg->display_hdr == 1 && data_log->num_brigade == 1) {
    	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,"Dump Incoming Headers");
    	dump_headers_from_table (r, r->headers_in);
    }


    if (cfg->max_brigade != 0 && data_log->num_brigade > cfg->max_brigade) {
	config_stop = 1;
    }


    if (APR_BRIGADE_EMPTY(bb)) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "Found an empty brigade");
	return ap_get_brigade(f->next, bb, mode, block, readbytes);
    }

    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		 "Entering brigade %i", data_log->num_brigade);
    APR_BRIGADE_FOREACH(e, bb) {

	const char *str = NULL;
	int length = 0;

	if (APR_BUCKET_IS_EOS(e)) {
	    seen_eos = 1;
	    break;
	}

	if (APR_BUCKET_IS_FLUSH(e)) {

	    break;
	}

	data_log->num_bucket++;

	if (cfg->max_bucket != 0 && data_log->num_bucket > cfg->max_bucket) {
	    config_stop = 1;
	}

	apr_bucket_read(e, &str, &length, APR_NONBLOCK_READ);

	if (length > 0) {
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0,
			 r->server, "Dumping bucket %i: %i bytes",
			 data_log->num_bucket, length);
	    dump_buffer(r, str, length);
	    data_log->total_read += length;
	}
	else {
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0,
			 r->server, "Can't find data in the bucket");

	}
    }


    if (seen_eos == 1 || config_stop == 1) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "End Logging - Removing log filter");
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "Read %i bytes of incoming data in %i brigade(s) and %i bucket(s)",
		     data_log->total_read, data_log->num_brigade,
		     data_log->num_bucket);
	ap_remove_input_filter(f);
    }

    return APR_SUCCESS;
}


static apr_status_t dump_outgoing_data_filter (ap_filter_t * f, apr_bucket_brigade * bb)
{
    request_rec *r = f->r;
    apr_bucket *e;
    int seen_eos = 0;
    int config_stop = 0;
    log_data_t data_log;
    apr_status_t rv;
    log_data_config_t cfg =
	ap_get_module_config(r->server->module_config, &mod_log_data_module);

    apr_pool_userdata_get((void **) &data_log, "OUT_DATA_LOG", r->pool);

    if (!data_log) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "Seems to be the first brigade - creating data log");
	alloc_data_log(r->pool, &data_log);
	data_log->num_bucket = 0;
	data_log->num_brigade = 0;
	data_log->total_read = 0;
	apr_pool_userdata_set(data_log, "OUT_DATA_LOG", NULL, r->pool);
    }

    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		 "Log Outgoing data");

    data_log->num_brigade++;

    if (cfg->display_hdr == 1 && data_log->num_brigade == 1) {
    	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,"Dump outgoing Headers");
    	dump_headers_from_table (r, r->headers_out);
    }


    if (cfg->max_brigade != 0 && data_log->num_brigade > cfg->max_brigade) {
	config_stop = 1;
    }


    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		 "Entering brigade %i", data_log->num_brigade);
    APR_BRIGADE_FOREACH(e, bb) {

	const char *str = NULL;
	int length = 0;

	if (APR_BUCKET_IS_EOS(e)) {
	    seen_eos = 1;
	    break;
	}

	if (APR_BUCKET_IS_FLUSH(e)) {

	    break;
	}

	data_log->num_bucket++;

	if (cfg->max_bucket != 0 && data_log->num_bucket > cfg->max_bucket) {
	    config_stop = 1;
	}

	apr_bucket_read(e, &str, &length, APR_NONBLOCK_READ);

	if (length > 0) {
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0,
			 r->server, "Dumping bucket %i: %i bytes",
			 data_log->num_bucket, length);
	    dump_buffer(r, str, length);
	    data_log->total_read += length;
	}
	else {
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0,
			 r->server, "Can't find data in the bucket");

	}
    }


    if (seen_eos == 1 || config_stop == 1) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "End Logging - Remove log filter");

	ap_log_error(APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, 0, r->server,
		     "Read %i bytes of outgoing data in %i brigade(s) and %i bucket(s)",
		     data_log->total_read, data_log->num_brigade,
		     data_log->num_bucket);

	ap_remove_output_filter(f);
    }

    return ap_pass_brigade(f->next, bb);
}


static const command_rec log_data_filter_cmds[] = {

    AP_INIT_FLAG("LogDataDisplayHdrs", log_data_set_display_hdr, NULL, OR_AUTHCFG,
                  "Ignore requests from the client for uncached content"),
    AP_INIT_TAKE1("LogDataLineLen", log_data_set_line_len, NULL, RSRC_CONF,
		  "Setup how many caracters a log line contains"),
    AP_INIT_TAKE1("LogDataMaxBrigades", log_data_set_max_brigade, NULL,
		  RSRC_CONF,
		  "Setup how many brigades will be logged"),
    AP_INIT_TAKE1("LogDataMaxBuckets", log_data_set_max_bucket, NULL,
		  RSRC_CONF,
		  "Setup how many buckets in each brigade will be logged"),
    {NULL}
};




static void register_hooks(apr_pool_t * p)
{
    ap_register_input_filter("LOG_INCOMING_DATA", dump_incoming_data_filter,
			     NULL, AP_FTYPE_CONTENT_SET);
    ap_register_output_filter("LOG_OUTGOING_DATA", dump_outgoing_data_filter,
    			     NULL, AP_FTYPE_CONTENT_SET);

}

module AP_MODULE_DECLARE_DATA mod_log_data_module = {
    STANDARD20_MODULE_STUFF,
    NULL,			/* dir config creater */
    NULL,			/* dir merger --- default is to override */
    create_log_data_server_config,	/* server config */
    NULL,			/* merge server config */
    log_data_filter_cmds,	/* command table */
    register_hooks,		/* register hooks */
};
