
/*
 * Mylo v0.2.2 - MySQL Logger module for Apache 1.3.x
 *
 * 2001, 2003 (C) yvind Grnnesby <oyving@orakel.ntnu.no>
 * This software is subject to GPL v2.
 *
 */

/*
 * Needed configuration in httpd.conf:
 * 
 *   - MyloHost hostname [portnr]
 *     the hostname and port to the database
 *
 *   - MyloUser username
 *     username to use with mysql
 *
 *   - MyloPassword password
 *     password for the mysql username
 *
 *   - MyloDBName dbname
 *     the database that contains the requests table
 *
 *   - MyloTable tablename
 *     the name of the table.  defaults to 'request'.
 *
 *   - MyloError (log, error)
 *     what to do in case of an error.  log just logs the error while
 *     error makes the module return 500.
 */


#include "httpd.h"
#include "http_log.h"
#include "http_config.h"

#include <mysql/mysql.h>
#include <stdio.h>

#define MYLO_VERSION "0.2.2"
#define MYLO_MAXQLEN 2048

static MYSQL *connection = NULL;

typedef struct {
char *host;
char *username;
char *password;
char *dbname;
char *table;
int   error;
int   port;
} mylo_server_cfg;

static enum {
    MYLO_LOG,
    MYLO_ERROR
} mylo_error;

module MODULE_VAR_EXPORT mylo_module;

static void  mylo_server_init(server_rec *c, pool *p);
static void *mylo_server_config(pool *p, server_rec *c);
static int   mylo_log(request_rec *r);
static void  mylo_child_init(server_rec *c, pool *p);
static void  mylo_child_exit(server_rec *c, pool *p);

static const char *mylo_set_host(cmd_parms *cmd, void *dummy, char *host, char *port);
static const char *mylo_set_user(cmd_parms *cmd, void *dummy, char *arg);
static const char *mylo_set_password(cmd_parms *cmd, void *dummy, char *arg);
static const char *mylo_set_dbname(cmd_parms *cmd, void *dummy, char *arg);
static const char *mylo_set_table(cmd_parms *cmd, void *dummy, char *arg);
static const char *mylo_set_error(cmd_parms *cmd, void *dummy, char *arg);

/* 
 * Commands and their associated functions.
 */
static command_rec mylo_cmds[] = {
    { "MyloHost", mylo_set_host, NULL,
      RSRC_CONF, TAKE12, "mysql server hostname and portnr" },
    { "MyloUser", mylo_set_user, NULL,
      RSRC_CONF, TAKE1, "mysql username" },
    { "MyloPassword", mylo_set_password, NULL,
      RSRC_CONF, TAKE1, "mysql password" },
    { "MyloDBName", mylo_set_dbname, NULL,
      RSRC_CONF, TAKE1, "mysql database" },
    { "MyloTable", mylo_set_table, NULL,
      RSRC_CONF, TAKE1, "mysql table name" },
    { "MyloError", mylo_set_error, NULL,
      RSRC_CONF, TAKE1, "mylo error handling" },
    { NULL }
};

static const char *mylo_set_host(cmd_parms *cmd, void *dummy, char *host, char *port)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *) 
	ap_get_module_config(s->module_config, &mylo_module);

    if (port != NULL) {
        cfg->port = atoi(port);
    }
    cfg->host = host;
    return NULL;
}

static const char *mylo_set_user(cmd_parms *cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *)
	ap_get_module_config(s->module_config, &mylo_module);
    cfg->username = arg;
    return NULL;
}

static const char *mylo_set_password(cmd_parms *cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *)
	ap_get_module_config(s->module_config, &mylo_module);
    cfg->password = arg;
    return NULL;
}

static const char *mylo_set_dbname(cmd_parms *cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *)
	ap_get_module_config(s->module_config, &mylo_module);
    cfg->dbname = arg;
    return NULL;
}

static const char *mylo_set_table(cmd_parms *cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *)
	ap_get_module_config(s->module_config, &mylo_module);
    cfg->table = arg;
    return NULL;
}

static const char *mylo_set_error(cmd_parms *cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    mylo_server_cfg *cfg = (mylo_server_cfg *)
	ap_get_module_config(s->module_config, &mylo_module);

    if (strcmp("log", arg) == 0) {
        cfg->error = MYLO_LOG;
    } else if (strcmp("error", arg) == 0) {
        cfg->error = MYLO_ERROR;
    } else {
        return "Incorrect value for MyloError";
    }

    return NULL;
}

/* 
 * Module definition and the associated functions / variables
 */
module MODULE_VAR_EXPORT mylo_module = {
    STANDARD_MODULE_STUFF,
    mylo_server_init,		/* initializer */
    NULL,                        /* dir config creater */
    NULL,                	/* dir merger --- default is to override */
    mylo_server_config,	        /* server config */
    NULL,			/* merge server config */
    mylo_cmds,	                /* command table */
    NULL,                        /* handlers */
    NULL,			/* filename translation */
    NULL,                        /* check_user_id */
    NULL,		        /* check auth */
    NULL,			/* check access */
    NULL,			/* type_checker */
    NULL,			/* fixups */
    mylo_log,			/* logger */
    NULL,			/* header parser */
    mylo_child_init,		/* child_init */
    mylo_child_exit,	        /* child_exit */
    NULL          	        /* post read-request */
};

static void mylo_server_init(server_rec *c, pool *p)
{
    ap_add_version_component("Mylo/" MYLO_VERSION);
}

static void *mylo_server_config(pool *p, server_rec *c)
{
    mylo_server_cfg *cfg = (mylo_server_cfg *) ap_pcalloc(p, sizeof *cfg);

    /* set the default values for configuration */
    cfg->host = "localhost";
    cfg->username = "www";
    cfg->password = "foobar";
    cfg->dbname = "logs";
    cfg->table = "request";
    cfg->error = MYLO_LOG;
    cfg->port = 0;

    return cfg;
}

static int mylo_log(request_rec *r)
{
    char query[MYLO_MAXQLEN], reqstr[MYLO_MAXQLEN], eident[MYLO_MAXQLEN];
    char ervalue[MYLO_MAXQLEN];
    char *logname, *ident, *rvalue;
    int length;
    mylo_server_cfg *cfg = 
	(mylo_server_cfg *) ap_get_module_config(r->server->module_config, 
                                             &mylo_module);

    if (mysql_ping(connection) != 0) {
        ap_log_error(APLOG_MARK, APLOG_ERR, r->server,
                     "mylo: mysql connection error: %s", 
                     mysql_error(connection));
        if (cfg->error == MYLO_ERROR) {
            return SERVER_ERROR;
        }
    } else {

        memset(query, 0, MYLO_MAXQLEN);
        memset(reqstr, 0, MYLO_MAXQLEN);
        memset(eident, 0, MYLO_MAXQLEN);
        memset(ervalue, 0, MYLO_MAXQLEN);

        /* escape the query string from the client */
        length = strlen(r->the_request);
        length = length < MYLO_MAXQLEN ? length : MYLO_MAXQLEN-1;
        mysql_real_escape_string(connection, reqstr, r->the_request, length);

        /* escape the identd logname */
        logname = ap_get_remote_logname(r);
        if (logname == NULL) {
            logname = "-";
        } else {
            length = strlen(logname);
            length = length < MYLO_MAXQLEN ? length : MYLO_MAXQLEN-1;
            mysql_real_escape_string(connection, eident, logname, length);
            logname = eident;
        }
    

        /* finally escape the HTTP user information */
        rvalue = r->connection->user;
        if (rvalue == NULL) {
            rvalue = "-";   
        } else if (strlen(rvalue) == 0) {
            rvalue = "\"\"";
        } else {
            length = strlen(rvalue);
            length = length < MYLO_MAXQLEN ? length : MYLO_MAXQLEN-1;
            mysql_real_escape_string(connection, ervalue, rvalue, length);
            rvalue = ervalue;
        }

        if (snprintf(query, MYLO_MAXQLEN,
                     "insert into %s values "\
                     "('%s','%s','%s','%s','%d','%s','%d','%d')",
                     cfg->table,
                     r->server->server_hostname,
                     r->hostname,
                     logname,
                     rvalue,
                     r->request_time,
                     reqstr,
                     r->status,
                     r->bytes_sent) == -1) {
            ap_log_error(APLOG_MARK, APLOG_ERR, r->server,
                         "mylo: mysql query error: query too long.");
            if (cfg->error == MYLO_ERROR) {
                return SERVER_ERROR;
            }
        } else {
            if (mysql_real_query(connection, query, MYLO_MAXQLEN) != 0) {
                ap_log_error(APLOG_MARK, APLOG_ERR, r->server,
                             "mylo: mysql query error: %s",
                             mysql_error(connection));
                if (cfg->error == MYLO_ERROR) {
                    return SERVER_ERROR;
                }
            }
        }
    }
    return OK;
}

static void mylo_child_init(server_rec *c, pool *p)
{
    mylo_server_cfg *cfg;

    cfg = (mylo_server_cfg *) ap_get_module_config(c->module_config, 
                                                   &mylo_module);
    
    connection = mysql_init(connection);
    if (mysql_real_connect(connection,
                           cfg->host,
                           cfg->username,
                           cfg->password,
                           cfg->dbname,
                           cfg->port,
                           NULL, 0) == NULL) {
        ap_log_error(APLOG_MARK, APLOG_ERR, c, 
                     "mylo: mysql conection error: %s", 
                     mysql_error(connection));
    }
}

static void mylo_child_exit(server_rec *c, pool *p)
{
    mysql_close(connection);
}
