/*  This file is part of mod_pcgi2 - apache module for handling PCGI protocol.
    Author: Oleg Broytmann <phd@phd.pp.ru>
    Copyright (C) 1999-2003 PhiloSoft Design.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This file contains code from pcgi-wrapper.

Copyright (c) 1998, Digital Creations, Fredericksburg, VA, USA.  All
rights reserved. This software includes contributions from Jeff Bauer.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

  o Redistributions of source code must retain the above copyright
    notice, this list of conditions, and the disclaimer that follows.

  o Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions, and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.

  o All advertising materials mentioning features or use of this
    software must display the following acknowledgement:

      This product includes software developed by Digital Creations
      and its contributors.

  o Neither the name of Digital Creations nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.


THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS *AS IS* AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||
  - 2.0a5  29 October 1998
    - (brian) fixed arg type mismatch for semctl

  - 2.0a4  10 August 1998
    - (jeff) fixed Win32 local socket host address issues

  - 2.0a4  8 August 1998
    - (jeff) added Win32 support
*/

/* This file contains code from AMK's mod_pcgi.c
 * mod_pcgi: Apache module to handle the Persistent CGI protocol
 *
 * v0.00
 *
 * Author: A.M. Kuchling <akuchlin@mems-exchange.org>
 * http://starship.python.net/crew/amk/
 */


#include <httpd.h>
#include <http_config.h>
#include <http_log.h>
#include <http_main.h>
#include <http_protocol.h>
#include <util_script.h>

#include "pcgi.h"


#ifdef MOD_PCGI2_DEBUG
#include <stdio.h>
#define debug(msg) fprintf(stderr, "%s, %d, %s: %s\n", __FILE__, __LINE__, __FUNCTION__, (msg))
#define debug2(fmt, msg, param) fprintf(stderr, fmt, msg, param)
#define debug3(fmt, msg, param1, param2) fprintf(stderr, fmt, msg, param1, param2)

#define show_pcgi_resource(r, from) { \
    debug2("pcgi resource attributes from %s%s:\n", from, ""); \
    debug2("  %-20s: %s\n", "r->sw_info", r->sw_info); \
    debug2("  %-20s: %s\n", "r->sw_name", r->sw_name); \
    debug2("  %-20s: %s\n", "r->sw_home", r->sw_home); \
    debug2("  %-20s: %s\n", "r->sw_exe", r->sw_exe); \
    debug2("  %-20s: %s\n", "r->procpath", r->procpath); \
    debug2("  %-20s: %s\n", "r->sockpath", r->sockpath); \
    debug2("  %-20s: %s\n", "r->modpath", r->modpath); \
    debug2("  %-20s: %s\n", "r->pubpath", r->pubpath); \
    debug2("  %-20s: %d\n", "r->procid", r->procid); \
    debug2("  %-20s: %s\n", "r->insertPath", r->insertPath); \
    debug2("  %-20s: %s\n", "r->pythonPath", r->pythonPath); \
    debug2("  %-20s: %d\n", "r->sockport", r->sockport); \
    debug2("  %-20s: %s\n", "r->sockhost", r->sockhost); \
    debug2("  %-20s: %d\n", "r->displayErrors", r->displayErrors); \
    debug2("  %-20s: %d\n", "CloseFileDescriptors", CloseFileDescriptors); \
    debug2("  %-20s: %s\n", "r->root", r->root); \
    debug2("  %-20s: %d\n", "r->cut_slashes", r->cut_slashes); \
    debug2("  %-20s: %s\n", "r->location", r->location); \
}

#else

#define debug(msg)
#define debug2(fmt, msg, param)
#define debug3(fmt, msg, param1, param2)
#define show_pcgi_resource(r, from)
#endif


extern int CloseFileDescriptors;
int pcgiAssignCloseFileDescriptors(pcgiResource *r, const char *val);


/* Define our module */
#ifdef APACHE2
module AP_MODULE_DECLARE_DATA pcgi2_module;
#else
module MODULE_VAR_EXPORT pcgi2_module;
#endif

#define MOD_PCGI2_VERSION "(mod_pcgi2/2.0.2; PCGI/2.0a5)"

#ifdef APACHE2
static int init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
    ap_add_version_component(p, MOD_PCGI2_VERSION);
    return OK;
}

#else

static void init(server_rec *s, pool *p)
{
#if MODULE_MAGIC_NUMBER >= 19980527
   ap_add_version_component(MOD_PCGI2_VERSION);
#endif
}
#endif


/* Init/merge per-server and per-directory configs */
static pcgiResource *init_pcgiResource(pool *p)
{
    pcgiResource *r = (pcgiResource *) ap_pcalloc(p, sizeof(pcgiResource));
    r->conn = -1;
    r->pcgi_env = ap_make_table(p, 4);
    return r;
}


/* Merge resource records */
#define CREATE_pcgi_mergef(name, addr, maxsz) static void pcgi_merge_##name(pcgiResource *r, pcgiResource *base, pcgiResource *overrides) \
{ \
   if (overrides->addr[0]) { \
      strncpy(r->addr, overrides->addr, maxsz); \
   } else if (base->addr[0]) { \
      strncpy(r->addr, base->addr, maxsz); \
   } \
}

CREATE_pcgi_mergef(swname, sw_name, MAXSZ)
CREATE_pcgi_mergef(swhome, sw_home, MAXSZ)
CREATE_pcgi_mergef(swexe, sw_exe, MAXSZ)
CREATE_pcgi_mergef(errlog, errlog, MAXSZ)
CREATE_pcgi_mergef(host, sockhost, MAXSZ)
CREATE_pcgi_mergef(modPath, modpath, MAXSZ)
CREATE_pcgi_mergef(procPath, procpath, MAXSZ)
CREATE_pcgi_mergef(sockPath, sockpath, MAXSZ)
CREATE_pcgi_mergef(pubpath, pubpath, MAXSZ)
CREATE_pcgi_mergef(pythonPath, pythonPath, MAXPATH)
CREATE_pcgi_mergef(insertPath, insertPath, MAXPATH)
CREATE_pcgi_mergef(root, root, MAXSZ)

static void pcgi_merge_displayErrors(pcgiResource *r, pcgiResource *base, pcgiResource *overrides)
{
   if (overrides->displayErrors) {
      r->displayErrors = overrides->displayErrors;
   } else if (base->displayErrors) {
      r->displayErrors = base->displayErrors;
   }
}

static void pcgi_merge_port(pcgiResource *r, pcgiResource *base, pcgiResource *overrides)
{
   if (overrides->sockport) {
      r->sockport = overrides->sockport;
   } else if (base->sockport) {
      r->sockport = base->sockport;
   }
}


static void pcgi_merge_slashes(pcgiResource *r, pcgiResource *base, pcgiResource *overrides)
{
   debug3("merge cutslashes: old=%d, base=%d, over=%d\n", r->cut_slashes, base->cut_slashes, overrides->cut_slashes);
   if (overrides->cut_slashes) {
      r->cut_slashes = overrides->cut_slashes;
   } else if (base->cut_slashes) {
      r->cut_slashes = base->cut_slashes;
   }
}


static void pcgi_merge_location(pcgiResource *r, pcgiResource *base, pcgiResource *overrides)
{
   debug3("merge location: old=%s, base=%s, over=%s\n", r->location, base->location, overrides->location);
   if (overrides->location) {
      r->location = overrides->location;
   } else if (base->location) {
      r->location = base->location;
   }
}


/* Utility function used to copy pcgi_env table into req->subprocess_env */
static int copy_pcgi_env(void *p, const char *key, const char *value)
{
  table *env = (table *)p;
  debug2("copy pcgi_env: %s=%s\n", key, value);
  ap_table_setn(env, key, value);
  return 1;
}


static pcgiResource *merge_pcgiResources(pool *p, pcgiResource *base, pcgiResource *overrides)
{
    pcgiResource *r = init_pcgiResource(p);
    pcgi_merge_swname(r, base, overrides);
    pcgi_merge_swhome(r, base, overrides);
    pcgi_merge_swexe(r, base, overrides);
    pcgi_merge_errlog(r, base, overrides);
    pcgi_merge_host(r, base, overrides);
    pcgi_merge_modPath(r, base, overrides);
    pcgi_merge_procPath(r, base, overrides);
    pcgi_merge_sockPath(r, base, overrides);
    pcgi_merge_pubpath(r, base, overrides);
    pcgi_merge_pythonPath(r, base, overrides);
    pcgi_merge_insertPath(r, base, overrides);
    pcgi_merge_displayErrors(r, base, overrides);
    pcgi_merge_port(r, base, overrides);
    pcgi_merge_root(r, base, overrides);
    ap_table_do(copy_pcgi_env, r->pcgi_env, base->pcgi_env, NULL);
    ap_table_do(copy_pcgi_env, r->pcgi_env, overrides->pcgi_env, NULL);
    pcgi_merge_slashes(r, base, overrides);
    pcgi_merge_location(r, base, overrides);

    show_pcgi_resource(r, "merge");
    return r;
}


/* static void *create_pcgi_server_config(pool *p, server_rec *s)
{
    debug("create_pcgi_server_config");
    debug2("%s for server %s\n", "create_pcgi_server_config", s->server_hostname);
    return (void *)init_pcgiResource(p);
}

static void *merge_pcgi_server_config(pool *p, void *basev, void *overridesv)
{
    debug("merge_pcgi_server_config");
    return (void *)merge_pcgiResources(p, (pcgiResource *)basev, (pcgiResource *)overridesv);
} */


static void *create_pcgi_dir_config(pool *p, char *d)
{
    pcgiResource *r = init_pcgiResource(p);

    debug("create_pcgi_dir_config");
    #ifdef MOD_PCGI2_DEBUG
    if (d)
       debug2("%s for location %s\n", "create_pcgi_dir_config", d);
    else
       debug2("%s for location (null)%s\n", "create_pcgi_dir_config", "");
    #endif

    if (d) {
       if (strcmp(d, "/") != 0) { /* Count cut_slashes for only non-root locations */
          int i, cs = 0;
          for (i = 0; i < strlen(d); i++)
             if (d[i] == PATHSEP) cs++;
          debug2("counting cutslashes for location %s: %d\n", d, cs);
          r->cut_slashes = cs;
       }
       r->location = (char *)ap_pstrdup(p, d);
    }
    return (void *)r;
}

static void *merge_pcgi_dir_config(pool *p, void *basev, void *overridesv)
{
    debug("merge_pcgi_dir_config");
    return (void *)merge_pcgiResources(p, (pcgiResource *)basev, (pcgiResource *)overridesv);

}


/* Parse command parameters */
#define CREATE_pcgi_setf(name, addr, maxsz) static const char *pcgi_set_##name(cmd_parms *parms, void *r, const char *arg) \
{ \
   strncpy(((pcgiResource *)r)->addr, arg, maxsz); \
   return NULL; \
}

CREATE_pcgi_setf(swname, sw_name, MAXSZ)
CREATE_pcgi_setf(swhome, sw_home, MAXSZ)
CREATE_pcgi_setf(swexe, sw_exe, MAXSZ)
CREATE_pcgi_setf(errlog, errlog, MAXSZ)
CREATE_pcgi_setf(host, sockhost, MAXSZ)
CREATE_pcgi_setf(modPath, modpath, MAXSZ)
CREATE_pcgi_setf(procPath, procpath, MAXSZ)
CREATE_pcgi_setf(sockPath, sockpath, MAXSZ)
CREATE_pcgi_setf(pubpath, pubpath, MAXSZ)
CREATE_pcgi_setf(pythonPath, pythonPath, MAXPATH)
CREATE_pcgi_setf(insertPath, insertPath, MAXPATH)
CREATE_pcgi_setf(root, root, MAXSZ)

static const char *pcgi_set_closeFds(cmd_parms *parms, void *r, const char *arg)
{
   if ((pcgiAssignCloseFileDescriptors((pcgiResource *)r, arg)) < 0)
      return (char *)(-1);
   return NULL;
}

static const char *pcgi_set_displayErrors(cmd_parms *parms, void *r, const char *arg)
{
   ((pcgiResource *)r)->displayErrors = pcgiTruthValue(arg[0]);
   return NULL;
}

static const char *pcgi_set_port(cmd_parms *parms, void *r, const char *arg)
{
   ((pcgiResource *)r)->sockport = atoi(arg);
   return NULL;
}

static const char *pcgi_set_env(cmd_parms *parms, void *r, const char *key, const char *value)
{
   debug2("pcgi_set_env: %s=%s\n", key, value);
   ap_table_setn(((pcgiResource *)r)->pcgi_env, key, value);
   return NULL;
}


#ifndef APACHE2
# define AP_INIT_TAKE1(directive, func, mconfig, where, help) \
    { directive, func, mconfig, where, TAKE1, help }
# define AP_INIT_TAKE2(directive, func, mconfig, where, help) \
    { directive, func, mconfig, where, TAKE2, help }
#endif

static const command_rec pcgi_cmds[] =
{
   AP_INIT_TAKE1("PCGI_SOFTWARE_NAME", pcgi_set_swname, NULL, OR_FILEINFO,
      "Software name; used to construct pid/scoket filenames"),

   AP_INIT_TAKE1("PCGI_SOFTWARE_HOME", pcgi_set_swhome, NULL, OR_FILEINFO,
      "Root directory of your PCGI installation (where var directory exists)"),

   AP_INIT_TAKE1("PCGI_SOFTWARE_EXE", pcgi_set_swexe, NULL, OR_FILEINFO,
      "Full path to your python interpreter without any option; defaults to /usr/local/bin/python1.4"),

   AP_INIT_TAKE1("PCGI_CLOSE_FDS", pcgi_set_closeFds, NULL, OR_FILEINFO,
      "M?"),

   AP_INIT_TAKE1("PCGI_DISPLAY_ERRORS", pcgi_set_displayErrors, NULL, OR_FILEINFO,
      "Boolean flag whether to hide or display errors/tracbacks"),

   AP_INIT_TAKE1("PCGI_ERROR_LOG", pcgi_set_errlog, NULL, OR_FILEINFO,
      "Full path to error log"),

   AP_INIT_TAKE1("PCGI_EXE", pcgi_set_swexe, NULL, OR_FILEINFO,
      "Alias for PCGI_SOFTWARE_EXE"),

   AP_INIT_TAKE1("PCGI_HOST", pcgi_set_host, NULL, OR_FILEINFO,
      "TCP/IP host (not used - PCGI protocol can currently work only through UNIX domain sockets)"),

   AP_INIT_TAKE1("PCGI_MODULE_PATH", pcgi_set_modPath, NULL, OR_FILEINFO,
      "Additional paths to search for publisher"),

   AP_INIT_TAKE1("PCGI_NAME", pcgi_set_swname, NULL, OR_FILEINFO,
      "Alias for PCGI_SOFTWARE_NAME"),

   AP_INIT_TAKE1("PCGI_PID_FILE", pcgi_set_procPath, NULL, OR_FILEINFO,
      "Full path to pid file; if not defined defaults to '%s/var/%s.pid' % (sw_home, sw_name)"),

   AP_INIT_TAKE1("PCGI_PORT", pcgi_set_port, NULL, OR_FILEINFO,
      "TCP/IP port (not used - PCGI protocol can currently work only through UNIX domain sockets)"),

   AP_INIT_TAKE1("PCGI_PUBLISHER", pcgi_set_pubpath, NULL, OR_FILEINFO,
      "Path to publsiher"),

   AP_INIT_TAKE1("PCGI_SOCKET_FILE", pcgi_set_sockPath, NULL, OR_FILEINFO,
      "Full path to UNIX socket file; defaults to '%s/var/%s.soc' % (sw_home, sw_name)"),

   AP_INIT_TAKE1("PCGI_PYTHONPATH", pcgi_set_pythonPath, NULL, OR_FILEINFO,
      "Additional directories for sys.path"),

   AP_INIT_TAKE1("PCGI_INSERT_PATH", pcgi_set_insertPath, NULL, OR_FILEINFO,
      "Additional paths to search for publisher"),

   AP_INIT_TAKE1("PCGI_WORKING_DIR", pcgi_set_insertPath, NULL, OR_FILEINFO,
      "Alias for PCGI_INSERT_PATH"),

   AP_INIT_TAKE1("PCGI_ROOT", pcgi_set_root, NULL, OR_FILEINFO,
      "Set root object for this location; default is real root (/)"),

   AP_INIT_TAKE2("PCGI_SETENV", pcgi_set_env, NULL, OR_FILEINFO,
      "Set environment variable for this location"),

    {NULL}
};

static int onError(char *estatus, char *emsg, pcgiResource *r)
{
    if (r != NULL)
    {
      ap_log_error(APLOG_MARK, APLOG_ERR,
#ifdef APACHE2
            0,
#endif
            r->server, "PCGI Error: [%s] (%s) %s", estatus, r->errmsg, emsg);

#ifdef UNIX
        if(r->conn != -1)    close(r->conn);
        cleanup();
#endif
#ifdef WIN32
        if (r->conn)
        {
            shutdown(r->conn, 1);
            closesocket(r->conn);
            WSACleanup();
        }
#endif
    }
    return HTTP_INTERNAL_SERVER_ERROR;
}


/*
 * Some lines of code copied from AMK's mod_pcgi.c
 */

/* Utility function used to sum up the lengths of the strings in a table */
static int env_adder(void *p, const char *key, const char *value)
{
  long *sz_env = (long *) p;
  (*sz_env) += strlen(key) + 1 + strlen(value) + 1;
  debug2("debug environment: %s=%s\n", key, value);
  return 1;
}

/* Utility function used to append key/value pairs from the environment */
static int env_pair_append(void *ptr, const char *key, const char *value)
{
  /*pcgiResource *pcgi = (pcgiResource *)p;*/
  char **pp = (char **)ptr;
  char *p = *pp;
  int l = strlen(key);
  memcpy(p, key, l); p+=l; *p='='; p++;

  l = strlen(value);
  memcpy(p, value, l); p+=l; *p=0; p++;

  *pp = p;
  return 1;
}

/* Utility function used to retrieve a single line from a buffer */
static int getsfunc_STRING(char *buf, int len, void *f)
{
  pcgiResource *pcgi = (pcgiResource *)f;
  char *p = pcgi->p_output;
  int i, sz_output = pcgi->sz_output;

  for(i=0; i<sz_output && i<(len-1); i++)
    {
      buf[i] = p[i];
      if (p[i] == '\n') {i++; break;}
    }
  buf[i] = '\0';
  pcgi->sz_output -= i;
  pcgi->p_output += i;

  return i;
}


/* That's finally the handler */
static int pcgi_handler(request_rec *req)
{
   char *p      = NULL;
   long l       = 0;
   char t[HDRLEN+1];
   struct stat statbuf;
   pcgiResource *r;

   #ifdef APACHE2
   if (strcmp(req->handler, "pcgi-handler")) {
      return DECLINED;
   }
   #endif

   debug("pcgi_handler");

   r = (pcgiResource *)ap_get_module_config(req->per_dir_config, &pcgi2_module);
   r->server = req->server; /* for onError */
   r->request = req; /* for pcgiStartProc */
   show_pcgi_resource(r, "handler");

   /* Fill req->subprocess_env */
   ap_add_common_vars(req);
   ap_add_cgi_vars(req);

   /* Add the 'Authorization' header - copied from AMK's mod_pcgi.c */
   {
     const char *auth = ap_table_get(req->headers_in, "Authorization");
     if (auth)
       ap_table_setn(req->subprocess_env, "HTTP_CGI_AUTHORIZATION", auth);
   }

   /* Change PATH_INFO if we have PCGI_ROOT */
   if (r->root[0])
   {
     int lr = strlen(r->root);
     int lr2 = lr;
     int lp;

     char *p;
     char *path_info = (char *)ap_pstrdup(req->pool, ap_table_get(req->subprocess_env, "REQUEST_URI"));

     #ifdef MOD_PCGI2_DEBUG
     if (path_info)
        debug2("Old PATH_INFO: %s%s\n", path_info, "");
     else
        debug2("Old PATH_INFO: %s%s\n", "", "");
     #endif

     if (r->root[lr-1] == PATHSEP) { lr2--; }
     if (path_info)
         lp = strlen(path_info);
     else
         lp = 0;

     if (lr2 + lp > 0) {
        if((p = ap_palloc(req->pool, lr2 + lp + 1)) == NULL)
        {
           return onError(E_500, "Error reallocating PATH_INFO", r);
        }

        strcpy(p, r->root);
        if (r->root[lr-1] == PATHSEP) p[lr2] = '\0';

        /* UGLY DIRTY HACK 1 HERE !!! */
        if (r->cut_slashes && path_info) {
           int i, j = 0;
           debug2("cut slashes: %d%s\n", r->cut_slashes, "");
           for (i = 0; i < strlen(path_info) && j <= r->cut_slashes; i++)
              if (path_info[i] == PATHSEP) j++;
           if (i < strlen(path_info))
              path_info += --i;
           else
              path_info[0] = '\0';
        }
        if (path_info) strcat(p, path_info);

        /* UGLY DIRTY HACK 2 HERE !!! */
        {
           int i;
           for (i = 0; i < strlen(p); i++)
              if (p[i] == '?') {
                 p[i] = '\0';
                 break;
              }
         }
         ap_unescape_url(p);

        debug2("New PATH_INFO: %s%s\n", p, "");
        ap_table_setn(req->subprocess_env, "PATH_INFO", p);
     }
   }

   if (r->location && (strcmp(r->location, "/") != 0)) { /* Set SCRIPT_NAME for non-root location */
     debug2("New SCRIPT_NAME: %s%s\n", r->location, "");
     ap_table_setn(req->subprocess_env, "SCRIPT_NAME", r->location);
   } else {
     debug2("New SCRIPT_NAME: %s%s\n", "(empty)", "");
     ap_table_setn(req->subprocess_env, "SCRIPT_NAME", "");
   }

   /* Copy r->pcgi_env to current req->subprocess_env - this will copy all env vars set by PCGI_SetEnv */
   ap_table_do(copy_pcgi_env, req->subprocess_env, r->pcgi_env, NULL);

    /*
    //  If the location of the publisher was not specified, try to
    //  locate it.
    */
    if (!r->pubpath[0])
    {
        pcgiAssignPublisher(r);
    }

    if (!r->pubpath[0])
    {
        strcpy(r->errmsg, "unable to determine the publisher location");
        return onError(ERR101_FAILURE_DURING_START, r->errmsg, r);
    }
    else
    {
        if (stat(r->pubpath, &statbuf) == -1)
        {
            sprintf(r->errmsg, "missing publisher: %s", r->pubpath);
            return onError(ERR101_FAILURE_DURING_START, r->errmsg, r);
        }
        else
        {
            if (!(statbuf.st_mode & S_IREAD))
            {
                sprintf(r->errmsg, "publisher read error: %s", r->pubpath);
                return onError(ERR101_FAILURE_DURING_START, r->errmsg, r);
            }
        }
    }

    /*
    //  Assign defaults, where necessary, for backward compatibility.
    */
    if (r->sw_name[0] && r->sw_home[0])
    {
        if (!r->sw_exe[0])
        {
            strcpy(r->sw_exe,"/usr/local/bin/python1.4");
        }
        if (!r->procpath[0])
        {
            sprintf(r->procpath,"%s/var/%s.pid", r->sw_home,r->sw_name);
        }
        if (!r->sockpath[0])
        {
            sprintf(r->sockpath,"%s/var/%s.soc", r->sw_home,r->sw_name);
        }
    }

    /*
    // Other than r->pubpath, the other required attributes to complete
    // the process are: r->procpath & r->sockpath.
    */
    if (!r->sockpath[0])
    {
        strcpy(r->errmsg, "missing parameter: PCGI_SOCKET_FILE");
        return onError(ERR101_FAILURE_DURING_START, r->errmsg, r);
    }
    if (!r->procpath[0])
    {
        strcpy(r->errmsg, "missing parameter: PCGI_PID_FILE");
        return onError(ERR101_FAILURE_DURING_START, r->errmsg, r);
    }

    /*
    // Attempt to connect
    */
    debug("Connecting...");
    if ((r->conn = pcgiConnect(r)) < 0)
    {
        if(pcgiVerifyProc(r) < 0)
        {
            if(pcgiStartProc(r) < 0)
            {
                if (!r->errmsg[0])
                    strcpy(r->errmsg, ERR101_FAILURE_DURING_START);
                return onError(E_500, "Failed to start resource", r);
            }
            if ((r->conn=pcgiConnect(r)) < 0)
            {
                if (!r->errmsg[0])
                    strcpy(r->errmsg, ERR102_FAILURE_DURING_CONNECT);
                return onError(E_503, strerror(errno), r);
            }
        }
        else
        {
            if (!r->errmsg[0])
                strcpy(r->errmsg, ERR103_UNABLE_VERIFY_RUNNING);
            return onError(E_503, "pcgiVerifyProc failed", r);
        }
    }

    /*
    // Send environment - again, copied from AMK's mod_pcgi.c
    */
    debug("Sending environment...");
   ap_table_do(env_adder, &r->sz_env, req->subprocess_env, NULL);
   r->sz_env *= sizeof(char);
   if((r->p_env=ap_palloc(req->pool, r->sz_env + HDRLEN*sizeof(char))) == NULL)
     {
       return onError(E_500, "Error allocating env", r);
     }

    p=r->p_env;
    sprintf(p, HDRFMT, r->sz_env);

   r->sz_env += HDRLEN*sizeof(char);
   if (p[0] != '0')
     {
       if (!r->errmsg[0])
         strcpy(r->errmsg, ERR123_BAD_ENV_HEADER);
       return onError(E_500, "Error creating header for environment", r);
     }

   p+=HDRLEN;
   ap_table_do(env_pair_append, &p, req->subprocess_env, NULL);

#ifdef WIN32
      if (pcgiWriteSocket(r->conn,r->p_env,r->sz_env) != r->sz_env)
#endif
#ifdef UNIX
        if (pcgiWrite(r->conn,r->p_env,r->sz_env) != r->sz_env)
#endif
          {
            if (!r->errmsg[0])
              strcpy(r->errmsg, ERR104_ENVIRONMENT_SEND);
            return onError(E_500, "Error sending env", r);
          }

    /*
    // Send stdin
    */
    p = (char *)ap_table_get(req->subprocess_env, "CONTENT_LENGTH");
    #ifdef MOD_PCGI2_DEBUG
    if (p)
       debug2("Sending stdin %s bytes...%s\n", p, "");
    #endif
    if((p != NULL) && ((r->sz_input=atol(p)) > 0))
    {
        if((r->p_input=ap_palloc(req->pool,sizeof(char) * r->sz_input)+(sizeof(char)*HDRLEN))==NULL)
        {
            return onError(E_500, "Error allocating stdin", r);
        }
        p=r->p_input;
        sprintf(p, HDRFMT, r->sz_input);
        r->sz_input += HDRLEN*sizeof(char);

        if (p[0] != '0')
        {
            if (!r->errmsg[0])
                strcpy(r->errmsg, ERR124_BAD_STDIN_HEADER);
            return onError(E_500, "Error allocating stdin", r);
        }

        p+=HDRLEN;

        /* Read from stdin, copied from an example on page 547 of the
           O'Reilly book on Apache modules */
        {
          int rc;
          if ( (rc=ap_setup_client_block(req, REQUEST_CHUNKED_ERROR)) != OK) {
            return rc;
          }

          if (ap_should_client_block(req)) {
            char buffer[HUGE_STRING_LEN];
            int rsize, len_read, rpos=0;
            long length = req->remaining;

            #ifndef APACHE2
            ap_hard_timeout("pcgi_read_stdin", req);
            #endif
            while ( (len_read = ap_get_client_block(req, buffer, sizeof(buffer))) > 0 ) {
              #ifndef APACHE2
              ap_reset_timeout(req);
              #endif
              if ((rpos + len_read) > length) rsize = length-rpos;
              else rsize = len_read;
              memcpy(p+rpos, buffer, rsize);
              rpos += rsize;
            }
            #ifndef APACHE2
            ap_kill_timeout(req);
            #endif
          }
        }
    }
    else
    {
        /* No CONTENT_LENGTH environment variable */
        r->sz_input = HDRLEN*sizeof(char);
        r->p_input = ap_palloc(req->pool, r->sz_input);
        sprintf(r->p_input, HDRFMT, (long)0);
    }

    /*
    // Send environment and stdin
    */
    debug("Sending to socket...");
#ifdef WIN32
      if (pcgiWriteSocket(r->conn,r->p_input,r->sz_input) != r->sz_input)
#endif
#ifdef UNIX
        if (pcgiWrite(r->conn,r->p_input,r->sz_input) != r->sz_input)
#endif
          {
            if (!r->errmsg[0])
              strcpy(r->errmsg, ERR105_STDIN_SEND);
            return onError(E_500, "Error sending stdin", r);
          }

#ifdef WIN32
shutdown(r->conn, 1); /* shutdown the socket so we can receive */
#endif

    /*
    // Receive stdout and stderr
    */
    debug("Receiving from socket...");
    memset(t, 0, sizeof(t));
#ifdef WIN32
    if (!pcgiReadSocket(r->conn, t, HDRLEN))
#endif
#ifdef UNIX
    if (!pcgiRead(r->conn, t, HDRLEN))
#endif
    {
        if (!r->errmsg[0])
            strcpy(r->errmsg, ERR106_STDOUT_READ_HEADER);
        return onError(E_503, strerror(errno), r);
    }

    if (strlen(t) <= 0)
    {
        if (!r->errmsg[0])
            sprintf(r->errmsg, "%s (%d)", ERR107_BAD_STDOUT_STRLEN, strlen(t));
       return onError(E_503, strerror(errno), r);
    }

    if (t[0] != '0')
    {
      /* XXX - Later: process this as out-of-bound stdin data */
        sprintf(r->errmsg, "t[0] = %d", (int) t[0]);
        return onError(E_500, "unexpected out-of-bound data in stdin", r);
    }
    else
    {
        /*ap_log_error(APLOG_MARK, APLOG_ERR,
#ifdef APACHE2
            0,
#endif
         * req->server, "[debug] received string `%s'\n", t);*/
        r->sz_output=atol(t);
        l=(sizeof(char) * r->sz_output) + sizeof(char);
        r->p_output=(char *)ap_palloc(req->pool, l);
        memset(r->p_output,0,l);
#ifdef WIN32
        if (pcgiReadSocket(r->conn,r->p_output,r->sz_output) != r->sz_output)
#endif
#ifdef UNIX
        if (pcgiRead(r->conn,r->p_output,r->sz_output) != r->sz_output)
#endif
        {
            if (!r->errmsg[0])
                strcpy(r->errmsg, ERR108_STDOUT_READ_BODY);
            return onError(E_500, "Error receiving stdout", r);
        }
    }

    t[0]='\0';
#ifdef WIN32
    if (!pcgiReadSocket(r->conn, t, HDRLEN))
#endif
#ifdef UNIX
    if (!pcgiRead(r->conn, t, HDRLEN))
#endif
    {
        if (!r->errmsg[0])
            strcpy(r->errmsg, ERR109_STDERR_READ_HEADER);
        return onError(E_500, "Error receiving stderr size", r);
    }
    if (strlen(t) <= 0)
    {
        if (!r->errmsg[0])
            sprintf(r->errmsg, "%s (%d)", ERR110_BAD_STDERR_STRLEN, strlen(t));
        return onError(E_500, "Error receiving stderr size", r);
    }

    if (t[0] != '0')
    {
      /* XXX - Later: process this as out-of-bound stderr data */
        sprintf(r->errmsg, "t[0] = %d", (int) t[0]);
        return onError(E_500, "unexpected out-of-bound data in stderr", r);
    }
    else
    {
        r->sz_error=atol(t);
        if (r->sz_error > 0)
        {
            l=(sizeof(char) * r->sz_error) + sizeof(char);
            r->p_error=(char *)ap_palloc(req->pool, l);
            memset(r->p_error,0,l);
#ifdef WIN32
            if (pcgiReadSocket(r->conn, r->p_error, r->sz_error) != r->sz_error)
#endif
#ifdef UNIX
            if (pcgiRead(r->conn, r->p_error, r->sz_error) != r->sz_error)
#endif
            {
                if (!r->errmsg[0])
                    strcpy(r->errmsg, ERR111_STDERR_READ_BODY);
                return onError(E_500, "Error receiving stderr", r);
            }
        }
    }
#ifdef UNIX
    close(r->conn);
#endif
#ifdef WIN32
    closesocket(r->conn);
#endif

    /*
    // Return stdout and stderr to server
    */
    if (r->sz_output != 0)
    {
      int sent;
      char buffer[HUGE_STRING_LEN+1];
      int scan_err_res;

      scan_err_res = ap_scan_script_header_err_core(req, buffer, getsfunc_STRING, r);
      if (scan_err_res != OK && scan_err_res != HTTP_NOT_MODIFIED) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
#ifdef APACHE2
            0,
#endif
           req->server, "error while scanning PCGI script headers");
           return HTTP_INTERNAL_SERVER_ERROR;
      }

      ap_send_http_header(req);

      if (!req->header_only) {
        /* Send the rest of the HTTP body */
        sent = ap_rwrite(r->p_output, r->sz_output, req);
        if (sent != r->sz_output) {
          ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
#ifdef APACHE2
            0,
#endif
                       req->server, "error during ap_rwrite of stdout");
        }
      }
    }

    if (r->sz_error != 0)
    {
      int sent = ap_rwrite(r->p_error, r->sz_error, req);
      if (sent != r->sz_error) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR,
#ifdef APACHE2
            0,
#endif
                     req->server, "error during ap_rwrite of stderr");

      }
    }

   cleanup();

   return OK;
}


#ifdef UNIX
void pcgi_child(pcgiResource *r)
{
  char **env;
  char *p = NULL;
  request_rec *req = r->request;
  table *e = req->subprocess_env;

  if ((p=strrchr(r->sw_exe, PATHSEP))==NULL)
  {
      p = r->sw_exe;
  }
  else
  {
      p++;
  }

  ap_add_common_vars(req);
  ap_add_cgi_vars(req);

  ap_table_setn(e, "PCGI_ERROR_LOG", r->errlog);
  ap_table_setn(e, "PCGI_HOST", r->sockhost);
  /*  ap_table_setn(e, "PCGI_PORT", r->); */
  ap_table_setn(e, "PCGI_MODULE_PATH", r->modpath);
  ap_table_setn(e, "PCGI_NAME", r->sw_name);
  ap_table_setn(e, "PCGI_PID_FILE", r->procpath);
  ap_table_setn(e, "PCGI_SOCKET_FILE", r->sockpath);

  env = ap_create_environment(req->pool, req->subprocess_env);

  ap_error_log2stderr(req->server);
  req->filename = r->sw_exe;
  req->args = r->pubpath;

  debug2("Starting subproces `%s %s'\n", r->sw_exe, r->pubpath);
  debug2("   env: %s=%s\n", "PCGI_ERROR_LOG", r->errlog);
  debug2("   env: %s=%s\n", "PCGI_HOST", r->sockhost);
  debug2("   env: %s=%s\n", "PCGI_MODULE_PATH", r->modpath);
  debug2("   env: %s=%s\n", "PCGI_NAME", r->sw_name);
  debug2("   env: %s=%s\n", "PCGI_PID_FILE", r->procpath);
  debug2("   env: %s=%s\n", "PCGI_SOCKET_FILE", r->sockpath);

  ap_cleanup_for_exec();
  execle(r->sw_exe, p, r->pubpath, (char *)0, env);

  ap_log_error(APLOG_MARK, APLOG_ERR,
#ifdef APACHE2
            0,
#endif
        NULL, "exec of %s failed", req->filename);
  exit(-1);
}

/*
// pcgiStartProc: manages starting a pcgi resource.
*/
int pcgiStartProc(pcgiResource *r)
{
    pid_t pid;
    UNION_SEMUN arg;
    arg.val=0;

    /* Make sure another wrapper isn't already doing a restart */
    if ((r->lock=semget(101, 1, 0700 | IPC_CREAT | IPC_EXCL)) == -1)
    {
        if (errno == EACCES)
            strcpy(r->errmsg, ERR117_LOCK_ERROR_EACCES);
        else if (errno == EEXIST)
            strcpy(r->errmsg, ERR118_LOCK_ERROR_EEXIST);
        else if (errno == EINVAL)
            strcpy(r->errmsg, ERR119_LOCK_ERROR_EINVAL);
        else if (errno == ENOENT)
            strcpy(r->errmsg, ERR120_LOCK_ERROR_ENOENT);
        else if (errno == ENOSPC)
            strcpy(r->errmsg, ERR121_LOCK_ERROR_ENOSPC);
        else
            sprintf(r->errmsg, "%s, %d", ERR122_LOCK_ERROR_OTHER, errno);
        return(-1);
    }

    /* XXX the semaphore won't get cleaned up if we abort */

    /* If the old socket file exists and we have write permission,
       attempt to remove it. */
    if (r->sockport == 0 && access(r->sockpath, W_OK) == 0)
    {
        unlink(r->sockpath);
    }

    /* Fork a child which forks a child -- this is so that
       init will inherit the grandchild (so it won't die) */
    if ((pid = fork()) < 0)
    {
        semctl(r->lock, 1, IPC_RMID, arg);
        return(-1);
    }
    else if (pid == 0)
    {
        if ((pid = fork()) < 0)
        {
            return(-1);
        }
        else if (pid > 0)
        {   /* Let child know we're going away so it can start */
            kill(pid, SIGUSR1);
            exit(0);
        }

        setsid();
        chdir("/");

        /* Platform oddities... */
        if (CloseFileDescriptors)
        {
            close(STDIN_FILENO);
            close(STDOUT_FILENO);
            close(STDERR_FILENO);
        }


        pcgi_child(r);
    }

    /* Wait for the first child to finish */
    if (waitpid(pid, NULL, 0) < 0)
    {   semctl(r->lock, 1, IPC_RMID, arg);
        return(-1);
    }

    /*
    // Release restart lock!
    */
    semctl(r->lock, 1, IPC_RMID, arg);
    return 0;
}
#endif


#ifdef APACHE2
static void register_hooks(apr_pool_t *p)
{
    ap_hook_post_config(init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_handler(pcgi_handler,NULL,NULL,APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA pcgi2_module =
{
    STANDARD20_MODULE_STUFF,
    create_pcgi_dir_config,         /* create per-directory config structure */
    merge_pcgi_dir_config,         /* merge per-directory config structures */
    NULL,			/* create per-server config structure */
    NULL,			/* merge per-server config structures */
    pcgi_cmds,    /* command apr_table_t */
    register_hooks		/* register hooks */
};

#else

static const handler_rec pcgi_handlers[] =
{
   {"pcgi-handler", pcgi_handler},
   {"application/pcgi", pcgi_handler},
   {NULL}
};

module MODULE_VAR_EXPORT pcgi2_module =
{
    STANDARD_MODULE_STUFF,
    init,         /* initializer */
    create_pcgi_dir_config,         /* create per-directory config structure */
    merge_pcgi_dir_config,         /* merge per-directory config structures */
    NULL,         /* create per-server config structure */
    NULL,         /* merge per-server config structures */
    pcgi_cmds,    /* command table */
    pcgi_handlers,/* handlers */
    NULL,         /* translate_handler */
    NULL,         /* check_user_id */
    NULL,         /* check auth */
    NULL,         /* check access */
    NULL,         /* type_checker */
    NULL,         /* pre-run fixups */
    NULL,         /* logger */
#if MODULE_MAGIC_NUMBER >= 19970103
    NULL,         /* header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
    NULL,         /* child_init */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
    NULL,         /* child_exit */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
    NULL          /* post read-request */
#endif
};
#endif
