
/* Copyright (c) 1995, 1996, 1997, 1998, 1999, 2001, 2002, 2003, 2004 by Arkkra Enterprises */
/* All rights reserved */

/*
There are 3 classes of location variables:

	A. Those associated with a specific staff of a specific score.
	This includes those for GRPSYLs and NOTES.

	B. Those associated with a specific score. These are for bars.

	C. Those associated with the current page. These are the builtin
	variables such as _win and _page, or absolute coordinates.

Type B variables, associated with bars, are extra strange. The y of a bar
in not particularly useful. If the bar line happens to wind up at the end
of a score, special consideration applies, because the right side of the
bar is effectively at the beginning of the next score after the clefsig.
So here are the rules:

1. If a bar does not occur at the end of a score, the location variable
associated with it, if any, is handled normally.

2. If a bar does fall on the end of a score, if the x coordinate derived from
a location variable associated with that bar in
an INPCOORD, after offsetting, comes out to be either left of the x of
the bar, or equal to the x of the bar, it will be handled normally.

3. If a bar falls on the end of a score, and
if the x coordinate derived from an INPCOORD, after offsetting, comes out
to be right of the x of the bar,
the x coordinate will be recalculated using the pseudo
bar at the beginning of the following score,
and if the y coordinate of the same INPCOORD is also associated
with the same bar coordinate, it will also be recalculated from the pseudo bar.

4. If rule 3 would normally apply, but there is no following score, rule 3
will be ignored, and the coordinate will be used as is.

A PRHEAD contains only a single location variable, so it should always
be taken just as is.

Lines and curves are more exciting, since they can have multiple
coordinates, and thus may need to be split into 2 or more pieces
across scores and/or pages. Lines are a degenerate case of curves,
so if we can deal with curves, we've got it made.

For drawing a specific curve, here are the rules:

1. If any coordinate is associated with a staff that is invisible, the entire
curve will be ignored.

2. Type C variables are always used as is, never themselves causing splitting.
Taking any adjacent pair of points in a curve, if either of them is of type C,
the line segment between those 2 points will not be split.

3. If all variables of type A and B are on the same score of the same page,
then the curve can be printed as is, with no splitting needed.

4. If the x and y components of a single INPCOORD are associated with
different scores, this will be an error condition.

5. If the x and y components of a single INPCOORD are associated with
different staffs, but the same score, the point will be treated as if
it were associated with the staff associated with the y coordinate.

6. If 2 adjacent points of a curve are associated with different
scores, the line segment must be split. The number of segments that will
need to be generated will be equal to the number of FEEDs between
the coordinates plus one. 

7. Splitting will only be done to forward scores. In other words, if the
coordinates of a curve would require splitting part of the curve onto a
preceeding score, that will be an error. This is to keep things simpler,
since I can't think of any times this restriction would cause a problem.

8. If a segment needs to be split, the first piece will extend in the
x direction from the first point to 0.1 inch left of the right edge of the
score associated with the first point.
However, if the starting x is already at the right edge of the score, a line of length 0.1 inches will be drawn instead.
The last piece of the split line segment will extend in
the x direction from the pseudo bar of the clefsig.

8a. If there are additional scores
between the one associated with the beginning point and the one associated
with the endpoint, for each intervening score a line will be drawn with
its x coordinates from the pseudo bar to the right margin.

9. To calculate the y coordinates of each piece of a split line segment,
there are several cases. First the easy case, where the y coordinates of
the beginning and ending point are both associated with the same staff.
Conceptionally, the scores are lined up on a single line without score
feeds. The slope of the line is then calculated. The y coordinates of
the derived points are then calculated using this slope. Thus, for example,
if the ending y coordinate would be A inches from of the beginning y coordinate
in the x direction (if they were on the same score),
and the line segment is split into 2 segments, with the first having
a length in the x direction of B and the second having a length in the x
direction of C, the y coordinate of the end of the first segment would be
y[begin] + (A/(B+C)) * B, and the y coordinate of the beginning of the second
piece would be y[end] - (A/(B+C)) * C.

10. If the y coordinates of the 2 points are associated with different staffs.
the slope is calculated based on the distance of the endpoints from their
respective staffs. Then for each segment, the slope and endpoints are adjusted
based on the ratio of the distance between the two staffs on the current score
relative to the widest distance.

11. For purposes of determining y coordinates, the y and n values of a bar
are considered to be associated with the top visible score, and the s value
is considered to be associated with the bottom visible score.
Then rules 9 and 10 above are applied as for with type A coordinates.
 */

#include "defines.h"
#include "structs.h"
#include "globals.h"

static int Total_pages = 1;		/* how many pages of output */

/* if lines must be added for intervening scores, save info about them */
struct SEGINFO {
	struct MAINLL *mll_p;	/* FEED where line segment must go */
	double xlength;		/* x length prior to current score */
	struct SEGINFO *next;	/* linked list */
};
struct SEGINFO *Seginfo_p;

static void gather_coord_info P((void));
static void save_coord_info P((struct COORD_INFO *coord_info_p,
		int coordtype, int page, int score, int staff,
		struct MAINLL *mll_feed_p, int vis));
static void coord_staff P((struct STAFF *staff_p, int page,
		int score, struct MAINLL *mll_feed_p));
static void split_lines_and_curves P((void));
static void chkline P((struct MAINLL *mll_p));
static void coordcheck P((struct COORD_INFO *x_info_p,
		struct COORD_INFO *y_info_p, char *fname, int lineno));
static void add_segment P((struct SEGINFO *seginfo_p, double slope,
		double y_offset, int staffno1, int linetype));
static double find_effXlength P((double seg1xlen, double seg2xlen,
		struct COORD_INFO *x1info_p, struct COORD_INFO *x2info_p,
		int save_feed_info));
static void svfeed P((struct MAINLL *mll_feed_p, double xlength));
static int eff_staff P((struct COORD_INFO *yinfo_p));
static double getYstaff P((struct MAINLL *mll_p, int staffno));
static void chkcurve P((struct MAINLL *mll_p));
static int bulgedir P((struct CURVE *curve_p, int index, char *inputfile,
		int inputlineno));
static int cmpcoords P((struct CURVE *curve_p, int p1, int p2));
static int abs2rel P((int vtype));
static void add_crv_seg P((struct SEGINFO *seginfo_p, double slope,
		double y_offset, int staffno1, int curvetype, int bulge,
		char *filename, int lineno));
static int is_invis P((struct COORD_INFO *cinfo_p));
static int is_builtin P((struct COORD_INFO *cinfo_p));
static void move2correct_page P((void));
static void move_it P((struct MAINLL *m_p, struct MAINLL *im_p, int page));
static void move2pseudo P((void));
static void do_pseudo P((struct INPCOORD *inpc_p, struct MAINLL *mll_p));
static void fix_inpcoords P((struct MAINLL *mll_p));
static void adj_coord P((struct INPCOORD *coord_p, struct MAINLL *mll_p,
		struct INPCOORD *prev_coord_p));
static void calc_bulge P((struct CURVE *curve_p, char *fname, int lineno,
		int is_split, struct MAINLL *mll_p));


/* during parse phase, a table of coordinates associated with location
 * variables was built. After all the positioning has been done, we need
 * to go through the main list and stuff off of it checking each coordinate.
 * If the coordinate is pointed to by something else, we'll need to
 * save some info about it. Then we have to go through the main list
 * again and for each line and curve, see whether it needs to be split
 * into pieces. If so, add LINE or CURVE structs at appropriate places.
 */


void
fix_locvars()

{
	/* first get info about all coordinates with loc variables */
	gather_coord_info();

	/* move things to pseudo-bar if necessary */
	move2pseudo();

	/* split any lines and curves that need to be split */
	split_lines_and_curves();

	/* move anything that is on the wrong page */
	move2correct_page();
}


/* go through everything looking for coordinates. For each one found, if
 * there is a location tag pointing at it, save info about what the coord
 * is associated with (bar, note, or group), what page, score and
 * staff it's on, etc. */

static void
gather_coord_info()

{
	struct MAINLL *mll_p;		/* to walk through list */
	short page = 1;			/* which page we're on */
	short score = 0;		/* which staff on current page */
	struct MAINLL *mll_feed_p;	/* FEED info for current score */
	struct COORD_INFO *coord_info_p;
	struct COORD_INFO *last_bar_coord_info_p;	/* info about the
					 * most recent bar line, in case we
					 * need to attach information about
					 * the pseudo bar at the beginning
					 * of the following score */


	debug(32, "gather_coord_info");

	initstructs();
	last_bar_coord_info_p = (struct COORD_INFO *) 0;
	/* We know that because of how the main list is set up, we will never
	 * actually access mll_feed_p without setting it first, but compilers
	 * aren't smart enough to know that, and some picky compilers warn
	 * that mll_feed_p could be used without being set, so shut them up.
	 */
	mll_feed_p = (struct MAINLL *) 0;

	for (mll_p = Mainllhc_p; mll_p != (struct MAINLL *) 0;
						mll_p = mll_p->next) {
		switch (mll_p->str) {

		case S_FEED:
			/* keep track of which page and score we're on */
			if (mll_p->u.feed_p->pagefeed == YES) {
				page++;
				score = 1;
				Total_pages++;
			}
			else {
				score++;
			}
			if (IS_CLEFSIG_FEED(mll_p)) {
				mll_feed_p = mll_p;
			}
			break;

		case S_BAR:
			/* if bar is pointed to, save info about it */
			if ((coord_info_p = find_coord(mll_p->u.bar_p->c))
						!= (struct COORD_INFO *) 0) {

				save_coord_info(coord_info_p, CT_BAR,
							page, score, 0,
							mll_feed_p, YES);
				last_bar_coord_info_p = coord_info_p;
			}

			else {
				/* no reference to this bar, so no need to
				 * attach pseudo bar info */
				last_bar_coord_info_p = (struct COORD_INFO *) 0;
			}
			break;

		case S_CLEFSIG:
			if (mll_p->u.clefsig_p->bar_p != (struct BAR *) 0) {
				if (last_bar_coord_info_p != (struct COORD_INFO *) 0) {
					/* point bar at end of previous score
					 * to the  pseudo bar on this score */
					last_bar_coord_info_p->pseudo_bar_p
						= mll_p->u.clefsig_p->bar_p;
				}

				/* always save info, because a split curve may
				 * need to refer to it */
				add_coord(mll_p->u.clefsig_p->bar_p->c, CT_BAR);
				coord_info_p = find_coord(mll_p->u.clefsig_p->bar_p->c);
				save_coord_info(coord_info_p, CT_BAR,
					page, score, 0, mll_feed_p, YES);
			}
			break;

		case S_STAFF:
			/* will have to get info for both GRPSYLs and NOTES. */
			coord_staff(mll_p->u.staff_p, page, score, mll_feed_p);
			break;

		case S_SSV:
			/* keep track of VISIBLE status */
			asgnssv(mll_p->u.ssv_p);
			break;

		default:
			/* nothing else is of interest at this point */
			break;
		}
	}
}


/* fill in the COORD_INFO table with information about a coordinate.  */

static void
save_coord_info(coord_info_p, coordtype, page, score, staff, mll_feed_p, vis)

struct COORD_INFO *coord_info_p;/* where to add -- assumed
				 * to be non-NULL */
int coordtype;			/* CT_BAR, CT_NOTE, etc */
int page;
int score;
int staff;
struct MAINLL *mll_feed_p;	/* MAINLL containing FEED
				 * associated with score */
int vis;			/* YES if visible */

{
	if (coord_info_p == (struct COORD_INFO *) 0) {
		pfatal("invalid coordinate information");
	}

	/* make sure this phase matches parse phase */
	if ((coord_info_p->flags & coordtype) == 0) {
		pfatal("coordinate type mismatch");
	}

	/* save relevant info */
	coord_info_p->page = (short) page;
	coord_info_p->scorenum = (short) score;
	coord_info_p->staffno = (short) staff;
	coord_info_p->mll_feed_p = mll_feed_p;
	if (vis == NO) {
		coord_info_p->flags |= CT_INVISIBLE;
	}
}


/* given a STAFF struct, save relevant info about all the GRPSYL
 * and NOTE coordinates */

static void
coord_staff(staff_p, page, score, mll_feed_p)

struct STAFF *staff_p;		/* get info from here */
int page;
int score;
struct MAINLL *mll_feed_p;	/* FEED associated with this score */

{
	struct GRPSYL *gs_p;
	struct COORD_INFO *coord_info_p;
	int vis;		/* YES if staff is visible */
	register int n;		/* to walk through NOTE lists */
	register int v;		/* walk through voices/verses */


	/* do for each voice */
	for (v = 0; v < MAXVOICES; v++) {

		vis = vvpath(staff_p->staffno, v + 1, VISIBLE)->visible;
		/* for each GRPSYL in the list */
		for (gs_p = staff_p->groups_p[v]; gs_p != (struct GRPSYL *) 0;
				gs_p = gs_p->next) {

			/* check its coordinate */
			if ((coord_info_p = find_coord(gs_p->c))
					!= (struct COORD_INFO *) 0) {
				save_coord_info(coord_info_p, CT_GRPSYL,
						page, score, gs_p->staffno,
						mll_feed_p, vis);
			}

			/* if has notes, check each note coordinate */
			for (n = 0; n < gs_p->nnotes; n++) {

				if ((coord_info_p = find_coord(gs_p->notelist[n].c))
						!= (struct COORD_INFO *) 0) {

					save_coord_info(coord_info_p, CT_NOTE,
							page, score,
							gs_p->staffno,
							mll_feed_p, vis);
				}
			}
		}
	}
}


/* go down main list. For any lines and curves, see if they need to be
 * split */

static void
split_lines_and_curves()

{
	struct MAINLL *mll_p;		/* walk through main list */


	debug(16, "split_lines_and_curves");

	initstructs();
	for (mll_p = Mainllhc_p; mll_p != (struct MAINLL *) 0;
						mll_p = mll_p->next) {

		switch(mll_p->str) {

		case S_LINE:
			fix_inpcoords(mll_p);
			chkline(mll_p);
			break;

		case S_CURVE:
			fix_inpcoords(mll_p);
			chkcurve(mll_p);
			break;

		case S_SSV:
			asgnssv(mll_p->u.ssv_p);
			break;

		default:
			/* ignore everything else */
			break;
		}
	}
}


/* check whether a LINE needs to be split. If so, split it */

static void
chkline(mll_p)

struct MAINLL *mll_p;	/* points to LINE */

{
	struct COORD_INFO *x1info_p, *y1info_p;	/* info about coordinates
				* referenced for the beginning of the line */
	struct COORD_INFO *x2info_p, *y2info_p;	/* same for end of line */
	struct LINE *line_p;			/* the line being processed */
	struct MAINLL *new_mll_p;		/* new main list struct to add
						 * if line has to be split */
	struct LINE *end_line_p;		/* new LINE struct to hang off
						 * of new_mll_p if the line
						 * has to be split */
	double offset;
	struct MAINLL *mll_clefsig_p;		/* clefsig before a continued
						 * line segment */
	struct MAINLL *m_p;			/* for finding BAR */
	char *fname;				/* file name for messages */
	int lineno;				/* line # for messages */
	double seg1xlen, seg2xlen;		/* lengths of split segments */
	double effective_x_len;			/* effective horizontal
						 * distance of line, adding
						 * the split segments */
	double slope;				/* of effective line */
	int p1staff, p2staff;			/* effective staff associated
						  * with y coord of line ends */
	struct SEGINFO *seg_p;			/* walk through segment list */
	struct SEGINFO *to_free_p;		/* which is to be freed */


	/* if we added this line internally, it's already split, so no
	 * more to check on it */
	if (mll_p->inputlineno <= 0) {
		return;
	}

	Seginfo_p = (struct SEGINFO *) 0;

	/* get relevant info about each referenced coordinate */
	line_p = mll_p->u.line_p;
	x1info_p = find_coord(line_p->start.hor_p);
	y1info_p = find_coord(line_p->start.vert_p);
	x2info_p = find_coord(line_p->end.hor_p);
	y2info_p = find_coord(line_p->end.vert_p);

	if (x1info_p == (struct COORD_INFO *) 0
				|| y1info_p == (struct COORD_INFO *) 0
				|| x2info_p == (struct COORD_INFO *) 0
				|| y2info_p == (struct COORD_INFO *) 0) {
		/* must be an absolute coordinate */
		return;
	}

	fname = mll_p->inputfile;
	lineno = mll_p->inputlineno;

	/* rule 1: if any invisible, ignore */
	if ( is_invis(x1info_p) || is_invis(y1info_p) || is_invis(x2info_p)
						|| is_invis(y2info_p) ) {
		/* not to be printed, so remove from main list */
		unlinkMAINLL(mll_p);
		/* don't free the space, since this way the function that
		 * called us can still do mll_p->next to get to the next
		 * item in list.  The space will never get reclaimed, but
		 * this case will be hit so rarely anyway, who cares
		 * about a few dead bytes? */
		return;
	}

	/* rule 2:
	 * if there are any references to a builtin variable (like _cur)
	 * then there will be no split */
	if ( is_builtin(x1info_p) || is_builtin(y1info_p)
			|| is_builtin(x2info_p) || is_builtin(y2info_p) ) {
		return;
	}

	/* rule 3:
	 * if all references are on same page and score, no split needed */
	if ( (x1info_p->scorenum == y1info_p->scorenum)
			&& (x1info_p->scorenum == x2info_p->scorenum)
			&& (x1info_p->scorenum == y2info_p->scorenum)
			&& (x1info_p->page == y1info_p->page)
			&& (x1info_p->page == x2info_p->page)
			&& (x1info_p->page == y2info_p->page) ) {
		return;
	}

	/* rule 4:
	 * If x and y of a single INPCOORD are associated with different
	 * scores, we give up. (coordcheck ufatals if x and y are on
	 * different scores.)
	 */
	coordcheck(x1info_p, y1info_p, fname, lineno);
	coordcheck(x2info_p, y2info_p, fname, lineno);

	/* rule 5:
	 * if x and y are associated with different staffs,
	 * make effective staff that of the y coordinate. */
	/* figure out which staff the beginning is associated with */
	p1staff = eff_staff(y1info_p);

	/* figure out which staff end of line is associated with */
	p2staff = eff_staff(y2info_p);

	/* rule 6:
	 * Arrrgh! The line will have to be split. No specific code to do
	 * for this rule...the mere fact that we are here indicates rule 6
	 * has been satisfied */

	/* rule 7:
	 *  Make sure x2 is not behind x1. */
	if (x2info_p->page < x1info_p->page ||
			(x2info_p->page == x1info_p->page &&
			x2info_p->scorenum < x1info_p->scorenum)) {
		l_ufatal(fname, lineno,
				"can't draw line backwards to previous score");
	}

	/* So... there will have to be at least 1 more LINE struct
	 * (more if the end is more than 1 score away) */
	new_mll_p = newMAINLLstruct(S_LINE, -1);
	new_mll_p->inputfile = mll_p->inputfile;
	end_line_p = new_mll_p->u.line_p;
	end_line_p->linetype = (short) line_p->linetype;

	/* the new LINE will have its end equal to what the original LINE had */
	end_line_p->end = line_p->end;

	/* Start out with end of first segment the same as its
	 * start. Later, we'll add appropriate x and y offsets. */
	line_p->end = line_p->start;

	/* start out with last segment's beginning the same as its end.
	 * In a bit, we'll adjust the x and y appropriately. */
	end_line_p->start = end_line_p->end;

	/* rule 8:
	 * finding the x's of the new pieces isn't too bad... */

	/* the end x of the first segment is just like the beginning x,
	 * but offset to the east far enough to
	 * reach the end of the score. */
	seg1xlen = PGWIDTH - eff_rightmargin(mll_p) - inpc_x( &(line_p->start),
					fname, lineno );

	/* handle bizarre case of beginning being too far right to deal
	 * with properly */
	if (seg1xlen < 0.1) {
		seg1xlen = 0.1;
	}
	/* convert inches to stepsizes, which is how offset are stored */
	line_p->end.hsteps += seg1xlen / STEPSIZE;

	/* the begin x of the last segment is at the pseudo-bar */
	/* The relevant clefsig should be immediately after the FEED
	 * associated with y2 */
	mll_clefsig_p = y2info_p->mll_feed_p->next;
	if (mll_clefsig_p->str != S_CLEFSIG) {
		pfatal("missing clefsig info after newscore");
	}

	/* fill in x of beginning of final segment based on the pseudo-bar */
	end_line_p->start.hor_p = mll_clefsig_p->u.clefsig_p->bar_p->c;
	end_line_p->start.htype = AX;
	end_line_p->start.hsteps = 0.0;
	end_line_p->start.counts = 0.0;

	/* effective distance in x direction will be the sum of the lengths of
	 * the first and last line segments and any intervening. We already
	 * know the length of the first segment and and now
	 * determine the lengths of the last segment. */
	seg1xlen = inpc_x( &(line_p->end), fname, lineno)
			- inpc_x( &(line_p->start), fname, lineno);
	seg2xlen = inpc_x( &(end_line_p->end), fname, lineno)
			- inpc_x( &(end_line_p->start), fname, lineno);

	/* rule 8a */
	/* check for intervening scores and find the effective length in
	 * the X direction. */
	effective_x_len = find_effXlength(seg1xlen, seg2xlen, x1info_p,
							x2info_p, YES);

	/* now find y values */

	/* figure out the first segment y relative to the effective staff */
	for (m_p = x1info_p->mll_feed_p; m_p != (struct MAINLL *) 0;
							m_p = m_p->next) {
		if (m_p->str == S_STAFF &&
				m_p->u.staff_p->staffno == p1staff) {
			break;
		}
	}
	offset = inpc_y( &(line_p->start), fname, lineno)
			- m_p->u.staff_p->c[AY];

	/* rule 9:
	 * First we tackle the easy (relatively speaking) case of both
	 * coordinates being associated with the same staff. */
	if (p1staff == p2staff) {

		/* calculate y values based on slope */
		slope = ((end_line_p->end.vert_p[RY]
				+ end_line_p->end.vsteps * STEPSIZE) -
				(line_p->start.vert_p[RY]
				+ line_p->start.vsteps * STEPSIZE))
				/ effective_x_len;

		/* use the slope to the end y of the first segment and
		 * begin y of the last segment, converted to stepsizes */
		line_p->end.vsteps += (slope * seg1xlen) / STEPSIZE;
		end_line_p->start.vsteps -= (slope * seg2xlen) / STEPSIZE;

		/* if need more than 2 line segments
		 * do the rest of them */
		for (seg_p = Seginfo_p; seg_p != (struct SEGINFO *) 0;  ) {

			add_segment(seg_p, slope, offset, p1staff,
							line_p->linetype);

			/* move on the next segment in list, if any. First
			 * remember current one so we can free it, then move
			 * to next, then free the one we just finished with */
			to_free_p = seg_p;
			seg_p = seg_p->next;
			FREE(to_free_p);
		}
	}

	else {
		/* ends are associated with different staffs */
		double y1, y2;

		/* find two slopes, one for the beginning line segment, one
		 * for the end. For each, base the slope on the distance
		 * between the two effective staffs, adjusted by the
		 * appropriate offset from those staffs. */
		y1 = end_line_p->end.vert_p[RY]
				+ end_line_p->end.vsteps * STEPSIZE;
		y2 = line_p->start.vert_p[RY]
				+ line_p->start.vsteps * STEPSIZE;
		slope = ( (getYstaff(y1info_p->mll_feed_p, p2staff) + y1) -
			(getYstaff(y1info_p->mll_feed_p, p1staff) + y2) )
			/ effective_x_len;
		line_p->end.vsteps += (slope * seg1xlen) / STEPSIZE;

		slope = ( (getYstaff(y2info_p->mll_feed_p, p2staff) + y1) -
			(getYstaff(y2info_p->mll_feed_p, p1staff) + y2) )
			/ effective_x_len;
		end_line_p->start.vsteps -= (slope * seg2xlen) / STEPSIZE;

		/* if need more than 2 line segments
		 * do the rest of them */
		for (seg_p = Seginfo_p; seg_p != (struct SEGINFO *) 0;  ) {

			slope = ( (getYstaff(seg_p->mll_p, p2staff) + y1) -
				(getYstaff(seg_p->mll_p, p1staff) + y2) )
				/ effective_x_len;

			add_segment(seg_p, slope, offset, p1staff,
							line_p->linetype);

			/* move on the next segment in list, if any. First
			 * remember current one so we can free it, then move
			 * to next, then free the one we just finished with */
			to_free_p = seg_p;
			seg_p = seg_p->next;
			FREE(to_free_p);
		}
	}

	/* link end_line_p into proper place in main list */
	/* this will be right before the first BAR after the FEED associated
	 * with y2 */
	for (m_p = mll_clefsig_p->next; m_p->str != S_BAR; m_p = m_p->next) {
		;
	}
	insertMAINLL(new_mll_p, m_p->prev);
}


/* check if location variables associated with an x and y point to at least
 * the same score on the same page. If not, give up */

static void
coordcheck(x_info_p, y_info_p, fname, lineno)

struct COORD_INFO *x_info_p;
struct COORD_INFO *y_info_p;
char *fname;
int lineno;

{
	if (x_info_p == (struct COORD_INFO *) 0 ||
				y_info_p == (struct COORD_INFO *) 0) {
		pfatal("coordinate not in table\n");
	}

	if ( (x_info_p->flags & CT_BUILTIN) || (y_info_p->flags & CT_BUILTIN)) {
		/* if any reference to builtin tag, leave as is */
		return;
	}

	if ( (x_info_p->scorenum != y_info_p->scorenum)
			|| (x_info_p->page != y_info_p->page) ) {
		l_ufatal(fname, lineno,
			"x and y cannot be associated with different scores");
	}
}


/* given info about a coord, return its effective staff. This is the staff
 * associated with the info if any, otherwise the top visible staff */

static int
eff_staff(yinfo_p)

struct COORD_INFO *yinfo_p;

{
	int staff;


	if (yinfo_p->staffno != 0) {
		staff = yinfo_p->staffno;
	}
	else {
		/* use top visible staff as effective staff */
		for (staff = 1; staff <= Score.staffs; staff++) {
			if (svpath(staff, VISIBLE)->visible == YES) {
				break;
			}
		}
	}
	return(staff);
}


/* find the total effective length of a line or curve, accounting for all
 * intervening scores. For each intermediate score, if the save_feed_info
 * flag is set, save away information for use in adding
 * a line or curve for that score */

static double
find_effXlength(seg1xlen, seg2xlen, x1info_p, x2info_p, save_feed_info)

double seg1xlen;	/* length of first part */
double seg2xlen;	/* length of last part */
struct COORD_INFO *x1info_p;	/* info about beginning point */
struct COORD_INFO *x2info_p;	/* info about last point */
int save_feed_info;	/* if YES, do svfeed() call, otherwise not */

{
	double effective_x_len;
	struct MAINLL *m_p;	/* to search main list */


	/* start out with length of first segment */
	effective_x_len = seg1xlen;

	/* check if there might be one or more intervening scores. If the
	 * end point is on the next page, there might be. If both are on
	 * the same page, with the first having a scorenum greater than
	 * the first one plus one, then there is an intervening score
	 * for sure. */
	if (x2info_p->page > x1info_p->page ||
			(x2info_p->page == x1info_p->page &&
			x2info_p->scorenum > x1info_p->scorenum + 1)) {
		/* search forward in main list. Every time we find
		 * a matching newscore that isn't the one associated with
		 * the last segment, save info to be able to
		 * add an intervening line. Also add the length of that line
		 * to the effective x length. */
		for (m_p = x1info_p->mll_feed_p->next;
				m_p != (struct MAINLL *) 0; m_p = m_p->next) {
			if (IS_CLEFSIG_FEED(m_p)) {
				if (m_p == x2info_p->mll_feed_p) {
					/* hurray! We found the score with
					 * the last line segment. No more to
					 * add */
					break;
				}
				else {
					/* need to add another line segment */
					if (m_p->next != (struct MAINLL *) 0 &&
						m_p->next->str == S_CLEFSIG &&
						m_p->next->u.clefsig_p->bar_p
						!= (struct BAR *) 0) {
					   if (save_feed_info == YES) {
						svfeed(m_p, effective_x_len);
					   }
					   effective_x_len += PGWIDTH
					      - eff_rightmargin(m_p) -
					      m_p->next->u.clefsig_p->bar_p->c[AX];
					}
					else {
						pfatal("error in main list while splitting lines");
					}
				}
			}
		}
	}

	/* add in length of final segment */
	effective_x_len += seg2xlen;

	return(effective_x_len);
}


/* allocate SEGINFO and fill it in  */

static void
svfeed(mll_feed_p, xlength)

struct MAINLL *mll_feed_p;
double xlength;

{
	struct SEGINFO *new_p;


	MALLOC(SEGINFO, new_p, 1);
	new_p->mll_p = mll_feed_p;
	new_p->xlength = xlength;

	/* link onto list */
	new_p->next = Seginfo_p;
	Seginfo_p = new_p;
}


/* add LINE for intervening scores */

static void
add_segment(seginfo_p, slope, y_offset, staffno1, linetype)

struct SEGINFO *seginfo_p;
double slope;
double y_offset;	/* offset from staff of beginning point */
int staffno1;	/* staff associated with y of beginning */
int linetype;

{
	struct MAINLL *m_p;		/* index through main list */
	struct MAINLL *new_mll_p;	/* points to new LINE */
	struct LINE *new_line_p;	/* LINE connected to new_mll_p */
	double xleng;			/* distance to end in x direction */


	/* create a new LINE */
	new_mll_p = newMAINLLstruct(S_LINE, -1);
	new_line_p = new_mll_p->u.line_p;
	new_line_p->linetype = linetype;

	/* x coords of the line are at the pseudobar and the rightmargin. We
	 * get to the right margin by adding the correct number of stepsizes
	 * from the pseudobar */
	new_line_p->start.hor_p = seginfo_p->mll_p->next->u.clefsig_p->bar_p->c;
	new_line_p->start.htype = AX;
	new_line_p->start.hsteps = 0.0;
	new_line_p->start.counts = 0.0;
	new_line_p->end.hor_p = new_line_p->start.hor_p;
	new_line_p->end.htype = AX;
	xleng = PGWIDTH - eff_rightmargin(seginfo_p->mll_p)
					 - new_line_p->start.hor_p[AX];
	new_line_p->end.hsteps = xleng / STEPSIZE;
	new_line_p->end.counts = 0.0;

	/* find staff coord info */
	for (m_p = seginfo_p->mll_p; m_p != (struct MAINLL *) 0;
							m_p = m_p->next) {
		if (m_p->str == S_STAFF && m_p->u.staff_p->staffno == staffno1) {
			break;
		}
	}
	/* y coords are determined from the slope */
	new_line_p->start.vert_p = m_p->u.staff_p->c;
	new_line_p->start.vtype = AY;
	new_line_p->start.vsteps = (y_offset + (slope * seginfo_p->xlength))
							/ STEPSIZE;
	new_line_p->end.vert_p = m_p->u.staff_p->c;
	new_line_p->end.vtype = AY;
	new_line_p->end.vsteps = (y_offset +
			(slope * (xleng + seginfo_p->xlength))) / STEPSIZE;

	/* link into proper place in main list */
	/* this will be right before the first BAR after the FEED */
	for (m_p = seginfo_p->mll_p->next; m_p != (struct MAINLL *) 0;
						m_p = m_p->next) {
		if (m_p->str == S_BAR) {
			break;
		}
	}

	if (m_p == (struct MAINLL *) 0) {
		pfatal("couldn't find bar while adding line segment");
	}

	insertMAINLL(new_mll_p, m_p->prev);
}


/* given a MAINLL and a staff number, return the absolute Y of the staff
 * searching forward from the MAINLL */

static double
getYstaff(mll_p, staffno)

struct MAINLL *mll_p;
int staffno;

{
	for (    ; mll_p != (struct MAINLL *) 0; mll_p = mll_p->next) {
		if (mll_p->str == S_STAFF) {
			if (mll_p->u.staff_p->staffno == staffno) {
				return(mll_p->u.staff_p->c[AY]);
			}
		}
	}
	pfatal("couldn't find Y of staff");
	/*NOTREACHED*/
	return(0.0);
}


/* check whether a CURVE needs to be split. If so, split it */

static void
chkcurve(mll_p)

struct MAINLL *mll_p;	/* points to CURVE */

{
	struct CURVE *curve_p;
	struct COORD_INFO *x1info_p, *y1info_p, *x2info_p, *y2info_p;
	int bulge;		/* 1 for UP or -1 for DOWN */
	register int n;
	int j;
	int curscore, curpage;	/* current score and page */
	int is_split = NO;
	int p1staff, p2staff;	/* staff associate with each endpoint */
	struct MAINLL *new_mll_p;	/* place for 2nd part of split curve */
	struct MAINLL *m_p;		/* to find place in list to insert */
	struct CURVE *new_crv_p;	/* points for second part */
	int ncoord1, ncoord2;	/* number of coords in each piece */
	int add1, add2;		/* 1 if need to add another point to the
				 * first of second piece of the curve,
				 * 0 if not */
	float offset;
	struct MAINLL *mll_clefsig_p;	/* clefsig for score where part of
					 * a curve goes */
	double seg1xlen, seg2xlen;	/* length of begin and end parts */
	double effective_x_len;		/* total length in X direction */
	double slope;			/* of line */
	int index1, index2;		/* into coordlist array */
	char *fname;
	int lineno;
	struct SEGINFO *seg_p, *to_free_p;	/* to deal with curves for
					 * intermediate scores */
	double addedx;		/* x of endpoint that we added */
	double userx;		/* x of nearest user-defined point */
	double y1, y2;


	curve_p = mll_p->u.curve_p;
	fname = mll_p->inputfile;
	lineno = mll_p->inputlineno;
	curscore = curpage = -1;

	Seginfo_p = (struct SEGINFO *) 0;

	for (n = 0; n < curve_p->ncoord; n++) {

		x1info_p = find_coord( curve_p->coordlist[n].hor_p);
		y1info_p = find_coord( curve_p->coordlist[n].vert_p);

		if (x1info_p == (struct COORD_INFO *) 0 ||
				y1info_p == (struct COORD_INFO *) 0) {
			/* must be an absolute coordinate */
			continue;
		}

		/* rule 1: if any coordinate on the list
		 * is associated with something
		 * invisible, ignore the whole curve. */
		if ( is_invis(x1info_p) || is_invis(y1info_p) ) {

			/* as with lines, unlink so we don't print, but
			 * don't free, so that calling function can still
			 * follow the ->next pointer */
			unlinkMAINLL(mll_p);
			return;
		}

		/* rule 4. Check that x and y are on same score */
		coordcheck(x1info_p, y1info_p, fname, lineno);

		/* rule 3 checking. See if all on same score/page */
		if (curpage == -1) {
			curscore = x1info_p->scorenum;
			curpage = x1info_p->page;
		}
		else {
			if (curscore != x1info_p->scorenum ||
					curpage != x1info_p->page) {
				is_split = YES;
			}
		}
	}

	/* If this curve was specified using bulge parameters, go calculate
	 * the intermediate points */
	if (curve_p->nbulge > 0) {
		calc_bulge(curve_p, fname, lineno, is_split, mll_p);
		/* some INPCOORDs might well point off the page, so fix those */
		fix_inpcoords(mll_p);
	}

	/* finish rule 3 checking. If all were on same score, we are done */
	if (is_split == NO) {
		return;
	}

	/* go through curve points a pair at a time */
	for (n = 0; n < curve_p->ncoord - 1; n++) {
		x1info_p = find_coord( curve_p->coordlist[n].hor_p);
		y1info_p = find_coord( curve_p->coordlist[n].vert_p);
		x2info_p = find_coord( curve_p->coordlist[n + 1].hor_p);
		y2info_p = find_coord( curve_p->coordlist[n + 1].vert_p);

		if (x1info_p == (struct COORD_INFO *) 0
				|| y1info_p == (struct COORD_INFO *) 0
				|| x2info_p == (struct COORD_INFO *) 0
				|| y2info_p == (struct COORD_INFO *) 0) {
			/* absolute coordinate */
			continue;
		}

		/* rule 2. If any builtin variable used,
		 * no split of this segment */
		if ( is_builtin(x1info_p) || is_builtin(y1info_p) ||
				is_builtin(x2info_p) || is_builtin(y2info_p) ) {
			continue;
		}

		/* rule 6. If both ends of segment are on same page/score
		 * no split of this segment */
		if ( (x1info_p->scorenum == y1info_p->scorenum)
				&& (x1info_p->scorenum == x2info_p->scorenum)
				&& (x1info_p->scorenum == y2info_p->scorenum)
				&& (x1info_p->page == y1info_p->page)
				&& (x1info_p->page == x2info_p->page)
				&& (x1info_p->page == y2info_p->page) ) {
			continue;
		}

		/* rule 7. Only split to forward score */
		if (x2info_p->page < x1info_p->page ||
				(x2info_p->page == x1info_p->page &&
				x2info_p->scorenum < x1info_p->scorenum)) {
			l_ufatal(mll_p->inputfile, mll_p->inputlineno,
				"can't draw curve backwards to previous score");
		}

		/* if we're here, segment must be split */

		/* figure out if curve generally bulges up or down */
		bulge = bulgedir(curve_p, n, mll_p->inputfile,
							mll_p->inputlineno);

		/* get effective staffs */
		p1staff = eff_staff(y1info_p);
		p2staff = eff_staff(y2info_p);

		/* set up first part of split curve. It will have as many
		 * coords as we have so far, unless that is only 2, in which
		 * case we have to add another, because a curve must have at
		 * least three points */
		if (n == 0) {
			ncoord1 = 3;
			add1 = 1;
		}
		else {
			ncoord1 = n + 2;
			add1 = 0;
		}
		/* similarly, the second portion has as many points as are
		 * left, or a minimum of 3 */
		if (curve_p->ncoord - n == 2) {
			ncoord2 = 3;
			add2 = 1;
		}
		else {
			ncoord2 = curve_p->ncoord - n;
			add2 = 0;
		}

		/* Split off the second part into a separate curve */
		new_mll_p = newMAINLLstruct(S_CURVE, mll_p->inputlineno);
		new_mll_p->inputfile = mll_p->inputfile;
		new_crv_p = new_mll_p->u.curve_p;
		new_crv_p->curvetype = curve_p->curvetype;
		new_crv_p->ncoord = (short) ncoord2;
		MALLOC (INPCOORD, new_crv_p->coordlist, ncoord2);

		/* copy second part into second curve. Copy backwards from
		 * the end, but don't fill in the first point of it, because
		 * we still need to calculate that */
		for (ncoord2--, j = curve_p->ncoord - 1; ncoord2 > 0 + add2;
							ncoord2--, j--) {
			new_crv_p->coordlist[ncoord2] = curve_p->coordlist[j];
		}

		/* realloc space for first part of curve, with just the
		 * points needed */
		REALLOC(INPCOORD, curve_p->coordlist, ncoord1);
		curve_p->ncoord = (short) ncoord1;

		/* find new endpoints for the split ends */

		/* the end x of the first segment is just like the beginning x,
		 * but offset to the east far enough to
		 * reach the end of the score. */
		curve_p->coordlist[ncoord1 - 1] = curve_p->coordlist[0];
		offset = PGWIDTH - eff_rightmargin(mll_p)
				- inpc_x( &(curve_p->coordlist[0]),
				fname, lineno );

		/* handle bizarre case of beginning being too far right to deal
		 * with properly */
		if (offset < 0.1) {
			offset = 0.1;
		}
		/* convert inches to stepsizes,
		 * which is how offset are stored */
		curve_p->coordlist[ncoord1 - 1].hsteps
						+= offset / STEPSIZE;

		/* the begin x of the last segment is at the pseudo-bar */
		/* The relevant clefsig should be immediately after the FEED
	 	 * associated with y2 */
		mll_clefsig_p = y2info_p->mll_feed_p->next;
		if (mll_clefsig_p->str != S_CLEFSIG) {
			pfatal("missing clefsig info after newscore");
		}


		/* fill in x of beginning of final part based
		 * on the pseudo-bar */
		new_crv_p->coordlist[0].hor_p
				= mll_clefsig_p->u.clefsig_p->bar_p->c;
		new_crv_p->coordlist[0].htype = AX;
		new_crv_p->coordlist[0].hsteps = 0.0;
		new_crv_p->coordlist[0].counts = 0.0;

		/* If the first user defined point on the subsequent score
		 * is extremely close to the pseudo-bar where we want to
		 * start this segment, or worse yet, is west of it
		 * (because they specified a negative offset that makes the
		 * curve bend back into the preceeding measure),
		 * we move the beginning point that we added,
		 * to make it 0.1 inch west of the user's point.
		 * This is not the same remedial action as we take later for
		 * the somewhat similar case at the end of the preceeding score.
		 * The argument for this lack of symmetry is that when
		 * carrying out from the end of a score, we don't want to
		 * spill out into the margin--it's better to end the curve
		 * a tiny bit too early. On the other hand, at the beginning
		 * of a score, there is probably some room in the
		 * clef/key/time area to allow starting somewhat earlier
		 * than the pseudo-bar and still look okay.
		 * Also, if the user did do some crazy curve that would
		 * bend back into the preceeding measure, it's just too
		 * hard to try to do anything about that on the preceeding
		 * score, but we can make it bend back prior to the
		 * pseudobar, which can sort of honor what they asked for.
		 */
		addedx = inpc_x( &(new_crv_p->coordlist[0]), fname, lineno );
		userx = inpc_x( &(new_crv_p->coordlist[1+add2]), fname, lineno );
		if (userx - addedx < 0.1) {
			new_crv_p->coordlist[0].hsteps =
					(-0.1 + (userx - addedx)) / STEPSIZE;
		}

		/* use the last user defined point (the one immediately
		 * before the one or two points we just added to the
		 * first part) as a reference point */
		index1 = (add1 ? 0 : curve_p->ncoord - 2);
		/* similarly, use first user-defined point of last part */
		index2 = (add2 ? 2 : 1);

		/* find y values for split ends  */

		/* copy vertical info from nearest point, will adjust vsteps
		 * later as needed based on slope */
		curve_p->coordlist[ncoord1 - 1].vert_p =
				curve_p->coordlist[index1].vert_p;
		curve_p->coordlist[ncoord1 - 1].vtype =
				curve_p->coordlist[index1].vtype;
		curve_p->coordlist[ncoord1 - 1].vsteps =
				curve_p->coordlist[index1].vsteps;
		new_crv_p->coordlist[0].vert_p =
				new_crv_p->coordlist[index2].vert_p;
		new_crv_p->coordlist[0].vtype =
				new_crv_p->coordlist[index2].vtype;
		new_crv_p->coordlist[0].vsteps =
				new_crv_p->coordlist[index2].vsteps;

		/* first need to find effective length in X direction for
		 * determining slope */
		seg1xlen = offset;
		seg2xlen = inpc_x( &(new_crv_p->coordlist[index2]), fname, lineno)
			- inpc_x( &(new_crv_p->coordlist[0]), fname, lineno);
		effective_x_len = find_effXlength(seg1xlen, seg2xlen, x1info_p,
						x2info_p, YES);

		/* figure out the first part's y relative
		 * to the effective staff */
		for (m_p = x1info_p->mll_feed_p; m_p != (struct MAINLL *) 0;
							m_p = m_p->next) {
			if (m_p->str == S_STAFF &&
					m_p->u.staff_p->staffno == p1staff) {
				break;
			}
		}
		offset = inpc_y( &(curve_p->coordlist[index1]),
				fname, lineno) - m_p->u.staff_p->c[AY];

		/* find two slopes, one for the beginning line segment,
		 * one for the end. For each, base the slope
		 * on the distance between the two effective staffs,
		 * adjusted by the appropriate offset
		 * from those staffs. */
		y1 = new_crv_p->coordlist[index2].vert_p[RY] +
				new_crv_p->coordlist[index2].vsteps * STEPSIZE;
		y2 = curve_p->coordlist[index1].vert_p[RY]
				+ curve_p->coordlist[index1].vsteps * STEPSIZE;
		slope = ((getYstaff(y1info_p->mll_feed_p, p2staff) + y1)
			    - (getYstaff(y1info_p->mll_feed_p, p1staff) + y2))
			    / effective_x_len;
		curve_p->coordlist[ncoord1 - 1].vsteps
					+= (slope * seg1xlen) / STEPSIZE;

		slope = ((getYstaff(y2info_p->mll_feed_p, p2staff) + y1) -
				(getYstaff(y2info_p->mll_feed_p, p1staff) + y2))
				/ effective_x_len;
				new_crv_p->coordlist[0].vsteps
					-= (slope * seg2xlen) / STEPSIZE;

		/* if need more than 2 curve segments
		 * do the rest of them */
		for (seg_p = Seginfo_p; seg_p != (struct SEGINFO *) 0;  ) {

			slope = ((getYstaff(seg_p->mll_p, p2staff) + y1)
				    - (getYstaff(seg_p->mll_p, p1staff) + y2))
				    / effective_x_len;

			add_crv_seg(seg_p, slope, offset, p1staff,
				curve_p->curvetype, bulge, fname, lineno);

			/* move on the next segment in list, if any.
			 * First remember current one so we can free it,
			 * then move to next, then free the one
			 * we just finished with */
			to_free_p = seg_p;
			seg_p = seg_p->next;
			FREE(to_free_p);
		}

		/* If there was a user-defined point extremely close to
		 * where we did the split (which is moderately likely,
		 * since they may well specific a point at a bar line),
		 * or even worse, if the one we added somehow came out
		 * to the left of the user-defined point,
		 * the curve could end up looking very strange since it
		 * contains a very tiny segment. So in that case we discard
		 * the extra point we added as the end of the split place,
		 * and just use the user-defined point.
		 */
		addedx = inpc_x( &(curve_p->coordlist[ncoord1 - 1]), fname, lineno);
		userx = inpc_x( &(curve_p->coordlist[ncoord1 - 2]), fname, lineno);
		if (add1 == 0 && (fabs(addedx - userx) < 0.1 || userx > addedx)) {
			if (ncoord1 == 3) {
				/* If discarding a point would get us down
				 * to only two points, we'll discard the
				 * user's point, by marking that we need to
				 * fill in an extra point in the middle
				 * (at subscript [1]). The ending point we added
				 * at [2] is so close that no one should
				 * notice. */
				add1 = 1;
			}
			else {
				/* We already had more than 3 points,
				 * so we'll just ignore the extra one we
				 * added. The previous user-defined point
				 * is close enough to where it should end.
				 * It isn't worth the trouble to reclaim
				 * the extra array element; just let it leak.
				 */
				ncoord1--;
				curve_p->ncoord = ncoord1;
			}
		}

		/* if first part of curve ended up with only a single segment,
		 * need to add another point in the middle to make the
		 * required minimum of 3 points for a curve. So copy the
		 * first point, adjust the x to be halfway between the first
		 * and last point, and adjust the y to be halfway between the
		 * the y's of the endpoint, offset by a little bit to get
		 * a bend in the curve. */
		if (add1 == 1) {
			curve_p->coordlist[1] = curve_p->coordlist[0];
			curve_p->coordlist[1].hsteps =
					(curve_p->coordlist[0].hsteps
					+ curve_p->coordlist[2].hsteps) / 2.0;
			curve_p->coordlist[1].counts =
					(curve_p->coordlist[0].counts
					+ curve_p->coordlist[2].counts) / 2.0;
			/* the square root is to make the amount of bulge
			 * proportional to the x length, 1 stepsize for a
			 * piece 1 inch long, less for shorter pieces,
			 * more for longer pieces */
			curve_p->coordlist[1].vsteps =
					(curve_p->coordlist[0].vsteps
					+ curve_p->coordlist[2].vsteps) / 2.0
					+ (bulge * sqrt(seg1xlen * Score.scale_factor));
		}

		/* similarly for the ending part of curve */
		if (add2 == 1) {
			new_crv_p->coordlist[1] = new_crv_p->coordlist[0];
			new_crv_p->coordlist[1].hsteps =
					(seg2xlen / 2.0) / STEPSIZE;
			new_crv_p->coordlist[1].counts = 0.0;
			/* the square root is to make the amount of bulge
			 * proportional to the x length, 1 stepsize for a
			 * piece 1 inch long, less for shorter pieces,
			 * more for longer pieces */
			new_crv_p->coordlist[1].vsteps =
					(new_crv_p->coordlist[0].vsteps
					+ new_crv_p->coordlist[2].vsteps) / 2.0
					+ (bulge * sqrt(seg2xlen * Score.scale_factor));
		}


		/* link new_mll_p into proper place in main list */
		/* this will be right before the first BAR after
		 * the FEED associated with y2 */
		for (m_p = mll_clefsig_p->next; m_p->str != S_BAR;
							m_p = m_p->next) {
			;
		}
		insertMAINLL(new_mll_p, m_p->prev);

		/* If the rest of the curve requires further splitting,
		 * we do that now, then break out of this loop */
		chkcurve(new_mll_p);
		break;
	}
}


/* try to determine whether a user-defined curve generally bulged upward
 * or downward and return 1 for up or -1 for down as appropriate.
 * If intermediate points
 * seem to be mainly higher than the endpoints it is probably up, if they
 * tend to be below the endpoints, it is probably down. */

static int
bulgedir(curve_p, index, inputfile, inputlineno)

struct CURVE *curve_p;
int index;		/* check bulge dir between this point in array and
			 * the next one */
char *inputfile;	/* where curve was defined */
int inputlineno;

{
	int retval = 0;

	
	if (index == 0 || index == curve_p->ncoord - 2) {
		/* if checking an end of the curve, we use the two end
		 * segments to guess the direction */
		retval += cmpcoords(curve_p, 0, 1);
		retval += cmpcoords(curve_p, curve_p->ncoord - 1,
					curve_p->ncoord - 2);
		/* if more than 3 points in curve, we can use the adjacent
		 * segment on the one side where there is an adjacent segment
		 * as another reference point */
		if (curve_p->ncoord > 3) {
			if (index == 0) {
				retval += cmpcoords(curve_p, 1, 2);
			}
			else {
				retval += cmpcoords(curve_p,
						curve_p->ncoord - 2,
						curve_p->ncoord - 3);
			}
		}
	}
	else {
		/* for a segment in the middle, use the segments on
		 * either side for reference */
		retval += cmpcoords(curve_p, index - 1, index);
		retval += cmpcoords(curve_p, index + 2, index + 1);
		/* if that was inconclusive, try using the endpoints */
		if (retval == 0) {
			retval += cmpcoords(curve_p, 0, 1);
			retval += cmpcoords(curve_p, curve_p->ncoord - 1,
					curve_p->ncoord - 2);
		}
	}

	if (retval == 0) {
		/**** eventually try more drastic measures to try to deduce
		 *** the direction??? It's debatable about whether this should
		 * be a ufatal or pfatal. The program should be smart enough
		 * to figure out the direction, but probably can't be that
		 * smart for just any arbitrary curve shape
		 * the user tries to throw at it, and
		 * user can probably always manage to get what they want by
		 * specifying enough points, so make ufatal. */
		l_ufatal(inputfile, inputlineno,
			"can't determine curve bend direction; try specifying more points");
	}
	return (retval > 0 ? 1 : -1);
}


/* return 1 if point p1 appears to be below point p2. Return -1 if p1 appears
 * to be above point p2. Return 0 if can't tell */

static int
cmpcoords(curve_p, p1, p2)

struct CURVE *curve_p;
int p1;
int p2;

{
	struct COORD_INFO *y1info_p, *y2info_p;
	int staff1, staff2;
	double y1, y2;


	/* check the two points */
	y1info_p = find_coord(curve_p->coordlist[p1].vert_p);
	y2info_p = find_coord(curve_p->coordlist[p2].vert_p);

	if ((y1info_p == (struct COORD_INFO *) 0)
				|| (y2info_p == (struct COORD_INFO *) 0)) {
		pfatal("couldn't find coord info in cmpcoords");
	}

	/* if on same score, can compare the absolute Y values */
	if (y1info_p->mll_feed_p == y2info_p->mll_feed_p) {
		y1 = inpc_y( &(curve_p->coordlist[p1]), (char *) 0, -1);
		y2 = inpc_y( &(curve_p->coordlist[p2]), (char *) 0, -1);
		if (y1 < y2) {
			return(1);
		}
		else if (y2 < y1) {
			return(-1);
		}
	}
	else {
		/* weren't on same score. See if associated with same staff.
		 * If so, we can compare the relative Y values. If associated
		 * with different staffs, if second point is with lower staff
		 * it probably bulges downward. */
		staff1 = eff_staff(y1info_p);
		staff2 = eff_staff(y2info_p);
		if (staff1 == staff2) {
			y1 = curve_p->coordlist[p1].vert_p
				[abs2rel(curve_p->coordlist[p1].vtype)]
				+ (curve_p->coordlist[p1].vsteps * STEPSIZE
				* svpath(staff1, STAFFSCALE)->staffscale);
			y2 = curve_p->coordlist[p2].vert_p
				[abs2rel(curve_p->coordlist[p2].vtype)]
				+ (curve_p->coordlist[p2].vsteps * STEPSIZE
				* svpath(staff2, STAFFSCALE)->staffscale);
			if (y1 < y2) {
				return(1);
			}
			else if (y2 < y1) {
				return(-1);
			}
		}
		else if (staff1 < staff2) {
			/* first point higher, bends down */
			return(-1);
		}
		else {
			return(1);
		}
	}
	return(0);
}


/* given a coord type of AY, AN, or AS, return its relative coord type,
 * that is RY, RN, or RS respectively */

static int
abs2rel(vtype)

int vtype;	/* AY, AN, or AS */

{
	switch(vtype) {
	case AY:
		return(RY);
	case AN:
		return(RN);
	case AS:
		return(RS);
	default:
		pfatal("illegal coordinate type in abs2rel");
	}
	/*NOTREACHED*/
	return(0);
}


/* add CURVE for intervening scores */

static void
add_crv_seg(seginfo_p, slope, y_offset, staffno1, curvetype, bulge, filename, lineno)

struct SEGINFO *seginfo_p;
double slope;
double y_offset;	/* offset from staff of beginning point */
int staffno1;	/* staff associated with y of beginning */
int curvetype;
int bulge;	/* 1 for bulge up, -1 for bulge down */
char *filename;	/* where original curve was defined */
int lineno;	/* where original curve was defined */

{
	struct MAINLL *m_p;		/* index through main list */
	struct MAINLL *new_mll_p;	/* points to new LINE */
	struct CURVE *new_crv_p;	/* CURVE connected to new_mll_p */
	double xleng;			/* distance to end in x direction */


	/* create a new CURVE */
	new_mll_p = newMAINLLstruct(S_CURVE, lineno);
	new_mll_p->inputfile = filename;
	new_crv_p = new_mll_p->u.curve_p;
	new_crv_p->curvetype = (short) curvetype;
	new_crv_p->ncoord = 3;
	MALLOC (INPCOORD, new_crv_p->coordlist, 3);

	/* x coords of the curve ends are at the pseudobar and the rightmargin.
	 * The middle point, appropriately enough, is in the middle. We
	 * get to the right margin by adding the correct number of stepsizes
	 * from the pseudobar */
	new_crv_p->coordlist[0].hor_p = seginfo_p->mll_p->next->u.clefsig_p->bar_p->c;
	new_crv_p->coordlist[0].htype = AX;
	new_crv_p->coordlist[0].hsteps = 0.0;
	new_crv_p->coordlist[0].counts = 0.0;
	new_crv_p->coordlist[2].hor_p = new_crv_p->coordlist[0].hor_p;
	new_crv_p->coordlist[2].htype = AX;
	xleng = PGWIDTH - eff_rightmargin(seginfo_p->mll_p)
					- new_crv_p->coordlist[0].hor_p[AX];
	new_crv_p->coordlist[2].hsteps = xleng / STEPSIZE;
	new_crv_p->coordlist[2].counts = 0.0;
	new_crv_p->coordlist[1].hor_p = new_crv_p->coordlist[0].hor_p;
	new_crv_p->coordlist[1].htype = AX;
	new_crv_p->coordlist[1].hsteps = (xleng / 2.0) / STEPSIZE;
	new_crv_p->coordlist[1].counts = 0.0;

	/* find staff coord info */
	for (m_p = seginfo_p->mll_p; m_p != (struct MAINLL *) 0;
							m_p = m_p->next) {
		if (m_p->str == S_STAFF && m_p->u.staff_p->staffno == staffno1) {
			break;
		}
	}
	/* y coords are determined from the slope. Offset the endpoint by 1
	 * STEPSIZE and the middle point by 1 STEPSIZE in the opposite
	 * direction to get a little bulge */
	new_crv_p->coordlist[0].vert_p = m_p->u.staff_p->c;
	new_crv_p->coordlist[0].vtype = AY;
	new_crv_p->coordlist[0].vsteps = (y_offset - (bulge * STEPSIZE)
				+ (slope * seginfo_p->xlength)) / STEPSIZE;
	new_crv_p->coordlist[2].vert_p = m_p->u.staff_p->c;
	new_crv_p->coordlist[2].vtype = AY;
	new_crv_p->coordlist[2].vsteps = (y_offset - (bulge * STEPSIZE) +
			(slope * (xleng + seginfo_p->xlength))) / STEPSIZE;

	/* add middle point, bulging by 2 stepsizes */
	new_crv_p->coordlist[1].vert_p = m_p->u.staff_p->c;
	new_crv_p->coordlist[1].vtype = AY;
	new_crv_p->coordlist[1].vsteps = (y_offset + (bulge * STEPSIZE) +
		(slope * (seginfo_p->xlength + (xleng / 2.0))))
		/ STEPSIZE;

	/* link into proper place in main list */
	/* this will be right before the first BAR after the FEED */
	for (m_p = seginfo_p->mll_p->next; m_p != (struct MAINLL *) 0;
							m_p = m_p->next) {
		if (m_p->str == S_BAR) {
			break;
		}
	}

	if (m_p == (struct MAINLL *) 0) {
		pfatal("couldn't find bar when adding curve segment");
	}

	insertMAINLL(new_mll_p, m_p->prev);
}


/* return YES if given coordinate is invisible */

static int
is_invis(cinfo_p)

struct COORD_INFO *cinfo_p;

{
	/* It is invisible if explictly marked as such, or if
	 * it is not a builtin, but never had its page set differently
	 * than the initial default of zero. */
	if ((cinfo_p->flags & CT_INVISIBLE) ||
			(cinfo_p->page == 0 && is_builtin(cinfo_p) == NO)) {
		return(YES);
	}
	else {
		return(NO);
	}
}


/* return YES if given coordinate is a builtin location variable */

static int
is_builtin(cinfo_p)

struct COORD_INFO *cinfo_p;

{
	return((cinfo_p->flags & CT_BUILTIN) ? YES : NO); 
}


/* go through list and see if any variables were defined on one page and
 * used on another. If so, move them. */

static void
move2correct_page()

{
	int page = 1;				/* current page */
	struct MAINLL *m_p;			/* index through main list */
	struct MAINLL **insertp_p;		/* where, on each page, to
						 * insert items moved from
						 * other pages */
	struct COORD_INFO *info_p, *info1_p;	/* to see what page the item
						 * is supposed to be on */
	struct PRINTDATA *pr_p;			/* index through print list */
	struct PRINTDATA **pr_del_p_p;		/* where to delete item from
						 * list when moving */
	struct MAINLL *next_p;			/* which to check next */
	int xabs, yabs;				/* YES if x or y is absolute or
						 * builtin-relative coord */


	/* allocate array for saving where to insert things to be moved.
	 * There is no page 0, so leave extra element for that */
	CALLOC(MAINLL *, insertp_p, Total_pages + 1);


	for (m_p = Mainllhc_p; m_p != (struct MAINLL *) 0; ) {

		/* save what will be the next to check, in case the current
		 * one gets moved */
		next_p = m_p->next;

		switch(m_p->str) {

		case S_BAR:
			if (insertp_p[page] == (struct MAINLL *) 0) {
				/* find first bar on that page and save the
				 * main list struct right before that. That
				 * is where we will move anything that has to
				 * be moved to this page */
				insertp_p[page] = m_p->prev;
			}
			break;

		case S_FEED:
			if (m_p->u.feed_p->pagefeed == YES) {
				page++;
			}
			break;

		case S_LINE:
			/* only check user defined lines */
			if (m_p->inputlineno != -1) {
				info_p = find_coord(m_p->u.line_p->start.hor_p);
				if (info_p != (struct COORD_INFO *) 0 &&
						info_p->page != page &&
						info_p->flags != CT_BUILTIN) {
					move_it(m_p, insertp_p[info_p->page],
							info_p->page);
				}
			}
			break;

		case S_CURVE:
			/* only check user defined curves */
			if (m_p->inputlineno != -1) {
				info_p = find_coord(m_p->u.curve_p->
							coordlist[0].hor_p);
				if (info_p != (struct COORD_INFO *) 0 &&
						info_p->page != page &&
						info_p->flags != CT_BUILTIN) {
					move_it(m_p, insertp_p[info_p->page],
							info_p->page);
				}
			}
			break;

		case S_PRHEAD:
			for (pr_p = m_p->u.prhead_p->printdata_p, pr_del_p_p =
						&(m_p->u.prhead_p->printdata_p);
						pr_p != (struct PRINTDATA *) 0;
						pr_del_p_p = &(pr_p->next)) {

				/* find out about x and y portions */
				info_p = find_coord(pr_p->location.hor_p);
				info1_p = find_coord(pr_p->location.vert_p);

				/* figure out if x and y are absolute or
				 * associated with builtins */
				xabs = yabs = NO;
				if (info_p == (struct COORD_INFO *) 0) {
					xabs = YES;
				}
				else if (info_p->flags & CT_BUILTIN) {
					xabs = YES;
				}
				if (info1_p == (struct COORD_INFO *) 0) {
					yabs = YES;
				}
				else if (info1_p->flags & CT_BUILTIN) {
					yabs = YES;
				}
				/* if both x and y are absolute coordinates,
				 * don't move it */
				if ((xabs == YES) && (yabs == YES)) {
					pr_p = pr_p->next;
					continue;
				}

				/* if both x and y are not absolute, make sure
				 * they are associated with same staff */
				if ((xabs == NO) && (yabs == NO)) {
					coordcheck(info_p, info1_p,
							pr_p->inputfile,
							pr_p->inputlineno);
				}

				/* normally we'll check for moving based on x.
				 * (most of the time x and y will be on the same
				 * page, so we can use either.) However, if
				 * x happens to be the one that is absolute,
				 * use y instead */
				if ((xabs == YES) && (yabs == NO)) {
					info_p = info1_p;
				}

				if (info_p->page != page ) {
					struct MAINLL *new_mll_p;
					struct PRINTDATA *save_p;

					/* moving a PRINTDATA is harder than
					 * moving a line or curve, because,
					 * there could be a list, so we have
					 * surgically remove this one from the
					 * list and graft onto a new PRHEAD */
					new_mll_p = newMAINLLstruct(S_PRHEAD,
						m_p->u.prhead_p->printdata_p->
						inputlineno);
					new_mll_p->inputfile = m_p->u.prhead_p->							printdata_p->inputfile;
					new_mll_p->u.prhead_p->printdata_p
							= pr_p;

					/* save link for continuing for loop */
					save_p = pr_p->next;

					/* patch up linked list */
					*pr_del_p_p = pr_p->next;
					pr_p->next = (struct PRINTDATA *) 0;

					/* If there is a page to move it to,
					 * move it there. If page is zero,
					 * it must be associated with something
					 * invisible, so discard it. */
					if (info_p->page != 0) {
						/* move to correct page */
						if (insertp_p[info_p->page] ==
									0) {
							l_ufatal(pr_p->inputfile,
								pr_p->inputlineno,
								"forward reference to location tag");
						}
						insertMAINLL(new_mll_p,
							insertp_p[info_p->page]); 
					}
					else {
						FREE(new_mll_p);
					}

					/* prepare for next time through loop */
					pr_p = save_p;
				}
				else {
					pr_p = pr_p->next;
				}
			}

			/* if all moved, can discard */
			if (m_p->u.prhead_p->printdata_p
						== (struct PRINTDATA *) 0) {
				unlinkMAINLL(m_p);
			}
			break;

		default:
			break;
		}

		m_p = next_p;
	}
	FREE(insertp_p);
}


/* move given MAINLL to specified place */

static void
move_it(m_p, im_p, page)

struct MAINLL *m_p;	/* move this */
struct MAINLL *im_p;	/* insert here */
int page;		/* if page 0, move to oblivion */

{
	unlinkMAINLL(m_p);
	if (page == 0) {
		/* must be invisible, so discard it */
		FREE(m_p);
		return;
	}
	if (im_p == (struct MAINLL *) 0) {
		l_ufatal(m_p->inputfile, m_p->inputlineno,
				"forward reference to location tag");
	}
	insertMAINLL(m_p, im_p);
}


/* Go through main list.
 * If there are any INPCOORDs that have a hor_p of a bar, and have a positive
 * x offset, and that bar is at the end of a score and thus has a pseudo-bar
 * on the following score, move the hor_p to point to the pseudo-bar instead.
 * If the y of the same INPCOORD also pointed to the same bar before, move it
 * as well, otherwise leave it as is. */

static void
move2pseudo()

{
	struct MAINLL *m_p;	/* walk through main list */
	struct PRINTDATA *pr_p;	/* walk through list of print commands */
	int n;			/* index through curve coordinates */


	/* go through main list */
	for (m_p = Mainllhc_p; m_p != (struct MAINLL *) 0; m_p = m_p->next) {

		if (m_p->str == S_LINE) {
			/* handle start and end points of line */
			do_pseudo( &(m_p->u.line_p->start), m_p );
			do_pseudo( &(m_p->u.line_p->end), m_p );
		}

		else if (m_p->str == S_CURVE) {
			/* do each point of curve */
			for (n = m_p->u.curve_p->ncoord - 1; n >= 0; n--) {
				do_pseudo( &(m_p->u.curve_p->coordlist[n]), m_p);
			}
		}

		else if (m_p->str == S_PRHEAD) {
			/* do each print command */
			for (pr_p = m_p->u.prhead_p->printdata_p;
					pr_p != (struct PRINTDATA *) 0;
					pr_p = pr_p->next) {
				do_pseudo( &(pr_p->location), m_p );
			}
		}
	}
}


/* given an INPCOORD, if it has a hor_p of a bar, and has a positive
 * x offset, and that bar is at the end of a score and thus has a pseudo-bar
 * on the following score, move the hor_p to point to the pseudo-bar instead.
 * If the y of the same INPCOORD also pointed to the same bar before, move it
 * as well, otherwise leave it as is. */

static void
do_pseudo(inpc_p, mll_p)

struct INPCOORD *inpc_p;
struct MAINLL *mll_p;

{
	struct COORD_INFO *info_p;


	if ((info_p = find_coord(inpc_p->hor_p)) == (struct COORD_INFO *) 0) {
		/* probably an absolute coordinate */
		return;
	}

	/* if x is associated with a bar... */
	if (info_p->flags & CT_BAR) {
		/* and that bar has an associated pseudo bar... */
		if (info_p->pseudo_bar_p != (struct BAR *) 0) {
			/* and the x value after adding offsets is into
			 * the right margin area... */
			if (inpc_x(inpc_p, (char *) 0, -1)
					> PGWIDTH - eff_rightmargin(mll_p)) {
				/* if y of INPCOORD was also associated with
				 * the same bar, move it to pseudo-bar */
				if (inpc_p->hor_p == inpc_p->vert_p) {
					inpc_p->vert_p = info_p->pseudo_bar_p->c;
				}
				/* move x to pseudo-bar */
				inpc_p->hor_p = info_p->pseudo_bar_p->c;
			}
		}
	}
}


/* Given a line or curve, fix any INPCOORD that end up off the margin */

static void
fix_inpcoords(mll_p)

struct MAINLL *mll_p;

{
	int n;		/* index through curve points */


	if (mll_p->str == S_CURVE) {
		for (n = 0; n < mll_p->u.curve_p->ncoord; n++) {
			adj_coord( & (mll_p->u.curve_p->coordlist[n]), mll_p,
				((n > 0) ? &(mll_p->u.curve_p->coordlist[n-1])
				: (struct INPCOORD *) 0) );
		}
	}
	else if (mll_p->str == S_LINE) {
		adj_coord( &(mll_p->u.line_p->start), mll_p,
				(struct INPCOORD *) 0);
		adj_coord( &(mll_p->u.line_p->end), mll_p,
				&(mll_p->u.line_p->start) );
	}
}


/* If x of INPCOORD ends up off the page, change the INPCOORD
 * to be on the following score, using that score's pseudo-bar
 * as the reference. */

static void
adj_coord(coord_p, mll_p, prev_coord_p)

struct INPCOORD *coord_p;	/* what to potentially adjust */
struct MAINLL *mll_p;	/* points to the line or curve containing coord_p */
struct INPCOORD *prev_coord_p;	/* previous coord if any, else NULL */

{
	struct MAINLL *m_p;		/* for finding thing in main list */
	float x, y;
	float prev_x, prev_y;		/* location of prev_coord_p */
	struct INPCOORD temp_coord;	/* reference if prev_coord_p is NULL */
	float right_margin_x;		/* PGWIDTH - eff_rightmargin */
	float staff_y = 0.0;
	struct COORD_INFO *xinfo_p, *yinfo_p;	/* for finding which staff,
					 * clefsig, etc is associated with
					 * the point */
	struct BAR *bar_p;		/* pseudo-bar */
	int staffno;


	/* don't bother with invisible points. */
	if (prev_coord_p != (struct INPCOORD *) 0) {
		xinfo_p = find_coord(prev_coord_p->hor_p);
		yinfo_p = find_coord(prev_coord_p->vert_p);
		if (xinfo_p == (struct COORD_INFO *) 0
				|| yinfo_p == (struct COORD_INFO *) 0) {
			return;
		}

		if (is_invis(xinfo_p) == YES || is_invis(yinfo_p) == YES) {
			/* things with invisible points are ignored */
			return;
		}
	}

	xinfo_p = find_coord(coord_p->hor_p);
	yinfo_p = find_coord(coord_p->vert_p);
	if (xinfo_p == (struct COORD_INFO *) 0
				|| yinfo_p == (struct COORD_INFO *) 0) {
		return;
	}

	if (is_invis(xinfo_p) == YES || is_invis(yinfo_p) == YES) {
		return;
	}

	x = inpc_x(coord_p, (char *) 0, -1);
	y = inpc_y(coord_p, mll_p->inputfile, mll_p->inputlineno);
	prev_x = prev_y = 0.0;  /* avoid bogus "used before set" warning */

	/* Check for points being too close together. If user specifies the
	 * same point for both endpoints of a line, or something like that,
	 * PostScript might get asked to divide by zero. */
	if (prev_coord_p != (struct INPCOORD *) 0) {
		prev_x = inpc_x(prev_coord_p, mll_p->inputfile, mll_p->inputlineno);
		prev_y = inpc_y(prev_coord_p, mll_p->inputfile, mll_p->inputlineno);

		if ( (fabs(x - prev_x) < .0001) && (fabs(y - prev_y) < .0001)) {
			l_ufatal(mll_p->inputfile, mll_p->inputlineno,
				"points too close together");
		}
	}
	
	/* Find the x value and see if it is off the right of the page.
	 * Pretend we don't know the file/lineno, because that way if it is
	 * off the page, no error message will be printed, which is what we
	 * want, since we hope to be able to patch things up so it isn't
	 * off the page anymore. */
	if (x < PGWIDTH) {
		/* this one is okay as is */
		return;
	}

	/* Get the staff associated with the y */
	staffno = eff_staff(yinfo_p);

	/* Find the pseudo bar of the next score and the y of the staff */
	for (m_p = xinfo_p->mll_feed_p->next; m_p != (struct MAINLL *) 0;
						m_p = m_p->next) {

		if (m_p->str == S_STAFF && m_p->u.staff_p->staffno == staffno) {
			staff_y = m_p->u.staff_p->c[AY];
		}

		if (IS_CLEFSIG_FEED(m_p)) {
			/* pseudo-bar will be in CLEFSIG right after this */
			break;
		}
	}
	if (m_p == (struct MAINLL *) 0) {
		/* no future score. Give up trying to fix this one */
		return;
	}

	/* Use the pseudo-bar as reference */
	bar_p = m_p->next->u.clefsig_p->bar_p;

	/* If there was a previous point, we will use that as a reference
	 * point, otherwise make a temporary point that is the same
	 * as the current point but with no x or y offset. */
	if (prev_coord_p == (struct INPCOORD *) 0) {
		temp_coord = *coord_p;
		temp_coord.hsteps = 0.0;
		temp_coord.counts = 0.0;
		temp_coord.vsteps = 0.0;
		prev_coord_p = &temp_coord;
		prev_x = inpc_x(prev_coord_p, mll_p->inputfile, mll_p->inputlineno);
		prev_y = inpc_y(prev_coord_p, mll_p->inputfile, mll_p->inputlineno);
	}

	/* Use the pseudo-bar for y */
	coord_p->vert_p = bar_p->c;
	coord_p->vtype = AY;

	/* Pretend to draw a line from reference point to the current point,
	 * and calculate, using similar triangles, what the y would be
	 * where that line would hit the right margin. */
	right_margin_x = PGWIDTH - eff_rightmargin(m_p);
	y = prev_y + ((y - prev_y) * (right_margin_x - prev_x)) / (x - prev_x);
	/* Now adjust to be relative to the staff's y */
	y = y - staff_y;

	/* on the following score, where we are moving the INPCOORD,
	 * find the y of the appropriate staff */
	for (m_p = m_p->next; m_p != (struct MAINLL *) 0; m_p = m_p->next) {
		if (m_p->str == S_STAFF && m_p->u.staff_p->staffno == staffno) {
			/* The new y for the INPCOORD is the staff's y offset
			 * from the score's y, plus the relative offset
			 * found above. */
			y += m_p->u.staff_p->c[AY] - bar_p->c[AY];
			break;
		}
		if (m_p->str == S_BAR) {
			l_ufatal(mll_p->inputfile, mll_p->inputlineno,
				"curve is associated with staff %d, which does not exists", staffno);
		}
	}

	coord_p->vsteps = y / (STEPSIZE * svpath(staffno, STAFFSCALE)->staffscale);

	/* change the INPCOORD x to point to the pseudo-bar's coord array */
	coord_p->hor_p = bar_p->c;
	coord_p->htype = AX;
	coord_p->counts = 0.0;
	/* x offset will be the excess that sticks beyond the right edge
	 * of the score */
	coord_p->hsteps = (x - right_margin_x)
			/ (STEPSIZE * svpath(staffno, STAFFSCALE)->staffscale);

	/* If the original was really, really far off the page, even the
	 * moved version may still be off the page, so try again. Eventually
	 * either we should get within the current score or run off the end
	 * of the song and have to give up. */
	adj_coord(coord_p, mll_p, prev_coord_p);
}


/* For manual curves that use "bulge" values, figure out the intermediate
 * points for curves, and put them in the coordlist, getting rid of the
 * bulgelist.
 */

static void
calc_bulge(curve_p, fname, lineno, is_split, mll_p)

struct CURVE *curve_p;	/* curve defined using bulge */
char *fname;
int lineno;
int is_split;	/* YES if goes across at least one FEED */
struct MAINLL *mll_p;	/* for finding effective margin */

{
	double x1, y1;			/* start point location */
	double x2, y2;			/* end point location */
	double xlen, ylen;		/* distances between endpoints */
	double seg1xlen, seg2xlen;	/* lengths of parts of curve on
					 * first and last score when split */
	double sintheta, costheta;	/* for rotation */
	double length;			/* between endpoints */
	double segX, segY;		/* distance to intermediate point */
	int p1staff, p2staff;		/* staff associated with each point */
	struct INPCOORD *coordlist_p;	/* the new calculated curve */
	int n;				/* index through bulge points */
	int nbulge;			/* how many bulge points specified */
	double staffscale;
	struct COORD_INFO *x1info_p, *y1info_p;	/* to get staff info, etc */
	struct COORD_INFO *x2info_p, *y2info_p;


	nbulge = curve_p->nbulge;

	/* The calculated curve will have the 2 endpoints plus nbulge
	 * intermediate points */
	MALLOC (INPCOORD, coordlist_p, 2 + nbulge);

	/* The endpoints just get copied. All the inner points will
	 * be calculated relative to the first, so for now we copy all of
	 * the INPCOORD data from the first point into them, then later
	 * we will overwrite the offset values appropriately. */
	for (n = 0; n < nbulge + 1; n++) {
		coordlist_p[n] = curve_p->coordlist[0];
	}
	coordlist_p[nbulge + 1] = curve_p->coordlist[1];

	/* Find relevant information about the endpoints */
	x1 = inpc_x( &(curve_p->coordlist[0]), fname, lineno);
	y1 = inpc_y( &(curve_p->coordlist[0]), fname, lineno);
	x2 = inpc_x( &(curve_p->coordlist[1]), fname, lineno);
	y2 = inpc_y( &(curve_p->coordlist[1]), fname, lineno);
	
	x1info_p = find_coord( curve_p->coordlist[0].hor_p);
	y1info_p = find_coord( curve_p->coordlist[0].vert_p);
	x2info_p = find_coord( curve_p->coordlist[1].hor_p);
	y2info_p = find_coord( curve_p->coordlist[1].vert_p);

	/* If all coordinates associated with staffs having the same
	 * staffscale value, then use that, otherwise use score value
	 */
	if ((staffscale = svpath(x1info_p->staffno, STAFFSCALE)->staffscale) !=
			svpath(y1info_p->staffno, STAFFSCALE)->staffscale
			|| staffscale !=
			svpath(x2info_p->staffno, STAFFSCALE)->staffscale
			|| staffscale !=
			svpath(y2info_p->staffno, STAFFSCALE)->staffscale) {
		staffscale = Score.staffscale;
	}

	/* Find the length of the line segment
	 * that would go straight between the two endpoints. To do this,
	 * we get the x and y distances to use with Pythagorean theorem */
	if (is_split == NO) {
		/* if all on same score, easier to find a and y */
		xlen = (x2 - x1);
		ylen = (y2 - y1);
	}
	else {
		/* Split curves take more work. First find x length
		 * on the score containing the first part of the curve */
		seg1xlen = PGWIDTH - eff_rightmargin(mll_p) -
			inpc_x( &(curve_p->coordlist[0]), fname, lineno );

		/* Find the x length of the score containing the last past.
		 * To do this, have to find the pseudo-bar inside the
		 * appropriate CLEFSIG, which should be on the main list
		 * immediately following the FEED
		 * of the score containing the ending x coordinate. */
		seg2xlen = inpc_x( &(curve_p->coordlist[1]), fname, lineno) -
			x2info_p->mll_feed_p->next->u.clefsig_p->bar_p->c[AX];

		/* Finally, add in the x lengths of any intervening scores */
		xlen = find_effXlength(seg1xlen, seg2xlen,
						x1info_p, x2info_p, NO);

		/* Now we need the distance in the y direction. First, the
		 * easy case, when both endpoints are associated with the
		 * same staff */
		p1staff = eff_staff(y1info_p);
		p2staff = eff_staff(y2info_p);
		if (p1staff == p2staff) {
			/* y length is the relative Y of the INPCOORD
			 * of the final point, offset by its vsteps, minus
			 * the similar value of the beginning point. */
			ylen = (curve_p->coordlist[1].vert_p
				[abs2rel(curve_p->coordlist[1].vtype)]
				+ curve_p->coordlist[1].vsteps
				* STEPSIZE * staffscale)
				- (curve_p->coordlist[0].vert_p
				[abs2rel(curve_p->coordlist[0].vtype)]
				+ curve_p->coordlist[0].vsteps
				* STEPSIZE * staffscale);
		}
		else {
			/* The endpoints are associated with different staffs.
			 * The y distance between these two staffs may vary
			 * from score to score, so to get things really
			 * accurate, we'd have to adjust the y proportionally
			 * on each intervening score, which may require adding
			 * lots of intermediate points and lots of complicated
			 * calculations which may or may not look much better
			 * than doing something more simple. So just do
			 * something fairly simple: Find the distance between
			 * each endpoint's y and its staff's y, and subtract
			 * those two distances to get an approximate ylen.
			 * As long as the distance between the two staffs is
			 * somewhat similar on both scores,
			 * which is likely to be the case,
			 * the results should be pretty good. */
			ylen = (y1info_p->mll_feed_p->next->u.clefsig_p->bar_p->c[AY] - y1) -
				(y2info_p->mll_feed_p->next->u.clefsig_p->bar_p->c[AY] - y2);
		}
	}

	/* Find distance between the endpoints */
	length = sqrt( SQUARED(xlen) + SQUARED(ylen) );

	/* Guard again divide by zero */
	if (length < 0.0001) {
		l_ufatal(fname, lineno, "curve endpoints too close together");
	}

	/* We find the intermediate points as if the line were horizontal,
	 * then rotate it, so need the sine and cosine for rotation.
	 */
	sintheta = ylen / length;
	costheta = xlen / length;

	/* Calculate the position of each inner point.  */
	for (n = 1; n <= nbulge; n++) {
		/* horizontal offset is based on a fraction of the length
		 * between endpoints: 1/2 if there is one bulge value,
		 * 1/3 and 2/3 if there are two values, etc, so use
		 * n/(nbulge + 1) to get the unrotated x lengths.
		 * Use the bulge value (which is already in stepsizes)
		 * for the unrotated y length.
		 * Then to do the rotation, use
		 * x' = x costheta - y sintheta
		 * y' = y costheta + x sintheta
		 */
		segX = (length * ((double) n / (double)(nbulge + 1)))
						/ (STEPSIZE * staffscale);
		segY = curve_p->bulgelist[n-1];
		coordlist_p[n].hsteps += (segX * costheta) - (segY * sintheta);
		coordlist_p[n].vsteps += (segY * costheta) + (segX * sintheta);
	}

	/* free the old coord list, which just had the endpoints */
	FREE(curve_p->coordlist);

	/* replace the old coordlist with the newly calculated one */
	curve_p->coordlist = coordlist_p;
	curve_p->ncoord = 2 + nbulge;

	/* don't need bulgelist anymore, since it has been converted to
	 * regular curve. */
	FREE(curve_p->bulgelist);
	curve_p->bulgelist = 0;
	curve_p->nbulge = 0;
}
