/* Copyright (c) 1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005 by Arkkra Enterprises */
/* All rights reserved */
/*
 * Name:	restsyl.c
 *
 * Description:	This file contains functions for setting the relative
 *		horizontal and vertical coordinates of all groups that
 *		contain a rest or space (grpcont != GC_NOTES), and the relative
 *		horizontal coordinates of syllables.  It then completes
 *		the relative horizontal work by setting the relative
 *		horizontal coords of chords.  But before it does that last
 *		step, it scales all the relative coords set so far according
 *		to staffscale.
 */

#include "defines.h"
#include "structs.h"
#include "globals.h"

#define PK_LEFT		(0)
#define PK_RIGHT	(1)
#define PK_CENTER	(2)
#define PK_NONE		(3)

static double highcoord P((struct GRPSYL *gs_p, struct GRPSYL *altgs_p));
static double lowcoord P((struct GRPSYL *gs_p, struct GRPSYL *altgs_p));
static void procrests P((struct CHORD *ch_p, struct STAFF *staff_p,
		struct GRPSYL *gs_p, double limhigh, double limlow));
static void procspaces P((struct GRPSYL *gs_p));
static struct GRPSYL *finalgroupproc P((struct GRPSYL *gs1_p));
static int v3pack P((struct GRPSYL *g_p[], int numgrps));
static void fixclef P((struct GRPSYL *gs1_p));
static void restsize P((struct GRPSYL *gs_p, float *wid_p, float *asc_p,
		float *des_p));
static void procsyls P((struct GRPSYL *gs_p));
static void apply_staffscale P((void));
static void relxchord P((void));
static double effwest P((struct MAINLL *mainll_p, struct CHORD *ch_p,
		struct GRPSYL *gs_p));
static double effeast P((struct CHORD *ch_p, struct GRPSYL *gs_p));
static struct CHORD *prevchord P((struct MAINLL *mainll_p, struct CHORD *ch_p));
static struct GRPSYL *nextchsyl P((struct GRPSYL *gs_p, struct CHORD *ch_p));
static struct GRPSYL *prevchsyl P((struct GRPSYL *gs_p,
		struct CHORD *prevch_p));
static void pedalroom P((void));
static struct CHORD *closestchord P((double count, struct CHORD *firstch_p));
static double rightped P((int pedstyle, int pedchar));
static double leftped P((int pedstyle, int pedchar));
static void fixspace P((void));

/*
 * Name:        restsyl()
 *
 * Abstract:    Sets all relative coords for rests and spaces, and horizontal
 *		ones for syllables; set relative horz. coords of chords.
 *
 * Returns:     void
 *
 * Description: This function loops through the main linked list, finding every
 *		STAFF structure.  For groups, it calls procrests() to process
 *		every rest in the linked list, and procspaces() to process
 *		every space in the linked list.  For syllables, it calls
 *		procsyls().  At the end it calls apply_staffscale() to apply
 *		staffscale to all the relative coords set so far, and then
 *		relxchord() to set the relative horizontal coords of chords.
 */

void
restsyl()

{
	register struct MAINLL *mainll_p; /* point item in main linked list */
	struct MAINLL *mll_p;		/* another MLL pointer */
	struct STAFF *staff_p;		/* point at a staff */
	struct STAFF *stafflist[MAXSTAFFS + 1];	/* point to this meas's staffs*/
	float limhigh[MAXSTAFFS + 1];	/* high y coord of limit of groups */
	float limlow[MAXSTAFFS + 1];	/* low y coord of limit of groups */
	int vscheme;			/* voice scheme */
	int v;				/* index into verse headcell array */
	struct CHORD *ch_p;		/* point at a chord */
	struct GRPSYL *gs1_p;		/* point at a group */


	debug(16, "restsyl");
	initstructs();


	/*
	 * Loop down the main linked list looking for each chord list
	 * headcell.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		if (mainll_p->str == S_SSV) {
			asgnssv(mainll_p->u.ssv_p);
			continue;
		}

		if (mainll_p->str != S_CHHEAD)
			continue;	/* skip everything but chord HC */

		/*
		 * For each visible staff in this measure, find the high and
		 * low limits of the relevant voices, so that we know what to
		 * avoid when placing rests in other voices.
		 */
		for (mll_p = mainll_p->next; mll_p->str == S_STAFF;
				mll_p = mll_p->next) {

			staff_p = mll_p->u.staff_p;
			stafflist[staff_p->staffno] = staff_p;
			vscheme = svpath(staff_p->staffno, VSCHEME)->vscheme;
			if (staff_p->visible == NO) {
				continue;
			}

			/*
			 * If there is more than one voice, each voice's rests
			 * have to worry about avoiding other voices' notes.
			 * So find how high voice 2 and how low voice 1, get,
			 * in this measure.  But voice 3 can "stand in" for
			 * either, so pass that in too.  If it's null, that's
			 * okay, the subroutines handle that.
			 */
			if (vscheme != V_1) {
				limhigh[staff_p->staffno] =
					highcoord(staff_p->groups_p[1],
					staff_p->groups_p[2]);
				limlow[staff_p->staffno] =
					lowcoord(staff_p->groups_p[0],
					staff_p->groups_p[2]);
			}
		}

		/*
		 * Loop through each chord in this list.
		 */
		for (ch_p = mainll_p->u.chhead_p->ch_p; ch_p != 0;
					ch_p = ch_p->ch_p) {
			/*
			 * Loop through the linked list of GRPSYLs hanging off
			 * this chord.  Skip the syllables; just deal with the
			 * groups.  Upon finding the first group on a staff
			 * (which could be for any of the voices, since not all
			 * might be present in this chord), call procrests and
			 * finalgroupproc to process the groups.
			 */
			gs1_p = ch_p->gs_p;
			for (;;) {
				/* find first group on a staff */
				while (gs1_p != 0 &&
						gs1_p->grpsyl == GS_SYLLABLE)
					gs1_p = gs1_p->gs_p;
				if (gs1_p == 0)
					break;

				/*
				 * Call procrests() to place any rest in the
				 * voice on this staff in this chord.
				 */
				procrests(ch_p, stafflist[gs1_p->staffno],
						gs1_p, limhigh[gs1_p->staffno],
						limlow[gs1_p->staffno]);

				/* set gs1_p to after this staff's groups */
				gs1_p = finalgroupproc(gs1_p);
			}
		}
	}

	initstructs();

	/*
	 * Loop once for each item in the main linked list.  Now that we're all
	 * done with notes and rests, do the spaces and syllables.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		if (mainll_p->str == S_SSV) {

			asgnssv(mainll_p->u.ssv_p);

		} else if (mainll_p->str == S_STAFF &&
					mainll_p->u.staff_p->visible == YES) {

			/* for each voice that exists, process the spaces */
			for (v = 0; v < MAXVOICES; v++) {
				if (mainll_p->u.staff_p->groups_p[v] != 0) {
					procspaces(mainll_p->u.staff_p->
							groups_p[v]);
				}
			}

			/* set relative horizontal coords for syllables */
			for (v = 0; v < mainll_p->u.staff_p->nsyllists; v++) {
				procsyls(mainll_p->u.staff_p->syls_p[v]);
			}
		}
	}

	/* scale all relative coords set so far according to staffscale */
	apply_staffscale();

	/* now we are ready to set relative horizontal coords for all chords */
	relxchord();
}

/*
 * Name:        highcoord()
 *
 * Abstract:    Find highest relative y coord of a group in one GRPSYL list.
 *
 * Returns:     void
 *
 * Description: This function goes down one of the linked lists of GRPSYLs,
 *		one that is for groups, not syllables, and finds the highest
 *		relative y coordinate of any group containing notes.  If there
 *		are no notes but there are rests, it returns 0.  If there are
 *		only spaces, it returns -100.  The answer, though, is rounded
 *		off to the nearest staff line.  Besides the primary linked list
 *		of GRPSYLs, it also looks for GRPSYLs in the alternate list
 *		(voice 3 if it exists) and considers ones that are "standing in"
 *		for the first list's voice.
 */

static double
highcoord(gs_p, altgs_p)

register struct GRPSYL *gs_p;	/* starts pointing at first GRPSYL in list */
register struct GRPSYL *altgs_p;/* first GRPSYL of voice 3, if any */

{
	float result;
	int normvoice;		/* main voice we are dealing with, 1 or 2 */
	float edge;		/* of a group in the other voice */


	debug(32, "highcoord file=%s line=%d", gs_p->inputfile,
			gs_p->inputlineno);
	result = -100;		/* init as if only spaces */
	normvoice = gs_p->vno;	/* remember the voice we're dealing with */

	/* 
	 * Loop through all groups (even grace), moving result up when
	 * something higher is found.  Rests count as 0 (the middle line).
	 */
	for ( ; gs_p != 0; gs_p = gs_p->next) {
		switch (gs_p->grpcont) {
		case GC_NOTES:
			edge = gs_p->c[RN];
			/* if wrong way stem, account for it as best we can */
			if (gs_p->stemdir == UP) {
				edge += (stemroom(gs_p) - 1.0) * STEPSIZE;
			}
			if (edge > result) {
				result = edge;
			}
			break;
		case GC_REST:
			if (result < 0)
				result = 0;
			break;
		/* ignore spaces */
		}
	}

	/*
	 * Look at every GRPSYL in voice 3, if any.  If it is "standing in" for
	 * the normal voice, move result up if need be.
	 */
	for ( ; altgs_p != 0; altgs_p = altgs_p->next) {
		if (altgs_p->pvno == normvoice) {
			switch (altgs_p->grpcont) {
			case GC_NOTES:
				if (altgs_p->c[RN] > result)
					result = altgs_p->c[RN];
				break;
			case GC_REST:
				if (result < 0)
					result = 0;
				break;
			/* ignore spaces */
			}
		}
	}

	return (nearestline(result));
}

/*
 * Name:        lowcoord()
 *
 * Abstract:    Find lowest relative y coord of a group in one GRPSYL list.
 *
 * Returns:     void
 *
 * Description: This function goes down one of the linked lists of GRPSYLs,
 *		one that is for groups, not syllables, and finds the lowest
 *		relative y coordinate of any group containing notes.  If there
 *		are no notes but there are rests, it returns 0.  If there are
 *		only spaces, it returns 100.  The answer, though, is rounded
 *		off to the nearest staff line.  Besides the primary linked list
 *		of GRPSYLs, it also looks for GRPSYLs in the alternate list
 *		(voice 3 if it exists) and considers ones that are "standing in"
 *		for the first list's voice.
 */

static double
lowcoord(gs_p, altgs_p)

register struct GRPSYL *gs_p;	/* starts pointing at first GRPSYL in list */
register struct GRPSYL *altgs_p;/* first GRPSYL of voice 3, if any */

{
	float result;
	int normvoice;		/* main voice we are dealing with, 1 or 2 */
	float edge;		/* of a group in the other voice */


	debug(32, "lowcoord file=%s line=%d", gs_p->inputfile,
			gs_p->inputlineno);
	result = 100;		/* init as if only spaces */
	normvoice = gs_p->vno;	/* remember the voice we're dealing with */

	/* 
	 * Loop through all groups (even grace), moving result down when
	 * something lower is found.  Rests count as 0 (the middle line).
	 */
	for ( ; gs_p != 0; gs_p = gs_p->next) {
		switch (gs_p->grpcont) {
		case GC_NOTES:
			edge = gs_p->c[RS];
			/* if wrong way stem, account for it as best we can */
			if (gs_p->stemdir == DOWN) {
				edge -= (stemroom(gs_p) - 1.0) * STEPSIZE;
			}
			if (edge < result) {
				result = edge;
			}
			break;
		case GC_REST:
			if (result > 0)
				result = 0;
			break;
		/* ignore spaces */
		}
	}

	/*
	 * Look at every GRPSYL in voice 3, if any.  If it is "standing in" for
	 * the normal voice, move result up if need be.
	 */
	for ( ; altgs_p != 0; altgs_p = altgs_p->next) {
		if (altgs_p->pvno == normvoice) {
			switch (altgs_p->grpcont) {
			case GC_NOTES:
				if (altgs_p->c[RS] < result)
					result = altgs_p->c[RS];
				break;
			case GC_REST:
				if (result > 0)
					result = 0;
				break;
			/* ignore spaces */
			}
		}
	}

	return (nearestline(result));
}

/*
 * Name:        procrests()
 *
 * Abstract:    Sets relative coordinates for rests in one CHORD/STAFF.
 *
 * Returns:     void
 *
 * Description: This function is given the top GRPSYL on a STAFF in a CHORD.
 *		It sets the relative coordinates of each rest GRPSYL on this
 *		STAFF/CHORD.
 */

static void
procrests(ch_p, staff_p, gs1_p, limhigh, limlow)

struct CHORD *ch_p;		/* the chord we are in */
struct STAFF *staff_p;		/* the staff we are processing */
struct GRPSYL *gs1_p;		/* point at top GRPSYL in chord */
double limhigh;			/* highest relative y coord below v1 */
double limlow;			/* lowest relative y coord above v2 */

{
	RATIONAL endtime;	/* time at the end of the rest */
	struct GRPSYL *g_p[MAXVOICES + 1]; /* index by vno, point at GRPSYL */
	struct GRPSYL *gs_p;	/* a GRPSYL we are now working on */
	struct GRPSYL *ogs_p;	/* other voice to be considered */
	float wid, asc, des;	/* width, ascent, and descent of a rest */
	int vscheme;		/* voice scheme */
	int restsabove;		/* are these rests above another voice? */
	int stafflines;		/* no. of lines in staff */
	int v;			/* voice number */
	float y;		/* relative y coord for this rest */


	debug(32, "procrests file=%s line=%d limhigh=%f limlow=%f",
		gs1_p->inputfile, gs1_p->inputlineno,
		(float)limhigh, (float)limlow);


	/* get voice scheme and number of lines in staff */
	vscheme = svpath(gs1_p->staffno, VSCHEME)->vscheme;
	stafflines = svpath(gs1_p->staffno, STAFFLINES)->stafflines;

	/* set pointers to all nonspace groups in this chord on this staff */
	for (v = 1; v <= MAXVOICES; v++) {
		g_p[v] = 0;
	}
	for (gs_p = gs1_p; gs_p != 0 && gs_p->staffno == gs1_p->staffno &&
			    gs_p->grpsyl == GS_GROUP; gs_p = gs_p->gs_p) {
		if (gs_p->grpcont != GC_SPACE) {
			g_p[gs_p->vno] = gs_p;
		}
	}

	y = 0.0;	/* to avoid useless 'used before set' warning */
	ogs_p = 0;	/* to avoid useless 'used before set' warning */

	/*
	 * Loop through each possible voice, setting its coords if it is a rest.
	 */
	for (v = 1; v <= MAXVOICES; v++) {

		gs_p = g_p[v];

		if (gs_p == 0 || gs_p->grpcont != GC_REST) {
			continue;
		}

		/* find the time at the end of the rest */
		endtime = radd(ch_p->starttime, gs_p->fulltime);

		/* find width, ascent, and descent of the rest */
		restsize(gs_p, &wid, &asc, &des);

		/*
		 * Find out if another voice needs to be considered in the
		 * placement of this rest, and set ogs_p to that voice's first
		 * GRPSYL if so.
		 */
		ogs_p = 0;
		if (vscheme == V_2FREESTEM ||
				vscheme == V_3FREESTEM && gs_p->pvno != 3) {
			ogs_p = gs_p->pvno == 1 ? staff_p->groups_p[1] :
						  staff_p->groups_p[0];
		}

		/* 
		 * Find the RY of the rest.
		 */
		if (vscheme == V_1 || (vscheme == V_2FREESTEM ||
				vscheme == V_3FREESTEM) &&
				hasspace(ogs_p, ch_p->starttime, endtime)) {
			/*
			 * There is either only 1 voice, or we are 2f/3f and the
			 * other voice is all spaces during this time.  Usually
			 * RY should be 0.  But for one-line staffs, whole
			 * rest characters need to be lowered so that they hang
			 * under the line.
			 */
			if (stafflines == 1 && gs_p->basictime == 1) {
				y = -2 * STEPSIZE;
			} else {
				y = 0;
			}
		} else {
			/*
			 * We are 2o, or 2f/3f with notes/rests in the other
			 * voice that we must avoid hitting.  Set up the
			 * relative y coord, based on whether gs_p is acting
			 * as v1 or v2.  We also have to set up restsabove
			 * for use below.
			 */
			restsabove = NO;	/* default value for now */
			switch (gs_p->pvno) {
			case 1:
				y = limhigh < -4 * STEPSIZE ?
						0 : limhigh + 4 * STEPSIZE;
				restsabove = ! hasspace(staff_p->groups_p[1],
						Zero, Maxtime);
				/* also check for v3 groups acting as v2 */
				for (ogs_p = staff_p->groups_p[2];
						ogs_p != 0 && restsabove == NO;
						ogs_p = ogs_p->next) {
					if (ogs_p->pvno == 2 &&
					ogs_p->grpcont != GC_SPACE) {
						restsabove = YES;
						break;
					}
				}
				break;
			case 2:
				y = limlow > 4 * STEPSIZE ?
						0 : limlow - 4 * STEPSIZE;
				break;
			}

			/*
			 * Usually RY should be the y was set above.  But
			 * if this is the upper voice, half rests and longer
			 * should be lower to fall within the staff when
			 * feasible, since they don't take much space
			 * vertically and we don't want needless ledger lines.
			 * (But nothing should ever be lowered if already on
			 * the center line.)  Short rests need to be moved
			 * away from the other voice by varying amounts,
			 * depending on how tall they are.  Quad whole rests
			 * below need to be raised a notch.
			 */
			if (restsabove == YES) {
				/* lower whole & double only if above middle */
				if (gs_p->basictime <= 2 && y > 0)
					y -= 2 * STEPSIZE;
				if (gs_p->basictime >= 16)
					y += 2 * STEPSIZE;
				if (gs_p->basictime == 256)
					y += 2 * STEPSIZE;
			} else {
				if (gs_p->basictime >= 128)
					y -= 2 * STEPSIZE;
				if (gs_p->basictime == -1)
					y += 2 * STEPSIZE;
			}
		}

		/*
		 * If restdist was set by the user, use that instead of
		 * whatever we calculated above.
		 */
		if (gs_p->restdist != NORESTDIST) {
			y = gs_p->restdist * STEPSIZE;
		}

		/* set all the relative coords */
		gs_p->c[RX] = 0;
		gs_p->c[RE] = wid / 2;
		gs_p->c[RW] = -wid / 2 - gs_p->padding -
				vvpath(gs_p->staffno, gs_p->vno, PAD)->pad;
		gs_p->c[RY] = y;
		gs_p->c[RN] = y + asc;
		gs_p->c[RS] = y - des;

		/* if there are dot(s), add their widths to the east side */
		if (gs_p->dots > 0) {
			gs_p->c[RE] += gs_p->dots * (width(FONT_MUSIC,
					DFLT_SIZE, C_DOT) + 2 * STDPAD);
		}
	}
}

/*
 * Name:        procspaces()
 *
 * Abstract:    Sets relative coordinates for spaces in one GRPSYL list.
 *
 * Returns:     void
 *
 * Description: This function goes down one of the linked lists of GRPSYLs,
 *		one that is for groups, not syllables, and sets the relative
 *		coordinates for each space found.  Usually these coords will
 *		be left as 0, the way they were calloc'ed, but not when there
 *		is padding or uncompressible spaces.
 */

static void
procspaces(gs_p)

register struct GRPSYL *gs_p;	/* starts pointing at first GRPSYL in list */

{
	static float half_us_width;	/* half width of uncompressible space*/
	char headchar;			/* char representing a note head */
	int headfont;			/* music font for head char */


	/*
	 * Loop, setting all relative coords of spaces, except that if they are
	 * to be zero there's no need to set them, since calloc zeroed them.
	 * The vertical ones are always zero, and so is RX.
	 */
	for ( ; gs_p != 0; gs_p = gs_p->next) {
		if (gs_p->grpcont != GC_SPACE)
			continue;

		if (gs_p->uncompressible == YES) {
			/*
			 * If this is the first time in here, set this to half
			 * a blank quarter note head plus standard pad.
			 */
			if (half_us_width == 0.0) {
				headchar = nheadchar(get_shape_num("blank"),
						4, UP, &headfont);
				half_us_width = width(headfont, DFLT_SIZE,
						headchar) / 2.0 + STDPAD;
			}

			/* center the imaginary note head */
			gs_p->c[RE] = half_us_width / 2;
			gs_p->c[RW] = -half_us_width / 2;

			/* apply global user requested padding; notice that
			 * normal spaces (s) don't get this */
			gs_p->c[RW] -= vvpath(gs_p->staffno,
						gs_p->vno, PAD)->pad;
		}

		/* add any user requested padding */
		gs_p->c[RW] -= gs_p->padding;
	}
}

/*
 * Name:        finalgroupproc()
 *
 * Abstract:    Do final processing of groups.
 *
 * Returns:     pointer to the first GRPSYL after these groups, 0 if none
 *
 * Description: This function is given the GRPSYL for the first (topmost) voice
 *		that is on this staff in this chord.  It find what other
 *		GRPSYLs exist.  For all the nonspace groups, it applies any
 *		horizontal offsets needed.
 */

static struct GRPSYL *
finalgroupproc(gs1_p)

struct GRPSYL *gs1_p;		/* first voice on this staff in this chord */

{
	struct GRPSYL *g_p[MAXVOICES];	/* point at nonspace voices' groups */
	struct GRPSYL *gs_p;		/* point at groups in the chord */
	struct GRPSYL *last_p;		/* point at last nonspace group */
	int numgrps;			/* how many nonspace groups are here */
	int staffno;			/* staff these groups are on */
	int n;				/* loop variable */
	float offset;			/* of each when + and - are used */
	float edge;			/* west or east coord of group 1 or 2 */
	int pack;			/* optimization for voice 3 */


	staffno = gs1_p->staffno;	/* remember staffno of first group */
	numgrps = 0;			/* no groups found yet */
	last_p = 0;			/* no last nonspace group yet */

	/* find all groups in this chord on this staff; remember nonspaces */
	for (gs_p = gs1_p; gs_p != 0 && gs_p->staffno == staffno &&
			    gs_p->grpsyl == GS_GROUP; gs_p = gs_p->gs_p) {
		if (gs_p->grpcont != GC_SPACE) {
			g_p[numgrps++] = gs_p;
			last_p = gs_p;
		}
	}

	/*
	 * If all groups on this staff were spaces, just make sure clef is
	 * marked correctly and return, though it's unlikely we have a clef
	 * change before a space.
	 */
	if (numgrps == 0) {
		fixclef(gs1_p);
		return (gs_p);
	}

	/* nothing to do for tab, since "ho" is ignored and rests invisible */
	if (is_tab_staff(g_p[0]->staffno))
		return (gs_p);

	/* for any voice with a user supplied offset value, apply it now */
	for (n = 0; n < numgrps; n++) {
		if (g_p[n]->ho_usage == HO_VALUE)
			shiftgs(g_p[n], g_p[n]->ho_value * STEPSIZE);
	}

	/*
	 * If both voices 1 and 2 are nonspace, handle any ho "+" or "-".
	 */
	if (numgrps >= 2 && g_p[0]->vno == 1 && g_p[1]->vno == 2) {
		/*
		 * Verify and fix offsets.  We did this in setgrps.c for note
		 * groups so that compatible note groups could then be handled
		 * together.  But we need to check again in case rest groups
		 * are involved.
		 */
		vfyoffset(g_p);

		/*
		 * Check each of these 2 groups:  If it has "+" or "-" and the
		 * other one doesn't, shift it to be next to the other one on
		 * the appropriate side.
		 */
		for (n = 0; n < 2; n++) {
			if ((g_p[n]->ho_usage == HO_LEFT ||
			     g_p[n]->ho_usage == HO_RIGHT) &&
			  ! (g_p[1-n]->ho_usage == HO_LEFT ||
			     g_p[1-n]->ho_usage == HO_RIGHT)) {

				if (g_p[n]->ho_usage == HO_LEFT) {
					shiftgs(g_p[n],
					g_p[1-n]->c[RW] - g_p[n]->c[RE]);
				} else {
					shiftgs(g_p[n],
					g_p[1-n]->c[RE] - g_p[n]->c[RW]);
				}
			}
		}

		/*
		 * If one has "+" and one has "-", shift them each by half of
		 * the amount of space needed to avoid a collision.
		 */
		if (g_p[0]->ho_usage == HO_LEFT &&
		    g_p[1]->ho_usage == HO_RIGHT) {

			offset = (g_p[0]->c[RE] - g_p[1]->c[RW]) / 2.0;
			shiftgs(g_p[0], -offset);
			shiftgs(g_p[1], offset);
		}
		if (g_p[0]->ho_usage == HO_RIGHT &&
		    g_p[1]->ho_usage == HO_LEFT) {

			offset = (g_p[1]->c[RE] - g_p[0]->c[RW]) / 2.0;
			shiftgs(g_p[0], offset);
			shiftgs(g_p[1], -offset);
		}

	} else if (g_p[0]->vno != 3) {
		/*
		 * If only one of groups 1 and 2 is nonspace, check whether it
		 * has "+" or "-", and warn if so.
		 */
		if (g_p[0]->ho_usage == HO_LEFT || g_p[0]->ho_usage == HO_RIGHT)
		{
			l_warning(
				g_p[0]->inputfile, g_p[0]->inputlineno,
				"voice %d cannot have horizontal offset '%c' since voice %d is not present; ignoring it",
				g_p[0]->vno,
				g_p[0]->ho_usage == HO_LEFT ? '-' :'+',
				3 - g_p[0]->vno);

			g_p[0]->ho_usage = HO_NONE;
		}
	}

	/*
	 * If voice 3 and at least one other voice exist here, and the user
	 * didn't state an offset value for voice 3, offset it next to the
	 * other voices, on the left or right, as requested.  But exclude the
	 * case where voice 3 was being treated as 1 or 2, by checking pvno
	 * instead of vno.
	 */
	if (numgrps > 1 && last_p->pvno == 3 && last_p->ho_usage != HO_VALUE) {
		/*
		 * See if we can pack v3 tightly against v1 and v2.  (This will
		 * not be allowed if ho_usage != HO_NONE for any voice, or any
		 * other of many conditions doesn't hold true.)
		 */
		pack = v3pack(g_p, numgrps);
		if (pack != PK_NONE) {
			/*
			 * Yes, we can; shift v3 a little if necessary.  Make
			 * it so that v3's stem is one stepsize away from the
			 * group that its stem is pointing toward.
			 */
			switch (pack) {
			case PK_LEFT:
				/* since v3 is on left, v2 must exist, and is
				 * the voice preceding v3 in g_p */
				shiftgs(last_p, -STEPSIZE +
					widest_head(last_p) / 2.0 +
					g_p[numgrps-2]->c[RW]);
				break;
			case PK_RIGHT:
				/* since v3 is on right, v1 must exist, and is
				 * the first voice in g_p */
				shiftgs(last_p, STEPSIZE +
					g_p[0]->c[RE] -
					widest_head(last_p) / 2.0);
				break;
			/* for PK_CENTER, nothing to do */
			}

		} else if (last_p->ho_usage == HO_LEFT) {

			/* find leftmost edge of the other voice(s) */
			edge = g_p[0]->c[RW];
			for (n = 1; n < numgrps - 1; n++) {
				if (g_p[n]->c[RW] < edge)
					edge = g_p[n]->c[RW];
			}
			/* set right edge of voice 3 == left edge of others */
			shiftgs(last_p, edge - last_p->c[RE]);

		} else { /* HO_RIGHT, or HO_NONE which defaults to HO_RIGHT */

			/* find rightmost edge of the other voice(s) */
			edge = g_p[0]->c[RE];
			for (n = 1; n < numgrps - 1; n++) {
				if (g_p[n]->c[RE] > edge)
					edge = g_p[n]->c[RE];
			}
			/* set left edge of voice 3 == right edge of others */
			shiftgs(last_p, edge - last_p->c[RW]);
		}
	} else if (g_p[0]->vno == 3 && (g_p[0]->ho_usage == HO_LEFT ||
					g_p[0]->ho_usage == HO_RIGHT)) {
		/*
		 * If the first (and thus only) voice is 3, it should not have
		 * ho "+" or "-".
		 */
		l_warning(
			g_p[0]->inputfile, g_p[0]->inputlineno,
			"voice 3 cannot have horizontal offset '%c' since voices 1 and 2 are not present; ignoring it",
			g_p[0]->ho_usage == HO_LEFT ? '-' :'+');

		g_p[0]->ho_usage = HO_NONE;
	}

	/* in case of midmeasure clef change, make sure it's marked right */
	fixclef(gs1_p);

	/* return the first GRPSYL after the groups we processed */
	return (gs_p);
}

/*
 * Name:        v3pack()
 *
 * Abstract:    Decide whether v3 can be packed tighter than the default.
 *
 * Returns:     PK_NONE		no, it can't
 *		PK_LEFT		pack tightly on left
 *		PK_RIGHT	pack tightly on right
 *		PK_CENTER	pack in the center
 *
 * Description: This function decides whether the voice 3 group can be packed
 *		in more tightly against voices 1 and 2 than the usual default
 *		of just putting v3's group's rectangle to the right of the
 *		other voices.  If there seems to be any danger that v3 would
 *		collide with v1 or v2, it gives up and returns PK_NONE.  It
 *		could be made a lot more sophisticated and not give up so soon
 *		in many cases.  However many of these improvements can't be
 *		done very well at this stage of the game, where we don't know
 *		yet about stem lengths, beam positions, etc.
 */

static int
v3pack(g_p, numgrps)

struct GRPSYL *g_p[];		/* point at nonspace voices' groups */
int numgrps;			/* how many nonspace groups are here */

{
	struct GRPSYL *gs_p;		/* point at a group */
	struct GRPSYL *v3_p;		/* point at v3's group */
	struct NOTE *v3note_p;		/* v3 note that neighbors other voice*/
	struct NOTE *onote_p;		/* v1/v2 note that neighbors v3 */
	float north;			/* highest coord of note or acc */
	float south;			/* lowest coord of note or acc */
	float topdesc;			/* descent of acc of top group */
	float botasc;			/* ascent of acc of bottom group */
	int v3hasacc, otherhasacc;	/* do v3 and other voice have acc(s)?*/
	int pack;
	int n;				/* loop variable */
	int k;				/* loop variable */


	/* either v1 or v2 must be nonspace */
	if (numgrps == 1) {
		return (PK_NONE);
	}

	/* point at v3's group for convenience */
	v3_p = g_p[numgrps - 1];

	/* set up what the answer will be if we can apply the optimization */
	if (v3_p->basictime >= 2) {
		/* there is a stem, so offset such that stem will avoid v1/v2 */
		if (v3_p->stemdir == UP) {
			pack = PK_RIGHT;
		} else {	/* DOWN */
			pack = PK_LEFT;
		}
	} else {
		pack = PK_CENTER;	/* no stem, so we can center v3 */
	}

	/* v3 must not be standing in for v1 or v2 */
	if (v3_p->pvno != 3) {
		return (PK_NONE);
	}

	/* if v3 would be on left, it must not have a flag or be start of beam*/
	if (pack == PK_LEFT && (v3_p->basictime >= 8 && v3_p->beamloc == NOITEM
				|| v3_p->beamloc == STARTITEM)) {
		return (PK_NONE);
	}

	/* if v3 would be on right, it must not have grace groups preceding */
	if (pack == PK_RIGHT && v3_p->prev != 0 &&
				v3_p->prev->grpvalue == GV_ZERO) {
		return (PK_NONE);
	}

	/* v3 cannot have slashes or alternation */
	if (v3_p->slash_alt != 0) {
		return (PK_NONE);
	}

	/*
	 * Loop through all voices, checking for rule violations.  We do it
	 * in reverse so that we know v3 is notes (the first check) before
	 * checking the other voices.
	 */
	for (n = numgrps - 1; n >= 0; n--) {
		gs_p = g_p[n];	/* set to current voice for convenience */

		/* voice must be notes, and not measure repeat */
		if (gs_p->grpcont != GC_NOTES || gs_p->is_meas) {
			return (PK_NONE);
		}

		/* voice cannot have user requested horizontal offset */
		if (gs_p->ho_usage != HO_NONE) {
			return (PK_NONE);
		}

		/* voice cannot have a "with" list */
		if (gs_p->nwith != 0) {
			return (PK_NONE);
		}

		/* voice cannot have a roll */
		if (gs_p->roll != NOITEM) {
			return (PK_NONE);
		}

		/* do voice specific checks */
		switch (gs_p->vno) {

		case 1:
			/* stem must be up */
			if (gs_p->stemdir != UP) {
				return (PK_NONE);
			}

			/* find neighboring notes of v1 and v3 */
			v3note_p = &v3_p->notelist[0];
			onote_p = &gs_p->notelist[gs_p->nnotes - 1];

			/* neighboring notes in v1 & v3 must not be too close */
			if (onote_p->stepsup < v3note_p->stepsup + 2 ||
			onote_p->stepsup == v3note_p->stepsup + 2 &&
			pack != PK_CENTER &&
			(v3_p->basictime < 1 || gs_p->basictime < 1)) {
				return (PK_NONE);
			}

			/* if 2 steps apart and on lines and v3 would not be on
			 * right, v1 can't have dots and v3 can't unless it
			 * is on the right */
			if (onote_p->stepsup == v3note_p->stepsup + 2 &&
			EVEN(v3note_p->stepsup) &&
			(gs_p->dots != 0 ||
			pack != PK_RIGHT && v3_p->dots != 0)) {
				return (PK_NONE);
			}

			/*
			 * Find the lowest extension of any accidental in v1.
			 * If no accidentals, the initial value for "south"
			 * will remain.  It's not good enough to check accs
			 * only on the neighboring notes, because some of them
			 * stick out pretty far.  We have to go through these
			 * gyrations because group boundaries do not consider
			 * accidentals that stick out up or down.
			 */
			otherhasacc = NO;
			south = onote_p->c[RY] - STEPSIZE + 0.001;
			for (k = 0; k < gs_p->nnotes; k++) {
				accdimen(&gs_p->notelist[k], (float *)0,
						&topdesc, (float *)0);
				if (gs_p->notelist[k].c[RY] - topdesc < south) {
					south = gs_p->notelist[k].c[RY]
							- topdesc;
				}
				if (gs_p->notelist[k].accidental != '\0') {
					otherhasacc = YES;
				}
			}
			/* similarly, find highest extension of v3 accs */
			v3hasacc = NO;
			north = v3note_p->c[RY] + STEPSIZE - 0.001;
			for (k = 0; k < v3_p->nnotes; k++) {
				accdimen(&v3_p->notelist[k], &botasc,
						(float *)0, (float *)0);
				if (v3_p->notelist[k].c[RY] + botasc > north) {
					north = v3_p->notelist[k].c[RY]
							+ botasc;
				}
				if (v3_p->notelist[k].accidental != '\0') {
					v3hasacc = YES;
				}
			}
			/* if v1 and v3 overlap due to acc(s), fail */
			if (south < north) {
				switch (pack) {
				case PK_RIGHT:
					if (v3hasacc == YES) {
						return (PK_NONE);
					}
					break;
				case PK_CENTER:
					if (v3hasacc == YES &&
					    otherhasacc == YES) {
						return (PK_NONE);
					}
					break;
				case PK_LEFT:
					if (otherhasacc == YES) {
						return (PK_NONE);
					}
					break;
				}
			}

			/* if left or right offset, neighboring notes in v1 &
			 * v3 must not have parentheses when accs exist */
			if ((pack != PK_CENTER || v3hasacc == YES || otherhasacc == YES) &&
			   (v3note_p->note_has_paren || onote_p->note_has_paren)) {
				return (PK_NONE);
			}

			break;

		case 2:
			if (gs_p->stemdir != DOWN) {
				return (PK_NONE);
			}

			/* find neighboring notes of v2 and v3 */
			v3note_p = &v3_p->notelist[v3_p->nnotes - 1];
			onote_p = &gs_p->notelist[0];

			/* neighboring notes in v1 & v3 must not be too close */
			if (onote_p->stepsup > v3note_p->stepsup - 2 ||
			onote_p->stepsup == v3note_p->stepsup - 2 &&
			pack != PK_CENTER &&
			(v3_p->basictime < 1 || gs_p->basictime < 1)) {
				return (PK_NONE);
			}

			/* if 2 steps apart and on lines and v3 would not be on
			 * right, neither can have dots */
			if (onote_p->stepsup == v3note_p->stepsup - 2 &&
					EVEN(v3note_p->stepsup) &&
					pack != PK_RIGHT &&
					(gs_p->dots != 0 || v3_p->dots != 0)) {
				return (PK_NONE);
			}

			/*
			 * Find the highest extension of any accidental in v2.
			 * If no accidentals, the initial value for "north"
			 * will remain.
			 */
			otherhasacc = NO;
			north = onote_p->c[RY] + STEPSIZE - 0.001;
			for (k = 0; k < gs_p->nnotes; k++) {
				accdimen(&gs_p->notelist[k], &botasc,
						(float *)0, (float *)0);
				if (gs_p->notelist[k].c[RY] + botasc > north) {
					north = gs_p->notelist[k].c[RY]
							+ botasc;
				}
				if (gs_p->notelist[k].accidental != '\0') {
					otherhasacc = YES;
				}
			}
			/* similarly, find highest extension of v3 accs */
			v3hasacc = NO;
			south = v3note_p->c[RY] - STEPSIZE + 0.001;
			for (k = 0; k < v3_p->nnotes; k++) {
				accdimen(&v3_p->notelist[k], (float *)0,
						&topdesc, (float *)0);
				if (v3_p->notelist[k].c[RY] - topdesc < south) {
					south = v3_p->notelist[k].c[RY]
							- topdesc;
				}
				if (v3_p->notelist[k].accidental != '\0') {
					v3hasacc = YES;
				}
			}
			/* if v2 and v3 overlap due to acc(s), fail */
			if (south < north) {
				switch (pack) {
				case PK_RIGHT:
					if (v3hasacc == YES) {
						return (PK_NONE);
					}
					break;
				case PK_CENTER:
					if (v3hasacc == YES &&
					    otherhasacc == YES) {
						return (PK_NONE);
					}
					if (v3hasacc == YES &&
					gs_p->nnotes >= 2 &&
					onote_p->stepsup ==
					gs_p->notelist[1].stepsup + 1) {
						return (PK_NONE);
					}
					break;
				case PK_LEFT:
					if (otherhasacc == YES) {
						return (PK_NONE);
					}
					break;
				}
			}

			/* if left or right offset, neighboring notes in v2 &
			 * v3 must not have parentheses when accs exist */
			if ((pack != PK_CENTER || v3hasacc == YES || otherhasacc == YES) &&
			   (v3note_p->note_has_paren || onote_p->note_has_paren)) {
				return (PK_NONE);
			}

			break;
		}
	}

	/* all checks passed, so return the answer */
	return (pack);
}

/*
 * Name:        fixclef()
 *
 * Abstract:    If midmeasure clef change at this chord, mark in right GRPSYL.
 *
 * Returns:     void
 *
 * Description: This function is given the GRPSYL for the first (topmost) voice
 *		that is on this staff in this chord.  If the clef changed at
 *		this time value, locllnotes() in setnotes.c will have set the
 *		"clef" field in each of the GRPSYLs in this chord on this
 *		staff (actually in their first preceding grace group, if
 *		any).  But it should only be set in the GRPSYL that has the
 *		westernmost west boundary.  So this function erases it from
 *		any other GRPSYLs.
 */

static void
fixclef(gs1_p)

struct GRPSYL *gs1_p;		/* starts at first voice on this staff */

{
	struct GRPSYL *g_p[MAXVOICES];	/* point at voices' groups */
	struct GRPSYL *gs_p;		/* point at groups in the chord */
	int numgrps;			/* how many groups are in the chord */
	struct GRPSYL *westgs_p;	/* remember westernmost */
	int staffno;			/* staff number */
	int n;				/* loop variable */


	staffno = gs1_p->staffno;	/* remember staffno of first group */

	/* point at all groups in this chord on this staff */
	numgrps = 0;			/* no groups found yet */
	for (gs_p = gs1_p; gs_p != 0 && gs_p->staffno == staffno &&
			    gs_p->grpsyl == GS_GROUP; gs_p = gs_p->gs_p) {
		g_p[numgrps++] = gs_p;
	}

	/*
	 * For each that is preceded by grace group(s), change the pointer to
	 * point at the first in that sequence of grace groups.  Any clef
	 * change would occur at that group.
	 */
	for (n = 0; n < numgrps; n++) {
		while (g_p[n]->prev != 0 && g_p[n]->prev->grpvalue == GV_ZERO) {
			g_p[n] = g_p[n]->prev;
		}
	}

	/* if clef not marked in first, it's not marked in any, so return */
	if (g_p[0]->clef == NOCLEF) {
		return;
	}

	westgs_p = 0;	/* prevent useless "used before set" warning */

	/*
	 * Find the westernmost group of notes, if any; if it's a tie, use the
	 * first one.  We don't want to put clefs in front of rests, spaces, or
	 * mrpt, unless we have no choice.
	 */
	for (n = 0; n < numgrps; n++) {
		if (g_p[n]->grpcont == GC_NOTES && g_p[n]->is_meas == NO) {
			if (westgs_p == 0 || g_p[n]->c[RW] < westgs_p->c[RW]) {
				westgs_p = g_p[n];
			}
		}
	}
	/* we have no choice; arbitrarily choose the first voice */
	if (westgs_p == NULL) {
		westgs_p = g_p[0];
	}

	/* erase clef from all but the group found above */
	for (n = 0; n < numgrps; n++) {
		if (g_p[n] != westgs_p) {
			g_p[n]->clef = NOCLEF;
		}
	}

	/* if there were no notes, there will be no clef here */
	if (westgs_p == 0) {
		return;
	}

	/* move western boundary of GRPSYL to allow room to print the clef */
	westgs_p->c[RW] -= clefwidth(westgs_p->clef, YES) + CLEFPAD;

	/*
	 * If this is a grace group, we also have to alter its main group's
	 * boundary, because the main group's boundary needs to enclose all
	 * its grace groups.
	 */
	for (gs_p = westgs_p; gs_p->grpvalue == GV_ZERO; gs_p = gs_p->next) {
		;
	}
	if (gs_p != westgs_p) {
		gs_p->c[RW] -= clefwidth(westgs_p->clef, YES) + CLEFPAD;
	}
}

/*
 * Name:        restsize()
 *
 * Abstract:    Find the size of a rest.
 *
 * Returns:     void
 *
 * Description: This function is given a GRPSYL which is a rest.  It returns
 *		the width, ascent, and descent through pointers.
 */

static void
restsize(gs_p, wid_p, asc_p, des_p)

register struct GRPSYL *gs_p;	/* the GRPSYL containing the rest */
float *wid_p, *asc_p, *des_p;	/* return width, ascent, and descent of rest */

{
	char rchar;		/* char for the rest */
	int size;		/* font size */


	/* multirest has no music character; just return the answer */
	if (gs_p->basictime < -1) {
		*wid_p = MINMULTIWIDTH;
		*asc_p = 2 * STEPSIZE;
		*des_p = 2 * STEPSIZE;
		return;
	}

	/* on a tab staff rests are invisible, so set to a very small size */
	if (is_tab_staff(gs_p->staffno)) {
		*wid_p = *asc_p = *des_p = 0.01;
		return;
	}

	/*
	 * The "normal" rest case.  Find the name of the character.  Then get
	 * the width, ascent, and descent of the rest.
	 */
	rchar = restchar(gs_p->basictime);
	size = (gs_p->grpsize == GS_NORMAL ? DFLT_SIZE : SMALLSIZE);
	*wid_p = width(FONT_MUSIC, size, rchar);
	*asc_p = ascent(FONT_MUSIC, size, rchar);
	*des_p = descent(FONT_MUSIC, size, rchar);
}

/*
 * Name:        procsyls()
 *
 * Abstract:    Sets relative horizontal coords for syllables in 1 GRPSYL list.
 *
 * Returns:     void
 *
 * Description: This function goes down one of the linked lists of GRPSYLs,
 *		one that is for syllables, not groups, and sets the relative
 *		horizontal coordinates for each syllable found.
 */

static void
procsyls(gs_p)

register struct GRPSYL *gs_p;	/* starts pointing at first GRPSYL in list */

{
	float wid_b4_syl;	/* width of leading non-lyrics */
	float wid_real_syl;	/* width of actual lyric */
	float wid_after_syl;	/* width of trailing non-lyrics */
	float lyricsalign;	/* fraction of syl to go left of chord center*/
	int font, size;		/* of the last char in a syllable */
	char lc;		/* last char of syllable */


	debug(32, "procsyls file=%s line=%d", gs_p->inputfile,
			gs_p->inputlineno);
	/* find what fraction of each syl should go left of center of chord */
	lyricsalign = svpath(gs_p->staffno, LYRICSALIGN)->lyricsalign;

	/*
	 * Set coords for every syllable.  A syllable can consist of 3 parts.
	 * The middle part is the actual lyric.  The optional first and last
	 * parts are surrounded in the user's input by angle brackets.  The
	 * syllable is to be positioned such that "lyricsalign" of the middle
	 * part goes to the left of the chord's center, and the rest goes to
	 * the right; unless sylposition is set, in which case the left edge of
	 * the actual lyric is offset by that many points from the chord's
	 * center.  Then adjust the east side for padding purposes.
	 */
	for ( ; gs_p != 0; gs_p = gs_p->next) {
		sylwidth(gs_p->syl, &wid_b4_syl, &wid_real_syl, &wid_after_syl);

		gs_p->c[RX] = 0;
		if (gs_p->sylposition == NOSYLPOSITION) {
			gs_p->c[RW] = -lyricsalign * wid_real_syl - wid_b4_syl;
			gs_p->c[RE] = (1 - lyricsalign) * wid_real_syl +
					wid_after_syl;
		} else {
			gs_p->c[RW] = gs_p->sylposition * POINT - wid_b4_syl;
			gs_p->c[RE] = gs_p->sylposition * POINT + wid_real_syl +
					wid_after_syl;
		}

		/* get last char of syl; if null syl don't alter RE any more */
		lc = last_char(gs_p->syl);
		if (lc == '\0')
			continue;

		/*
		 * If this is not the last syllable of the measure, and it
		 * doesn't end in '-', leave space for a blank after it, to
		 * separate it from the next syllable.
		 */
		if ( gs_p->next != 0 && lc != '-') {
			end_fontsize(gs_p->syl, &font, &size);
			gs_p->c[RE] += width(font, size, ' ');
		}

		/*
		 * If this is the last syllable of the measure, and it ends in
		 * '-', back up a space, letting the '-' go into the bar line.
		 */
		if ( gs_p->next == 0 && lc == '-' ) {
			end_fontsize(gs_p->syl, &font, &size);
			gs_p->c[RE] -= width(font, size, ' ');
		}
	}
}

/*
 * Name:        apply_staffscale()
 *
 * Abstract:    Scale all relative coordinates according to staffscale.
 *
 * Returns:     void
 *
 * Description:	Throughout Mup, we are able to almost entirely avoid dealing
 *		with the "scale" parameter, by the following trick:  We pretend
 *		the paper is a different size than it really is, by the inverse
 *		of the "scale" factor, place and print everything at standard
 *		size, and then at the end apply the scale to everything at
 *		once, in PostScript.  (Margins are exempt from scaling, hence
 *		the EFF_* macros to cancel it out.)
 *
 *		But for the "staffscale" parameter, this kind of trick only
 *		works up to a point.  As long as we are dealing only with
 *		relative coords on one staff at a time, as we have up to this
 *		point in the program, we can ignore staffscale.  But now we're
 *		about to start dealing with chord coords, and chords span
 *		staffs.  So the jig is up.
 *
 *		This function goes through all the relative coords set so far,
 *		and scales them according to staffscale.  It also scales the
 *		font sizes in strings.  From this point on staffscale must
 *		always be considered.
 */

static void
apply_staffscale()

{
	struct MAINLL *mainll_p;	/* point at items in main linked list*/
	struct STAFF *staff_p;		/* point at a staff structure */
	register float staffscale;	/* current staffscale */
	register struct GRPSYL *gs_p;	/* point at groups */
	register struct NOTE *note_p;	/* point at notes */
	struct STUFF *stuff_p;		/* point at a stuff structure */
	int n;				/* loop variable */
	int v;				/* voice number, 0 to 2 */


	debug(16, "apply_staffscale");

	initstructs();

	/*
	 * Loop down the main linked list looking for each staff.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		switch (mainll_p->str) {
		case S_SSV:
			/* apply SSVs to keep staffscale up to date */
                        asgnssv(mainll_p->u.ssv_p);
			continue;

		case S_STAFF:
			staff_p = mainll_p->u.staff_p;
			break;		/* break out to handle staffs */

		default:
			continue;	/* nothing to do */
		}

		/* get staffscale for this staff in this measure */
		staffscale = svpath(staff_p->staffno, STAFFSCALE)->staffscale;

		/* go through each voice this staff has */
		for (v = 0; v < MAXVOICES; v++) {

			/* and each group in each voice */
			for (gs_p = staff_p->groups_p[v]; gs_p != 0;
					gs_p = gs_p->next) {

				/* scale the group's relative coords */
				gs_p->c[RX] *= staffscale;
				gs_p->c[RN] *= staffscale;
				gs_p->c[RY] *= staffscale;
				gs_p->c[RS] *= staffscale;

				/* but don't disturb this E,W constant value */
				/* (see setgrps.c and abshorz.c) */
				if (gs_p->c[RE] != TEMPMRPTWIDTH / 2.0) {
					gs_p->c[RE] *= staffscale;
					gs_p->c[RW] *= staffscale;
				}

				gs_p->xdotr *= staffscale;

				/* usually we're done caring about padding */
				/*  by now, but not always, so scale it */
				gs_p->padding *= staffscale;

				if (gs_p->grpcont == GC_NOTES) {
					for (n = 0; n < gs_p->nnotes; n++) {
						note_p = &gs_p->notelist[n];

						/* scale note's rel. coords */
						note_p->c[RW] *= staffscale;
						note_p->c[RX] *= staffscale;
						note_p->c[RE] *= staffscale;
						note_p->c[RN] *= staffscale;
						note_p->c[RY] *= staffscale;
						note_p->c[RS] *= staffscale;

						note_p->waccr *= staffscale;
						note_p->ydotr *= staffscale;
						note_p->wlparen *= staffscale;
						note_p->erparen *= staffscale;

						/* this isn't really scaling,
						 * but it's a convenient place
						 * to undo CSS_STEPS */
						if (gs_p->stemto == CS_ABOVE &&
						    n <= gs_p->stemto_idx) {
							gs_p->notelist[n].stepsup -= CSS_STEPS;
							gs_p->notelist[n].ydotr -=
							CSS_STEPS * STEPSIZE * staffscale;
						} else if (gs_p->stemto == CS_BELOW &&
						    n >= gs_p->stemto_idx) {
							gs_p->notelist[n].stepsup += CSS_STEPS;
							gs_p->notelist[n].ydotr +=
							CSS_STEPS * STEPSIZE * staffscale;
						}
					}
				}

				for (n = 0; n < gs_p->nwith; n++) {
					(void)resize_string(gs_p->withlist[n],
						staffscale,
						gs_p->inputfile,
						gs_p->inputlineno);
				}
			}
		}

		/* scale the syllables' coords and font sizes */
		for (v = 0; v < staff_p->nsyllists; v++) {
			for (gs_p = staff_p->syls_p[v]; gs_p != 0;
							gs_p = gs_p->next) {
				gs_p->c[RW] *= staffscale;
				gs_p->c[RX] *= staffscale;
				gs_p->c[RE] *= staffscale;
				gs_p->c[RN] *= staffscale;
				gs_p->c[RY] *= staffscale;
				gs_p->c[RS] *= staffscale;

				(void)resize_string(gs_p->syl, staffscale,
					gs_p->inputfile, gs_p->inputlineno);
			}
		}

		/* scale the STUFF structures' font sizes */
		/* (their coords won't be set until we get to stuff.c) */
		for (stuff_p = staff_p->stuff_p; stuff_p != 0;
				stuff_p = stuff_p->next) {
			if (stuff_p->string != 0) {
				(void)resize_string(
					stuff_p->string,
					stuff_p->all == YES ? Score.staffscale
							: staffscale,
					stuff_p->inputfile,
					stuff_p->inputlineno);
			}
		}
	}
}

/*
 * Name:        relxchord()
 *
 * Abstract:    Set relative horizontal coordinates of each chord.
 *
 * Returns:     void
 *
 * Description: This function goes through the chord lists, and for each chord,
 *		sets its horizontal relative coordinates, by going down the
 *		list of GRPSYLs hanging off it.
 */

static void
relxchord()

{
	struct CHORD *ch_p;		/* point at a chord */
	struct CHORD *pch_p;		/* point at previous chord */
	struct CHORD *ppch_p;		/* point at chord before that */
	struct MAINLL *mainll_p;	/* point at items in main linked list*/
	struct GRPSYL *gs_p;		/* point at groups */
	struct GRPSYL *nsyl_p;		/* point at next syl */
	struct GRPSYL *psyl_p;		/* point at previous syl */
	float stealable;		/* from previous chord */
	float eff;			/* effective coord */


	debug(16, "relxchord");
	initstructs();

	/*
	 * Loop down the main linked list looking for each chord list headcell.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		/* need to keep "pad" up to date */
		if (mainll_p->str == S_SSV)
			asgnssv(mainll_p->u.ssv_p);

		if (mainll_p->str != S_CHHEAD)
			continue;	/* skip everything but chord HC */

		/*
		 * Pretend that all the chords will be jammed tightly together,
		 * starting at absolute coordinate zero.  We set absolute
		 * coords here for the benefit of effwest(), but they will be
		 * overwritten with their true values later in abshorz().
		 */
		mainll_p->u.chhead_p->ch_p->c[AW] = 0.0; /* west of 1st chord*/

		/*
		 * First, loop forwards through the chord list, setting the
		 * boundaries and widths of each chord based only on its
		 * groups.  The chord is to extend outwards just enough to
		 * contain every group.
		 */
		for (ch_p = mainll_p->u.chhead_p->ch_p; ch_p != 0;
					ch_p = ch_p->ch_p) {

			/* start chord as if 0 width */
			ch_p->c[RX] = 0;
			ch_p->c[RE] = 0;
			ch_p->c[RW] = 0;

			/* loop through groups, expanding chord when necessary*/
			for (gs_p = ch_p->gs_p; gs_p != 0; gs_p = gs_p->gs_p) {
				if (gs_p->grpsyl == GS_GROUP) {
					/*
					 * If last chord in measure, add pad
					 * parameter on right side of groups;
					 * but not for collapseable spaces (s).
					 */
					if (ch_p->ch_p == 0 &&
					   (gs_p->grpcont != GC_SPACE ||
					   gs_p->uncompressible == YES)) {
						gs_p->c[RE] += vvpath(gs_p->
						staffno, gs_p->vno, PAD)->pad *
						svpath(gs_p->staffno,
						STAFFSCALE)->staffscale;
					}

					eff = effwest(mainll_p, ch_p, gs_p);
					if (eff < ch_p->c[RW])
						ch_p->c[RW] = eff;
					eff = effeast(ch_p, gs_p);
					if (eff > ch_p->c[RE])
						ch_p->c[RE] = eff;
				}
			}

			/* store width; will be updated later to include syls */
			ch_p->width = ch_p->c[RE] - ch_p->c[RW];

			/* set phony absolute coords for effwest() */
			ch_p->c[AX] = ch_p->c[AW] - ch_p->c[RW];
			ch_p->c[AE] = ch_p->c[AX] + ch_p->c[RE];
			if (ch_p->ch_p != 0)
				ch_p->ch_p->c[AW] = ch_p->c[AE];
		}

		/*
		 * Loop again through each chord in this list, this time
		 * expanding chords when necessary to include eastward
		 * extensions of syllables.  Work right to left, so that when
		 * a syllable steals space from the following chord, the
		 * following chord has already been widened eastwards, if it
		 * needed to be, based on its syllables.
		 */
		/* find last chord in the chord LL */
		ch_p = mainll_p->u.chhead_p->ch_p;	/* first chord */
		while (ch_p->ch_p != 0)
			ch_p = ch_p->ch_p;

		/* loop backwards through them (too bad there's no back ptr) */
		for ( ; ch_p != 0; ch_p = prevchord(mainll_p, ch_p)) {
			/*
			 * Loop through the linked list of GRPSYLs hanging off
			 * this chord, altering RE when finding a syl that
			 * sticks out farther.  There is one exception to
			 * this.  If a syllable extends farther east than any
			 * one so far, a test is made so that it can steal
			 * space from the following chord if that chord has
			 * no syllable there.
			 */
			for (gs_p = ch_p->gs_p; gs_p != 0; gs_p = gs_p->gs_p) {

				/* if not a syl or not sticking out east */
				if (gs_p->grpsyl != GS_SYLLABLE ||
						gs_p->c[RE] <= ch_p->c[RE])
					continue;

				/* syl seems to be sticking out east */

				/*
				 * If this is the last chord in the measure,
				 * the chord boundary must include the syl.
				 */
				if (ch_p->ch_p == 0) {
					ch_p->c[RE] = gs_p->c[RE];
					continue;
				}

				/*
				 * The syl is sticking out east of the current
				 * chord boundary, and this is not the last
				 * chord in the measure.  See if the next
				 * chord contains the next syl of this verse.
				 * If not, there's an empty space there, and
				 * we can let the current syl steal space from
				 * the next chord.
				 */
				nsyl_p = nextchsyl(gs_p, ch_p);
				if (nsyl_p == 0) {
					/*
					 * Next chord has no syl here.  We can
					 * steal its space.  If the syl is so
					 * long that even that isn't enough
					 * room, we'll force the current chord
					 * boundary far enough out to contain
					 * the excess.
					 */
					if (gs_p->c[RE] > ch_p->c[RE] +
							ch_p->ch_p->width) {
						ch_p->c[RE] = gs_p->c[RE] -
							ch_p->ch_p->width;
					}
				} else {
					/*
					 * Next chord has a syl, so we can't
					 * steal its space.  Extend this chord.
					 */
					ch_p->c[RE] = gs_p->c[RE];
				}
			}

			/* revise width; will be revised again later */
			ch_p->width = ch_p->c[RE] - ch_p->c[RW];

		} /* end of backwards loop through chords in this measure */

		/*
		 * Loop again through each chord in this list, this time
		 * expanding chords when necessary to include westward
		 * extensions of syllables.  Work left to right, so that when
		 * a syllable steals space from the preceding chord, the
		 * preceding chord has already been widened westwards, if it
		 * needed to be, based on its syllables.
		 */
		for (ch_p = mainll_p->u.chhead_p->ch_p; ch_p != 0;
					ch_p = ch_p->ch_p) {
			/*
			 * Loop through the linked list of GRPSYLs hanging off
			 * this chord, altering RW when finding a syl that
			 * sticks out farther.  There is one exception to
			 * this.  If a syllable extends farther west than any
			 * one so far, a test is made so that it can steal
			 * space from the following chord if that chord has
			 * no syllable there.
			 */
			for (gs_p = ch_p->gs_p; gs_p != 0; gs_p = gs_p->gs_p) {

				/* if not a syl or not sticking out west */
				if (gs_p->grpsyl != GS_SYLLABLE ||
						gs_p->c[RW] >= ch_p->c[RW])
					continue;

				/* syl seems to be sticking out west */

				/*
				 * If this is the first chord in the measure,
				 * the chord boundary must include the syl.
				 */
				if (prevchord(mainll_p, ch_p) == 0) {
					ch_p->c[RW] = gs_p->c[RW];
					continue;
				}

				/*
				 * The syl is sticking out west of the current
				 * chord boundary, and this is not the first
				 * chord in the measure.  See if the previous
				 * chord contains the previous syl of this
				 * verse, or if the one before that stole
				 * space from it.  If not, there's an empty
				 * space there, and we can let the current syl
				 * steal space from the previous chord.  Even
				 * if the previous chord has no syl but the one
				 * before stole some space from it, we can use
				 * the part of the space it didn't steal.
				 */
				/* get prev chord, & prev syl in this verse */
				pch_p = prevchord(mainll_p, ch_p);
				psyl_p = prevchsyl(gs_p, pch_p);

				if (psyl_p == 0) {
					/* first, assume all of the previous */
					/*  chord's width is available */
					stealable = pch_p->width;

					/*
					 * Get the chord before the previous.
					 * If it exists and contains a syl,
					 * syl may already be stealing space
					 * east of it, in which case we can
					 * only steal what's left over.
					 */
					ppch_p = prevchord(mainll_p, pch_p);
					if (ppch_p != 0) {
						psyl_p = prevchsyl(gs_p,
								ppch_p);

						if (psyl_p != 0 && psyl_p->c[RE]
						> ppch_p->c[RE]) {
							stealable -= (psyl_p->
							c[RE] - ppch_p->c[RE]);
						}
					}

					/*
					 * If our syl needs more space than is
					 * available for stealing, widen our
					 * chord by the necessary amount.
					 */
					if (gs_p->c[RW] < ch_p->c[RW] -
								stealable)
						ch_p->c[RW] = gs_p->c[RW] +
								stealable;
				} else {
					/*
					 * Prev chord has a syl, so we can't
					 * steal its space.  Extend this chord.
					 */
					ch_p->c[RW] = gs_p->c[RW];
				}
			}

			/* final revision of width */
			ch_p->width = ch_p->c[RE] - ch_p->c[RW];

		} /* end of forwards loop through chords in this measure */

	} /* end of loop through each CHHEAD in main linked list */

	pedalroom();		/* make room for "Ped." and "*" if need be */

	fixspace();		/* set a width for certain space chords */
}

/*
 * Name:        effwest()
 *
 * Abstract:    Find the effective west boundary of a group.
 *
 * Returns:     the RW to be used for the group
 *
 * Description: This function returns an "effective" RW for the given group.
 *		Sometimes this is just the true RW.  But if the previous chord
 *		has no groups on this staff, we pretend it is a smaller number,
 *		so that our group can overlap horizonally with previous
 *		that have no groups on this staff chord(s).
 */

static double
effwest(mainll_p, ch_p, gs_p)

struct MAINLL *mainll_p;	/* point at MLL item for this chord */
struct CHORD *ch_p;		/* point at this chord */
struct GRPSYL *gs_p;		/* point at this group */

{
	struct CHORD *pch_p;	/* point at previous chord */
	struct CHORD *ech_p;	/* point at earlier chord */
	struct GRPSYL *pgs_p;	/* point a group in previous chord */
	float small;		/* small number to be used */
	int found;		/* found a chord with a group on our staff? */
	float ourax;		/* tentative value for our chord's AX */
	float temp;		/* temp variable */


	pch_p = prevchord(mainll_p, ch_p);	/* find previous chord */

	/* if we are the first chord, return our group's true RW */
	if (pch_p == 0)
		return (gs_p->c[RW]);

	/* set default to -1.5 stepsize */
	small = -1.5 * STEPSIZE * svpath(gs_p->staffno, STAFFSCALE)->staffscale;

	/* if already closer to 0 than "small", return true RW */
	if (gs_p->c[RW] > small)
		return (gs_p->c[RW]);

	/*
	 * Loop through the previous chord's GRPSYLs to see if it has any
	 * groups on this staff.  If so, return our true RW.  We'd like to
	 * have a fancier algorithm here, to see if such a group truly would
	 * run into ours, but stem lengths haven't been set yet, so we don't
	 * know the vertical extent of groups yet.
	 */
	for (pgs_p = pch_p->gs_p; pgs_p != 0; pgs_p = pgs_p->gs_p) {
		/* skip cases where there can't be any interference */
		if (pgs_p->staffno > gs_p->staffno)
			break;		/* nothing more could be on our staff*/
		if (pgs_p->staffno < gs_p->staffno)
			continue;	/* ignore if wrong staff */
		if (pgs_p->grpsyl == GS_SYLLABLE)
			continue;	/* ignore if not a group */

		/* found a group, return our true RW */
		return (gs_p->c[RW]);
	}

	/*
	 * There is no group on our staff in the preceding chord.  We'd like to
	 * let our group overlap into that space if necessary.  But there
	 * might be a group in some earlier chord, and if there are enough dots
	 * on it, or enough accidentals on our group, they could still
	 * interfere.  Find the first earlier chord, looking right to left,
	 * that has a group neighboring our group.
	 */
	found = NO;
	for (ech_p = prevchord(mainll_p, pch_p); ech_p != 0;
				ech_p = prevchord(mainll_p, ech_p)) {

		for (pgs_p = ech_p->gs_p; pgs_p != 0; pgs_p = pgs_p->gs_p) {

			if (pgs_p->staffno > gs_p->staffno)
				break;	/* nothing more could be on our staff*/
			if (pgs_p->staffno < gs_p->staffno)
				continue;	/* ignore if wrong staff */
			if (pgs_p->grpsyl == GS_SYLLABLE)
				continue;	/* ignore if not a group */

			/* we found a group on our group's staff */
			found = YES;
			break;
		}
		if (found == YES)
			break;
	}

	if (ech_p == 0)
		pfatal("no preceding group in effwest()");

	/*
	 * Since there could be multiple voices on this staff, there could be
	 * multiple groups on this staff in the chord we found.  Loop through
	 * each of them, keeping track of the max value our chord's AX would
	 * have to be to keep our group from overlapping that group.
	 */
	ourax = 0.0;
	for ( ; pgs_p != 0 && pgs_p->staffno == gs_p->staffno &&
			pgs_p->grpsyl == GS_GROUP; pgs_p = pgs_p->gs_p) {

		temp = ech_p->c[AX] + pgs_p->c[RE] - gs_p->c[RW];
		if (temp > ourax)
			ourax = temp;
	}

	/* find what that value for our AX would make our RW be */
	temp = ch_p->c[AW] - ourax;

	/* return that amount, but not more than "small" */
	return (MIN(temp, small));
}

/*
 * Name:        effeast()
 *
 * Abstract:    Find the effective east boundary of a group.
 *
 * Returns:     the RE to be used for the group
 *
 * Description: This function returns an "effective" RE for the given group.
 *		Sometimes this is just the true RE.  But if the next chord
 *		has no groups on this staff, we pretend it is a smaller number,
 *		so that our group can overlap horizonally with the next chord.
 *		Don't worry about colliding with a group in a later chord;
 *		effwest() will handle that when processing that later group.
 */

static double
effeast(ch_p, gs_p)

struct CHORD *ch_p;		/* point at this chord */
struct GRPSYL *gs_p;		/* point at this group */

{
	struct CHORD *nch_p;	/* point at next chord */
	struct GRPSYL *ngs_p;	/* point a group in next chord */
	float small;		/* small number to be used */
	float onestep;		/* a stepsize, scaled */


	nch_p = ch_p->ch_p;	/* find next chord */

	/* if we are the last chord, return our group's true RE */
	if (nch_p == 0)
		return (gs_p->c[RE]);

	/* set default to 1.5 stepsize */
	onestep = STEPSIZE * svpath(gs_p->staffno, STAFFSCALE)->staffscale;
	small = 1.5 * onestep;

	/* if already closer to 0 than "small", return true RE */
	if (gs_p->c[RE] < small)
		return (gs_p->c[RE]);

	/*
	 * Loop through the next chord's GRPSYLs to see if it has any
	 * groups on this staff.  If so, return our true RE.
	 */
	for (ngs_p = nch_p->gs_p; ngs_p != 0; ngs_p = ngs_p->gs_p) {
		/* skip cases where there can't be any interference */
		if (ngs_p->staffno > gs_p->staffno)
			break;		/* nothing more could be on our staff*/
		if (ngs_p->staffno < gs_p->staffno)
			continue;	/* ignore if wrong staff */
		if (ngs_p->grpsyl == GS_SYLLABLE)
			continue;	/* ignore if not a group */

		/* found a group, return our true RE */
		return (gs_p->c[RE]);
	}

	return (small);
}

/*
 * Name:        prevchord()
 *
 * Abstract:    Find chord preceding the given one.
 *
 * Returns:     pointer to previous chord, or 0 if none
 *
 * Description: This function is given a pointer to a chord headcell and a
 *		chord in that list.  It finds the preceding chord, returning
 *		it, or 0 if none.  If chord linked lists were doubly linked,
 *		we wouldn't have to go through this aggravation.
 */

static struct CHORD *
prevchord(mainll_p, ch_p)

struct MAINLL *mainll_p;	/* ptr to current syllable */
struct CHORD *ch_p;		/* ptr to current chord */

{
	register struct CHORD *prevch_p;


	prevch_p = mainll_p->u.chhead_p->ch_p;	/* get first chord in list */

	/* if current chord is first chord, there is none before it */
	if (prevch_p == ch_p)
		return (0);

	/* loop until we find it, then return */
	while (prevch_p->ch_p != ch_p)
		prevch_p = prevch_p->ch_p;
	return (prevch_p);
}

/*
 * Name:        nextchsyl()
 *
 * Abstract:    Find following syllable if it is in the next chord.
 *
 * Returns:     pointer to next syllable, or 0 if none
 *
 * Description: This function is given a pointer to a syllable, and the chord
 *		it is in.  It looks in the next chord, to see if there is a
 *		syllable there that follows this syllable.  If there is, it
 *		returns it.  Otherwise it returns 0.
 *		Note:  if the next syllable is was given as a space, it counts
 *		as if it weren't there at all (return 0).
 */

static struct GRPSYL *
nextchsyl(gs_p, ch_p)

struct GRPSYL *gs_p;	/* ptr to current syllable */
struct CHORD *ch_p;	/* ptr to current chord */

{
	struct GRPSYL *nextgs_p;	/* point, looking for next syl */


	/* if last chord in measure, return no next syllable */
	if (ch_p->ch_p == 0)
		return (0);

	/*
	 * Look down next chord until we hit either the end, or the syllable
	 * that follows the given one.  Return what was found.
	 */
	for (nextgs_p = ch_p->ch_p->gs_p;
			nextgs_p != 0 && nextgs_p != gs_p->next;
			nextgs_p = nextgs_p->gs_p)
		;

	/* if syl doesn't exist or is a space, return 0 */
	if (nextgs_p == 0 || nextgs_p->syl == 0)
		return (0);

	return (nextgs_p);
}

/*
 * Name:        prevchsyl()
 *
 * Abstract:    Find preceding syllable if it is in the previous chord.
 *
 * Returns:     pointer to previous syllable, or 0 if none
 *
 * Description: This function is given a pointer to a syllable, and the chord
 *		it is in.  It looks in the previous chord, to see if there is a
 *		syllable there that precedes this syllable.  If there is, it
 *		returns it.  Otherwise it returns 0.
 *
 *		Note:  if the prev syllable is given as a space, it counts
 *		as if it weren't there at all (return 0).
 *
 *		Also note:  unlike nextchsyl, this function compares against
 *		not only the given GRPSYL, but the also the previous GRPSYL.
 *		It has to, because it is sometimes called with the previous
 *		chord, and sometimes with the one before that.
 */

static struct GRPSYL *
prevchsyl(gs_p, prevch_p)

struct GRPSYL *gs_p;		/* ptr to current syllable */
struct CHORD *prevch_p;		/* ptr to previous chord */

{
	struct GRPSYL *prevgs_p;	/* point, looking for next syl */


	/* if first chord in measure, return no previous syllable */
	if (prevch_p == 0)
		return (0);

	/*
	 * Look down previous chord until we hit either the end, or the syllable
	 * that precedes the given one.  Return what was found.
	 * "Precede" here means either directly precedes, or precedes in two
	 * steps.
	 */
	for (prevgs_p = prevch_p->gs_p;
			prevgs_p != 0 && ! (prevgs_p == gs_p->prev ||
			(gs_p->prev != 0 && prevgs_p == gs_p->prev->prev));
			prevgs_p = prevgs_p->gs_p)
		;

	/* if syl doesn't exist or is a space, return 0 */
	if (prevgs_p == 0 || prevgs_p->syl == 0)
		return (0);

	return (prevgs_p);
}

/*
 * Name:        pedalroom()
 *
 * Abstract:    Increase some chords' width to make room for pedal characters.
 *
 * Returns:     void
 *
 * Description: This function tries to make room for "Ped." and "*", so that
 *		they don't overwrite each other.  For each "pedstar" style
 *		pedal mark, it finds the chord it's closest to.  If two of them
 *		are on neighboring chords, it may widen the left chord to
 *		provide enough room.  The problem is, the best it can do is
 *		assume that the pedal marks are exactly aligned with their
 *		closest chords.  It doesn't do anything about marks that are
 *		not on neighboring chords, since that would be quite a bit
 *		more work and would rarely be necessary.  Worst of all, if two
 *		marks' closest chords are the same chord, nothing can be done.
 */

static void
pedalroom()

{
	struct MAINLL *mainll_p;	/* point at items in main linked list*/
	struct CHHEAD *chhead_p;	/* point at a chord head cell */
	struct STAFF *staff_p;		/* point at a staff */
	struct STUFF *stuff_p;		/* point at a stuff */
	struct CHORD *pedch_p;		/* point at a chord near a pedal mark*/
	struct CHORD *opedch_p;		/* point at prev chord near pedal */
	int pedstyle;			/* P_* */
	int pedchar, opedchar;		/* current and previous pedal char */
	int font, size;			/* of a pedal char */
	char *string;			/* for pedal char */
	float needed;			/* amount of room needed */


	debug(16, "pedalroom");
	initstructs();

	chhead_p = 0;		/* prevent useless 'used before set' warning */

	/*
	 * Loop down the main linked list looking for each chord list headcell.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		switch (mainll_p->str) {
		case S_SSV:
			/* need to keep pedstyle and timeden up to date */
			asgnssv(mainll_p->u.ssv_p);
			continue;	/* go to next MLL structure */

		case S_CHHEAD:
			/* remember this measure's chord list */
			chhead_p = mainll_p->u.chhead_p;
			continue;	/* go to next MLL structure */

		case S_STAFF:
			pedstyle = svpath(mainll_p->u.staff_p->staffno,
					PEDSTYLE)->pedstyle;
			if (pedstyle != P_LINE) {
				staff_p = mainll_p->u.staff_p;
				break;		/* break out and handle this */
			}

			continue;	/* not pedstar, ignore this staff */

		default:
			continue;	/* skip everything else */
		}

		/*
		 * At this point we are at a staff that has a pedstyle that
		 * uses "Ped." and "*".  Loop down the stuff list, looking for
		 * pedal marks.
		 */
		opedch_p = 0;		/* no pedal mark yet in measure */
		opedchar = '\0';/* prevent useless 'used before set' warning */
		for (stuff_p = staff_p->stuff_p; stuff_p != 0;
					stuff_p = stuff_p->next) {
			/*
			 * If it is not a pedal stuff, or it has no character,
			 * like a continuation from the previous score, skip.
			 */
			if (stuff_p->stuff_type != ST_PEDAL ||
			    stuff_p->string == 0)
				continue;
			/*
			 * Find the chord that is closest to this pedal mark,
			 * and which character this pedal mark is.
			 * But following the usual policy of applying "steps"
			 * offsets only after everything else is done, we
			 * ignore start.steps and use only start.count.
			 */
			pedch_p = closestchord(stuff_p->start.count,
					chhead_p->ch_p);
			font = stuff_p->string[0];
			size = stuff_p->string[1];
			string = stuff_p->string + 2;
			pedchar = next_str_char(&string, &font, &size) & 0xff;

			/* if first pedal mark in measure, nothing more to do*/
			if (opedch_p == 0) {
				/* remember as previous chord with pedal */
				opedch_p = pedch_p;
				opedchar = pedchar;
				continue;
			}

			/*
			 * If this pedal mark and the previous one are by
			 * neighboring chords, assume these marks are exactly
			 * aligned with their chords.  Make sure the east half
			 * of the previous chord plus the west half of this
			 * chord is enough room for them.  If it isn't, enlarge
			 * the east half of the previous chord.  (Note: RW is
			 * negative, so it must be negated.)
			 */
			if (pedch_p == opedch_p->ch_p) {
				needed = rightped(pedstyle, opedchar) +
					 leftped(pedstyle, pedchar);
				if (stuff_p->all == YES) {
					needed *= Score.staffscale;
				} else {
					needed *= svpath(staff_p->staffno,
						STAFFSCALE)->staffscale;
				}
				if (opedch_p->c[RE] - pedch_p->c[RW] < needed){
					opedch_p->c[RE] = needed +
							pedch_p->c[RW];
					opedch_p->width = opedch_p->c[RE] -
							opedch_p->c[RW];
				}
			}

			/* remember previous chord with pedal, and its char */
			opedch_p = pedch_p;
			opedchar = pedchar;
		}
	}
}

/*
 * Name:        closestchord()
 *
 * Abstract:    Find closest chord to given time value.
 *
 * Returns:     pointer to the closest chord
 *
 * Description: This function finds the CHORD in the given linked list that is
 *		closest, timewise, to the given count number.
 */

static struct CHORD *
closestchord(count, firstch_p)

double count;			/* which count of the measure */
struct CHORD *firstch_p;	/* first CHORD in this measure */

{
	RATIONAL reqtime;	/* time requested */
	struct CHORD *ch_p;	/* point along chord list */
	struct CHORD *och_p;	/* (old) point along chord list */


	/* if at or before the first count, it's closest to first group */
	if (count <= 1)
		return (firstch_p);

	/* get requested time to nearest tiny part of a count, in lowest terms*/
	reqtime.n = 4 * MAXBASICTIME * (count - 1) + 0.5;
	reqtime.d = 4 * MAXBASICTIME * Score.timeden;
	rred(&reqtime);

	/*
	 * Loop through the chord list.  As soon as a chord starts at or after
	 * the requested time value, check whether the requested time is closer
	 * to the new chord's time, or the previous chord's.  Return the
	 * closest one.
	 */
	for (och_p = firstch_p, ch_p = och_p->ch_p; ch_p != 0;
				och_p = ch_p, ch_p = ch_p->ch_p) {
		if (GE(ch_p->starttime, reqtime)) {
			if (GT( rsub(reqtime, och_p->starttime),
					rsub(ch_p->starttime, reqtime) ))
				return (ch_p);
			else
				return (och_p);
		}
	}

	/* requested time is after last chord; return last chord */
	return (och_p);
}

/*
 * Name:        rightped()
 *
 * Abstract:    Find the size of the right side of a pedstar pedal char.
 *
 * Returns:     the size
 *
 * Description: This function finds the size of the part of the given pedal
 *		character (pedstar style) that is to the right of where it
 *		should be centered.
 */

static double
rightped(pedstyle, pedchar)

int pedstyle;			/* pedstar or alt pedstar */
int pedchar;			/* the given char */

{
	switch (pedchar) {
	case C_BEGPED:
		return (strwidth(Ped_start) / 2.0);
	case C_PEDAL:
		if (pedstyle == P_PEDSTAR)
			return (strwidth(Ped_start) - ped_offset());
		else /* P_ALTPEDSTAR */
			return (strwidth(Ped_start) / 2.0);
	case C_ENDPED:
		return (strwidth(Ped_stop) / 2.0);
	default:
		pfatal("bad pedal character passed to rightped()");
	}
	return (0);	/* to keep lint happy */
}

/*
 * Name:        leftped()
 *
 * Abstract:    Find the size of the left side of a pedstar pedal char.
 *
 * Returns:     the size
 *
 * Description: This function finds the size of the part of the given pedal
 *		character (pedstar style) that is to the left of where it
 *		should be centered.
 */

static double
leftped(pedstyle, pedchar)

int pedstyle;			/* pedstar or alt pedstar */
int pedchar;			/* the given char */

{
	switch (pedchar) {
	case C_BEGPED:
		return (strwidth(Ped_start) / 2.0);
	case C_PEDAL:
		if (pedstyle == P_PEDSTAR)
			return (strwidth(Ped_stop) + ped_offset());
		else /* P_ALTPEDSTAR */
			return (strwidth(Ped_start) / 2.0);
	case C_ENDPED:
		return (strwidth(Ped_stop) / 2.0);
	default:
		pfatal("bad pedal character passed to leftped()");
	}
	return (0);	/* to keep lint happy */
}

/*
 * Name:        fixspace()
 *
 * Abstract:    Reset width, if need be, for chords of all spaces.
 *
 * Returns:     void
 *
 * Description: This function loops through chord lists, looking for each chord
 *		that is all spaces (as shown by the fact that its width is 0).
 *		If any voice has a nonspace during the time value of this
 *		chord, we reset its width to a very small positive number, to
 *		prevent abshorz from treating it like it "deserves" no width
 *		(rather than what its duration would imply).
 */

static void
fixspace()

{
	struct CHORD *ch_p;		/* point at a chord */
	struct MAINLL *mainll_p;	/* point at items in main linked list*/
	struct MAINLL *m2_p;		/* another pointer down the MLL */
	int crunch;			/* should chord be crunched to 0 width*/
	int v;				/* voice number, 0 or 1 */


	debug(16, "fixspace");
	/*
	 * Loop down the main linked list looking for each chord list headcell.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		if (mainll_p->str != S_CHHEAD)
			continue;	/* skip everything but chord HC */

		/*
		 * Loop through the chord list, looking for all-space chords.
		 * Skip the first one in the list; we always want that one to
		 * be crunched to zero width, since there's nothing earlier
		 * that could extend into it.
		 */
		for (ch_p = mainll_p->u.chhead_p->ch_p->ch_p; ch_p != 0;
					ch_p = ch_p->ch_p) {

			if (ch_p->width != 0)
				continue;	/* skip nonspace chord */

			crunch = YES;		/* init to crunch */

			/* loop through every staff and every voice in meas */
			for (m2_p = mainll_p->next; m2_p->str == S_STAFF;
					m2_p = m2_p->next) {

				for (v = 0; v < MAXVOICES && m2_p->u.staff_p->
							groups_p[v] != 0; v++) {

					/* if voice has nonspace, don't crunch*/
					if ( ! hasspace(m2_p->u.staff_p->
					groups_p[v], ch_p->starttime,
					radd(ch_p->starttime, ch_p->duration))){
						crunch = NO;
						break;
					}
				}

				if (crunch == NO)
					break;
			}

			/* if should not crunch, set nonzero width to stop it */
			if (crunch == NO)
				ch_p->width = 0.001;

		} /* for every chord in this measure */

	} /* looping through MLL, dealing with chord headcells */
}
