/* Copyright (c) 1995, 1996, 1997, 1998, 2002, 2003, 2004 by Arkkra Enterprises */
/* All rights reserved */
/*
 * Name:	mkchords.c
 *
 * Description:	This file contains functions for creating CHORD linked lists
 *		and erasing invisible voices.
 */

#include "defines.h"
#include "structs.h"
#include "globals.h"


static void swingmidi P((void));
static struct GRPSYL *voicevis P((struct GRPSYL *gs_p));

/*
 * Name:        makechords()
 *
 * Abstract:    Set up the linked lists of chords.
 *
 * Returns:     void
 *
 * Description: This function scans through the main linked list looking
 *		for STAFF structures.  It joins the GRPSYL structures in
 *		the lists that they head, into perpendicular linked lists,
 *		allocating a CHORD to head each of these.  It also links
 *		the CHORDs for each measure together into a linked list.
 *
 *		While doing the above, it also calls voicevis() to check if a
 *		voice should be invisible, and if so change it to a measure
 *		space.
 */


void
makechords()

{
	struct MAINLL *mainll_p;	/* point at item in main linked list */
	struct MAINLL *mainch_p;	/* for headcell of a chord list */

	RATIONAL *vtime;		/* cum. time of each voice and verse */
	struct GRPSYL **grpsyl_p;	/* pointers along GRPSYL lists */
	struct STAFF *staff_p;		/* point at a staff structure */

	RATIONAL mintime;		/* the minimum vtime */
	register int num;		/* no. of visible voices/verses */
	register int v;			/* index into grpsyl_p[] */
	struct CHORD *ch_p;		/* pointer to current chord */
	struct CHORD *och_p;		/* pointer to old chord */
	struct GRPSYL *gs_p;		/* pointer to current group/syllable */
	int n;				/* loop variable */
	int firstgs;			/* flag for first group/syllable */
	int quit;			/* flag for being done */


	debug(16, "makechords");
	/*
	 * If we are generating MIDI, we may need to change lengths (fulltimes)
	 * of groups.  This has to be done now, before we link groups together
	 * into chords.
	 */
	if (Doing_MIDI == YES) {
		swingmidi();
	}

	gs_p = 0;		/* keep lint happy; will be set before used */

	/* malloc enough of these for all voices and verses */
	MALLOC(rational, vtime, MAXSTAFFS * (MAXVOICES + Maxverses));
	MALLOC(GRPSYL *, grpsyl_p, MAXSTAFFS * (MAXVOICES + Maxverses));

	mainll_p = Mainllhc_p;		/* point at first thing in main LL */

	initstructs();			/* clean out old SSV info */

	/*
	 * Loop once for each measure in the input.
	 */
	for (;;) {
		num = 0;		/* number of linked lists in measure */

		/*
		 * Look for the first structure in this measure that points off
		 * to a linked list of groups/syllables.  If we hit the end of
		 * the main linked list, we're all done, so return.
		 */
		while (mainll_p != 0 && mainll_p->str != S_STAFF) {
			if (mainll_p->str == S_SSV)
				asgnssv(mainll_p->u.ssv_p);
			mainll_p = mainll_p->next;
		}

		if (mainll_p == 0) {
			FREE(vtime);
			FREE(grpsyl_p);
			return;
		}

		/*
		 * We've found another measure with STAFF in it.  Allocate
		 * a chord headcell for this measure, and put it in the
		 * main linked list.
		 */
		mainch_p = newMAINLLstruct(S_CHHEAD, 0);
		insertMAINLL(mainch_p, mainll_p->prev);

		/*
		 * Look for the last STAFF structure in the measure.  While
		 * doing this, point at first element of all the group/syllable
		 * linked lists, and keep count of them.  Ignore invisible
		 * staffs.  Skip over any grace groups.
		 */
		while (mainll_p != 0 && mainll_p->str == S_STAFF) {
			staff_p = mainll_p->u.staff_p;

			if (staff_p->visible == YES) {

				/* do all the voices on this staff */
				for (n = 0; n < MAXVOICES &&
				staff_p->groups_p[n] != 0; n++) {
					/*
					 * Zap voice if invisible.  Set both
					 * of these to point at the first
					 * GRPSYL, which will be a new one if
					 * we zapped it.
					 */
					grpsyl_p[num] = staff_p->groups_p[n] =
					voicevis(staff_p->groups_p[n]);

					/* skip leading grace groups */
					while (grpsyl_p[num] != 0 &&
					grpsyl_p[num]->grpvalue == GV_ZERO)
						grpsyl_p[num] = grpsyl_p[num]->next;
					if (grpsyl_p[num] == 0)
						pfatal("nothing but grace groups found");
					num++;
				}

				/* do all the verses on this staff */
				for (n = 0; n < staff_p->nsyllists; n++)
					grpsyl_p[num++] = staff_p->syls_p[n];
			}
			mainll_p = mainll_p->next;
		}

		/*
		 * Set up the first chord from the first note in each
		 * voice/verse.  Its linked list of GRPSYLs will include
		 * the first GRPSYL in every linked list off of a visible
		 * STAFF.
		 */
		MALLOC(CHORD, ch_p, 1);
		mainch_p->u.chhead_p->ch_p = ch_p; /* point at first chord */
		ch_p->ch_p = 0;		/* only member on list so far */
		ch_p->starttime = Zero; /* start time = any voice */

		/* point headcell at first and set its first group's time */
		ch_p->gs_p = grpsyl_p[0];
		vtime[0] = grpsyl_p[0]->fulltime;

		/* for each remaining one, point prev one at it & set time */
		for (v = 1; v < num; v++) {
			grpsyl_p[v-1]->gs_p = grpsyl_p[v];
			vtime[v] = grpsyl_p[v]->fulltime;
		}
		grpsyl_p[num-1]->gs_p = 0;	/* terminate linked list */

		/* point at second GRPSYL in each voice/verse, if any */
		for (v = 0; v < num; v++)
			grpsyl_p[v] = grpsyl_p[v]->next;

		/*
		 * Loop until groups/syllables in the voices/verses are used
		 * up.  Form a chord for each time at which any voice/verse
		 * has a GRPSYL structure, though ignore grace groups.
		 */
		for (;;) {
			/*
			 * If every GRPSYL currently pointed at by a grpsyl_p[]
			 * is the last in its list (the measure), there are no
			 * more chords in this measure, and quit will remain
			 * YES.
			 */
			quit = YES;		/* first assume "quit" */
			for (v = 0; v < num; v++) {
				/* find next item (if any) not a grace group */
				while (grpsyl_p[v] != 0 &&
				       grpsyl_p[v]->grpsyl == GS_GROUP &&
				       grpsyl_p[v]->grpvalue == GV_ZERO) {

					grpsyl_p[v] = grpsyl_p[v]->next;
				}

				/* check if voice/verse has another item */
				if (grpsyl_p[v] != 0) {
					quit = NO; /* yes, so don't quit yet */
				}
			}

			/* if time to quit, skip rest of loop, and get out */
			if (quit == YES)
				break;

			/*
			 * At least one voice/verse has another note in it.
			 * Find the earliest time at which something changes.
			 */
			mintime = vtime[0];
			for (v = 1; v < num; v++)
				if (LT(vtime[v], mintime))
					mintime = vtime[v];

			/* allocate memory for another chord */
			och_p = ch_p;	/* remember where previous chord is */
			MALLOC(CHORD, ch_p, 1);
			och_p->ch_p = ch_p;	/* point previous chord at it*/
			ch_p->ch_p = 0;		/* terminate in case last */

			ch_p->starttime = mintime; /* starting time for chord*/

			/*
			 * Form a new linked list.  The head cell is the new
			 * chord, and the list connects it to all the groups/
			 * syllables that start at this time.
			 */
			firstgs = YES;
			for (v = 0; v < num; v++) {
				if (EQ(vtime[v], mintime)) {
					/*
					 * This voice/verse has a grpsyl at
					 * this time.  Make the previous one
					 * point at it, set its pointer to 0
					 * in case it turns out to be the last,
					 * and add its length to vtime[v].
					 */
					if (firstgs == YES) {
						/* point headcell at first */
						ch_p->gs_p = grpsyl_p[v];
						firstgs = NO;
					} else {
						/* point previous one at ours*/
						gs_p->gs_p = grpsyl_p[v];
					}

					/* set gs_p to point at our new one */
					gs_p = grpsyl_p[v];

					vtime[v] = radd(vtime[v],
							gs_p->fulltime);

					/* get next GRPSYL in voice/verse */
					grpsyl_p[v] = gs_p->next;
				}
			}

			gs_p->gs_p = 0;		/* terminate linked list */
		}

		/*
		 * Set the duration of each chord in this measure.  It's the
		 * next chord's start time minus this chord's start time.
		 * But for the last chord, it's the time signature minus
		 * this chord's start time.  Also set pseudodur, which is a
		 * function of the duration.  The amount of width the chord
		 * "deserves" to be allocated is proportional to pseudodur.
		 */
		for (ch_p = mainch_p->u.chhead_p->ch_p; ch_p->ch_p != 0;
				ch_p = ch_p->ch_p) {
			ch_p->duration = rsub(ch_p->ch_p->starttime,
						ch_p->starttime);
			ch_p->pseudodur = pow(RAT2FLOAT(ch_p->duration),
						Score.packexp);
		}
		ch_p->duration = rsub(Score.time, ch_p->starttime);
		ch_p->pseudodur = pow(RAT2FLOAT(ch_p->duration), Score.packexp);
	}
}

/*
 * Name:	swingmidi()
 *
 * Abstract:	Alter groups' time to implement the "swingunit" parameter.
 *
 * Returns:	void
 *
 * Description:	This function loops through every GRPSYL in every voice,
 *		adjusting their fulltime so that they will start and end at
 *		different times, if necessary, to follow the "swingunit" parm.
 *		Each measure is divided into durations of "swingunit", starting
 *		at the beginning.  (Usually the timesig divided by swingunit
 *		will be an integer, but if not, the last piece will be shorter.)
 *		The time where one group ends and the next group starts will be
 *		altered in either of these two circumstances:
 *		1. The current boundary time is halfway into a swingunit, and
 *		   each group is at least half a swingunit long.
 *		2. The current boundary time is 3/4 of the way into a swingunit,
 *		   and the first group is at least 3/4 of a swingunit long, and
 *		   and the second group is at least 1/4 of a swingunit long.
 *		In both of these cases, the fulltimes are altered so that the
 *		meeting point is 2/3 of the way into the swingunit.
 */

void
swingmidi()
{
	struct MAINLL *mainll_p;	/* point along main linked list */
	struct GRPSYL *gs_p;		/* point along a GRPSYL list */
	struct GRPSYL *prev_p;		/* the GRPSYL before gs_p */
	int vidx;			/* voice index, 0 to MAXVOICES-1 */
	RATIONAL quot;			/* quotient */
	RATIONAL swingunit;		/* from SSV */
	RATIONAL starttime;		/* offset into measure of gs_p */
	RATIONAL halfswing;		/* swingunit/2 */
	RATIONAL sixthswing;		/* swingunit/6 */
	RATIONAL twelfthswing;		/* swingunit/12 */
	RATIONAL threefourthsswing;	/* 3/4 * swingunit */
	RATIONAL onefourthswing;	/* 1/4 * swingunit */
	static RATIONAL six = {6,1};
	static RATIONAL twelve = {12,1};


	initstructs();

	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		switch (mainll_p->str) {
		case S_SSV:
			/* need to keep swingunit up to date */
			asgnssv(mainll_p->u.ssv_p);
			continue;
		case S_STAFF:
			break;	/* break out and handle this staff */
		default:
			continue;
		}

		/* loop through every voice on this staff */
		for (vidx = 0; vidx < MAXVOICES; vidx++) {

			swingunit = vvpath(mainll_p->u.staff_p->staffno,
					vidx + 1, SWINGUNIT)->swingunit;

			/* skip this voice if swingunit was not set */
			if (EQ(swingunit, Zero)) {
				continue;
			}

			/* various rationals we will need in the loop below */
			halfswing = rdiv(swingunit, Two);
			threefourthsswing = rmul(swingunit, Three_fourths);
			onefourthswing = rmul(swingunit, One_fourth);
			sixthswing = rdiv(swingunit, six);
			twelfthswing = rdiv(swingunit, twelve);

			/* accumulate starttime */
			starttime = Zero;

			/* for lint; we'll never really check it when it's 0 */
			prev_p = 0;

			/* find first nongrace group in voice (0 if none) */
			gs_p = mainll_p->u.staff_p->groups_p[vidx];
			gs_p = gs_p != 0 && gs_p->grpvalue == GV_ZERO ?
					nextnongrace(gs_p) : gs_p;

			/* loop through every nongrace group in this voice */
			for ( ; gs_p != 0; gs_p = nextnongrace(gs_p)) {

				quot = rdiv(starttime, swingunit);

				/* set starttime for the following group here
				 * because we may alter fulltime below */
				starttime = radd(starttime, gs_p->fulltime);

				/* handle case 1 (see prolog above) */
				if (quot.d == 2 &&
				    GE(gs_p->fulltime, halfswing) &&
				    GE(prev_p->fulltime, halfswing)) {

					prev_p->fulltime = radd(
						prev_p->fulltime, sixthswing);
					gs_p->fulltime = rsub(
						gs_p->fulltime, sixthswing);
				}

				/* handle case 2 (see prolog above) */
				if (quot.d == 4 && quot.n % 4 == 3 &&
				    GE(gs_p->fulltime, onefourthswing) &&
				    GE(prev_p->fulltime, threefourthsswing)) {

					prev_p->fulltime = rsub(
						prev_p->fulltime, twelfthswing);
					gs_p->fulltime = radd(
						gs_p->fulltime, twelfthswing);
				}

				prev_p = gs_p;
			}
		}
	}
}

/*
 * Name:        voicevis()
 *
 * Abstract:    If this voice is to be invisible, make it a measure space.
 *
 * Returns:     pointer to first GRPSYL; this equals the input pointer if the
 *		voice is visible, else it points at the new measure space GRPSYL
 *
 * Description: This function finds out if the given voice is supposed to be
 *		invisible this measure.  If so, it throws away the current
 *		GRPSYL list, replacing it with a single GRPSYL that is a
 *		measure space.  See the big comment in svpath() in ssv.c.
 */


static struct GRPSYL *
voicevis(gs_p)

struct GRPSYL *gs_p;		/* first GRPSYL in this voice's linked list */

{
	struct GRPSYL *ngs_p;	/* pointer to the new GRPSYL */


	/*
	 * At this point we know that the command line -s option allows this
	 * staff to be printed, and in fact this staff will be printed, because
	 * at least one of its voice(s) is supposed to be.  (We know this
	 * because where we were called from, staff_p->visible == YES, and that
	 * was set by calling svpath().)  If the staff weren't being printed,
	 * we wouldn't need to bother wiping out its invisible voice(s).
	 */

	/*
	 * This voice must be changed to a measure space if the -s option says
	 * it should be invisible, or if the SSVs say so.  The vvpath function
	 * checks both things.
	 */
	if (vvpath(gs_p->staffno, gs_p->vno, VISIBLE)->visible == NO) {

		/*
		 * Allocate a new GRPSYL and make it a measure space, with the
		 * same inputlineno etc. as the first in the current list.
		 */
		ngs_p = newGRPSYL(GS_GROUP);

		ngs_p->inputlineno = gs_p->inputlineno;
		ngs_p->inputfile = gs_p->inputfile;
		ngs_p->staffno = gs_p->staffno;
		ngs_p->vno = gs_p->vno;
		ngs_p->grpsyl = GS_GROUP;
		ngs_p->is_meas = YES;
		ngs_p->basictime = -1;

		/* in one compiler the following had to be done separately */
		/* like this, because it couldn't do the structure assignment */
		ngs_p->fulltime.n = Score.time.n;
		ngs_p->fulltime.d = Score.time.d;

		ngs_p->grpcont = GC_SPACE;
		ngs_p->prev = 0;
		ngs_p->next = 0;
		ngs_p->gs_p = 0;

		/* throw away the old GRPSYL list */
		free_grpsyls(gs_p);

		return (ngs_p);	/* ret pointer to the measure space */
	}

	return (gs_p);	/* ret pointer to the original, unchanged GRPSYL */
}
