/* Copyright (c) 1995, 1996, 1997, 1998, 2000, 2001, 2003, 2004, 2005 by Arkkra Enterprises */
/* All rights reserved */
/*
 * Name:	setnotes.c
 *
 * Description:	This file contains functions for setting relative vertical
 *		locations of notes.  It also sets relative vertical locations
 *		of the groups that contain notes, considering only the notes.
 *		It also sets the directions of stems.
 */

#include "defines.h"
#include "structs.h"
#include "globals.h"

static void locnotes P((void));
static void locllnotes P((struct MAINLL *mll_p, int v,
		struct MAINLL *nextbar_p));
static void chktabcollision P((struct GRPSYL *start_p));
static void intertab P((struct GRPSYL *gs_p, struct MAINLL *mll_p));
static void setstems P((void));
static void setonestem P((struct GRPSYL *gs_p));
static void setopstem P((struct GRPSYL *gs1_p, struct GRPSYL *gs2_p));
static void setfreestem P((struct GRPSYL *gs1_p, struct GRPSYL *gs2_p));
static void set1freestem P((struct GRPSYL *this_p, struct GRPSYL *other_p,
		int stemdir));
static void setbeamedstem P((struct GRPSYL *start_p, int stemdir));
static void dobunch P((struct GRPSYL *start_p, struct GRPSYL *end_p));
static void dograce P((struct GRPSYL *gs1_p, struct GRPSYL *gs2_p));
static int setv3stem P((struct GRPSYL *gs_p, int stemdir));
static int dov3bunch P((struct GRPSYL *start_p, struct GRPSYL *end_p,
		int stemdir));
static void setheads P((void));
static void setvoiceheads P((struct MAINLL *mll_p, struct GRPSYL *gs_p,
		int stafflines, short *shapes, int is_tab, int allx_hsi,
		int sharps, int keylet));
static void fixoneline P((void));

/*
 * Name:        setnotes()
 *
 * Abstract:    Sets the relative vert coords of each note, and stem dir.
 *
 * Returns:     void
 *
 * Description: This function calls subroutines to set the relative vertical
 *		coordinates of each note and each note group (considering only
 *		the note heads in it at this point), stem directions, and which
 *		notehead characters to use.
 */

void
setnotes()

{
	debug(16, "setnotes");

	locnotes();
	setstems();
	setheads();
	fixoneline();
}

/*
 * Name:        locnotes()
 *
 * Abstract:    Sets the relative vertical coordinates of each note.
 *
 * Returns:     void
 *
 * Description: This function loops through the main linked list, finding every
 *		STAFF structure.  It calls a subroutine to process each list of
 *		list of GRPSYLs for groups (not syllables).
 */

static void
locnotes()

{
	register struct MAINLL *mainll_p; /* point item in main linked list */
	int v;				/* index to voice linked lists */
	int did_a_voice;		/* have we processed a voice in meas?*/
	struct TIMEDSSV *tssv_p;	/* point along a timed SSV list */
	struct MAINLL *nextbar_p;	/* the next bar in the MLL */


	debug(16, "locnotes");
	initstructs();			/* clean out old SSV info */

	did_a_voice = NO;	/* prevent useless "used before set" warning */
	nextbar_p = 0;		/* prevent useless "used before set" warning */

	/*
	 * Loop through the main linked list, processing voices.  MLL SSVs are
	 * assigned when encountered.  But we also have to handle timed SSVs,
	 * because they may change the clef.  This algorithm would be simpler
	 * if we called setssvstate() after each voice (to undo the timed
	 * SSVs), and then always reapplied them when we get to the next bar.
	 * But to save time, we don't undo them after the last voice, and so
	 * usually don't have to reassign them at the bar (unless all the
	 * visible voices had measure repeats, and so we never assigned any).
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		switch (mainll_p->str) {
		case S_SSV:
			asgnssv(mainll_p->u.ssv_p);
			break;

		case S_CHHEAD:
			/* find the next bar line */
			for (nextbar_p = mainll_p; nextbar_p->str != S_BAR;
					nextbar_p = nextbar_p->next) {
				;
			}
			/* we haven't processed a voice in this measure yet */
			did_a_voice = NO;
			break;

		case S_STAFF:
			/* if invisible, there is nothing to do */
			if (mainll_p->u.staff_p->visible == NO) {
				break;
			}

			/* loop through the voice(s), and process each list */
			for (v = 0; v < MAXVOICES && mainll_p->u.staff_p
					->groups_p[v] != 0; v++) {

				/* meas rpt/rest/space have no notes to do */
				if (mainll_p->u.staff_p->groups_p[v]->is_meas
						== YES) {
					continue;
				}

				/*
				 * If this is not the first voice we've done in
				 * this measure, and there are timed SSVs,
				 * locllnotes() assigned them when we were in
				 * there for the previous voice we did.  So set
				 * the SSVs back to the state they were in at
				 * the start of the measure.
				 */
				if (did_a_voice == YES && nextbar_p->u.bar_p->
							timedssv_p != 0){
					setssvstate(mainll_p);
				}
				locllnotes(mainll_p, v, nextbar_p);
				did_a_voice = YES;
			}
			break;

		case S_BAR:
			if (did_a_voice == NO) {
				for (tssv_p = mainll_p->u.bar_p->timedssv_p;
						tssv_p != 0;
						tssv_p = tssv_p->next) {
					asgnssv(&tssv_p->ssv);
				}
			}
			break;
		}
	}
}

/*
 * Name:        locllnotes()
 *
 * Abstract:    Set the "stepsup" field for the notes in one GRPSYL list.
 *
 * Returns:     void
 *
 * Description: This function goes down one of the linked lists of GRPSYLs,
 *		one that is for groups, not syllables, and sets the stepsup
 *		value.
 */

static void
locllnotes(mll_p, v, nextbar_p)

struct MAINLL *mll_p;		/* point at the MLL struct voice hangs off */
int v;				/* voice to loop through */
struct MAINLL *nextbar_p;	/* point at MLL for the next bar line */

{
	register int upfromc4;	/* steps up from middle C */
	register int n;		/* loop through all notes in a group */
	int s;			/* staff number */
	int slines;		/* lines in this staff */
	int clef;		/* the clef currently in operation */
	int newclef;		/* the new clef, in case it changes */
	struct GRPSYL *gs_p;	/* starts pointing at first GRPSYL in list */
	struct TIMEDSSV *tssv_p;/* point along a timed SSV list */
	RATIONAL offset;	/* current group's offset into measure */


	s = mll_p->u.staff_p->staffno;
	debug(32, "locllnotes file=%s line=%d staff=%d vidx=%d",
			mll_p->inputfile, mll_p->inputlineno, s, v);
	slines = svpath(s, STAFFLINES)->stafflines;

	/* find the initial clef for this staff */
	clef = svpath(s, CLEF)->clef;

	/* point at the first timed SSV for this measure, if there is one */
	tssv_p = nextbar_p->u.bar_p->timedssv_p;
	offset = Zero;		/* first group's offset into measure */

	/* loop through every group in this voice */
	for (gs_p = mll_p->u.staff_p->groups_p[v]; gs_p != 0;
				gs_p = gs_p->next) {

		/* if no timed SSVs, don't waste time doing the following */
		if (tssv_p != 0) {
			/* assign timed SSVs before current offset */
			while (tssv_p != 0 && LT(tssv_p->time_off, offset)) {
				asgnssv(&tssv_p->ssv);
				tssv_p = tssv_p->next;
			}

			/* get clef state just before this group */
			clef = svpath(s, CLEF)->clef;

			/* assign timed SSVs at current offset */
			while (tssv_p != 0 && EQ(tssv_p->time_off, offset)) {
				asgnssv(&tssv_p->ssv);
				tssv_p = tssv_p->next;
			}

			/* get clef for this group */
			newclef = svpath(s, CLEF)->clef;

			/*
			 * If the clef changed at this time, set it in GRPSYL.
			 * This could happen with multiple voices on the staff.
			 * If so, we'll later erase clef from all but one; but
			 * the choice depends on the coords, which we don't
			 * know yet, so that is done later.  The erasing is
			 * done in eraseclef() in restsyl.c.
			 */
			if (newclef != clef) {
				clef = newclef;
				gs_p->clef = clef;
			}

			/* add our group's dur to get ready for next iteration*/
			offset = radd(offset, gs_p->fulltime);
		}

		/* nothing more to do for rests or spaces */
		if (gs_p->grpcont != GC_NOTES)
			continue;

		/*
		 * We found a group consisting of notes, normal or tablature.
		 * First handle the tablature case.
		 */
		if (clef == TABCLEF) {
			/*
			 * Make sure this voice's notes don't collide with
			 * some later voice's notes.
			 */
			chktabcollision(gs_p);

			for (n = 0; n < gs_p->nnotes; n++) {
				/*
				 * Set stepsup to be on the appropriate string.
				 */
				/* calc steps up from middle of staff */
				gs_p->notelist[n].stepsup = slines
					- 1 - 2 * gs_p->notelist[n].STRINGNO;

			}

			continue;
		}

		/*
		 * We found a non-tablature group consisting of notes.  For
		 * each note, find the number of steps it is up from middle C
		 * and from the center line of the staff.  (However, for 1-line
		 * staffs, we assume center line for now.)
		 * For CSS notes, we apply an offset to keep it far from the
		 * normal notes.  This is so that setgrps.c will understand
		 * that CSS and non-CSS notes in a group never interfere.
		 * Later, absvert.c will find the true stepsup on the other
		 * staff.
		 */
		for (n = 0; n < gs_p->nnotes; n++) {
			/* set steps up from middle line of staff */
			if (slines == 5) {
				/* get steps up from middle C */
				upfromc4 = (gs_p->notelist[n].octave - 4) * 7 +
				Letshift[ gs_p->notelist[n].letter - 'a' ];

				gs_p->notelist[n].stepsup = upfromc4
						+ clef - ALTO;
				if (gs_p->stemto == CS_ABOVE &&
						n <= gs_p->stemto_idx) {
					gs_p->notelist[n].stepsup += CSS_STEPS;
				} else if (gs_p->stemto == CS_BELOW &&
						n >= gs_p->stemto_idx) {
					gs_p->notelist[n].stepsup -= CSS_STEPS;
				}
			} else {
				/* 1-line staff; assume center line for now */
				gs_p->notelist[n].stepsup = 0;
			}
		}
	}

	/*
	 * Assign any timed SSVs that came after the last group, so that we are
	 * in the right state for the next measure (if we are the last voice).
	 */
	while (tssv_p != 0) {
		asgnssv(&tssv_p->ssv);
		tssv_p = tssv_p->next;
	}
}

/*
 * Name:        chktabcollision()
 *
 * Abstract:    Error if this GRPSYL conflicts with others on this staff.
 *
 * Returns:     void
 *
 * Description: This function checks for collisions between notes in this
 *		GRPSYL and notes in GRPSYLs of later voices in this chord on
 *		this staff.  On a tab staff, no two voices are allowed to use
 *		the same string at the same time.  If the frets are different,
 *		it would be impossible to play, and it seems best to disallow
 *		it even if they agreed.  So if this happens, do an l_ufatal.
 */

static void
chktabcollision(start_p)

struct GRPSYL *start_p;		/* first voice on this staff in this chord */

{
	int sv[MAXTABLINES];	/* which voice, if any, is using each string */
	int sidx;		/* string index, starting at 0 */
	struct GRPSYL *gs_p;	/* a GRPSYL on this staff in this chord */
	int n;			/* loop through notes (frets) in GRPSYL */


	/* if this chord has no more voices on this staff, return */
	if (start_p->gs_p == 0 ||
	    start_p->gs_p->grpsyl == GS_SYLLABLE ||
	    start_p->gs_p->staffno != start_p->staffno)
		return;

	/* we care only about notes (frets); rests and spaces are invisible */
	if (start_p->grpcont != GC_NOTES)
		return;

	/* init each string to an invalid voice number */
	for (sidx = 0; sidx < MAXTABLINES; sidx++) {
		sv[sidx] = 0;
	}

	/*
	 * Loop from this voice through the last voice on this staff that has
	 * a GRPSYL in this chord.  Don't worry about preceding voices; they
	 * already were in here and were checked against all these voices.
	 */
	for (gs_p = start_p; gs_p != 0 && start_p->gs_p->grpsyl == GS_GROUP &&
			gs_p->staffno == start_p->staffno; gs_p = gs_p->gs_p) {

		/* put each note into array, checking if string already used */
		for (n = 0; n < gs_p->nnotes; n++) {

			if (sv[(int)gs_p->notelist[n].STRINGNO] != 0) {

				l_ufatal(start_p->inputfile,
					 start_p->inputlineno,
					 "voices %d and %d on staff %d are using the \"%s\" string at the same time",
					 sv[(int)gs_p->notelist[n].STRINGNO],
					 gs_p->vno,
					 gs_p->staffno,
					 stringname(gs_p->notelist[n].STRINGNO,
						gs_p->staffno));
			}

			sv[(int)gs_p->notelist[n].STRINGNO] = gs_p->vno;
		}
	}
}

/*
 * Name:        intertab()
 *
 * Abstract:    Do additional work between tablature groups.
 *
 * Returns:     void
 *
 * Description: This function does checks to prevent certain bend sequences
 *		on a tab staff.  (It's unclear how such things would be drawn.)
 *		Also, when it finds the end of a single consecutive bend, it
 *		alters the previously set northern group boundaries of the
 *		groups, so that the arrows pointing at the bend strings will go
 *		up and down appropriately.
 *
 *		This function is called only with groups that have real bends
 *		(regular or prebends).
 */
#define	MAXBDIST	20	/* no. of unique bend distances in a sequence*/

static void
intertab(gs_p, mll_p)

struct GRPSYL *gs_p;		/* point at current tablature group */
struct MAINLL *mll_p;		/* point at the main LL struct it hangs off */

{
	RATIONAL bdist[MAXBDIST];	/* array of bend distances */
	RATIONAL bd;			/* a bend distance */
	int bdidx;			/* index into table of bend distances*/
	struct GRPSYL *nextgs_p;	/* point at the next GRPSYL */
	struct GRPSYL *gs2_p;		/* point at earlier GRPSYLs */
	struct MAINLL *mll2_p;		/* needed for crossing bar lines */
	int count, count2;		/* count numbers of bends */
	int n, k, j;			/* loop variables */
	int idx;			/* index into a notelist */
	int bad;			/* was a bad thing found? */


	/* count how many nonnull bends end at this group, remember last one */
	count = 0;
	idx = 0;		/* prevent useless 'used before set' warning */
	for (n = 0; n < gs_p->nnotes; n++) {
		if (HASREALBEND(gs_p->notelist[n])) {
			count++;
			idx = n;		/* remember where bend is */
		}
	}

	/* enforce restrictions on the following group, if there is one */
	mll2_p = mll_p;		/* we don't want to disturb mll_p */
	nextgs_p = nextgrpsyl(gs_p, &mll2_p);
	count2 = 0;	/* how many nonnull nonprebend bends are in *nextgs_p */

	if (nextgs_p != 0 && nextgs_p->grpcont == GC_NOTES) {

		bad = NO;	/* init to "nothing is bad" */

		for (n = 0; n < nextgs_p->nnotes; n++) {

			/* if this note has a nonnull nonprebend bend */
			if (HASREALBEND(nextgs_p->notelist[n]) &&
			    nextgs_p->notelist[n].FRETNO == NOFRET) {

				count2++;
				if (count > 1) {
					l_ufatal(gs_p->inputfile,
						gs_p->inputlineno,
						"no bend (other than a release) is allowed to follow a multiple string bend");
				}

				if (count2 > 1) {
					l_ufatal(gs_p->inputfile,
						gs_p->inputlineno,
						"only single string bends are allowed to be consecutive");
				}

				if (nextgs_p->notelist[n].STRINGNO !=
						gs_p->notelist[idx].STRINGNO) {
					bad = YES;
				}
			}
		}
		/*
		 * We check "bad" here instead of inside the above loop,
		 * because we want to give priority to the "only single string
		 * bends . . ." message if that condition is happening.
		 */
		if (bad == YES) {
			l_ufatal(gs_p->inputfile, gs_p->inputlineno,
				"consecutive bends must be on the same string");
		}
	}

	/*
	 * We know the current group has bend(s).  If the following group has
	 * a nonnull nonprebend bend, just return now.  We will handle this
	 * bend sequence when we find the last nonnull bend in it.
	 */
	if (count2 > 0)
		return;

	/*
	 * Loop backwards through the sequence of bends.  The start should be
	 * either a bend following no nonnull bend, or a prebend.  While
	 * searching, build a table of all the unique bend distances.  Usually
	 * we break out by finding a group with a nonnull bend, which means we
	 * went one too far with gs2_p, and gs_p is the start of the sequence.
	 * But if we hit the start, gs2_p will become 0 and we get out of the
	 * loop naturally.  Again, gs_p is the start of the sequence.
	 */
	bdidx = 0;	/* number of unique bend distances found */
	gs2_p = gs_p;
	while (gs2_p != 0) {
		/* find which note, if any, has the bend in this group */
		for (n = 0; n < gs2_p->nnotes; n++) {
			if (HASREALBEND(gs2_p->notelist[n])) {
				bd = ratbend(&gs2_p->notelist[n]);
				break;
			}
		}

		if (n < gs2_p->nnotes) {
			/*
			 * We found a nonnull bend.  Search the bdist array to
			 * see if this value has already occurred.  Get out
			 * when the value is found, or when we find a greater
			 * value (the list is in ascending order).
			 */
			for (k = 0; k < bdidx; k++) {
				if (GE(bdist[k], bd))
					break;
			}
			if (k == bdidx) {
				/* bd > everything in the array */
				/* add it at the end */
				bdist[k] = bd;
				bdidx++;
			} else if (GT(bdist[k], bd)) {
				/* bd should be put at this position */
				/* move all later ones down a notch */
				for (j = bdidx - 1; j >= k; j--)
					bdist[j+1] = bdist[j];
				bdist[k] = bd;
				bdidx++;
			}
			/* else bd is already in the table */

			if (bdidx >= MAXBDIST)
				pfatal("too many unique bend distances in sequence of bends");
			/* if this bend was a prebend, break */
			if (gs2_p->notelist[n].FRETNO != NOFRET) {
				gs_p = gs2_p;	/* series starts at prebend */
				break;
			}
		} else {
			/* there was no bend; start at the following group; */
			/* gs_p is now the beginning of the sequence */
			break;
		}

		/*
		 * It was a nonprebend bend.  Point gs2_p to the preceding
		 * group, remember the one we just looked at in gs_p, and keep
		 * looping.
		 */
		gs_p = gs2_p;
		gs2_p = prevgrpsyl(gs2_p, &mll_p);
	}

	/*
	 * Loop forward through these groups.  For each one, alter its northern
	 * boundary according to where its bend distance occurs in the bdist
	 * table.  This will cause the print phase to print the bend strings
	 * at varying heights so that the arrows will bend up and down as
	 * appropriate.
	 */
	while (gs_p != nextgs_p && gs_p != 0) {
		/* find which note has the bend in this group, get distance */
		for (n = 0; n < gs_p->nnotes; n++) {
			if (HASREALBEND(gs_p->notelist[n])) {
				bd = ratbend(&gs_p->notelist[n]);
				break;
			}
		}
		/* find distance in table, raise RN proportionally to index */
		for (n = 0; n < bdidx; n++) {
			if (EQ(bdist[n], bd)) {
				gs_p->c[RN] += 3.0 * STEPSIZE * TABRATIO * n;
				break;
			}
		}

		gs_p = nextgrpsyl(gs_p, &mll_p);
	}
}

/*
 * Name:        setstems()
 *
 * Abstract:    Sets stem direction for each group.
 *
 * Returns:     void
 *
 * Description: This function sets the stem direction for each group, based
 *		on the voice scheme at the time and other factors.
 */

static void
setstems()

{
	/* remember the previous stem direction of voice 3 on each staff */
	short v3stemdir[MAXSTAFFS + 1];

	int staffno;			/* staff number */
	int n;				/* loop variable */
	register struct MAINLL *mainll_p; /* point at main linked list item */
	int vscheme;			/* current voice scheme */


	debug(16, "setstems");
	initstructs();			/* clean out old SSV info */

	/* set initial default direction of voice 3 stems to be UP */
	for (n = 1; n <= MAXSTAFFS; n++)
		v3stemdir[n] = UP;

	/*
	 * Loop once for each item in the main linked list.  Apply any SSVs
	 * that are found.  Call subroutines to process linked lists of
	 * groups.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {
		if (mainll_p->str == S_SSV) {

			asgnssv(mainll_p->u.ssv_p);

		} else if (mainll_p->str == S_STAFF &&
				mainll_p->u.staff_p->visible == YES &&
				! is_mrpt(mainll_p->u.staff_p->groups_p[0])) {
			/*
			 * We've found a visible staff, which will have one
			 * or more voices, depending on the voice scheme.
			 */
			staffno = mainll_p->u.staff_p->staffno;
			vscheme = svpath(staffno, VSCHEME)->vscheme;

			switch (vscheme) {
			case V_1:
				/*
				 * There's only one voice on this staff, so
				 * call a routine to decide which way to point
				 * each stem.  It handles both grace & nongrace.
				 */
				setonestem(mainll_p->u.staff_p->groups_p[0]);
				break;

			case V_2OPSTEM:
				/*
				 * There are two voices on this staff, and
				 * the stems are always supposed to point
				 * opposite.  Call a routine to mark their
				 * stem directions.  It handles both nongrace
				 * and grace.
				 */
				setopstem(mainll_p->u.staff_p->groups_p[0],
					  mainll_p->u.staff_p->groups_p[1]);
				break;

			case V_2FREESTEM:
				/*
				 * There are two voices on this staff, and
				 * the stems are free to point either way
				 * when one voice is a space.  Call routines
				 * to mark their stem directions; first
				 * nongrace, then grace.
				 */
				setfreestem(mainll_p->u.staff_p->groups_p[0],
					    mainll_p->u.staff_p->groups_p[1]);
				dograce(mainll_p->u.staff_p->groups_p[0],
					mainll_p->u.staff_p->groups_p[1]);

				break;

			case V_3OPSTEM:
				/*
				 * This is just like V_2OPSTEM for the first
				 * two voices, but also allows a voice 3.
				 */
				setopstem(mainll_p->u.staff_p->groups_p[0],
					  mainll_p->u.staff_p->groups_p[1]);
				v3stemdir[staffno] = setv3stem(
					mainll_p->u.staff_p->groups_p[2],
					v3stemdir[staffno]);
				break;

			case V_3FREESTEM:
				/*
				 * This is just like V_2FREESTEM for the first
				 * two voices, but also allows a voice 3.
				 */
				setfreestem(mainll_p->u.staff_p->groups_p[0],
					    mainll_p->u.staff_p->groups_p[1]);
				dograce(mainll_p->u.staff_p->groups_p[0],
					mainll_p->u.staff_p->groups_p[1]);
				v3stemdir[staffno] = setv3stem(
					mainll_p->u.staff_p->groups_p[2],
					v3stemdir[staffno]);

				break;
			}
		}
	}
}

/*
 * Name:        setonestem()
 *
 * Abstract:    Sets stem direction for each group in a linked list for V_1.
 *
 * Returns:     void
 *
 * Description: This function sets the stem direction for each group in a
 *		linked list for a voice/measure, for the case where there is
 *		only one voice on the staff.
 */

static void
setonestem(gs_p)

struct GRPSYL *gs_p;	/* starts pointing at the first GRPSYL in a list */

{
	register struct GRPSYL *start_p, *end_p; /* first and last of a set */


	debug(32, "setonestem file=%s line=%d", gs_p->inputfile,
			gs_p->inputlineno);
	/*
	 * Loop once for each bunch of groups that must be stemmed the same
	 * way.  A beamed group must all be stemmed the same way, but nonbeamed
	 * notes are independent.
	 */
	start_p = gs_p;
	for (;;) {
		/*
		 * Find next group that has nongrace notes.  While doing this,
		 * set the stemdir for any grace groups encountered.  For V_1,
		 * grace stems always go up.
		 */
		while (start_p != 0 && (start_p->grpcont != GC_NOTES ||
					start_p->grpvalue == GV_ZERO)) {
			if (start_p->grpcont == GC_NOTES) /* must be grace */
				start_p->stemdir = UP;
			start_p = start_p->next;
		}
		if (start_p == 0)	/* get out if no more this measure */
			break;

		/* if this group is not beamed, handle it, and point at next */
		if (start_p->beamloc == NOITEM) {
			dobunch(start_p, start_p->next);
			start_p = start_p->next;
			continue;
		}

		/*
		 * Find end of this beamed group, setting grace groups UP.  If
		 * this is a cross staff beamed group, we may be starting at an
		 * INITEM or even the ENDITEM, since on this staff STARTITEM
		 * may have been a space.  But that doesn't matter; we still
		 * look for ENDITEM, whether or not it's also a space; and
		 * dobunch handles these cases.
		 */
		for (end_p = start_p; end_p != 0 &&
		(end_p->grpvalue == GV_ZERO || end_p->beamloc != ENDITEM);
		end_p = end_p->next) {
			if (end_p->grpvalue == GV_ZERO)
				end_p->stemdir = UP;
		}
		if (end_p == 0)
			pfatal("beamed group is not terminated");

		/* handle this bunch of groups, and point at next */
		dobunch(start_p, end_p->next);
		start_p = end_p->next;
	}
}

/*
 * Name:        setopstem()
 *
 * Abstract:    Sets stemdir for v1 or v2 groups for V_2OPSTEM/V_3OPSTEM.
 *
 * Returns:     void
 *
 * Description: This function sets the stem direction for each group in
 *		2 linked lists for a staff/measure, for the case where
 *		the linked list is for voice 1 or voice 2 and stems are always
 *		supposed to be opposed.  This function does both grace and
 *		nongrace groups.  For this vscheme, they act the same.
 *		The user can force the stems against the normal direction,
 *		except that the parse phase blocks any forcing of grace groups.
 */

static void
setopstem(gs1_p, gs2_p)

register struct GRPSYL *gs1_p;	/* starts at first GRPSYL in voice 1 list */
register struct GRPSYL *gs2_p;	/* starts at first GRPSYL in voice 2 list */

{
	debug(32, "setopstem file=%s line=%d", gs1_p->inputfile,
			gs1_p->inputlineno);
	/* mark first voice's stems up */
	while (gs1_p != 0) {
		/* if notes or starttime (needed for CSB), mark direction */
		if (gs1_p->grpcont == GC_NOTES || gs1_p->beamloc == STARTITEM) {
			/* if grace, or not in beamed group, try to set UP */
			if (gs1_p->grpvalue == GV_ZERO ||
			    gs1_p->beamloc == NOITEM) {
				/* if not forced by user, set UP */
				if (gs1_p->stemdir == UNKNOWN) {
					gs1_p->stemdir = UP;
				}
			} else if (gs1_p->beamloc == STARTITEM) {
				/* do same for all groups in beamed set */
				setbeamedstem(gs1_p, UP);
			}
		}
		gs1_p = gs1_p->next;
	}

	/* mark second voice's stems down */
	while (gs2_p != 0) {
		/* if notes or starttime (needed for CSB), mark direction */
		if (gs2_p->grpcont == GC_NOTES || gs2_p->beamloc == STARTITEM) {
			/* if grace, or not in beamed group, try to set DOWN */
			if (gs2_p->grpvalue == GV_ZERO ||
			    gs2_p->beamloc == NOITEM) {
				/* if not forced by user, set DOWN */
				if (gs2_p->stemdir == UNKNOWN) {
					gs2_p->stemdir = DOWN;
				}
			} else if (gs2_p->beamloc == STARTITEM) {
				/* do same for all groups in beamed set */
				setbeamedstem(gs2_p, DOWN);
			}
		}
		gs2_p = gs2_p->next;
	}
}

/*
 * Name:        setfreestem()
 *
 * Abstract:    Sets stemdir for each group in 2 linked lists for V_2FREESTEM.
 *
 * Returns:     void
 *
 * Description: This function sets the stem direction for each (nongrace)
 *		group in 2 linked lists for a staff/measure, for the case
 *		where there are two voices on the staff and the stems are
 *		allowed to point either way for one voice when the other
 *		voice has a space.
 */

static void
setfreestem(gs1_p, gs2_p)

struct GRPSYL *gs1_p;	/* starts pointing at first GRPSYL in voice 1 list */
struct GRPSYL *gs2_p;	/* starts pointing at first GRPSYL in voice 2 list */

{
	debug(32, "setfreestem file=%s line=%d", gs1_p->inputfile,
			gs1_p->inputlineno);
	/* call to handle first voice, then call to handle second voice */
	set1freestem(gs1_p, gs2_p, UP);
	set1freestem(gs2_p, gs1_p, DOWN);
}

/*
 * Name:        set1freestem()
 *
 * Abstract:    Sets stemdir for v1 or v2 groups for V_2FREESTEM/V_3FREESTEM.
 *
 * Returns:     void
 *
 * Description: This function sets the stem direction for each (nongrace)
 *		group in one linked list for a staff/measure, for the case
 *		where the linked list is for voice 1 or voice 2 and stems are
 *		allowed to point either way for one voice when the other
 *		voice has a space.  The function sets the directions just
 *		for "this" voice; the other voice is only used as a reference
 *		(we need to check when it has spaces).
 */

static void
set1freestem(this_p, other_p, stemdir)

struct GRPSYL *this_p;	/* starts pointing at first GRPSYL in linked list */
			/* for the voice whose stems we are now setting */
struct GRPSYL *other_p;	/* starts pointing at first GRPSYL in linked list */
			/* for the other voice */
int stemdir;		/* which way the stem must point if forced */

{
	register struct GRPSYL *start_p, *end_p; /* first and last of a set */
	RATIONAL vtime, vtime2;		/* elapsed time this measure */


	debug(32, "set1freestem file=%s line=%d stemdir=%d", this_p->inputfile,
			this_p->inputlineno, stemdir);
	vtime = Zero;			/* init to no time elapsed */

	/*
	 * Loop once for each bunch of groups in this voice that must be
	 * stemmed the same way.  A beamed group must all be stemmed the same
	 * way, but nonbeamed notes are independent.
	 */
	start_p = this_p;
	for (;;) {
		/*
		 * Find next group that has nongrace notes, accumulating
		 * elapsed time.  This code depends on grace notes having
		 * zero duration.
		 */
		while (start_p != 0 && (start_p->grpcont != GC_NOTES ||
					start_p->grpvalue == GV_ZERO)) {
			vtime = radd(vtime, start_p->fulltime);
			start_p = start_p->next;
		}
		if (start_p == 0)	/* get out if no more this measure */
			break;

		/* if this group is not beamed, handle it, and point at next */
		if (start_p->beamloc == NOITEM) {
			vtime2 = radd(vtime, start_p->fulltime);

			if (hasspace(other_p, vtime, vtime2)) {
				/* other voice has space; decide stem */
				dobunch(start_p, start_p->next);
			} else {
				/*
				 * The other voice has notes/rests; force the
				 * the direction, unless the user has already
				 * forced it.
				 */
				if (start_p->stemdir == UNKNOWN) {
					start_p->stemdir = (short)stemdir;
				}
			}

			start_p = start_p->next;
			vtime = vtime2;
			continue;
		}

		/* find end of this beamed group, ignoring grace groups */
		vtime2 = vtime;
		for (end_p = start_p; end_p != 0 &&
		(end_p->grpvalue == GV_ZERO || end_p->beamloc != ENDITEM);
		end_p = end_p->next)
			vtime2 = radd(vtime2, end_p->fulltime);
		if (end_p == 0)
			pfatal("beamed group is not terminated");
		vtime2 = radd(vtime2, end_p->fulltime);	/* add in final note */

		/* handle this bunch of groups, and point at next */
		if (hasspace(other_p, vtime, vtime2)) {
			/* other voice has space; decide our stems */
			dobunch(start_p, end_p->next);
		} else {
			/* other voice has notes/rests; this forces ours */
			setbeamedstem(start_p, stemdir);
		}

		vtime = vtime2;
		start_p = end_p->next;
	}
}

/*
 * Name:        setbeamedstem()
 *
 * Abstract:    Sets stem direction in beamed set, favoring one direction.
 *
 * Returns:     void
 *
 * Description: This function is given the first group in a nongrace beamed
 *		set.  It sets all the stem directions, the same way of course.
 *		It sets them to the given stemdir, unless the user has
 *		overridden the direction.
 */

static void
setbeamedstem(start_p, stemdir)

struct GRPSYL *start_p;	/* point at the first GRPSYL in beamed set */
int stemdir;		/* which way we will try to point the stems */

{
	struct GRPSYL *g_p;		/* point into the set */
	int forcedir;			/* direction forced by user */


	forcedir = UNKNOWN;		/* no forcing yet */

	/* look for groups in this set where the user has forced stemdir */
	for (g_p = start_p; g_p != 0; g_p = g_p->next) {
		/* consider only nongrace note groups */
		if (g_p->grpcont != GC_NOTES || g_p->grpvalue == GV_ZERO) {
			continue;
		}
		/* if user forced the stemdir */
		if (g_p->stemdir != UNKNOWN) {
			if (forcedir == UNKNOWN) {
				/* first such occurrence; remember it */
				forcedir = g_p->stemdir;
			} else if (g_p->stemdir != forcedir) {
				/* any later occurrence must agree */
				l_warning(g_p->inputfile, g_p->inputlineno,
				"cannot have both 'up' and 'down' stem in same set of beamed or 'alt'-ed note groups");
				forcedir = g_p->stemdir; /* use latest */
			}
		}
		if (g_p->beamloc == ENDITEM) {
			break;
		}
	}

	/* if user forced any stems, we'll go with that direction */
	if (forcedir != UNKNOWN) {
		stemdir = forcedir;
	}

	/* set all the stems in this set */
	for (g_p = start_p; g_p != 0; g_p = g_p->next) {
		if (g_p->grpcont != GC_NOTES || g_p->grpvalue == GV_ZERO) {
			continue;
		}
		g_p->stemdir = stemdir;
		if (g_p->beamloc == ENDITEM) {
			break;
		}
	}
}
/*
 * Name:        dobunch()
 *
 * Abstract:    Sets stem direction for a single group or a beamed set.
 *
 * Returns:     void
 *
 * Description: This function is given a single (nongrace) group, or a set
 *		of them that will be beamed together, for the case where
 *		the stems are allowed to go either way.  It decides which
 *		way is best, and sets the result.
 */

static void
dobunch(start_p, end_p)

struct GRPSYL *start_p;	/* starts pointing at the first GRPSYL in a bunch */
struct GRPSYL *end_p;	/* starts pointing after the last GRPSYL in a bunch */

{
	register struct GRPSYL *gs_p;	/* point along list of them */
	int lonesum;		/* sum of offsets of single notes from center*/
	int topsum;		/* sum of offsets of top notes from center */
	int botsum;		/* sum of offsets of bottom notes from center*/
	int insum;		/* sum of offsets of inner notes from center */
	int n;			/* loop counter */
	int stemdir;		/* answer of where stems should point */


	/*
	 * Loop through all (nongrace) notes in these group(s), adding up
	 * the offsets of their outer notes from the center line.  For groups
	 * that have only one note, count this in lonesum.  For other groups,
	 * count the top notes and bottom notes separately.  We consider only
	 * outer notes in these counters, and we count single note groups
	 * separately to avoid counting the same note twice.  But to be able to
	 * breaks ties in the best way, we keep a separate counter for inner
	 * notes of groups that have 3 or more notes.
	 * While doing this, also keep track of whether the user requested a
	 * specific stem direction on any of these groups.  If so, there must
	 * not be any contradictions between what they asked for.
	 */
	lonesum = topsum = botsum = insum = 0;
	stemdir = UNKNOWN;	/* user hasn't asked for anything yet */
	for (gs_p = start_p; gs_p != end_p; gs_p = gs_p->next) {
		/*
		 * Consider only note groups.  Cross staff beaming can have
		 * spaces in the list of groups, and rests need to be skipped.
		 */
		if (gs_p->grpcont == GC_NOTES && gs_p->grpvalue == GV_NORMAL) {
			if (gs_p->stemdir != UNKNOWN) {
				if (stemdir == UNKNOWN) {
					stemdir = gs_p->stemdir;
				} else if (gs_p->stemdir != stemdir) {
					l_warning(gs_p->inputfile,
					gs_p->inputlineno,
					"cannot have both 'up' and 'down' stem in same set of beamed or 'alt'-ed note groups");
					stemdir = gs_p->stemdir;
				}
			}

			if (gs_p->nnotes == 1) {
				lonesum += gs_p->notelist[0].stepsup;
			} else {
				topsum += gs_p->notelist[0].stepsup;
				botsum += gs_p->notelist[ gs_p->nnotes - 1 ].
						stepsup;
			}

			/* this loop happens only if >= 3 notes in the group */
			for (n = 1; n < gs_p->nnotes - 1; n++ ) {
				insum += gs_p->notelist[n].stepsup;
			}
		}
	}

	/*
	 * If the user requested a stem direction, that's what they will get,
	 * for 5-line regular staffs, but for 1-line regular staffs stems are
	 * always UP and for tablature staffs, always DOWN.  For tab staffs, the
	 * parse phase blocks any user requests for stemdir, so we don't have
	 * to cover that in the warning and error messages here.
	 *
	 * For a regular 5-line staff where the user didn't specify, if we are
	 * involved in cross staff beaming, the direction defaults such that
	 * the beam ends up between the two staffs; else, these rules apply:
	 * If lonesum + topsum + botsum is positive, the "average" outer note
	 * in these group(s) is above the center line, so the stems should go
	 * down.  If negative, they should go up.  In case of tie, they should
	 * go down, unless we can break the tie by using the inner notes.
	 * For 1-line staff, the stem should go up, regardless.
	 */
	if (svpath(start_p->staffno, STAFFLINES)->stafflines == 5 &&
			is_tab_staff(start_p->staffno) == NO) {
		if (stemdir == UNKNOWN) {
			switch (start_p->beamto) {
			case CS_ABOVE:		/* bm with staff above */
				stemdir = UP;
				break;
			case CS_BELOW:		/* bm with staff below */
				stemdir = DOWN;
				break;
			case CS_SAME:		/* no cross staff beaming */
				/* normal case: base on note distances */
				if (lonesum + topsum + botsum > 0)
					stemdir = DOWN;
				else if (lonesum + topsum + botsum < 0)
					stemdir = UP;
				else
					stemdir = insum >= 0 ? DOWN : UP;
				break;
			}
		}
	} else if (is_tab_staff(start_p->staffno) == YES) {
		stemdir = DOWN;
	} else {
		if (stemdir == DOWN)
			l_ufatal(start_p->inputfile, start_p->inputlineno,
			"cannot specify 'down' stem on voice 1 or 2 of a one-line staff");
		if (stemdir == UP)
			l_warning(start_p->inputfile, start_p->inputlineno,
			"stem direction should not be specified on voice 1 or 2 of a one-line staff");
		stemdir = UP;	/* in case it was UNKNOWN */
	}

	/* mark all groups (doesn't hurt to mark rests and spaces too) */
	for (gs_p = start_p; gs_p != end_p; gs_p = gs_p->next) {
		if (gs_p->grpvalue == GV_NORMAL)
			gs_p->stemdir = (short)stemdir;
	}
}

/*
 * Name:        dograce()
 *
 * Abstract:    Sets stem direction for a single grace group.
 *
 * Returns:     void
 *
 * Description:	This function sets stem directions for grace groups when the
 *		vscheme V_2FREESTEM.  (The V_1 and V_2OPSTEM cases were handled
 *		along with nongrace groups.)  If the next nongrace group occurs
 *		at a time when the other voice has a space, the grace stem goes
 *		up (like V_1), else the same as the main group (like V_2OPSTEM).
 *		For the first voice, these rules boil down to the fact that
 *		graces stems are always up.  The second voice can end up
 *		going either way.
 */

static void
dograce(gs1_p, gs2_p)

register struct GRPSYL *gs1_p;	/* starts at first GRPSYL in voice 1 list */
register struct GRPSYL *gs2_p;	/* starts at first GRPSYL in voice 2 list */

{
	register struct GRPSYL *gs_p;	/* point along list of them */
	RATIONAL vtime;			/* elapsed time in measure */
	static RATIONAL tiny = {1, 4 * MAXBASICTIME};


	/* for the first voice, mark all grace stems up */
	for (gs_p = gs1_p; gs_p != 0; gs_p = gs_p->next) {
		if (gs_p->grpvalue == GV_ZERO)
			gs_p->stemdir = UP;
	}

	/*
	 * For the 2nd voice, loop though all groups.  For each nongrace group,
	 * accumulate the fulltime.  For each grace group, find out if the
	 * other voice has a space at the moment the following nongrace group
	 * starts.  If so, treat as V_1.  If not, treat as V_2OPSTEM.
	 */
	vtime = Zero;
	for (gs_p = gs2_p; gs_p != 0; gs_p = gs_p->next) {
		if (gs_p->grpvalue == GV_NORMAL) {
			vtime = radd(vtime, gs_p->fulltime);
		} else {
			/* does other voice have space? */
			if (hasspace(gs1_p, vtime, radd(vtime, tiny)) == YES) {
				gs_p->stemdir = UP;
			} else {
				gs_p->stemdir = DOWN;
			}
		}
	}
}

/*
 * Name:        setv3stem()
 *
 * Abstract:    Sets stem direction for each group in a linked list for voice 3.
 *
 * Returns:     default stem direction after this measure
 *
 * Description: This function sets the stem direction for each group in a
 *		linked list for a voice/measure that is for voice 3.  Voice 3
 *		ignores the other voices.
 */

static int
setv3stem(gs_p, stemdir)

struct GRPSYL *gs_p;	/* starts pointing at the first GRPSYL in a list */
int stemdir;		/* stem direction of the previous group */

{
	register struct GRPSYL *start_p, *end_p; /* first and last of a set */


	debug(32, "setv3stem file=%s line=%d", gs_p->inputfile,
			gs_p->inputlineno);
	/*
	 * Loop once for each bunch of groups that must be stemmed the same
	 * way.  A beamed group must all be stemmed the same way, but nonbeamed
	 * notes are independent.
	 */
	start_p = gs_p;
	for (;;) {
		/*
		 * Find next group that has nongrace notes.  While doing this,
		 * set the stemdir for any grace groups encountered.  For voice
		 * 3, grace stems always go up.
		 */
		while (start_p != 0 && (start_p->grpcont != GC_NOTES ||
					start_p->grpvalue == GV_ZERO)) {
			if (start_p->grpcont == GC_NOTES) /* must be grace */
				start_p->stemdir = UP;
			start_p = start_p->next;
		}
		if (start_p == 0)	/* get out if no more this measure */
			break;

		/* if this group is not beamed, handle it, and point at next */
		if (start_p->beamloc == NOITEM) {
			stemdir = dov3bunch(start_p, start_p->next, stemdir);
			start_p = start_p->next;
			continue;
		}

		/*
		 * Find end of this beamed group, setting grace groups UP.
		 * Note that voice 3 does not allow cross staff beaming.
		 */
		for (end_p = start_p; end_p != 0 &&
		(end_p->grpvalue == GV_ZERO || end_p->beamloc != ENDITEM);
		end_p = end_p->next) {
			if (end_p->grpvalue == GV_ZERO)
				end_p->stemdir = UP;
		}
		if (end_p == 0)
			pfatal("beamed group is not terminated");

		/* handle this bunch of groups, and point at next */
		stemdir = dov3bunch(start_p, end_p->next, stemdir);
		start_p = end_p->next;
	}

	return (stemdir);
}

/*
 * Name:        dov3bunch()
 *
 * Abstract:    Sets stem dir for a single group or a beamed set on voice 3.
 *
 * Returns:     stem direction that was chosen
 *
 * Description: This function is given a single (nongrace) group, or a set
 *		of them that will be beamed together, for voice 3.  It decides
 *		which stemdir is needed, and sets it for each group.
 */

static int
dov3bunch(start_p, end_p, stemdir)

struct GRPSYL *start_p;	/* starts pointing at the first GRPSYL in a bunch */
struct GRPSYL *end_p;	/* starts pointing after the last GRPSYL in a bunch */
int stemdir;		/* stem direction of the previous group */

{
	register struct GRPSYL *gs_p;	/* point along list of them */
	int userdir;			/* stemdir requested by user */


	/*
	 * Loop through all groups in this bunch, keeping track of any user-
	 * specified direction.  Grace groups are forced to UP but are other-
	 * wise ignored.  Nongrace groups could be rests, so ignore them.
	 */
	userdir = UNKNOWN;	/* user hasn't asked for anything yet */
	for (gs_p = start_p; gs_p != end_p; gs_p = gs_p->next) {
		if (gs_p->grpvalue == GV_ZERO) {
			gs_p->stemdir = UP;	/* grace group */
		} else if (gs_p->grpcont == GC_NOTES) {
			if (gs_p->stemdir != UNKNOWN) {	/* user request */
				if (userdir == UNKNOWN) {
					userdir = gs_p->stemdir;
				} else if (gs_p->stemdir != userdir) {
					l_warning(gs_p->inputfile,
					gs_p->inputlineno,
					"cannot have both 'up' and 'down' stem in same set of beamed or 'alt'-ed note groups");
					userdir = gs_p->stemdir;
				}
			}
		}
	}

	/* if user requested a direction, we will use that, else keep previous*/
	if (userdir != UNKNOWN)
		stemdir = userdir;

	/* mark all nongrace groups; it doesn't hurt to mark rests */
	for (gs_p = start_p; gs_p != end_p; gs_p = gs_p->next) {
		if (gs_p->grpvalue == GV_NORMAL)
			gs_p->stemdir = (short)stemdir;
	}

	return (stemdir);
}

/*
 * Name:        setheads()
 *
 * Abstract:    Set headshape, headfont, headchar, and coords for all notes.
 *
 * Returns:     void
 *
 * Description: This function sets the headshape, headfont, and headchar for
 *		all notes.  (However, the headchar is changed later, in
 *		setgrps.c, in certain cases where two GRPSYLs share a note.)
 *		It also sets the relative vertical coords of the notes and
 *		their groups.  We waited until now to do this so that stemdir
 *		would be known.
 */

static void
setheads()

{
	struct MAINLL *mainll_p;	/* point at main linked list item */
	struct STAFF *staff_p;		/* point at a STAFF */
	int stafflines;			/* lines in a tablature staff */
	int is_tab;			/* is this a tablature staff? */
	int sharps;			/* in the key sig */
	char keylet;			/* letter of the key, assuming major */
	short *shapes;			/* 7 shapes for the 7 notes */
	int vidx;			/* voice index */
	int allx_hsi;			/* headshape index for allx */


	debug(16, "setheads");
	initstructs();			/* clean out old SSV info */

	/* just in case we'll need it later */
	allx_hsi = get_shape_num("allx");

	/*
	 * Loop once for each item in the main linked list.  Apply any SSVs
	 * that are found.  For each voice on each staff, call setvoiceheads
	 * to do the the work.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		if (mainll_p->str == S_SSV) {
			/* apply the SSV and go to the next item */
			asgnssv(mainll_p->u.ssv_p);
			continue;
		}

		/* deal only with visible staffs that aren't measure rpts */
		if (mainll_p->str != S_STAFF ||
				mainll_p->u.staff_p->visible == NO ||
				is_mrpt(mainll_p->u.staff_p->groups_p[0])) {
			continue;
		}

		/*
		 * We found a staff to work on.  Set up some variables we'll
		 * be needing.
		 */
		staff_p = mainll_p->u.staff_p;
		stafflines = svpath(staff_p->staffno, STAFFLINES)->stafflines;
		is_tab = svpath(staff_p->staffno, CLEF)->clef == TABCLEF;

		/*
		 * Find the key letter.  We don't care about any sharp or flat
		 * in the key name, just the letter.  For tab it's meaningless,
		 * but that's okay.
		 */
		sharps = eff_key(staff_p->staffno);
		keylet = Circle[(sharps + 1 + 7) % 7];

		/* loop through every possible voice on this staff */
		for (vidx = 0; vidx < MAXVOICES; vidx++) {

			/* point at array of headshapes for this voice */
			shapes = vvpath(staff_p->staffno, vidx + 1,
					NOTEHEADS)->noteheads;

			setvoiceheads(mainll_p, staff_p->groups_p[vidx],
				stafflines, shapes, is_tab, allx_hsi, sharps,
				keylet);
		}
	}
}

/*
 * Name:        setvoiceheads()
 *
 * Abstract:    Set headshape, headfont, headchar, and coords for one GRPSYL.
 *
 * Returns:     void
 *
 * Description: This function sets the headshape, headfont, and headchar for
 *		one GRPSYL.  (However, the headchar is changed later, in
 *		setgrps.c, in certain cases where two GRPSYLs share a note.)
 *		It also sets the relative vertical coords of the notes and
 *		the group.
 */

static void
setvoiceheads(mll_p, gs_p, stafflines, shapes, is_tab, allx_hsi, sharps, keylet)

struct MAINLL *mll_p;		/* point at the main LL struct gs_p hangs off */
struct GRPSYL *gs_p;		/* starts at start of GRPSYL list */
int stafflines;			/* lines in a tablature staff */
short *shapes;			/* 7 shapes for the 7 notes */
int is_tab;			/* is this a tablature staff? */
int allx_hsi;			/* headshape index for allx */
int sharps;			/* in the key sig */
int keylet;			/* letter of the key, assuming major */

{
	float bendheight;		/* total height of bend numbers */
	int havebend;			/* any bends in this group? */
	int n;				/* loop variable */
	int i;				/* temp variable */
	int hfont;			/* font of note head */
	int hchar;			/* char of note head */
	float vhalf;			/* half the vert size of note head */
	int stepsup;			/* local copy */


	/* loop through every GRPSYL in voice (may be none) */
	for ( ; gs_p != 0; gs_p = gs_p->next) {

		/* we only care about notes, not rest/space */
		if (gs_p->grpcont != GC_NOTES) {
			continue;
		}

		bendheight = 0;		/* init to no bends */
		havebend = NO;

		/*
		 * Loop through every note in the GRPSYL, setting its
		 * headshape, head font/char, and coords.
		 */
		for (n = 0; n < gs_p->nnotes; n++) {

			/* if there is no note-level override... */
			if (gs_p->notelist[n].headshape == HS_UNKNOWN) {

				/* set to group-level override if present */
				gs_p->notelist[n].headshape = gs_p->headshape;

				/*
				 * If still no setting (which is the usual
				 * case), set according to what the SSVs said.
				 * Set i to how far note is above the tonic
				 * (assuming a major key).  Tab uses tonic.
				 * Then get shape from array.
				 */
				if (gs_p->notelist[n].headshape == HS_UNKNOWN) {

					if (is_tab) {
						i = 0;	/* arbitrary */
					} else {
						i = (gs_p-> notelist[n].letter
							+ 7 - keylet) % 7;
					}

					gs_p->notelist[n].headshape = shapes[i];
				}
			}

			/*
			 * Now that we know the stepsup (set in locllnotes())
			 * and the headshape, we can set the note's coords.
			 */
			if (is_tab && gs_p->notelist[n].headshape != allx_hsi) {

				/* handle tab (except when it's an X) */

				gs_p->notelist[n].c[RY] = gs_p->notelist[n].
					stepsup * TABRATIO * STEPSIZE;

				if (gs_p->notelist[n].FRETNO == NOFRET) {
					/* set RN and RS the same as RY */
					gs_p->notelist[n].c[RN] =
						gs_p->notelist[n].c[RY];
					gs_p->notelist[n].c[RS] =
						gs_p->notelist[n].c[RY];
				} else {
					/*
					 * Set vertical coordinates of the
					 * "note" (fret number).  It is to be
					 * centered on the appropriate line.
					 */
					vhalf = strheight(fret_string(&gs_p->
						notelist[n], gs_p)) / 2.0;
					gs_p->notelist[n].c[RN] =
						gs_p->notelist[n].c[RY] + vhalf;
					gs_p->notelist[n].c[RS] =
						gs_p->notelist[n].c[RY] - vhalf;
				}

			} else {

				/* handle non-tab and tab X-notes */

				/* find & store music font and char */
				hchar = nheadchar(gs_p->notelist[n].headshape,
					gs_p->basictime, gs_p->stemdir, &hfont);
				gs_p->notelist[n].headchar = hchar;
				gs_p->notelist[n].headfont = hfont;

				/* half the height of the note head */
				vhalf = height(hfont, gs_p->notelist[n].notesize
					== GS_NORMAL ? DFLT_SIZE : SMALLSIZE,
					hchar) / 2;

				/*
				 * Set actual relative vertical coords.  We need
				 * to recalculate the original stepsup, which
				 * was modified for CSS notes, because absvert.c
				 * needs to know what the note's coords would
				 * have been if it hadn't been CSS.  Sigh.
				 */
				stepsup = gs_p->notelist[n].stepsup;
				switch (gs_p->stemto) {
				case CS_ABOVE:
					if (n <= gs_p->stemto_idx) {
						stepsup -= CSS_STEPS;
					}
					break;
				case CS_BELOW:
					if (n >= gs_p->stemto_idx) {
						stepsup += CSS_STEPS;
					}
					break;
				}
				gs_p->notelist[n].c[RY] = stepsup * STEPSIZE *
					(is_tab ? TABRATIO : 1.0);

				gs_p->notelist[n].c[RN] =
					gs_p->notelist[n].c[RY] + vhalf;

				gs_p->notelist[n].c[RS] =
					gs_p->notelist[n].c[RY] - vhalf;
			}

			if (is_tab) {
				/*
				 * If there was a real bend, add to total height
				 * of the bend numbers.
				 */
				if (HASREALBEND(gs_p->notelist[n])) {
					bendheight += strheight(bend_string(
						&gs_p->notelist[n])) + STDPAD;
				}

				/* if any bend at all, remember it */
				if (HASBEND(gs_p->notelist[n])) {
					havebend = YES;
				}
			}
		}

		/*
		 * Set the group's coords.
		 */
		if (is_tab) {
			/*
			 * Set the group's north based on the top of the top
			 * bend number if there is one, otherwise the top of
			 * the top fret number.  We leave 3 "tab stepsizes" of
			 * white space between the staff and the lowest bend
			 * number, for the arrow.
			 */
			if (havebend == NO) {	/* no bends present */
				/* there must be frets, since no bends */
				gs_p->c[RN] = gs_p->notelist[0].c[RN] + STDPAD;
			} else {		/* bend(s) present */
				gs_p->c[RN] = (stafflines + 2) *
					STEPSIZE * TABRATIO + bendheight;
			}

			/*
			 * Set the group's south based on the bottom of the
			 * bottom fret number if there is one, otherwise the
			 * middle of the staff.
			 */
			if (gs_p->nnotes == 0) {	/* no frets present */
				gs_p->c[RS] = 0;
			} else {			/* frets present */
				gs_p->c[RS] = gs_p->notelist
					[ gs_p->nnotes - 1 ].c[RS] - STDPAD;
			}

			/* if bends, do work between this and other groups */
			if (bendheight > 0) {
				intertab(gs_p, mll_p);
			}
		} else {
			/*
			 * Non-tab: use the outermost non-CSS notes, but pad.
			 * If all notes are CSS, then set RN and RS to zero.
			 */
			switch (gs_p->stemto) {
			case CS_SAME:
				gs_p->c[RN] = gs_p->notelist
					[0].c[RN] + STDPAD;
				gs_p->c[RS] = gs_p->notelist
					[gs_p->nnotes-1].c[RS] - STDPAD;
				break;
			case CS_ABOVE:
				if (gs_p->stemto_idx == gs_p->nnotes - 1) {
					gs_p->c[RN] = gs_p->c[RS] = 0.0;
				} else {
					gs_p->c[RN] = gs_p->notelist
					   [gs_p->stemto_idx+1].c[RN] + STDPAD;
					gs_p->c[RS] = gs_p->notelist
					   [gs_p->nnotes-1].c[RS] - STDPAD;
				}
				break;
			case CS_BELOW:
				if (gs_p->stemto_idx == 0) {
					gs_p->c[RN] = gs_p->c[RS] = 0.0;
				} else {
					gs_p->c[RN] = gs_p->notelist
					   [0].c[RN] + STDPAD;
					gs_p->c[RS] = gs_p->notelist
					   [gs_p->stemto_idx-1].c[RS] - STDPAD;
				}
				break;
			}
		}
	}
}

/*
 * Name:        fixoneline()
 *
 * Abstract:    Fix stemsup and vertical coord for notes on one-line staffs.
 *
 * Returns:     void
 *
 * Description: stepsup and notes' vertical coords are set in locllnotes().
 *		For one-line staffs, it assumes the notes are on the line.
 *		But if the notes are not to be on the line, that isn't right.
 *		It depends on the stem direction.  Now that we have set the
 *		stem direction, we need to correct this info.
 */

static void
fixoneline()

{
	struct MAINLL *mainll_p;	/* point at main linked list item */
	struct STAFF *staff_p;		/* point at a STAFF */
	struct GRPSYL *gs_p;		/* point along a GRPSYL list */
	int v;				/* voice number, 0 or 1 */


	debug(16, "fixoneline");
	initstructs();			/* clean out old SSV info */

	/*
	 * Loop once for each item in the main linked list.  Apply any SSVs
	 * that are found.  Move notes that are not to be on the line.
	 */
	for (mainll_p = Mainllhc_p; mainll_p != 0; mainll_p = mainll_p->next) {

		if (mainll_p->str == S_SSV) {
			/* apply the SSV and go to the next item */
			asgnssv(mainll_p->u.ssv_p);
			continue;
		}

		/* deal only with visible staffs that aren't measure rpts */
		if (mainll_p->str != S_STAFF ||
				mainll_p->u.staff_p->visible == NO ||
				is_mrpt(mainll_p->u.staff_p->groups_p[0])) {
			continue;
		}

		staff_p = mainll_p->u.staff_p;

		/* deal only with non-tab one-line staffs */
		if (svpath(staff_p->staffno, STAFFLINES)->stafflines != 1 ||
				svpath(staff_p->staffno, CLEF)->clef
				== TABCLEF) {
			continue;
		}

		/*
		 * Loop through voices 1 and 2, and process each list.  Note
		 * that voice 3 is always on the line, so we don't need to do
		 * anything to it.
		 */
		for (v = 0; v < NORMVOICES && staff_p->groups_p[v] != 0; v++) {

			/* change stepsup from 0 only if notes not on the line*/
			if (vvpath(staff_p->staffno, v + 1, ONTHELINE)->
					ontheline == YES) {
				continue;
			}

			for (gs_p = staff_p->groups_p[v]; gs_p != 0;
					gs_p = gs_p->next) {

				/* only notes are to be changed */
				if (gs_p->grpcont != GC_NOTES) {
					continue;
				}

				/* move up or down a step based on stem dir */
				if (gs_p->stemdir == UP) {
					gs_p->notelist[0].stepsup = 1;
					gs_p->notelist[0].c[RY] = STEPSIZE;
					gs_p->notelist[0].c[RN] += STEPSIZE;
					gs_p->notelist[0].c[RS] += STEPSIZE;
				} else {
					gs_p->notelist[0].stepsup = -1;
					gs_p->notelist[0].c[RY] = -STEPSIZE;
					gs_p->notelist[0].c[RN] -= STEPSIZE;
					gs_p->notelist[0].c[RS] -= STEPSIZE;
				}
			}
		}
	}
}
