/*
 * msg.c - class routines to handle debug and error messages
 * $Id: errMessage.c,v 1.1 2000/01/29 20:44:58 kline Exp kline $
 *  adapted primarily from Kjetil T. Homme's p.d. errmessage code.
 */
extern		"C"
{
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h> 
#include <string.h>
#include "copyright.h"

}
#include "errMessage.h"


/* 
 * Default levels for masking out messages.
 */
severity_t msg_level[MSG_CATEGORIES] = {
    5, 5, 0, 0, 0,
};

/* The string prepended to each message. You may use date directives
 * like %T for time (see strftime(3)). Extension: You can use %s for
 * the severity level.
 */
const char *msg_preface[MSG_CATEGORIES] = 
{
    "Trace\t",
    "Debug\t",
    "Warning [%Y-%m-%d %H:%M.%S] level %s\n\t",
    "Error [%Y-%m-%d %H:%M.%S] level %s\n\t",
    "Fatal [%Y-%m-%d %H:%M.%S] level %s\n\t",
    "\n\tAborting\t",
};
const char xcategory_t[] = { 0, 1, 2, 3, 4 };

/* How many messages to accept in a given class before calling msgExit.
 * 0 means never exit.
 */
ecategory_t msg_category;	/* global used for communication from macro */
severity_t msg_severity;	/* global used for communication from macro */
static int msg_parse_preface[MSG_CATEGORIES];	/* really Boolean */

#ifdef WWW
void errMessage::
msg_init(severity_t level)
{
    short int i;

    // move init line to emsg(); 
    severity_level = level;
    initialized = true;
    for (i = MSG_CATEGORIES; i--;)
	msg_parse_preface[i] = strchr(msg_preface[i], '%') ? 1 : 0;
}

void errMessage::
msg_init(severity_t level, unsigned int n)
{
    short int i;

    msg_warnings = n;
    severity_level = level;
    initialized = true;
    for (i = MSG_CATEGORIES; i--;)
	msg_parse_preface[i] = strchr(msg_preface[i], '%') ? 1 : 0;
} 
#endif


void errMessage::
msg_printf(const char *fmt, ...)
{
    char    buff[1024];
    va_list args;
    register short int i, prelen, msglen;

    for (i = MSG_CATEGORIES; i--;)
        msg_parse_preface[i] = strchr(msg_preface[i], '%') ? 1 : 0;

    if (msg_parse_preface[msg_category]) {
        struct tm *tm;
	time_t     clock;
	char      *fmt;

        fmt = parse_preface(msg_preface[msg_category]);
        time(&clock);
        tm = localtime(&clock);
        prelen = strftime(buff, sizeof(buff), fmt, tm);
        if (prelen == 0)		/* Error condition from strftime */
            msg_fatal("Buffer too small");
        free(fmt);
    } else {
        strncpy(buff, msg_preface[msg_category], sizeof(buff));
	prelen = strlen(msg_preface[msg_category]);
    }

    va_start(args, fmt);
#if defined(_POSIX_SOURCE)
    msglen = vsprintf(buff + prelen, fmt, args);

    if (prelen + msglen > sizeof(buff))
        msg_fatal("Buffer too small");
#else
    /* BSD vsprintf() returns the first argument - can't be used
     * for detecting buffer over-runs.
     */
    (void) vsprintf(buff + prelen, fmt, args);
#endif

    va_end(args);
    fprintf(stderr, "%s\n", buff);
  if ( msg_category >= 2 && (errno != 0) )
  {
#ifdef __FreeBSD__
	fprintf(stderr, "\tPossible explanation: %s\n", strerror(errno) );
#else
	perror("\tPossible explaination: ");
	errno = 0;  // reset it explicitly here...
#endif
  }
}


char * errMessage::
parse_preface(const char *Template) 
{
    char                    *fmt;
    size_t                   fmt_size;
    ptrdiff_t                p = 0;
    enum { normal, percent } state = normal;

    fmt_size = strlen(Template) + 1;
    fmt = (char *) malloc(fmt_size);
    if (fmt == NULL)
        msg_fatal("Out of memory");

    for (; *Template; Template++) {
        if (p > fmt_size) {
            fmt_size += 16;
            if ((fmt = (char *)realloc(fmt, fmt_size)) == NULL)
                msg_fatal("Out of memory");
        }
        fmt[p++] = *Template;
        if (state == percent) {
            if (*Template == 's')
#ifdef __FreeBSD__
                p += sprintf(fmt + p - 2, "%d", msg_severity) - 2;
#else
                p = strchr(sprintf(fmt + p - 2, "%d", msg_severity), 0) - fmt;
#endif

            state = normal;
        } else if (*Template == '%')
            state = percent;
    }
    fmt[p] = 0;
    return fmt;
}

void errMessage::
msg_fatal(char *msg)
{
    fprintf(stderr, "Fatal error: %s\n", msg);
    exit(1);
}
 

#ifdef __E_TEST__
main()
{
  errMessage program;
  FILE *fp;

 printf("xgat[3] = (%d)\n", xcategory_t[3] );
 program.setProgname("", "etest");

 fprintf(stderr, "progname = [%s]\n", program.getProname());
 program.emsg(error, 3, ("Should print and exit"), _fl_ );
 if ((fp = fopen("foo", "r")) == NULL)
    program.emsg(warning, "fopen failed to open foo");
 program.emsg(error, "fopen failed to open foo", _fl_);
 program.emsg(nprint, "Testing, testing....", _fl_);
 program.emsg(debug, 5, "Testing2....", _fl_);
 //program.emsg(doabort, "Test of abort.", _fl_);
}
#endif   // __E_TEST__

/*  NEED one like below ... WITH *  severity_t severity;
 *  AND:: add the errno stuff from msg_printf().
 */

void errMessage::
emsg(ecategory_t category, severity_t severity, const char *fmt, ...)
{
    va_list args;
    char buff[8096];
    unsigned short msglen;

    doAbort = (category == doabort) ? true : false;
    if (category == nprint) return;
    do {
        if (severity >= msg_level[category]) {
            msg_category = category;
            msg_severity = severity;
            va_start(args, fmt);
#if defined(_POSIX_SOURCE)
            msglen = vsprintf(buff, fmt, args);

            if (msglen > sizeof(buff))
                msg_fatal("Buffer too small");
#else
    /* BSD vsprintf() returns the first argument - can't be used
     * for detecting buffer over-runs.
     */
            (void) vsprintf(buff, fmt, args);
#endif
            va_end(args);
            fprintf(stderr, "%s\n", buff);
            if (msg_warnings && --msg_warnings == 0) { 
                msgExit(category); 
            } 
	    if (doAbort) abort();
      }
    } while (0);
}




void errMessage::
emsg(ecategory_t category, const char *fmt, ...)
{
    va_list args;
    char buff[8096];
unsigned short msglen;

    //puts("Entering emsg(category, fmt, ...");
    doAbort = (category == doabort) ? true : false;
    //if (category == doabort) doAbort = true, errno = 0;
    if (category == nprint) return;
    do { 
	//puts("In do-while");
            msg_category = category; 
        va_start(args, fmt);
#if defined(_POSIX_SOURCE)
    msglen = vsprintf(buff, fmt, args);

    if (msglen > sizeof(buff))
        msg_fatal("Buffer too small");
#else
    /* BSD vsprintf() returns the first argument - can't be used
     * for detecting buffer over-runs.
     */
    (void) vsprintf(buff, fmt, args);
#endif
    va_end(args);
    fprintf(stderr, "%s\n", buff);


            if (msg_warnings && --msg_warnings == 0) { 
                msgExit(category); 
            } 
	    if (doAbort) abort();
    } while (0);
}

#ifdef WW
void errMessage::
emsg(ecategory_t category, severity_t severity, const char *args) 
{
    doAbort = (category == doabort) ? true : false;
    //if (category == doabort) doAbort = true, errno = 0;
    if (category == nprint) return;
    do { 
        if (severity >= msg_level[category]) { 
            msg_category = category; 
            msg_severity = severity; 
            msg_printf(args); 
            if (msg_warnings && --msg_warnings == 0) { 
                msgExit(category); 
            } 
	    if (doAbort) abort();
        } 
    } while (0);
}
#endif
void errMessage::
emsg(ecategory_t category, const char *args, char *file, int line)
{
    doAbort = (category == doabort) ? true : false;
    //if (category == doabort) doAbort = true, errno = 0;
    if (category == nprint) return;
    do { 
            msg_category = category; 
            msg_printf (args); 
	    if (line) printf("\tFilename: %s, line %d\n", file, line);
	    if (doAbort) abort();
	    printf("msg_warnings = (%d)\n", msg_warnings);
            if (msg_warnings && --msg_warnings == 0) { 
                msgExit(category); 
            } 
    } while (0);
}



void errMessage::
emsg(ecategory_t category, severity_t severity, const char *args, 
  char *file, int line)
{
    doAbort = (category == doabort) ? true : false;
    //if (category == doabort) doAbort = true, errno = 0;
    if (category == nprint) return;
    do {
        if (severity >= msg_level[category])
        { 
            msg_category = category;
            msg_severity = severity; 
            msg_printf (args);  
            if (line) printf("\tFilename: %s, line %d\n", file, line);
	    if (doAbort) abort();
            printf("msg_warnings = (%d)\n", msg_warnings);
            if (msg_warnings && --msg_warnings == 0) {
                msgExit(category);
            }
       }
    } while (0); 
}


void errMessage::
msgExit(ecategory_t category) 
{
  const char *max = "    The maximum number of error messages,";
  const char *reached = " has been reached, exiting";
  fprintf(stderr, "%s %d, %s\n", max, msg_warnings, reached);
  exit(category);
}
