/* colors.c - Mandelbrot Explorer (colormap setup and rotation)
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */ 

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/SmeBSB.h>
#include "mxp.h"

extern double drand48();

extern short depth;

/* externals from mandel.c */
extern void refresh();

/* externals from xmandelf.c */
extern Widget toplevel;
extern Widget color_label;
extern Widget file_popup;
extern Widget status_pu;
extern Pixmap marker;

/* externals from draw.c */
extern Display *draw_d;
extern Visual *draw_v;

/* externals from menu.c */
extern struct menud *scheme_menu;
extern struct menud *seq_menu;

void setup_cmap();
void rotate_cmap();
void setcolors();

XColor colorDefs[512];
Colormap cmap;
int colors;
int maxcolors;
int color_mode = 0;
int color_shift = 0;
int reserved = 0; 
int last_bias = 0;
int need_cmap_upd;
int bias;
int mode;
int red_shift;
int green_shift;
int blue_shift;

/*
 * create a private colormap
 */
void init_cmap(dp)
Display *dp;
{
	int i;
	long mask;

	draw_v = DefaultVisual(dp, DefaultScreen(dp));
	if (depth == 8) {
		maxcolors = maxcolors;
		cmap = XCreateColormap(dp, DefaultRootWindow(dp),
			    draw_v, AllocAll);
	} else {
		maxcolors = 512;
		mask = draw_v->red_mask;
		for (i=0; i<32 ; i++) {
			if (mask & 1) {
				break;
			}
			mask = mask >> 1;
		}
		red_shift = i;

		mask = draw_v->green_mask;
		for (i=0; i<32 ; i++) {
			if (mask & 1) {
				break;
			}
			mask = mask >> 1;
		}
		green_shift = i;

		mask = draw_v->blue_mask;
		for (i=0; i<32 ; i++) {
			if (mask & 1) {
				break;
			}
			mask = mask >> 1;
		}
		blue_shift = i;
	}
}

/*
 * Set color mode and shift
 */
void color_seq(w, val)
Widget w;
int val;
{
	Arg arg[1];

	/*
	 * Set/clear menu markers. A value of 8 is special cased and limits
	 * colors to a single sequence, regardless of iterations.
	 */
	XtSetArg(arg[0], XtNleftBitmap, None);
	switch (val) {
	case 0:
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
		XtSetValues(seq_menu->entry[8].button, arg, 1);
		XtSetValues(seq_menu->entry[9].button, arg, 1);
		XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
		mode = 1;
		color_shift = val;
		XtSetArg(arg[0], XtNleftBitmap, marker);
		XtSetValues(seq_menu->entry[val].button, arg, 1);
		break;
	case 8:
		mode = 2;
		XtSetValues(seq_menu->entry[9].button, arg, 1);
		XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
		XtSetArg(arg[0], XtNleftBitmap, marker);
		XtSetValues(seq_menu->entry[8].button, arg, 1);
		break;
	case 9:
		mode = 0;
		XtSetValues(seq_menu->entry[8].button, arg, 1);
		XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
		XtSetArg(arg[0], XtNleftBitmap, marker);
		XtSetValues(seq_menu->entry[9].button, arg, 1);
		break;
	}
	refresh();
}

/*
 * Set color scheme
 */
void color_scheme(w, val)
Widget w;
int val;
{
	Arg arg[1];

	XtSetArg(arg[0], XtNleftBitmap, None);
	XtSetValues(scheme_menu->entry[color_mode].button, arg, 1);
	XtSetArg(arg[0], XtNleftBitmap, marker);
	XtSetValues(scheme_menu->entry[val].button, arg, 1);
	color_mode = val;
	setup_cmap(draw_d, DefaultScreen(draw_d), colors);
	refresh();
} 

/*
 * Fill in color map
 */
void setup_cmap(disp, scr, count)
Display *disp;
int scr;
int count;
{
	int i, r=0, g=0, b=0, res, state;

	/*
	 * Fill in all unused color cells from the default colormap
	 */
	res = maxcolors - count;
		if (res > reserved) {
		for (i=0; i<res; i++) {
			colorDefs[i].pixel = i;
			XQueryColor(disp, DefaultColormap(disp, scr),
				&colorDefs[i]);
			colorDefs[i].flags = DoRed | DoGreen | DoBlue;
		}
	}
	reserved = res;

	/*
	 * Create various color schemes
	 */
	switch(color_mode) {
	case 0:
		count /= 9;
		r = count;
		b = 0;
		break;
	case 1:
		count /= 6;
		r = count;
		b = 0;
		break;
	case 2:
	case 3:
		count /= 3;
		r = count;
		b = 0;
		break;
	case 4:
		count /= 2;
		r = 0;
		b = count/2;
		break;
	case 5:
		count /= 2;
		r = count/2;
		b = count/2;
		break;
	case 6:
		count /= 4;
		b = count;
		r = 0;
		break;
	case 7:
	case 8:
		count /= 4;
		r = count;
		b = 0;
		break;
	}
	g = 0;
	state = 1;
	for (i=res; i<maxcolors; i++) {
		switch(color_mode) {
		case 0:
			/* Default */
			switch(state) {
			case 1:
				if (++g == count) {
					state++;
				}
				break;
			case 2:
				if (--r == 0) {
					state++;
				}
				break;
			case 3:
				if (++b == count) {
					state++;
				}
				break;
			case 4:
				if (--g == 0) {
					state++;
				}
				break;
			case 5:
				if (++r == count) {
					state++;
				}
				break;
			case 6:
				if (--b == 0) {
					state = 7;
				}
				break;
			case 7:
				if (b < count) {
					b++;
					g++;
				} else {
					state = 8;
				}
				break;
			case 8:
				r--;
				g--;
				b--;
				if (b <= 0) {
					state = 9;
				}
				break;
			case 9:
				if (++r == count) {
					state = 1;
				}
			}
			break;
		case 1:
			/* Bright colors */
			switch(state) {
			case 1:
				if (++g == count) {
					state++;
				}
				break;
			case 2:
				if (--r == 0) {
					state++;
				}
				break;
			case 3:
				if (++b == count) {
					state++;
				}
				break;
			case 4:
				if (--g == 0) {
					state++;
				}
				break;
			case 5:
				if (++r == count) {
					state++;
				}
				break;
			case 6:
				if (--b == 0) {
					state = 1;
				}
				break;
			}
			break;
		case 2:
			/* Dark colors */
			switch(state) {
			case 1:
				if (--r == 0) {
					state++;
				}
				g = count - r;
				break;
			case 2:
				if (--g == 0) {
					state++;
				}
				b = count - g;
				break;
			case 3:
				if (--b == 0) {
					state = 1;
				}
				r = count - b;
				break;
			}
			break;
		case 3:
			/* Sharp */
			switch(state) {
			case 1:
				if (++g == count) {
					r = 0;
					state++;
				}
				break;
			case 2:
				if (++b == count) {
					g = 0;
					state++;
				}
				break;
			case 3:
				if (++r == count) {
					b = 0;
					state = 1;
				}
				break;
			}
			break;
		case 4:
			/* Blue - White */
			switch(state) {
			case 1:
				if (b++ >= count) {
					b = count;
					state = 2;
				}
				break;
			case 2:
				if (r < count) {
					r++;
					g++;
				}
			}
			break;
		case 5:
			/* Purple - White */
			switch(state) {
			case 1:
				r++;
				if (b++ >= count) {
					b = count;
					r = count;
					state = 2;
				}
				break;
			case 2:
				if (g < count) {
					g++;
				}
				break;
			}
			break;
		case 6:
			/* Blue - Green */
			switch(state) {
			case 1:
				if (++g == count) {
					state++;
				}
				break;
			case 2:
				if (--b == 0) {
					state++;
				}
				break;
			case 3:
				if (++b == count) {
					state++;
				}
				break;
			case 4:
				if (--g == 0) {
					state = 1;
				}
				break;
			}
			break;
		case 7:
			/* Red - Blue */
			switch(state) {
			case 1:
				if (++b == count) {
					state++;
				}
				break;
			case 2:
				if (--r == 0) {
					state++;
				}
				break;
			case 3:
				if (++r == count) {
					state++;
				}
				break;
			case 4:
				if (--b == 0) {
					state = 1;
				}
				break;
			}
			break;
		case 8:
			/* Red - Green */
			switch(state) {
			case 1:
				if (++g == count) {
					state++;
				}
				break;
			case 2:
				if (--r == 0) {
					state++;
				}
				break;
			case 3:
				if (++r == count) {
					state++;
				}
				break;
			case 4:
				if (--g == 0) {
					state = 1;
				}
				break;
			}
			break;
		case 9:
			/* Gray scale */
			r = i;
			g = r;
			b = r;
			break;
		case 10:
			/* Gray scale reverse */
			r = count - i;
			g = r;
			b = r;
			break;
		}
		colorDefs[i].pixel = i;
		colorDefs[i].flags = DoRed | DoGreen | DoBlue;
		switch (depth) {
		case 8:
			colorDefs[i].red = r * 0xffff / count;
			colorDefs[i].green = g * 0xffff / count;
			colorDefs[i].blue = b * 0xffff / count;
			break;
		case 16:
			colorDefs[i].red = r * 0x1f / count;
			colorDefs[i].green = g * 0x3f / count;
			colorDefs[i].blue = b * 0x1f / count;
			break;
		case 24:
			colorDefs[i].red = r * 0xff / count;
			colorDefs[i].green = g * 0xff / count;
			colorDefs[i].blue = b * 0xff / count;
		}
	}
	if (depth == 8) {
		XStoreColors(disp, cmap, colorDefs, maxcolors);
	}
	rotate_cmap(disp, 0);
}

/*
 * Rotate colormap 
 */
void rotate_cmap(disp, flag)
Display *disp;
{
	int i, n, idx;
	int r[maxcolors], g[maxcolors], b[maxcolors];

	if (last_bias == bias) {
		return;
	}

	n = bias - last_bias;
	last_bias = bias;

	if (n >= colors) {
		n = n % colors;
	}

	for (i=reserved; i<maxcolors; i++) {
		r[i] = colorDefs[i].red;
		g[i] = colorDefs[i].green;
		b[i] = colorDefs[i].blue;
	}
	for (i=reserved; i<maxcolors; i++) {
		idx = i + n;
		if (idx >= maxcolors) {
			idx = n + reserved + i - maxcolors;
		} else if (idx < reserved) {
			idx = n + colors + i;
		} 
		colorDefs[idx].red = r[i];
		colorDefs[idx].green = g[i];
		colorDefs[idx].blue = b[i];
	}
	if (depth == 8) {
		XStoreColors(disp, cmap, colorDefs, maxcolors);
	} else {
		if (flag) {
			refresh();
		}
	}
}

/*
 * Increment number of colors
 */
void color_inc()
{
	colors++;
	if (colors > maxcolors-2) {
		colors = maxcolors-2;
	}
	setcolors();
	need_cmap_upd++;
}

/*
 * Decrement number of colors
 */
void color_dec()
{
	colors--;
	if (colors < 6) {
		colors = 6;
	}
	need_cmap_upd++;
	setcolors();
}

/*
 * Increment color bias
 */
void bias_inc()
{
        bias++;
	rotate_cmap(draw_d, 1);
}

/*
 * Decrement color bias
 */
void bias_dec()
{
        bias--;
        if (bias < 0) {
                bias = maxcolors-1;
        }
	rotate_cmap(draw_d, 1);
}

/*
 * Set number of colors label
 */
void setcolors()
{
	Arg arg;
        char string[20];

        sprintf(string, "%3d", colors);
        arg.name = XtNlabel;
        arg.value = (XtArgVal) string;
        XtSetValues(color_label, &arg, 1);
}
