/* file.c - Mandelbrot Explorer (file I/O, save/load setup, gif's)
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */ 

#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Dialog.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include "lug.h"
#include "lugfnts.h"
#include "mxp.h"

extern char *getenv();

/* externals from colors.c */
extern int colors;
extern int color_mode;
extern int color_shift;
extern int bias;
extern int mode;
extern int reserved;
extern XColor colorDefs[];

/* externals from mandel.c */
extern int iter;
extern double mag;
extern int *dpall;
extern struct zoomd *zoomp;
extern void resetmandel();
extern void refresh();
extern void redo();
extern void event_proc();

/* externals from draw.c */
extern void resize();
extern int aspect_opt;
extern Widget drawform;

/* externals from xmandelf.c */
extern short depth;
extern void create_draw();
extern Dimension wwidth;
extern Dimension wheight;
extern Widget toplevel;
extern Widget file_popup;
extern Widget file_text;
extern Widget error_pu;
extern Widget errorm;

void filename_cb();
void pop_down();
void error_msg();

struct setup set, init_set;
char fname[MAX_NAME];
char fname1[MAX_NAME];
void (*cfunct)();
void read_rc_file();

/*
 * Write current settings to a file
 */
void write_setup(fn, flag)
char *fn;
int flag;
{
	int fd;
	char text[MAX_NAME];

	if (flag == 1) {
		strcpy(fname1, fn);
	}
	if ((fd = open(fn, O_WRONLY|O_CREAT, 0644)) < 0) {
		sprintf(text, "Error: Can't open %s ", fn);
		error_msg(toplevel, text);
		return;
	}
	set.mode = mode;
	set.color_mode = color_mode;
	set.color_shift = color_shift;
	set.colors = colors;
	set.bias = bias;
	set.iter = iter;
	set.win_wid = wwidth;
	set.win_hgt = wheight;
	set.aspect_opt = aspect_opt;
	if (zoomp) {
		set.upper_x = zoomp->ux;
		set.lower_x = zoomp->lx;
		set.upper_y = zoomp->uy;
		set.lower_y = zoomp->ly;
	}
	set.mag = mag;

	if (write(fd, &set, sizeof set) != sizeof set) {
		sprintf(text, "Error: Can't write %s ", fn);
		error_msg(toplevel, text);
	}
	if (flag == 0) {
		/*
	         * Save  a copy of the setup info 
		 */
	        memcpy(&init_set, &set, sizeof(struct setup)); 
	}
	close(fd);
}

/*
 * Read settings from the specified file
 */
void read_setup(fn, setup)
char *fn;
int setup;
{
	int fd;
	char text[MAX_NAME];

	strcpy(fname1, fn);
	if ((fd = open(fn, O_RDONLY)) < 0) {
		sprintf(text, "Error: Can't open %s ", fn);
		error_msg(toplevel, text);
		return;
	}
	if (read(fd, &set, sizeof set) != sizeof set) {
		close(fd);
		sprintf(text, "Error: Can't read %s ", fn);
		error_msg(toplevel, text);
		return;
	}
	close(fd);
	if (setup) {
		if (wwidth != set.win_wid || wheight != set.win_hgt) {
			resize(set.win_wid, set.win_hgt, -1);
		}
		resetmandel(0);
	}
	return;
}

/*
 * Get a filename via a popup
 * The routine to be called after the filename has been entered is
 * passed as "fun".
 */
void get_filename(w, name, fun)
Widget w;
char *name;
void (*fun)();
{
	Arg arg[2];
        Position x, y;
        Dimension height;

	cfunct = fun;

	strcpy(fname, name);
	strcat(fname, "\n");
	XtSetArg(arg[0], XtNstring, fname);
        XtSetValues(file_text, arg, 1);

        XtSetArg (arg[1], XtNheight, &height);
        XtGetValues(w, arg, 2);

        XtTranslateCoords (w, 0, height, &x, &y);

        XtSetArg (arg[0], XtNx, x);
        XtSetArg (arg[1], XtNy, y);
        XtSetValues(file_popup, arg, 2);
        XtPopup(file_popup, XtGrabExclusive);
} 

/*
 * Display an error message via a popup
 */
void error_msg(w, name)
Widget w;
char *name;
{
	Arg arg[2];
        Position x, y;

        XtSetArg (arg[0], XtNlabel, name);
        XtSetValues(errorm, arg, 1);

        XtTranslateCoords (w, 0, 0, &x, &y);

        XtSetArg (arg[0], XtNx, x);
        XtSetArg (arg[1], XtNy, y);
        XtSetValues(error_pu, arg, 2);
        XtPopup(error_pu, XtGrabExclusive);
} 

/*
 * The get filename callback
 */
void filename_cb(w, a)
Widget w;
caddr_t a;
{
	int i;
	char *st = NULL;

	XtPopdown(file_popup);

	/*
	 * Process the entered filename, strip leading spaces and newlines
	 */
	for (i=0; i<MAX_NAME; i++) {
		if (!isspace(fname[i]) && fname[i] != '\n') {
			st = &fname[i];
			break;
		}
	}
	for (; i<MAX_NAME; i++) {
		if (fname[i] == '\n') {
			fname[i] = '\0';
		}
	}
	(*cfunct)(st, 1);
}

void pop_down(w, wgt)
Widget w;
Widget wgt;
{
	XtPopdown(wgt);
}

/*
 * Create a gif of the current mandelbrot image
 */
void create_gif(fn)
char *fn;
{
	int i;
        bitmap_hdr hdr;
	color_map *cmap;

        allocatebitmap(&hdr, wwidth, wheight, 8, 0);

	for (i=0; i<wwidth*wheight; i++) {
                if (dpall[i] == iter) {
                        hdr.r[i] = 0;
                } else {
                        switch (mode) {
                        case 0:
                                hdr.r[i] = ITER_2_COL_0(dpall[i])+1-reserved;
                                break;

                        case 1:
                                hdr.r[i] = ITER_2_COL_1(dpall[i])+1-reserved;
                                break;
                        }
                }
	}
	cmap = (color_map *)hdr.cmap;
	cmap[0][0] = 0;
	cmap[0][1] = 0;
	cmap[0][2] = 0;
	for (i=0; i<colors; i++) {
		cmap[i+1][0] = colorDefs[i+reserved].red;
		cmap[i+1][1] = colorDefs[i+reserved].green;
		cmap[i+1][2] = colorDefs[i+reserved].blue;
	}
	write_gif_file(fn, &hdr); 
	freebitmap(&hdr);
}

/*
 * Dump mandelbrot image in a format that can be read directly by xplot
 * Great for doing 3d renderings.
 */
void dump_data()
{
	int ix, iy;
	double val, scale, log();
	FILE *fd, *fopen();

	scale = (double)iter / (double)wwidth;
	fd = fopen("mxp.data", "w");
	for (ix = 0; ix < wwidth; ix++) {
		for (iy = 0; iy < wheight; iy++) {
			val = (double)dpall[ix+(iy*wwidth)];
			fprintf(fd, "%f %d %d\n", log(val/scale)*12.0, ix, iy);
		}
		fprintf(fd, "\n");
	}
	fclose(fd);
}

void save_setup()
{
	/*
	 * Write current settings to rc file
	 */
	sprintf(fname, "%s/%s", getenv("HOME"), RC_FILE);
	write_setup(fname, 0);
}

void save_setup_as(w)
Widget w;
{
	if (fname1[0] == 0) {
		get_filename(w, "mxp1.rc", write_setup);
	} else {
		get_filename(w, fname1, write_setup);
	}
}

/*
 * Clear defaults by removing rc file
 */
void clear_defaults()
{
	char text[MAX_NAME];

	sprintf(text, "%s/%s", getenv("HOME"), RC_FILE);
	unlink(text);
	read_rc_file();
	if (wwidth != set.win_wid || wheight != set.win_hgt) {
		resize(set.win_wid, set.win_hgt, -1);
	}
	resetmandel(0);
}

/*
 * Read current settings from rc file
 */
void read_rc_file()
{
	int fd;
	char text[MAX_NAME];

	sprintf(text, "%s/%s", getenv("HOME"), RC_FILE);
	if ((fd = open(text, O_RDONLY)) >= 0) {
		if (read(fd, &set, sizeof set) == sizeof set) {
			close(fd);
		}
		close(fd);
	} else {
		/*
		 * No rc file, plug in defaults
		 */
		set.mode = 1;
		set.color_mode = 0;
		set.color_shift = 0;
		set.colors = 100;
		set.bias = 59;
		set.iter = 800;
		set.win_wid = 512;
		set.win_hgt = 384;
		set.aspect_opt = 0;
		set.upper_x = 1.25; 
		set.lower_x = -2.75;
		set.upper_y = 1.5;
		set.lower_y = -1.5;
		set.mag = 1.0;
		set.aspect = 1.333333;
	}
	color_mode = set.color_mode;
	color_shift = set.color_shift;
	mode = set.mode;
	aspect_opt = set.aspect_opt;
	colors = set.colors;
	bias = set.bias;
	iter = set.iter;

	/*
         * Save  a copy of the setup info 
         */
        memcpy(&init_set, &set, sizeof(struct setup)); 
}

void load_setup(w)
Widget w;
{
	if (fname1[0] == 0) {
		get_filename(w, "mxp1.rc", read_setup);
	} else {
		get_filename(w, fname1, read_setup);
	}
}

void write_image(w)
Widget w;
{
	get_filename(w, "mandel.gif", create_gif);
}
