/* mandel.c - Mandelbrot Explorer (mandelbrot generation, event handler)
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */ 

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/SmeBSB.h>
#include <stdio.h>
#include <math.h>
#include <sys/time.h>
#include "mxp.h"
#ifdef MPI
#include <mpi.h>
#endif

extern void free();

/* Externals from menu.c */
extern struct menud *scheme_menu;
extern struct menud *seq_menu;

/* Externals from color.c */
extern XColor colorDefs[];
extern int colors;
extern int color_mode;
extern int color_shift;
extern int mode;
extern int bias;
extern int need_cmap_upd;
extern int reserved;
extern void setcolors();
extern void setup_cmap();
extern int red_shift;
extern int green_shift;
extern int blue_shift;
extern int last_bias;

/* Externals from draw.c */
extern Widget drawform;
extern Window draw_win;
extern GC draw_gc;
extern Screen *draw_Screen;
extern Display *draw_d;
extern Visual *draw_v;
extern XImage *xip;
extern XImage *xipa;
extern char *dp;
extern int *dpall;
extern short *dpref;
extern void resize();
extern short bypp;

/* Externals from xmandelf.c */
extern XtAppContext ctx;
extern Widget iter_label, status_pu;
extern Widget toplevel;
extern Widget st_time, st_pct, st_pixsec, st_pixels, st_mflops;
extern Widget st_ux, st_uy, st_lx, st_ly, st_cx, st_cy, st_mag, st_size;
extern Dimension wwidth;
extern Dimension wheight;
extern Cursor crosshair_xcr;
extern Cursor watch_xcr; 
extern Cursor hand_xcr; 
extern Pixmap marker; 
extern Pixel black_pix; 
extern short depth;
extern int stats_flag;

/* Externals from file.c */
extern struct setup set, init_set;

void refresh();
void setiter();
void mandel();
void generate();
void upd_stats();
int color_xlate();
char *funits();

#ifdef MPI
extern int mpi_rank;
extern int mpi_size;
extern int lines;
extern struct wtab *workt;
extern int msg_len;
extern int *msg_buf;
int done;
int segs;
int widx;
struct wtab *wt;
struct mdef md;
void do_block(struct mdef mmd);
Boolean mpi_work();
void mpi_start();
#else
Boolean do_line();
#endif

char suff[4] = { 'K', 'M', 'G', 'T'};
struct zoomd *zoomp = NULL, *zoompn;
int iter;
int iter_change;
int drawing;
int need_refresh;
int initialized=0;
int display;
int diy;
double flops;
int work_id;
int esec;
double dy, dx, incy, incx;
struct timeval tps, tpe;
double cmag;
double mag;
Arg arg[1];
#ifdef MPI
MPI_Status status;
#endif

/*
 * Reset all values to initial except for image size
 */
void resetmandel(flag)
int flag;
{
	double adj, asp;

	if (zoomp) {        /* reset to beginning by popping zps off stack */
		while ((zoompn = zoomp->zp)) {
			free(zoomp);
			zoomp = zoompn;
		}
	} else {                    /* This is first time */
		zoomp = (struct zoomd *) malloc(sizeof (struct zoomd));
		zoomp->zp = NULL;    /* NULL means last in stack - don't pop */
	}

	/*
	 * If flag is set get the initial settings
	 */
	if (flag) {
		memcpy(&set, &init_set, sizeof(struct setup));
	}

	/*
	 * Setup co-ordinates and adjust for aspect ratio
	 */
	zoomp->lx = set.lower_x; 
	zoomp->ux = set.upper_x;
	zoomp->uy = set.upper_y;
	zoomp->ly = set.lower_y;
	asp = (double)wwidth / (double)wheight;
	adj = (((asp / set.aspect) - 1.0) * (zoomp->ux - zoomp->lx)) / 2.0;
	zoomp->ux += adj;
	zoomp->lx -= adj;
	zoomp->aspect = asp;
	set.aspect = asp;

	/*
	 * Set all menu markers to reflect the initial values
	 */
	mode = set.mode;
	XtSetArg(arg[0], XtNleftBitmap, None);
        if (mode == 0) {
                XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
                XtSetArg(arg[0], XtNleftBitmap, marker);
                XtSetValues(seq_menu->entry[8].button, arg, 1);
        } else {
                XtSetValues(seq_menu->entry[8].button, arg, 1);
                XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
		color_shift = set.color_shift;
                XtSetArg(arg[0], XtNleftBitmap, marker);
                XtSetValues(seq_menu->entry[color_shift].button, arg, 1);
        }
        XtSetArg(arg[0], XtNleftBitmap, None);
        XtSetValues(scheme_menu->entry[color_mode].button, arg, 1);
	color_mode = set.color_mode;
        XtSetArg(arg[0], XtNleftBitmap, marker);
        XtSetValues(scheme_menu->entry[color_mode].button, arg, 1);

	colors = set.colors;
	bias = set.bias;
	iter = set.iter;
	mag = set.mag;
	cmag = 0.0;
	need_cmap_upd = 1;
	last_bias = 0;
	setiter();
	setcolors();
	/*
	 * Now draw the picture
	 */
	mandel(M_CLEAR);
}

/*
 * Event handler for drawform events
 */
void event_proc(w, call_data, evnt)
Widget w;
caddr_t call_data;
XEvent *evnt;
{
	static int width, height;
	static int winx0,winy0,winx1,winy1;
	static int x0, y0, x1, y1;
	double tw, th, f;
	static int rubberband = 1;
	char text[40];

	switch(evnt->type) {
	case Expose:
		/*
		 * Only refresh screen if this is the last of a multi-part
		 * expose. Also defer refreshes until after the image has
		 * been drawn.
		 */
		if (evnt->xexpose.count == 0) {
			if (drawing == 0) {
				refresh();
			} else {
				need_refresh = 1;
			}
		}

		break;

	case ConfigureNotify:
		/*
		 * Resize triggered by stretching with the mouse
		 */
		if ((evnt->xconfigure.width == wwidth &&
			evnt->xconfigure.height == wheight) ||
			initialized == 0) {
			break;
		}
		resize(evnt->xconfigure.width, evnt->xconfigure.height, -1);
		mandel(M_CLEAR);
		break;

	case ButtonPress:
		if (evnt->xbutton.button == Button1) {
			/*
			 * Set the starting co-ordinates for a zoom
			 */
			x0 = winx0 = evnt->xbutton.x;
			y0 = winy0 = evnt->xbutton.y;
			rubberband = 0;
			width = height = 0;

			/*
			 * Set values in the status window
			 */
			XtSetArg(arg[0], XtNlabel, text);
			f = zoomp->lx + (((double)x0 / (double)wwidth) *
				(zoomp->ux - zoomp->lx));
			sprintf(text, "UX   %- 9g", f);
			XtSetValues(st_ux, arg, 1);
			f = zoomp->uy - (((double)y0 / (double)wheight) *
				(zoomp->uy - zoomp->ly));
			sprintf(text, "UY   %- 9g", f);
			XtSetValues(st_uy, arg, 1);
			sprintf(text, "X");
			XtSetValues(st_cx, arg, 1);
			sprintf(text, "Y");
			XtSetValues(st_cy, arg, 1);
		} else {
			/*
			 * If this was not button 1 then make sure that
			 * the command and status windows are on top.
			 */
			XRaiseWindow(draw_d, XtWindow(toplevel));
			if (stats_flag) {
				XRaiseWindow(draw_d, XtWindow(status_pu));
			}
		}
		break;

	case ButtonRelease:
		/*
		 * Complete the zoom when button 1 is released
		 */
		if (evnt->xbutton.button != Button1) {
			break;
		}
		winx1 = evnt->xbutton.x;
		winy1 = evnt->xbutton.y;
		if ((height = winy1 - winy0) < 0)
			height = - height;
		width = (int)(((float)wwidth / (float)wheight) * (float)height);
		x0 = (winx1 > winx0) ? winx0 : winx0 - width;
		y0 = (winy1 > winy0) ? winy0 : winy0 - height;
		x1 = x0 + width;
		y1 = y0 + height;
		rubberband = 1;
		XSetFunction(draw_d, draw_gc, GXcopy);
		tw = zoomp->ux - zoomp->lx;
		th = zoomp->uy - zoomp->ly;
		zoompn = (struct zoomd *) malloc(sizeof (struct zoomd));
		zoompn->zp = zoomp;
		zoompn->ux = zoomp->lx + ((double)x1/(double)wwidth) * tw;
		zoompn->lx = zoomp->lx + ((double)x0/(double)wwidth) * tw;
		zoompn->ly = zoomp->uy - ((double)y1/(double)wheight) * th;
		zoompn->uy = zoomp->uy - ((double)y0/(double)wheight) * th;
		zoompn->mag = (zoomp->uy - zoomp->ly) /
		    (zoompn->uy - zoompn->ly);
		zoompn->aspect = set.aspect;
		zoomp = zoompn;
		mag *= zoomp->mag;
		cmag = zoomp->mag;
		mandel(M_CLEAR);
		break;

	case MotionNotify:
		if (evnt->xmotion.state == Button1Mask) {
			/*
			 * While button 1 is down draw a rectangle to
			 * identify the zoom area
			 */
			if (rubberband) break;
			XSetForeground(draw_d, draw_gc, 0x55);
			XSetFunction(draw_d, draw_gc, GXxor);
			XDrawRectangle(draw_d,draw_win, draw_gc, x0, y0,
				width, height);
			winx1 = evnt->xbutton.x;
			winy1 = evnt->xbutton.y;
			if ((height = winy1 - winy0) < 0)
				height = - height;
			width = (int)(((float)wwidth / (float)wheight) *
				(float)height);
			x0 = (winx1 > winx0) ? winx0 : winx0 - width;
			y0 = (winy1 > winy0) ? winy0 : winy0 - height;
			XDrawRectangle(draw_d,draw_win, draw_gc, x0, y0,
				width, height);
	
			if (!stats_flag) {
				break;
			}
			/*
			 * Update values in the status window
			 */
			tw = zoomp->ux - zoomp->lx;
			XtSetArg(arg[0], XtNlabel, text);
			f = zoomp->lx + (((double)(x0 + width) /
				(double)wwidth) * tw);
			sprintf(text, "LX   %- 9g", f);
			XtSetValues(st_lx, arg, 1);
			f = zoomp->uy - (((double)(y0 + height) /
				(double)wheight) * tw);
			sprintf(text, "LY   %- 9g", f);
			XtSetValues(st_ly, arg, 1);
			if (x0 != winx0) {
				f = zoomp->lx + (((double)x0 /
					(double)wwidth) * tw);
				sprintf(text, "UX   %- 9g", f);
				XtSetValues(st_ux, arg, 1);
				f = zoomp->uy - (((double)y0 /
					(double)wheight) * tw);
				sprintf(text, "UY   %- 9g", f);
				XtSetValues(st_uy, arg, 1);
			}
		} else {
			if (!stats_flag || drawing) {
				break;
			}
			/*
			 * If not button is pressed show the co-ordinates
			 * for the pointer position.
			 */
			XtSetArg(arg[0], XtNlabel, text);
			f = zoomp->lx + (((double)evnt->xbutton.x /
				(double)wwidth) * (zoomp->ux - zoomp->lx));
			sprintf(text, "X    %- 9g", f);
			XtSetValues(st_cx, arg, 1);
			f = zoomp->uy - (((double)evnt->xbutton.y /
				(double)wheight) * (zoomp->uy - zoomp->ly));
			sprintf(text, "Y    %- 9g", f);
			XtSetValues(st_cy, arg, 1);
		}
		break;

	case LeaveNotify:
		if (!stats_flag) {
			break;
		}
		/*
		 * Blank the X and Y co-ordinates when the pointer leaves
		 * the window.
		 */
		XtSetArg(arg[0], XtNlabel, text);
		sprintf(text, "X");
		XtSetValues(st_cx, arg, 1);
		sprintf(text, "Y");
		XtSetValues(st_cy, arg, 1);
		break;

	default:
		break;
	}
}

/*
 * Unzoom to the previous zoom co-ordinates
 */
void mooz()
{
	double adj;

	if (zoomp == NULL) { 
		XBell(draw_d, 0); 
		return; 
	}
	if (zoomp->zp == NULL) { 
		XBell(draw_d, 0); 
		return; 
	}
	mag /= zoomp->mag;
	cmag = zoomp->mag * -1;
	if ((zoompn = zoomp->zp)) {   /* last on stack */
		free(zoomp);
		zoomp = zoompn;
	}
	adj = (((set.aspect / zoomp->aspect) - 1.0) * (zoomp->ux - zoomp->lx)) / 2.0;
	zoomp->ux += adj;
	zoomp->lx -= adj;
	mandel(M_CLEAR);
}

/*
 * Redraw the picture by either re-generating or refreshing from an
 * internal copy with possible colormap changes.
 */
void redo()
{
	if (iter_change) {
		mandel(M_NONE);
		iter_change = 0;
		return;
	}
	if (need_cmap_upd) {
		setup_cmap(draw_d, DefaultScreen(draw_d), colors);
		need_cmap_upd = 0;
	}
	refresh();
}

void setiter()
{
	char text[20];

	sprintf(text, " %6d", iter);
	XtSetArg(arg[0], XtNlabel, text);
	XtSetValues(iter_label, arg, 1);
}

void increment()
{
	iter += (int)((float)iter * 0.05);
	if (iter > 999999) {
		iter = 999999;
	}
	iter_change++;
	setiter();
}

void decrement()
{
	iter -= (int)((float)iter * 0.05);
	if (iter < 20) {
		iter = 20;
	}
	iter_change++;
	setiter();
}

/*
 * Redraw picture from an internal copy of the generated data
 */
void refresh()
{
	int i, j, k;

	if (initialized == 0) {
		resetmandel(0);
		initialized = 1;
		return;
	}
       for (j=0, k=0; k<wheight; k++) {
               for (j=k*wwidth*bypp, i=0; i<wwidth; i++) {
                       j = color_xlate(dpall[i+k*wwidth], dpref, j);
		}
	}
	if (display) {
		XPutImage(draw_d,draw_win,draw_gc,xipa,0,0,0,0, wwidth,wheight);
	}
}
	

/*
 * Generate mandelbrot picture
 */
void mandel(opts)
int opts;
{
	int j;
	double x, y;
	double ix, iy;
	char text[40];

	if (opts&M_CLEAR) {
		 XClearWindow(draw_d, draw_win);
	}
	if (opts&M_NODISP) {
		display = 0;
	} else {
		display = 1;
	}
	if (need_cmap_upd) {
		setup_cmap(draw_d, DefaultScreen(draw_d), colors);
		need_cmap_upd = 0;
	}

	ix = (zoomp->ux - zoomp->lx)/(double)wwidth;
        iy = (zoomp->uy - zoomp->ly)/(double)wheight;
        y = zoomp->uy - iy;
        x = zoomp->lx + ix;
	dy = y;
	dx = x;
	incy = iy;
	incx = ix;
	diy = 0;
	flops = 0;
	esec = 0;
	drawing = 1;
	need_refresh = 0;

	/*
	 * Set values in the status window
	 */
	XtSetArg(arg[0], XtNlabel, text);
	sprintf(text,"Size     %4dx%d", wwidth, wheight);
	XtSetValues(st_size, arg, 1);
	sprintf(text,"Zoom       %6s", funits(mag, 6, 2));
	XtSetValues(st_mag, arg, 1);
	sprintf(text, "Pixels %10d", wheight * wwidth);
	XtSetValues(st_pixels, arg, 1);
	sprintf(text, "UX   %- 9g", zoomp->ux);
	XtSetValues(st_ux, arg, 1);
	sprintf(text, "UY   %- 9g", zoomp->uy);
	XtSetValues(st_uy, arg, 1);
	sprintf(text, "LX   %- 9g", zoomp->lx);
	XtSetValues(st_lx, arg, 1);
	sprintf(text, "LY   %- 9g", zoomp->ly);
	XtSetValues(st_ly, arg, 1);
	sprintf(text,"M-Flops        ");
	XtSetArg(arg[0], XtNlabel, text);
	XtSetValues(st_mflops, arg, 1);

	gettimeofday(&tps, NULL);
#ifdef MPI
	mpi_start();
#endif

	/*
	 * We setup a work_proc to do one line at a
	 * time. This keeps all of the buttons active and allows for
	 * periodic updates of status values
	 */
	if (work_id != 0) {
		XtRemoveWorkProc(work_id);
	}

	/*
	 * The drawing may be done either syncronous or async
	 */
#ifdef MPI
	if (opts&M_SYNC) {
		while(mpi_work() == FALSE);
	} else {
		work_id = XtAppAddWorkProc(ctx, mpi_work, 0);
	}
#else
	if (opts&M_SYNC) {
		while(do_line() == FALSE);
	} else {
		work_id = XtAppAddWorkProc(ctx, do_line, 0);
	}
#endif
}


#ifdef MPI
void do_block(struct mdef mmd)
{
        register int i;
        int ix, bidx, line;
        complex z, z0;
        register double temp, temp2, temp3;
        char text[40];

	for (bidx=0, line=mmd.sline; line<mmd.sline+mmd.nline; line++) {
	        z0.im = mmd.disty - (double)line * mmd.incy;
		for (ix = 0; ix < mmd.wwidth; ix++) {
			/*
			 * Determine value for a pixel
			 */
			z0.re = mmd.distx + (double)ix * mmd.incx;
			temp2 = 0;
			temp3 = 0;
			z.re = 0;
			z.im = 0;
			for (i=0; (i<mmd.iter) && (temp2+temp3 < 4.0); ++i){
				temp = z.re;
				temp2 = temp * temp;
				temp3 = z.im * z.im;
				z.re = temp2 - temp3 + z0.re;
				z.im = 2 * z.im * temp + z0.im;
			}
			msg_buf[bidx++] = i;
		}
        }
}
#else
/*
 * Create one line of the picture and update status values when done
 */
Boolean do_line()
{
	register int i, j;
	int ix, idx;
	complex z, z0;
	double temp, temp2, temp3;
	double ttime;
	char text[40];

	z0.im = dy - (double)diy * incy;
	idx = diy * wwidth;
	for (j=0, ix = 0; ix < wwidth; ix++) {
		/*
		 * Determine value for a pixel
		 */
		z0.re = dx + (double)ix * incx;
		temp2 = 0;
		temp3 = 0;
		z.re = 0;
		z.im = 0;
		for (i=0; (i<iter) && (temp2+temp3 < 4.0); ++i){
			temp = z.re;
			temp2 = temp * temp;
			temp3 = z.im * z.im;
			z.re = temp2 - temp3 + z0.re;
			z.im = 2 * z.im * temp + z0.im;
		}
		if (display) {
			j = color_xlate(i, dp, j);
		}
		dpall[idx + ix] = i;
		flops += (double)i;
	}
	if (display) {
		XPutImage(draw_d,draw_win,draw_gc,xip,0,0,0,diy,wwidth,1);
	}

	/*
	 * Update stats on every 12th line
	 */
	if (stats_flag && (diy % 12) == 0) {
		upd_stats();
	}

	/*
	 * Are we done ?
	 */
	if (++diy < wheight) {
		return(FALSE);
	}

	/*
	 * All done, update status
	 */
	upd_stats();
	flops *= 9.0;
	flops += (double)(wwidth * wheight * 3);
	ttime = (double)(tpe.tv_sec - tps.tv_sec) * 1000000.0;
	ttime += (double)(tpe.tv_usec - tps.tv_usec);
	flops = flops / ttime;
	sprintf(text,"M-Flops %9.2f", flops);
	XtSetArg(arg[0], XtNlabel, text);
	XtSetValues(st_mflops, arg, 1);

	work_id = 0;
	drawing = 0;
	/*
	 * If we got an expose event while drawing do the refresh now
	 */
	if (need_refresh) {
		refresh();
	}
	return(TRUE);
}
#endif

int color_xlate(int val, char *p, int idx) {

	int k=0, color;

	if (val == iter) {
		switch(depth) {
		case 32:
			p[idx++] = 0;
		case 24:
			p[idx++] = 0;
		case 16:
			p[idx++] = 0;
			p[idx++] = 0;
			break;
		case 8:
			p[idx++] = black_pix;
		}
	} else {
		switch (mode) {
		case 0:
			k = ITER_2_COL_0(val);
			break;

		case 1:
			k = ITER_2_COL_1(val);
			break;
		case 2:
			val = sqrt(val);
			k = (val % colors) + reserved;
			break;
		}
		if (depth == 8) {
			p[idx++] = k;
			return(idx);
		}

		color = ((colorDefs[k].blue << blue_shift) &
				draw_v->blue_mask) |
			((colorDefs[k].green << green_shift) &
				draw_v->green_mask) |
			((colorDefs[k].red << red_shift) &
				draw_v->red_mask);
		switch(depth) {
		case 32:
			p[idx++] = (color & 0xff000000) >> 24; 
		case 24:
			p[idx++] = (color & 0xff0000) >> 16; 
		case 16:
			p[idx++] = (color & 0xff00) >> 8; 
			p[idx++] = color & 0xff;
		}
	}
	return(idx);
}

void upd_stats()
{
	int min, sec;
	double usec;
	char text[40];

	XtSetArg(arg[0], XtNlabel, text);
	gettimeofday(&tpe, NULL);
	usec = (double)(tpe.tv_sec - tps.tv_sec) * 1000000.0;
	usec += (double)(tpe.tv_usec - tps.tv_usec);
	sprintf(text, "Pix/Sec %9.0f", (double)(diy * wwidth)*1000000.0/usec);
	XtSetValues(st_pixsec, arg, 1);
	min = (int)(usec / 60000000.0);
	usec -= (double)min * 60000000.0;
	sec = (int)(usec / 1000000.0);
	usec -= ((double)sec * 1000000.0);
	sprintf(text, "Time %4d:%.2d.%.4d", min, sec, (int)usec/100);
	XtSetValues(st_time, arg, 1);
#ifdef MPI
	sprintf(text, "Complete    %4d%%", (done * 100) / (segs-1));
#else
	sprintf(text, "Complete    %4d%%", (diy * 100) / wheight);
#endif
	XtSetValues(st_pct, arg, 1);
}

/*
 * Display a floating point value with units, highly readable
 */
char *funits(val, size, prec)
float val;
int size;
int prec;
{
	register int i, neg = 1.0;
	register float n, k, a;
	static char tval[16];
	double pow();

	/*
	 * If negative, we need to reserve a digit for the "-", make val 
	 * positive and remember that the number was negative.
	 */
	if (val < 0) {
		size--;
		val *= -1.0;
		neg = -1.0;
	}
	/*
	 * If less than 1.0 then just print with max or specified precision
	 */
	if (val < 1.0) {
		if (prec) {
			if (prec > size-2) {
				prec = size -2;
			}
			sprintf(tval, "%*.*f", size, prec, val*neg);
		} else {
			sprintf(tval, "%*.*f", size, size-2, val*neg);
		}
		return(tval);
	}

	/*
	 * Determine the maximum value that will fit in "size" digits (n)
	 * and calculate a rounding value that will allow us to predict the
	 * size of the rounded result
	 */
	n = (float)pow(10.0, (double)size);
	a = val / 2 * 10.0/n;

	/*
	 * If the number can be represented by the available digits, determine
	 * the maximum precision
	 */
	if (val <= n-.5) {
		for (k=n/10, i=0; ; k/=10, i++) {
			if (val+a > k) {
				if (i) {
					i--;
				}
				if (prec && prec < i) {
					i = prec;
				}

				sprintf(tval, "%*.*f", size, i, val*neg);
				return(tval);
			}
		}
	}

	/*
	 * Not enough digits, divide by 1000's until we have enough digits.
	 * Add a suffix to indicate the units.
	 */
	n /= 10;
	for (i=0, k=1000; i<4; i++, k*=1000) {
		if ((val+a)/k < n) {
			sprintf(tval, "%*.0f%c", size-1, val*neg/k, suff[i]);
			return(tval);
		}
	}

	/*
	 * Too big to represent with this scheme, return *'s.
	 */
	strcpy(tval, "************");
	tval[size] = 0;
	return(tval);
}

#ifdef MPI
void mpi_start() {
	int i;

	for (segs=0, wt=workt, i=0; i < wheight; wt++, segs++) {
		wt->stat = 0;
		wt->sline = i;
		if (i + lines > wheight) {
			wt->nline = wheight - i;
		} else {
			wt->nline = lines;
		}
		wt->id = 0;
		i += lines;
	}

	md.msg_len = msg_len;
	/* Send a work segment to each of the slaves */
	for (widx=0, i=1; i<mpi_size && widx<segs; widx++, i++) {
		md.wwidth = wwidth;
		md.iter = iter;
		md.sline = workt[widx].sline;
		md.nline = workt[widx].nline;
		md.distx = dx;
		md.disty = dy;
		md.incx = incx;
		md.incy = incy;
		MPI_Send(&md, sizeof(md), MPI_CHAR, i, widx, MPI_COMM_WORLD);
		workt[widx].stat = 1;
		workt[widx].id = i;
	}
	done = 0;
}

Boolean mpi_work() {
	int i, k, j, idx;
	int line, bidx;
	double ttime;
	char text[40];

	MPI_Recv(msg_buf, msg_len, MPI_INT, MPI_ANY_SOURCE,
		MPI_ANY_TAG, MPI_COMM_WORLD, &status);
	i = status.MPI_SOURCE;
	j = status.MPI_TAG;
	
	/* If we have more work to dispatch, send it now */
	if (widx < segs) {
		md.wwidth = wwidth;
		md.iter = iter;
		md.sline = workt[widx].sline;
		md.nline = workt[widx].nline;
		md.distx = dx;
		md.disty = dy;
		md.incx = incx;
		md.incy = incy;
		MPI_Send(&md, sizeof(md), MPI_CHAR, i, widx,
			MPI_COMM_WORLD);
		workt[widx].stat = 1;
		workt[widx].id = i;
		widx++;
	}
	workt[j].stat = 2;

	for (bidx=0, line=workt[j].sline;
			line<workt[j].sline+workt[j].nline; line++) {
		idx = line * wwidth;
		for (k=0, i=0; i<wwidth; i++) {
			if (display) {
				k = color_xlate(msg_buf[bidx], dp, k);
			}
			dpall[idx + i] = msg_buf[bidx];
			flops += (double)msg_buf[bidx++];
		}
		if (display) {
			XPutImage(draw_d,draw_win,draw_gc,xip,0,0,0,
				line, wwidth,1);
		}
	}

	upd_stats();
	if (++done < segs) {
		return(FALSE);
	}
	/*
	 * All done, update status
	 */
	upd_stats();
	flops *= 9.0;
	flops += (double)(wwidth * wheight * 3);
	ttime = (double)(tpe.tv_sec - tps.tv_sec) * 1000000.0;
	ttime += (double)(tpe.tv_usec - tps.tv_usec);
	flops = flops / ttime;
	sprintf(text,"M-Flops %9.2f", flops);
	XtSetArg(arg[0], XtNlabel, text);
	XtSetValues(st_mflops, arg, 1);

	return (TRUE);
}

sched()
{
	static int len=0;
	int tag;


	while (1) {
		MPI_Recv(&md, sizeof(md), MPI_CHAR, 0, MPI_ANY_TAG,
			MPI_COMM_WORLD, &status);
		tag = status.MPI_TAG;
		if (md.msg_len != len) {
			free(msg_buf);
			if ((msg_buf = (int *)malloc(md.msg_len * sizeof(int)))
					== NULL) {
               			printf("malloc 0 failed\n");
		                exit(1);
			}
			len = md.msg_len;
		}

		do_block(md);

		MPI_Send(msg_buf, len, MPI_INT, 0, tag, MPI_COMM_WORLD);
	}
}
#endif
