/********************************************************************************
NDPMon - Neighbor Discovery Protocol Monitor
Copyright (C) 2006 MADYNES Project, LORIA - INRIA Lorraine (France)

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

Author Info:
  Name: Thibault Cholez
  Mail: thibault.cholez@esial.uhp-nancy.fr

Maintainer:
  Name: Frederic Beck
  Mail: frederic.beck@loria.fr

MADYNES Project, LORIA-INRIA Lorraine, hereby disclaims all copyright interest in
the tool 'NDPMon' (Neighbor Discovery Protocol Monitor) written by Thibault Cholez.

Olivier Festor, Scientific Leader of the MADYNEs Project, 20 August 2006
***********************************************************************************/

#include "monitoring_ra.h"

/*Test if the RA comes from a router with IP6 address specified in the
 *configuration file.
 *@return: 0=ok, not 0=pb
 */
int watch_ra_ip(char* buffer, struct ether_header* eptr, struct ip6_hdr* ipptr)
{
	router_list_t *tmp = routers;
	char ip_address[IP6_STR_SIZE];

	while(tmp != NULL)
	{
		/* RA supposed to come from a LLA */
		if(IN6_ARE_ADDR_EQUAL(&ipptr->ip6_src,&(tmp->lla)))
			return 0;

		ipv6_ntoa(ip_address, ipptr->ip6_src);
		sprintf (buffer, "wrong router ip %s %s", ether_ntoa((struct ether_addr*) (eptr->ether_shost)), ip_address);
		notify(2, buffer);
		return 2;

		tmp = tmp->next;
	}

	return 0;
}


/*Test if the RA comes from a router with MAC address specified in the
 *configuration file.
 *@return: 0=ok, not 0=pb
 */
int watch_ra_mac(char* buffer, struct ether_header* eptr, struct ip6_hdr* ipptr)
{
	router_list_t *tmp = routers;
	char ip_address[IP6_STR_SIZE];
	char* mac_address;

	mac_address= (char*)ether_ntoa((struct ether_addr*) (eptr->ether_shost));

	while(tmp != NULL)
	{
		/* RA supposed to come from a LLA */
		if(!MEMCMP(eptr->ether_shost,&(tmp->mac),sizeof(struct ether_addr)))
			return 0;

		ipv6_ntoa(ip_address, ipptr->ip6_src);
		sprintf (buffer, "wrong router mac %s %s", mac_address, ip_address);
		notify(2, buffer);
		return 2;

		tmp = tmp->next;
	}

	return 0;
}


/*Test if the prefix specified in RA is right according to the configuration
 *file
 */
int watch_ra_prefix(char* buffer,  const u_char* packet, struct ether_header* eptr, struct ip6_hdr* ipptr, int packet_len)
{
	router_list_t *tmp = routers;
	char prefix[IP6_STR_SIZE];
	const u_char* pos;
	struct nd_opt_hdr* optptr;  /*netinet/icmp6.h*/
	struct nd_opt_prefix_info* preptr = NULL;
	int find=0;

	/*We have to search the prefix option among the others NA options*/
	pos = packet + ETHERNET_SIZE + IPV6_SIZE + sizeof(struct nd_router_advert);
	optptr = (struct nd_opt_hdr*) ( pos ); 

	while((optptr->nd_opt_type!=0)&&((u_char*)optptr < (packet+packet_len)))
	{
		if(optptr->nd_opt_type ==  ND_OPT_PREFIX_INFORMATION)
		{

			preptr = (struct nd_opt_prefix_info*) optptr;
			ipv6pre_ntoa(prefix, preptr->nd_opt_pi_prefix);

			while(tmp != NULL)
			{
				if(router_has_prefix(routers, tmp->lla, tmp->mac, preptr->nd_opt_pi_prefix, preptr->nd_opt_pi_prefix_len))
					find = 1;

				tmp = tmp->next;
			}

			if(!find)
			{
				char ip_address[IP6_STR_SIZE];
				ipv6_ntoa(ip_address, ipptr->ip6_src);
				sprintf (buffer, "wrong prefix %s %s %s", prefix,(char*)ether_ntoa((struct ether_addr*) (eptr->ether_shost)), ip_address);
				notify(2, buffer);
				return 2;
			}
		}

		/*Next option field*/
		pos += (optptr->nd_opt_len)*8;
		optptr = (struct nd_opt_hdr*) ( pos ); 

	}

	return 0;
}

int watch_ra(char* buffer,  const u_char* packet, struct ether_header* eptr, struct ip6_hdr* ipptr, int packet_len)
{
	int found_router = 0, ret = 0;
	struct ether_addr *src_eth;
	char ip_address[IP6_STR_SIZE];

	src_eth = (struct ether_addr *) eptr->ether_shost;
	ipv6_ntoa(ip_address, ipptr->ip6_src);

	found_router = is_router_in(routers, ipptr->ip6_src, *src_eth);

	if(learning)
	{
		const u_char* pos;
		struct nd_opt_hdr* optptr;  /*netinet/icmp6.h*/
		struct nd_opt_prefix_info* preptr = NULL;
		char prefix[INET6_ADDRSTRLEN];

		/*We have to search the prefix option among the others NA options*/
		pos = packet + ETHERNET_SIZE + IPV6_SIZE + sizeof(struct nd_router_advert);
		optptr = (struct nd_opt_hdr*) ( pos ); 
		while((optptr->nd_opt_type!=0)&&((u_char*)optptr < (packet+packet_len)))
		{
			if(optptr->nd_opt_type ==  ND_OPT_PREFIX_INFORMATION)
			{
				preptr = (struct nd_opt_prefix_info*) optptr;
				ipv6pre_ntoa(prefix, preptr->nd_opt_pi_prefix);
				break;
			}

			/*Next option field*/
			pos += (optptr->nd_opt_len)*8;
			optptr = (struct nd_opt_hdr*) ( pos );
		}

		if(!found_router)
		{
			add_router(&routers, ipptr->ip6_src, *src_eth);
			add_prefix(&routers, ipptr->ip6_src, *src_eth, preptr->nd_opt_pi_prefix, preptr->nd_opt_pi_prefix_len);
		}
		else
		{
			if( !router_has_prefix(routers, ipptr->ip6_src, *src_eth, preptr->nd_opt_pi_prefix, preptr->nd_opt_pi_prefix_len) )
				add_prefix(&routers, ipptr->ip6_src, *src_eth, preptr->nd_opt_pi_prefix, preptr->nd_opt_pi_prefix_len);
		}

		return 0;
	}

	if(!found_router)
	{
		int found_mac = is_router_mac_in(routers, *src_eth);
		int found_lla = is_router_lla_in(routers, ipptr->ip6_src);

		if( found_mac && found_lla)
		{
			/* valid MAC and IP, but not together */
			sprintf (buffer, "wrong couple IP/MAC %s %s in RA", (char*)ether_ntoa(src_eth), ip_address);
			notify(2, buffer);
			ret = 2;
		}
		else if( found_mac && !found_lla)
		{
			/* wrong IP */
			sprintf (buffer, "wrong router ip %s %s", (char*)ether_ntoa(src_eth), ip_address);
			notify(2, buffer);
			ret = 2;
		}
		else if( !found_mac && found_lla)
		{
			/* wrong MAC */
			sprintf (buffer, "wrong router mac %s %s", (char*)ether_ntoa(src_eth), ip_address);
			notify(2, buffer);
			ret = 2;
		}
	}
	else
	{
		const u_char* pos;
		struct nd_opt_hdr* optptr;  /*netinet/icmp6.h*/
		struct nd_opt_prefix_info* preptr;
		char prefix[INET6_ADDRSTRLEN];

		/*We have to search the prefix option among the others NA options*/
		pos = packet + ETHERNET_SIZE + IPV6_SIZE + sizeof(struct nd_router_advert);
		optptr = (struct nd_opt_hdr*) ( pos ); 

		while((optptr->nd_opt_type!=0)&&((u_char*)optptr < (packet+packet_len)))
		{
			if(optptr->nd_opt_type ==  ND_OPT_PREFIX_INFORMATION)
			{
				preptr = (struct nd_opt_prefix_info*) optptr;
				ipv6pre_ntoa(prefix, preptr->nd_opt_pi_prefix);
				/* Check prefix */
				if( !router_has_prefix(routers, ipptr->ip6_src, *src_eth, preptr->nd_opt_pi_prefix, preptr->nd_opt_pi_prefix_len) )
				{
					char ip_address[IP6_STR_SIZE];
					ipv6_ntoa(ip_address, ipptr->ip6_src);
					sprintf (buffer, "wrong prefix %s %s %s", prefix,(char*)ether_ntoa((struct ether_addr*) (eptr->ether_shost)), ip_address);
					notify(2, buffer);
					ret = 2;
				}

			}
			/*Next option field*/
			pos += (optptr->nd_opt_len)*8;
			optptr = (struct nd_opt_hdr*) ( pos );
		}
	}

	return ret;
}
