/********************************************************************************
NDPMon - Neighbor Discovery Protocol Monitor
Copyright (C) 2006 MADYNES Project, LORIA - INRIA Lorraine (France)

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

Author Info:
  Name: Thibault Cholez
  Mail: thibault.cholez@esial.uhp-nancy.fr

Maintainer:
  Name: Frederic Beck
  Mail: frederic.beck@loria.fr

MADYNES Project, LORIA-INRIA Lorraine, hereby disclaims all copyright interest in
the tool 'NDPMon' (Neighbor Discovery Protocol Monitor) written by Thibault Cholez.

Olivier Festor, Scientific Leader of the MADYNEs Project, 20 August 2006
***********************************************************************************/

#include "routers.h"
#include "neighbors.h"
#include "parser.h"
#include "ndpmon_defs.h"
#include "ndpmon.h"

int DEBUG = 0;

router_list_t *routers = NULL;
neighbor_list_t *neighbors = NULL;
char admin_mail[128] = "";
char syslog_facility[16] = "";
int ignor_autoconf = 0;
char config_path[128] = "/usr/local/ndpmon/config_ndpmon.xml";
char cache_path[128] = "/usr/local/ndpmon/neighbor_list.xml";
char dtd_path[128] = "/usr/local/ndpmon/neighbor_list.dtd";
char dtd_config_path[128] = "/usr/local/ndpmon/config_ndpmon.dtd";

#ifdef _MACRESOLUTION_
manufacturer_t *manuf = NULL;
#endif

int learning = 0;

static pcap_t* descr = NULL;


/*Function called each time that a packet pass the filter and is captured*/
void callback(u_char *args,const struct pcap_pkthdr* hdr,const u_char*
        packet)
{

	static time_t last_save_time=0;/*to periodicaly save the neighbor cache*/
	time_t current= time(NULL);

	struct ether_header *eptr;  /* net/ethernet.h */
	struct ip6_hdr* ipptr;     /*netinet/ip6.h*/
	struct icmp6_hdr* icmpptr;  /*netinet/icmp6.h*/
	struct ip6_ext* opt_hdr;

	struct nd_router_solicit* rsptr;  
	struct nd_router_advert* raptr;
	struct nd_neighbor_solicit* nsptr;
	struct nd_neighbor_advert* naptr;
	struct nd_redirect* rdptr;

	int result=-1;
	int new_eth =0;/*Use to detect dad dos attack, 1 if NA annouces a new station*/
	int type_58 = 0;
	char message[255];

	const time_t* time = (const time_t*) &(hdr->ts).tv_sec;


	if(DEBUG)
	{
		/* General info on the paquet */
		fprintf(stderr,"length of this packet: %d\n", hdr->len);
		fprintf(stderr,"Recieved at: %s", (char*)ctime(time));
	}

	/* Info from the ethernet layer */
	eptr = (struct ether_header *) packet;
	if(DEBUG)
	{
		print_eth(*eptr);
	}

	/* Info from the IPV6 layer */
	if (ntohs (eptr->ether_type) ==  ETHERTYPE_IPV6)
	{
		if(DEBUG)
		{
			fprintf(stderr,"Ethernet type hex:%x dec: it's an IPv6 packet\n", ntohs(eptr->ether_type));
		}

		ipptr = (struct ip6_hdr*)(packet + ETHERNET_SIZE);
		
		if(DEBUG)
		{
			print_ip6hdr(*ipptr);
		}


		/*Jumping optional headers if there are*/
		if(ipptr->ip6_nxt != 58)
		{
			opt_hdr = (struct ip6_ext*)(packet + ETHERNET_SIZE + IPV6_SIZE);
			if(DEBUG)
			{
				fprintf(stderr,"next option header : %d\n",opt_hdr->ip6e_nxt);
			}

			while((opt_hdr->ip6e_nxt != 58) &&((u_char*)opt_hdr < (packet+hdr->len)))
			{
				/*cf rfc 2460: opt_hdr->ip6e_len 
				  8-bit unsigned integer.  Length of the extension
				  header in 8-octet units, not including the first
				  8 octets.
				  */
				opt_hdr = (struct ip6_ext*)((u_char*)opt_hdr + (1 + opt_hdr->ip6e_len)*8);
				if(DEBUG)
				{
					fprintf(stderr,"next option header : %d\n",opt_hdr->ip6e_nxt);
				}
			}
			if(opt_hdr->ip6e_nxt == 58)
			{
				icmpptr = (struct icmp6_hdr*)((u_char*)opt_hdr + (1 + opt_hdr->ip6e_len)*8);
				/*Info from the ICMPv6 layer*/
				if(DEBUG)
				{
					fprintf(stderr,"IP type after ext headers: %d, it's an ICMPv6 packet\n",opt_hdr->ip6e_nxt);
				}
				type_58 = 1;
			}
		}
		else
		{
			icmpptr = (struct icmp6_hdr*)(packet + ETHERNET_SIZE + IPV6_SIZE);
			/*Info from the ICMPv6 layer*/
			if(DEBUG)
			{
				fprintf(stderr,"IP type: %d, it's an ICMPv6 packet\n",ipptr->ip6_nxt);
			}
			type_58 = 1;
		}


		if(type_58)
		{
			struct ether_addr *src_eth;

			if(DEBUG)
			{
				fprintf(stderr,"ND type: %d\n",(icmpptr->icmp6_type));
			}

			/*General verifications*/
			result = watch_eth_mismatch(message, packet, eptr, ipptr, icmpptr, hdr->len);
			result =  watch_eth_broadcast(message, eptr, ipptr);
			result =  watch_ip_broadcast(message, eptr, ipptr);
			result =  watch_bogon(message, eptr, ipptr);

			switch (icmpptr->icmp6_type)
			{
				case  ND_ROUTER_SOLICIT:
					fprintf(stderr,"----- ND_ROUTER_SOLICIT -----\n");
					rsptr = (struct nd_router_solicit*) (packet + ETHERNET_SIZE + IPV6_SIZE);
					if(DEBUG)
					{
						print_rs(*rsptr);
					}

					/*RS with node addr Should be used to build cache so:*/
					if(!IN6_IS_ADDR_UNSPECIFIED(&ipptr->ip6_src))
					{
						src_eth = (struct ether_addr *) eptr->ether_shost;
						new_station(&neighbors,*src_eth,ipptr->ip6_src, &new_eth);
					}
					break;

				case ND_ROUTER_ADVERT:
					fprintf(stderr,"----- ND_ROUTER_ADVERT -----\n");
					src_eth = (struct ether_addr *) eptr->ether_shost;
					raptr = (struct nd_router_advert*) (packet + ETHERNET_SIZE + IPV6_SIZE);
					if(DEBUG)
					{
						print_ra(*raptr);
					}

					result = new_station(&neighbors,*src_eth,ipptr->ip6_src, &new_eth);
					result = watch_ra(message, packet, eptr, ipptr, hdr->len);

					break;

				case ND_NEIGHBOR_SOLICIT:
					fprintf(stderr,"----- ND_NEIGHBOR_SOLICIT -----\n");
					nsptr = (struct  nd_neighbor_solicit*) (packet + ETHERNET_SIZE + IPV6_SIZE);
					if(DEBUG)
					{
						print_ns(*nsptr);
					}
					/*NS with node addr Should be used to build cache so:*/
					if(!IN6_IS_ADDR_UNSPECIFIED(&ipptr->ip6_src))
					{
						src_eth = (struct ether_addr *) eptr->ether_shost;
						new_station(&neighbors,*src_eth,ipptr->ip6_src, &new_eth);
					}
					watch_dad(eptr, ipptr, nsptr);
					break;

				case ND_NEIGHBOR_ADVERT:
					fprintf(stderr,"----- ND_NEIGHBOR_ADVERT -----\n");
					src_eth = (struct ether_addr *) eptr->ether_shost;
					naptr = (struct nd_neighbor_advert*) (packet + ETHERNET_SIZE + IPV6_SIZE );
					if(DEBUG)
					{
						print_na(*naptr);
					}

					new_station(&neighbors,*src_eth,ipptr->ip6_src, &new_eth);
					result = watch_dad_dos(message, eptr, ipptr, naptr, new_eth);
					result = watch_R_flag(message, eptr, ipptr, naptr);

					break;

				case ND_REDIRECT:
					rdptr = (struct nd_redirect*) (packet + ETHERNET_SIZE + IPV6_SIZE);
					print_rd(*rdptr);

					result = watch_rd_src(message, eptr, ipptr);
					break;
				case 128:
					printf ("Echo request: %d\n", icmpptr->icmp6_type);
					break;
				case 129:
					printf ("Echo reply: %d\n", icmpptr->icmp6_type);
					break;
				case 1:
					printf ("Address Unreachable: %d\n", icmpptr->icmp6_type);
					break;
				default:
					printf ("Unknown ICMPv6 type: %d\n", icmpptr->icmp6_type);
					break;

			}/*end switch*/
		}
		else 
			fprintf(stderr,"IP type:%d , is not an ICMPv6 packet\n", ipptr->ip6_nxt);
	}
	else
		fprintf(stderr,"Ethernet type %x is not an IPv6 packet\n", ntohs(eptr->ether_type));
	fprintf(stderr,"------------------\n\n");

	if(difftime(current, last_save_time)> 2*60)
	{
		write_cache();
		last_save_time = current;
	}


}/*end callback*/


void usage()
{
	fprintf(stderr,"Usage: ndpmon [ -i interfacename ] [ -f config_file ] [-e config-dtd] [ -F filter ] [ -n number ] [ -L ]  [-g neighbor_file] [-d neighbor_dtd] [-v] [-h]\nPlease refer to manpage for more details.\n");
	exit(1);
}


/*To write cache before exiting*/
void handler(int n)
{
	if(learning)
		write_config();

	write_cache();
	fprintf(stderr,"\nInterrupted ;) \n");

	/* cleanup */
	pcap_close(descr);
	syslog(LOG_NOTICE,"Program Stopped...");
	closelog();
	/* free data structures */
	clean_routers(&routers);
	clean_neighbors(&neighbors);
#ifdef _MACRESOLUTION_
	clean_manufacturer(&manuf);
#endif
	exit(0);
}



/*To display properly the network address and device's mask */
void interface_spec(char* interface, bpf_u_int32 netp, bpf_u_int32 maskp)
{

	struct in_addr addr;
	char *net; /* network address */
	char *mask;/* network mask */

	if(DEBUG)
	{
		fprintf(stderr,"Interface: %s\n", interface);
	}

	addr.s_addr = netp;
	net = inet_ntoa(addr);
	if(net == NULL)
	{
		fprintf(stderr,"Problem with net adress"); 
		exit(1);
	}

	if(DEBUG)
	{
		fprintf(stderr,"Net: %s\n",net);
	}

	addr.s_addr = maskp;
	mask = inet_ntoa(addr);
	if(mask == NULL)
	{
		fprintf(stderr,"Problem with mask"); 
		exit(1);
	}

	if(DEBUG)
	{
		fprintf(stderr,"Mask: %s\n",mask);
		fprintf(stderr,"\n");
	}
}


int main(int argc,char **argv)
{ 

	char *interface; /* name of the interface/device to use */ 
	char *filter; /* filter to select the parckets to grab */
	int nb_packet = 0;
	int op = 0;

	char errbuf[PCAP_ERRBUF_SIZE];
	struct bpf_program fp;/* string which contains the filter expression */
	bpf_u_int32 maskp; /* mask  */
	bpf_u_int32 netp; /* ip */
	u_char* args = NULL; /* for the callback function */

	interface = NULL;
	filter = "icmp6";
	nb_packet=0;/*all packets are captured until kill*/

	memset(errbuf,0,PCAP_ERRBUF_SIZE);

	while ((op = getopt(argc, argv, "i:vhF:n:f:e:g:d:L")) != EOF)
		switch (op)
		{

			case 'i':
				interface = optarg;
				fprintf(stderr,"interface: %s \n", interface);
				break;
			case 'F':
				filter = optarg;
				fprintf(stderr,"filter expression: %s \n", filter);
				break;
			case 'n':
				nb_packet = atoi(optarg);
				fprintf(stderr,"nb packets: %d \n", nb_packet);
				break;
			case 'f':
				strcpy(config_path,optarg);
				fprintf(stderr,"config file path: %s \n", config_path);
				break;
			case 'e':
				strcpy(dtd_config_path, optarg);
				fprintf(stderr,"config dtd file path: %s \n", dtd_config_path);
				break;
			case 'g':
				strcpy(cache_path, optarg);
				fprintf(stderr,"cache file path: %s \n", cache_path);
				break;
			case 'd':
				strcpy(dtd_path, optarg);
				fprintf(stderr,"neighbor dtd file path: %s \n", dtd_path);
				break;
			case 'L':
				learning = 1;
				fprintf(stderr,"learning phase\n");
				break;

			case 'v':
				DEBUG = 1;
				fprintf(stderr,"Debug mode on\n");
				break;

			case 'h':
				usage();
				break;

			default:
				usage();
				break;
		}
	
	fprintf(stderr,"\n");

	signal(SIGINT,handler);
	signal(SIGQUIT,handler);
	signal(SIGTERM,handler);


	/* if the device isn't specified */
	if ( (interface == NULL) && ((interface = pcap_lookupdev(errbuf)) == NULL))
	{
		fprintf(stderr,"%s\n",errbuf); exit(1); 
	}

	/* pcap get information on the interface */
	pcap_lookupnet(interface,&netp,&maskp,errbuf);
	interface_spec(interface,netp,maskp);


	/* open device for reading */
	descr = pcap_open_live(interface,BUFSIZ,1,-1,errbuf);
	if(descr == NULL)
	{
		fprintf(stderr,"pcap_open_live(): %s\n",errbuf); exit(1);
	}



	/* using the filter */
	if(pcap_compile(descr,&fp,filter,0,netp) <0)
	{ 
		fprintf(stderr,"Error calling pcap_compile %s \n", pcap_geterr(descr));
		exit(1); 
	}

	if(pcap_setfilter(descr,&fp) == -1)
	{
		fprintf(stderr,"Error setting filter\n"); exit(1);
	}

#ifdef _MACRESOLUTION_
	read_manuf_file("/usr/local/ndpmon/plugins/mac_resolv/manuf",&manuf);
#endif

	/* opening and parsing the configuration xml files*/
	parse_config();
	parse_cache();
	set_alarm(!learning);

	if(DEBUG)
	{
		fprintf(stderr,"Routers' List Parsed:\n");
		print_routers(routers);
		fprintf(stderr,"Neighbors' List Parsed:\n");
		print_neighbors(neighbors);
	}

	/* then we can capture the packets */ 
	pcap_loop(descr,nb_packet,callback,args);

	write_cache();

	/* cleanup */
	pcap_freecode(&fp);
	pcap_close(descr);

	fprintf(stdout,"Finished! \n");
	/*printf ( "using libpcap: %s \n", pcap_lib_version());*/

	return 0;

}/*end main*/


