C PROGRAMMER'S GUIDE FOR NEFU TEST DEVELOPMENT

INTRODUCTION

    The purpose of this file is to show a C programmer how to use the nefu
Test Development Kit (TDK) to develop and debug a binary nefu test.  Tests
developed with the TDK can be quickly integrated in to the main nefu source.

    You must build nefu's main source before you can use this kit.

TEST DEVELOPMENT

    The nefu TDK is used to create a stand alone program that exercises a
test on a user specified machine.  Once built, the tdk requires the following
syntax:

Usage: ./tdk machine test [arg ...]

    The first argument to the tester program must be the target machine.  The
second argument must be the test we wish to execute.  The remainder of the
arguments are optional and will be passed to the test.

    All nefu tests have an initilization function and a test function.  The
initilization function is responsable for test argument checking and useage,
the test function is where protocol transaction code resides.

INITILIZATION FUNCTION

    Most nefu tests do not require arguments and should simply use the built
in function init_noargs.  Tests with arguments need their own initilization
routine to insure that they're used properly.  The prototype that a test's
initilization routine must conform to is as follows:

char *init_mytest( struct test *t );

    Initilization routines return a NULL string if test t's arguments conform
to it's useage.  If there was an input error, init_mytest should return a
test useage description.

TEST FUNCTION

    To create a new test for nefu, modify the function test_tdk located in
tdk.c.  It's prototype is as follows:

int test_tdk( struct test *t, struct report *r );

    All tests written for nefu are two argument functions.  The first
argument, struct test *t, contains optional arguments and internet
information for the target.  The second argument, struct report *r, is used
for error message reporting.

    Tests written for nefu should return a predefined integer value once a
result has been determined.  The three valid codes are T_UP, T_DOWN, and
T_MAYBE_DOWN.  T_UP is returned if the test was successful.  T_DOWN should be
returned if a test fails in a conclusive manner, such as having a connection
refused or if the service tested indicates an error.  T_MAYBE_DOWN is
generated when the test was unsuccessful, but the point of failure could not
be determined, such as when a connection times out.

    Tests written for nefu must return programatic control back to their
parent function.  Tests may not contain exit calls.  Programatically unsound
tests integrated in the main nefu source have the potential to cause
monitoring outages.

    In the event of a system error, such as the failure of a support function
or system call, the error text should reported via the reporting API and the
function should return T_MAYBE_DOWN.

THE TEST STRUCTURE

#include <sys/types.h>
#include <netinet/in.h>
#include "nefu.h"

struct test {
    char		*t_name
    struct machine	*t_machine;
    char		*t_argv[];
    int			t_argc;
    struct sockaddr_in	t_sin;
};

    The test data structure contains information that is nessecary in writing
nefu tests.  t_name conatins that name of the test.  t_machine is a pointer
to the test's machine structure, detailed below.  t_argc and t_argv[] are
test argument counters and vector, respectively.  t_sin is a sockaddr_in
filled with the machine's address and the test's port.

struct machine {
    char		*m_name;
};

    The only item of interest in the machine structure is m_name, the name of
the machine we are testing.

ERROR REPORTING & THE REPORT STRUCTURE

int report_printf( struct report *r, char *fmt, ... );

    The report_printf() function is used to report diagnostic error messages
in the event of a system error or test failure.  For nefu test authoring
purposes, simply use the report structure provided as report_printf's first
argument.  fmt is a control string similar to the printf format string.
Currently only %s, %m, and %d are supported.  Additional arguments should be
variables appropriate for their placeholders in the format string.

    Additional information, such as performance statistics, may be logged via
syslog.

TIME SUPPORT FUNCTIONS

int time_start( struct timeval *tv );
int time_end( struct timeval *tv );
int time_minus( struct timeval *total, struct timeval *minus );

    The duration of an event can be timed using time_start and time_end.
time_end will modify a timeval to show how much time has passed since it was
initilized with time_start.  time_minus subtracts the ammount of time
contained in the minus structure from the total.  All time functions return 0
if successful, -1 if there is a system failure or a value out of range.

TEST DEVELOPMENT IN THE TDK SOURCE:
    Create a test for nefu, use ldap as an example:

    1.  ttable.c:
	    Add an entry for the new test.  Entries have four fields:
		1.  test name for nefu.conf
		2.  default port, 0 if not applicable
		3.  test function
		4.  test initilization function

		EXAMPLE:
		    { "ldap", 389, test_ldap, init_ldap }

    2.  ttable.h:
	    All nefu tests have the same prototype, add the new test to
		the prototype list.

		EXAMPLE:
		    TESTPROTO( test_ldap );
	    
	    If the test has an initilization routine, list it's prototype
		here as well.  Only tests with arguments will require an
		initilization routine.  Note that alternate ports are
		specified in nefu.conf along with the test, so port
		information is not considered an argument.

    3.  Makefile:
	    Add ldap.o object file to TESTEROBJ

	    Add additional software libraries to LIBS

    4.  ldap.c:
	    Write a test that exercises the LDAP protocol.  Note port
		information is stored in t->t_sin.sin_port.

    5.  build and test:
	    make
	    ./tdk test.machine.edu ldap
    
    6.  integrate new test in to main source:
	    If the test requires no additional software, it can now be simply
		installed in to the main source.  If the test requires
		additional libraries, autoconf must be used to build the test
		depending on library availability.


SIMPLE TEST INTEGRATION (NO ADDITIONAL LIBRARIES):

    Tests created with the TDK can be easily integrated in to nefu's main
source by following the example below.

    1.  From nefu's main source directory, copy and rename the new test's
	    source file from the TDK to libtest.  Note that we have used the
	    name "mytest" for example purposes only.

	    cp TDK/mytest.c libtest/mytest.c

    2.  Edit libtest/Makefile.in: add "mytest.c" to SRC, "mytest.o" to OBJ.

    3.  Edit ttable.c to add a new entry to the testtable array.

	    { "mytest", default_port, test_mytest, init_mytest },

    4.  Edit ttable.h.  Append a new TESTPROTO for test_mytest.  Append a new
	    INITPROTO for init_mytest, if any.

    5.  Rebuild nefu


TEST INTEGRATION WITH MASTER SOURCE USING AUTOCONF FOR LIBRARY DETECTION:

    1.  From nefu's main source directory, copy and rename the new test's
	    source file from the TDK to libtest.  Note that we have used the
	    name "mytest" for example purposes only.

	    cp TDK/mytest.c libtest/mytest.c

    2.	configure.ac:
	    CHECK_LDAP
	
    3.	aclocal.m4:
	    add macro "CHECK_LDAP".  if configure finds ldap:
		LIBS: add required ldap libs
		CFLAGS: add required ldap cflags
		LDFLAGS: add required ldap ldflags
		TEST_SRC: add required ldap source files
		TEST_OBJ: add required ldap object files
		HAVE_LDAP: AC_DEFINE variable for use in nefu source

    4.	ttable.c
	    add ifdef for optional compiling

    5.	ttable.h
	    add ifdef for optional compiling
