/**********          ntp.c          **********/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <sys/time.h>
#include <syslog.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include "nefu.h"

#define WAIT 	4
#define TRIES	4
#define EPOCH	((unsigned)2208988800) /* ((70*365) + (70/4)) * 24 * 60 * 60 */

struct ntptime {
    unsigned	nt_sec;
    unsigned	nt_fsec;
};

struct sntp {
    unsigned		sn_li:2;
    unsigned		sn_vn:3;
    unsigned		sn_mode:3;
    unsigned		sn_stratum:8;
    unsigned		sn_poll:8;
    unsigned		sn_precision:8;
    unsigned		sn_rodelay;
    unsigned		sn_rodispersion;
    unsigned		sn_reidentifier;
    struct ntptime	sn_retimestamp;
    struct ntptime	sn_ortimestamp;
    struct ntptime	sn_rxtimestamp;
    struct ntptime	sn_txtimestamp;
    unsigned char	sn_authenticator[ 96 ];
};

    int
test_ntp( struct test *t, struct report *r )
{
    int			i, rc, s, fromlen;
    int			off;
    unsigned		nt_sec;
    struct sntp		query, resp;
    fd_set		fdset;
    struct sockaddr_in	from;
    struct timeval	tv, tv_send, tv_select, tv_recv;
    char		*direction;

    memset( &query, 0, sizeof( struct sntp ));
    query.sn_vn = 1;
    query.sn_mode = 3;

    if (( s = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 ) {
	report_printf( r, "test_ntp socket: %m" );
	return( T_MAYBE_DOWN );
    }

    for ( i = 0; i < TRIES; i++ ) {
	if ( gettimeofday( &tv_send, NULL ) < 0 ) {
	    report_printf( r, "gettimeofday: %m" );
	    close( s );
	    return( T_MAYBE_DOWN );
	}

	if (( sendto( s, (char *)&query,
		sizeof( struct sntp ) - sizeof( query.sn_authenticator ),
		0, (struct sockaddr *)&t->t_sin,
		sizeof( struct sockaddr_in ))) < 0 ) {
	    report_printf( r, "test_ntp sendto: %m" );
	    close( s );
	    return( T_MAYBE_DOWN );
	}

	tv.tv_sec = WAIT;
	tv.tv_usec = 0;

	while ( 1 ) {
	    FD_ZERO( &fdset );
	    FD_SET( s, &fdset );

	    if ( gettimeofday( &tv_select, NULL ) < 0 ) {
		report_printf( r, "gettimeofday: %m" );
		close( s );
		return( T_MAYBE_DOWN );
	    }

	    if (( rc = select( s + 1, &fdset, NULL, NULL, &tv )) < 0 ) {
		report_printf( r, "test_dns select: %m" );
		close( s );
		return( T_MAYBE_DOWN );
	    }

	    if ( rc == 0 ) {
		break;
	    } else {
		if ( gettimeofday( &tv_recv, NULL ) < 0 ) {
		    report_printf( r, "gettimeofday: %m" );
		    close( s );
		    return( T_MAYBE_DOWN );
		}

		memset( &from, 0, sizeof( struct sockaddr_in ));
		fromlen = ( sizeof( struct sockaddr_in ));
		if (( rc = recvfrom( s, (char *)&resp, sizeof( resp ), 0, 
			(struct sockaddr *)&from, &fromlen )) < 0 ) {
		    report_printf( r, "test_ntp recvfrom: %m" );
		    close( s );
		    return( T_MAYBE_DOWN );
		}

		/*
		 * If we get a response from an unknown address,
		 * just keep going.
		 */
		if ( t->t_sin.sin_addr.s_addr == from.sin_addr.s_addr ) {
		    if ( resp.sn_li == 3 ||
			    ( resp.sn_txtimestamp.nt_sec == 0 &&
			    resp.sn_txtimestamp.nt_fsec == 0 )) {
			report_printf( r, "Not synchronized" );
			close( s );
			return( T_DOWN );
		    } else {
			nt_sec = resp.sn_txtimestamp.nt_sec - EPOCH;
			if ( nt_sec > tv_recv.tv_sec ) {
			    off = nt_sec - tv_recv.tv_sec;
			    direction = "ahead";
			} else {
			    off = tv_recv.tv_sec - nt_sec;
			    direction = "behind";
			}
			if ( off > 30 ) {
			    report_printf( r, "%d seconds %s", off, direction );
			    close( s );
			    return( T_DOWN );
			} else {
			    time_minus( &tv_recv, &tv_send );

			    syslog( LOG_INFO, "%s %s %s %d %ld.%.6ld %d",
				    t->t_rlist, t->t_machine->m_name,
				    t->t_name, i + 1, tv_recv.tv_sec,
				    tv_recv.tv_usec, off );

			    close( s );
			    return( T_UP );		/* good response */
			}
		    }
		}

		/*
		 * This isn't right for linux, et al, that change
		 * the time value passed to select()  XXX
		 */
		if ( time_minus( &tv_recv, &tv_select ) < 0 ) {
		    report_printf( r, "recv'd packet before it was sent" );
		    close( s );
		    return( T_MAYBE_DOWN );
		}
		if ( time_minus( &tv, &tv_recv ) < 0 ) {
		    break;
		}
	    }
	}
    }

    report_printf( r, "Timeout" );
    close( s );
    return( T_MAYBE_DOWN );
}
