/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: flowprobe.c,v 1.13.2.8 2004/08/18 09:41:24 jura Exp $ */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/stat.h>
#include <paths.h>
#include <time.h>
#include <signal.h>
#include <net/ethernet.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <arpa/inet.h>

#ifdef LINUX
#define __FAVOR_BSD 1
        extern "C" {
        #include <pcap.h>

        #ifndef ETHERTYPE_VLAN
                #define ETHERTYPE_VLAN 0x8100 /* IEEE 802.1Q VLAN tagging */
        #endif
        #ifndef ETHERTYPE_IP
                #define ETHERTYPE_IP 0x0800  /* IP protocol */
        #endif
        #ifndef ETHERMTU
                #define ETHERMTU 1500
        #endif
	#undef DLT_LINUX_SLL
	#ifndef DLT_LINUX_SLL
		/* Based on TCPDUMP code */
		#define DLT_LINUX_SLL 113
		#define SLL_HDR_LEN 16 /* total header length */
		#define SLL_ADDRLEN 8 /* length of address field */

		struct sll_header {
			u_int16_t sll_pkttype; /* packet type */
			u_int16_t sll_hatype; /* link-layer address type */
			u_int16_t sll_halen; /* link-layer address length */
			u_int8_t sll_addr[SLL_ADDRLEN]; /* link-layer address */
			u_int16_t sll_protocol; /* protocol */
		};
	#endif
        }
#else
#include <pcap.h>
#endif

#define NETFLOW_CLASS
#include "netflow.h"


//---------------------------------------------------------------------------------
void termination(int signal);
void pcap_callback(unsigned char *args, const struct pcap_pkthdr *hdr, const unsigned char *packet);
void PrintUsage(FILE *);

int offset;
pcap_t *pcap;
unsigned cap_len;

class NetFlow *Flow;

char *destination=NULL;
char *dst_host=NULL;
unsigned short dst_port=20001;
u_char debug=0, quiet=0;

void Debug(char *msg,...);

//---------------------------------------------------------------------------------
int main(int argc, char **argv){
	char *rule=NULL;
	char *interface=NULL;
	struct bpf_program pf;
	bpf_u_int32 net, mask;
	char errbuf[PCAP_ERRBUF_SIZE];
struct ip *ip;
	register const struct ether_header *ep;
#ifdef LINUX
	register const struct sll_header   *sllp;
#endif
	u_short ether_type;
	unsigned active=0,inactive=0,expired=0, promisc=1;
	unsigned char *packet;
	unsigned status;
	unsigned datalink;

int op;

	while((op=getopt(argc, argv, "hqpde:r:i:1:2:")) != EOF){
	switch(op){
		case 'h': PrintUsage(stdout);
				  exit(-1);
				  break;
		case 'd': debug=1;
				  break;
		case 'q': quiet=1;
				  break;
		case 'p': promisc=0;
				  break;
		case 'e': destination=(char *)malloc(strlen(optarg)+1);
				  strcpy(destination, optarg);
				  break;
		case 'r': rule=(char *)malloc(strlen(optarg)+1);
				  strcpy(rule, optarg);
				  break;
		case 'i': interface=(char *)malloc(strlen(optarg)+1);
				  strcpy(interface, optarg);
				  break;
		case '1': active=atoi(optarg);
				  break;
		case '2': inactive=atoi(optarg);
				  break;
		default: break;
		}
	}

	printf("FlowProbe 1.1 (c) 2003 Anton Vinokurov, anton@netams.com\n");
	printf("This is part of NeTAMS project, http://www.netams.com/ \n");

	if (!interface && !quiet) { PrintUsage(stderr); exit(-1); }  
	if (!destination) dst_host="127.0.0.1"; 
	else { 
		char *c; c=strchr(destination, ':');
		if (c) { dst_port=atoi(c+1); c[0]='\0'; dst_host=destination; }
		else dst_host=destination;
	}
//	if (!rule) rule="ip";
	if (!active) active=10*60; // 1 min
	if (!inactive) inactive=1*60; // 10 min

	if (!quiet) {
		printf("Exporting to:\t%s:%d\n", dst_host, dst_port);
		printf("Interface:\t%s\n", interface);
		if(rule) printf("Rule:\t\t%s\n", rule);
		if(!promisc) printf("Promisc mode is off\n");
		printf("Active timeout:\t%d seconds\n", active);
		printf("Inact timeout:\t%d seconds\n", inactive);
	}

	signal(SIGINT, (sig_t)termination);

	pcap=pcap_open_live(interface, MAX_PKT_SIZE, promisc, 100, errbuf);
	if (!pcap) { 
		if (!quiet) fprintf(stderr, "failed to open pcap interface: %s\n", errbuf);	
		exit(1);	
	}

	if (-1==pcap_lookupnet(interface, &net, &mask, errbuf)) {
	        net=0;
                mask=0;
		if (!quiet) fprintf(stderr, "failed to lookup net pcap interface: %s\n", errbuf);	
	}
	
	if(rule) {
		if (-1==pcap_compile(pcap, &pf, rule, 1, net)) { 
			if (!quiet) fprintf(stderr, "failed to compile pcap rule: %s\n", pcap_geterr(pcap));	
			exit(3);	
		}
		if (-1==pcap_setfilter(pcap, &pf)) { 
			if (!quiet) fprintf(stderr, "failed to setfilter pcap rule: %s\n", pcap_geterr(pcap));	
			exit(4);	
		}
	}
	
	datalink=pcap_datalink(pcap);
	switch (datalink){ 
		case DLT_NULL:
			offset=0;
			if (!quiet) fprintf(stderr, "Libpcap:\tBSD loopback\n");
			break;
		case DLT_EN10MB:
			offset=ETHER_HDR_LEN;
			if (!quiet) fprintf(stderr, "Libpcap:\tethernet interface\n");
			break;
		case DLT_PPP:
			offset=4; // PPP_HDRLEN
			if (!quiet) fprintf(stderr, "Libpcap:\tPPP interface\n");
			break;
#ifdef LINUX
		case DLT_LINUX_SLL:
			offset=SLL_HDR_LEN; // PPP_HDRLEN
			if (!quiet) fprintf(stderr, "Libpcap: PPP(SLL) interface\n");
			break;
#endif
		default:
			offset=0;
			if (!quiet) fprintf(stderr, "Libpcap:\tunknown interface\n");
			break;
	}

        Flow=new NetFlow(dst_host,dst_port);
        Flow->SetTimeouts(active,inactive,expired);

	if (!debug) {
		daemon(0, 1); 
		chdir(_PATH_VARTMP);
	}

	while(1) {
		status=pcap_dispatch(pcap, 1, pcap_callback, (unsigned char*)&packet);
		if(status <1) {
                	if(status==1)
               			if(!quiet) printf("pcap dispatch error: %s\n", pcap_geterr(pcap));
        		continue;
        	}
		Flow->Expiresearch();
		switch(datalink) {
			case DLT_EN10MB:
				ep = (struct ether_header *)packet;
				ether_type = ntohs(ep->ether_type);
				packet += ETHER_HDR_LEN;
				break;
#ifdef LINUX
			case DLT_LINUX_SLL:
				if(cap_len < SLL_HDR_LEN) /* Oops */
					continue;
				sllp = (struct sll_header *)packet;
				ether_type = ntohs(sllp->sll_protocol);
				if(ether_type <= ETHERMTU) /* Oops */
					continue;
				packet += SLL_HDR_LEN;
				break;
#endif
			default:
				ether_type = 0xFFFF;
				break;
		}
 	
		if (ether_type!=0xFFFF) {       
recurse:
			if (ether_type == ETHERTYPE_VLAN) {
				ether_type = ntohs(*(u_int16_t *)(packet + 2));
				packet += 4;
				if(ether_type > ETHERMTU)
					goto recurse;
			}
			if(ether_type != ETHERTYPE_IP) continue;
			ip = (struct ip *)packet;
        	} else 
			ip = (struct ip*) (packet + offset);

		Flow->Processpacket(ip);
	}
} 		
//---------------------------------------------------------------------------------
void pcap_callback(unsigned char *args, const struct pcap_pkthdr *hdr, const unsigned char *packet){
	unsigned char **pkt=(unsigned char**)args;
	*pkt=(unsigned char*)packet;
	cap_len=hdr->len;
}

//---------------------------------------------------------------------------------
void termination(int signal){
	Flow->Status();
	delete Flow;
	pcap_close(pcap);
	exit(0);
}
//---------------------------------------------------------------------------------

void PrintUsage(FILE *f){
	fprintf(f, "Usage: flowprobe {options}\nwhere {options} are:\n \
	-h\t print help screen and exit\n\
	-q\t quiet output\n\
	-d\t debug output\n\
	-p\t turn off promisc mode\n\
	-e export_to\t IP address to export flows to\n\
	-r rule\t\t libpcap rule to capture packets\n\
	-i interface\t network interface to listen\n\
	-1 active_timeout\t active flow timeout (sec.)\n\
	-2 inactive_timeout\t inactive flow timeout (sec.)\n");
}
//---------------------------------------------------------------------------------
void Debug(char *msg,...)
{
#ifdef DEBUG
    	va_list args;
    	char *params;

    	if (!debug || quiet) return;
    	params=(char *)malloc(2049);
    	va_start(args, msg);
    	vsprintf(params, msg, args);
    	va_end(args);

    	fprintf(stdout, "[%lu]: %s", time(NULL), params);
    	fflush(stdout);

    	free(params);
#endif
}
//---------------------------------------------------------------------------------
