/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***
*************************************************************************/
/* $Id: netams.h,v 1.51.2.11 2004/05/06 10:35:03 jura Exp $ */

#ifndef _NETAMS_H_
#define _NETAMS_H_

// Global includes from standard libraries
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <time.h>
#include <math.h>
#include <sys/resource.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#ifdef LINUX
#define __FAVOR_BSD 1
#endif

#ifdef PVM
#include <pvm3.h>
#endif

#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <net/ethernet.h>
#include <net/if_arp.h>
#include <sys/ioctl.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <paths.h>
#include <stdarg.h>
#include <signal.h>
#include <sched.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <limits.h>
#ifdef LINUX
	extern "C" {
	#include <pcap.h>
		#include <netinet/ether.h>
	}
#else
#include <pcap.h>
#endif

// Includes of local definitions
class Connection;
class Service;
class User;
class ConnectionsList;
class ServicesList;
class UsersList;
class Policy;
class PdList;
class NetUnit;
class FIFO;
class ScheduleList;
class PrefixTree;
class IPTree;
struct policy_data;
struct alert;
struct ServiceAlerter_cfg;
struct sLoginData;
struct sQuotaData;
struct Monitor_cfg;
struct Quota_cfg;
struct Login_cfg;
extern char UNKNOWN_REFERENCE[];

enum match { MATCH_NONE=0, MATCH_SRC=1, MATCH_DST=2, MATCH_BOTH=3 };

typedef struct ds_rule {
	unsigned number;
	char *rule_str;
	ds_rule *next;
	} ds_rule;
typedef struct Html_cfg;


#include "common.h"
#include "connections.h"
#include "services.h"
#include "security.h"
#include "users.h"
#include "policy_prefix.h"
#include "policy.h"
#include "net_units.h"
#include "iptree.h"
#include "alert_report.h"
#include "messages_fifo.h"
#include "st_any.h"

#define ADD     1
#define REMOVE  0

#define DEBUG_TO_LOG	1
#define DEFAULT_READ_TIMEOUT 1000
#define SCHED_MIN_SLEEP_TIME 999

//default delays for different services
#define PROCESSOR_DELAY 60000 //msecs
#define PROCESSOR_LIFETIME 60000  //msecs
#define QUOTACTL_DELAY 55000  //msecs
#define WEBLOGIN_DELAY 110000 //msecs

#define KILOBYTE 1024
#define GENERATE_CHILD_HTML


#define PARSE_KILL	-1
#define PARSE_EXIT	-2
#define PARSE_RELOAD	-3
#define PARSE_SHUTDOWN	-4
#define PARSE_OK	1

#define OID_USER	1
#define OID_UNIT	2
#define OID_POLICY	3
#define OID_REPORT	6
#define OID_TASK	8

/////////////////////////////////////////////////////////////////////////////////////////
// Versions Management //////////////////////////////////////////////////////////////////
extern const char aaa_fw_software_name[];
extern const unsigned int aaa_fw_major_version;
extern const unsigned int aaa_fw_minor_version;
extern const unsigned int aaa_fw_build_version;
extern const unsigned int aaa_fw_build_version_local;
extern const char aaa_fw_build_time[];
extern const char aaa_fw_build_person[];

//////////////////////////////////////////////////////////////////////////////////////////
extern int flag_log, flag_quiet, flag_nodaemon;
extern FILE *LOGFILE;
extern char *config_file_name;
extern const char path_to_config[];
extern const char path_to_log[];
extern char *empty;
extern struct timeval program_start;

extern ConnectionsList Connections;
extern ServicesList Services;
extern UsersList Users;
extern ScheduleList Sched;
extern NetUnitsList Units;
extern PolicyList PolicyL;
extern Connection *cInternal;
extern Service *sMain;
extern Service *sSched;
extern User *uInternal;
extern FIFO Mux_in;
extern int global_return_code;

//////////////////////////////////////////////////////////////////////////////////////////
char *timeU2T(time_t time, char *buf);
time_t timeT2U(char *buf, time_t *time);
char *bytesQ2T(unsigned long long bytes, char *buf);
unsigned long long bytesT2Q(unsigned long long *bytes, char *buf);

void termination(int signal);

//////////////////////////////////////////////////////////////////////////////////////////
unsigned aCommand(Connection *conn, char *cmd);
void cSave(Connection *conn);
void cDebug(Connection *conn, char *param[32], int no_flag);
void cShowConfig(FILE *f, Connection *conn);
void cRotate(Connection *conn,char *param[32]);

//////////////////////////////////////////////////////////////////////////////////////////
// server service functions
void sServerInit(Service *s);
void sServerProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sServerListCfg(Service *s, FILE *f);
//////////////////////////////////////////////////////////////////////////////////////////
// data-source service functions
void sDSInit(Service *s);
void sDSProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sDSListCfg(Service *s, FILE *f);
unsigned sDS_AcctIp(Service *s, struct ip *ip, struct timeval *tv, unsigned long len, unsigned skip_fw_check=0);
unsigned sDS_AcctNetFlow(Service *s, struct ip *ip, struct timeval *tv);

#define  MAX_PKT_SIZE    0xffff

typedef struct pcap_data {
	pcap_t *h;
	struct bpf_program filter;
	struct pcap_pkthdr pkthdr;
	int datalink;
	int offset;
	char *errbuf;
	bpf_u_int32 net;
	bpf_u_int32 mask;
	unsigned char *pkt;
	} pcap_data;

typedef struct ServiceDS_cfg {
	Service *sLookup;
	policy_target_type type;
	unsigned long long total_packets;
	unsigned long seq_id;
	double delay; // cumulative delay accumulated for all packets
	long skewdelay; // measured in microseconds and based on previous value
	int pc[10];
	long bc[10];
	char *src_cmd;
	unsigned divert; //if(divert) we are diverting
	unsigned long s_addr; //address from we accept netflow packets
	unsigned port;
	ds_rule *root;
	oid id;
	class IPTree *IPtree;

#ifdef LINUX
	#ifndef IPTBL_NONE
	struct ipq_handle *ipq;
	#endif
#endif

	struct sockaddr_in ds;
	int socketid;
	struct pcap_data *pcap;
	} ServiceDS_cfg;

void dsMailHookAdd(Service *s);
void dsMailHookRemove(Service *s);
void dsMail(Service *s, struct ip *ip);

//////////////////////////////////////////////////////////////////////////////////////////
// storage service functions
void sStorageInit(Service *s);
void sStorageProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sStorageListCfg(Service *s, FILE *f);
enum storage_type { HASH, MY_SQL, POSTGRES, ORACLE, UNKNOWN=10000 };
enum storage_category { ALL, SUMMARY, RAW };
typedef struct st_unit {
	oid id;
	storage_category cat;
	FIFO *in;
	FIFO *out;
	Service *s;
	st_unit *next,*prev;
	} st_unit;
typedef struct ServiceStorage_cfg {
	storage_type type;
	char *hostname;
	char *username;
	char *password;
	char *dbname; // 03.11.2002, default to 'netams'
	char *socket;
	unsigned port;
	FIFO *in;
	FIFO *out;
	char *query;
	void *fd1;  // for raw
	void *fd2;  // for summary
	} ServiceStorage_cfg;

//////////////////////////////////////////////////////////////////////////////////////////
// processor service functions
void sProcessorProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sProcessorInit(Service *s);
unsigned sProcessorData(Connection *conn, char *param[32]);
unsigned sProcessorRead(Connection *conn, char *param[32]);
unsigned cShowStat(Connection *conn, char *param[32]);
int sProcessorRawRead(Message *msg, long timeout);
void sProcessorListCfg(Service *s, FILE *f);
void cShowProcessor(Connection *conn);
void cShowVersion(Connection *conn);
void sPConstructFillMessage(oid netunit, oid policy, policy_data *pdata);
unsigned cDefault(Connection *conn, char *param[], int no_flag);
unsigned cRestrict(Connection *conn, char *param[], int no_flag);
unsigned cRegister(Connection *conn, char *param[], int no_flag);

enum restrict_type { DROP=1, PASS=0, PASS_LOCAL=2 };
typedef struct ServiceProcessor_cfg {
	struct st_unit *st_root;
	NetUnit *def;
	unsigned delay; // in milliseconds
	unsigned lifetime; // in milliseconds
	restrict_type restrict_local, restrict_all;
	char *access_script;
	} ServiceProcessor_cfg;

extern ServiceProcessor_cfg ProcessorCfg;
//////////////////////////////////////////////////////////////////////////////////////////
// scheduler service classes and functions
class Task {
	public:
		oid id;
		time_t when;
		char *action;
		char *str_interval;
		Task *next;
		int visible;
		Task();
		~Task();
		};

class ScheduleList {
	public:
		Task *root;
		Task *last;
		unsigned num_tasks;
		pthread_mutex_t *lock;
		ScheduleList();
		~ScheduleList();
		void Insert(Task *t);
		void Delete(oid id);
		void Delete(Task *t);
		Task *getTask(char *action);
		Task *getTaskById(oid id);
		void listTasks(Connection *conn);
		void listTasksCfg(FILE *f);
		unsigned getTasks() { return num_tasks; }
		};

unsigned cTask(Connection *conn, char *param[], int no_flag);
unsigned cShowScheduleList(Connection *conn);
time_t getInterval(char *s);

void sSchedulerInit(Service *s);
void *sScheduler(void *ss);
void sSchedulerCancel(void *v);
void *sSchedulerTask(void *v);
//////////////////////////////////////////////////////////////////////////////////////////
// html service functionality
enum html_lang { HTML_LANG_EN, HTML_LANG_RU };
enum cpages_type { CPAGES_ALL, CPAGES_GROUPS, CPAGES_NONE };
typedef struct Html_cfg {
	char *path;
	html_lang lang;
	char *run_period;
	int is_distributed;
	cpages_type cpages;
	char *url;
	int is_htaccess;
	} Html_cfg;
void sHtmlInit(Service *s);
void sHtmlProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sHtmlListCfg(Service *s, FILE *f);
void *sHtml(void *ss);
void sHtmlCancel(void *v);
void cHtml(Connection *conn, char *param[32], int no_flag);
void sHtmlAction(Html_cfg *cfg);
int guess_load(int x);
//////////////////////////////////////////////////////////////////////////////////////////
// monitor virtual service functionality
void sMonitorProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sMonitorInit(Service *s);
void sMonitorCancel(void *v);
void cShowMonitor(Connection *conn);
void sMonitorListCfg(Service *s, FILE *out);
enum monitor_to { MON_FILE, MON_STORAGE, MON_NONE };
typedef struct Monitor_cfg {
	monitor_to to;
	char *name;
	unsigned items;
	unsigned storage_no;
	Service *st;
	char *query;
	void *fd;
	unsigned commit;
	} Monitor_cfg;

void aMonitor(NetUnit *u, struct ip *ip, struct timeval *tv, unsigned long len);
//////////////////////////////////////////////////////////////////////////////////////////
// web login service functionality
#ifndef MAX_WL_UNITS
#define MAX_WL_UNITS 32
#endif
typedef struct WLdata {
	WLdata *next;
	User *user;
	time_t timeout;
	char *timeout_s;
	time_t inact;
	char *inact_s;
	SysPolicy pol;
	oid pol_perm;
	NetUnit *units[MAX_WL_UNITS];
	unsigned totalunits;
	unsigned multi;
	} WLdata;

typedef struct Weblogin_cfg {
	WLdata *WLroot;
	unsigned delay;
} Weblogin_cfg;

void sWLInit(Service *s);
void sWLProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sWLListCfg(Service *s, FILE *f);
void *sWL(void *s);
void sWLCancel(void *);
void sWLOpenUnit(Connection *conn, WLdata *w, int i, const char *user);
void cShowTimeouts(Connection *conn);

//////////////////////////////////////////////////////////////////////////////////////////
// PVM gateway service functionality
typedef struct PVM_cfg {
	//nothing yet
} PVM_cfg;
void sPVMInit(Service *s);
void sPVMProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sPVMListCfg(Service *s, FILE *f);
void *sPVM(void *s);
void sPVMCancel(void *);
//////////////////////////////////////////////////////////////////////////////////////////
// login service functionality
typedef struct sLoginData {
	int c_state; //0=closed, 1=opened
	int d_state; //0=closed, 1=opened
	int strict; //0=allow open from anywhere, 1=open from the same IP
	time_t opened; //time when opened
	time_t inact; // unit-specific inactivity timeout
	time_t abs; // unit-specific absolute timeout
	struct in_addr ip_from;
	struct in_addr last_ip_from;
	struct ether_addr mac_from;
	struct ether_addr last_mac_from;
	char *password;
	} sLoginData;


typedef struct Login_cfg {
	time_t default_inact;
	time_t min_inact;
	time_t max_inact;
	time_t default_abs;
	time_t min_abs;
	time_t max_abs;
	int min_passwd_length;
	int storage;
	int set_user_ip;
	unsigned delay;
	Service *st;
	char *query;
	void *fd;
	pthread_rwlock_t *rwlock; // for multiple threads access same config data
} Login_cfg;

#define S_LOGIN_DEF_default_inact		0
#define S_LOGIN_DEF_default_abs 		0
#define S_LOGIN_DEF_max_inact			12*60*60
#define S_LOGIN_DEF_min_inact			60
#define S_LOGIN_DEF_max_abs				24*12*60*60
#define S_LOGIN_DEF_min_abs				60
#define S_LOGIN_DEF_min_passwd_length	3
#define S_LOGIN_DEF_delay				60

void sLgInit(Service *s);
void sLgProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sLgListCfg(Service *s, FILE *f);
void *sLogin(void *s);
void sLgCancel(void *);
void cShowLogin(Connection *conn, char *param[32]);
//////////////////////////////////////////////////////////////////////////////////////////
void cAccessScriptCall(restrict_type action, NetUnit *u, char *param=NULL);
//////////////////////////////////////////////////////////////////////////////////////////
// quota control service functionality

typedef struct qstat {
	unsigned long long in;
	unsigned long long out;
	unsigned long long sum;
	unsigned long long softin;
	unsigned long long softout;
	unsigned long long softsum;
	} qstat;

typedef struct sQuotaData {
	Policy *policy;
	qstat h, d, w, m, t;
	unsigned int soft_treshold;
	int active;
	int notified;
	int blocked;
	time_t blocked_time;
	oid nso, nho, nro;
	unsigned nss, nhs, nrs;
	} sQuotactlData;


#define S_QUOTA_DEF_arraysize 2
typedef struct Quota_cfg {
	Policy *default_policy;
	unsigned int default_soft_treshold;
	int storage;
	unsigned delay;
	Service *st;
	void *fd;
	char *query;
	
	pthread_rwlock_t *rwlock; // for multiple threads access same config data 

	oid notify_soft[S_QUOTA_DEF_arraysize];
	oid notify_hard[S_QUOTA_DEF_arraysize];
	oid notify_return[S_QUOTA_DEF_arraysize];
} Quota_cfg;

#define S_QUOTA_DEF_soft_treshold 80
#define S_QUOTA_DEF_delay 60
#define S_QUOTA_DEF_notify_soft 1
#define S_QUOTA_DEF_notify_hard 1
#define S_QUOTA_DEF_notify_return 1

void sQuotaInit(Service *s);
void sQuotaProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sQuotaListCfg(Service *s, FILE *f);
void cShowQuota(Connection *conn, char *param[32]);

void sQuInit(Service *s);
void sQuProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sQuListCfg(Service *s, FILE *f);
void *sQuota(void *s);
void sQuCancel(void *);
void cShowQuota(Connection *conn, char *param[32]);
void sQuotaGetValue(int *i, char *param[], qstat *q);
//////////////////////////////////////////////////////////////////////////////////////////
// quotactl control service functionality

typedef struct Qdef {
        char *name;
        Policy *policy;
        qstat h, d, w, m, t;
        qUlist *root;
        Qdef *prev,*next;

        char *sys_policy_name;
        SysPolicy sys_policy;
        oid sys_policy_perm;

        } Qdef;

typedef struct Quotactl_cfg {
        Qdef *root;
        pthread_rwlock_t *rwlock;
        User *soft_reach; User *hard_reach; User *reset_back;
        int user_soft_reach; int user_hard_reach; int user_reset_back;
        unsigned delay; // in milliseconds
        } Quotactl_cfg;

void sQuotactlInit(Service *s);
void sQuotactlProcessCfg(char *param[32], Connection *conn, unsigned no_flag);
void sQuotactlListCfg(Service *s, FILE *f);
void cShowQuotactl(Connection *conn, char *param[32]);

void *sQuotactl(void *s);
void sQuotactlCancel(void *);
void sQuotaSendAlert(Service *s,NetUnit *u, Qdef *q, int dir);
void sQuotactlGetValue(int *i, char *param[], qstat *q);

//////////////////////////////////////////////////////////////////////////////////////////
#endif /* !_NETAMS_H_ */
