/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: policy.c,v 1.24.2.6 2004/05/06 10:35:04 jura Exp $ */

#include "netams.h"

void sPConstructReadMessages(oid netunit, policy_data *pdata);
/////////////////////////////////////////////////////////////////////////
// Policy class
Policy::Policy(policy_type t) {
	id=newOid(OID_POLICY);
	name=NULL;
	type=t;
	next=NULL;
	target_string=NULL;
 	memset(&target, 0, sizeof(policy_target));
	target.unit_id=0; target.unit_name=NULL;
	target.target_type=PT_UNKNOWN;
	target.check_type=PC_UNKNOWN;
	target.file=NULL;
	}

Policy::~Policy() {
	if(name) aFree(name);
	if (target.check_type==PC_FILE)  delete target.file;
	aFree(target_string);
	}

void Policy::setName(char *n){
	if (name) aFree(name);
	name=set_string(n);
}

unsigned Policy::setTarget(char **tgt, int *i){
	if (!strncasecmp("ip", tgt[*i+1], 2)) {
		target_string=(char*)aMalloc(3); 
		strcpy(target_string, "ip");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=255; // ip
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		return 1;
	}
	else if (!strncasecmp("icmp", tgt[*i+1], 4)) {
		target_string=(char*)aMalloc(5); 
		strcpy(target_string, "icmp");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=1; // icmp
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		return 1;
	}
	else if (!strncasecmp("tcp-http", tgt[*i+1], 8)) {
		target_string=(char*)aMalloc(9); 
		strcpy(target_string, "tcp-http");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=IPPROTO_TCP; // tcp
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		target.src_ports[0]=80;		target.dst_ports[0]=80;
		target.src_ports[1]=8080;	target.dst_ports[1]=8080;
		target.src_ports[2]=81;		target.dst_ports[2]=81;
		target.src_ports[3]=3128;	target.dst_ports[3]=3128;
		target.src_ports[4]=443;	target.dst_ports[4]=443;
		target.src_ports[5]=0;		target.dst_ports[5]=0;
		return 1;
	}
	else if (!strncasecmp("tcp-ports", tgt[*i+1], 9) || !strncasecmp("udp-ports", tgt[*i+1], 9)) {	// target tcp-ports s25 22 ...
		target_string=(char*)aMalloc(11); 
		if (!strncasecmp("tcp-ports", tgt[*i+1], 9)) {
			strcpy(target_string, "tcp-ports");
			target.proto=IPPROTO_TCP; 
		}
		else {
			strcpy(target_string, "udp-ports");
			target.proto=IPPROTO_UDP; 
		}
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		int j=0, ports;
		while (tgt[j+*i+2]!=empty && j<IP_FW_MAX_PORTS) {
			if (tgt[j+*i+2][0]=='s' || tgt[j+*i+2][0]=='d' || tgt[j+*i+2][0]=='b') ports=atoi(tgt[j+*i+2]+1);
			else ports=atoi(tgt[j+*i+2]);
			if (!ports) return 0;
			switch (tgt[j+*i+2][0]) {
				case 's':
					target.src_ports[j]=ports;
					target.dst_ports[j]=0;
					break;
				case 'd':
					target.src_ports[j]=0;
					target.dst_ports[j]=ports;
					break;
				default:
					target.src_ports[j]=ports;
					target.dst_ports[j]=ports;
					break;
			}
			j++;
		}
		(*i)+=j;
		return 1;
	}
	else if (!strncasecmp("nbt", tgt[*i+1], 8)) {  // 445 137 138 139 
		target_string=(char*)aMalloc(4); 
		strcpy(target_string, "nbt");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=IPPROTO_TCP; // sorry but only TCP is implemented
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		target.src_ports[0]=445;		target.dst_ports[0]=445;
		target.src_ports[1]=137;	target.dst_ports[1]=137;
		target.src_ports[2]=138;		target.dst_ports[2]=138;
		target.src_ports[3]=139;	target.dst_ports[3]=139;
		target.src_ports[4]=0;	target.dst_ports[4]=0;
		return 1;
	}
	else if (!strncasecmp("tcp", tgt[*i+1], 3)) {
		target_string=(char*)aMalloc(4); 
		strcpy(target_string, "tcp");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=IPPROTO_TCP; // tcp
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
//		if (!strcasecmp("src-port", tgt[*i+2])) setTargetPorts(TGT_SRC_PORT, &target, tgt, *i); 
		return 1;
	}
	else if (!strncasecmp("udp", tgt[*i+1], 3)) {
		target_string=(char*)aMalloc(4); 
		strcpy(target_string, "udp");
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_IP;
		target.proto=IPPROTO_UDP; // udp
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		return 1;
	}
	else if (!strncasecmp("units", tgt[*i+1], 5)) {	// syntax is follows: target units {oid 12345|name AAAA}
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_UNIT;
		target.proto=0; // ip
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;
		
		if (!strcasecmp("oid", tgt[*i+2])) {
			target.unit_id=strtol(tgt[*i+3], NULL, 16);
		}
		else if (!strcasecmp("name", tgt[*i+2])) {
			target.unit_name=set_string(tgt[*i+3]);
		}
		else return 0;

		target_string=(char*)aMalloc(strlen("units")+strlen(tgt[*i+2])+strlen(tgt[*i+3])+3); 
		sprintf(target_string, "%s %s %s", "units", tgt[*i+2], tgt[*i+3]);
		(*i)+=2;
		return 1;
	}
	else if (!strncasecmp("file", tgt[*i+1], 4)) {
		target.file = new PrefixFile();
		target.target_type=PT_IP_TRAFFIC;
		target.check_type=PC_FILE;
		target.proto=0; // ip
		target.src.s_addr=target.dst.s_addr=INADDR_ANY;
		target.src_mask.s_addr=target.dst_mask.s_addr=INADDR_ANY;

		if (tgt[*i+2]!=empty) {
			target.file->Load(tgt[*i+2]);
			target_string=(char*)aMalloc(strlen("file")+strlen(tgt[*i+2])+2); 
			sprintf(target_string, "%s %s", "file", tgt[*i+2]);
			(*i)+=1;
			return 1;
		}
		else return 0;
	}
	else if (!strncasecmp("mailstat", tgt[*i+1], 8)) {
		target_string=(char*)aMalloc(9); 
		strcpy(target_string, "mailstat");
		target.target_type=PT_MAIL_STAT;
		target.check_type=PC_IP;
		return 1;
	}
	else return 0;
}

char *Policy::getTarget(char *buf){
	if (buf && target_string) {
		sprintf(buf, "%s ", target_string);
		for (int i=0; i<IP_FW_MAX_PORTS && (target.src_ports[i] || target.dst_ports[i]); i++) {
			if (target.src_ports[i]==target.dst_ports[i]) sprintf(buf+strlen(buf), "%d ", target.src_ports[i]);
			else if (target.src_ports[i]) sprintf(buf+strlen(buf), "s%d ", target.src_ports[i]);
			else if (target.dst_ports[i]) sprintf(buf+strlen(buf), "d%d ", target.dst_ports[i]);
			} 
		return buf;
		}
	else if (target_string) return target_string;
	else return "";
	}

unsigned short Policy::Check(struct ip *ip, match mf){
	unsigned short res=0;

	if (target.target_type==PT_IP_TRAFFIC) {

		switch(target.check_type){
			case PC_IP: {
				
				aDebug(DEBUG_DS_IP, " check: IP %s target: %s\n", name, target_string);

/* it's not work actually
				if (target.src.s_addr) {
					if (target.src.s_addr!=ip->ip_src.s_addr) ms++;
					}
				if (target.dst.s_addr) {
					if (target.dst.s_addr!=ip->ip_dst.s_addr) md++;
					}
*/
				if(target.proto && (target.proto==ip->ip_p || target.proto==255))  
					if ( (target.proto==IPPROTO_UDP || target.proto==IPPROTO_TCP) && (target.src_ports[0]!=0 || target.dst_ports[0]!=0)) {
						struct tcphdr *th;
                				th=(struct tcphdr *)((unsigned char *)ip + ip->ip_hl*4);
						for (int i=0; target.src_ports[i]; i++) 
							if (ntohs(th->th_sport)==target.src_ports[i]) {
								res=1; 
								break; //we meet condition - no need to continue  
								       // policy matches packet anyway	
							}
						for (int i=0; target.dst_ports[i]; i++) 
							if (ntohs(th->th_dport)==target.dst_ports[i]) {
								res=1;
								break;
							}
					} else res=1;
			}

			case PC_UNIT: {

				if (target.unit_id || target.unit_name) {
					NetUnit *ut=NULL;
					//there possible to organize more quick but it's rare case
					if (target.unit_id) ut=Units.getUnitById(target.unit_id);
					else if (target.unit_name) ut=Units.getUnit(target.unit_name);
			
					if (!ut) { 
						aDebug(DEBUG_DS_IP, "   IP 'units' policy but '%s' not exist\n", target_string);
						return 0;
						}
					match utmf=ut->Check(ip);
					if(utmf!=MATCH_NONE && (mf|utmf==MATCH_BOTH)) res=1;
					}
				break;
				}
			case PC_FILE: {
				if (target.file->list==NULL) {
					aLog(D_WARN, "prefix check but file is not loaded\n"); 
					return MATCH_NONE;
				}
				switch(mf) {
					case MATCH_SRC:
						res=target.file->Check(ip->ip_dst.s_addr);
						break;
					case MATCH_DST:
						res=target.file->Check(ip->ip_src.s_addr);
						break;
					case MATCH_BOTH:
						res=target.file->Check(ip->ip_src.s_addr);
						if(!res) res=target.file->Check(ip->ip_dst.s_addr);
						break;
					default:
						aLog(D_WARN, "Matched unit with mf=MATCH_NONE\n");
				}
				break;
			}
			default : break;
		} //switch 
			
			#ifdef DEBUG
                        struct tcphdr *th;
			char buf1[32], buf2[32];
                	strcpy(buf1, inet_ntoa(ip->ip_src)); strcpy(buf2, inet_ntoa(ip->ip_dst));
                        th=(struct tcphdr *)((unsigned char *)ip + ip->ip_hl*4);
			aDebug(DEBUG_DS_IP, "Policy check: IP %d s:%s:%u d:%s:%u %d mf=%u res=%u\n", ip->ip_p, buf1, ntohs(th->th_sport), buf2, ntohs(th->th_dport), ntohs(ip->ip_len), mf, res);
			#endif
	} // if

	return res;
}

/////////////////////////////////////////////////////////////////////////
// PolicyList class
PolicyList::PolicyList(){
	root=last=NULL;
	num_policies=0;
	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(rwlock, NULL);
	}

PolicyList::~PolicyList(){
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
	}

void PolicyList::Insert(Policy *s){
	pthread_rwlock_wrlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)
		if (d==s) { 
			pthread_rwlock_unlock(rwlock);
			return;
			}
	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	num_policies++;
	pthread_rwlock_unlock(rwlock);
	}

void PolicyList::Delete(Policy *s){
	pthread_rwlock_wrlock(rwlock);
	Policy *d, *p;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;

			num_policies--;
			Units.DeletePolicyElsewhere(s);
			break;
			}
		p=d; 
	 	}
	pthread_rwlock_unlock(rwlock);
	}

void PolicyList::DeleteAll(){
	pthread_rwlock_wrlock(rwlock);
	Policy *d,*p;
	for (d=root; d!=NULL; d=p) {
			p=d->next;
			delete d;
	 	}
	pthread_rwlock_unlock(rwlock);
	}

void PolicyList::Delete(oid id){
	Policy *p=getPolicyById(id);
	if (p) Delete(p);
	}

Policy* PolicyList::getPolicy(char *name){
	pthread_rwlock_rdlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (!strcmp(d->name, name)) {
			pthread_rwlock_unlock(rwlock);
			return d;
			}
	 	}
	pthread_rwlock_unlock(rwlock);
	return NULL;
	}

Policy* PolicyList::getPolicyById(oid id){
	pthread_rwlock_rdlock(rwlock);
	Policy *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->id==id) {
			pthread_rwlock_unlock(rwlock);
			return d;
			}
	 	}
	pthread_rwlock_unlock(rwlock);
	return NULL;
	}

/////////////////////////////////////////////////////////////////////////
unsigned cPolicy(Connection *conn, char *param[], int no_flag){
	Policy *ut;
	int i=2;
	if (!strcmp(param[2], "name")) ut=PolicyL.getPolicy(param[3]);
	else if (!strcmp(param[2], "oid")) ut=PolicyL.getPolicyById(strtol(param[3], NULL, 16));
	else { aParse(conn, "policy %s name/oid unspecified\n", param[1]); return PARSE_OK; }

//-----------------------------------------------------------------------
	if (!strcasecmp(param[1], "acct")) {
		
		if (!ut && !no_flag) { 
			ut = new Policy(POLICY_ACCT);
			PolicyL.Insert(ut); 
			aParse(conn, "policy accounting %06X created\n", ut->id); 
			}
		
		if (ut && no_flag){
			aParse(conn, "policy accounting %06X deleted\n", ut->id);
			PolicyL.Delete(ut);
			delete ut;
			return PARSE_OK;
			}

		while (param[i]!=empty) {
	 		if (strcasecmp(param[i], "name")==0) { 
				ut->setName(param[i+1]);
				aParse(conn, "policy %06X name set: %s\n", ut->id, ut->name);

			} else if (strcasecmp(param[i], "oid")==0) { 
				ut->id=strtol(param[i+1], NULL, 16);
				aParse(conn, "policy %06X oid set\n", ut->id);

			} else if (strcasecmp(param[i], "target")==0) {
				int j=ut->setTarget(param, &i); 
				if (j) aParse(conn, "policy %06X target set: %s\n", ut->id, ut->getTarget(NULL));
				else aParse(conn, "policy %06X target invalid\n", ut->id); 

			} else if (strcasecmp(param[i], "do")==0) {
				if (ut->target.check_type==PC_FILE) {
			   		aParse(conn, "policy %06X file do: '%s'\n", ut->id, param[i+1]);
			   		ut->target.file->Do(conn, param[i+1], param[i+2]);
			   		}
			   	else aParse(conn, "policy %06X action unsupported!\n", ut->id);	

			} else aParse(conn, "policy %06X command unknown: %s\n", ut->id, param[i]); 

			i=i+2;
			}
		return PARSE_OK;
		}
//-----------------------------------------------------------------------
	else if (!strcasecmp(param[1], "fw")) {
		
		if (!ut && !no_flag) { 
			ut = new Policy(POLICY_FW);
			PolicyL.Insert(ut); 
			aParse(conn, "policy firewalling %06X created\n", ut->id); 
			}
		
		if (ut && no_flag){
			aParse(conn, "policy firewalling %06X deleted\n", ut->id);
			PolicyL.Delete(ut);
			delete ut;
			return PARSE_OK;
			}

		while (param[i]!=empty) {
	 		if (strcasecmp(param[i], "name")==0) { 
				ut->setName(param[i+1]);
				aParse(conn, "policy %06X name set: %s\n", ut->id, ut->name);

			} else if (strcasecmp(param[i], "oid")==0) { 
				ut->id=strtol(param[i+1], NULL, 16);
				aParse(conn, "policy %06X oid set\n", ut->id);

			} else if (strcasecmp(param[i], "target")==0) {
				int j=ut->setTarget(param, &i); 
				if (j) aParse(conn, "policy %06X target set: %s\n", ut->id, ut->getTarget(NULL));
				else aParse(conn, "policy %06X target invalid\n", ut->id); 

			} else if (strcasecmp(param[i], "do")==0) {
				if (ut->target.check_type==PC_FILE) {
			   		aParse(conn, "policy %06X file do: '%s'\n", ut->id, param[i+1]);
			   		ut->target.file->Do(conn, param[i+1], param[i+2]);
			   		}
			   	else aParse(conn, "policy %06X action unsupported!\n", ut->id);	

			} else aParse(conn, "policy %06X command unknown\n", ut->id); 

			i=i+2;
			}
		return PARSE_OK;
		}
//-----------------------------------------------------------------------
	else aParse(conn, "policy type unknown\n");

	return PARSE_OK;
	}

/////////////////////////////////////////////////////////////////////////
unsigned cShowPolicy(Connection *conn){
	pthread_rwlock_rdlock(PolicyL.rwlock);
	Policy *d;
	char buf[64];

	fprintf(conn->stream_w, "%4s | %6s | %15s | %-20s\n", "TYPE", "OID", "NAME", "PARAMS");
	for (d=PolicyL.root; d!=NULL; d=d->next)	{

		if (d->type==POLICY_ACCT) fprintf(conn->stream_w, "%4s | ", "acct");
		else if (d->type==POLICY_FW) fprintf(conn->stream_w, "%4s | ", "fw");
		else fprintf(conn->stream_w, "%4s | ", "<\?\?>");

		fprintf(conn->stream_w, "%06X | %15s | ", d->id, d->name?d->name:"<\?\?>");
		if (d->target.target_type==PT_IP_TRAFFIC) fprintf(conn->stream_w, "target: %-12s", d->getTarget(buf));

		fprintf(conn->stream_w, "\n");
		}

	pthread_rwlock_unlock(PolicyL.rwlock);
	return PARSE_OK;
	}

/////////////////////////////////////////////////////////////////////////
// PdList class
PdList::PdList(){
	root=last=NULL;
	num_policies=0;
	lock=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
	}

PdList::~PdList(){
	pthread_mutex_destroy(lock);
	aFree(lock);
	}

policy_data *PdList::Add(Policy *p, unsigned inv, unsigned brk, unsigned place) {
	pthread_mutex_lock(lock);
	policy_data *d;	
	policy_data *after=NULL;
	unsigned curr_place=0;
	if (!place) place=999;
	for (d=root; d!=NULL; d=d->next) {
		if (d->policy==p) { 
			pthread_mutex_unlock(lock);
			return NULL;
			}
		curr_place++;
		if (curr_place<place) after=d;
		}

	policy_data *np = new policy_data;
	memset(np, 0, sizeof(policy_data));
	np->inv=inv;
	np->brk=brk;
	np->next=NULL;
	np->timestamp=0;		   
	np->policy=p;
	np->check=np->match=0; 
	
	struct time_counters tc;
	PrepareTimeCounters(&tc);
	FillTimeCounters(np,&tc);
	np->t.from=time(NULL);

	if (root==NULL) { root=np; last=np; }  // empty list
	else if (after==NULL) { np->next=root; root=np; } // first
	else if (after==last) { last->next=np; last=np; } // last
	else { np->next=after->next; after->next=np; }
	
	num_policies++;
	pthread_mutex_unlock(lock);
	return np;
	}

void PdList::Delete(Policy *s) {
	pthread_mutex_lock(lock);
	policy_data *d, *p;
	for (d=root; d!=NULL; d=d->next)	{
		if (d->policy==s) {
			if (d==root && d==last ) root=last=NULL;
			else if (d==root) root=d->next;
			else if (d==last) { last=p; last->next=NULL; }
			else p->next=d->next;

			num_policies--;
			break;
			}
		p=d; 
	 	}
	pthread_mutex_unlock(lock);
	}

void PdList::List(Connection *conn){
	pthread_mutex_lock(lock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next)	{
		fprintf(conn->stream_w, "%s(%06X) ", d->policy->name, d->policy->id);
	 	}
	fprintf(conn->stream_w, "\n");
	pthread_mutex_unlock(lock);
	}

void PdList::ListForCfg(FILE *f){
	pthread_mutex_lock(lock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next)	{
		fprintf(f, "%s ", d->policy->name);
	 	}
	fprintf(f, "\n");
	pthread_mutex_unlock(lock);
	}

void PdList::SetForUnit(policy_type pt, NetUnit *u){
	pthread_mutex_lock(lock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next)	{
		if (pt==POLICY_ACCT) {
		 	u->ap.Add(d->policy); 
			aParse(cInternal, "unit %06X acct policy %s added\n", u->id, d->policy->name); 
			sPConstructReadMessages(u->id, u->ap.last);	
		}
		else if (pt==POLICY_FW) {
		 	u->fp.Add(d->policy); 
			aParse(cInternal, "unit %06X fw policy %s added\n", u->id, d->policy->name); 
			sPConstructReadMessages(u->id, u->fp.last);	
		}
	}
	pthread_mutex_unlock(lock);
}

policy_data *PdList::Get(Policy *p){
	pthread_mutex_lock(lock);
	policy_data *d, *r=NULL;
	for (d=root; d!=NULL; d=d->next) if (d->policy==p) r=d;
	pthread_mutex_unlock(lock);
	return r;
	}

time_t PdList::LastUsed(){
	pthread_mutex_lock(lock);
	policy_data *d;
	time_t l=0;
	for (d=root; d!=NULL; d=d->next) 
		if (d->timestamp > l) l=d->timestamp;
	pthread_mutex_unlock(lock);
	return l;
	}
void PdList::ClearLastUsed(){
	pthread_mutex_lock(lock);
	policy_data *d;
	for (d=root; d!=NULL; d=d->next) d->timestamp=0;
	pthread_mutex_unlock(lock);
	}
/////////////////////////////////////////////////////////////////////////
void PolicyDataUpdate(match m, policy_target_type type, policy_data *p, unsigned long len, struct timeval *tv){
	time_t t = tv->tv_sec;

	switch (type) {
		case PT_IP_TRAFFIC:
			p->timestamp=t;
			if (p->flow.from==0) p->flow.from=t;
			p->t.to=p->m.to=p->w.to=p->d.to=p->h.to=p->flow.to=t;

			if (m==MATCH_DST || m==MATCH_BOTH) { 
				p->t.in+=len;
				p->m.in+=len;			 
				p->w.in+=len;
				p->d.in+=len;
				p->h.in+=len;
				p->flow.in+=len;
				}

			if (m==MATCH_SRC || m==MATCH_BOTH) { 
				p->t.out+=len;
				p->m.out+=len;
				p->w.out+=len;
				p->d.out+=len;
				p->h.out+=len;
				p->flow.out+=len;
				}
		
			break;

		default:
			// we cannot account this non-ip data yet!
			break;
		}
	}
/////////////////////////////////////////////////////////////////////////
void PolicyAdd(NetUnit *u, int *i, policy_type po, Connection *conn, char *param[32], int no_flag){
	Policy *p;
	char *c_param;
	unsigned inv, brk, place=0;
	while (param[(*i)+1]!=empty) {
		inv=0; brk=0;
		(*i)++; 
		c_param=param[*i];
		if (c_param[0]=='!') { inv=1; c_param++; }
		if (c_param[0]=='%') { brk=1; c_param++; }
		if (c_param[0]=='!') { inv=1; c_param++; }
		
		if (!place) { 
			if ((place=atoi(c_param))!=0) continue; else place=0;
			}
		
		p=PolicyL.getPolicy(c_param);
		policy_data *pdc;
		if (p)  {
			if (po==POLICY_ACCT) { 
				if (no_flag) {
					u->ap.Delete(p);
					aParse(conn, "unit %06X acct %spolicy %s removed\n", u->id, inv?"inverted ":"", c_param); 
				}
				else {
					if ((pdc=u->ap.Add(p, inv, brk, place))) { 
						aParse(conn, "unit %06X acct %spolicy %s added <%u>\n", u->id, inv?"inverted ":"", c_param, place); 
						// we should construct a request to fill the policy_data structure
						sPConstructReadMessages(u->id, pdc);
					}
				}
			}
			else if (po==POLICY_FW) { 
				if (no_flag) {
					u->fp.Delete(p);
					aParse(conn, "unit %06X fw %spolicy %s removed\n", u->id, inv?"inverted ":"", c_param); 
				}
				else {
					if ((pdc=u->fp.Add(p, inv, brk, place))){
						aParse(conn, "unit %06X fw %s%spolicy %s added <%u>\n", u->id, inv?"inverted ":"", brk?"break ":"", c_param, place); 
						// we should construct a request to fill the policy_data structure
						sPConstructReadMessages(u->id, pdc);
					}
				}
			}
		} // policy is defined
		else { 
			aParse(conn, "unit %06X policy '%s' undefined\n", u->id, c_param);
			(*i)-=2;
			return;
		}
		place=0;

		}
	}

/////////////////////////////////////////////////////////////////////////

char *SysPolicy_name[32];
void aSysPolicyInit(){
	SysPolicy_name[SP_NONE]="";
	SysPolicy_name[SP_ALLOW]="sys-allow";
	SysPolicy_name[SP_DENY_NONE]="sys-deny";
	SysPolicy_name[SP_DENY_MONEY]="sys-deny-money";
	SysPolicy_name[SP_DENY_QUOTA]="sys-deny-quota";
	SysPolicy_name[SP_DENY_MACVIOL]="sys-deny-macviol";
	SysPolicy_name[SP_DENY_AUTH]="sys-deny-auth";
	SysPolicy_name[SP_ALLOW_AUTH]="sys-allow-auth";
	SysPolicy_name[SP_LOC_NONE]="sys-local";
	SysPolicy_name[SP_LOC_MONEY]="sys-local-money";
	SysPolicy_name[SP_LOC_QUOTA]="sys-local-quota";
	SysPolicy_name[SP_LOC_MACVIOL]="sys-local-macviol";
	SysPolicy_name[SP_LOC_AUTH]="sys-local-auth";
	SysPolicy_name[SP_LX_NONE]="";
	SysPolicy_name[SP_LX_MONEY]="money";
	SysPolicy_name[SP_LX_QUOTA]="quota";
	SysPolicy_name[SP_LX_MACVIOL]="macviol";
	SysPolicy_name[SP_LX_AUTH]="auth";
	SysPolicy_name[SP_LX_AUTH+1]=NULL;
}

void SetSysPolicy(char *p, NetUnit *u, Connection *conn){
	char *p1, *p2;
	char *s1=NULL, *s2=NULL;
	oid id;
	SysPolicy prev=u->sys_policy;
	p1=p+4;
	if (!p1) { aParse(conn, "incorrect NetUnit sys-* value\n"); return; }

	p2=strchr(p1, '-'); 
	if (p2)  {
		p2++;
		s1=(char*)aMalloc(p2-p1+2); strncpy(s1, p1, p2-p1-1);
		s2=(char*)aMalloc(strlen(p2)+2); strncpy(s2, p2, strlen(p2));
	}
	else {
		s1=(char*)aMalloc(strlen(p1)+2); strncpy(s1, p1, strlen(p1));
		s2=(char*)aMalloc(2); strncpy(s2, "", 1);
	}

	//aParse(conn, "SysPolicy got '%s'-'%s'\n", s1, s2);
	id=strtol(s1, NULL, 16);

	if (strcasecmp(s1, "allow")==0) {
		if (strcasecmp(s2, "")==0) { u->sys_policy=SP_NONE; u->sys_policy_name=SysPolicy_name[SP_ALLOW]; }
		else if (strcasecmp(s2, "auth")==0) { u->sys_policy=SP_ALLOW_AUTH; u->sys_policy_name=SysPolicy_name[SP_ALLOW_AUTH]; }
		else aParse(conn, "System policy for %06X allow: is incorrect\n", u->id);
	}
	else if (strcasecmp(s1, "deny")==0) {
		if (strcasecmp(s2, "")==0) { u->sys_policy=SP_DENY_NONE; u->sys_policy_name=SysPolicy_name[SP_DENY_NONE]; }
		else if (strcasecmp(s2, "money")==0) { u->sys_policy=SP_DENY_MONEY; u->sys_policy_name=SysPolicy_name[SP_DENY_MONEY]; }
		else if (strcasecmp(s2, "quota")==0) { u->sys_policy=SP_DENY_QUOTA; u->sys_policy_name=SysPolicy_name[SP_DENY_QUOTA]; }
		else if (strcasecmp(s2, "macviol")==0) { u->sys_policy=SP_DENY_MACVIOL; u->sys_policy_name=SysPolicy_name[SP_DENY_MACVIOL]; }
		else if (strcasecmp(s2, "auth")==0) { u->sys_policy=SP_DENY_AUTH; u->sys_policy_name=SysPolicy_name[SP_DENY_AUTH]; }
		else aParse(conn, "System policy for %06X deny: is incorrect\n", u->id);
	}
	else if (strcasecmp(s1, "local")==0) {
		if (strcasecmp(s2, "")==0) { u->sys_policy=SP_LOC_NONE; u->sys_policy_name=SysPolicy_name[SP_LOC_NONE]; }
		else if (strcasecmp(s2, "money")==0) { u->sys_policy=SP_LOC_MONEY; u->sys_policy_name=SysPolicy_name[SP_LOC_MONEY]; }
		else if (strcasecmp(s2, "quota")==0) { u->sys_policy=SP_LOC_QUOTA; u->sys_policy_name=SysPolicy_name[SP_LOC_QUOTA]; }
		else if (strcasecmp(s2, "macviol")==0) { u->sys_policy=SP_LOC_MACVIOL; u->sys_policy_name=SysPolicy_name[SP_LOC_MACVIOL]; }
		else if (strcasecmp(s2, "auth")==0) { u->sys_policy=SP_LOC_AUTH; u->sys_policy_name=SysPolicy_name[SP_LOC_AUTH]; }
		else aParse(conn, "System policy for %06X local: is incorrect\n", u->id);
	}
	else if (id) {
		u->sys_policy_perm=id;
		if (strcasecmp(s2, "")==0) { u->sys_policy=SP_LOC_NONE; u->sys_policy_name=SysPolicy_name[SP_LX_NONE]; }
		else if (strcasecmp(s2, "money")==0) { u->sys_policy=SP_LOC_MONEY; u->sys_policy_name=SysPolicy_name[SP_LX_MONEY]; }
		else if (strcasecmp(s2, "quota")==0) { u->sys_policy=SP_LOC_QUOTA; u->sys_policy_name=SysPolicy_name[SP_LX_QUOTA]; }
		else if (strcasecmp(s2, "macviol")==0) { u->sys_policy=SP_LOC_MACVIOL; u->sys_policy_name=SysPolicy_name[SP_LX_MACVIOL]; }
		else if (strcasecmp(s2, "auth")==0) { u->sys_policy=SP_LOC_AUTH; u->sys_policy_name=SysPolicy_name[SP_LX_AUTH]; }
		else aParse(conn, "System policy for %06X %06X: is incorrect\n", u->id, id);
	}
	else aParse(conn, "System policy for %06X *: is incorrect\n", u->id);
				
	if (prev!=u->sys_policy) {
		if (s2[0]) aParse(conn, "System policy for %06X set to '%s-%s'\n", u->id, s1, s2);
		else aParse(conn, "System policy for %06X set to '%s'\n", u->id, s1);
	}

	aFree(s1); aFree(s2); 
}
/////////////////////////////////////////////////////////////////////////
void sPConstructReadMessages(oid netunit, policy_data *pdata) {
	char prefix[5] = { 'H', 'D', 'W', 'M', 'T'};
	Message *msg;
	time_t t=time(NULL);

	for(unsigned i=0;i<5;i++) {
		msg= new Message();
		msg->pdata=pdata;
		msg->unblock=NULL;
		msg->type=READ;
                msg->ts=t;
		msg->ap=pdata->policy->id;
		msg->netunit=netunit;
		msg->id=0;
		msg->prefix=prefix[i];
		Mux_in.Push(msg);
	}
	aDebug(DEBUG_PROC_MUX, "P<-P H unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->h.from);
	aDebug(DEBUG_PROC_MUX, "P<-P D unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->d.from);
	aDebug(DEBUG_PROC_MUX, "P<-P W unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->w.from);
	aDebug(DEBUG_PROC_MUX, "P<-P M unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->m.from);
	aDebug(DEBUG_PROC_MUX, "P<-P T unit:%06X acct:%06X from:%lu\n", msg->netunit, msg->ap, msg->pdata->t.from);

	return;
}
/////////////////////////////////////////////////////////////////////////
