/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_login.c,v 1.22.2.15 2004/08/17 06:10:14 jura Exp $ */

#include "netams.h"

#ifdef FREEBSD
	#include <sys/sysctl.h>
	#include <net/if_dl.h>
	#include <net/if_types.h>
	#include <net/route.h>
	#include <netinet/if_ether.h>
#endif

void sLgCheckDb(int id, Login_cfg *c);
void sLgObtainDbData(NetUnit *u, Login_cfg *c);
void sLgSetCfg(char *param[32], Connection *conn, Login_cfg *c);
void sLgLoginCfg(char *param[32], Connection *conn, int no_flag, Login_cfg *c);
void sLgObtainMac(struct in_addr *u, struct ether_addr *current_mac);
void sLgRestoreLogin(Login_cfg *cfg);

#define LG_LOGIN        0x01
#define LG_LOGOUT       0x02
#define LG_FORCE        0x04
#define LG_SET_USER_IP  0x08

void sLgLogin(Connection *conn, NetUnit *u, struct in_addr *ip, Login_cfg *cfg, u_char flag);
//////////////////////////////////////////////////////////////////////////////////////////
void sLgInit(Service *s){
	Login_cfg *cfg = (Login_cfg*)aMalloc(sizeof(Login_cfg));
	cfg->default_inact=S_LOGIN_DEF_default_inact;
	cfg->default_abs=S_LOGIN_DEF_default_abs;
	cfg->max_inact=S_LOGIN_DEF_max_inact;
	cfg->min_inact=S_LOGIN_DEF_min_inact;
	cfg->max_abs=S_LOGIN_DEF_max_abs;
	cfg->min_abs=S_LOGIN_DEF_min_abs;
	cfg->min_passwd_length=S_LOGIN_DEF_min_passwd_length;
	cfg->delay=S_LOGIN_DEF_delay;
	cfg->storage=0;
	cfg->set_user_ip=0;
	cfg->fd=NULL;
	cfg->st=NULL;
	cfg->query=(char*)aMalloc(256);
	cfg->rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
        pthread_rwlock_init(cfg->rwlock, NULL);

	s->cfg=cfg;
	pthread_create(&(s->t_id), NULL, &sLogin, s);

}

//////////////////////////////////////////////////////////////////////////////////////////
void sLgProcessCfg(char *param[32], Connection *conn, unsigned no_flag){

	Service *s=NULL;
	if(!(s=Services.getServiceByName("login",s))) return;

	Login_cfg *cfg=(Login_cfg*)s->cfg;
	time_t t;

	if (!strcasecmp(param[0], "")) {
		}

	else if (!strcasecmp(param[0], "default-inact")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->default_inact=t;
		aParse(conn, "default inactivity timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "default-abs")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->default_abs=t;
		aParse(conn, "default absolute timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "max-inact")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->max_inact=t;
		aParse(conn, "maximum inactivity timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "min-inact")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->min_inact=t;
		aParse(conn, "minimum inactivity timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "max-abs")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->max_abs=t;
		aParse(conn, "maximum absolute timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "min-abs")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->min_abs=t;
		aParse(conn, "minimum absolute timeout set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "min-passwd-length")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->min_passwd_length=t;
		aParse(conn, "minimum password length set to %ld symbols\n", t);
		}

	else if (!strcasecmp(param[0], "set-user-ip")) {
		if (!strcasecmp(param[1], "yes") || strtol(param[1], NULL, 10)==1) {
			aParse(conn, "unit type user ip address will be set to current ip\n");
			cfg->set_user_ip=1;
			}
		else {
			aParse(conn, "unit type user ip address will be NOT set\n");
			cfg->set_user_ip=0;
			}
		}

	else if (!strcasecmp(param[0], "delay")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->delay=t;
		aParse(conn, "login check delay set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "storage")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->storage=(int)t;
		aParse(conn, "storage number set to %d\n", (int)t);
		}

	else if (!strcasecmp(param[0], "set")) sLgSetCfg(param, conn, cfg);

	else if (!strcasecmp(param[0], "login")) sLgLoginCfg(param, conn, no_flag, cfg);

	else if (!strcasecmp(param[0], "logout")) sLgLoginCfg(param, conn, 1, cfg);

	else aParse(conn, "unknown login command: %s\n", param[0]);
	
}

//////////////////////////////////////////////////////////////////////////////////////////
void sLgListCfg(Service *s, FILE *f){

	Login_cfg *cfg=(Login_cfg*)s->cfg;
	
	if (!cfg->rwlock) return;
        pthread_rwlock_rdlock(cfg->rwlock);

	fprintf(f, "service login %d\n", s->instance);
	if (cfg->default_inact!=S_LOGIN_DEF_default_inact) fprintf(f, "default-inact %ld\n", cfg->default_inact);
	if (cfg->default_abs!=S_LOGIN_DEF_default_abs) fprintf(f, "default-abs %ld\n", cfg->default_abs);
	if (cfg->max_inact!=S_LOGIN_DEF_max_inact) fprintf(f, "max-inact %ld\n", cfg->max_inact);
	if (cfg->min_inact!=S_LOGIN_DEF_min_inact) fprintf(f, "min-inact %ld\n", cfg->min_inact);
	if (cfg->max_abs!=S_LOGIN_DEF_max_abs) fprintf(f, "max-abs %ld\n", cfg->max_abs);
	if (cfg->min_abs!=S_LOGIN_DEF_min_abs) fprintf(f, "min-abs %ld\n", cfg->min_abs);
	if (cfg->min_passwd_length!=S_LOGIN_DEF_min_passwd_length) fprintf(f, "min-passwd-length %d\n", cfg->min_passwd_length);
	if (cfg->delay!=S_LOGIN_DEF_delay) fprintf(f, "delay %d\n", cfg->delay);
	fprintf(f, "storage %d\n", cfg->storage);
	if (cfg->set_user_ip) fprintf(f, "set-user-ip yes\n");

	fprintf(f, "\n");
	
	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void *sLogin(void *ss){
	Service *s=(Service*)ss;
	Login_cfg *cfg=(Login_cfg*)s->cfg;
	NetUnit *u;
	time_t now;

	aLog(D_INFO, "service login:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sLgCancel, s);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	s->t_id=pthread_self();

	// now we should sleep before starting service
	s->Sleep();

	cfg->st = Services.getService("storage", cfg->storage);
    	if (cfg->st==NULL) { 
		aLog(D_WARN, "login service requires storage %d to be up, skipping\n", cfg->storage);  
		goto S_LOGIN_CLEANUP_POP;
	}

	while(!(cfg->fd=stOpenSql(cfg->st,ST_CONN_LOGIN))) {
		aLog(D_WARN, "Service login:%u can't obtain data from storage:%u\n",s->instance,cfg->st->instance);
		sleep(60);
	}
	
	pthread_rwlock_wrlock(cfg->rwlock);
	pthread_rwlock_rdlock(Units.rwlock);
	for (u=Units.root; u!=NULL; u=u->next){
		stLgObtainDbData(cfg,u); // for each unit, ask the database and fill the u->logindata structure
		if (u->logindata) u->logindata->c_state=u->logindata->d_state; // reset the state value to default
	}
	pthread_rwlock_unlock(Units.rwlock);
	if(cfg->fd) {stCloseSql(cfg->st,cfg->fd); cfg->fd=NULL;}
	pthread_rwlock_unlock(cfg->rwlock);
	//actually we deno't need cfg->query after that today
	
	sLgRestoreLogin(cfg); //restore login status back

	aLog(D_DEBUG, "service login:%u checking every %d seconds\n", s->instance, cfg->delay);

	while (1) {
		aDebug(DEBUG_LOGIN, "Checking logins (every %d seconds)\n", cfg->delay);
		now=time(NULL);
		
		//probably there might be much faster but suspect there might be locks optherwise
		pthread_rwlock_wrlock(cfg->rwlock);	
		pthread_rwlock_rdlock(Units.rwlock);
		for (u=Units.root; u!=NULL; u=u->next) 
		   if (u->logindata && u->logindata->c_state) {

			if (u->logindata->abs && (now - u->logindata->opened) > u->logindata->abs) {
				aDebug(DEBUG_LOGIN, "Unit %06X absolute timeout reached: now=%ld, opened=%ld\n", u->id, now, u->logindata->opened);
				sLgLogin(cInternal, u, NULL, cfg, (LG_LOGOUT|LG_FORCE));
			}
			if ((u->logindata->inact && u->ap.LastUsed() && (now - u->ap.LastUsed()) > u->logindata->inact)
				|| (u->logindata->inact && !u->ap.LastUsed() && (now - u->logindata->opened) > u->logindata->inact)) {
				aDebug(DEBUG_LOGIN, "Unit %06X inact timeout reached: now=%ld, opened=%ld, lastused=%ld\n", u->id, now, u->logindata->opened, u->ap.LastUsed());
				sLgLogin(cInternal, u, NULL, cfg, (LG_LOGOUT|LG_FORCE));
			}
		   }

		pthread_rwlock_unlock(Units.rwlock);
		pthread_rwlock_unlock(cfg->rwlock);

		s->Sleep(cfg->delay*1000); // login timeouts will be checked every cfg->delay seconds
	}
S_LOGIN_CLEANUP_POP:
	pthread_cleanup_pop(0);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////////////////////
void sLgRestoreLogin(Login_cfg *cfg) {
        sLoginData *ld;
        NetUnit *u;

        pthread_rwlock_rdlock(Units.rwlock);

        for (u=Units.root; u!=NULL; u=u->next) {
                ld=u->logindata;
                if(!ld) continue;

		if(ld->c_state)
			sLgLogin(cInternal, u, NULL, cfg, (LG_LOGIN|LG_FORCE)); 
		else 
			sLgLogin(cInternal, u, NULL, cfg, (LG_LOGOUT|LG_FORCE));
        }

        pthread_rwlock_unlock(Units.rwlock);
}
//////////////////////////////////////////////////////////////////////////////////////////
void sLgCancel(void *v){
	Service *s = (Service*)v;
	Login_cfg *cfg = (Login_cfg*)s->cfg;
	aFree(cfg->query);
	pthread_rwlock_destroy(cfg->rwlock);
        aFree(cfg->rwlock);
	aFree(s->cfg);
	aLog(D_INFO, "cancelling service login:%u\n", s->instance);
}
//////////////////////////////////////////////////////////////////////////////////////////
void cShowLogin(Connection *conn, char *param[32]){
	NetUnit *u, *ut=NULL;
	oid t;

	int total=0, enabled=0, opened=0, closed=0;
	static char buf1[32], buf2[32];

	if (!strcasecmp(param[2], "oid")) {
		t=strtol(param[3], NULL, 16);
		ut=Units.getUnitById(t);
		}
	else if (!strcasecmp(param[2], "name")) {
		ut=Units.getUnit(param[3]);
		}

	Service *s=Services.getServiceByName("login",NULL);
        if(!s) return;
	Login_cfg *cfg=(Login_cfg*)s->cfg;

	pthread_rwlock_rdlock(cfg->rwlock);
	pthread_rwlock_rdlock(Units.rwlock);
	for (u=Units.root; u!=NULL; u=u->next){
		total++;
		if (u->logindata) {
			enabled++;
			if (u->logindata->c_state) opened++; else closed++;
			if ((ut && ut==u) || !ut) {
				fprintf(conn->stream_w, "OID: %06X (%s) %s, inact:%ld, abs:%ld\n", u->id, u->name?u->name:"<\?\?>", u->logindata->c_state?"OPENED":"CLOSED", u->logindata->inact, u->logindata->abs);
				if (u->logindata->c_state) fprintf(conn->stream_w, " OP:%s, LU:%s, %s[%s]\n", timeU2T(u->logindata->opened, buf1), timeU2T(u->ap.LastUsed(), buf2), inet_ntoa(u->logindata->ip_from), ether_ntoa(&u->logindata->mac_from));
			}
		}
	}
	pthread_rwlock_unlock(Units.rwlock);
	pthread_rwlock_unlock(cfg->rwlock);
	fprintf(conn->stream_w, "Total units: %d, enabled: %d, opened: %d, closed:%d\n", total, enabled, opened, closed);

}

//////////////////////////////////////////////////////////////////////////////////////////
void sLgSetCfg(char *param[32], Connection *conn, Login_cfg *c){
	NetUnit *u;
	oid t;
	time_t t_inact, t_abs;

	if (!strcasecmp(param[1], "oid")) {
		t=strtol(param[2], NULL, 16);
		u=Units.getUnitById(t);
		if (!u) { aParse(conn, "unit with oid=%06X is not exist\n", t); return; }
	}
	else if (!strcasecmp(param[1], "name")) {
		u=Units.getUnit(param[2]);
		if (!u) { aParse(conn, "unit with name=%s is not exist\n", param[2]); return; }
	}
	else { aParse(conn, "oid must be specified\n"); return; }
	
	Service *s=Services.getServiceByName("login",NULL);
        if(!s) return;
	Login_cfg *cfg=(Login_cfg*)s->cfg;

	pthread_rwlock_wrlock(cfg->rwlock);
	
	if (!u->logindata) {
		u->logindata = (sLoginData*)aMalloc(sizeof(sLoginData));
		u->logindata->inact=c->default_inact;
		u->logindata->abs=c->default_abs;
		u->logindata->opened=0L;
		bzero(&u->logindata->ip_from, sizeof (struct in_addr));
		bzero(&u->logindata->mac_from, sizeof (struct ether_addr));
		u->logindata->d_state=u->logindata->c_state=u->logindata->strict=0;
		u->logindata->password=NULL;
	}

	int i=3;
	while (param[i]!=empty){
		if (!strcasecmp(param[i], "inact")){
			t_inact=strtol(param[i+1], NULL, 10);
			if (t_inact != 0 && (c->min_inact > t_inact || t_inact > c->max_inact)) aParse(conn, "invalid inact value range, setting to default\n");
			else u->logindata->inact=t_inact;
		}
		else if (!strcasecmp(param[i], "abs")){
			t_abs=strtol(param[i+1], NULL, 10);
			if (t_abs != 0 && (c->min_abs > t_abs || t_abs > c->max_abs)) aParse(conn, "invalid abs value range, setting to default\n");
			else u->logindata->abs=t_abs;
		}
		else if (!strcasecmp(param[i], "password")){
			int j=strlen(param[i+1]);
			if (j < c->min_passwd_length) {
				aParse(conn, "password length is too small (real %d need at least %d)\n", j, c->min_passwd_length);
				return;
			}
			u->logindata->password=set_string(param[i+1]);
		}
		else if (!strcasecmp(param[i], "ip")) {
			inet_aton(param[i+1], &u->logindata->ip_from);
		}
		else if (!strcasecmp(param[i], "mac")){
			struct ether_addr *o_mac;
			o_mac=ether_aton(param[i+1]);
			if (o_mac) memcpy(&u->logindata->mac_from, o_mac, sizeof (struct ether_addr));
		}
		else if (!strcasecmp(param[i], "strict")){
			u->logindata->strict=1; i--;
		}
		else if (!strcasecmp(param[i], "nostrict")){
			u->logindata->strict=0; i--;
		}
		i+=2;
	}

	aDebug(DEBUG_LOGIN, "set/got: oid=%06X, inact:%ld, abs:%ld, pass:%s, d_s=%d, c_s=%d, st=%d\n", u->id, u->logindata->inact, u->logindata->abs, u->logindata->password?u->logindata->password:"-", u->logindata->d_state, u->logindata->c_state, u->logindata->strict);
	
	stLgSetCfg(c,u);
	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void sLgLoginCfg(char *param[32], Connection *conn, int no_flag, Login_cfg *c){
	NetUnit *u;
	oid t;
	char *password=NULL;
	struct in_addr o_ip;
        struct ether_addr o_mac;
        u_char flag=no_flag?LG_LOGOUT:LG_LOGIN;

        bzero(&o_ip, sizeof (struct in_addr));
        bzero(&o_mac, sizeof (struct ether_addr));

	if (!strcasecmp(param[1], "oid")) {
		t=strtol(param[2], NULL, 16);
		u=Units.getUnitById(t);
		if (!u) { aParse(conn, "unit with oid=%06X is not exist\n", t); return; }
	}
	else if (!strcasecmp(param[1], "name")) {
		u=Units.getUnit(param[2]);
		if (!u) { aParse(conn, "FAIL: unit with name=%s is not exist\n", param[2]); return; }
	}
	else { aParse(conn, "FAIL: unit oid/name must be specified\n"); return; }

	if (!u->logindata) { aParse(conn, "FAIL: login service is not running for that unit\n"); return; }
	
	int i=3;
	
	while (param[i]!=empty){
		if (!strcasecmp(param[i], "ip")){
			inet_aton(param[i+1], &o_ip);
		}
		else if (!strcasecmp(param[i], "mac")){
			struct ether_addr *mac;
			mac=ether_aton(param[i+1]);
			if (!memcmp(mac, "\0\0\0\0\0\1", sizeof (struct ether_addr))) flag|=LG_FORCE;
		}
		else if (!strcasecmp(param[i], "password")){
			password=param[i+1];
		}
		i+=2;
	}
	Service *s=Services.getServiceByName("login",NULL);
        if(!s) return;
	Login_cfg *cfg=(Login_cfg*)s->cfg;

	pthread_rwlock_wrlock(cfg->rwlock);

	aDebug(DEBUG_LOGIN, "login/got: oid=%06X, pass:%s(%s), ip=%s, mac=%s\n", u->id, password?password:"-", u->logindata->password?u->logindata->password:"-", inet_ntoa(o_ip), ether_ntoa(&o_mac));

	if (!password || !u->logindata->password) { 
		aParse(conn, "FAIL: password is not set for %06X\n", u->id); 
		goto END; 
	}
	if (strcmp(password,u->logindata->password)) { 
		aParse(conn, "FAIL: password incorrect\n"); 
		goto END; 
	}
	sLgLogin(conn, u, &o_ip, cfg, flag);
END:
        pthread_rwlock_unlock(cfg->rwlock);
}
//////////////////////////////////////////////////////////////////////////////////////////
void sLgLogin(Connection *conn, NetUnit *u, struct in_addr *ip_src, Login_cfg *cfg, u_char flag) {
	struct ether_addr zero_mac;
        struct ether_addr current_mac;
        sLoginData *ld=u->logindata;
        struct in_addr *ip;

        bzero(&zero_mac, sizeof (struct ether_addr));
        bzero(&current_mac, sizeof (struct ether_addr));

        if(ip_src) ip=ip_src;
        else ip=(in_addr*)"\0\0\0\0";
	
	if(!(flag&LG_FORCE)) {
		// strict login processing
		if (ld->strict && (ld->ip_from.s_addr!=ip->s_addr)) {
                        aParse(conn, "FAIL: login is not allowed\n");
                        return;
                }

		// mac address processing
		aDebug(DEBUG_LOGIN, "MAC: checking %s for mac address...\n", inet_ntoa(*ip));
                sLgObtainMac(ip, &current_mac);
                memcpy(&ld->last_mac_from, &current_mac, sizeof (struct ether_addr));

                // mac address processing
                if (memcmp(&ld->mac_from, &zero_mac, sizeof (struct ether_addr))) { // unit's MAC is not zero
                        if (memcmp(&ld->mac_from, &current_mac, sizeof (struct ether_addr))) { // unit's MAC != current
                                aDebug(DEBUG_LOGIN, "MAC: not match database: %s\n", ether_ntoa(&ld->mac_from));
                                aParse(conn, "FAIL: login is not allowed\n");
                                return;
                        }
                }
	} else 
		aDebug(DEBUG_LOGIN, "MAC: magic mac address is specified\n");
	
	if(flag&LG_LOGIN) { // login
		u->unit2trees(REMOVE);
		cAccessScriptCall(DROP, u, "LOGIN OFF");

		ld->c_state=1; // open the unit
		ld->opened=time(NULL);
		memcpy(&ld->last_ip_from, ip, sizeof (struct in_addr));
		if (cfg->set_user_ip && u->type==NETUNIT_USER) {
			NetUnit_user *uu = (NetUnit_user*)u;
                       	if(ip_src) memcpy(&uu->ip, ip, sizeof (struct in_addr));
                       	aDebug (DEBUG_LOGIN, "unit user ip address is set to %s\n", inet_ntoa(*ip));
		}
		u->unit2trees(ADD);
		cAccessScriptCall(PASS, u, "LOGIN ON");
		aDebug(DEBUG_LOGIN, "login success from ip:%s, mac:%s\n", inet_ntoa(*ip), ether_ntoa(&current_mac));
                aParse(conn, "OK: login success from ip:%s, mac:%s\n", inet_ntoa(*ip), ether_ntoa(&current_mac));
	}
	else if(flag&LG_LOGOUT) { // logout
		u->unit2trees(REMOVE);
		cAccessScriptCall(DROP, u, "LOGIN OFF");
		u->ap.ClearLastUsed(); //clear last_used	

                aDebug(DEBUG_LOGIN, "login success from ip:%s, mac:%s\n", inet_ntoa(*ip), ether_ntoa(&current_mac));
                aParse(conn, "OK: login success from ip:%s, mac:%s\n", inet_ntoa(*ip), ether_ntoa(&current_mac));
	
		ld->c_state=0; // close the unit
		if (cfg->set_user_ip && u->type==NETUNIT_USER) {
			NetUnit_user *uu = (NetUnit_user*)u;
			memset(&uu->ip, 0, sizeof (struct in_addr));
			aDebug (DEBUG_LOGIN, "unit user ip address is cleared\n");
		}
	}
}
//////////////////////////////////////////////////////////////////////////////////////////
#define ROUNDUP(a) ((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
void sLgObtainMac(struct in_addr *u, struct ether_addr *current_mac){

#ifdef FREEBSD

	char *e;
	int mib[6];
	size_t needed;
	char *lim, *buf, *next;
	struct rt_msghdr *rtm;
	struct sockaddr_inarp *sin;
	struct sockaddr_dl *sdl;

	mib[0] = CTL_NET; mib[1] = PF_ROUTE; mib[2] = 0; mib[3] = AF_INET; mib[4] = NET_RT_FLAGS; mib[5] = RTF_LLINFO;
	if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0) return;
	if ((buf = (char*)aMalloc(needed)) == NULL) return;
	if (sysctl(mib, 6, buf, &needed, NULL, 0) < 0) return;

	lim = buf + needed;

	for (next = buf; next < lim; next += rtm->rtm_msglen) {
		rtm = (struct rt_msghdr *)next;
		sin = (struct sockaddr_inarp *)(rtm + 1);
		sdl = (struct sockaddr_dl *)((char *)sin + ROUNDUP(sin->sin_len));
		e=LLADDR(sdl);
		if (sin->sin_addr.s_addr==u->s_addr && memcmp(e, "\0\0\0\0\0\0", sizeof(struct ether_addr)))  {
			memcpy(current_mac, e, sizeof (struct ether_addr));
			aDebug(DEBUG_LOGIN, "MAC for ip: %s is: %s found in ARP\n", inet_ntoa(*u), ether_ntoa((struct ether_addr*)e));
			aFree(buf);
			return;
		}
	}

	aDebug(DEBUG_LOGIN, "MAC for ip:%s is not found in ARP\n", inet_ntoa(*u));
	aFree(buf);

#endif

#ifdef LINUX

	char *buf;
	char b_ip[32], b_hw[32], b_trash[32];
	struct in_addr in;
	unsigned char *e;
	FILE *arp;


	if ((buf = (char*)aMalloc(256)) == NULL) return;

	arp=fopen("/proc/net/arp", "rt");
	if (arp==NULL) { aDebug(DEBUG_LOGIN, "failed to open /proc/net/arp\n"); return; }

	if (NULL==fgets(buf, 255, arp)) return;

	while (!feof(arp)){
		if (NULL==fgets(buf, 255, arp)) break;
		bzero(b_ip, 31); bzero(b_hw, 8);
		sscanf(buf, "%s%s%s%s", b_ip, b_trash, b_trash, b_hw);
		if (inet_aton(b_ip, &in) && ((struct ether_addr*)e=ether_aton(b_hw))){
			if (u->s_addr==in.s_addr && memcmp(e, "\0\0\0\0\0\0", sizeof(struct ether_addr))) {
				memcpy(current_mac, e, sizeof (struct ether_addr));
				aDebug(DEBUG_LOGIN, "MAC for ip: %s is: %s found in ARP\n", inet_ntoa(*u), ether_ntoa((struct ether_addr*)e));
				aFree(buf);
				return;
			}
		}
	}
#endif

	return;
}
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
