/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: s_processor.c,v 1.18.2.11 2004/05/06 10:35:04 jura Exp $ */

#include "netams.h"

void *sProcessor(void *ss);
void sProcessorCancel(void *);
Message* sPConstructFillMessageRaw(char prefix, oid netunit, oid policy, policy_data *pdata, time_t t);
void sPConstructStoreMessages(oid netunit, policy_data *pd, time_t t);
void sProcessorListCfgUnit(NetUnit *d, FILE *f);

NetUnitsList Units;
PolicyList PolicyL;
FIFO Mux_in(10000);
Service *Processor;
//////////////////////////////////////////////////////////////////////////
ServiceProcessor_cfg ProcessorCfg;
//////////////////////////////////////////////////////////////////////////
void sProcessorInit(Service *s){
	Processor=s;
	ServiceProcessor_cfg *cfg=&ProcessorCfg;
	cfg->st_root=NULL;
	cfg->def=NULL;
	cfg->def=new NetUnit(NETUNIT_UNKNOWN);
	cfg->delay=PROCESSOR_DELAY; // in milliseconds
	cfg->lifetime=PROCESSOR_LIFETIME;
	cfg->restrict_all=DROP;
	cfg->restrict_local=PASS;
	cfg->access_script=NULL;
	s->cfg=(void*)cfg;
	pthread_create(&(s->t_id), NULL, &sProcessor, s);
}
//////////////////////////////////////////////////////////////////////////
void sProcessorProcessCfg(char *param[32], Connection *conn, unsigned no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "unit")) cUnit(conn, param, no_flag);
	else if (!strcasecmp(param[0], "policy")) cPolicy(conn, param, no_flag);
	else if (!strcasecmp(param[0], "default")) cDefault(conn, param, no_flag);
	else if (!strcasecmp(param[0], "restrict")) cRestrict(conn, param, no_flag);
	else if (!strcasecmp(param[0], "storage")) cRegister(conn, param, no_flag);
	else if (!strcasecmp(param[0], "lookup-delay")) {
		unsigned delay=atol(param[1]);
		if (delay>1 && delay<24*60*60) {
			aParse(conn, "lookup delay is set to %u seconds\n", delay);
			cfg->delay=1000*delay;
		} 
		else aParse(conn, "lookup delay value invalid\n");
	}		
	else if (!strcasecmp(param[0], "flow-lifetime")) {
		unsigned lifetime=atol(param[1]);
		if (lifetime>1 && lifetime<24*60*60) {
			aParse(conn, "flow lifetime is set to %u seconds\n", lifetime);
			cfg->lifetime=1000*lifetime;
		} 
		else aParse(conn, "flow lifetime value invalid\n");
	}		
	else if (!strcasecmp(param[0], "access-script")) {
		aFree(cfg->access_script);
		cfg->access_script=set_string(param[1]);
		aParse(conn, "access control script name is set to '%s'\n", param[1]);
	}		
	else aParse(conn, "unknown processor command: %s\n", param[0]);
}
//////////////////////////////////////////////////////////////////////////
void *sProcessor(void *ss){
	Service *s=(Service*)ss;
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;
	Message *msg;
	Service *st_s;
	time_t current_time;
	struct timeval start, stop;
	NetUnit *tmp;
	policy_data *pd;
	st_unit *st;
	unsigned short pass;
	unsigned short fill=0;
	struct time_counters tc;

	aLog(D_INFO, "service processor thread started\n");
	pthread_cleanup_push(sProcessorCancel, s);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	Processor->t_id=pthread_self();
	
	// now we should sleep before starting actual processor
	Processor->Sleep();

	if (!cfg->st_root) aLog(D_WARN, "no storages registered!\n");
	//Initialize storages
	for(st=cfg->st_root;st!=NULL;st=st->next) {
		if (!(st_s=Services.getService("storage", st->id)))
                	aLog(D_WARN, "storage %d registered but never started\n", st->id);
                else {
                	st->in=((ServiceStorage_cfg*)st_s->cfg)->in;
                        st->out=((ServiceStorage_cfg*)st_s->cfg)->out;
                        st->s=st_s;
                        aLog(D_INFO, "using storage:%d, in=%p, out=%p, service_cfg=%p\n", st->id, st->in, st->out, st->s->cfg);
                        if(st==cfg->st_root) aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",st->id);
               	}
	} 

	// loop for incoming data forever
	while(1) {
		//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// previous worked as a part of DS_LOOKUP service
		gettimeofday(&start, NULL);
		time(&current_time);
		
		if (unsigned(current_time - aGetActual('H'))*1000 <= 2*cfg->delay) {
			fill=1;
			PrepareTimeCounters(&tc);
		}	

		pthread_rwlock_rdlock(Units.rwlock);

		for (tmp=Units.root; tmp!=NULL; tmp=tmp->next) {
			for (pd=tmp->ap.root; pd!=NULL; pd=pd->next) {
				if ( ( fill || (unsigned(current_time - pd->flow.from)*1000 >= cfg->lifetime) )&& (pd->flow.in!=0 || pd->flow.out!=0 )) {
					
					sPConstructStoreMessages(tmp->id, pd, current_time);
					pd->flow.in=pd->flow.out=0; 
					pd->flow.from=pd->flow.to=0;
				}
				if (fill) FillTimeCounters(pd,&tc);
			}

		}
		pthread_rwlock_unlock(Units.rwlock);

		gettimeofday(&stop, NULL);

		double d_start=start.tv_sec+ (double)start.tv_usec/1000000;
		double d_stop=stop.tv_sec+ (double)stop.tv_usec/1000000;
		aDebug(DEBUG_PROC_MUX, "lookup takes %.4f  seconds\n", d_stop-d_start);
		// previous worked as a part of DS_LOOKUP service
		//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

		while((msg=Mux_in.Pop())) {
			pass=0;
			switch (msg->type) {
				case SOURCE: 
					aDebug(DEBUG_PROC_MUX, "SOURCE messages NOT allowed!\n");
					delete msg;
					break; // message was: data-source
				case STORE: 
					for (st=cfg->st_root;st!=NULL;st=st->next) {
					   	msg->id=st->id;
						switch (msg->prefix){
							case 'T':
							case 'M':
							case 'W':
							case 'D':
							case 'H':
								if (st->cat==ALL || st->cat==SUMMARY) {
									st->in->Push(msg);
									pass=1;
								}
							   	break;	
							case 'F':
								if (st->cat==ALL || st->cat==RAW) {
									st->in->Push(msg);
									pass=1;
								}
								break;
							} //switch
						aDebug(DEBUG_PROC_MUX, "P->ST %c st:%u unit:%06X acct:%06X from:%lu to:%lu\n", msg->prefix, msg->id, msg->netunit, msg->ap, msg->data.from, msg->data.to);	
					} // for all registered storages
					break;
				case READ: 
					// only ONE storage process READ request, this default to st_root for now
					st=cfg->st_root;
                                        if (st) { 
						msg->id=st->id;
						switch (msg->prefix){
							case 'T':
							case 'M':
							case 'W':
							case 'D':
							case 'H':
								if (st->cat==ALL || st->cat==SUMMARY){
								   	st->out->Push(msg);
									pass=1;
								}
								break;	
							case 'F':
								if (st->cat==ALL || st->cat==RAW){
								   	st->out->Push(msg);
									pass=1;
								}
								break;
						} //switch
						aDebug(DEBUG_PROC_MUX, "P<-ST %c st:%u unit:%06X acct:%06X to now\n", msg->prefix, msg->id, msg->netunit, msg->ap);
					}  
					break;
				default:
					aDebug(DEBUG_PROC_MUX, "out <%u> (unknown message type)\n", msg->type);
					delete msg;
					break;
			} // this message processed
			if(!pass) delete msg;
		} // all messages processed
		//wakeup all storages
		for (st_unit *st=cfg->st_root;st!=NULL;st=st->next) st->s->Wakeup();

		Processor->Sleep(cfg->delay);
	} // infinite while
	// we will never reach this point
	pthread_cleanup_pop(0);
	return NULL;
}
//////////////////////////////////////////////////////////////////////////
void sProcessorCancel(void *v){
	Service *s=(Service*)v;
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;
	aLog(D_INFO, "cancelling service processor, %ld items active\n", Mux_in.num_items);
	st_unit *ptr,*st=cfg->st_root;
	while(st) {
		ptr=st;
		st=st->next;
		aFree(ptr);
	}
	cfg->st_root=NULL;
	if(cfg->access_script) aFree(cfg->access_script);
	PolicyL.DeleteAll();
	delete cfg->def;
}
//////////////////////////////////////////////////////////////////////////
unsigned sProcessorData(Connection *conn, char *param[32]){
	
   	if (!strcasecmp(param[1], "?")) { cHelp(conn, 0, "data"); return PARSE_OK; }
	
	Message *msg = new Message();
	msg->type=SOURCE;
	msg->data.from=msg->data.to=msg->data.in=msg->data.out=0;
	msg->ts=time(NULL);
	msg->pdata=NULL;

	// message format:
	// data {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {IN} {OUT}
	msg->prefix=param[1][0];
	// msg_p->ds=conn->conn_id;
	msg->id=0; //temporary; when switching to data-source mode, we should register it!
	msg->netunit=strtol(param[2], NULL, 16);
	msg->ap=strtol(param[3], NULL, 16);
	sscanf(param[4], "%ld", &msg->data.from);
	sscanf(param[5], "%ld", &msg->data.to);
	sscanf(param[6], "%qu", &msg->data.in);
	sscanf(param[7], "%qu", &msg->data.out);

	Mux_in.Push(msg);
	aDebug(DEBUG_PROC_MUX, "in fl:%c unit:%06X acct:%06X from:%ld to:%ld in:%qu out:%qu\n", msg->prefix, msg->netunit, msg->ap, msg->data.from, msg->data.to, msg->data.in, msg->data.out);
	Processor->Wakeup();
	  
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
unsigned sProcessorRead(Connection *conn, char *param[32]){
	long timeout;

   	if (!strcasecmp(param[1], "?")) { cHelp(conn, 0, "read"); return PARSE_OK; }
	
	Message *msg = new Message();
	msg->type=READ;
	msg->ts=time(NULL);
	msg->pdata=NULL;

	// message format:
	// data {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {TIMEOUT}
	msg->prefix=param[1][0];
	
	if (param[6]!=empty) sscanf(param[6], "%ld", &timeout);
	else timeout=DEFAULT_READ_TIMEOUT; 
	

	msg->id=0; // later processor will change this field and put the actual storage ID
	msg->netunit=strtol(param[2], NULL, 16);
	msg->ap=strtol(param[3], NULL, 16);
	sscanf(param[4], "%ld", &msg->data.from);
	sscanf(param[5], "%ld", &msg->data.to);
	
	Mux_in.Push(msg);
	Processor->Wakeup();

	int i=sProcessorRawRead(msg, timeout);

	// how to deal with timeouted messages ?

	if (i && i==ETIMEDOUT) aParse(conn, "Query timeout\n");
	else if (!i) {
		fprintf(conn->stream_w, "prefix:%c unit:%06X acct:%06X from:%ld to:%ld timeout:%ld\nin:%qu out:%qu\n", msg->prefix, msg->netunit, msg->ap, msg->data.from, msg->data.to, timeout, msg->data.in, msg->data.out);
		aDebug(DEBUG_PROC_MUX, "read[%d], fl:%c unit:%06X acct:%06X from:%ld to:%ld timeout:%ld\nin:%qu out:%qu\n", i, msg->prefix, msg->netunit, msg->ap, msg->data.from, msg->data.to, timeout, msg->data.in, msg->data.out);
		delete msg;
	}
	else aParse(conn, "Unknown error\n");  
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
int sProcessorRawRead(Message *msg, long timeout){
	pthread_mutex_t *unblock_m=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
	pthread_cond_t *unblock_c=(pthread_cond_t*)aMalloc(sizeof (pthread_cond_t));
	pthread_mutex_init(unblock_m, NULL);
	pthread_cond_init(unblock_c, NULL);

	msg->unblock=unblock_c;
	struct timespec ts;
	struct timeval tv;
	gettimeofday(&tv, NULL);

	TIMEVAL_TO_TIMESPEC(&tv, &ts);
	ts.tv_sec+=(timeout/1000);
	ts.tv_nsec+=1000000*(time_t)(timeout-1000*((long double)timeout/1000));
	if (ts.tv_nsec>1000000000) { 
		ts.tv_sec++; 
		ts.tv_nsec-=1000000000; 
	}
	pthread_mutex_lock(unblock_m);
	int i=pthread_cond_timedwait(unblock_c, unblock_m, &ts);
	pthread_mutex_unlock(unblock_m);
	
	pthread_cond_destroy(unblock_c); aFree(unblock_c);
	pthread_mutex_destroy(unblock_m); aFree(unblock_m);

	return i;
}
//////////////////////////////////////////////////////////////////////////
unsigned cShowStat(Connection *conn, char *param[32]){
	NetUnit *u;
	unsigned points;
	policy_data *pd;
	Message *msg;

	if (!strcasecmp(param[2], "unit")) {
		if (!strcasecmp(param[3], "*")) u=NULL;
		else {
			u=Units.getUnitById(strtol(param[3], NULL, 16));
			if (!u) u=Units.getUnit(param[3]);
			if (!u) { aParse(conn, "unit not found: %s\n", param[3]); return 0; }
			}
		}
	else { aParse(conn, "show stat command invalid: %s\n", param[2]); return 0; }

	switch(param[4][0]){
		case 'M': case 'm': case 'W': case 'w': case 'D': case 'd': case 'H': case 'h':
		break;
		default: { aParse(conn, "show stat period invalid: %s\n", param[4]); return 0; }
		}

	if (strcasecmp(param[5], "to") || strcasecmp(param[6], "now")) { aParse(conn, "show stat time invalid: %s %s\n", param[5], param[6]); return 0;	}

	points=atoi(param[7]); 
	if (!points) { aParse(conn, "show stat points invalid: %s %s\n", param[7]); return 0; }

	aParse(conn, "unit %s [%c] to->now <%u>\n", u?u->name:"*", param[4][0], points);
	
	Service *s;
        if(!(ProcessorCfg.st_root && (s=ProcessorCfg.st_root->s))) {
                aParse(conn, "There is no storage to use\n");
                return 0;
	}

	msg = new Message();
        msg->type=STAT;
        msg->stat=(stat_data*)aMalloc(sizeof(stat_data));
        msg->prefix=param[4][0];

	msg->id=0;
        msg->ts=time(NULL);     msg->pdata=NULL; msg->netunit=u->id;
        msg->stat->num_points=msg->stat->num_policies=0;
        msg->stat->in=NULL; msg->stat->out=NULL;

	aDebug(DEBUG_PROC_MUX, "%c unit:%06X STAT\n", msg->prefix, msg->netunit);

	fprintf(conn->stream_w, "%s %06X %lu %lu %u\n", u->name?u->name:"<>", u->id, 0L, msg->ts, points);
	for (pd=u->ap.root; pd!=NULL; pd=pd->next) 
		fprintf(conn->stream_w, "%s ", pd->policy->name?pd->policy->name:"<>");
	fprintf(conn->stream_w, "\n");

	if (stStatSql(s,msg)) {
		//fprintf(conn->stream_w, "got %u lines each %u policies\n", msg->stat->num_points, msg->stat->num_policies);
		for (unsigned i=0; i<msg->stat->num_points; i++) {
			for (unsigned j=0; j<msg->stat->num_policies; j++) {
				fprintf(conn->stream_w, "%qu %qu ", msg->stat->in[j*msg->stat->num_points+i], msg->stat->out[j*msg->stat->num_points+i]);
			}
			fprintf(conn->stream_w, "\n");
		}

	}
	fprintf(conn->stream_w, "\n");
	delete msg;
	return 1;
}
//////////////////////////////////////////////////////////////////////////
void sProcessorListCfg(Service *s, FILE *f){
	char buf[64];
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)s->cfg;

	fprintf(f, "service processor %d\n", s->instance);

	if(cfg->delay != PROCESSOR_DELAY) fprintf(f, "lookup-delay %d\n", cfg->delay/1000);
	if(cfg->lifetime != PROCESSOR_LIFETIME) fprintf(f, "flow-lifetime %d\n", cfg->lifetime/1000);

	// first, policy rules
	{
	pthread_rwlock_rdlock(PolicyL.rwlock);
	Policy *d;

	for (d=PolicyL.root; d!=NULL; d=d->next) {
		fprintf(f, "policy ");
		if (d->type==POLICY_ACCT) fprintf(f, "acct ");
		else if (d->type==POLICY_FW) fprintf(f, "fw ");
		else fprintf(f, "<\?\?>");

		fprintf(f, "oid %06X ", d->id);
		if (d->name) fprintf(f, "name %s ", d->name);

		if (d->target.target_type==PT_IP_TRAFFIC) fprintf(f, "target %-12s", d->getTarget(buf));
		fprintf(f, "\n");
	}

	pthread_rwlock_unlock(PolicyL.rwlock);
	}

	// default parameters
	if (cfg->def->ap.root) { fprintf(f, "default acct-policy "); cfg->def->ap.ListForCfg(f); }
	if (cfg->def->fp.root) { fprintf(f, "default fw-policy "); cfg->def->fp.ListForCfg(f); } 

	// restrict parameters
	fprintf(f, "restrict all %s local %s\n", cfg->restrict_all?"drop":"pass", cfg->restrict_local?"drop":"pass");

	// next, print NetUnits
	{
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d;

	for (d=Units.root; d!=NULL; d=d->next) {
		if (d->type==NETUNIT_GROUP) {
				NetUnit_group *h = (NetUnit_group*)d;
				fprintf(f, "unit group oid %06X ", h->id);
				if (h->name) fprintf(f, "name %s ", h->name);
				if (h->email) fprintf(f, "email %s ", h->email);
				sProcessorListCfgUnit(d, f);
				fprintf(f, "\n");
		}
	} //for

	for (d=Units.root; d!=NULL; d=d->next) {
		if(d->type==NETUNIT_GROUP) continue;
		fprintf(f, "unit %s oid %06X ", netunit_type_name[d->type], d->id);
		if (d->name) fprintf(f, "name %s ", d->name);
		if (d->email) fprintf(f, "email %s ", d->email);

		switch (d->type) {
			case NETUNIT_HOST: {
				NetUnit_host *h = (NetUnit_host*)d;
				fprintf(f, "ip %s ", inet_ntoa(h->ip));
				} break;
			case NETUNIT_CLUSTER: { 
				NetUnit_host *t;
				NetUnit_cluster *h = (NetUnit_cluster*)d;
				for (t=h->root; t!=NULL; t=t->next)	fprintf(f, "ip %s ", inet_ntoa(t->ip));
				} break;
			case NETUNIT_GROUP: continue;
			case NETUNIT_NET: {
				NetUnit_net *h = (NetUnit_net*)d;
				fprintf(f, "ip %s ", inet_ntoa(h->ip));	
				fprintf(f, "mask %s ", inet_ntoa(h->mask));
				} break;
			case NETUNIT_USER: {
                                NetUnit_user *u = (NetUnit_user*)d;
				if(u->ip.s_addr!=0) fprintf(f, "ip %s ", inet_ntoa(u->ip));
				if(u->password) fprintf(f, "password %s ", u->password);
				if(u->real_name) fprintf(f, "real_name \"%s\" ", u->real_name);
                                } break;
			default: 
				continue;
		} //switch

		sProcessorListCfgUnit(d, f);
		fprintf(f, "\n");
	} //for

	pthread_rwlock_unlock(Units.rwlock);
	} //netunits
	
	{ // data-sources and storages
	for(st_unit *st=cfg->st_root;st!=NULL;st=st->next) 
		if (st->s) { 
			fprintf(f, "storage %d ", st->id);
			if (st->cat==ALL) fprintf(f, "all\n"); 
			else if (st->cat==RAW) fprintf(f, "raw\n"); 
			else if (st->cat==SUMMARY) fprintf(f, "summary\n"); 
			else fprintf(f, "??\n"); 
		}
	}

	if (cfg->access_script) fprintf(f, "access-script \"%s\"\n", cfg->access_script);

	fprintf(f, "\n");
}
//////////////////////////////////////////////////////////////////////////
void sProcessorListCfgUnit(NetUnit *d, FILE *f){
	policy_data *cpd;

	DSList *dsl=d->getDSList();
	if (dsl){
		fprintf(f, "ds-list ");
		for (; dsl!=NULL; dsl=dsl->next) {
			if (dsl!=d->dsl_root) fprintf(f, ",");
			fprintf(f, "%u", dsl->id);
		}
		fprintf(f, " ");
	}

	if (d->parent && d->parent->name) fprintf(f, "parent %s ", d->parent->name);
	if (d->sys_policy!=SP_NONE) { 
		char *buf=NULL; 
		buf=ShowSysPolicy(buf, d->sys_policy,d->sys_policy_perm, d->sys_policy_name);    
		fprintf(f, "%s ", buf);
		aFree(buf);
	}
	if (d->nlp) fprintf(f, "no-local-pass ");
	if (d->ap.root) {
		int header_printed=0, already_default; 
		policy_data *cpx;

		for (cpd=d->ap.root; cpd!=NULL; cpd=cpd->next) {
			already_default=0;
			for (cpx=ProcessorCfg.def->ap.root; cpx!=NULL; cpx=cpx->next) if (cpx->policy->id==cpd->policy->id) already_default=1;
			if (!already_default) {
				if (!header_printed) { fprintf(f, "acct-policy "); header_printed=1; }
				fprintf(f, "%s%s%s ", cpd->inv?"!":"", cpd->brk?"%":"", cpd->policy->name?cpd->policy->name:"");
				/* ^^ [21.06.2002] ugly workaround. what we should do if policy name is undefined (NULL)? */
			}
		}
	}
	if (d->fp.root) { 
		int header_printed=0, already_default; 
		policy_data *cpx;

		for (cpd=d->fp.root; cpd!=NULL; cpd=cpd->next) {
			already_default=0;
			for (cpx=ProcessorCfg.def->fp.root; cpx!=NULL; cpx=cpx->next) if (cpx->policy->id==cpd->policy->id) already_default=1;
			if (!already_default) {
				if (!header_printed) { fprintf(f, "fw-policy "); header_printed=1; }
				fprintf(f, "%s%s%s ", cpd->inv?"!":"", cpd->brk?"%":"", cpd->policy->name?cpd->policy->name:"");
			}
		}
	}
}
//////////////////////////////////////////////////////////////////////////
void cShowProcessor(Connection *conn){
	FIFO *in, *out;
	ServiceProcessor_cfg *cfg=&ProcessorCfg;
	
	fprintf(conn->stream_w, "INPUT  Multiplexer\n");
	fprintf(conn->stream_w, "\t current: %u\tmax: %u\ttotal: %u\n", Mux_in.num_items, Mux_in.max_items, Mux_in.total_items); 
	fprintf(conn->stream_w, "OUTPUT Multiplexers\n");
	for(st_unit *st=cfg->st_root;st!=NULL;st=st->next) {
		if (st->s) {
			in=st->in;
			out=st->out;
			fprintf(conn->stream_w, "   Storage %d type ", st->s->instance);
			switch (((ServiceStorage_cfg*)(st->s->cfg))->type) {
				case HASH: fprintf(conn->stream_w, "HASH"); break;
				case MY_SQL: fprintf(conn->stream_w, "MySQL"); break;
				case POSTGRES: fprintf(conn->stream_w, "Postgres"); break;
				default: fprintf(conn->stream_w, "UNKNOWN"); break;
			}
			fprintf(conn->stream_w, "\n");
			fprintf(conn->stream_w, "\t in current: %u\tmax: %u\ttotal: %u\n", in->num_items, in->max_items, in->total_items); 
			fprintf(conn->stream_w, "\tout current: %u\tmax: %u\ttotal: %u\n", out->num_items,	out->max_items, out->total_items); 
		}
	}
	fprintf(conn->stream_w, "Default acct policy: "); cfg->def->ap.List(conn);
	fprintf(conn->stream_w, "Default   fw policy: "); cfg->def->fp.List(conn);
}
//////////////////////////////////////////////////////////////////////////
void sPConstructStoreMessages(oid netunit, policy_data *pd, time_t t){
	char prefix[6] = { 'F', 'H', 'D', 'W', 'M', 'T' };
	Message *msg;

	for(unsigned i=0;i<6;i++) {
		msg= new Message();
		msg->ts=t;
		msg->ap=pd->policy->id;
		msg->netunit=netunit;
		msg->pdata=NULL;
		msg->prefix=prefix[i];
		msg->type=STORE;
		switch(prefix[i]){
			case 'F':
				memcpy(&msg->data, (struct pstat *)&pd->flow, sizeof (struct pstat));
				break;
			case 'M':
				memcpy(&msg->data, (struct pstat *)&pd->m, sizeof (struct pstat));
				break;
			case 'W':
				memcpy(&msg->data, (struct pstat *)&pd->w, sizeof (struct pstat));
				break;
			case 'D':
				memcpy(&msg->data, (struct pstat *)&pd->d, sizeof (struct pstat));
				break;
			case 'H':
				memcpy(&msg->data, (struct pstat *)&pd->h, sizeof (struct pstat));
				break;
			case 'T':
				memcpy(&msg->data, (struct pstat *)&pd->t, sizeof (struct pstat));
				break;
		}
		aDebug(DEBUG_PROC_MUX, "DS->P %c in unit:%06X acct:%06X from:%lu to:%lu in:%qu out:%qu\n",msg->prefix, msg->netunit, msg->ap, msg->data.from, msg->data.to, msg->data.in, msg->data.out);
		Mux_in.Push(msg);
	}
}
//////////////////////////////////////////////////////////////////////////
unsigned cDefault(Connection *conn, char *param[], int no_flag){

	int i=1;

	if (!strcmp(param[1], "acct-policy")) PolicyAdd(ProcessorCfg.def, &i, POLICY_ACCT, conn, param, no_flag);
	else if (!strcmp(param[1], "fw-policy")) PolicyAdd(ProcessorCfg.def, &i, POLICY_FW, conn, param, no_flag);
	else { aParse(conn, "default action name unspecified\n", param[1]); return PARSE_OK; }

	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
unsigned cRestrict(Connection *conn, char *param[], int no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;
	int i=1;

	while (param[i]!=empty) {
		if (!strcmp(param[i], "all")) {
			if (!strcmp(param[i+1], "drop")){
				cfg->restrict_all=DROP;
				aParse(conn, "restricting ALL traffic to DROP\n");
			}
			else if (!strcmp(param[i+1], "pass")){
				cfg->restrict_all=PASS;
				aParse(conn, "restricting ALL traffic to PASS\n");
			}
			else aParse(conn, "restrict ALL traffic command invalid!\n");
		}
		else if (!strcmp(param[i], "local")) {
			if (!strcmp(param[i+1], "drop")){
				cfg->restrict_local=DROP;
				aParse(conn, "restricting LOCAL traffic to DROP\n");
			}
			else if (!strcmp(param[i+1], "pass")){
				cfg->restrict_local=PASS;
				aParse(conn, "restricting LOCAL traffic to PASS\n");
			}
			else aParse(conn, "restrict LOCAL traffic command invalid!\n");
		}

		else aParse(conn, "restrict command '%s' invalid!\n", param[i]);

		i+=2;
	}
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
unsigned cRegister(Connection *conn, char *param[], int no_flag){
	ServiceProcessor_cfg *cfg=(ServiceProcessor_cfg*)conn->service->cfg;
	unsigned id=atoi(param[1]);
	st_unit *st,*last=NULL;
	for (st=cfg->st_root;st!=NULL;st=st->next) {
		if (st->id==id) break;
                last=st;
        }

        if(no_flag)
        	if(st) {
                	aParse(conn, "unregistering storage:%d\n", id);
                        aLog(D_INFO, "unregistering storage:%d\n", id);
                        if(st==cfg->st_root) {
                        	cfg->st_root=st->next;
                                if(cfg->st_root) {
                                	aParse(conn, "using storage:%u as source for READ and STAT requests\n",cfg->st_root->id);
                                	aLog(D_INFO, "using storage:%u as source for READ and STAT requests\n",cfg->st_root->id);
                        	}
                        } else
                        	st->prev->next=st->next;
                        if(st->next) st->next->prev=st->prev;
                        aFree(st);
                } else
                	aParse(conn, "no such storage %d or storage %d not registered\n",id,id);
	else
       		if(st) {
               	 	aParse(conn, "reconfiguring storage %d\n",id);
                        if (!strcasecmp(param[2], "all")) st->cat=ALL;
                        else if (!strcasecmp(param[2], "summary")) st->cat=SUMMARY;
                        else if (!strcasecmp(param[2], "raw")) st->cat=RAW;
                } else {
                	aParse(conn, "registering storage: %u\n", id);
                        st=(st_unit*)aMalloc(sizeof(st_unit));
                        st->id=id;
                        st->in=st->out=NULL;
                        st->s=NULL;
                       	if(!cfg->st_root) {
                        	cfg->st_root=st;
                                aParse(conn, "using storage:%u as source for READ and STAT requests\n",id);
                        } else
                        	last->next=st;
                        st->next=NULL;
                        st->prev=last;
                        if (!strcasecmp(param[2], "all")) st->cat=ALL;
                        else if (!strcasecmp(param[2], "summary")) st->cat=SUMMARY;
                        else if (!strcasecmp(param[2], "raw")) st->cat=RAW;
                }
	return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

