/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: users.c,v 1.9.2.3 2004/05/06 10:35:07 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
User::User(char *n, int h){
	name=set_string(n);
	real_name=email=password=crypted=NULL;
	next=NULL;
	permissions=UPERM_NONE;
	hidden=h;
	id=newOid(OID_USER);
	mode=USER_HUMAN;
}

User::~User(){
	aFree(name);
	aFree(real_name);
	aFree(email);
	aFree(password);
	aFree(crypted);
}

void User::setName(char *s) { aFree(name); name=set_string(s);Users.changed_user=1; }
void User::setRealName(char *s) { aFree(real_name); real_name=set_string(s); }
void User::setEmail(char *s) { aFree(email); email=set_string(s);}
void User::setPassword(char *s) { 

	aFree(password); 
	// with new password encryption technology we must do following:
	// if there is plain text password given here, free old password string variable,
	// next prepare crypted form, and call setCrypted
	char *t=crypt(s, "$1$");
	if (t) setCrypted(t);
	Users.changed_pw=1;  
	}

void User::setCrypted(char *s) { aFree(crypted); crypted=set_string(s);}

void User::setPermissions(char *s) { 
	for (int i=0; i<=255; i++) if (permissions_list[i] && s && !strcmp(s, permissions_list[i])) permissions=i;
	}

unsigned User::getPermissions() { return permissions; }
	 
//////////////////////////////////////////////////////////////////////////
UsersList::UsersList(){
	root=last=NULL;
	num_users=0;
	changed_pw=changed_user=1;
	lock=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
	}

UsersList::~UsersList(){
	pthread_mutex_destroy(lock);
	aFree(lock);
	}

void UsersList::Insert(User *s){
	pthread_mutex_lock(lock);
	User *d;
	for(d=root; d!=NULL; d=d->next)
		if (d==s) { 
			pthread_mutex_unlock(lock);
			return;
			}
	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	num_users++;
	pthread_mutex_unlock(lock);
	}	

void UsersList::Delete(char *p){
		User *s = getUser(p);
		if (!s) Delete(s);
		}

void UsersList::Delete(User *s){
	pthread_mutex_lock(lock);
	User *d, *p;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;

			delete s;
			num_users--;
			break;
			}
		p=d; 
	 	}
	pthread_mutex_unlock(lock);
	}

User *UsersList::getUser(char *p){
	pthread_mutex_lock(lock);
	User *d;
	for(d=root; d!=NULL; d=d->next)
		if (!strcmp(d->name, p)) { pthread_mutex_unlock(lock); return d; }
	pthread_mutex_unlock(lock);
	return NULL;
	}

User *UsersList::getUserById(oid id){
	pthread_mutex_lock(lock);
	User *d;
	for(d=root; d!=NULL; d=d->next)
		if (d->id == id) { pthread_mutex_unlock(lock); return d; }
	pthread_mutex_unlock(lock);
	return NULL;
	}

void UsersList::listUsers(Connection *conn){
	pthread_mutex_lock(lock);
	User *t;
	fprintf(conn->stream_w, "%6s | %4s | %10s | %20s | %10s\n", "OID", "MODE", "NAME", "REAL NAME", "PERMIT");
	for(t=root; t!=NULL; t=t->next){
		if (!t->hidden) {
			fprintf(conn->stream_w, "%06X |", t->id);
			switch (t->mode) {
				case USER_HUMAN:		fprintf(conn->stream_w, " %4s |", "U"); break; 
				case USER_DATASOURCE:	fprintf(conn->stream_w, " %4s |", "DS"); break; 
				case USER_STORAGE:		fprintf(conn->stream_w, " %4s |", "ST"); break; 
				default: 				fprintf(conn->stream_w, " %4s |", "<\?\?>"); break; 
				}
			fprintf(conn->stream_w, " %10s | %20s | %10s\n", t->name, t->real_name?t->real_name:"<>", permissions_list[t->permissions]);
			}
		}
	pthread_mutex_unlock(lock);
	}

void UsersList::listUsersCfg(FILE *f){
	pthread_mutex_lock(lock);
	User *t;
	for(t=root; t!=NULL; t=t->next){
		if (!t->hidden) {
			fprintf(f, "user oid %06X", t->id);
			if (t->name) fprintf(f, " name %s", t->name);
			if (t->real_name) fprintf(f, " real-name \"%s\"", t->real_name);
			if (t->password) fprintf(f, " password %s", t->password);
			if (t->crypted) fprintf(f, " crypted %s", t->crypted);
			if (t->email) fprintf(f, " email %s", t->email);
			fprintf(f, " permit %s\n", permissions_list[t->permissions]);
			}
		}
	pthread_mutex_unlock(lock);
	}

void UsersList::listUsersHtml(Connection *conn){
	pthread_mutex_lock(lock);
	User *t;
	for (t=root; t!=NULL; t=t->next)
		if (!t->hidden && t->permissions >= UPERM_SHOWSTAT_HIS) 
			fprintf(conn->stream_w, " %s", t->name);
	pthread_mutex_unlock(lock);
	}

void UsersList::listPasswordsHtml(Connection *conn){
	pthread_mutex_lock(lock);
	User *t;
	for (t=root; t!=NULL; t=t->next)
		if (!t->hidden && t->permissions >= UPERM_SHOWSTAT_HIS) {
			if (t->password) fprintf(conn->stream_w, "%s:%s\n", t->name, crypt(t->password, "$1$"));
			if (t->crypted) fprintf(conn->stream_w, "%s:%s\n", t->name, t->crypted);
			}
	pthread_mutex_unlock(lock);
	}

//////////////////////////////////////////////////////////////////////////
unsigned cUser(Connection *conn, char *param[], int no_flag){
	User *u=NULL;
	
	if (!strcasecmp(param[1], "?")) { cHelp(conn, no_flag, "user"); return PARSE_OK; }
	
	else if (!strcmp(param[1], "name")) u=Users.getUser(param[2]);
	else if (!strcmp(param[1], "oid")) u=Users.getUserById(strtol(param[2], NULL, 16));
	else { aParse(conn, "user command unknown\n"); return PARSE_OK; }
	
	if (!u && !no_flag) { 
		u = (User*)aMalloc(sizeof(User)); 
		User *t= new User(param[2]); 
		memcpy(u, t, sizeof(User));
		Users.Insert(u); 
		aParse(conn, "user %s created\n", u->name); 
		}

	if (u && no_flag){
		aParse(conn, "user %s deleted\n", u->name);
		Users.Delete(u);
		u->~User();
		return PARSE_OK;
		}

	int i=3;
	while (param[i]!=empty) {
	 	if (strcasecmp(param[i], "real-name")==0) { 
			u->setRealName(param[i+1]);
			aParse(conn, "user %s real-name set: %s\n", u->name, u->real_name);

			} else if (strcasecmp(param[i], "name")==0) { 
			u->setName(param[i+1]);
			aParse(conn, "user %s name set: %s\n", u->name, u->name);

			} else if (strcasecmp(param[i], "oid")==0) { 
			u->id=strtol(param[i+1], NULL, 16);
			aParse(conn, "user %s oid set: %06X\n", u->name, u->id);

			} else if (strcasecmp(param[i], "email")==0) { 
			u->setEmail(param[i+1]);
			aParse(conn, "user %s email set: %s\n", u->name, u->email);

			} else if (strcasecmp(param[i], "password")==0) { 
			u->setPassword(param[i+1]);
			aParse(conn, "user %s password set: %s\n", u->name, u->password?u->password:"<>");

			} else if (strcasecmp(param[i], "crypted")==0) { 
			u->setCrypted(param[i+1]);
			aParse(conn, "user %s crypted password set: %s\n", u->name, u->crypted);

			} else if (strcasecmp(param[i], "permit")==0) { 
			u->setPermissions(param[i+1]);
			aParse(conn, "user %s permissions set: %s\n", u->name, permissions_list[u->getPermissions()]);
			conn->permissions=u->getPermissions();
			} 
			
			else aParse(conn, "user %s command unknown\n", u->name); 

		i=i+2;
		}
	return PARSE_OK;
	}
//////////////////////////////////////////////////////////////////////////
unsigned cShowUsers(Connection *conn){
	Users.listUsers(conn);
	return PARSE_OK;
	}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
