/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: common.c,v 1.32.2.6 2004/05/06 10:35:02 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////////////////////
static char *help[] = {

"?",  "top level commands",
	"auth", "authenticates a user to open corresponding unit",
	"data", "send raw data to processor service",
	"debug", "turning on or off debugging on some action",
	"exit", "finish this client session and disconnect",
	"kill", "kill the daemon very quickly, with last data loss",
	"mode", "switch the client operation mode",	
  	"monitor", "look and store specified unit activity",						   
	"oid", "generate new oid",
	"read", "get raw data from processor service",
	"reload", "gracefully shutdown and restart the daemon",
	"save", "write the running config to a file",
	"schedule", "schedule an action to do it intime",
	"service", "set up service",
	"show", "shows various system parameters", 
	"shutdown", "gracefully shut down the daemon",
	"user", "users management",
	NULL, NULL,

"show", "shows various system parameters",
	"config", "running configuration",
	"connections", "active connections to server",
	"list", "NetUnits list with applied policy list",
	"monitor", "monitor information", 
	"policy", "policy list",
	"processor", "processor service status",
	"schedule", "active scheduled actions",
	"units", "NetUnits list",
	"users", "users list",
	"version", "software version and current status",
	NULL, NULL,

"show config", "running configuration",
   "<cr>", "running configuration",
   "brief", "the same but without NetUnits",
   NULL, NULL,

"show list", "NetUnits list with applied policy list",
   "<cr>", "NetUnits list with applied policy list",
   "full", "the same with flow counters",
   NULL, NULL,

"mode", "switch the client operation mode",
   "data-source", "data source mode",
   "human", "user i/o mode (default)",
   "storage", "storage mode",
   NULL, NULL,

"oid", "generate new oid",
	"host", "for unit host",
    "net", "for unit net",
    "group", "for unit group",
    "cluster", "for unit cluster",
    NULL, NULL,

"data", "send raw data to processor service \ndata {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {IN} {OUT}",
   "PREFIX", "'F','M','W','D','H' - record prefix", 
   "NETUNIT_OID", "NetUnit OID, for which this record is entered", 
   "AP_OID", "Accounting Policy OID", 
   "FROM", "UNIXTIME, record start time", 
   "TO", "UNIXTIME, record stop time", 
   "IN", "LONG LONG, bytes in", 
   "OUT", "LONG LONG, bytes out",
   NULL, NULL,

"read", "get raw data from processor service \nread {PREFIX} {NETUNIT_OID} {AP_OID} {FROM} {TO} {TIMEOUT}",
   "PREFIX", "'F','T','M','W','D','H' - record prefix", 
   "NETUNIT_OID", "NetUnit OID, for which this record is entered", 
   "AP_OID", "Accounting Policy OID", 
   "FROM", "UNIXTIME, record start time", 
   "TO", "UNIXTIME, record stop time", 
   "TIMEOUT", "Query timeout, in milliseconds",
   NULL, NULL,

"schedule", "schedule an action to do it intime\nschedule oid {OID} time {TIME} action {ACTION}",
	"TIME", "time to schedule this action",
	"ACTION", "action to perform",
	NULL, NULL,

NULL,
	NULL
 } ;

char UNKNOWN_REFERENCE[]="<..>";

//////////////////////////////////////////////////////////////////////////////////////////
// Memory management /////////////////////////////////////////////////////////////////// 
static quad_t bytes_allocated=0;
static int times_allocated=0;
static int times_freed=0;
static int times_freed_null=0;
pthread_mutex_t mem_lock = PTHREAD_MUTEX_INITIALIZER;

void *aMalloc(size_t size, int critical){
	pthread_mutex_lock(&mem_lock);
	void *res;
	res=malloc(size+1);
	memset(res, 0, size+1);
	if (res==NULL) {
		if (critical) aLog(D_CRIT, "malloc of %d bytes failed!\n", size);  //will never returns
		else aLog(D_WARN, "malloc of %d bytes failed!\n", size); 
		}
	times_allocated++;
	bytes_allocated+=size;
	pthread_mutex_unlock(&mem_lock);
	return res;
	}

void aFree(void *ptr){
	pthread_mutex_lock(&mem_lock);
	if (!ptr) times_freed_null++;
	else { 
		times_freed++;
		free(ptr);
		}
	pthread_mutex_unlock(&mem_lock);
	}

quad_t aGetBytesAllocated() { return bytes_allocated; }
unsigned aGetTimesAllocated() { return times_allocated; }
unsigned aGetTimesFreed() { return times_freed; }
unsigned aGetTimesFreedNull() { return times_freed_null; }

//////////////////////////////////////////////////////////////////////////////////////////
// Logging Management ///////////////////////////////////////////////////////////////////
pthread_mutex_t parse_lock = PTHREAD_MUTEX_INITIALIZER;
void aParse(Connection *conn, char const *fmt, ...){
	pthread_mutex_lock(&parse_lock);
	va_list ap;

	if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) fprintf(conn->stream_w, "parse: ");
    va_start(ap, fmt);
	if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) vfprintf(conn->stream_w, fmt, ap);
    va_end(ap);
	pthread_mutex_unlock(&parse_lock);
	}

pthread_mutex_t log_lock = PTHREAD_MUTEX_INITIALIZER;
void aLog(unsigned level, char const *fmt, ...){

	time_t t;
	struct timeval tv;
	char *t_T;
	char *s_where, *s_state;
	va_list ap;
	
	switch(level){
                case D_INFO:    s_state="INFO"; break;
                case D_ERR:     s_state="ERR"; break;
                case D_CRIT:    s_state="CRIT"; break;
                case D_WARN:    s_state="WARN"; break;
                case D_DEBUG:   
#ifndef DEBUG
				return;
#endif 
				s_state="DEBUG"; 
				break;
                default:        s_state=UNKNOWN_REFERENCE; break;
	}
	
	pthread_mutex_lock(&log_lock);

	t_T=(char *)aMalloc(30); bzero(t_T, 29);
	s_where=Services.getServiceByThr(pthread_self());
	if (!strcmp(s_where, UNKNOWN_REFERENCE)) 
		s_where=Connections.getConnectionByThr(pthread_self()); 

	time(&t); timeU2T(t, t_T);
	gettimeofday(&tv, NULL);

	if (flag_nodaemon && !flag_quiet) 
		fprintf(stdout, "%s.%04u %s [%s]: ", t_T, (unsigned)((double)tv.tv_usec/100), s_where, s_state);
	if (flag_log) 
		fprintf(LOGFILE, "%s.%04u %s [%s]: ", t_T, (unsigned)((double)tv.tv_usec/100), s_where, s_state);
  
    va_start(ap, fmt);
	if (flag_nodaemon && !flag_quiet) vfprintf(stdout, fmt, ap);
	if (flag_log) vfprintf(LOGFILE, fmt, ap);
    va_end(ap);

	aFree(t_T); 
	pthread_mutex_unlock(&log_lock);
	if (level==D_ERR) termination(0);
	if (level==D_CRIT) termination(-1);
}
//////////////////////////////////////////////////////////////////////////////////////////
void termination(int signal){
	sleep(1); //we should wait until sMain is sleeping, this is not guarantee this
	printf("\n");
	int i=sMain->Wakeup();
	if (i) ; //exit(-1);
	}
/////////////////////////////////////////////////////////////////////////////////////////////
char *timeU2T(time_t time, char *buf){
	struct tm tm;
	if (buf==NULL) buf=(char *)aMalloc(25);
	localtime_r(&time, &tm);
	sprintf(buf, "%02d.%02d.%4d %02d:%02d:%02d", tm.tm_mday, tm.tm_mon+1, tm.tm_year+1900, tm.tm_hour, tm.tm_min, tm.tm_sec);
	return buf;
	}
///////////////////////////////////////////////////////////////////////////
time_t timeT2U(char *buf, time_t *time){
	struct tm tm;
	time_t t;
	sscanf(buf, "%02d.%02d.%4d %02d:%02d:%02d", &tm.tm_mday, &tm.tm_mon, &tm.tm_year, &tm.tm_hour, &tm.tm_min, &tm.tm_sec);
	tm.tm_mon = tm.tm_mon-1;
	tm.tm_year = tm.tm_year-1900;
	tm.tm_isdst = -1;
	t=mktime(&tm);
	if (time!=NULL) memcpy(time, &t, sizeof(time_t));
	return t;
	}
/////////////////////////////////////////////////////////////////////////////////////////////
char *bytesQ2T(unsigned long long bytes, char *buf){
if (buf==NULL) buf=(char*)aMalloc(12);
if (bytes<KILOBYTE) sprintf(buf, "%qu", bytes); 
else if (bytes>=KILOBYTE && bytes<KILOBYTE*KILOBYTE) sprintf(buf, "%.3fK", (double)(bytes)/(KILOBYTE));
else if (bytes>=KILOBYTE*KILOBYTE && bytes <KILOBYTE*KILOBYTE*KILOBYTE) sprintf(buf, "%.3fM", (double)(bytes)/(KILOBYTE*KILOBYTE));
else if (bytes>=KILOBYTE*KILOBYTE*KILOBYTE && bytes <(double)KILOBYTE*KILOBYTE*KILOBYTE*KILOBYTE) sprintf(buf, "%.3fG", (double)(bytes)/((double)KILOBYTE*KILOBYTE*KILOBYTE));
else aLog(D_WARN, "bytesQ2T called with enormous value: %qu\n", bytes);
return buf;
}
//////////////////////////////////////////////////////////////////////////////////////////
unsigned long long bytesT2Q(unsigned long long *bytes, char *buf){
if (buf==NULL) { *bytes=0; return *bytes; }
double b;
sscanf(buf, "%lf", &b);
if (strchr(buf, 'K')!=NULL) b*=KILOBYTE;
if (strchr(buf, 'M')!=NULL) b*=KILOBYTE*KILOBYTE;
if (strchr(buf, 'G')!=NULL) b*=KILOBYTE*KILOBYTE*KILOBYTE;
*bytes=(unsigned long long)b;
return *bytes;
} 
//////////////////////////////////////////////////////////////////////////////////////////
oid newOid(unsigned id){
	oid result;
	
	for(unsigned j=0;j<25;j++) {
		result=unsigned(id*65536 + (double)random()/65536);
		if (Units.getUnitById(result) || PolicyL.getPolicyById(result) || \
		    Sched.getTaskById(result) || Users.getUserById(result) ) 
			continue;
		// no report id checking since they are static
		return result;
	}

	aLog(D_WARN, "newOid cannot construct new id on %u\n", id);
	return 0;
}
//////////////////////////////////////////////////////////////////////////////////////////
void PrepareTimeCounters(struct time_counters *tc) {

        time_t t1;
        struct tm *tm1;

        t1=time(NULL);
        tm1=(struct tm *)aMalloc(sizeof (struct tm));

        localtime_r(&t1, tm1);
        tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_isdst=-1;
        tc->ht=mktime(tm1);

        tm1->tm_hour=0; tm1->tm_isdst=-1;
        tc->dt=mktime(tm1);

        if (tm1->tm_wday==0) tm1->tm_wday=7;
        tm1->tm_mday+=(8-tm1->tm_wday)-7;
        tc->wt=mktime(tm1);

        localtime_r(&t1, tm1);
        tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_hour=0; tm1->tm_isdst=-1;
        tm1->tm_mday=1; // tm1->tm_mon-=1; if (tm1->tm_mon<0) {         tm1->tm_year--; tm1->tm_mon+=12; }
        tc->mt=mktime(tm1);
	
	aFree(tm1);
}
//////////////////////////////////////////////////////////////////////////////////////////
void FillTimeCounters(policy_data *np, struct time_counters *tc){

	time_t t2;

	t2=tc->ht;
	if (np->h.from!=t2) {
		np->h.from=t2;
		np->h.in=np->h.out=0;
	}

	t2=tc->dt;
	if (np->d.from!=t2) {
		np->d.from=t2;
		np->d.in=np->d.out=0;
	}

	t2=tc->wt;
	if (np->w.from!=t2) {
		np->w.from=t2;
		np->w.in=np->w.out=0;
	}

	t2=tc->mt;
	if (np->m.from!=t2) {
		np->m.from=t2;
		np->m.in=np->m.out=0;
	}

}

//////////////////////////////////////////////////////////////////////////////////////////
// Debugging management
char *debug_list[64];
char *debug_list_help[64]; 
pthread_mutex_t debug_lock = PTHREAD_MUTEX_INITIALIZER;

void aDebug(Debug d, char const *fmt, ...){
#ifdef DEBUG
	pthread_mutex_lock(&debug_lock);
	Connection *conn;
	va_list ap;

	for	(conn=Connections.root; conn!=NULL; conn=conn->next) if (debug_list[d] && aDebugIsSet(conn->debug, d)) {
		if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) fprintf(conn->stream_w, "|%s: ", debug_list[d]);
	#ifdef DEBUG_TO_LOG
		if (flag_log) fprintf(LOGFILE, "|%s: ", debug_list[d]);
	#endif
    	va_start(ap, fmt);
		if (conn->stream_w && !(flag_quiet&&(conn==cInternal))) vfprintf(conn->stream_w, fmt, ap);
		conn->Flush();
	#ifdef DEBUG_TO_LOG
		if (flag_log) vfprintf(LOGFILE, fmt, ap);
	#endif
    	va_end(ap);
		}

	pthread_mutex_unlock(&debug_lock);
	return;
#endif	
}

int aDebugAdd(Debug &d, char *str){
	for (unsigned i=0; i<64; i++){
		if (debug_list[i] && !strcmp(str, debug_list[i])) {
			if (i==DEBUG_NONE) d=0;
			else if (i==DEBUG_ALL) d=~0;
			else d |= (Debug)1<<i;
			return 1;
			}
		}
	return 0;
	}

int aDebugRm(Debug &d, char *str){
	for (unsigned int i=0; i<64; i++){
		if (debug_list[i] && !strcmp(str, debug_list[i])) {
			if (i==DEBUG_NONE) d=~0;
			else if (i==DEBUG_ALL) d=0;
			else d &=~ (Debug)1<<i;
			return 1;
			}
		}
	return 0;
	}

int aDebugIsSet(Debug d, unsigned u){
	if (u==DEBUG_NONE && d==0) return 1;
	else if (u==DEBUG_ALL && d==~0) return 1;
	else if (u!=DEBUG_NONE && (d&((Debug)1<<u))) return 1; 
	return 0;
	}

void aDebugInit(){
	for (int i=0; i<=63; i++) { debug_list[i]=NULL;	debug_list_help[i]=NULL; }
	debug_list[DEBUG_NONE]="none"; debug_list_help[DEBUG_NONE]="turn off all debugging info";             
	debug_list[DEBUG_COMMAND]="command"; debug_list_help[DEBUG_COMMAND]="entered command processing";         
	debug_list[DEBUG_PARSE]="parse"; debug_list_help[DEBUG_PARSE]="entered command parse";
	debug_list[DEBUG_SLEEP]="sleep"; debug_list_help[DEBUG_SLEEP]="services going in/out sleep mode";
	debug_list[DEBUG_SERVER]="server"; debug_list_help[DEBUG_SERVER]="commands server activity";
	debug_list[DEBUG_PROC_MUX]="proc_mux"; debug_list_help[DEBUG_PROC_MUX]="processor multiplexer activity";
	debug_list[DEBUG_DS_IP]="ds_ip"; debug_list_help[DEBUG_DS_IP]="data sources, IP-packets based";
	debug_list[DEBUG_STORAGE]="storage"; debug_list_help[DEBUG_STORAGE]="storage activity";
	debug_list[DEBUG_ALERT]="alert"; debug_list_help[DEBUG_ALERT]="system alerts";
	debug_list[DEBUG_SCHED]="scheduler"; debug_list_help[DEBUG_SCHED]="scheduled actions";
	debug_list[DEBUG_NF]="netflow"; debug_list_help[DEBUG_NF]="netflow packets processing";
	debug_list[DEBUG_HTML]="html"; debug_list_help[DEBUG_HTML]="html static pages generation";
	debug_list[DEBUG_MONITOR]="monitor"; debug_list_help[DEBUG_MONITOR]="specified units packet headers";
	debug_list[DEBUG_LOGIN]="login"; debug_list_help[DEBUG_LOGIN]="user login events";
	debug_list[DEBUG_QUOTA]="quota"; debug_list_help[DEBUG_QUOTA]="quota control events";
	debug_list[DEBUG_IPTREE]="iptree"; debug_list_help[DEBUG_IPTREE]="iptree engine events";
	debug_list[DEBUG_FLOW]="flow"; debug_list_help[DEBUG_FLOW]="ip flow engine events";
	debug_list[DEBUG_ALL]="all"; debug_list_help[DEBUG_ALL]="turn on all debugging info (be careful!)";     
	}
//////////////////////////////////////////////////////////////////////////////////////////
void cHelp(Connection *conn, int no_flag, char *str){
	
	aDebug(DEBUG_PARSE, "help %s%s\n", no_flag?"NO ":"", str);

	int printed=0;

	if (!strcasecmp(str, "debug")) {
		fprintf(conn->stream_w, "turning %s debugging on following action:\n", no_flag?"OFF":"ON");
		for (int i=0; i<=63; i++) if (debug_list[i])
			fprintf(conn->stream_w, "   %12s | %s\n", debug_list[i], debug_list_help[i]);		
		fprintf(conn->stream_w, "\n"); printed=1;
		}
	else {
		int i=0;
		while(help[i]) {
			if (!strcasecmp(str, help[i])) {
				fprintf(conn->stream_w, "%s\n", help[i+1]);	i+=2;
				while(help[i]) {
					fprintf(conn->stream_w, "   %12s | %s\n", help[i], help[i+1]);	i+=2;
					}
				fprintf(conn->stream_w, "\n"); printed=1;
				}
			else { while(help[i]) i+=2; }
			i+=2;
			}
		}
	if (!printed) fprintf(conn->stream_w, "no help is available on: %s\n", str);
	return;
	}

//////////////////////////////////////////////////////////////////////////////////////////
void aShowVersion(FILE *f) {
	fprintf(f, "%s version %d.%d(%d", aaa_fw_software_name, aaa_fw_major_version, aaa_fw_minor_version, aaa_fw_build_version);
	if (aaa_fw_build_version_local) fprintf(f, ".%d", aaa_fw_build_version_local);
	fprintf(f, ") %s / %s\n", aaa_fw_build_person, aaa_fw_build_time);
	}

//////////////////////////////////////////////////////////////////////////////////////////
void cShowVersion(Connection *c){
	aShowVersion(c->stream_w);

	struct rusage res;
	struct timeval now;
	unsigned long days, hours, mins, secs;
	time_t t_secs;
	double mksecs;

	gettimeofday(&now, NULL);
	if (0!=getrusage(RUSAGE_SELF, &res)) { aParse(c, "resourse usage retrival failed: %s\n", strerror(errno)); return; }

	t_secs=(now.tv_sec - program_start.tv_sec);
	days=t_secs/(60*60*24);
	hours=(t_secs-days*60*60*24)/(60*60);
	mins=(t_secs-(days*60*60*24)-(hours*60*60))/60;
	secs=t_secs-(days*60*60*24)-(hours*60*60)-mins*60;
	mksecs=(double)(now.tv_usec - program_start.tv_usec)/1000000;
	fprintf(c->stream_w, "Run time:");
	if (days>0) fprintf(c->stream_w, " %ld days", days);
	if (hours>0) fprintf(c->stream_w, " %ld hours", hours);
	if (mins>0) fprintf(c->stream_w, " %ld mins", mins);
	fprintf(c->stream_w, " %.4f secs\n", fabs((double)secs+mksecs));

	t_secs=(res.ru_stime.tv_sec);
	days=t_secs/(60*60*24);
	hours=(t_secs-days*60*60*24)/(60*60);
	mins=(t_secs-(days*60*60*24)-(hours*60*60))/60;
	secs=t_secs-(days*60*60*24)-(hours*60*60)-mins*60;
	mksecs=(double)(res.ru_stime.tv_usec)/1000000;
	fprintf(c->stream_w, "System time:");
	if (days>0) fprintf(c->stream_w, " %ld days", days);
	if (hours>0) fprintf(c->stream_w, " %ld hours", hours);
	if (mins>0) fprintf(c->stream_w, " %ld mins", mins);
	fprintf(c->stream_w, " %.4f secs\n", (double)secs+mksecs);

	fprintf(c->stream_w, "Average CPU/system load: %3.2f%%\n", 100*((float)(res.ru_stime.tv_sec)+(float)(res.ru_stime.tv_usec)/1000000)/((float)(now.tv_sec - program_start.tv_sec)+(float)(now.tv_usec - program_start.tv_usec)/1000000)); 
	fprintf(c->stream_w, "Process ID: %d RES: %ldK\n", getpid(), res.ru_maxrss); 
	fprintf(c->stream_w, "Memory allocated: %qu (%d), freed (%d) (%d NULL) [%d used]\n", aGetBytesAllocated(), aGetTimesAllocated(), aGetTimesFreed(), aGetTimesFreedNull(), aGetTimesAllocated() - aGetTimesFreed());
	fprintf(c->stream_w, "\n");

	fprintf(c->stream_w, "Total objects:\n");
	
	fprintf(c->stream_w, "   NetUnits: %d\n", Units.num_units);
	fprintf(c->stream_w, "   Policies: %d\n", PolicyL.num_policies);
	fprintf(c->stream_w, "   Services: %d\n", Services.getServices());
	fprintf(c->stream_w, "   Users: %d\n", Users.getUsers());
	fprintf(c->stream_w, "   Connections: %d active, %d total\n", Connections.getConnectionsNum(), Connections.getConnectionsLastId());
	fprintf(c->stream_w, "   Scheduled tasks: %d\n", Sched.num_tasks);
	{	
	        Service *s=NULL;
        	while((s=Services.getServiceByName("alerter",s))) {
                	ServiceAlerter_cfg *cfg=(ServiceAlerter_cfg*)s->cfg;
                	fprintf(c->stream_w, "Alerter %u queue max: %u, current: %u\n", s->instance, cfg->queue->max_items, cfg->queue->num_items);
        	}

	}
	fprintf(c->stream_w, "\nServices info:\n");
	
	Services.listInfo(c->stream_w);

	}
	
//////////////////////////////////////////////////////////////////////////////////////////
time_t aGetActual(char prefix){
	time_t t1;
	struct tm *tm1;
	time_t actual;

	t1=time(NULL);
	tm1=(struct tm *)aMalloc(sizeof (struct tm));
	localtime_r(&t1, tm1);


	switch(prefix) {
		case 'T': actual=t1; break;
		case 'F': actual=t1; break;
		case 'M': {
					tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_hour=0; tm1->tm_isdst=-1;
					tm1->tm_mday=1; // tm1->tm_mon-=1; if (tm1->tm_mon<0) { 	tm1->tm_year--; tm1->tm_mon+=12; }
					actual=mktime(tm1);
					break;
					}
		case 'W': {
					tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_hour=0; tm1->tm_isdst=-1; 
					if (tm1->tm_wday==0) tm1->tm_wday=7;
					tm1->tm_mday+=(8-tm1->tm_wday)-7;
					actual=mktime(tm1);
					break;
					}
		case 'D': {
					tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_hour=0; tm1->tm_isdst=-1;
					actual=mktime(tm1);
					break;
					}
		case 'H': {
					tm1->tm_sec=0; tm1->tm_min=0; tm1->tm_isdst=-1;
					actual=mktime(tm1);
					break;
					}
		}

	aFree(tm1);
	return actual;
	}

//////////////////////////////////////////////////////////////////////////////////////////
void print_to_string(char **string, char const *fmt, ...){

	va_list ap;
	char *result;
	char *ret;

   	va_start(ap, fmt);
	if (-1==vasprintf(&ret, fmt, ap)) { va_end(ap); return; }
   	va_end(ap);

	if (*string) {
		result=(char*)aMalloc(strlen(*string)+strlen(ret)+1);
		sprintf(result, "%s%s", *string, ret);
		free(ret);
		aFree(*string); 
		*string=result;
	}
	else  {
		(*string)=(char*)aMalloc(strlen(ret)+1);
		strcpy((*string), ret);
		free(ret);
	}

}
//////////////////////////////////////////////////////////////////////////////////////////
char *set_string(char *string) {
	char *res=NULL;
	if(string) {
		res=(char*)aMalloc(strlen(string)+1);
		strcpy(res, string);
	}
	return res;
}
//////////////////////////////////////////////////////////////////////////////////////////
// command execution
// if buffer==NULL, we don't need in command output; else print_to_string onto it
void cExec(char *cmd, char **buffer){

	Connection *conn = new Connection("exec", CONN_FD_VIRT);
	conn->permissions=UPERM_ALL;
	Connections.Insert(conn);

	aCommand(conn, cmd);
	
	fflush(conn->stream_w);

//	aDebug(DEBUG_PARSE, "cExec res: \"%s\"\n", conn->buffer?conn->buffer:"??"); 

	if (conn->buffer && buffer) print_to_string(buffer, "%s", conn->buffer);

	Connections.Delete(conn);

	}

//////////////////////////////////////////////////////////////////////////////////////////
char *ShowSysPolicy(char *buf, SysPolicy p, oid perm, char *name){
	if (buf==NULL) buf=(char*)aMalloc(32);

	if (p!=SP_NONE) { 
		if (perm) snprintf(buf, 31, "sys-%06X%s%s", perm, name[0]?"-":"", name);
		else snprintf(buf, 31, "%s", name);
		}
	else snprintf(buf, 31, "%s", "");
	return buf;
	}
//////////////////////////////////////////////////////////////////////////////////////////
int cNewOid(Connection *c, char *type) {
    int t;
    
    if (!strcasecmp(type, "host"))
        t = NETUNIT_HOST;
    else if (!strcasecmp(type, "net"))
        t = NETUNIT_NET;
    else if (!strcasecmp(type, "group"))
        t = NETUNIT_GROUP;
    else if (!strcasecmp(type, "cluster"))
        t = NETUNIT_CLUSTER;
    else {
        aParse(c, "invalid unit type specified, assuming NETUNIT_UNKNOWN\n");
        t = NETUNIT_UNKNOWN;
	    }

    fprintf(c->stream_w, "%06X\n", newOid(t));
    return PARSE_OK;
}
//////////////////////////////////////////////////////////////////////////////////////////
char* binary(unsigned long num) {
	char *buf=(char*)aMalloc(34);
	for(unsigned i=0;i<32;i++) 
		if(num&(1<<(31-i))) buf[i]='1';
		else buf[i]='0';
	aFree(buf);
	return buf;
}
//////////////////////////////////////////////////////////////////////////////////////////
void cAccessScriptCall(restrict_type action, NetUnit *u, char *p){
	if (!ProcessorCfg.access_script) return;
	char *buffer;
	int i;

	char *param;
	if (u->type==NETUNIT_HOST) {
		print_to_string(&param, "%s %s", inet_ntoa(((NetUnit_host*)u)->ip), p);
	}
	else if (u->type==NETUNIT_USER) {
		print_to_string(&param, "%s %s", inet_ntoa(((NetUnit_user*)u)->ip), p);
	}
	else {
		param=set_string(p);
	}

	int len = strlen(ProcessorCfg.access_script) + 10 + strlen(param);
	buffer = (char*)aMalloc(len);
	oid oid=u->id;

	switch (action) {
		case DROP:	snprintf(buffer, len, "%s DENY %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
		case PASS:	snprintf(buffer, len, "%s ALLOW %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
		case PASS_LOCAL:	snprintf(buffer, len, "%s LOCAL %06X %s", ProcessorCfg.access_script, oid, param?param:"");
			break;
	}

	i=system(buffer);
	aLog(D_INFO, "cAccessScriptCall oid %06X action %d system:%d\n", oid, action, i);
	aFree(param);
	aFree(buffer);
}
//////////////////////////////////////////////////////////////////////////////////////////
unsigned cShowPerf(Connection *conn, char *param[32]) {
	char *filename=NULL;
	FILE *out;

	if (param[2]!=empty) filename=param[2];
	if (!strcasecmp(param[3], "header")) {
		out=fopen(filename, "wt");
		if (!out) return 1;
		aShowVersion(out);
		/*
		TOD		Time-Of-Day
		RTM		Run Time
		STM		System Time
		LOAD	System Load
		RES		Resources Usage
		*/
		fprintf(out, "TOD\tRTM\tSTM\tLOAD\tRES\t");
		Services.showPerf(out, 1);
		fprintf(out, "\n");

		} else {
		out=fopen(filename, "at");
		if (!out) return 1;
		}

	struct rusage res;
	struct timeval now;
	time_t t_secs;
	double mksecs;
	static char t_T[32];
	time_t t;

	time(&t); timeU2T(t, t_T);
	gettimeofday(&now, NULL);

	//TOD
	fprintf(out, "%s.%04u\t", t_T, (unsigned)((double)now.tv_usec/100));

	//RTM
	t_secs=(now.tv_sec - program_start.tv_sec);
	fprintf(out, "%ld\t", t_secs);

	//STM
	if (0!=getrusage(RUSAGE_SELF, &res)) { fclose(out); return 2; }
	t_secs=(res.ru_stime.tv_sec);
	mksecs=(double)(res.ru_stime.tv_usec)/1000000;
	fprintf(out, "%f\t", t_secs+mksecs); 

	//LOAD
	fprintf(out, "%3.2f\t", 100*((float)(res.ru_stime.tv_sec)+(float)(res.ru_stime.tv_usec)/1000000)/((float)(now.tv_sec - program_start.tv_sec)+(float)(now.tv_usec - program_start.tv_usec)/1000000)); 

	//RES
	fprintf(out, "%ld\t", res.ru_maxrss);

	//show performance counters for all services which provides such an info
	Services.showPerf(out);

	fprintf(out, "\n");	fclose(out);
	return 0;
	}
//////////////////////////////////////////////////////////////////////////////////////////


