/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: net_units.c,v 1.23.2.13 2004/05/06 10:35:03 jura Exp $ */

#include "netams.h"

char *netunit_type_name[NETUNIT_TYPES_NUM]={ "unknown", "user", "host", "cluster", "net", "group" };

/////////////////////////////////////////////////////////////////////////
// NetUnit class
NetUnit::NetUnit(netunit_type t) {
	id=newOid(OID_UNIT);
	name=NULL;
	parent=NULL;
	type=t;
	next=NULL;

	if (ProcessorCfg.def) {
		ProcessorCfg.def->ap.SetForUnit(POLICY_ACCT, this);
		ProcessorCfg.def->fp.SetForUnit(POLICY_FW, this);
		}
	
	dsl_root=NULL;
	nlp=0;
	monitor=NULL;

	sys_policy_name=SysPolicy_name[SP_NONE]; 
	sys_policy=SP_NONE; 
	sys_policy_perm=0;
	open=0;
	logindata=NULL;
	quotadata=NULL;
	changed=1;
	
	email=NULL;
}

NetUnit::~NetUnit() {
	this->unit2trees(REMOVE);
        DSList *ptr,*dsl=dsl_root;
        while(dsl) {
		ptr=dsl;
                dsl=dsl->next;
                aFree(ptr);
        }
	aFree(logindata);
	aFree(name);
}

void NetUnit::setName(char *n){
	aFree(name);
	name=set_string(n);
	changed=1;
	}

match NetUnit::Check(void *s){
	switch(type) {
		case NETUNIT_HOST: return ((NetUnit_host*)this)->Check((struct ip*)s); break;
		case NETUNIT_CLUSTER: return ((NetUnit_cluster*)this)->Check((struct ip*)s); break;
		case NETUNIT_NET: return ((NetUnit_net*)this)->Check((struct ip*)s); break;
		case NETUNIT_GROUP: return ((NetUnit_group*)this)->Check((struct ip*)s); break;
		case NETUNIT_USER: return ((NetUnit_user*)this)->Check((struct ip*)s); break;
		default: break;						 
		}
	return MATCH_NONE;
	}

void NetUnit::unit2tree(IPTree *iptree,unsigned flag) {
	        switch(type) {
                case NETUNIT_HOST: return ((NetUnit_host*)this)->unit2tree(iptree,flag); break;
                case NETUNIT_CLUSTER: return ((NetUnit_cluster*)this)->unit2tree(iptree,flag); break;
                case NETUNIT_NET: return ((NetUnit_net*)this)->unit2tree(iptree,flag); break;
                case NETUNIT_USER: return ((NetUnit_user*)this)->unit2tree(iptree,flag); break;
                default: break;
                }
}

void NetUnit::unit2trees(unsigned flag) {
	Service *s=NULL;

        while((s=Services.getServiceByName("data-source",s))) 
		if(this->checkDSList(s->instance)) {
			IPTree *iptree=((ServiceDS_cfg*)s->cfg)->IPtree;
                	this->unit2tree(iptree,flag);	
		}
}

int NetUnit::setDSList(unsigned ds_id){
	DSList *dsl;
	for(dsl=dsl_root;dsl!=NULL;dsl=dsl->next)
		if (dsl->id==ds_id) return -1; // we will not add existing DS to list
		
	dsl=(DSList*)aMalloc(sizeof(DSList));
	dsl->prev=NULL;
	dsl->id=ds_id;
	if(dsl_root) dsl_root->prev=dsl;
	dsl->next=dsl_root;
	dsl_root=dsl;
	return 0;
	}

int NetUnit::clearDSList(unsigned ds_id){
	for (DSList *dsl=dsl_root;dsl!=NULL;dsl=dsl->next) 
		if (dsl->id==ds_id) {
			if(dsl==dsl_root) dsl_root=dsl->next;
			else dsl->prev->next=dsl->next;
			if(dsl->next) dsl->next->prev=dsl->prev;
			aFree(dsl);
			break;
			}
		
	return 0;
	}

struct DSList *NetUnit::getDSList(){
	return dsl_root;
	}

int NetUnit::checkDSList(unsigned ds_id){
	if(!dsl_root) return 2;
	for(DSList *dsl=dsl_root;dsl!=NULL;dsl=dsl->next)
                if (dsl->id==ds_id) return 1;
	return 0;
	} 
/////////////////////////////////////////////////////////////////////////
// NetUnitsList class
NetUnitsList::NetUnitsList(){
	root=last=NULL;
	num_units=0;
	tries_lock=tries_lock_failed=0;

	rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
	pthread_rwlock_init(rwlock, NULL);
}

NetUnitsList::~NetUnitsList(){
	pthread_rwlock_destroy(rwlock);
	aFree(rwlock);
}

void NetUnitsList::Insert(NetUnit *s){
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)
		if (d==s) { 
			if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return;
			}
	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	num_units++;
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void NetUnitsList::Delete(NetUnit *s){
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	NetUnit *d, *p;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;
			num_units--;
			break;
			}
		p=d; 
	 	}

	for(d=root; d!=NULL; d=d->next)	if (d->parent==s) d->parent=NULL;

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void NetUnitsList::Delete(oid id){
	NetUnit *u;
	u=getUnitById(id);
	if (u) Delete(u);
}

NetUnit* NetUnitsList::getUnit(char *name){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (!strcmp(d->name, name)) {
			if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return d;
		}
 	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return NULL;
}

NetUnit* NetUnitsList::getUnitById(oid id){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->id==id) {
			if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return d;
		}
 	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return NULL;
}

void NetUnitsList::DeletePolicyElsewhere(Policy *p){
	NetUnit *d;
	policy_data *cpd;
	tries_lock++;
	int err=pthread_rwlock_trywrlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	for (d=root; d!=NULL; d=d->next) {
		for (cpd=d->ap.root; cpd!=NULL; cpd=cpd->next) if (cpd->policy==p) d->ap.Delete(p);
		for (cpd=d->fp.root; cpd!=NULL; cpd=cpd->next) if (cpd->policy==p) d->fp.Delete(p);
	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

unsigned NetUnitsList::setMon(void *value, oid id, char *name){
	int res=0;
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (id && d->id==id) { d->monitor=value; res=1; }
		if (name && !strcmp(name, d->name)) { d->monitor=value; res=1; }
		if (!id && !name) { d->monitor=value; res=1; }
	 	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return res;
	}

void NetUnitsList::showMon(FILE *out, void *to,unsigned action){ //action: 1-for show, 0- for config
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	NetUnit *d;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->monitor && d->monitor==to && d->name) {
			if (action) fprintf(out, "%s(%06X) ", d->name, d->id);	
			else fprintf(out, "monitor unit %06X\n", d->id);
			}
	 	}

	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return;
	}

void NetUnitsList::listPasswordsHtml(Connection *conn){
	int err=pthread_rwlock_tryrdlock(rwlock); 
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	NetUnit *d;
	NetUnit_user *u;
	for(d=root; d!=NULL; d=d->next)	
		if (d->type==NETUNIT_USER) {
			u = (NetUnit_user*)d;
			if (u->password) fprintf(conn->stream_w, "%s:%s\n", d->name, crypt(u->password, "$1$"));
			}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return;
	}

/////////////////////////////////////////////////////////////////////////
// NetUnit_cluster class
NetUnit_cluster::NetUnit_cluster() : NetUnit(NETUNIT_CLUSTER) {
	root=last=NULL;
	references=0;
	lock=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t));
	pthread_mutex_init(lock, NULL);
	}

NetUnit_cluster::~NetUnit_cluster(){
	pthread_mutex_destroy(lock);
	aFree(lock);
	}

void NetUnit_cluster::Add(struct in_addr ip){
	pthread_mutex_lock(lock);
	NetUnit_host *d, *s;
	for(d=root; d!=NULL; d=d->next)
		if (d->ip.s_addr==ip.s_addr) { 
			pthread_mutex_unlock(lock);
			return;
			}
	s=(NetUnit_host *)aMalloc(sizeof(NetUnit_host));
	s->ip=ip;
	s->next=NULL;

	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	references++;
	pthread_mutex_unlock(lock);
	}

void NetUnit_cluster::Remove(struct in_addr ip){
	pthread_mutex_lock(lock);
	NetUnit_host *d, *p;
	for(d=root; d!=NULL; d=d->next)	{
		if (d->ip.s_addr==ip.s_addr) {
			if (d==root && d==last ) root=last=NULL;
			else if (d==root) root=d->next;
			else if (d==last) { last=p; last->next=NULL; }
			else p->next=d->next;

			references--;
			aFree(d);
			break;
			}
		p=d; 
	 	}
	pthread_mutex_unlock(lock);
	}

match NetUnit_cluster::Check(struct ip *ipc) {
	unsigned rs=0, rd=0;
	NetUnit_host *d;
	pthread_mutex_lock(lock);
	for (d=root; d!=NULL; d=d->next) {
		if (ntohl(d->ip.s_addr)==ntohl(ipc->ip_src.s_addr)) rs++;
		if (ntohl(d->ip.s_addr)==ntohl(ipc->ip_dst.s_addr)) rd++;
		}
	pthread_mutex_unlock(lock);
	if (rs && rd) return MATCH_BOTH;
	else if (rs) return MATCH_SRC;
	else if (rd) return MATCH_DST;
	else return MATCH_NONE;
	}

void NetUnit_cluster::unit2tree(IPTree *iptree,unsigned flag) {
	for(NetUnit_host *h=root;h!=NULL;h=h->next)
		iptree->addr2tree(h->ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////
// NetUnit_group class
NetUnit_group::NetUnit_group() : NetUnit(NETUNIT_GROUP){
	references=0;
	}

match NetUnit_group::Check(struct ip *ipc) {
	NetUnit *t;
	unsigned rs=0, rd=0;
	match m;
	
	for (t=Units.root; t!=NULL; t=t->next)	
		if (t->parent==this) {
			m=t->Check(ipc);
			if (m==MATCH_BOTH) { rs++; rd++; }
			else if (m==MATCH_SRC) rs++;
			else if (m==MATCH_DST) rd++;
			}
				
	if (rs && rd) return MATCH_BOTH;
	else if (rs) return MATCH_SRC;
	else if (rd) return MATCH_DST;
	else return MATCH_NONE;
	}

/////////////////////////////////////////////////////////////////////////
// NetUnit_host class
NetUnit_host::NetUnit_host() : NetUnit(NETUNIT_HOST) {
	ip.s_addr=0;
	next=NULL;
	}

match NetUnit_host::Check(struct ip *ipc) {
	unsigned rs=0, rd=0;
	if (ntohl(ip.s_addr)==ntohl(ipc->ip_src.s_addr)) rs++;
	if (ntohl(ip.s_addr)==ntohl(ipc->ip_dst.s_addr)) rd++;

	if (rs && rd) return MATCH_BOTH;
	else if (rs) return MATCH_SRC;
	else if (rd) return MATCH_DST;
	else return MATCH_NONE;
	}

void NetUnit_host::unit2tree(IPTree *iptree,unsigned flag) {
	iptree->addr2tree(ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
NetUnit_net::NetUnit_net() : NetUnit(NETUNIT_NET) {
	ip.s_addr=0;
	mask.s_addr=INADDR_BROADCAST;
	}

match NetUnit_net::Check(struct ip *ipc){
	unsigned rs=0, rd=0;
	
	//aDebug(DEBUG_DS_IP, "netunit_net chk <%s>: src:%lu, dst:%lu, net1:%lu, net2:%lu\n", name, ntohl(ipc->ip_src.s_addr), ntohl(ipc->ip_dst.s_addr), ntohl(ip.s_addr), ntohl(ip.s_addr)+INADDR_BROADCAST-ntohl(mask.s_addr) );

	if (ntohl(ip.s_addr)<ntohl(ipc->ip_src.s_addr) && ntohl(ipc->ip_src.s_addr)<(ntohl(ip.s_addr)+INADDR_BROADCAST-ntohl(mask.s_addr))) rs++;
	if (ntohl(ip.s_addr)<ntohl(ipc->ip_dst.s_addr) && ntohl(ipc->ip_dst.s_addr)<(ntohl(ip.s_addr)+INADDR_BROADCAST-ntohl(mask.s_addr))) rd++;

	if (rs && rd) return MATCH_BOTH;
	else if (rs) return MATCH_SRC;
	else if (rd) return MATCH_DST;
	else return MATCH_NONE;
	}

void NetUnit_net::unit2tree(IPTree *iptree,unsigned flag) {
	unsigned m=0;
	unsigned long tmp=mask.s_addr;
	
	for(unsigned i=0;i<32;i++) {
        	if(tmp&1) m++;
		tmp=tmp>>1;
	}
        iptree->addr2tree(ip.s_addr,m,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
// NetUnit_user class
NetUnit_user::NetUnit_user() : NetUnit(NETUNIT_USER) {
        ip.s_addr=0;
        next=NULL;
	real_name=password=NULL;
        }

match NetUnit_user::Check(struct ip *ipc) {
        unsigned rs=0, rd=0;
	if(ip.s_addr==0) return MATCH_NONE;
        if (ntohl(ip.s_addr)==ntohl(ipc->ip_src.s_addr)) rs++;
        if (ntohl(ip.s_addr)==ntohl(ipc->ip_dst.s_addr)) rd++;

        if (rs && rd) return MATCH_BOTH;
        else if (rs) return MATCH_SRC;
        else if (rd) return MATCH_DST;
        else return MATCH_NONE;
        }

void NetUnit_user::unit2tree(IPTree *iptree,unsigned flag) {
	if (ip.s_addr) iptree->addr2tree(ip.s_addr,32,(NetUnit*)this,flag);
}
/////////////////////////////////////////////////////////////////////////
int cUnit(Connection *conn, char *param[], int no_flag){
	NetUnit *u;
	int i=2;
	u_char type=0;
	char *type_name;
	if (!strcmp(param[2], "name")) u=Units.getUnit(param[3]);
	else if (!strcmp(param[2], "oid")) u=Units.getUnitById(strtol(param[3], NULL, 16));
	else { aParse(conn, "unit %s name/oid unspecified\n", param[1]); return PARSE_OK; }


	for(u_char j=0;j<NETUNIT_TYPES_NUM;j++) {
		if (strcasecmp(param[1], netunit_type_name[j])==0) {
			type=j;
			type_name=netunit_type_name[type];
			break;
		}
	}
	if(!type) {
		aParse(conn, "unit type unknown\n");
		return PARSE_OK;
	}
	
	
	NetUnit_host *host=(NetUnit_host*)u;
	NetUnit_user *user=(NetUnit_user*)u;
	NetUnit_net *net=(NetUnit_net*)u;
	NetUnit_cluster *cluster=(NetUnit_cluster*)u;
	NetUnit_group *group=(NetUnit_group*)u;

	if (!u && !no_flag) {
                switch(type) {
                        case NETUNIT_HOST:
                                host= new NetUnit_host();
                                u=(NetUnit*)host;
                                break;
                        case NETUNIT_USER:
                                user= new NetUnit_user();
                                u=(NetUnit*)user;
                                break;
                        case NETUNIT_NET:
                                net= new NetUnit_net();
                                u=(NetUnit*)net;
                                break;
                        case NETUNIT_GROUP:
                                group= new NetUnit_group();
				u=(NetUnit*)group;
                                break;
                        case NETUNIT_CLUSTER:
                                cluster= new NetUnit_cluster();
                                u=(NetUnit*)cluster;
                                break;
                        default:
                                return PARSE_OK; // we already sure we have correct type
                }
		Units.Insert(u);
                aParse(conn, "unit %s %06X created\n", type_name, u->id);
        } else if (!u && no_flag) {
                aParse(conn, "unit %s does not exist\n", type_name);
                return PARSE_OK;
        } else if (u && no_flag) {
                aParse(conn, "unit %s %06X deleted\n", type_name, u->id);
                Units.Delete(u);
                delete u;
                return PARSE_OK;
        }

        while (param[i]!=empty) {
                no_flag=0;
                if (strcasecmp(param[i], "no")==0) {
                        no_flag=1;
                        i++;
                }

                if (strstr(param[i], "sys-")==param[i]) { // possibly sys-* here
                        SetSysPolicy(param[i], u, conn);
                        i-=1;

                } else if (strcasecmp(param[i], "acct-policy")==0) {
                        PolicyAdd(u, &i, POLICY_ACCT, conn, param, no_flag);

                } else if (strcasecmp(param[i], "fw-policy")==0) {
                        PolicyAdd(u, &i, POLICY_FW, conn, param, no_flag);

                } else if (strcasecmp(param[i], "ip")==0) {
                        u->unit2trees(REMOVE);
                        switch(type) {
                                case NETUNIT_CLUSTER:
                                        struct in_addr addr;
                                        inet_aton(param[i+1], &addr);
                                        if(no_flag) cluster->Remove(addr);
                                        else cluster->Add(addr);
                                        break;
                                case NETUNIT_HOST:
                                        if(no_flag) host->ip.s_addr=0;
                                        else inet_aton(param[i+1], &host->ip);
                                        break;
                                case NETUNIT_USER:
                                        if(no_flag) user->ip.s_addr=0;
                                        else inet_aton(param[i+1], &user->ip);
                                        break;
                                case NETUNIT_NET:
                                        if(no_flag) net->ip.s_addr=0;
                                        else inet_aton(param[i+1], &net->ip);
                                        break;
                                default:
                                        aParse( conn, "IP for %s not supported\n", type_name);
                                        break;
                        }
                        u->unit2trees(ADD);
                        aParse(conn, "%s %06X ip set\n", type_name, u->id);

                } else if (type == NETUNIT_NET && strcasecmp(param[i], "mask")==0) {
                                u->unit2trees(REMOVE);
                                inet_aton(param[i+1], &net->mask);
                                u->unit2trees(ADD);
                                aParse(conn, "net %06X mask set: %s\n", u->id, inet_ntoa(net->mask));

                } else if (strcasecmp(param[i], "name")==0) {
                        u->setName(param[i+1]);
                        aParse(conn, "%s %06X name set: %s\n", type_name, u->id, u->name);

                } else if (strcasecmp(param[i], "email")==0) {
                        aFree(u->email);
                        if(no_flag) {
                                u->email=NULL;
                                aParse(conn, "%s %06X email cleared\n", type_name, u->id);
                        } else {
                                u->email=set_string(param[i+1]);
                                aParse(conn, "%s %06X email set: %s\n", type_name, u->id, u->email);
                        }

                } else if (strcasecmp(param[i], "oid")==0) {
                        u->id=strtol(param[i+1], NULL, 16);
                        aParse(conn, "%s %06X oid set\n", type_name, u->id);

                } else if (strcasecmp(param[i], "ds-list")==0) {
                        u->unit2trees(REMOVE);
                        DSListAdd(u, &i, conn, param, no_flag);
                        u->unit2trees(ADD);

                } else if (strcasecmp(param[i], "parent")==0) {
                        NetUnit_group *gr;
                        if (no_flag) {
                                u->parent=NULL;
                                aParse(conn, "%s %06X parent cleared\n", type_name, u->id);
                        } else {
                                gr=(NetUnit_group*)Units.getUnit(param[i+1]);
                                if (gr && gr->name) {
                                        u->parent=gr;
                                        aParse(conn, "%s %06X added to group %s\n", type_name, u->id, gr->name);
                                }
                                else aParse(conn, "group %s for %s %06X not exist\n", param[i+1], type_name, u->id);
                        }

                } else if (strcasecmp(param[i], "no-local-pass")==0) {
                        if (no_flag) u->nlp=0; else u->nlp=1;
			aParse(conn, "%s %06X no-local-pass flag set: %u\n", type_name, u->id, u->nlp);
                        i-=1;

                } else if (type == NETUNIT_USER && strcasecmp(param[i], "password")==0) {
                        if(user->password) aFree(user->password);
                        if(no_flag) {
                                user->password=NULL;
                                aParse(conn, "Unit user %06X password cleared\n", u->id);
                        } else {
                                user->password=set_string(param[i+1]);
                                aParse(conn, "user %06X password set: %s\n", u->id, user->password);
                        }

                } else if (type == NETUNIT_USER && strcasecmp(param[i], "real_name")==0) {
                        if(user->real_name) aFree(user->real_name);
                        if(no_flag) {
                                user->real_name=NULL;
                                aParse(conn, "Unit user %06X real_name cleared\n", u->id);
                        } else {
                                user->real_name=set_string(param[i+1]);
                                aParse(conn, "user %06X real_name set: %s\n", u->id, user->real_name);
                        }

                } else {
                        aParse(conn, "%s %06X command '%s' unknown\n", type_name, u->id, param[i]);
                        i--;
                }

                i=i+2;
        }
        return PARSE_OK;
}
/////////////////////////////////////////////////////////////////////////
unsigned cShowUnits(Connection *conn, char *param[]){	   // "show units"
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d;
	char *parent_name;

	if (!strcasecmp(param[2], "syspolicy")) {
		fprintf(conn->stream_w, "%6s | %10s | %10s \n", "OID", "NAME", "SYSPOLICY");
		for (d=Units.root; d!=NULL; d=d->next) {
			fprintf(conn->stream_w, "%06X | %10s | ", d->id, d->name?d->name:"<\?\?>");
			switch (d->sys_policy) {
				case SP_NONE: fprintf(conn->stream_w, "none\n"); break;
				case SP_ALLOW: fprintf(conn->stream_w, "allow\n"); break;
				case SP_DENY_NONE: fprintf(conn->stream_w, "deny\n"); break;
				case SP_DENY_QUOTA: fprintf(conn->stream_w, "deny_quota\n"); break;
				case SP_DENY_AUTH: fprintf(conn->stream_w, "deny_auth\n"); break;
				default: fprintf(conn->stream_w, "unknown (%d)\n", d->sys_policy); break;
				}
			}
		}

	else if (!strcasecmp(param[2], "email")) {
		fprintf(conn->stream_w, "%6s | %10s | %10s \n", "OID", "NAME", "E-MAIL");
		for (d=Units.root; d!=NULL; d=d->next) {
			if (d->email) {
				fprintf(conn->stream_w, "%06X | %10s | %s\n", d->id, d->name?d->name:"<\?\?>", d->email);
			}
                }

	} else {	
		unsigned short show_type=0; //show units <type>
		unsigned short show_active=0; //show units users active
		for(unsigned i=0;i<6;i++) {
			if (!strcasecmp(param[2], netunit_type_name[i])){
				show_type = i;	
				break;
			}
		}
		if (!strcasecmp(param[3], "active")) show_active=1;

		fprintf(conn->stream_w, "%8s | %6s | %10s | %3s | %10s | %15s |%-20s\n", "TYPE", "OID", "NAME", "NLP", "PARENT", "EMAIL", "PARAMS");
		for(d=Units.root; d!=NULL; d=d->next)	{
			if(show_type && d->type!=show_type) continue;
			if(show_active) {
			 	if(d->type!=NETUNIT_USER) continue; 
				NetUnit_user *h=(NetUnit_user*)d;
				if(h->ip.s_addr==0) continue;
			}

			if (d->parent && d->parent->name) parent_name=d->parent->name; else parent_name="<>";
			fprintf(conn->stream_w, "%8s | %06X | %10s | %3s | %10s | %15s | ", netunit_type_name[d->type], d->id, d->name?d->name:"<\?\?>", d->nlp?" + ":"", parent_name,d->email?d->email:"");

			switch (d->type) {
			case NETUNIT_HOST: {
				NetUnit_host *h;
				h = (NetUnit_host*)d;
				fprintf(conn->stream_w, "IP: %-12s\n",inet_ntoa(h->ip));
				} break;
			case NETUNIT_USER: {
				NetUnit_user *h;
				h = (NetUnit_user*)d;
                                fprintf(conn->stream_w, "IP: %-12s\n", inet_ntoa(h->ip));
				} break;
			case NETUNIT_CLUSTER: { 
				NetUnit_cluster *h;
				NetUnit_host *t;
				h = (NetUnit_cluster*)d;
                                fprintf(conn->stream_w, "IPs: ");
				for (t=h->root; t!=NULL; t=t->next)	fprintf(conn->stream_w, "%s ", inet_ntoa(t->ip));
				fprintf(conn->stream_w, "\n");
				} break;
			case NETUNIT_GROUP: {
				NetUnit_group *h;
				NetUnit *t; 
				h = (NetUnit_group*)d;
				if(h->email) fprintf(conn->stream_w, "Sub: ");
				for (t=Units.root; t!=NULL; t=t->next)	if (t->parent==d && t->name) fprintf(conn->stream_w, "%s ", t->name);
				fprintf(conn->stream_w, "\n");
				} break;
			case NETUNIT_NET: {
				NetUnit_net *h;
				h = (NetUnit_net*)d;
				fprintf(conn->stream_w, "%s:", inet_ntoa(h->ip));
                                fprintf(conn->stream_w, "%s\n", inet_ntoa(h->mask));
				} break;
			default: 
				fprintf(conn->stream_w, "%8s | %06X\n", "<\?\?>", d->id);
				break;
			}
		}
	}

	pthread_rwlock_unlock(Units.rwlock);
	return PARSE_OK;
}
/////////////////////////////////////////////////////////////////////////
unsigned cShowUnitsList(Connection *conn, int isfull, char *p1, char *p2){ // "show list {full}"
	pthread_rwlock_rdlock(Units.rwlock);
	NetUnit *d, *d2=NULL;
	policy_data *cpd;
	char *parent_name, *t_str;
	t_str=(char *)aMalloc(32);

	if (p1!=empty && p2!=empty) {
		if (!strcasecmp(p1, "name")) d2=Units.getUnit(p2);
		else if (!strcasecmp(p1, "oid")) d2=Units.getUnitById(strtol(p2, NULL, 16));
	}

	for (d=Units.root; d!=NULL; d=d->next) {
  		
  		if (p1!=empty && p2!=empty && d!=d2) continue;
  		
  		fprintf(conn->stream_w, "OID: %06X Name: %-10s Type: %-8s", d->id, d->name, netunit_type_name[d->type]);
		if (d->nlp) fprintf(conn->stream_w, " NLP"); 
		if (d->parent && d->parent->name) parent_name=d->parent->name; else parent_name="<>";
		fprintf(conn->stream_w, " Parent: %-10s", parent_name);

		if (d->dsl_root) {
			fprintf(conn->stream_w, " DS-list: ");
			for (DSList *dsl=d->dsl_root;dsl!=NULL;dsl=dsl->next)
				 fprintf(conn->stream_w, "%u ", dsl->id);
		}

		fprintf(conn->stream_w, "\n");
				
		fprintf(conn->stream_w, " %-11s", "SYST policy");
		if (d->sys_policy==SP_NONE) fprintf(conn->stream_w, " is not set\n");
		else {
			char *buf=NULL; 
			buf=ShowSysPolicy(buf, d->sys_policy,d->sys_policy_perm, d->sys_policy_name);    
			fprintf(conn->stream_w, ": %s\n", buf);
			aFree(buf);
		} 

		fprintf(conn->stream_w, " %-11s", "  FW policy");
		if (d->fp.root)	fprintf(conn->stream_w, ": %-6s %-10s %-12s %-12s\n",  "OID", "NAME", "CHECK", "MATCH");
		else fprintf(conn->stream_w, " list is empty\n");

		for (cpd=d->fp.root; cpd!=NULL; cpd=cpd->next) {
			fprintf(conn->stream_w, "%-10s %s%s %06X %-10s %-12qu %-12qu\n", "", cpd->brk?"%":" ", cpd->inv?"!":" ", cpd->policy->id, cpd->policy->name?cpd->policy->name:"<\?\?>", cpd->check, cpd->match);
		}
						  
		fprintf(conn->stream_w, " %-11s", "ACCT policy");
		if (d->ap.root)	fprintf(conn->stream_w, ": %-6s %-10s %-12s %-12s\n",  "OID", "NAME", "CHECK", "MATCH");
		else fprintf(conn->stream_w, " list is empty\n");

		for (cpd=d->ap.root; cpd!=NULL; cpd=cpd->next) {
			fprintf(conn->stream_w, "%-10s %s%s %06X %-10s %-12qu %-12qu\n", "", cpd->brk?"%":" ", cpd->inv?"!":" ", cpd->policy->id, cpd->policy->name?cpd->policy->name:"<\?\?>", cpd->check, cpd->match);
			if (isfull) {
				timeU2T(cpd->flow.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", cpd->flow.from?t_str:"--.--.---- --:--:--",  "flow", cpd->flow.in, cpd->flow.out);
				timeU2T(cpd->t.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", t_str,  "total", cpd->t.in, cpd->t.out);
				timeU2T(cpd->m.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", t_str,  "month", cpd->m.in, cpd->m.out);
				timeU2T(cpd->w.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", t_str,  "week", cpd->w.in, cpd->w.out);
				timeU2T(cpd->d.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", t_str,  "day", cpd->d.in, cpd->d.out);
				timeU2T(cpd->h.from, t_str); fprintf(conn->stream_w, " %-19s %-10s in: %-12qu out: %-12qu\n", t_str,  "hour", cpd->h.in, cpd->h.out);
			}			
		}

		fprintf(conn->stream_w, "\n");
	}
	aFree(t_str);
	pthread_rwlock_unlock(Units.rwlock);
	return PARSE_OK;
}
/////////////////////////////////////////////////////////////////////////
void DSListAdd(NetUnit *u, int *i, Connection *conn, char *param[32], int no_flag){
	char *res, *ptr;

	aDebug(DEBUG_DS_IP, "NetUnit: DSL: adding DS list \"%s\"\n", param[(*i)+1]);

	int k=strlen(param[(*i)+1]);
	unsigned f;

	for (ptr=param[(*i)+1]; (ptr-param[(*i)+1])<k ;ptr=res+1){
		if(no_flag && ( !strcmp(ptr,"all") || !strcmp(ptr,"*")) ) {
			aParse(conn, "ds-list cleared for unit %06X\n", u->id);
			DSList *ptr,*dsl=u->dsl_root;
			while(dsl) {
				ptr=dsl;
				dsl=dsl->next;
				aFree(ptr);
			}
			u->dsl_root=NULL;
			return;
		}
		f=strtol(ptr, &res, 10);
		if (no_flag) {
			aParse(conn, "removing DS %u from list for unit %06X\n", f, u->id);
			u->clearDSList(f);
		}
		else {
			aParse(conn, "adding DS %u to list for unit %06X\n", f, u->id);
			u->setDSList(f);
		}
	}
}
/////////////////////////////////////////////////////////////////////////
