/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_datasource.c,v 1.44.2.15 2004/07/12 08:17:21 jura Exp $ */

#include "netams.h"
#include "flowmodule.h"

#ifdef LINUX
	extern "C" {
	#ifndef IPTBL_NONE
		#include <linux/netfilter.h>
		#include <libipq.h>
	#endif
	
	// Added by Marat Phattakhoff a.k.a. unReal http://marat.tps.uz/, admin@tps.uz
        #ifndef ETHERTYPE_VLAN
                #define ETHERTYPE_VLAN 0x8100 /* IEEE 802.1Q VLAN tagging */
        #endif
        #ifndef ETHERTYPE_IP
                #define ETHERTYPE_IP 0x0800  /* IP protocol */
        #endif
        #ifndef ETHERMTU
                #define ETHERMTU 1500
        #endif
	#undef DLT_LINUX_SLL
        #ifndef DLT_LINUX_SLL
                /* Based on TCPDUMP code */
                #define DLT_LINUX_SLL 113
                #define SLL_HDR_LEN 16 /* total header length */
                #define SLL_ADDRLEN 8 /* length of address field */

                struct sll_header {
                        u_int16_t sll_pkttype; /* packet type */
                        u_int16_t sll_hatype; /* link-layer address type */
                        u_int16_t sll_halen; /* link-layer address length */
                        u_int8_t sll_addr[SLL_ADDRLEN]; /* link-layer address */
                        u_int16_t sll_protocol; /* protocol */
                };
        #endif
	}
#endif

void *sDataSource(void *s);
// void *sDSLookup(void *s);
void sDSCancel(void *);
void sDS_RuleIp(int flag, Service *s, unsigned number, char *rule); // flag==0 eq ADD, ==1 eq REMOVE

void sDSHookInstall(Service *s);
void sDSHookRemove(Service *s);

//Message *sDSLookupConstructMsg(char prefix, oid ds, oid unit, oid ap, pstat s);
unsigned short sDSsystem(dsUlist *start,NetUnit *u);
unsigned short sDSfw(NetUnit *u, struct ip *packet, match mf);
void sDSacct(NetUnit *u, struct ip *packet, match mf, struct timeval *tv, unsigned long len);

void pcap_callback(unsigned char *args, const struct pcap_pkthdr *hdr, const unsigned char *packet);
/////////////////////////////////////////////////////////////////////////////////////////
//below obtained from flow-tools
#define FT_SO_RCV_BUFSIZE      (4*1024*1024) /* UDP recv socket buffer size */
int bigsockbuf(int fd, int dir, int size);
//////////////////////////////////////////////////////////////////////////
void sDSInit(Service *s){
	s->cfg = (ServiceDS_cfg*)aMalloc(sizeof(ServiceDS_cfg));

	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	cfg->type=PT_UNKNOWN;
	cfg->root=NULL;
	cfg->src_cmd=NULL;
	cfg->divert=0;
	cfg->s_addr=0;
	cfg->port=0;
	cfg->id=s->instance;
	cfg->delay=0; cfg->skewdelay=0;
	cfg->total_packets=0;
	cfg->seq_id=0;
	
	cfg->IPtree = new IPTree;
	//organize tree here
        Units.tries_lock++;
        int err=pthread_rwlock_tryrdlock(Units.rwlock);
        if (err==EBUSY) { Units.tries_lock_failed++; pthread_rwlock_rdlock(Units.rwlock); }

        for (NetUnit *u=Units.root; u!=NULL; u=u->next)
                if(u->checkDSList(s->instance)) u->unit2tree(cfg->IPtree,1);

        if (err!=EDEADLK) pthread_rwlock_unlock(Units.rwlock);

	pthread_create(&(s->t_id), NULL, &sDataSource, s);
}

//////////////////////////////////////////////////////////////////////////
void sDSProcessCfg(char *param[32], Connection *conn, unsigned no_flag){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)conn->service->cfg;

	if (!strcasecmp(param[0], "type")) {
		if (!strcasecmp(param[1], "ip-traffic")) {
			aParse(conn, "data-source type is set to ip-traffic\n");
			cfg->type=PT_IP_TRAFFIC;
		}
		else if (!strcasecmp(param[1], "netflow")) {
			aParse(conn, "data-source type is set to cisco NetFlow\n");
			cfg->type=PT_NETFLOW_TRAFFIC;
		}
		else if (!strcasecmp(param[1], "libpcap")) {
			aParse(conn, "data-source type is set to libpcap\n");
			cfg->type=PT_LIBPCAP_TRAFFIC;
		}
		else if (!strcasecmp(param[1], "flowmodule")) {
			aParse(conn, "data-source type is set to flow_module\n");
			cfg->type=PT_FM_TRAFFIC;
		}
		else if (!strcasecmp(param[1], "mailstat")) {
			aParse(conn, "data-source type is set to mailstat\n");
			cfg->type=PT_MAIL_STAT;
		}
		else aParse(conn, "data-source type is invalid\n");
	}
	else if (!strcasecmp(param[0], "rule")) {
		unsigned number=atol(param[1]);
		ds_rule *t=NULL, *tt;

		if (cfg->type==PT_MAIL_STAT) return;

		for (ds_rule *r=cfg->root; r!=NULL; r=r->next) {
			tt=r;
			if (r->number==number) t=r;
		}

		if (no_flag && t) {
			aParse(conn, "rule %d removed\n", number);
			sDS_RuleIp(1, conn->service, number, t->rule_str);
			aFree(t->rule_str);
			t->number=0;
		} //no actual chain removal!!!!!!!
		else if (!no_flag && t) {
			sDS_RuleIp(1, conn->service, number, t->rule_str);
			aFree(t->rule_str);
			t->rule_str=set_string(param[2]);
			aParse(conn, "rule %d replaced to %s\n", number, t->rule_str);
			sDS_RuleIp(0, conn->service, number, t->rule_str);
		}
		else if (!t) {
			t=new ds_rule;
			t->rule_str=set_string(param[2]);
			t->number=number;
			t->next=NULL;
			if (!cfg->root) cfg->root=t; else tt->next=t;
			aParse(conn, "rule %d set to %s\n", number, t->rule_str);
			sDS_RuleIp(0, conn->service, number, t->rule_str);
		}
	}
	else if (!strcasecmp(param[0], "source")) {
		if (cfg->type==PT_MAIL_STAT) {
			if (!strcasecmp(param[1], "sendmail")) {
				cfg->src_cmd=set_string("sendmail");
				aParse(conn, "source is set to sendmail\n");
			}
			else aParse(conn, "source is invalid\n");
			return;
		}

		if (!strcasecmp(param[1], "tee")) {
			unsigned port=atol(param[2]);
			if (!port) port=199;
			aParse(conn, "source is set to tee:%u\n", port);
			cfg->src_cmd=set_string("tee");
			cfg->port=port;
			cfg->divert=0;
		}
		else if (!strcasecmp(param[1], "divert")) {
			unsigned port=atol(param[2]);
			if (!port) port=199;
			aParse(conn, "source is set to divert:%u\n", port);
			cfg->src_cmd=set_string("divert");
			cfg->port=port;
			cfg->divert=1;
		}
		else if (INADDR_NONE!=inet_addr(param[1])) {
			struct in_addr ina;
			inet_aton(param[1], &ina);
			unsigned port=atol(param[2]);
			if (!port) port=20001;
			aParse(conn, "source is listening %s:%u\n", inet_ntoa(ina), port);
			cfg->src_cmd=set_string(param[1]);
			cfg->port=port;
			cfg->s_addr=ina.s_addr;
		}
		else if (cfg->type==PT_LIBPCAP_TRAFFIC) {
			aParse(conn, "source is listening interface %s\n", param[1]);
			cfg->src_cmd=set_string(param[1]);
			cfg->port=0;
		}
		else aParse(conn, "source is invalid\n");
	}
	else aParse(conn, "unknown data-source command: %s\n", param[0]);
}
//////////////////////////////////////////////////////////////////////////
void *sDataSource(void *ss){
	Service *s=(Service*)ss;

	aLog(D_INFO, "service data-source:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sDSCancel, s);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	s->t_id=pthread_self();

	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
//	cfg->sLookup = new Service("ds-lookup", s->instance);
//	Services.Insert(cfg->sLookup);

//	pthread_create(&(cfg->sLookup->t_id), NULL, sDSLookup, s);

	// now we should sleep before starting data source service exec
	s->Sleep();

	sDSHookInstall(s);

	socklen_t size_ds = sizeof(cfg->ds);
	unsigned len;
	struct timeval start, stop;
	double d_start=0, d_stop=0;
	unsigned char *packet = (unsigned char*) aMalloc(MAX_PKT_SIZE);
	struct ip *ip;
	int i,j; 
	for (i=0; i<10; i++) { cfg->pc[i]=cfg->bc[i]=0; }

	gettimeofday(&start, NULL); 

	switch (cfg->type) {


		case PT_IP_TRAFFIC:	// here we are assume that arrived data has an IP format
			{
			unsigned process_result;
			while (1) {

				#ifdef LINUX
				#ifndef IPTBL_NONE
					ipq_read(cfg->ipq, packet, MAX_PKT_SIZE, 0);
					gettimeofday(&start, NULL);
					if (ipq_message_type(packet)==IPQM_PACKET){
						ipq_packet_msg_t *m=ipq_get_packet(packet);
						ip = (struct ip*) m->payload;
						len = ntohs(ip->ip_len);
						process_result=sDS_AcctIp(s, ip, &start, len, 0);
						if (process_result) ipq_set_verdict(cfg->ipq, m->packet_id, NF_ACCEPT, 0, NULL);
						else ipq_set_verdict(cfg->ipq, m->packet_id, NF_DROP, 0, NULL);
						}
				#else /* IPTBL_NONE = yes*/
					aLog(D_ERR, "You are trying to run IPQ version on system without libipq support\n");
				#endif

				#else // FreeBSD
					unsigned skip_fw_check;
					len=recvfrom(cfg->socketid, packet, MAX_PKT_SIZE, 0, (struct sockaddr *)&(cfg->ds), &size_ds);
					gettimeofday(&start, NULL);
					ip = (struct ip*) packet;
					if (IN_MULTICAST((ntohl((ip->ip_dst).s_addr))) && cfg->ds.sin_addr.s_addr == 0) {
						setsockopt(cfg->socketid, IPPROTO_IP, IP_MULTICAST_IF, &(ip->ip_src), sizeof(ip->ip_src));
						skip_fw_check=1; // we will permit all multicasts
					} else //if we diverting skip_fw_check=!cfg->divert=!1=0
						skip_fw_check=!cfg->divert;
					len = ntohs(ip->ip_len);
					process_result=sDS_AcctIp(s, ip, &start, len, skip_fw_check);
					if (process_result && cfg->divert) sendto(cfg->socketid, packet, len, 0, (struct sockaddr *)&(cfg->ds), size_ds);

				#endif
				gettimeofday(&stop, NULL);
				d_start=start.tv_sec+ (double)start.tv_usec/1000000;
				d_stop=stop.tv_sec+ (double)stop.tv_usec/1000000;
				cfg->total_packets++;
				cfg->delay+=(d_stop-d_start);
				cfg->skewdelay = long(cfg->skewdelay*0.98 + (long)(1000000*(d_stop-d_start))*0.02); 
				i=start.tv_sec%10;
				cfg->pc[i]++; cfg->bc[i]+=len;
				j=i+1; if (j==10) j=0;
				cfg->pc[j]=0; cfg->bc[j]=0;
				aDebug(DEBUG_DS_IP, "ip packet %s in %d mcsec\n\n", process_result?"PASS":"DROP", (unsigned)(1000000*(d_stop-d_start)));
				}
			}
			break;

		case PT_NETFLOW_TRAFFIC:
			{
			while(1) {

				len=recvfrom(cfg->socketid, packet, MAX_PKT_SIZE, 0, (struct sockaddr *)&(cfg->ds), &size_ds);
				if ((cfg->ds.sin_addr.s_addr!=cfg->s_addr) || !cfg->s_addr) { aDebug(DEBUG_NF, "NetFlow packet from unknown source: %s\n", inet_ntoa(cfg->ds.sin_addr)); continue; }
				gettimeofday(&start, NULL);
				ip = (struct ip*) packet;
				sDS_AcctNetFlow(s, ip, &start);
				gettimeofday(&stop, NULL);
				d_start=start.tv_sec+ (double)start.tv_usec/1000000;
				d_stop=stop.tv_sec+ (double)stop.tv_usec/1000000;
				cfg->total_packets++;
				cfg->delay+=(d_stop-d_start);
				cfg->skewdelay = long(cfg->skewdelay*0.8 + (d_stop-d_start)*0.2); 
				aDebug(DEBUG_NF, "flow packet processed in %d mcsec\n\n", (unsigned)(1000000*(d_stop-d_start)));
				}

			}
			break;

		case PT_LIBPCAP_TRAFFIC:
			{
			register const struct ether_header *ep;
#ifdef LINUX
			register const struct sll_header   *sllp;
#endif
			u_short ether_type;
			u_int offset;
			int status;
			unsigned char *packet;

			if(!cfg->pcap->h) break;
			while(1) {
				status=pcap_dispatch(cfg->pcap->h, 1, pcap_callback, (unsigned char*)cfg->pcap);
				if(status <1) {
                        		if(status==1)
                                		aLog(D_WARN, "pcap dispatch error: %s\n", pcap_geterr(cfg->pcap->h));
                        		continue;
                		}
				packet=cfg->pcap->pkt;
				gettimeofday(&start, NULL);
				switch(cfg->pcap->datalink) {
		               	  case DLT_EN10MB:
                        		ep = (struct ether_header *)packet;
                        		ether_type = ntohs(ep->ether_type);
                        		packet += ETHER_HDR_LEN;
					break;
#ifdef LINUX
				  case DLT_LINUX_SLL:
                               		if(cfg->pcap->pkthdr.len < SLL_HDR_LEN) /* Oops */
                                     		continue;
                                	sllp = (struct sll_header *)packet;
                                	ether_type = ntohs(sllp->sll_protocol);
                                	if(ether_type <= ETHERMTU) /* Oops */
                                 	       continue;
                                	packet += SLL_HDR_LEN;
                                	break;
#endif
				   default:
					ether_type = 0xFFFF;
					break;
				}
				
				if (ether_type!=0xFFFF) { 
recurse:
                        		if (ether_type == ETHERTYPE_VLAN) {
                                		ether_type = ntohs(*(u_int16_t *)(packet + 2));
                                		packet += 4;
                                		if(ether_type > ETHERMTU)
                                        		goto recurse;
                        		}
                        		if(ether_type != ETHERTYPE_IP) continue;
                        		ip = (struct ip *)packet;
                		} else
                        		ip = (struct ip*) (packet + offset);

				len = ntohs(ip->ip_len);
				sDS_AcctIp(s, ip, &start, len, 1);
				gettimeofday(&stop, NULL);
				d_start=start.tv_sec+ (double)start.tv_usec/1000000;
				d_stop=stop.tv_sec+ (double)stop.tv_usec/1000000;
				cfg->total_packets++;
				cfg->delay+=(d_stop-d_start);
				cfg->skewdelay = long(cfg->skewdelay*0.98 + (long)(1000000*(d_stop-d_start))*0.02); 
				i=start.tv_sec%10;
				cfg->pc[i]++; cfg->bc[i]+=len;
				j=i+1; if (j==10) j=0;
				cfg->pc[j]=0; cfg->bc[j]=0;
				aDebug(DEBUG_DS_IP, "pcap packet processed in %d mcsec [%6ld]\n", (unsigned)(1000000*(d_stop-d_start)), cfg->total_packets, cfg->pcap->pkthdr.len);
				}

			}
			break;

		case PT_FM_TRAFFIC:
			{
			dsFlowModule(s);
			}
			break;

		case PT_MAIL_STAT: {
			dsMail(s, ip);
			}
			break;

		default:	// this source type is not catched yet!
			break;
		}

	aFree(packet);
	pthread_cleanup_pop(0);
	return NULL;
	}
//////////////////////////////////////////////////////////////////////////
void pcap_callback(unsigned char *args, const struct pcap_pkthdr *hdr, const unsigned char *packet){
	pcap_data *a=(pcap_data*)args;
	a->pkt=(unsigned char*)packet;
	a->pkthdr.len=hdr->len;
	}
//////////////////////////////////////////////////////////////////////////
void sDSCancel(void *v){
	Service *s = (Service*)v;
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	aLog(D_INFO, "cancelling service data-source:%u\n", s->instance);
	ds_rule *ptr,*t=cfg->root;
	while(t) {
//		printf("removing %p (%p) %d %s\n", t, t->next, t->number, t->rule_str);
		sDS_RuleIp(1, s, t->number, t->rule_str);
		ptr=t; t=t->next;
		aFree(ptr);
		}

	sDSHookRemove(s);
	delete cfg->IPtree;
	aFree(cfg->src_cmd);
	aFree(cfg);
}

//////////////////////////////////////////////////////////////////////////
void sDSListCfg(Service *s, FILE *f){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

	fprintf(f, "service data-source %d\n", s->instance);
	if (cfg->type==PT_IP_TRAFFIC) fprintf(f, "type ip-traffic\n");
	else if (cfg->type==PT_NETFLOW_TRAFFIC) fprintf(f, "type netflow\n");
	else if (cfg->type==PT_LIBPCAP_TRAFFIC) fprintf(f, "type libpcap\n");
	else if (cfg->type==PT_FM_TRAFFIC) fprintf(f, "type flowmodule\n");
	else if (cfg->type==PT_MAIL_STAT) fprintf(f, "type mailstat\n");

	if (cfg->src_cmd && cfg->port) fprintf(f, "source %s %u\n", cfg->src_cmd, cfg->port);
	else if (cfg->src_cmd) fprintf(f, "source %s\n", cfg->src_cmd);
	for (ds_rule *r=cfg->root; r!=NULL; r=r->next) if (r->number && r->rule_str) fprintf(f, "rule %d \"%s\"\n", r->number, r->rule_str);
	fprintf(f, "\n");
	}

//////////////////////////////////////////////////////////////////////////
unsigned sDS_AcctIp(Service *s, struct ip *packet, struct timeval *tv, unsigned long len, unsigned skip_fw_check) {
	NetUnit *u, *up;
	IPTree *IPtree=((ServiceDS_cfg*)(s->cfg))->IPtree;
	dsUlist *start,*dsulist,*dsu;
	match mf;
	unsigned tmp;				   //for no-local-pass
	unsigned fw_res=ProcessorCfg.restrict_all; // 0=pass, 1=drop

	pthread_rwlock_rdlock(IPtree->rwlock);
	
	//here we organize units that matches packet
	start=IPtree->checktree(packet);

	// we should reorganize Units lock here to be more effective
	Units.tries_lock++;
	pthread_rwlock_rdlock(Units.rwlock);

	//aLog(D_INFO,"tut skip_fw_check=%u fw_res=%u start=%p\n",skip_fw_check,fw_res,start);
	if(!skip_fw_check)
	for(dsulist=start;dsulist!=NULL;dsulist=dsulist->link) {
		mf=dsulist->mf;
		for(dsu=dsulist;dsu!=NULL;dsu=dsu->next) {
			tmp=fw_res; //for no-local-pass
			u=dsu->u;
			// yes, this unit corresponds to this data packet
			aDebug(DEBUG_DS_IP, "packet matches unit %s (oid=%06X)\n", u->name, u->id);
			// we will perform the fw check for this unit

			if((fw_res=sDSsystem(start, u))) {
				aDebug(DEBUG_DS_IP, " fw check for this unit: SYS-DROP\n");
				goto ACCOUNTING;
			}
			if((fw_res=sDSfw(u, packet, mf))) {
				aDebug(DEBUG_DS_IP, " fw check for this unit: DROP\n");
				goto ACCOUNTING;
			}
			aDebug(DEBUG_DS_IP, " fw check for this unit: SYS-PASS\n");

			// now check the fw policy for all parents (groups)
			for	(up=u->parent; up!=NULL; up=up->parent)
				if (up->checkDSList(s->instance))  {
					// we will perform the fw check for this unit because it is parent
					aDebug(DEBUG_DS_IP, "packet matches parent %s (oid=%06X)\n", up->name, up->id);
					if((fw_res=sDSsystem(start, up))) {
						aDebug(DEBUG_DS_IP, " fw check for this parent: SYS-DROP\n");
						goto ACCOUNTING;
					}
					if((fw_res=sDSfw(up, packet, mf))) {
						aDebug(DEBUG_DS_IP, " fw check for this parent: DROP\n");
						goto ACCOUNTING;
					}
					aDebug(DEBUG_DS_IP, " fw check for this parent: PASS\n");
				}
			if(u->nlp) fw_res=tmp; //thus we activate no-local-pass
		}
	}
	else fw_res=0;

	//accounting here
ACCOUNTING:
	for(dsulist=start;dsulist!=NULL;dsulist=dsulist->link) {
		if(!fw_res) { //we need this even we drop packed to clear match flags for future use
			mf=dsulist->mf;
			for(dsu=dsulist;dsu!=NULL;dsu=dsu->next) {
				u=dsu->u;
				// now check the acct policy for this packet, and update counters
				// for this unit itself
				sDSacct(u, packet, mf, tv, len);

				// and for all parents
				for	(up=u->parent; up!=NULL; up=up->parent)
					if (up->checkDSList(s->instance)) {
					sDSacct(up, packet, mf, tv, len);
				}
			}
		}
		dsulist->mf=MATCH_NONE;
	}
	//we should reorganize locks here to be more effective
	pthread_rwlock_unlock(Units.rwlock);

	pthread_rwlock_unlock(IPtree->rwlock);
//	aDebug(DEBUG_DS_IP, "sDS_AcctIp return %d %u\n", fw_res, !fw_res);
	return !fw_res;
	}

//////////////////////////////////////////////////////////////////////////
void sDS_RuleIp(int flag, Service *s, unsigned number, char *rule) { // flag==0 eq ADD, ==1 eq REMOVE
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;
	char *buffer = (char *)aMalloc(255);
	if (cfg->type==PT_IP_TRAFFIC)
	switch (flag) {
		case 0: { // add a rule to the system

			#ifdef LINUX
			sprintf(buffer,"/sbin/iptables -I %s 2>&1 >/dev/null", rule);
			#else
			sprintf(buffer,"/sbin/ipfw add %u %s %u %s 2>&1 >/dev/null", number, cfg->src_cmd, cfg->port, rule);
			#endif
			aLog(D_INFO, "adding rule %u \"%s\"\n", number, rule);
			system(buffer);
			}
			break;

		case 1: { // remove a rule from the system

			#ifdef LINUX
			sprintf(buffer, "/sbin/iptables -D %s 2>&1 >/dev/null", rule);
			#else
			sprintf(buffer,"/sbin/ipfw delete %u 2>&1 >/dev/null", number);
			#endif

			aLog(D_INFO, "removing rule %u \"%s\"\n", number, rule);
			system(buffer);
			}
			break;

		default: break;
		}

	aFree(buffer);
	}

//////////////////////////////////////////////////////////////////////////
void sDSHookInstall(Service *s){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

	switch (cfg->type) {

		case PT_IP_TRAFFIC:	// here we are assume that arrived data has an IP format
			{

#ifdef LINUX
#ifndef IPTBL_NONE
	#ifdef IPTBL_PFSET
		cfg->ipq=ipq_create_handle(0, PF_INET);
	#else
		cfg->ipq=ipq_create_handle(0);
	#endif

	if (!cfg->ipq) aLog(D_ERR, "linux ipq handle failed, %s!\n", ipq_errstr());
	if(ipq_set_mode(cfg->ipq, IPQ_COPY_PACKET, MAX_PKT_SIZE)<0) {
                aLog(D_ERR, "ipq_set_mode failed: %s\n",ipq_errstr());
                return;
        }

#endif

#else // FreeBSD

	cfg->ds.sin_family = AF_INET;
	cfg->ds.sin_addr.s_addr = INADDR_ANY;
	cfg->ds.sin_port = htons(cfg->port);

	if ((cfg->socketid = socket (PF_INET, SOCK_RAW, IPPROTO_DIVERT)) < 0) aLog(D_ERR, "divert socket\n");

	if (bind(cfg->socketid, (struct sockaddr *)&(cfg->ds), sizeof(cfg->ds))) aLog(D_ERR, "bind divert socket: %d\n", cfg->socketid);

#endif
			} break;
		case PT_NETFLOW_TRAFFIC:
			{

			int reuseaddr_on=1, status;

			cfg->socketid=socket(PF_INET, SOCK_DGRAM, 0);
			if (cfg->socketid<0) { aLog(D_ERR, "failed to create netflow listening socket: %s", strerror(errno)); return; }
			memset(&cfg->ds, 0, sizeof(cfg->ds));

			cfg->ds.sin_family = AF_INET;
			cfg->ds.sin_addr.s_addr = INADDR_ANY;
			cfg->ds.sin_port = htons(cfg->port);

			// set options
			status=bigsockbuf(cfg->socketid, SO_RCVBUF, FT_SO_RCV_BUFSIZE);
			if(status<0) {
				aLog(D_ERR, "failed to setsockopt receive buffer: %s", strerror(errno));
				return;
			} else {
				aLog(D_INFO, "receive bufer set to %u\n", status);
			}

			setsockopt(cfg->socketid, SOL_SOCKET, SO_REUSEADDR, (void*)&reuseaddr_on, sizeof(reuseaddr_on));

			// bind socket with serv_addr
			status=bind(cfg->socketid, (struct sockaddr*)&cfg->ds, sizeof(cfg->ds));
			if (status<0) { aLog(D_ERR, "failed to bind netflow listening socket: %s", strerror(errno)); return; }

			} break;
		case PT_LIBPCAP_TRAFFIC:
			{
			cfg->pcap=(pcap_data*)aMalloc(sizeof(pcap_data));
			pcap_data *p=cfg->pcap;
			p->errbuf=(char*)aMalloc(256);
			p->h=pcap_open_live(cfg->src_cmd, MAX_PKT_SIZE-1, 1, 100, p->errbuf);
			if (!p->h) {
				aLog(D_ERR, "failed to open pcap interface: %s\n", p->errbuf);
				return;
			}
			if (-1==pcap_lookupnet(cfg->src_cmd, &p->net, &p->mask, p->errbuf)) {
				p->net=0;
				p->mask=0;
				aLog(D_WARN, "%s\n", p->errbuf);
			}
			
			if (cfg->root && cfg->root->rule_str) {
				if (-1==pcap_compile(p->h, &p->filter, cfg->root->rule_str, 1, p->net)) {
					aLog(D_ERR, "failed to compile pcap: %s\n", pcap_geterr(p->h));
					 return;
				}
				if (-1==pcap_setfilter(p->h, &p->filter)) {
					aLog(D_ERR, "failed to setfilter pcap: %s\n", pcap_geterr(p->h));
					return;
				}
			} else 
				aLog(D_WARN, "no rule for pcap, capturing all \n");

			p->datalink=pcap_datalink(p->h);
			switch (p->datalink){
				case DLT_NULL:
					p->offset=0;
					aLog(D_INFO, "Libpcap: BSD loopback\n");
					break;
				case DLT_EN10MB:
					p->offset=ETHER_HDR_LEN;
					aLog(D_INFO, "Libpcap: ethernet interface\n");
					break;
				case DLT_PPP:
					p->offset=4; // PPP_HDRLEN
					aLog(D_INFO, "Libpcap: PPP interface\n");
					break;
#ifdef LINUX
				case DLT_LINUX_SLL:
					p->offset=SLL_HDR_LEN; // PPP_HDRLEN
					aLog(D_INFO, "Libpcap: PPP(SLL) interface\n");
					break;
#endif
				default:
					p->offset=0;
					aLog(D_INFO, "Libpcap: unknown interface\n");
					break;
					}

			} break;
		case PT_FM_TRAFFIC:
			{
				dsFlowModuleAdd(s);
			} break;
		case PT_MAIL_STAT:
			{
				dsMailHookAdd(s);
			} break;
		default: break;
		} //switch

	aLog(D_INFO, "Hook is installed by ds:%u\n", cfg->id);
	}

//////////////////////////////////////////////////////////////////////////
void sDSHookRemove(Service *s){
	ServiceDS_cfg *cfg=(ServiceDS_cfg*)s->cfg;

	switch (cfg->type) {

		case PT_IP_TRAFFIC:	// here we are assume that arrived data has an IP format
			{
#ifdef LINUX
	#ifndef IPTBL_NONE
	ipq_destroy_handle(cfg->ipq);
	#endif
#else // FreeBSD

	shutdown(cfg->socketid, SHUT_RDWR);
	close(cfg->socketid);

#endif
			} break;
		case PT_NETFLOW_TRAFFIC:
			{

			shutdown(cfg->socketid, SHUT_RDWR);
			close(cfg->socketid);

			} break;

		case PT_LIBPCAP_TRAFFIC:
			{
			pcap_close(cfg->pcap->h);
			aFree(cfg->pcap->errbuf);
			aFree(cfg->pcap);
			} break;

		case PT_FM_TRAFFIC:
			{
			dsFlowModuleRemove(s);
			} break;

		case PT_MAIL_STAT:
			{
				dsMailHookRemove(s);
			} break;

		default: break;
		} // switch

	aLog(D_INFO, "Hook placed by ds:%u is removed\n", cfg->id);

	}

//////////////////////////////////////////////////////////////////////////
unsigned short sDSfw(NetUnit *u, struct ip *packet, match mf){
unsigned short fw_res=0;
policy_data *cpd;

fw_res=ProcessorCfg.restrict_local;

	for (cpd=u->fp.root; cpd!=NULL; cpd=cpd->next) {
		cpd->check++;
		if(cpd->policy->Check(packet, mf)) {
			if (cpd->inv) {
				cpd->match++;
				fw_res=0;
				if (cpd->brk) break;
			} else {
				cpd->match++;
				fw_res=1;
				break;
			}
		 } else {
			if (cpd->inv) {
				fw_res=1;
				break;
			} else
				fw_res=0;
		}
	}
	return fw_res;
}

//////////////////////////////////////////////////////////////////////////
unsigned short sDSsystem(dsUlist *start, NetUnit *u){
	NetUnit *up;

	switch (u->sys_policy) {
		case SP_NONE:
		case SP_ALLOW:
		case SP_ALLOW_AUTH:
			return 0;
		case SP_DENY_NONE:
		case SP_DENY_MONEY:
		case SP_DENY_QUOTA:
		case SP_DENY_MACVIOL:
		case SP_DENY_AUTH:
			return 1;
		default:
			for(dsUlist *dsulist=start;dsulist!=NULL;dsulist=dsulist->link)
				for(dsUlist *dsu=dsulist;dsu!=NULL;dsu=dsu->next) {
					up=dsu->u;
					if (up!=u) {
						if (!up->nlp) {
							if (u->sys_policy_perm) { if (u->sys_policy_perm==up->id) return 0; }
							else return 0;
						}
					}
				}
			return 1;
		}
	}

//////////////////////////////////////////////////////////////////////////
void sDSacct(NetUnit *u, struct ip *packet, match mf, struct timeval *tv, unsigned long len){
policy_data *cpd;
unsigned short monitor=0;

aDebug(DEBUG_DS_IP, " acct for unit: %s (oid=%06X) ---\n", u->name, u->id);
	for (cpd=u->ap.root; cpd!=NULL; cpd=cpd->next) {
		cpd->check++;
		if(cpd->policy->Check(packet,mf)) {
			if (!cpd->inv) {
				cpd->match++;
				monitor=1;
				PolicyDataUpdate(mf, PT_IP_TRAFFIC, cpd, len, tv);
				if (cpd->brk) break;
			}
		} else {
			if (cpd->inv) {
				cpd->match++;
				monitor=1;
				PolicyDataUpdate(mf, PT_IP_TRAFFIC, cpd, len, tv);
				if (cpd->brk) break;
			}
		}
	}

	if (u->monitor && monitor) aMonitor(u, packet, tv, len);
}
//////////////////////////////////////////////////////////////////////////
// obtained from flow-tools
// Copyright (c) 2001 Mark Fullmer and The Ohio State University
/*
 * function: bigsockbuf
 *
 * There is no portable way to determine the max send and receive buffers
 * that can be set for a socket, so guess then decrement that guess by
 * 2K until the call succeeds.  If n > 1MB then the decrement by .5MB
 * instead.
 *
 * returns size or -1 for error
*/
int bigsockbuf(int fd, int dir, int size)
{
  int n, tries;

  /* initial size */
  n = size;
  tries = 0;

  while (n > 4096) {

    if (setsockopt(fd, SOL_SOCKET, dir, (char*)&n, sizeof (n)) < 0) {

      /* anything other than no buffers available is fatal */
      if (errno != ENOBUFS) {
        return -1;
      }

      /* try a smaller value */

      if (n > 1024*1024) /* most systems not > 256K bytes w/o tweaking */
        n -= 1024*1024;
      else
        n -= 2048;

      ++tries;

    } else {
      return n;
    }

  } /* while */

  /* no increase in buffer size */
  return 0;

} /* bigsockbuf */
//////////////////////////////////////////////////////////////////////////
