/*************************************************************************
***	Authentication, authorization, accounting + firewalling package
***	(c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***
*************************************************************************/
/* $Id: s_quota.c,v 1.19.2.8 2004/05/06 10:35:05 jura Exp $ */

#include "netams.h"

void sQuCheckDb(int id, Quota_cfg *c);
void sQuObtainDbData(NetUnit *u, Quota_cfg *c);
void sQuSetCfg(char *param[32], Connection *conn, Quota_cfg *c);
void sQuSendAlert(NetUnit *u, int dir);

//////////////////////////////////////////////////////////////////////////////////////////
void sQuInit(Service *s){

	Quota_cfg *cfg = (Quota_cfg*)aMalloc(sizeof(Quota_cfg));
	cfg->default_soft_treshold=S_QUOTA_DEF_soft_treshold;
	cfg->default_policy=PolicyL.root;
	cfg->delay=S_QUOTA_DEF_delay;
	cfg->storage=0;
	cfg->st=NULL; 	
	cfg->fd=NULL;
	cfg->query=(char*)aMalloc(512);

	cfg->notify_soft[0]=S_QUOTA_DEF_notify_soft;
	cfg->notify_hard[0]=S_QUOTA_DEF_notify_hard;
	cfg->notify_return[0]=S_QUOTA_DEF_notify_return;
	for (int i=1; i<S_QUOTA_DEF_arraysize; i++) {
		cfg->notify_soft[i]=0;
		cfg->notify_hard[i]=0;
		cfg->notify_return[i]=0;
		}
	cfg->rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
        pthread_rwlock_init(cfg->rwlock, NULL);

	s->cfg=cfg;
	pthread_create(&(s->t_id), NULL, &sQuota, s);

	}

//////////////////////////////////////////////////////////////////////////////////////////
void sQuProcessCfg(char *param[32], Connection *conn, unsigned no_flag){

	Service *s=NULL;

	if(!(s=Services.getServiceByName("quota",s))) return;

	Quota_cfg *cfg=(Quota_cfg*)s->cfg;
	time_t t;
	
	pthread_rwlock_wrlock(cfg->rwlock);

	if (!strcasecmp(param[0], "")) {
		}

	else if (!strcasecmp(param[0], "policy")) {
		Policy *p;
		if (param[1]!=empty && (p=PolicyL.getPolicy(param[1]))) {
			aParse(conn, "default policy is set to %s\n", param[1]);
			cfg->default_policy=p;
			} else aParse(conn, "no such policy exist: %s\n", param[1]);
		}

	else if (!strcasecmp(param[0], "soft-treshold")) {
		unsigned int st=atoi(param[1]);
		if (st<0 || st>100) {
			aParse(conn, "invalid soft treshold value: %d (must be between 0 and 100)\n", st);
			}
		else {
			aParse(conn, "soft treshold set to %d\n", st);
			cfg->default_soft_treshold=st;
			}
		}

	else if (!strcasecmp(param[0], "delay")) {
		t=strtol(param[1], NULL, 10);
		if (t<1 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->delay=t;
		aParse(conn, "quota control check delay set to %ld seconds\n", t);
		}

	else if (!strcasecmp(param[0], "storage")) {
		t=strtol(param[1], NULL, 10);
		if (t<0 || t>60*60*24*7) return; // disallow negative and more than one week
		cfg->storage=(int)t;
		aParse(conn, "storage number set to %d\n", (int)t);
		}

	else if (!strcasecmp(param[0], "notify")) {
		User *us;
		if (!strcasecmp(param[1], "soft")) { int j=2;
			if (!strcasecmp(param[j], "<owner>")) { cfg->notify_soft[0]=1; j++; }
			us=Users.getUser(param[j]);
			if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
			if (us) { cfg->notify_soft[1]=us->id; }
			}
		if (!strcasecmp(param[1], "hard")) { int j=2;
			if (!strcasecmp(param[j], "<owner>")) { cfg->notify_hard[0]=1; j++; }
			us=Users.getUser(param[j]);
			if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
			if (us) { cfg->notify_hard[1]=us->id; }
			}
		if (!strcasecmp(param[1], "return")) { int j=2;
			if (!strcasecmp(param[j], "<owner>")) { cfg->notify_return[0]=1; j++; }
			us=Users.getUser(param[j]);
			if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
			if (us) { cfg->notify_return[1]=us->id; }
			}
		}

	else if (!strcasecmp(param[0], "set")) sQuSetCfg(param, conn, cfg);

	else aParse(conn, "unknown quota command: %s\n", param[0]);
	
	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void sQuListCfg(Service *s, FILE *f){

	Quota_cfg *cfg=(Quota_cfg*)s->cfg;

	if (!cfg->rwlock) return;
        pthread_rwlock_rdlock(cfg->rwlock);


	fprintf(f, "service quota %d\n", s->instance);
	if (cfg->default_soft_treshold!=S_QUOTA_DEF_soft_treshold) fprintf(f, "soft-treshold %d\n", cfg->default_soft_treshold);
	if (cfg->delay!=S_QUOTA_DEF_delay) fprintf(f, "delay %d\n", cfg->delay);
	if (cfg->default_policy) fprintf(f, "policy %s\n", cfg->default_policy->name);

	// default notify on soft quotas
	int j=0; for (int i=0; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_soft[i]) j++;
	if (j) {
		fprintf(f, "notify soft ");
		if (cfg->notify_soft[0]) fprintf(f, "<owner> ");
		for (int i=1; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_soft[i]) fprintf(f, "%06X ", cfg->notify_soft[i]);
		fprintf(f, "\n");
		}

	// default notify on hard quotas
	j=0; for (int i=0; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_hard[i]) j++;
	if (j) {
		fprintf(f, "notify hard ");
		if (cfg->notify_hard[0]) fprintf(f, "<owner> ");
		for (int i=1; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_hard[i]) fprintf(f, "%06X ", cfg->notify_hard[i]);
		fprintf(f, "\n");
		}

	// default notify on quotas return
	j=0; for (int i=0; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_return[i]) j++;
	if (j) {
		fprintf(f, "notify return ");
		if (cfg->notify_return[0]) fprintf(f, "<owner> ");
		for (int i=1; i<S_QUOTA_DEF_arraysize; i++) if (cfg->notify_return[i]) fprintf(f, "%06X ", cfg->notify_return[i]);
		fprintf(f, "\n");
		}

	fprintf(f, "storage %d\n", cfg->storage);

	fprintf(f, "\n");
	
	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void *sQuota(void *ss){
	Service *s=(Service*)ss;
	Quota_cfg *cfg=(Quota_cfg*)s->cfg;
//	time_t now;
	NetUnit *u;
	sQuotaData *q;
	policy_data *pd;
	unsigned q_viol, q_soft, update_required;
	static char *param[32]; param[0]="set"; param[1]="oid"; param[2]=(char*)aMalloc(32); param[3]=empty; param[4]=NULL;

	aLog(D_INFO, "service quota:%u thread started (%p)\n", s->instance, s->cfg);
	pthread_cleanup_push(sQuCancel, s);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	s->t_id=pthread_self();

	// now we should sleep before starting service
	s->Sleep();

	cfg->st = Services.getService("storage", cfg->storage);
    	if (cfg->st==NULL) {
		aLog(D_WARN, "quota service requires storage %d to be up, skipping\n", cfg->storage);
		goto S_QUOTA_CLEANUP_POP;
        }

	while(!(cfg->fd=stOpenSql(cfg->st,ST_CONN_QUOTA))) {
                aLog(D_WARN, "Service quota:%u can't obtain data from storage:%u\n",s->instance,cfg->st->instance);
                sleep(60);
        }
	
	pthread_rwlock_wrlock(cfg->rwlock);
	pthread_rwlock_rdlock(Units.rwlock);
	for (u=Units.root; u!=NULL; u=u->next){
		stQuObtainDbData(cfg, u); // for each unit, ask the database and fill the u->quotadata structure
		if(u->quotadata && u->quotadata->blocked) {
			aLog(D_INFO, "Restoring unit %06X (%s) violated quota\n", u->id, u->name?u->name:"<>");
			SetSysPolicy("sys-deny-quota", u, cInternal);
                        cAccessScriptCall(DROP, u, "QUOTA VIOLATE");
		}
	}
	pthread_rwlock_unlock(Units.rwlock);
	if(cfg->fd) {stCloseSql(cfg->st,cfg->fd); cfg->fd=NULL;}
	pthread_rwlock_unlock(cfg->rwlock);
	//actually we deno't need cfg->query after that today

	s->Sleep(cfg->delay*1000); // we should wait for loading 'summary' data

	aLog(D_DEBUG, "service quota:%u checking every %d seconds\n", s->instance, cfg->delay);

	while (1) {

		aDebug(DEBUG_QUOTA, "Checking quotas (every %d seconds)\n", cfg->delay);
//		now=time(NULL);
		
		//probably there might be much faster but suspect there might be locks optherwise
		pthread_rwlock_wrlock(cfg->rwlock);
		pthread_rwlock_rdlock(Units.rwlock);
		for (u=Units.root; u!=NULL; u=u->next) if (u->quotadata && u->quotadata->active) {

			q=u->quotadata;
			q_viol=q_soft=update_required=0;
			pd=u->ap.Get(q->policy);
			if (!pd) continue;

			if (q->h.in) if (pd->h.in>=q->h.in) q_viol=1;
			if (q->h.out) if (pd->h.out>=q->h.out) q_viol=1;
			if (q->h.sum) if ((pd->h.in+pd->h.out)>=q->h.sum) q_viol=1;
			if (q->h.in && q->soft_treshold) if (pd->h.in>=q->soft_treshold*q->h.in/100) q_soft=1;
			if (q->h.out && q->soft_treshold) if (pd->h.out>=q->soft_treshold*q->h.out/100) q_soft=1;
			if (q->h.sum && q->soft_treshold) if ((pd->h.in+pd->h.out)>=q->soft_treshold*q->h.sum/100) q_soft=1;

			if (q->d.in) if (pd->d.in>=q->d.in) q_viol=1;
			if (q->d.out) if (pd->d.out>=q->d.out) q_viol=1;
			if (q->d.sum) if ((pd->d.in+pd->d.out)>=q->d.sum) q_viol=1;
			if (q->d.in && q->soft_treshold) if (pd->d.in>=q->soft_treshold*q->d.in/100) q_soft=1;
			if (q->d.out && q->soft_treshold) if (pd->d.out>=q->soft_treshold*q->d.out/100) q_soft=1;
			if (q->d.sum && q->soft_treshold) if ((pd->d.in+pd->d.out)>=q->soft_treshold*q->d.sum/100) q_soft=1;

			if (q->w.in) if (pd->w.in>=q->w.in) q_viol=1;
			if (q->w.out) if (pd->w.out>=q->w.out) q_viol=1;
			if (q->w.sum) if ((pd->w.in+pd->w.out)>=q->w.sum) q_viol=1;
			if (q->w.in && q->soft_treshold) if (pd->w.in>=q->soft_treshold*q->w.in/100) q_soft=1;
			if (q->w.out && q->soft_treshold) if (pd->w.out>=q->soft_treshold*q->w.out/100) q_soft=1;
			if (q->w.sum && q->soft_treshold) if ((pd->w.in+pd->w.out)>=q->soft_treshold*q->w.sum/100) q_soft=1;

			if (q->m.in) if (pd->m.in>=q->m.in) q_viol=1;
			if (q->m.out) if (pd->m.out>=q->m.out) q_viol=1;
			if (q->m.sum) if ((pd->m.in+pd->m.out)>=q->m.sum) q_viol=1;
			if (q->m.in && q->soft_treshold) if (pd->m.in>=q->soft_treshold*q->m.in/100) q_soft=1;
			if (q->m.out && q->soft_treshold) if (pd->m.out>=q->soft_treshold*q->m.out/100) q_soft=1;
			if (q->m.sum && q->soft_treshold) if ((pd->m.in+pd->m.out)>=q->soft_treshold*q->m.sum/100) q_soft=1;

			if (q->t.in) if (pd->t.in>=q->t.in) q_viol=1;
			if (q->t.out) if (pd->t.out>=q->t.out) q_viol=1;
			if (q->t.sum) if ((pd->t.in+pd->t.out)>=q->t.sum) q_viol=1;
			if (q->t.in && q->soft_treshold) if (pd->t.in>=q->soft_treshold*q->t.in/100) q_soft=1;
			if (q->t.out && q->soft_treshold) if (pd->t.out>=q->soft_treshold*q->t.out/100) q_soft=1;
			if (q->t.sum && q->soft_treshold) if ((pd->t.in+pd->t.out)>=q->soft_treshold*q->t.sum/100) q_soft=1;

			// deal with hard quotas...
			if (!q_viol && q->blocked) { // viol=0, blocked=1 -> reset it back
				q->blocked=0;
				aLog(D_WARN, "unit %06X (%s) quota reset back\n", u->id, u->name?u->name:"<>");
				sQuSendAlert(u, 0);
				q->notified=0;
				SetSysPolicy("sys-allow", u, cInternal);
				cAccessScriptCall(PASS, u, "QUOTA RESET");
				update_required++;
				}
			else if (q_viol && !q->blocked) { // viol=1, blocked=0 -> block it
				q->blocked=1;
				aLog(D_WARN, "unit %06X (%s) violated quota\n", u->id, u->name?u->name:"<>");
				if (q->notified<2) sQuSendAlert(u, 1);
				q->notified=2;
				q->blocked_time=time(NULL);
				SetSysPolicy("sys-deny-quota", u, cInternal);
				cAccessScriptCall(DROP, u, "QUOTA VIOLATE");
				update_required++;
				}
			else if (!q_viol && !q->blocked) { // viol=0, blocked=0 -> pass
				}
			else  { // viol=1, blocked=1 -> do nothing
				}

			// deal with soft quotas...
			if (q_soft && !q->blocked && !q_viol) { // soft=1, blocked=0, viol=0 -> notify
				if (q->notified<1) {
					aLog(D_WARN, "unit %06X (%s) reached soft quota\n", u->id, u->name?u->name:"<>");
					sQuSendAlert(u, 2);
					q->notified=1;
					update_required++;
					}
				}

			if (update_required) {
				sprintf(param[2], "%06X", u->id);
				sQuSetCfg(param, cInternal, cfg);
				}

			} // for all units and in unit is quota-active

		pthread_rwlock_unlock(Units.rwlock);
		pthread_rwlock_unlock(cfg->rwlock);

		s->Sleep(cfg->delay*1000); // quota timeouts will be checked every cfg->delay seconds
	}

S_QUOTA_CLEANUP_POP:
	pthread_cleanup_pop(0);
	return NULL;
}

//////////////////////////////////////////////////////////////////////////////////////////
void sQuCancel(void *v){
	Service *s = (Service*)v;
	Quota_cfg *cfg=(Quota_cfg*)s->cfg;
	aFree(cfg->query);
	pthread_rwlock_destroy(cfg->rwlock);
	aFree(cfg->rwlock);
	aFree(cfg);

	aLog(D_INFO, "cancelling service quota:%u\n", s->instance);
}

//////////////////////////////////////////////////////////////////////////////////////////
void cShowQuota(Connection *conn, char *param[32]){
	NetUnit *u, *ut=NULL;
	User *us;
	oid t;
	sQuotaData *q;
	policy_data *pd;
	int total=0, enabled=0, active=0, blocked=0, notified=0, unit_spec=0;
	static char buf[32], buf2[32];

	Service *s=Services.getServiceByName("quota",NULL);
        if(!s) return;
//	Quota_cfg *cfg=(Quota_cfg*)s->cfg;

	if (!strcasecmp(param[2], "oid")) {
		t=strtol(param[3], NULL, 16);
		ut=Units.getUnitById(t);
		unit_spec=1;
	}
	else if (!strcasecmp(param[2], "name")) {
		ut=Units.getUnit(param[3]);
		unit_spec=1;
	}
	else if (!strcasecmp(param[2], "list")) {
		fprintf(conn->stream_w, "Units where quota is enabled:\n");
//		pthread_rwlock_rdlock(cfg->rwlock);
//		pthread_rwlock_rdlock(Units.rwlock);
		for (u=Units.root; u!=NULL; u=u->next) 
			if (u->quotadata) fprintf(conn->stream_w, "%s %06X ", u->name?u->name:"<\?\?>", u->id);
//		pthread_rwlock_unlock(Units.rwlock);
//		pthread_rwlock_unlock(cfg->rwlock);
		fprintf(conn->stream_w, "\n");
		return;
	}
//	int err1=pthread_rwlock_tryrdlock(cfg->rwlock);
//	if(err1==EBUSY) pthread_rwlock_rdlock(cfg->rwlock); // deadlock is here.
//	int err2=pthread_rwlock_tryrdlock(Units.rwlock);
//	if(err2==EBUSY) pthread_rwlock_rdlock(Units.rwlock);

	for (u=Units.root; u!=NULL; u=u->next){
		total++;
		if (!u->quotadata) continue;
		q=u->quotadata;
		pd=u->ap.Get(q->policy);
		if (!pd) continue;
		enabled++;
		if (q->active) active++;
		if (q->blocked) blocked++;
		if (q->notified) notified++;
		if (ut && ut!=u) continue; //unit specified
			
		fprintf(conn->stream_w, "OID: %06X (%s) policy: %s soft-treshold: %d%% %sNF:%d %s\n", u->id, u->name?u->name:"<\?\?>", q->policy->name, q->soft_treshold, q->active?"ACTIVE ":"", q->notified, q->blocked?"BLOCKED ":"");
		fprintf(conn->stream_w, "Notification: soft %s%s, hard %s%s, return %s%s\n", q->nss?"<owner> ":"", q->nso?((us=Users.getUserById(q->nso))?us->name:"??"):"", q->nhs?"<owner> ":"", q->nho?((us=Users.getUserById(q->nho))?us->name:"??"):"", q->nrs?"<owner> ":"", q->nro?((us=Users.getUserById(q->nro))?us->name:"??"):"") ;

		if (q->h.in && q->soft_treshold)		fprintf(conn->stream_w, "  HOUR   in: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.in, buf), bytesQ2T(q->soft_treshold*q->h.in/100, buf2), 10000*(double)pd->h.in/(q->h.in*q->soft_treshold), pd->h.in>=(double)q->soft_treshold/100*q->h.in?'-':'+');
		if (q->h.in)		fprintf(conn->stream_w, "  HOUR   in: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.in, buf), bytesQ2T(q->h.in, buf2), 100.0*(double)pd->h.in/q->h.in, pd->h.in>=q->h.in?'-':'+');
		if (q->h.out && q->soft_treshold)		fprintf(conn->stream_w, "  HOUR  out: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.out, buf), bytesQ2T(q->soft_treshold*q->h.out/100, buf2), 10000*(double)pd->h.out/(q->h.out*q->soft_treshold), pd->h.out>=(double)q->soft_treshold/100*q->h.out?'-':'+');
		if (q->h.out)		fprintf(conn->stream_w, "  HOUR  out: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.out, buf), bytesQ2T(q->h.out, buf2), 100.0*(double)pd->h.out/q->h.out, pd->h.out>=q->h.out?'-':'+');
		if (q->h.sum && q->soft_treshold)		fprintf(conn->stream_w, "  HOUR  sum: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.in+pd->h.out, buf), bytesQ2T(q->soft_treshold*q->h.sum/100, buf2), 10000*(double)(pd->h.in+pd->h.out)/(q->h.sum*q->soft_treshold), (pd->h.in+pd->h.out)>=(double)q->soft_treshold/100*q->h.sum?'-':'+');
		if (q->h.sum)		fprintf(conn->stream_w, "  HOUR  sum: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->h.in+pd->h.out, buf), bytesQ2T(q->h.sum, buf2), 100.0*(double)(pd->h.in+pd->h.out)/q->h.sum, (pd->h.in+pd->h.out)>=q->h.sum?'-':'+');

		if (q->d.in && q->soft_treshold)		fprintf(conn->stream_w, "  DAY	  in: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.in, buf), bytesQ2T(q->soft_treshold*q->d.in/100, buf2), 10000*(double)pd->d.in/(q->d.in*q->soft_treshold), pd->d.in>=(double)q->soft_treshold/100*q->d.in?'-':'+');
		if (q->d.in)		fprintf(conn->stream_w, "  DAY	  in: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.in, buf), bytesQ2T(q->d.in, buf2), 100.0*(double)pd->d.in/q->d.in, pd->d.in>=q->d.in?'-':'+');
		if (q->d.out && q->soft_treshold)		fprintf(conn->stream_w, "  DAY	 out: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.out, buf), bytesQ2T(q->soft_treshold*q->d.out/100, buf2), 10000*(double)pd->d.out/(q->d.out*q->soft_treshold), pd->d.out>=(double)q->soft_treshold/100*q->d.out?'-':'+');
		if (q->d.out)		fprintf(conn->stream_w, "  DAY	 out: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.out, buf), bytesQ2T(q->d.out, buf2), 100.0*(double)pd->d.out/q->d.out, pd->d.out>=q->d.out?'-':'+');
		if (q->d.sum && q->soft_treshold)		fprintf(conn->stream_w, "  DAY	 sum: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.in+pd->d.out, buf), bytesQ2T(q->soft_treshold*q->d.sum/100, buf2), 10000*(double)(pd->d.in+pd->d.out)/(q->d.sum*q->soft_treshold), (pd->d.in+pd->d.out)>=(double)q->soft_treshold/100*q->d.sum?'-':'+');
		if (q->d.sum)		fprintf(conn->stream_w, "  DAY	 sum: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->d.in+pd->d.out, buf), bytesQ2T(q->d.sum, buf2), 100.0*(double)(pd->d.in+pd->d.out)/q->d.sum, (pd->d.in+pd->d.out)>=q->d.sum?'-':'+');

		if (q->w.in && q->soft_treshold)		fprintf(conn->stream_w, "  WEEK   in: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.in, buf), bytesQ2T(q->soft_treshold*q->w.in/100, buf2), 10000*(double)pd->w.in/(q->w.in*q->soft_treshold), pd->w.in>=(double)q->soft_treshold/100*q->w.in?'-':'+');
		if (q->w.in)		fprintf(conn->stream_w, "  WEEK   in: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.in, buf), bytesQ2T(q->w.in, buf2), 100.0*(double)pd->w.in/q->w.in, pd->w.in>=q->w.in?'-':'+');
		if (q->w.out && q->soft_treshold)		fprintf(conn->stream_w, "  WEEK  out: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.out, buf), bytesQ2T(q->soft_treshold*q->w.out/100, buf2), 10000*(double)pd->w.out/(q->w.out*q->soft_treshold), pd->w.out>=(double)q->soft_treshold/100*q->w.out?'-':'+');
		if (q->w.out)		fprintf(conn->stream_w, "  WEEK  out: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.out, buf), bytesQ2T(q->w.out, buf2), 100.0*(double)pd->w.out/q->w.out, pd->w.out>=q->w.out?'-':'+');
		if (q->w.sum && q->soft_treshold)		fprintf(conn->stream_w, "  WEEK  sum: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.in+pd->w.out, buf), bytesQ2T(q->soft_treshold*q->w.sum/100, buf2), 10000*(double)(pd->w.in+pd->w.out)/(q->w.sum*q->soft_treshold), (pd->w.in+pd->w.out)>=(double)q->soft_treshold/100*q->w.sum?'-':'+');
		if (q->w.sum)		fprintf(conn->stream_w, "  WEEK  sum: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->w.in+pd->w.out, buf), bytesQ2T(q->w.sum, buf2), 100.0*(double)(pd->w.in+pd->w.out)/q->w.sum, (pd->w.in+pd->w.out)>=q->w.sum?'-':'+');

		if (q->m.in && q->soft_treshold)		fprintf(conn->stream_w, "  MONTH  in: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.in, buf), bytesQ2T(q->soft_treshold*q->m.in/100, buf2), 10000*(double)pd->m.in/(q->m.in*q->soft_treshold), pd->m.in>=(double)q->soft_treshold/100*q->m.in?'-':'+');
		if (q->m.in)		fprintf(conn->stream_w, "  MONTH  in: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.in, buf), bytesQ2T(q->m.in, buf2), 100.0*(double)pd->m.in/q->m.in, pd->m.in>=q->m.in?'-':'+');
		if (q->m.out && q->soft_treshold)		fprintf(conn->stream_w, "  MONTH out: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.out, buf), bytesQ2T(q->soft_treshold*q->m.out/100, buf2), 10000*(double)pd->m.out/(q->m.out*q->soft_treshold), pd->m.out>=(double)q->soft_treshold/100*q->m.out?'-':'+');
		if (q->m.out)		fprintf(conn->stream_w, "  MONTH out: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.out, buf), bytesQ2T(q->m.out, buf2), 100.0*(double)pd->m.out/q->m.out, pd->m.out>=q->m.out?'-':'+');
		if (q->m.sum && q->soft_treshold)		fprintf(conn->stream_w, "  MONTH sum: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.in+pd->m.out, buf), bytesQ2T(q->soft_treshold*q->m.sum/100, buf2), 10000*(double)(pd->m.in+pd->m.out)/(q->m.sum*q->soft_treshold), (pd->m.in+pd->m.out)>=(double)q->soft_treshold/100*q->m.sum?'-':'+');
		if (q->m.sum)		fprintf(conn->stream_w, "  MONTH sum: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->m.in+pd->m.out, buf), bytesQ2T(q->m.sum, buf2), 100.0*(double)(pd->m.in+pd->m.out)/q->m.sum, (pd->m.in+pd->m.out)>=q->m.sum?'-':'+');

		if (q->t.in && q->soft_treshold)		fprintf(conn->stream_w, "  TOTAL  in: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.in, buf), bytesQ2T(q->soft_treshold*q->t.in/100, buf2), 10000*(double)pd->t.in/(q->t.in*q->soft_treshold), pd->t.in>=(double)q->soft_treshold/100*q->t.in?'-':'+');
		if (q->t.in)		fprintf(conn->stream_w, "  TOTAL  in: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.in, buf), bytesQ2T(q->t.in, buf2), 100.0*(double)pd->t.in/q->t.in, pd->t.in>=q->t.in?'-':'+');
		if (q->t.out && q->soft_treshold)		fprintf(conn->stream_w, "  TOTAL out: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.out, buf), bytesQ2T(q->soft_treshold*q->t.out/100, buf2), 10000*(double)pd->t.out/(q->t.out*q->soft_treshold), pd->t.out>=(double)q->soft_treshold/100*q->t.out?'-':'+');
		if (q->t.out)		fprintf(conn->stream_w, "  TOTAL out: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.out, buf), bytesQ2T(q->t.out, buf2), 100.0*(double)pd->t.out/q->t.out, pd->t.out>=q->t.out?'-':'+');
		if (q->t.sum && q->soft_treshold)		fprintf(conn->stream_w, "  TOTAL sum: %s, softquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.in+pd->t.out, buf), bytesQ2T(q->soft_treshold*q->t.sum/100, buf2), 10000*(double)(pd->t.in+pd->t.out)/(q->t.sum*q->soft_treshold), (pd->t.in+pd->t.out)>=(double)q->soft_treshold/100*q->t.sum?'-':'+');
		if (q->t.sum)		fprintf(conn->stream_w, "  TOTAL sum: %s, hardquota %s ratio %.0f%% -> [%c]\n", bytesQ2T(pd->t.in+pd->t.out, buf), bytesQ2T(q->t.sum, buf2), 100.0*(double)(pd->t.in+pd->t.out)/q->t.sum, (pd->t.in+pd->t.out)>=q->t.sum?'-':'+');

	}
//	if(err2!=EDEADLK) pthread_rwlock_unlock(Units.rwlock);
//	if(err1!=EDEADLK) pthread_rwlock_unlock(cfg->rwlock);
	fprintf(conn->stream_w, "Total units: %d, enabled: %d, active: %d, blocked:%d, notified: %d\n", total, enabled, active, blocked, notified);
}

//////////////////////////////////////////////////////////////////////////////////////////
void sQuSetCfg(char *param[32], Connection *conn, Quota_cfg *c){
	NetUnit *u;
	oid t;
	sQuotaData *q;

	if (!strcasecmp(param[1], "oid")) {
		t=strtol(param[2], NULL, 16);
		u=Units.getUnitById(t);
		if (!u) { aParse(conn, "unit with oid=%06X is not exist\n", t); return; }
	}
	else if (!strcasecmp(param[1], "name")) {
		u=Units.getUnit(param[2]);
		if (!u) { aParse(conn, "unit with name=%s is not exist\n", param[2]); return; }
	}
	else { aParse(conn, "oid must be specified\n"); return; }
	
	Service *s=Services.getServiceByName("quota",NULL);
        if(!s) return;
//	Quota_cfg *cfg=(Quota_cfg*)s->cfg;

//	pthread_rwlock_wrlock(cfg->rwlock);

	if (!u->quotadata) {
		u->quotadata = (sQuotaData*)aMalloc(sizeof(sQuotaData));
		u->quotadata->active=1;
		u->quotadata->blocked=0;
		u->quotadata->blocked_time=0L;
		u->quotadata->policy=c->default_policy;
		u->quotadata->notified=0;
		u->quotadata->soft_treshold=c->default_soft_treshold;
		u->quotadata->nss=c->notify_soft[0]; u->quotadata->nso=c->notify_soft[1];
		u->quotadata->nhs=c->notify_hard[0]; u->quotadata->nho=c->notify_hard[1];
		u->quotadata->nrs=c->notify_return[0]; u->quotadata->nro=c->notify_return[1];
		bzero(&u->quotadata->h, sizeof (struct qstat));
		bzero(&u->quotadata->d, sizeof (struct qstat));
		bzero(&u->quotadata->w, sizeof (struct qstat));
		bzero(&u->quotadata->m, sizeof (struct qstat));
		bzero(&u->quotadata->t, sizeof (struct qstat));
	}

	q=u->quotadata;

	int i=3;
	while (param[i]!=empty) {

		if (strcasecmp(param[i], "policy")==0) {
			Policy *pol=PolicyL.getPolicy(param[i+1]);
			if (pol) {
				q->policy=pol;
				aParse(conn, "unit %06X quota policy set to %s (%06X)\n", u->id, pol->name?pol->name:"<>", pol->id);
			}
			else aParse(conn, "unit %06X quota policy %s undefined\n", u->id, param[i+1]);
		}
		else if (!strcasecmp(param[i], "soft-treshold")) {
			unsigned int st=atoi(param[i+1]);
			if (st<0 || st>100) {
				aParse(conn, "invalid soft treshold value: %d (must be betwaan 0 and 100)\n", st);
			}
			else {
				aParse(conn, "soft treshold set to %d\n", st);
				q->soft_treshold=st;
			}
		}
		else if (strcasecmp(param[i], "active")==0) {
			aParse(conn, "unit %06X quota is set to ACTIVE\n", u->id);
			q->active=1;
			if (q->blocked) SetSysPolicy("sys-deny-quota", u, conn);
			i--;
		}
		else if (strcasecmp(param[i], "inactive")==0) {
			aParse(conn, "unit %06X quota is set to INACTIVE\n", u->id);
			q->active=0;
			if (q->blocked) SetSysPolicy("sys-allow", u, conn);
			i--;
		}
		else if (!strcasecmp(param[i], "notify")) {
			User *us;
			int j=i+2;
			if (!strcasecmp(param[i+1], "soft")) {
				if (!strcasecmp(param[j], "<none>")) { q->nss=0; q->nso=0; }
				if (!strcasecmp(param[j], "<owner>")) { q->nss=1; j++; }
				us=Users.getUser(param[j]);
				if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
				if (us) { q->nso=us->id; }
			}
			if (!strcasecmp(param[i+1], "hard")) {
				if (!strcasecmp(param[j], "<none>")) { q->nhs=0; q->nho=0; }
				if (!strcasecmp(param[j], "<owner>")) { q->nhs=1; j++; }
				us=Users.getUser(param[j]);
				if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
				if (us) { q->nho=us->id; }
			}
			if (!strcasecmp(param[i+1], "return")) {
				if (!strcasecmp(param[j], "<none>")) { q->nrs=0; q->nro=0; }
				if (!strcasecmp(param[j], "<owner>")) { q->nrs=1; j++; }
				us=Users.getUser(param[j]);
				if (!us) { t=strtol(param[j], NULL, 16); us=Users.getUserById(t); }
				if (us) { q->nro=us->id; }
			}
			i=j;
		}
		else if (strcasecmp(param[i], "hour")==0) sQuotaGetValue(&i, param, &(q->h));
		else if (strcasecmp(param[i], "day")==0) sQuotaGetValue(&i, param, &(q->d));
		else if (strcasecmp(param[i], "week")==0) sQuotaGetValue(&i, param, &(q->w));
		else if (strcasecmp(param[i], "month")==0) sQuotaGetValue(&i, param, &(q->m));
		else if (strcasecmp(param[i], "total")==0) sQuotaGetValue(&i, param, &(q->t));

		i+=2;
	} // while

	aDebug(DEBUG_QUOTA, "set/got: oid=%06X\n", u->id);
        
       	stQuSetCfg(c,u);
//	pthread_rwlock_unlock(cfg->rwlock);
}

//////////////////////////////////////////////////////////////////////////////////////////
void sQuotaGetValue(int *i, char *param[], qstat *q){

	int j=*i+1;
	unsigned long long data;
	while (1) {
		bytesT2Q(&data, param[j]);

		if (strcasecmp(param[j+1], "in")==0) q->in=data;
		else if (strcasecmp(param[j+1], "out")==0) q->out=data;
		else if (strcasecmp(param[j+1], "sum")==0) q->sum=data;
/*		else if (strcasecmp(param[j+1], "soft-in")==0) q->softin=data;
		else if (strcasecmp(param[j+1], "soft-out")==0) q->softout=data;
		else if (strcasecmp(param[j+1], "soft-sum")==0) q->softsum=data;  */
		else break;

		j+=2;
	}
	*i=j-2;

}
//////////////////////////////////////////////////////////////////////////
void sQuSendAlert(NetUnit *u, int dir){  // dir=1:violates; =0:back; =2:soft_quota_reached
	if(!u->email) return; //no email - nowhere to send
	Service *ss=NULL;
	ss=Services.getServiceByName("alerter",ss);
	if (!ss) return;
	ServiceAlerter_cfg *alerter=(ServiceAlerter_cfg*)ss->cfg;
	if (!alerter->queue) return; // alerter service is not running

	Message *msg;
	alert *al;

	msg = new Message();
	msg->type=ALERT;
	al=(alert*)aMalloc(sizeof(alert));

	msg->al=al;
	al->sent=time(NULL);
	al->expired=al->sent+60*60; // one hour expire
	al->report_id=0x06101;
	al->tries=0;
	al->alert_id=alerter->queue->total_items+1; //here
	al->user_id[0]=0;

	char *subject, *message, *buffer;
	subject=message=NULL;
	buffer=(char*)aMalloc(255);
	timeU2T(time(NULL), buffer);

	print_to_string(&message, "This is automatically generated report by %s, version %d.%d(%d)\nTime: %s\n\n", aaa_fw_software_name, aaa_fw_major_version, aaa_fw_minor_version, aaa_fw_build_version, buffer);
	
	Service *s=Services.getServiceByName("quota",NULL);
	if(!s) return;
//     	Quota_cfg *cfg=(Quota_cfg*)s->cfg;
	
	//we do not need lock here because we locked it in sQuota already	

	switch (dir) {
		case 0:
//			if (cfg->reset_back) al->user_id[0]=cfg->reset_back->id; else al->user_id[0]=0;
			if (u->quotadata->nrs) { al->unit_id[0]=u->id; }
			al->user_id[0]=u->quotadata->nro;
			print_to_string(&subject, "Quota: unit %s quota RETURN", u->name?u->name:"<>");
			print_to_string(&message, "NeTAMS Quota service have detected quota RETURN for unit %s (%06X)\n", u->name?u->name:"<>", u->id);
			break;
		case 1:
//			if (cfg->hard_reach) al->user_id[0]=cfg->hard_reach->id; else al->user_id[0]=0;
			if (u->quotadata->nhs) { al->unit_id[0]=u->id; }
			al->user_id[0]=u->quotadata->nho;
			print_to_string(&subject, "Quota: unit %s quota VIOLATION", u->name?u->name:"<>");
			print_to_string(&message, "NeTAMS Quota service have detected quota HARD REACH for unit %s (%06X)\n", u->name?u->name:"<>", u->id);
			break;
		case 2:
//			if (cfg->soft_reach) al->user_id[0]=cfg->soft_reach->id; else al->user_id[0]=0;
			if (u->quotadata->nss) { al->unit_id[0]=u->id; }
			al->user_id[0]=u->quotadata->nso;
			print_to_string(&subject, "Quota: unit %s quota SOFT REACH", u->name?u->name:"<>");
			print_to_string(&message, "NeTAMS Quota service have detected quota SOFT REACH for unit %s (%06X)\n", u->name?u->name:"<>", u->id);
			break;
	}

	if (!(al->unit_id[0] + al->user_id[0])) { // no recipients, discard this alert
		aFree(subject); aFree(message); aFree(buffer);
		delete msg; aFree(al);
		aDebug(DEBUG_ALERT, "alert (quota) abandoned because of no recipients\n");
		return;
	}

	print_to_string(&message, "\n#############################");
	print_to_string(&message, "\nCurrent quota status follows:\n\n");
	snprintf(buffer, 254, "show quota oid %06X", u->id);
	cExec(buffer, &message);
	
	print_to_string(&message, "\n####################################");
	print_to_string(&message, "\nCurrent user traffic status follows:\n\n");
	snprintf(buffer, 245, "show list full oid %06X", u->id);
	cExec(buffer, &message);

	Service *sh=Services.getServiceByName("html", NULL);
	if (sh) {
		Html_cfg *shtml=(Html_cfg*)sh->cfg;
		print_to_string(&message, "\n##############################################################");
		print_to_string(&message, "\nYou can check your traffic statistics online by clicking here:\n");
		print_to_string(&message, "%sclients/%s/index.html\n\n", shtml->url, u->name);
	}
	
	al->data=NULL;
	print_to_string(&al->data, "%s\n%s\n", subject, message);
	aFree(subject); aFree(message); aFree(buffer);

	al->alert_id=(unsigned)alerter->queue->Push(msg);
	alerter->self->Wakeup();
	aDebug(DEBUG_ALERT, "alert (quota) %u complete, data is %d bytes\n", al->alert_id, strlen(al->data));
}
//////////////////////////////////////////////////////////////////////////


