/*************************************************************************
***     Authentication, authorization, accounting + firewalling package
***     (c) 1998-2003 Anton Vinokurov <anton@netams.com>
***		(c) 2003 NeTAMS Development Team
***		All rights reserved. See 'Copying' file included in distribution
***		For latest version and more info, visit this project web page
***		located at http://www.netams.com
***		 					
*************************************************************************/
/* $Id: services.c,v 1.17.2.5 2004/05/06 10:35:06 jura Exp $ */

#include "netams.h"

//////////////////////////////////////////////////////////////////////////
Service::Service(char *param1, unsigned i){
	name=set_string(param1);
	instance=i;
	next=NULL;
	t_id=(pthread_t)0;
	sleep_mutex=(pthread_mutex_t*)aMalloc(sizeof (pthread_mutex_t)); pthread_mutex_init(sleep_mutex, NULL);
	sleep_cond=(pthread_cond_t*)aMalloc(sizeof (pthread_cond_t)); pthread_cond_init(sleep_cond, NULL);
	sleep_state=0;
	cfg=(void*)NULL;
}

Service::~Service(){
	pthread_mutex_destroy(sleep_mutex); aFree(sleep_mutex);
	pthread_cond_destroy(sleep_cond); aFree(sleep_cond);
	aFree(name);
}

char *Service::getName(){ return name; }

void Service::setTID() { t_id = pthread_self(); }

void Service::Shutdown(){
	if(!strcmp(name,"monitor")) sMonitorCancel(this);
	else if (t_id) {
		pthread_cancel(t_id);
		pthread_join(t_id,NULL);
	}
}
	
int Service::Sleep(unsigned msec){	 // parameter in milliseconds!!!!
	sleep_state=1;
	
	aDebug(DEBUG_SLEEP, "going to sleep service %s for %d ms\n", this->name, msec);
	pthread_mutex_lock(sleep_mutex);

	if (msec) {
		struct timespec ts;
		struct timeval tv;
		gettimeofday(&tv, NULL);

		TIMEVAL_TO_TIMESPEC(&tv, &ts);
		ts.tv_sec+=(msec/1000);
		ts.tv_nsec+=1000000*(time_t)(msec-1000*((long double)msec/1000));
		if (ts.tv_nsec>1000000000) { ts.tv_sec++; ts.tv_nsec-=1000000000; }

		int i=pthread_cond_timedwait(sleep_cond, sleep_mutex, &ts);
		pthread_mutex_unlock(sleep_mutex);
		sleep_state=0;
		if (i && i!=ETIMEDOUT) aLog(D_WARN, "sleeping of %s failed: %s\n", this->name, strerror(i));
		else if (i && i==ETIMEDOUT) return 1;
		return 0;
		}
	else { 
		pthread_cond_wait(sleep_cond, sleep_mutex); 
		pthread_mutex_unlock(sleep_mutex);
		sleep_state=0;
		}
	return 0;
	}

int Service::Wakeup(){
	aDebug(DEBUG_SLEEP, "(%s) waking up service %s, sleep_state was %d\n", Services.getServiceByThr(pthread_self()), this->name, sleep_state);
	if (sleep_state==0) return 1;
	sleep_state=0;
	pthread_cond_signal(sleep_cond);
	return 0;
	}

void Service::Start(char *param[32]){
	if (param[1]) Start(param[1]);
	}

void Service::Start(char *param){
	if (!strcmp(param, "server")) sServerInit(this);
	else if (!strcmp(param, "processor")) sProcessorInit(this);
	else if (!strcmp(param, "storage")) sStorageInit(this);
	else if (!strcmp(param, "data-source")) sDSInit(this);
	else if (!strcmp(param, "alerter")) sAlerterInit(this);
	else if (!strcmp(param, "quotactl")) sQuotactlInit(this);
	else if (!strcmp(param, "quota")) sQuInit(this);
	else if (!strcmp(param, "weblogin")) sWLInit(this);
	else if (!strcmp(param, "html")) sHtmlInit(this);
	else if (!strcmp(param, "scheduler")) sSchedulerInit(this);
	else if (!strcmp(param, "pvmgate")) sPVMInit(this);
	else if (!strcmp(param, "login")) sLgInit(this);
	else if (!strcmp(param, "monitor")) sMonitorInit(this);
	else { aLog(D_WARN, "service %s undefined\n", param); return; }
	aLog(D_INFO, "service %s starting thread...\n", param);
	}

void Service::ProcessCfg(char *param[32], Connection *conn, unsigned no_flag){
	if (!strcmp(name, "server")) sServerProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "processor")) sProcessorProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "data-source")) sDSProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "storage")) sStorageProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "alerter")) sAlerterProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "quotactl")) sQuotactlProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "quota")) sQuProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "weblogin")) sWLProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "pvmgate")) sPVMProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "login")) sLgProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "html")) sHtmlProcessCfg(param, conn, no_flag);
	else if (!strcmp(name, "monitor")) sMonitorProcessCfg(param, conn, no_flag);
}

void Service::listCfg(FILE *f){
	if (!strcmp(name, "server")) sServerListCfg(this, f);
	else if (!strcmp(name, "processor")) sProcessorListCfg(this, f);
	else if (!strcmp(name, "data-source")) sDSListCfg(this, f);
	else if (!strcmp(name, "storage")) sStorageListCfg(this, f);
	else if (!strcmp(name, "alerter")) sAlerterListCfg(this, f);
	else if (!strcmp(name, "quotactl")) sQuotactlListCfg(this, f);
	else if (!strcmp(name, "quota")) sQuListCfg(this, f);
	else if (!strcmp(name, "weblogin")) sWLListCfg(this, f);
	else if (!strcmp(name, "pvmgate")) sPVMListCfg(this, f);
	else if (!strcmp(name, "login")) sLgListCfg(this, f);
	else if (!strcmp(name, "html")) sHtmlListCfg(this, f);
	else if (!strcmp(name, "monitor")) sMonitorListCfg(this, f);
	}

void Service::listInfo(FILE *f) {
	if (!strcmp(name, "data-source")) {
		ServiceDS_cfg *c=(ServiceDS_cfg*)cfg;
		char *pt;
		switch (c->type){
			case PT_IP_TRAFFIC: pt="IP_FILT"; break;
			case PT_NETFLOW_TRAFFIC: pt="NETFLOW"; break;
			case PT_LIBPCAP_TRAFFIC: pt="LIBPCAP"; break;
			default : pt="<\?\?>"; break;
			}
		fprintf(f, " Data-source ID=%d type %s source %s:%u loop %qu average %d mcsec\n", instance,  pt, c->src_cmd?c->src_cmd:"<>", c->port, c->total_packets, (unsigned)(1000000*(c->delay/c->total_packets)));
		int avg_pps=0; for (int i=0; i<10; i++) avg_pps+=c->pc[i]; avg_pps=avg_pps/9;
		long avg_bps=0; for (int i=0; i<10; i++) avg_bps+=c->bc[i]; avg_bps=avg_bps/9;
		fprintf(f, "    Perf: average skew delay %ld mcsec, PPS: %d, BPS: %ld\n", c->skewdelay, avg_pps, avg_bps);
		}
	else if (!strcmp(name, "storage")) {
		ServiceStorage_cfg *c=(ServiceStorage_cfg*)cfg;
		char *pt;
		switch (c->type){
			case HASH: pt="HASH"; break;
			case MY_SQL: pt="MYSQL"; break;
			case POSTGRES: pt="POSTGRES"; break;
			default : pt="<\?\?>"; break;
			}
		fprintf(f, " Storage ID=%d type %s wr_q %u/%u rd_q %u/%u\n", instance, pt, c->in->num_items, c->in->total_items, c->out->num_items, c->out->total_items); 
		}
	}

void Service::showPerf(FILE *f, int isheader) {
	if (!strcmp(name, "data-source")) {
		ServiceDS_cfg *c=(ServiceDS_cfg*)cfg;
		int avg_pps=0; for (int i=0; i<10; i++) avg_pps+=c->pc[i]; avg_pps=avg_pps/9;
		long avg_bps=0; for (int i=0; i<10; i++) avg_bps+=c->bc[i]; avg_bps=avg_bps/9;
		if (isheader) fprintf(f, "DsLOOP:%d\tDsAVG:%d\tDsDly:%d\tDsPps:%d\tDsBps:%d\t", instance, instance, instance, instance, instance);
		else fprintf(f, "%qu\t%d\t%ld\t%d\t%ld\t", c->total_packets, (unsigned)(1000000*(c->delay/c->total_packets)), c->skewdelay, avg_pps, avg_bps);
		}
	else if (!strcmp(name, "storage")) {
		ServiceStorage_cfg *c=(ServiceStorage_cfg*)cfg;
		if (isheader) fprintf(f, "StWRITE:%d\t", instance);
		else fprintf(f, "%u\t", c->in->total_items);
		}
	}
//////////////////////////////////////////////////////////////////////////
ServicesList::ServicesList(){
	root=last=NULL;
	num_services=0;
	tries_lock=tries_lock_failed=0;
	unknown_service=UNKNOWN_REFERENCE;
        rwlock=(pthread_rwlock_t*)aMalloc(sizeof (pthread_rwlock_t));
        pthread_rwlock_init(rwlock, NULL);
}

ServicesList::~ServicesList(){
        pthread_rwlock_destroy(rwlock);
        aFree(rwlock);
}

void ServicesList::Insert(Service *s){
	tries_lock++;
        int err=pthread_rwlock_trywrlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	Service *d;
	for(d=root; d!=NULL; d=d->next)
		if (d==s) {
		        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return;
		}

	if (root==NULL) root=s;
	else last->next=s;
	last=s; 
	num_services++;
//	aDebug("services", cInternal, "service <%s> [%d] inserted\n", s->getName(), num_services);

        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}	

void ServicesList::Delete(char *param1, unsigned i){
	Service *s = getService(param1, i);
	if (s) Delete(s);
}

void ServicesList::Delete(Service *s){
	tries_lock++;
        int err=pthread_rwlock_trywrlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_wrlock(rwlock); }

	Service *d, *p;
	for(d=root; d!=NULL; d=d->next)	{
		if (d==s) {
			if (s==root && s==last ) root=last=NULL;
			else if (s==root) root=s->next;
			else if (s==last) { last=p; last->next=NULL; }
			else p->next=s->next;

			num_services--;
			break;
		}
		p=d; 
	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

Service *ServicesList::getService(char *param1, unsigned i){
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	char *name=set_string(param1);
	Service *d;
	for(d=root; d!=NULL; d=d->next)
		if (!strcmp(d->getName(), name) && d->instance==i) { 
			if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			aFree(name); 
			return d; 
		}
	aFree(name);
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return NULL;
}

Service *ServicesList::getServiceByName(char *param1, Service *s) {
	tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	Service *d;
	char *name=set_string(param1);
	for(s?d=s->next:d=root; d!=NULL; d=d->next)
		if(!strcmp(d->getName(), name)) break;
			
	aFree(name);
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
        return d;
}

char *ServicesList::getServiceByThr(pthread_t t){
//	printf("ServicesList::getServiceByThr locked by %d\n", t);
	tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for(d=root; d!=NULL; d=d->next)
		if (pthread_equal(t, d->t_id)) {
			if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
			return d->name;
		}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
	return unknown_service;
}

void ServicesList::StartAll(Service *except){
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
        for (d=root; d!=NULL; d=d->next) {
			if (except && except==d) aLog(D_INFO, "wake up %s:%u service: SKIPPED!\n", d->getName(), d->instance);
			else aLog(D_INFO, "waking up %s:%u service: %s\n", d->getName(), d->instance, d->Wakeup()?"WAS_UP":"WAS_DOWN");
	}
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::ShutdownAll(Service *except){
	Service *s;
	tries_lock++;
	int err=pthread_rwlock_tryrdlock(rwlock);
	if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
        
	for(s=root;s!=NULL;s=s->next)
		if (except!=s) {
			aLog(D_INFO, "shutting down %s:%u service\n", s->getName(), s->instance);
			s->Shutdown();
		}
        
	if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::listCfg(FILE *f) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }
	
	Service *d;
	for (d=root; d!=NULL; d=d->next) d->listCfg(f); 
        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::listInfo(FILE *f) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for (d=root; d!=NULL; d=d->next) d->listInfo(f);

        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}

void ServicesList::showPerf(FILE *f, int isheader) {
        tries_lock++;
        int err=pthread_rwlock_tryrdlock(rwlock);
        if (err==EBUSY) { tries_lock_failed++; pthread_rwlock_rdlock(rwlock); }

	Service *d;
	for (d=root; d!=NULL; d=d->next) d->showPerf(f, isheader);

        if (err!=EDEADLK) pthread_rwlock_unlock(rwlock);
}
//////////////////////////////////////////////////////////////////////////
