#!/usr/bin/perl -w

# Newsgrab - The Newsgrabber by Jesper L. Nielsen <lyager@phunkbros.dk>
# Newsrc support added by Roger Knobbe <rogerk+newsgrab@tislabs.com>
#
# Released under the GNU public license.
#
# $Id: newsgrab.pl,v 1.11 2003/03/26 13:07:11 lyager Exp $
#

use strict;
use News::NNTPClient;
use News::Newsrc;
use Time::ParseDate;

# Added for authentication.
use Term::ReadKey;

$|=1;	# Set autoflush for progressbar
sub progress_sub($$);
sub xover_fetch($);
sub xpat_fetch($);
sub stat_print;
sub parse_newsgrabrc($);
sub fetch_newsgrabrc(;$);
sub probe_article($);
sub decode_ydec($);
sub decode_uudec($);


# Bind function for handling calls to Carp() in News::NNTPClient
#$SIG{__WARN__} = \&warn_sig;

# ------------------------------------------ global variable initialization ---

my $VERSION		= "0.4.0";
my $LIST		= 1;
my $NNTP_REGEX		= '';
my $NNTP_GROUP		= '';
my $NEWSGRAB_RC		= $ENV{HOME}."/.newsgrabrc";
my $SERVER		= '';
my $RAW			= '';
my $SORT		= 0;
my $SORT_DATE		= 0;
my $IGNORE_RC		= 0;
my $VERBOSE		= 0;
my $QUITE		= 0;
my $OUTDIR		= '.';

my $input_newsrc = '';
my $input_outdir = '';

# Gather parameters
while ($_ = shift(@ARGV)) {
	if (/^-h$/ || /^--help$/) {
		usage_sub();
		next;
	}
	if (/^-q$/) {
		$QUITE = 1;
		next;
	}
	if (/^--version$/) {
		version_sub();
		next;
	}
	if (/^-s$/ || /^--server$/) {
		$SERVER = shift(@ARGV);
		next;
	}
	if (/^-v$/ || /^--verbose$/) {
		$VERBOSE= 1;
		next;
	}
	if (/^-g$/ || /^--group$/) {
		$NNTP_GROUP = shift(@ARGV);
		next;
	}
	if (/^-n$/) {	# Specify a newsrc file
		$input_newsrc = shift(@ARGV);
		next;
	}
	if (/^-l$/ || /^--list$/) {
		print "Listing (-l) is now default, use -r to retrieven\n";
		next;
	}
	if (/^-r$/ || /^--retrieve$/) {
		$LIST = 0;
		next;
	}
	if (/^-i$/ || /^--ignore$/) {
		$IGNORE_RC = 1;
		next;
	}
	if (/^-c$/ || /^--conf$/) {
		$NEWSGRAB_RC = shift (@ARGV);
		next;
	}
	if (/^-o$/) {
		$input_outdir = shift(@ARGV);
		next;
	}
	if (/^--sort$/) {
		$SORT = 1;
		my $sort_after = shift(@ARGV);
		if ($sort_after eq 'd') {
			$SORT_DATE = 1;
		} else {
			usage_sub();
		}
		next;
	}
	if (/^--raw$/) {
		$RAW = 1;
		next;
	}
	# Failsafe, catch unrecognized arguments
	if (/^-/) {
		usage_sub();
	}
	$NNTP_REGEX = $_;
}

# --------------------------------------------------- read newsgrabrc file ---
# Get connectinfo if missing

my $server;
if (!parse_newsgrabrc($NEWSGRAB_RC)) {
	stat_print "Unable to open conf file $NEWSGRAB_RC, specify server.",
		"NOTIFY";
	# Ask for connection info
	print "Server: ";
	$server->{'hostname'}  = <STDIN>;
	chomp $server->{'hostname'};
	
	if (!$server->{'hostname'}) {
		print STDERR "Unable to connect without a hostname\n";
		exit(1);
	}

	$server->{'rc'} = 
		$ENV{HOME}."/.newsrc.newsgrab.".$server->{'hostname'};

	print "Port [119]: ";
	$server->{'port'}  = <STDIN>;
	chomp $server->{'port'};
	if (!$server->{'port'}) {
		# Assign default value
		$server->{'port'} = 119;
	}
	
	print "Username: ";
	$server->{'username'}  = <STDIN>;
	chomp $server->{'username'};
	
	print "Password: ";
	ReadMode('noecho');
	$server->{'password'} = ReadLine(0);
	ReadMode 0;		# Reset readmode
	chomp $server->{'password'};
	print "\n";

} else {
	if (!($server = fetch_newsgrabrc($SERVER))) {
		stat_print "Unable to fetch "
			.($SERVER ? "server with key '$SERVER'" : "default server")
			." from RC file.\n",
			"ERROR";
		exit;
	}
}

# ----------------------------------------------- Initialize runtime things ---

# If an output directory is specified, try to create this dir, and place 
# all binary files in there

if ($input_outdir) {
	# Create the directory for the files, if it doesn't exits.
	if (! -e $input_outdir) {
		if (!(mkdir $input_outdir, 0777)) {
			stat_print "Unable to create output dir '$input_outdir'",
				"ERROR";
			exit 1;
		}
	}
	
	# Overwrite the default location.
	$OUTDIR = $input_outdir;
}


# Check if an alternate newsrc file was specified on the command line
if ($input_newsrc) {
	# Overwrite the newsrc with the one from the commandline
	$server->{'rc'} = $input_newsrc;
}

# Load newsrc file
my $rc = new News::Newsrc;
stat_print "newsrc", "Reading newsrc '".$server->{'rc'}."'";
if ($rc->load($server->{'rc'})) {
	stat_print("newsrc", 
		"loaded ".$rc->num_groups." group(s) from '".$server->{'rc'}."'");
} else {
	stat_print "newsrc", "No groups in '".$server->{'rc'}."'";
}

# add group to newsrc file, if it doesn't exist.
if ($rc->exists($NNTP_GROUP)) {
	stat_print "newsrc", "$NNTP_GROUP loaded ok";	
} else {
	$rc->add_group($NNTP_GROUP);
}


# ------------------------------------------------------ connnect to server ---
print_plain("Connecting to ".($NNTP_GROUP ? $NNTP_GROUP." \@ " : "").$server->{'hostname'}.":".$server->{'port'}."\n");

my $connected=1;
my $c = new News::NNTPClient($server->{'hostname'}, $server->{'port'});
if (!$c->ok) {
	stat_print "Unable to connect", "Error";
	exit(1);
}

if ($server->{'username'}) {
	print_plain("Using authentication with username: ".$server->{'username'}."\n");
	$c->authinfo($server->{'username'},$server->{'password'});
	if (!$c->ok) {
		stat_print "Authentication error", "Error";
		exit(1);
	}
}

# ------------------------------------------------------------ main program ---	

# If no group, list all groups on server and EXIT
if (!$NNTP_GROUP) {
	print_plain("No group specified, listing all groups available on the server\n");
	my $newsgroup;
	foreach ($c->list('active')) {
		($newsgroup) = split(/[\t|\s]/);
		print_plain($newsgroup."\n");
	}
	exit(0);
}

# Take all multipart articles, and join them in a hash (one hash key
# presents each complete file)
if ($NNTP_REGEX) {
	print_plain("Finding all subjects in $NNTP_GROUP matching '$NNTP_REGEX'\n");
} else {
	print_plain("Finding all subjects in $NNTP_GROUP\n");
}
print_plain('-'x80);
print_plain("\n");
my ($first, $last) = $c->group($NNTP_GROUP);

# %arts is a collection of multipart messages
my %arts;
my @ordered_keys;
my @crctab;			# yEnc CRC table
if ($SORT) {
	%arts = xover_fetch($NNTP_REGEX);
	if ($SORT_DATE) {
		@ordered_keys = 
		   sort { $arts{$a}->{'date'} <=> $arts{$b}->{'date'} } keys %arts;
	} else {
		# Default is per key
		@ordered_keys = sort(keys %arts);
	}
} else {
	%arts = xpat_fetch($NNTP_REGEX);
	@ordered_keys = sort(keys %arts);
}

print_plain("\n");

# Now all the multiparted articals have an element in the
# %arts hash, and we can retrieve them, if they are complete
my ($key);
foreach $key (@ordered_keys) {
	# $key   : Multithread article subject

	# All articles for the multithread is located in
	# array $arts{$key}->{'arts'}
	#
	# Let's see if they are all there
	#

	if ($arts{$key}->{'total'} == scalar(keys(%{$arts{$key}->{'arts'}}))) {
		# If we have the same number of articles as the
		# 'total' number.

		$arts{$key}->{'complete'} = 'C';

		# Lightly check if already retrieved
		foreach my $message_no (keys(%{$arts{$key}->{'arts'}})) {
			if ($rc->marked($NNTP_GROUP, 
				$arts{$key}
					->{'arts'}
					->{$message_no}
					->{'message_id'})) {
				# Article was already retrived
				# no need to continue
				$arts{$key}->{'complete'} = 'R';
				last;
			}
				
			
		}
	}

	# Detemine wether to download file, or skip
	# because it's incomplete
	# There 3 the followin states:
	# I: Incomplete (all multipart messages not found)
	# R: Skipping file, coz we've already downloaded (Retrieved) it
	# C: Complete posting

	stat_print $key, $arts{$key}->{'complete'};
	if ($LIST) {
		#Skip if list mode
		next;
	} 
	if ( $arts{$key}->{'complete'} eq 'R') {
		# If already retrieved, skip unles $IGNORE_RC was set
		if (!$IGNORE_RC) {
			next;
		}
	}
	if ($arts{$key}->{'complete'} eq 'I') {
		# Generate incomplete log
		print_plain("SKIPPING: $key\n");
		# Skip if incomplete
		next;
	}

	# If we didn't provide the list option, perform a retriaval of the
	# posting
	my ($file, $mode, @art_lines);
	my @markable;
	my $skipping = 0;

	foreach my $art_no (sort(keys(%{$arts{$key}->{'arts'}}))) {
		my $art = $arts{$key}->{'arts'}->{$art_no};
		my $artnum = $art->{'message_id'};
		#print "Artno: $art_no mid: $artnum\n";
		next unless $artnum;

		# All articles in @markable will be added to the newsrc file
		push @markable, $artnum;

		if (!$skipping) {
			if (!$connected) {
				$c->connect;
				$c->group($NNTP_GROUP);
				$connected=1;
			}
			@art_lines = $c->body($artnum);
			if ($RAW) {
				# Take care of raw output, and take next;
				stat_print "newsgrab-".$artnum, "Rawwrite";
				open(FP, ">newsgrab-".$artnum) || die;
				print FP @art_lines;
				close (FP);
				next;
			}
			if (!($arts{$key}->{'type'})) {
				# Determine article type
				if (my $type = probe_article(\@art_lines)) {
					$arts{$key}->{'type'} = $type;
				} else {
					stat_print "Unable to probe $key", "Skipping";
					$skipping = 1;
					next;
				}
			}
			if ($arts{$key}->{'type'} eq "ART_YENC") {
				$skipping = decode_ydec(\@art_lines);
				if ($skipping) {
					stat_print "$key", "SKIPPING YDEC ($skipping)";
					next;
				}
			} elsif ($arts{$key}->{'type'} eq "ART_UUENC") {
				$skipping = decode_uudec(\@art_lines);
				if ($skipping) {
					stat_print "$key", "SKIPPING UUDEC ($skipping)";
					next;
				}
			} else {
				print STDERR "Unable to determine encoding of article: $key\n";
				exit(1);
			}

			# Done fetching, upgrade progress_bar
			#progress_sub($arts{$key}->{'total'});
			progress_sub($art_no, $arts{$key}->{'total'});
		}

	}
	# Mark articles as read in the newsrc file
	$rc->mark_list($NNTP_GROUP, \@markable);
	$rc->save;
}



exit(1);

# -------------------------------------------------------------- functions ---

sub usage_sub {

	print STDERR "Usage: $0 [OPTIONS] [-g group] [match]\n\n";
	print STDERR "  OPTIONS:\n";
	print STDERR "  -h: This help (and quit)\n";
	print STDERR "  -q: Quite mode, only output filenames retrieved\n";
	print STDERR "  -r: Retrive files that match <exp>\n";
	print STDERR "  -i: Ignore what .newsrc says\n";
	print STDERR "  -n: Specify an alternate newsrc file to use\n";
	print STDERR "  -s: Server to use from the RC file\n";
	print STDERR "  -o: Specify a directory to please retrieved files\n";
	print STDERR "  -c: Specify an alternative conf file (default: $NEWSGRAB_RC)\n";
	print STDERR "  --raw: Write article raw to file named 'newsgrab-<MessageID>'\n";
	print STDERR "  --version: Print version information, then quit\n";
	print STDERR "  --sort: Sort output by: 'd' for date\n";
	print STDERR "          (This require more header information to be\n";
	print STDERR "           retrieved, and is therefor slower)\n";
	print STDERR "\n";
	print STDERR "  -g <group>: Group to retrive from (no group lists all)\n";
	print STDERR "  match: Subject must contain this string (default: 'none')\n";
	print STDERR "\n";
	exit(0);
}

sub version_sub {
	print STDERR "NewsGrab Version $VERSION\n";
	exit(0);
}

sub warn_sig {
	print "Got unexpected answer from server ";
	if ($c) {
		print "(Code: ".$c->code.")";
	}
	print "\n";

	
	if ($c && ($c->code >= 200) && ($c->code < 300)) {
		return;
	}
	if ($c && ($c->code >= 500)) {
		print "Error: ".$c->message()."\n";
		$connected = 0;
	} else {
		if (!($c)) {
			# Connection object not created, we was unable to connect
			# to the news server
			print STDERR "Error: Unable to connect to server\n";
		}
		exit(1);
	}
}

sub print_filename { 
	if ($QUITE) { 
		print $_[0]."\n";
	}
}

# Just simple simple output
sub print_plain {
	if (!$QUITE) {
		print $_[0];
	}
}

sub stat_print_right {

	if ($QUITE) {
		return;
	}
	
	my ($line, $stat) = @_;

	# line_width is the space the 2 arguments can actually take
	my $column_width = 77;
	my $line_len = ($column_width-(length($stat))); 
	for (my $i = 0; $i < length($line); $i += $line_len) {
		if (!($i)) {
			printf "%s [%s]\n", substr($line, $i, $line_len), $stat;
		} else {
			printf "%s\n", substr($line, $i, $line_len), $stat;

		}
	}
}

sub stat_print {
	my ($line, $stat) = @_;

	if ($QUITE) {
		return;
	}
	
	# Column determination is such a hastle, lets just
	# print the line without making it so pretty
	printf "[%s] %s\n", $stat, $line;
	return;



	# line_width is the space the 2 arguments can actually take
	# Use COLUMNS as width or 80 chars as default
	my $column_width = ($ENV{COLUMNS} || 80) - 3;
	my $line_len = ($column_width-(length($stat))); 
	for (my $i = 0; $i < length($line); $i += $line_len) {
		if (!($i)) {
			printf "[%s] %s\n", $stat, substr($line, $i, $line_len);
		} else {
			# Remember to add the filling around $stat
			printf " "x(length($stat)+3);
			printf "%s\n", substr($line, $i, $line_len), $stat;

		}
	}
}


sub stat_print_old {
	my ($line, $stat) = @_;

	# line_width is the space the 2 arguments can actually take
	my $column_width = 77;
	my $line_len = ($column_width-(length($stat))); 
	# Cut $line of, if it's too long;
	$line = substr $line, 0, $line_len;
	printf "%-*s [%s]\n", $line_len, $line, $stat;
}


# BLOCK
{

my $last_res;
sub progress_sub($$) {
	return unless $VERBOSE;
	my ($art_no, $total_no) = @_;
	my $cols = 80;

	if (!$total_no) {
		$total_no = 1;
	}
	if ($art_no == 1) {
		# Reset the global
		$last_res = 0;
	}

	# Calculate chunksize (this could be cached)
	my $chunksize = ($cols / $total_no) * $art_no;

	# By now we should have drawn $chunksize * $art_no X's
	print "X" x ($chunksize - $last_res);
	$last_res = int($chunksize);
}

sub old_progress_sub($$) {

	return unless $VERBOSE;
	my ($current_no, $total_no) = @_;
	my $cols = 80;

	# First article, initialize values
	if ($current_no == 1) {
		$last_res = 0;
	}

	if (!($total_no)) {
		$total_no = 1;
	}

	my $res = ($current_no * $cols) / $total_no;
	#print "now X".($res - $last_res)."|$last_res\n";
	print "X" x ($res - $last_res);

	# Last article, set newline
	if ($current_no == $total_no) {
		print "x" x ($cols - $res);
		#print "$last_res | $res";
		print "\n";
	}
	$last_res = $res;
}
}

# ------------------------------------------ Article Header Fetch Functions ---
# Articales are returned as follows:
#
# $arts{KEY}->{'date'}      : Contains the date of the posting
#                           : (Note: only if the date is actually in the
#                           : subject listing
# $arts{KEY}->{'complete'}  : Multipart complete (C|R|I)
# $arts{KEY}->{'total'}     : Totalt number of messages in mulipart
# $arts{KEY}->{'message'}   : array of message IDs, there doesn't have
#                           : to be a message ID under each array element
#                           : if a message is missing from the multipart
# $arts{KEY}->{'m_count'}   : Is incremented each time an article is added to
#                           : the {'message'} array.
# $arts{KEY}->{'subj_pre'}  : Subject before multipart number
# $arts{KEY}->{'subj_post'} : Subject after multipart number
#
# KEY : A concationation of the subject of the message, excluding the numbers
#       indicting which part of the subject the article is (Cocatenation of
#	{'subj_pre'} and {'subj_post'}).
# 

sub xover_fetch($) {
	my $NNTP_REGEX = shift;
	my %arts;
	foreach my $head ($c->xover($first, $last)) {
		next unless ($head =~ /$NNTP_REGEX/);
	
		# From RFC 2980:
		# Each line of output will be formatted with the article number,
		# followed by each of the headers in the overview database or the
		# article itself (when the data is not available in the overview
		# database) for that article separated by a tab character.  The
		# sequence of fields must be in this order: subject, author, date,
		# message-id, references, byte count, and line count.  Other optional
		# fields may follow line count.  Other optional fields may follow line
		# count.  These fields are specified by examining the response to the
		# LIST OVERVIEW.FMT command.  Where no data exists, a null field must
		# be provided (i.e. the output will have two tab characters adjacent to
		# each other).  Servers should not output fields for articles that have
		# been removed since the XOVER database was created.
	
		my ($mid, $subject, $author, $date, $message_id, 
	    	$refernences, $byte_count, $line_count
	    	) = split(/\t/, $head);
	
		# Check status of the multipost
		if (!($subject =~ /^(.*[\(|\[]{1})([0-9]+)\/([0-9]+)([\)|\]]{1}.*)$/)) {
			next;
		}
		
		next unless ($1 || $4); # Skip, we can't make index
		my $key = "$1$4";
	
		#print "$2:$3|$head\n";
		if ($2 && $3) {
			# initialize
			if (!$arts{$key}) {
				$arts{$key}->{'date'} = parsedate($date);
				$arts{$key}->{'complete'} = 'I';	# Incomplete
				$arts{$key}->{'total'} = $3;
			}
			if (!$arts{$key}->{'arts'}->{scalar($2)}
			    || $arts{$key}->{'arts'}->{scalar($2)}->{'message_id'} < $mid) {
				# All is indexed by its multipart #, but
				# and in case of exact same 'Subject' line higher
				# message_id's superseeds lower message_ids
				$arts{$key}->{'arts'}->{scalar($2)}->{'subject'} = $subject;
				$arts{$key}->{'arts'}->{scalar($2)}->{'message_id'} = $mid;
			}
		} else {
			if (!$arts{$key}) {
				$arts{$key}->{'date'} = parsedate($date);
				$arts{$key}->{'complete'} = 'I';	# Incomplete
				$arts{$key}->{'total'} = 1;
			}
			$arts{$key}->{'arts'}->{'1'}->{'subject'} = $subject;
			$arts{$key}->{'arts'}->{'1'}->{'message_id'} = $mid;
		}
	
	}
	return %arts;
}
	
sub xpat_fetch($) {
	my $NNTP_REGEX = shift;
	my %arts;
	foreach my $head ($c->xpat("Subject", $first, $last, '*'.$NNTP_REGEX.'*')) {
		# Attemt to split result from XPAT into the following variables
		# $1: Name of article (Subject line)
		# $2: Part number
		# $3: Number of total parts
		# $4: Possible rest of subject
		#
		# Key for each hash is based on $5 appended to $2
		chomp($head);

		my ($mid, $subject) = split(/ /, $head, 2);
		if (!($subject =~ /^(.*)[\(|\[]{1}([0-9]+)\/([0-9]+)[\)|\]]{1}(.*)$/)) {
			next;
		}
		
		next unless ($1 || $4);	# Skip if we can't make index
		my $key = "$1$4";

		# Our index will be the subject minus the multipart section (meaning
		# beginning part of subject concatenated with the end part of the subject)

		if ($2 && $3) {
			# This is a multipart message
			if (!($arts{$key})) {
				#initialize
				$arts{$key}->{'complete'} = 'I';
				$arts{$key}->{'total'} = $3;
			}
			if (!$arts{$key}->{'arts'}->{scalar($2)}
			    || $arts{$key}->{'arts'}->{scalar($2)}->{'message_id'} < $mid) {
				# All is indexed by its multipart #, but
				# and in case of exact same 'Subject' line higher
				# message_id's superseeds lower message_ids
				$arts{$key}->{'arts'}->{scalar($2)}->{'subject'} = $subject;
				$arts{$key}->{'arts'}->{scalar($2)}->{'message_id'} = $mid;
			}
		} else {
			if (!$arts{$key}) {
				#initialize
				$arts{$key}->{'complete'} = 'I';
				$arts{$key}->{'total'} = 1;
			}
			$arts{$key}->{'arts'}->{'1'}->{'subject'} = $subject;
			$arts{$key}->{'arts'}->{'1'}->{'message_id'} = $mid;
		}
	
	}
	return %arts;
}

# ----------------------------------------------------------------------------
# RC Functions

{

my @servers;

#
# Puts all servers in an array, to be popped.. Array contains hashes like:
# $server->{'key'}      = SERVERKEY
#        ->{'hostname'} = HOSTNAME
#        ->{'port'}     = PORT
#        ->{'username'} = USERNAME
#        ->{'password'} = PASSWORD
#        ->{'rc'}       = RC File
#
# Returns number of servers available
#

sub parse_newsgrabrc($) {
	my $newsgrabrc = shift;

	if(open(FP, $newsgrabrc)) {
		stat_print "Success",
			"Opening newsgrabrc: $newsgrabrc";
	} else {
		stat_print "Failed", 
			"Opening newsgrabrc: $newsgrabrc";
		return;
	}

	while (<FP>) {
		next if (/^\s*#/);
		my %server;
		my ($nothing,
		 $name, 
		 $hostname,
		 $port,
		 $username,
		 $password
			) = split /\s/;
		$server{'key'} = $name;
		$server{'hostname'} = $hostname;
		$server{'port'} = ($port || 119);
		$server{'username'} = $username;
		$server{'password'} = $password;
		$server{'rc'} = 
			$ENV{HOME}."/.newsrc.newsgrab.".$hostname;
		push @servers, \%server;
	}
	
	close (FP);
	return scalar(@servers);
}

#
# Returning an element from the %servers hash, containing all servers
# from the newsgrabrc file.
#
# Returns:
# 	%server{'hostname'}
# 	       {'port'}
#	       {'username'}
#	       {'password'}
#

sub fetch_newsgrabrc(;$) {
	$key = shift;
	
	#
	# If a key for the servers was specified
	# return this server ('undef' if not found), or else
	# return the first in line.
	#

	if ($key) {
		foreach my $server (@servers) {
			if ($server->{'key'} =~ /^$key$/) {
				return $server;
			}
		}
		return;
	} else {
		return shift @servers;
	}
}

} # RC Features END

#
# sub probe_article
#
# Probe article checks lines in the article to determine what kind of
# encoding has been used. It returns a sticker that can be attached to the
# article hash.
#
sub probe_article($) {
	my @lines = @{$_[0]};
	while ($_ = shift(@lines)) {
		if (/^=ybegin/) {
			return "ART_YENC";
		}
		if (/^begin\s*\d*\s*.*/) {
			return "ART_UUENC"
		}
	}
	return 0;
}

# UU_decode
# Hmm! uudec doesn't give a filename in each article, we need some
# way to cope with this

{	# decode_uudec

my ($file, $mode);
sub decode_uudec($) {
	my @lines = @{$_[0]};
	while ($_ = shift(@lines)) {
		chomp;
		if(/^begin\s*(\d*)\s*(.*)/) {
			($mode, $file) = ($1, $2);
			if (-e "$OUTDIR/$file") {
				print STDERR "File: '$file' already exists. skipping\n";
				undef $file;
				undef $mode;
				return 1;
			} else {
				open(OUT, "> $OUTDIR/$file") || die "Couldn't create $OUTDIR/$file: $!\n";
				print_filename("$OUTDIR/$file");
				binmode(OUT);
			}
			next;
		}
		if (/^end/) {
			close (OUT);
			if (!($mode)) {
				stat_print "No mode supplied for file", "Warning";
			} elsif (!($file)) {
				stat_print "No filename to chmod().. Wierd", "Error";
			} else {
				chmod oct($mode), "$OUTDIR/$file";
			}
			# Set $file and $mode to undef, we have reached the end of this file
			undef $file;
			undef $mode;
		}
		# Failsafe.. Check if we have a filename here
		if ($file) {
			next if /[a-z]/;
			next unless int((((ord() - 32) & 077) + 2) / 3) == int(length() / 4);
			my $unpacked = unpack("u", $_); # This catches phony end lines
			if ($unpacked) {
				(print OUT $unpacked) || die "Couldn't write $OUTDIR/$file: $!\n";
			}
		} 
	}
	return 0;

}
} # decode_uudec

# Ydec decode

sub decode_ydec($) {
	
	my @lines = @{$_[0]};
	my ($ydec_part, $ydec_line, $ydec_size, $ydec_name, $ydec_pcrc, 
		$ydec_begin, $ydec_end);
	fill_crctab() unless @crctab;
	my $pcrc;
	while ($_ = shift(@lines)) {
		# Newlines a fakes and should not be decoded
		chomp;
		s/
//g;
		# If we've started decoding $ydec_name will be set
		if (!$ydec_name) {
			# Skip until beginning of yDecoded part
			next unless (/^=ybegin/);
			if (/ part=(\d+)/) {
				$ydec_part = $1;
			}
		
			if (/ size=(\d+)/) {
				$ydec_size = $1;
			} else {
				print STDERR "Mandatory field 'size' missing";
				return 1;
			}
			if (/ line=(\d+)/) {
				$ydec_line = $1;
			}
			if(/ name=(.*)$/) {
				$ydec_name = $1;
				$ydec_name =~ s/\s+$//g;	# Strip wierdo chars
				#print "Found attach ".$ydec_name." of size ".$ydec_size."\n";
			} else {
				print STDERR "Unknown attach name\n";
				return 1;
			}

			# Multipart messages contain more information on
			# the second lin
			if ($ydec_part) {
				$_ = shift(@lines);
				chomp;
				s/^M//g;
				if (/^=ypart/) {
					if (/ begin=(\d+)/) {
						# We need this to check if the size of this message
						# is correct
						$ydec_begin = $1;
						$pcrc = 0xffffffff;
						undef $ydec_pcrc;
					} else {
						print STDERR "No begin field found in part, ignoring\n";
						undef $ydec_part;
					}
					if (/ end=(\d+)/) {
						# We need this to calculate the size of this message
						$ydec_end = $1;
					} else {
						print STDERR "No end field found in part, ignoring";
						undef $ydec_part;
					}
				} else {
					print STDERR "Article described as multipart message, however "
								."it doesn't seem that way\n";
					undef $ydec_part;
				}
			}
			

			# Now make use of the variables
			# If this is a multipart message $ydec_part is defined
			# What we need to do if the $ydec_part is different from 1
			# we need to open the file for appending!
			if (-e "$OUTDIR/$ydec_name") {
				if (defined($ydec_part)) {
					if ($ydec_part != 1) {
						# If we have a multipart message, the file exists
						# and we are not at the first part, we should just
						# open the file as an append. We assume that this is
						# the multipart we were already processing
						#print "Opening $ydec_name for appending\n";
						if (!open(OUT, ">>$OUTDIR/$ydec_name")) {
							print STDERR "Couldn't open $OUTDIR/$ydec_name for appending: $!\n";
							return 1;
						}
					} else {
						# File exists, though this is the first part of the file
						# We are probably in the process of overwriting an older 
						# file
						print STDERR "File: '$OUTDIR/$ydec_name' alread exists. Skipping\n";
						return 1;
					}
				} else {
					# Message is not multipart, still the filename exists.
					# This shouldn't be, we are probably about to overwrite an
					# older message
					print STDERR "File: '$OUTDIR/$ydec_name' alread exists. Skipping\n";
					return 1;
				}
			} else {
				# File doesn't exist. We open it for writing O' so plain.
				if (!open(OUT, "> $OUTDIR/$ydec_name")) {
					print STDERR "Couldn't create $OUTDIR/$ydec_name: $!\n";
					return 1;
				} else {
					print_filename("$OUTDIR/$ydec_name");
				}
			}
			# Set binmode for the open filehandler
			binmode(OUT);
			# Excellent.. We have detirmed all the info for this file we
			# need.. Skip till next line, this should contain the real
			# data
			next;
		}

		# Looking for the end tag
		if (/^=yend/) {
			# We are done.. Check the sanity of article
			# and unset $ydec_name in case that there are more
			# ydecoded files in the same article
			if (/ part=(\d+)/) {
				if ($ydec_part != $1) {
					print STDERR "Part number '$1' different from beginning part '$ydec_part'\n";
					return 1;
				}
			}
			if (/ size=(\d+)/) {
				# Check size, but first calculate it
				my $size;
				if (defined($ydec_part)) {
					$size = ($ydec_end - $ydec_begin + 1);
				} else {
					$size = $ydec_size;
				}
				if ($1 != $size) {
					print STDERR "Size '$1' different from beginning size '$size'\n";
					return 1;
				}
			}
			if (/ pcrc32=([0-9a-f]+)/i) {
				if (defined($ydec_pcrc) && ($ydec_pcrc != $1)) {
					print STDERR "CRC '$1' different from beginning CRC '$ydec_pcrc'\n";
					return 1;
				}
				$ydec_pcrc = hex($1);
				$pcrc = $pcrc ^ 0xffffffff;
				if ( $pcrc == $ydec_pcrc ) {
				    print STDERR ("part $ydec_part, checksum OK\n");
				}
				else {
				    printf STDERR ("part $ydec_part, checksum mismatch 0x%08x, must be 0x%08x\n",
						   $pcrc, $ydec_pcrc);
				}

			}
			undef $ydec_name;
			# Dont encode the endline, we skip to the next line
			# in search for any more parts
			next;
		}

		# If we got here, we are within an encoded article, an
		# we will take meassures to decode it
		# We decode line by line
		# Decoder + CRC checking by jvromans@squirrel.nl.
		s/=(.)/chr(ord($1)+(256-64) & 255)/ge;
		my @all = map { ord($_)+(256-42) & 255 } split(//, $_);
		print OUT pack("C*", @all);
		foreach ( @all ) {
		    $pcrc = $crctab[($pcrc^$_)&0xff] ^ (($pcrc >> 8) & 0x00ffffff);
		}
	}

	close(OUT);
	return 0;
}

sub fill_crctab {
    @crctab = (
	0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
	0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
	0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
	0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
	0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
	0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
	0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
	0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
	0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
	0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
	0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
	0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
	0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
	0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
	0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
	0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
	0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
	0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
	0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
	0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
	0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
	0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
	0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
	0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
	0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
	0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
	0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
	0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
	0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
	0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
	0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
	0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
	0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
	0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
	0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
	0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
	0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
	0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
	0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
	0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
	0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
	0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
	0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d,
    );
}
