/*
 * getput.c -- file I/O routine
 *
 * Copyright (c) 1995-1997 by nir@mxa.meshnet.or.jp
 *
 * This file is part of npc.cgi source tree.
 * npc.cgi is free software; you can redistribute it and/or modify it
 * for any purpose.
 *
 * @(#)$Id: getput.c,v 2.4 1997/11/10 16:19:05 nir Rel $
 */

/*
 * $Log: getput.c,v $
 * Revision 2.4  1997/11/10 16:19:05  nir
 * Modify error_out() message.
 * Check index file mode whether it is writable or not.
 *
 * Revision 2.3  1997/11/09 22:59:45  nir
 * Line number is displayed when error occure in config file.
 * Fixed next string problems in IF condition and ELSIF block.
 *
 * Revision 2.2  1997/11/08 18:57:56  nir
 * Comprehensive rewiting of the IF conditonal routines. Divide the IF block
 * into several part of functions, as if_block(), if_condition(), if_then(),
 * eval_body(), according to functional property blocks.
 *
 * Revision 2.1  1997/11/05 11:56:34  nir
 * Add Copyright, Id and Log.
 * Extend configuration file syntax.
 * Append several TAGs and OPTIONs, i.e., IF, ELSIF, ELSE, ENDIF,
 * ENVironment and OPTion function, TRUE and FALSE pseudofunction,
 * and Mode option.
 *
 */

#include "npc.h"

INTERN int lockfile(int fildes, BOOLEAN lock, off_t size);
INTERN void if_block(FILE *index_file, char *tag, BOOLEAN eval);
INTERN BOOLEAN if_condition(FILE *index_file, char **p_next);
INTERN enum tag_id if_then(FILE *index_file, char **p_next, BOOLEAN eval);
INTERN BOOLEAN condition(enum tag_id id, char *param, char *str);
INTERN BOOLEAN is_option(enum tag_id id);
INTERN void eval_body(char *name, enum tag_id id, char *param);
INTERN BOOLEAN address_area(char *addr);
INTERN char *get_next_line(FILE *index_file);
INTERN char *clip(char *str);
INTERN char *unquote(char *str);
INTERN char *get_subtag(char *str);

INTERN int lines = 0;	/* config file input line number */

#if defined(_WIN32)
INTERN int lockfile(int fildes, BOOLEAN lock, off_t size) {
	int function;

	function = (lock) ? _LK_LOCK : _LK_UNLCK;
	return(locking(fildes, function, size));
}
#else
#define TIMEOUT 10

INTERN int lockfile(int fildes, BOOLEAN lock, off_t size) {
	struct flock fld;
	int timeout;

	fld.l_whence = SEEK_CUR;
	fld.l_start = 0L;
	fld.l_len = size;
	if (lock) {
		fld.l_type = F_WRLCK;
		timeout = TIMEOUT;
		while (fcntl(fildes, F_SETLK, &fld) < 0) {
			if ((errno != EACCES) && (errno != EAGAIN))
				return(BAD);
			if (--timeout < 0) {
				errno = EDEADLK;
				return(BAD);
			}
			sleep(1);
		}
		return(0);
	} else {
		fld.l_type = F_UNLCK;
		return(fcntl(fildes, F_SETLK, &fld));
	}
}
#endif

void update_count(FILE *index_file) {
	char buf[LINE_SIZE];
	off_t s_off, e_off, size;

	if ((gcontrol.number != BAD) || (gcontrol.random))
		return;
	if (gcontrol.location == NULL)
		error_out("Location Is Not Set");
	if (env.read_only)
		error_out("Index File Is Not Writable");
	size = 0L;
	LOOP {
		s_off = ftell(index_file);
		if (fgets(buf, LINE_SIZE, index_file) == NULL)
			break;
		clip(buf);
		if (strcmp(buf + MAX_WIDTH + 1, gcontrol.location) == 0) {
			e_off = ftell(index_file);
			size = e_off - s_off;
			break;
		}
	}
	if ((size == 0L) && (gcontrol.increment <= 0)) {
		gcontrol.number = 0;
		gcontrol.increment = 0;
		return;
	}
	fseek(index_file, s_off, SEEK_SET);
	if (lockfile(fileno(index_file), YES, size) < 0) {
		switch (errno) {
		case EDEADLK:
			error_out("Time Out");
		case EBADF:
			error_out("Bad File Descriptor");
		case EMFILE:
			error_out("No More File Descriptor");
		case ENOLCK:
			error_out("No More Locks");
		case EINVAL:
			error_out("Does Not Support Locking");
		case EINTR:
			error_out("Interrupted By A Signal");
		default:
			error_out("Index File Lock Error");
		}
	}
	if (size == 0L) {
		gcontrol.number = gcontrol.initial - 1;
	} else if (fgets(buf, LINE_SIZE, index_file) == NULL) {
		error_out("Cannot Get Counter Line");
	} else {
		gcontrol.number = atol(buf);
		fseek(index_file, s_off, SEEK_SET);
	}
	if ((gcontrol.number += gcontrol.increment) < 0) {
		gcontrol.number = 0;
		gcontrol.increment = 0;
	}
	fprintf(index_file, "%010d %s\n", gcontrol.number, gcontrol.location);
	fseek(index_file, s_off, SEEK_SET);
	lockfile(fileno(index_file), NO, size);
}


FILE *open_index(void) {
	char buf[LINE_SIZE];
	FILE *index_file;
	char *index_path;

#if defined(_WIN32)
	if ((env.index[0] == SEP) || (env.index[1] == ':')) {
#else
	if (env.index[0] == SEP) {
#endif
		index_path = env.index;
	} else {
		sprintf(buf, "%s%c%s", env.index_dir, SEP, env.index);
		index_path = buf;
	}
	if ((index_file = fopen(index_path, "r+")) != NULL) {
		env.read_only = NO;
	} else if ((index_file = fopen(index_path, "r")) != NULL) {
		env.read_only = YES;
	} else {
		error_out("Cannot Open Index File \"%s\"", env.index);
	}
	return(index_file);
}


void get_config(FILE *index_file) {
	char buf[LINE_SIZE];
	enum tag_id id;
	enum tag_id disable_id;
	char *str, *tag;

	lines = 1;
	if ((fgets(buf, LINE_SIZE, index_file) == NULL)
	 || (strncmp(buf, ID_MARK, sizeof(ID_MARK) - 1) != 0))
		error_out("%s(%d): BAD npc.idx File", env.index, lines);
	id = T_BAD;
	disable_id = T_BAD;
	LOOP {
		str = get_next_line(index_file);
		if (*str == '=') {
			str = get_subtag(tag = str + 1);
			switch (id = tag_search(tag)) {
			case T_COUNTER:
				return;
			case T_INDEX:
			case T_HELP:
				error_out("%s(%d): Cannot Use \"%s\" TAG",
					env.index, lines, tag);
			case T_IF:
				if_block(index_file, str, (disable_id == T_BAD));
				id = T_BAD;
				break;
			case T_ELSIF:
			case T_ELSE:
			case T_ENDIF:
			case T_OPTION:
			case T_TRUE:
			case T_FALSE:
				error_out("%s(%d): Unexpected \"%s\" TAG",
					env.index, lines, tag);
			case T_REFERER:
			case T_BROWSER:
				if ((disable_id == id)
				 || (disable_id == T_BAD))
					disable_id = id;
				break;
			case T_ADDRESS:
			case T_HOST:
				if ((disable_id == T_ADDRESS)
				 || (disable_id == T_BAD))
					disable_id = T_ADDRESS;
				break;
			case T_BAD:
				error_out("%s(%d): \"%s\" Is Unknown TAG Name",
					env.index, lines, tag);
			}
		} else if (id == T_BAD) {
			error_out("%s(%d): Unexpected Value Line \"%s\"",
				env.index, lines, str);
		} else {
			str = unquote(str);
			switch (id) {
			case T_REFERER:
			case T_BROWSER:
				if ((disable_id == id)
				 && (condition(id, NULL, str)))
					disable_id = T_BAD;
				break;
			case T_ADDRESS:
			case T_HOST:
				if ((disable_id == T_ADDRESS)
				 && (condition(id, NULL, str)))
					disable_id = T_BAD;
				break;
			default:
				if (disable_id == T_BAD)
					eval_body(env.index, id, str);
				break;
			}
		}
	}
}


INTERN void if_block(FILE *index_file, char *tag, BOOLEAN eval) {
	enum tag_id id;
	BOOLEAN valid, undone;
	char *str;

	undone = YES;
	str = tag;
	do {
		valid = if_condition(index_file, &str);
		id = if_then(index_file, &str, (valid && undone && eval));
		if (valid)
			undone = NO;
	} while (id == T_ELSIF);
	if (id == T_ELSE) {
		if (*(str = get_next_line(index_file)) != '=') {
			error_out("%s(%d): ELSE Does Not Need Conditional Line \"%s\"",
				env.index, lines, str);
		}
		str++;
		id = if_then(index_file, &str, (undone && eval));
	}
	if (id != T_ENDIF) {
		error_out("%s(%d): ENDIF Is Expected",
			env.index, lines, str);
	}
}


INTERN BOOLEAN if_condition(FILE *index_file, char **p_next) {
	char buf[LINE_SIZE];
	enum tag_id id;
	BOOLEAN valid;
	char *param, *str;

	param = get_subtag(*p_next);
	switch (id = tag_search(*p_next)) {
	case T_TRUE:
	case T_FALSE:
		break;
	case T_ENVIRONMENT:
		strcpy(buf, param);
		break;
	case T_OPTION:
		if ((id = tag_search(param)) == T_BAD)
			error_out("%s(%d): \"%s\" Is Unknown IF OPTION TAG",
				env.index, lines, param);
		if (!is_option(id))
			error_out("%s(%d): Unexpected \"%s\" TAG Exist In IF OPTION",
				env.index, lines, param);
		break;
	case T_BAD:
		error_out("%s(%d): \"%s\" Is Unknown IF TAG",
			env.index, lines, *p_next);
	default:
		if (!is_option(id))
			error_out("%s(%d): Unexpected \"%s\" TAG Exist In IF",
				env.index, lines, *p_next);
		break;
	}
	valid = (id == T_TRUE);
	while (*(str = get_next_line(index_file)) != '=') {
		str = unquote(str);
		if ((!valid) && condition(id, buf, str))
			valid = YES;
	}
	*p_next = str + 1;
	return(valid);
}


INTERN enum tag_id if_then(FILE *index_file, char **p_next, BOOLEAN eval) {
	enum tag_id id;
	char *tag, *str;

	tag = *p_next;
	LOOP {
		str = get_subtag(tag);
		switch (id = tag_search(tag)) {
		case T_ELSIF:
		case T_ELSE:
		case T_ENDIF:
			*p_next = str;
			return(id);
		case T_IF:
			if_block(index_file, str, eval);
			id = T_BAD;
			break;
		case T_COUNTER:
		case T_OPTION:
		case T_TRUE:
		case T_FALSE:
		case T_REFERER:
		case T_BROWSER:
		case T_ADDRESS:
		case T_HOST:
			error_out("%s(%d): Unexpected \"%s\" Tag Exist In IF Block",
				env.index, lines, tag);
		case T_INDEX:
		case T_HELP:
			error_out("%s(%d): Cannot Use \"%s\" TAG",
				env.index, lines, tag);
		case T_BAD:
			error_out("%s(%d): \"%s\" Is Unknown TAG Name",
				env.index, lines, tag);
		default:
			break;
		}
		str = get_next_line(index_file);
		if ((*str == '=') && (id != T_BAD)) {
			error_out("%s(%d): Missing Value Line Before \"%s\"",
				env.index, lines, str);
		} else if (*str == '=') {
			tag = str + 1;
		} else if (id == T_BAD) {
			error_out("%s(%d): Unexpected Value Line \"%s\"",
				env.index, lines, str);
		} else {
			str = unquote(str);
			if (eval)
				eval_body(env.index, id, str);
			if (*(str = get_next_line(index_file)) != '=') {
				error_out("%s(%d): Unexpected Another Value Line \"%s\"",
					env.index, lines, str);
			}
			tag = str + 1;
		}
	}
}


INTERN BOOLEAN is_option(enum tag_id id) {
	switch (id) {
	case T_ADDRESS:
	case T_ANIMATION:
	case T_BROWSER:
	case T_COLOR:
	case T_DELAY:
	case T_DIGITS:
	case T_HOST:
	case T_INCREMENT:
	case T_INDEX:
	case T_INITIAL:
	case T_LOCATION:
	case T_MODE:
	case T_NUMBER:
	case T_OFFSET:
	case T_PROGRESS:
	case T_RANDOM:
	case T_REFERER:
	case T_RESTRICTION:
	case T_TRANSPARENT:
	case T_UNIT:
	case T_WIDTH:
		return(YES);
	}
	return(NO);
}


INTERN BOOLEAN condition(enum tag_id id, char *param, char *str) {
	GCONTROL gc;
	char *value;

	str = unquote(str);
	switch (id) {
	case T_TRUE:
		return(YES);
	case T_FALSE:
		return(NO);
	case T_ENVIRONMENT:
		if ((value = getenv(param)) == NULL)
			return(NO);
		return(reg_strcmp(str, value));
	case T_INDEX:
		return(reg_strcmp(str, env.index));
	case T_REFERER:
		return(reg_strcmp(str, env.referer));
	case T_BROWSER:
		return(reg_strcmp(str, env.browser));
	case T_HOST:
		return(reg_strcmp(str, env.host));
	case T_ADDRESS:
		return(address_area(str));
	case T_ANIMATION:
		set_control(&gc, env.index, id, str);
		return(gc.gif_animation == gcontrol.gif_animation);
	case T_COLOR:
		set_control(&gc, env.index, id, str);
		return((gc.color[0] == gcontrol.color[0]) && (gc.color[1] == gcontrol.color[1]));
	case T_DELAY:
		set_control(&gc, env.index, id, str);
		return((gc.delay[0] == gcontrol.delay[0]) && (gc.delay[1] == gcontrol.delay[1]));
	case T_DIGITS:
		set_control(&gc, env.index, id, str);
		return(gc.digits == gcontrol.digits);
	case T_INCREMENT:
		set_control(&gc, env.index, id, str);
		return(gc.increment == gcontrol.increment);
	case T_INITIAL:
		set_control(&gc, env.index, id, str);
		return(gc.initial == gcontrol.initial);
	case T_LOCATION:
		set_control(&gc, env.index, id, str);
		return(reg_strcmp(gc.location, gcontrol.location));
	case T_MODE:
		set_control(&gc, env.index, id, str);
		return(gc.mode == gcontrol.mode);
	case T_NUMBER:
		set_control(&gc, env.index, id, str);
		return(gc.number == gcontrol.number);
	case T_OFFSET:
		set_control(&gc, env.index, id, str);
		return(gc.offset == gcontrol.offset);
	case T_PROGRESS:
		set_control(&gc, env.index, id, str);
		return(gc.progress == gcontrol.progress);
	case T_RANDOM:
		set_control(&gc, env.index, id, str);
		return(gc.random == gcontrol.random);
	case T_RESTRICTION:
		set_control(&gc, env.index, id, str);
		return(gc.restriction == gcontrol.restriction);
	case T_TRANSPARENT:
		set_control(&gc, env.index, id, str);
		return(gc.transparent == gcontrol.transparent);
	case T_UNIT:
		set_control(&gc, env.index, id, str);
		return(gc.unit == gcontrol.unit);
	case T_WIDTH:
		set_control(&gc, env.index, id, str);
		return(gc.width == gcontrol.width);
	}
	error_out("INTERNAL ERROR: condition: Invalid TAG ID %d", id);
}


INTERN void eval_body(char *name, enum tag_id id, char *param) {
	switch (id) {
	case T_DEBUG:
		debug_out(param);
	case T_ENVIRONMENT:
		env_out(param);
	case T_LOCATION:
		set_control(&gcontrol, name, id, new_str(param));
		break;
	default:
		set_control(&gcontrol, name, id, param);
		break;
	}
}


INTERN BOOLEAN address_area(char *addr) {
	int n, num;

	if ((env.addr[0] == 0)
	 && (env.addr[1] == 0)
	 && (env.addr[2] == 0)
	 && (env.addr[3] == 0))
		return(NO);
	n = 0;
	LOOP {
		if (*addr == '*') {
			addr++;
		} else if (*addr == '[') {
			addr++;
			if (((num = next_num(&addr)) < 0)
			 || (num > env.addr[n]))
				return(NO);
			if (*addr == '-') {
				addr++;
				if ((num = next_num(&addr)) < 0)
					return(NO);
			}
			if ((num < env.addr[n])
			 || (*addr != ']'))
				return(NO);
			addr++;
		} else if (((num = next_num(&addr)) < 0)
			|| (num != env.addr[n])) {
			return(NO);
		}
		if (++n >= 4)
			break;
		if (*(addr++) != '.')
			return(NO);
	}
	return(YES);
}


int next_num(char **pstr) {
	char *p;
	int c, n;

	p = *pstr;
	if (! isdigit(*p))
		return(BAD);
	do {
		++p;
	} while (isdigit(*p));
	c = *p;
	*p = EOS;
	n = atol(*pstr);
	*(*pstr = p) = c;
	return(n);
}


INTERN char *get_next_line(FILE *index_file) {
	LOCAL char buf[LINE_SIZE];
	char *str;

	do {
		lines++;
		if (fgets(buf, LINE_SIZE, index_file) == NULL)
			error_out("%s(%d): Unexpected EOF", env.index, lines);
		str = clip(buf);
	} while ((*str == EOS) || (*str == '#') || (*str == ';'));
	return(str);
}


INTERN char *clip(char *str) {
	char *p;

	while (isspace(*str))
		str++;
	p = str + strlen(str);
	while (p-- > str) {
		if (! isspace(*p))
			break;
		*p = EOS;
	}
	return(str);
}

INTERN char *unquote(char *str) {
	char *p;

	p = str + strlen(str) - 1;
	if ((p > str)
	 && (((*str == '\"') && (*p == '\"'))
	  || ((*str == '\'') && (*p == '\'')))) {
		str++;
		*p = EOS;
	}
	return(str);
}

INTERN char *get_subtag(char *str) {
	while (*str != EOS) {
		if (*str == '=') {
			*str = EOS;
			return(str + 1);
		}
		str++;
	}
	return(str);
}


void gwrite(unsigned char *buf, unsigned int num) {
	fwrite(buf, sizeof(unsigned char), num, stdout);
}


void gputc(unsigned int c) {
	putchar(c);
}


void gputw(unsigned int w) {
	gputc(w & 0xFF);
	gputc(w >> 8);
}

