/* oqtplayer.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OQTPlayer, a free video player using the QuickTime 
 * library.
 *
 * OQTPlayer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OQTPlayer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* OQTPlayer - A very simple video player showing QuickTime video using
 * SDL and OpenQuicktime. Multithreaded.
 *
 * Copyright (c) Antoine Mine' 2001
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <SDL.h>
#include <SDL_byteorder.h>
#include <SDL_thread.h>
#include <SDL_mutex.h>


#include <openquicktime/openquicktime.h>
#include <openquicktime/colormodels.h>

SDL_Surface *screen;

int n_audio, n_video, nb_channel;
int video_width = 100, video_height = 0;
int scaled_width = 100, scaled_height = 0;
int audio_rate, audio_length;
int video_rate, video_length;
quicktime_t* qtfile;

int debug = 0;

int fullscreen=0;
int screen_width, screen_height;
int pos_x = 0, pos_y = 0;


#define DEBUG(s) if (debug) s

/*
 */

typedef enum  {
  none,
  yuv420p,
  rgba32,
  rgb24
} colormodel;

colormodel cmodel = none;


/* frame buffer 
 * -----------
 */

#define NFRAMES 100

typedef struct {
  int dropped;           /* if 1, the frame is really not here */
  SDL_Surface*  surf;    /* for RGB/RGB playing */
  SDL_Overlay*  yuv;     /* for YUV playing */
} frame;

frame frames[NFRAMES];

volatile int frame_shown = -1;   /* frame beeing shown */
volatile int frame_computed = 0; /* frame beeing computed */
volatile int frame_seek = 0;     /* decompression must be preceeded by seek */

/* audio buffer 
 * ------------
 */

#define NCHUNKS  24

#define SAMPLES 4096

typedef struct {
  __s16* stream;
} chunk;
chunk chunks[NCHUNKS];

volatile int chunk_heard = -1;   /* audio chunk beeing heard */
volatile int chunk_computed = 0; /* audio chunk beeing computed */
volatile int chunk_seek = 0;     /* decompression must be preceeded by seek */

volatile int pause_audio = 0;    /* next audio tick will pause audio playback */

SDL_AudioSpec in;

/* Timing
 * ------
 */

double delay;                            /* normal delay between two frames */
Uint32 date_frame;                       /* when to show the next frame */
Uint32 last_date = 0, lastlast_date = 0; /* keep for monitoring fps */

volatile int video_dropping = 0; /* video decompression */
volatile int video_pausing = 0, audio_pausing = 0;  /* used to halt temporarily decompression */

SDL_TimerID timer = 0;

#define NDROP 7

const int drop_pattern[NDROP][NDROP] =
  { {0,0,0,0,0,0,0}, /* 0/7 */
    {1,0,0,0,0,0,0}, /* 1/7 */
    {1,0,0,1,0,0,0}, /* 2/7 */
    {1,0,0,1,0,1,0}, /* 3/7 */
    {1,0,1,1,0,1,0}, /* 4/7 */
    {1,0,1,1,0,1,1}, /* 5/7 */
    {1,1,1,1,0,1,1}  /* 6/7 */
  };

double show_limit = 1.5;   /* up to show_limit frames late are acceptable */
double drop_limit = 10.;   /* up to show_limit frames late => dropping */


/* Instrumentation
 * ---------------
 */
int inst_decomp = 0;
int inst_shown = 0;
int inst_dropped = 0;
int inst_late = 0;
int inst_very_late = 0;
int inst_seek = 0;
Uint32 inst_begin, inst_end;


/* decompression threads 
 * ---------------------
 */

SDL_Thread* video_decompress_thread;   
SDL_Thread* audio_decompress_thread;

SDL_mutex *video_compute_mutex, *audio_compute_mutex;
SDL_cond *video_next_cond, *audio_next_cond;
SDL_cond *video_computed_cond, *audio_computed_cond;


/* Print an error and quit */

void sdlerror()
{
  fprintf(stderr,"SDL error: %s\n",SDL_GetError());
  SDL_Quit();
  exit(1);
}

void customerror(const char* s)
{
  fprintf(stderr,"Error: %s\n",s);
  SDL_Quit();
  exit(1);
}


/* slider 
 * ------
 */

Uint32 color_black, color_white;

void update_slider (int frame)
{
  SDL_Rect rect;

  rect.x = pos_x; 
  rect.w = scaled_width;
  rect.y = pos_y+scaled_height;
  rect.h = 20;
  SDL_FillRect(screen,&rect,color_black);  
  rect.y = pos_y+scaled_height+8;
  rect.h = 4;
  SDL_FillRect(screen,&rect,color_white);  
  
  rect.y = pos_y+scaled_height;
  rect.h = 20;
  rect.x = pos_x+((double)(scaled_width-4)*((double)frame+1)/((double)video_length));
  rect.w = 4;
  SDL_FillRect(screen,&rect,color_white);  
}

/* decompression 
 * -------------
 */

int video_decompress_start (void* arg)
{
  unsigned char** decompress_row;
  int i;
  int dropping;
  int computed;

  decompress_row = (unsigned char**) malloc (video_height*sizeof(char*));
  if (!decompress_row) customerror("malloc error in video_decompress_start.");

  SDL_mutexP (video_compute_mutex);
  for (;;) {
    /* wait till the frame buffer is no longer full */
    if (frame_computed-frame_shown>NFRAMES-1 || video_pausing) {
      DEBUG(printf(" wait before computing frame %i\n",frame_computed));
      while (frame_computed-frame_shown>NFRAMES-1 || video_pausing)
	SDL_CondWait(video_next_cond,video_compute_mutex);
    }

    /* seek */
    if (frame_seek) {
      quicktime_set_video_position (qtfile, frame_computed, n_video);
      frame_seek = 0;
    }

    /* decompress the frame */
    dropping =   /* outside the mutex, video_dropping can change! */
      drop_pattern
      [(video_dropping>=NDROP)?(NDROP-1):video_dropping]
      [frame_computed%NDROP];
    computed = frame_computed; /* can also change outside mutex */

    SDL_mutexV (video_compute_mutex);
    DEBUG(printf(" computing frame %i\n",computed));
    inst_decomp++;
    if (n_video>=0) {
      switch (cmodel) {

      case rgba32:
	if (SDL_LockSurface(frames[computed%NFRAMES].surf)) sdlerror();
	for (i=0;i<video_height;i++) {
	  decompress_row[i] = (unsigned char*)frames[computed%NFRAMES].surf->pixels + i*video_width*4; }
	quicktime_decode_video (qtfile, 
				dropping ? BC_NONE : BC_RGBA8888, 
				decompress_row,
				n_video);
	SDL_UnlockSurface(frames[computed%NFRAMES].surf);
	break;

      case rgb24:
	if (SDL_LockSurface(frames[computed%NFRAMES].surf)) sdlerror();
	for (i=0;i<video_height;i++) {
	  decompress_row[i] = (unsigned char*)frames[computed%NFRAMES].surf->pixels + i*video_width*3; }
	quicktime_decode_video (qtfile, 
				dropping ? BC_NONE : BC_RGB888, 
				decompress_row,
				n_video);
	SDL_UnlockSurface(frames[computed%NFRAMES].surf);
	break;

      case yuv420p:
	if (SDL_LockYUVOverlay(frames[computed%NFRAMES].yuv)) sdlerror();
	decompress_row[0] = (unsigned char*)frames[computed%NFRAMES].yuv->pixels[0];
	decompress_row[2] = (unsigned char*)frames[computed%NFRAMES].yuv->pixels[1];
	decompress_row[1] = (unsigned char*)frames[computed%NFRAMES].yuv->pixels[2];	  
	quicktime_decode_video (qtfile, 
				dropping ? BC_NONE : BC_YUV420P,
				decompress_row,
				n_video);
	SDL_UnlockYUVOverlay(frames[computed%NFRAMES].yuv);
	break;
	
	
      }
      frames[computed%NFRAMES].dropped = dropping;
    }
    DEBUG(printf(" computed frame  %i\n",computed));
    SDL_mutexP (video_compute_mutex);
    
    /* signal there a new frame available in the buffer */
    if (!frame_seek) frame_computed++;
    SDL_CondSignal(video_computed_cond);
  }
  SDL_mutexV (video_compute_mutex);
  return 0;
}

int audio_decompress_start (void* arg)
{
  int i;
  int computed;
  __s16* audio_left;
  __s16* audio_right;
 
  audio_left  = (__s16*) malloc(SAMPLES*2);
  audio_right = (__s16*) malloc(SAMPLES*2);
  if (!audio_left || !audio_right) 
    customerror("malloc error in audio_decompress_start.");

  SDL_mutexP (audio_compute_mutex);
  for (;;) {
    if (chunk_computed-chunk_heard>NCHUNKS-1 || audio_pausing) {
      DEBUG(printf(" wait before computing chunk %i\n",chunk_computed));
      while (chunk_computed-chunk_heard>NCHUNKS-1 || audio_pausing)
	/* wait till the audio buffer is no longer full */
	SDL_CondWait(audio_next_cond,audio_compute_mutex);
    }

    /* seek */
    if (chunk_seek) {
      quicktime_set_audio_position (qtfile, chunk_computed*SAMPLES, n_audio);
      chunk_seek = 0;
    }

    /* decompress a chunk: SAMPLES samples of 2 bytes, each
       sample beeing mono or stereo */
    computed = chunk_computed; /* can change outside the mutex */
    SDL_mutexV (audio_compute_mutex);
    DEBUG(printf(" computing chunk %i\n",computed));
    if (nb_channel==1) {
      if (/*computed*SAMPLES>=audio_length ||*/
	  quicktime_decode_audio(qtfile,chunks[computed%NCHUNKS].stream,
				 NULL,SAMPLES,n_audio))
	memset (chunks[computed%NCHUNKS].stream,0,SAMPLES*2);
    }
    else if (nb_channel==2) {
      int i;
      __s16* s = chunks[computed%NCHUNKS].stream;
      __s16 *l = audio_left;
      __s16 *r = audio_right;
      longest pos = quicktime_audio_position (qtfile,n_audio);
      if (computed*SAMPLES>=audio_length ||
	  quicktime_decode_audio(qtfile,l,NULL,SAMPLES,2*n_audio)) {
	memset (chunks[computed%NCHUNKS].stream,0,SAMPLES*4);
      } else {
	quicktime_set_audio_position(qtfile,pos,n_audio);
	quicktime_decode_audio(qtfile,r,NULL,SAMPLES,2*n_audio+1);
	for (i=0;i<SAMPLES;i++) {
	  *s = *l; s++;l++;
	  *s = *r; s++;r++;
	}      
      }
    }
    DEBUG(printf(" computed chunk  %i\n",computed));
    SDL_mutexP (audio_compute_mutex);
    
    /* signal there a new chunk available in the buffer */
    if (!chunk_seek) chunk_computed++;
    SDL_CondSignal(audio_computed_cond);
  }
  SDL_mutexV (audio_compute_mutex);
  return 0;
}

/* Draw the current frame
 */

void show_frame(int frame)
{
  if (frame_shown>=video_length) return;
  if (n_video>=0) {
    SDL_Rect rect;
    rect.x = pos_x;
    rect.y = pos_y;
    rect.w = scaled_width;
    rect.h = scaled_height;

    switch (cmodel) {

    case rgba32:
    case rgb24:
      if (SDL_BlitSurface(frames[frame%NFRAMES].surf,NULL,screen,&rect)) sdlerror();
      update_slider(frame);
      SDL_UpdateRect(screen, 0, 0, 0, 0);
      break;
      
    case yuv420p: 
      if (SDL_DisplayYUVOverlay(frames[frame%NFRAMES].yuv,&rect)) sdlerror();
      update_slider(frame);
      SDL_UpdateRect(screen, pos_x, pos_y+scaled_height, scaled_width, 20);
      break;
      
    }
  } 
  else {
    update_slider(frame);
    SDL_UpdateRect(screen, 0, 0, 0, 0);
  }
}


/* Timers 
 * ------
 */

/* Timer asks for a new video frame */

/* if there is audio:
   - date_frame is set by the audio_tick procedure to synchronize,
   - if curtime > date_frame, drop frames (decompression AND showing
not every frame)
   - if curtime >> date_frame, pause decompression (so the next case is
willing to occur)
   - if there is no frame available, seek to the next interframe, set
frame_shown and date_frame accordingly
   - if the frame buffer is near empty, then also drop some frames

  if there is no audio, do as above without ever pausing decompression nor
skipping to a next keyframe
 */
  
Uint32 video_tick(Uint32 id, void* arg)
{
  Uint32 cur = SDL_GetTicks();
  Uint32 d = delay;

  DEBUG(printf("video tick, date=%i\n",cur));

  SDL_mutexP (video_compute_mutex);

  if (n_audio>=0) { pause_audio = 0; SDL_PauseAudio(0); }

  /* audio only */

  if (n_video<0) {
    frame_shown++;
    DEBUG(printf(" showing frame %i  (audio only)\n",frame_shown));
    show_frame(frame_shown);
    SDL_CondSignal (video_next_cond);
  }

  /* video, and there's a frame ready for us */

  else if (frame_computed>frame_shown+1) {

    /* too early ! */
    if (cur<date_frame-show_limit*delay) {
      SDL_mutexV (video_compute_mutex);
      return show_limit*delay;
    }

    frame_shown++;
    inst_shown++;
    
    /* show the frame, if not dropped */
    if (frames[frame_shown%NFRAMES].dropped ||
	drop_pattern
	[(video_dropping>=NDROP)?(NDROP-1):video_dropping]
	[frame_shown%NDROP]) {
      inst_dropped++;
      DEBUG(printf(" frame %i dropped\n",frame_shown));
    }
    else {
      DEBUG(printf(" showing frame %i\n",frame_shown));
      SDL_mutexV (video_compute_mutex);
      show_frame(frame_shown);
      SDL_mutexP (video_compute_mutex);
      SDL_CondSignal (video_next_cond);
    }

    /* we are in time */
    if (!date_frame || cur <= date_frame+show_limit*delay) {
      video_pausing = 0;
      audio_pausing = 0;
      if (frame_computed > frame_shown+NFRAMES/3) video_dropping = 0;
      else video_dropping = 1 + ((double)NDROP-1.)*
	((double)(frame_shown+NFRAMES/3-frame_computed))/
	((double)NFRAMES/3.+1.);
      if (video_dropping>=NDROP) video_dropping = NDROP-1;
      DEBUG(printf(" in time: cur=%i date_frame=%i (dropping %i/%i)\n",cur,date_frame,video_dropping,NDROP));
    }
    
    /* we are a little late: just try to drop frames */
    else if (cur <= date_frame+drop_limit*delay || n_audio<0) {
      audio_pausing = 0;
      video_pausing = 0;
      video_dropping = 1 + ((double)NDROP-1.)*
	((double)(cur-date_frame-delay*show_limit))/
	((double)(delay*drop_limit-delay*show_limit)+1.);
      if (video_dropping>=NDROP) video_dropping = NDROP-1;
      DEBUG(printf(" late: cur=%i date_frame=%i (dropping %i/%i)\n",cur,date_frame,video_dropping,NDROP));
      inst_late++;
    }

    /* we are in trouble: pause video decoding */
    else {
      DEBUG(printf(" very late: cur=%i date_frame=%i\n",cur,date_frame));
      audio_pausing = 1;
      video_pausing = 1;
      inst_very_late++;
    }

    cur = SDL_GetTicks();
    if (!date_frame) date_frame = cur;
    date_frame += delay;
    d = (cur < date_frame) ? (date_frame - cur) : 1;
    if (d>delay*show_limit) d = delay*show_limit;

    /* show fps */
    if (last_date && lastlast_date && last_date!=lastlast_date) {
      double lfps = 1000./((double)last_date-(double)lastlast_date+1.);
      DEBUG(printf(" latest=%2.2f fps\n",(double)lfps));
      fflush(stdout);
    } 
    else
      DEBUG(printf(" no fps info\n"));
    lastlast_date = last_date;
    last_date = cur;
  } 
  
  /* buffer empty => seek to next keyframe */
  else if (n_audio>=0) {
    const int min_delay = 6; /* allows this number of frame delay */
    int key = frame_shown+2;
    if (!date_frame) date_frame = cur;
    while (1) {
      key = quicktime_get_keyframe_after(qtfile, key, n_video);
      if (!key || key>=video_length || 
	  date_frame+(key-frame_shown)*delay>=cur+min_delay*delay) break;
      key++;
    }
    if (!key || key>=video_length) exit(0);
    date_frame += (key-frame_shown)*delay;
    printf("no frame available at date=%i shown=%i computed=%i skip to interframe=%i date=%i\n",cur,frame_shown,frame_computed,key,date_frame);
    frame_shown = key-1;
    frame_computed = key;
    frame_seek = 1;
    d = date_frame - cur;
    video_dropping = 0;
    audio_pausing = 0;
    video_pausing = 0;
    SDL_CondSignal (video_next_cond);
    inst_seek++;
  } else

    /* buffer empty and no audio => no skip to next keyframe */
    {
      date_frame = 0;
      d = delay*3;
      video_dropping = 0;
      audio_pausing = 0;
      video_pausing = 0;
    }

  SDL_mutexV (video_compute_mutex);
  return d+1;
}



/* SDL asks for a new audio frame */

/*
   - sets date_frame so that video is synmchronized with audio
   - if no more chunk, pause video/audio showing for a little time
*/

void audio_tick(void *userdata, Uint8 *stream, int len)
{
  DEBUG(printf("audio tick, date=%i\n",SDL_GetTicks()));

  if (len!=SAMPLES*nb_channel*2) 
    customerror("invalid buffer size in audio_tick.");
  
  SDL_mutexP (audio_compute_mutex);

  /* pause */
  if (pause_audio) {
    pause_audio = 0;
    memset (stream,0,len);
    SDL_PauseAudio(1);
  } else

  /* play chunk */
  if (chunk_computed>chunk_heard+1) {
    chunk_heard++;
    DEBUG(printf(" playing chunk %i\n",chunk_heard));
    memcpy(stream,chunks[chunk_heard%NCHUNKS].stream,len);
  } 

  /* no more data in buffer */
  else  {
    int date;
    double frame_delay = video_rate/2.; /* restarting after this many frames */

    /* play silence */
    memset(stream,0,len);

    /* pause audio/video playback */
    SDL_mutexP (video_compute_mutex);
    if (timer) {
      SDL_CondWait(video_next_cond,video_compute_mutex);
      SDL_RemoveTimer(timer);
    }
    SDL_PauseAudio(1);
    date = SDL_GetTicks() + frame_delay*delay;
    printf("no chunk available heard=%i computed=%i olddate=%i newdate=%i\n",chunk_heard,chunk_computed,date_frame,date);
    date_frame = date;
    SDL_mutexV (video_compute_mutex);
    if (timer) timer = SDL_AddTimer (delay*frame_delay,video_tick,NULL);
  }

  SDL_CondSignal (audio_next_cond);
  SDL_mutexV (audio_compute_mutex);
}


/* stream control 
 * --------------
 */

/* should be called when video is stopped */
/* you need to call play() to restart playing after bufferize */
void bufferize()
{  
  printf("  buffering ..."); fflush(stdout);
  if (n_video>=0) {
    SDL_mutexP (video_compute_mutex);
    while (frame_computed-frame_shown<NFRAMES*4/5-1 && frame_computed<video_length) {
      SDL_CondSignal(video_next_cond);
      SDL_CondWait(video_computed_cond,video_compute_mutex);
    }
    SDL_mutexV (video_compute_mutex);
  }
  if (n_audio>=0) {
    SDL_mutexP (audio_compute_mutex);
    while (chunk_computed-chunk_heard<NCHUNKS*4/5-1 && 
	   chunk_computed*SAMPLES<audio_length) {
      SDL_CondSignal(audio_next_cond);
      SDL_CondWait(audio_computed_cond,audio_compute_mutex);
    }
    SDL_mutexV (audio_compute_mutex);
  }
  printf("  done\n");
}

/* start playing after delay (+time for bufering) */
/* try hard to start audio & video synchronized!  */
/* buffer a little */
void play(int delay)
{
  if (delay<=0) delay = 1;
  SDL_LockAudio();
  SDL_mutexP (audio_compute_mutex);
  SDL_mutexP (video_compute_mutex);

  /* stop showing */
  if (timer) {
    SDL_CondWait(video_next_cond,video_compute_mutex);
    SDL_RemoveTimer(timer);
  }
  if (n_audio>=0) pause_audio = 1;

  /* start decoding & buffer a little */
  video_pausing = 0;
  audio_pausing = 0;
  video_dropping = 0;
  last_date = lastlast_date = 0;
  SDL_CondSignal(video_next_cond);
  SDL_CondSignal(audio_next_cond);
  if (n_video>=0) {
    while (frame_computed-frame_shown<NFRAMES/5+1 && frame_computed<video_length) {
      SDL_CondSignal(video_next_cond);
      SDL_CondWait(video_computed_cond,video_compute_mutex);
    }
  }
  SDL_mutexV (video_compute_mutex);
  if (n_audio>=0) {
    while (chunk_computed-chunk_heard<NCHUNKS/5+1 && 
	   chunk_computed*SAMPLES<audio_length) {
      SDL_CondSignal(audio_next_cond);
      SDL_CondWait(audio_computed_cond,audio_compute_mutex);
    }
  }
  SDL_mutexV (audio_compute_mutex);
  SDL_UnlockAudio();

  /* schedule showing */
  date_frame = SDL_GetTicks()+delay;
  if (n_audio>=0) {
    double sample = /* sample corresponding to frame shown */
      ((double)frame_shown+1.)*((double)audio_rate)/((double)video_rate);
    double middle = /* sample approximately corresponding to chunk heard */
      ((double)chunk_heard+1.)*SAMPLES;
    double extra_delay = /* frame should be delayed by extra_delay more */
      ((double)sample-(double)middle+SAMPLES/2)*1000./((double)audio_rate);
    if (extra_delay>=0) date_frame += extra_delay;
    else frame_shown -= extra_delay*video_rate/1000.;
    DEBUG(printf(" extra_delay = %i\n",(int)extra_delay));
  }
  DEBUG(printf("  play at date=%i in %i\n",date_frame,delay));
  timer = SDL_AddTimer(delay,video_tick,NULL); 
}

void stop()
{
  SDL_LockAudio();
  SDL_mutexP (audio_compute_mutex);
  SDL_mutexP (video_compute_mutex);
  if (timer) {
    SDL_CondWait(video_next_cond,video_compute_mutex);
    SDL_RemoveTimer(timer);
  }
  if (n_audio>=0) pause_audio = 1;
  timer = 0;
  last_date = lastlast_date = 0;
  video_pausing = 0;
  audio_pausing = 0;
  video_dropping = 0;
  SDL_mutexV (video_compute_mutex);
  SDL_mutexV (audio_compute_mutex);
  SDL_UnlockAudio();
}

/* you need to call play() to restart playing after seek */
/* if exact=0, then seeks to the nearest keyframe before (faster) */
void seek(int frame,
	  int exact)
{
  SDL_LockAudio();
  SDL_mutexP (audio_compute_mutex);
  SDL_mutexP (video_compute_mutex);

  /* video seeking */
  if (timer) {
    SDL_CondWait(video_next_cond,video_compute_mutex);
    SDL_RemoveTimer(timer);
  }
  timer = 0;
  video_pausing = 0;
  audio_pausing = 0;
  video_dropping = 0;
  last_date = lastlast_date = 0;
  frame_seek = 1;
  if (n_video<0) frame_computed = frame;
  else frame_computed = quicktime_get_keyframe_before(qtfile, frame, n_video);
  if (exact) frame_shown = frame-1;
  else frame_shown = frame_computed-1;
  
  /* audio seeking */
  if (n_audio>=0) {
    int sample = 
      ((double)frame)*((double)audio_rate)/((double)video_rate);
    chunk_seek = 1;
    chunk_computed = sample/SAMPLES;
    chunk_heard = chunk_computed-1;
    pause_audio = 1;
  }

  DEBUG(printf(" seeking to frame %i (keyframe=%i) chunk %i (cur=%i)\n",
	       frame, frame_computed, chunk_computed,SDL_GetTicks()));

  /* buffer one frame and show it immediately */
  if (n_video>=0) {
    while (frame_computed-frame_shown<2 && frame_computed<video_length) {
      SDL_CondSignal(video_next_cond);
      SDL_CondWait(video_computed_cond,video_compute_mutex);
    }
  }
  SDL_mutexV (video_compute_mutex);
  SDL_CondSignal(audio_next_cond);
  SDL_mutexV (audio_compute_mutex);
  show_frame(frame_shown+1);
  SDL_UnlockAudio();
}


/* main */

void quit()
{
  inst_end = SDL_GetTicks();
  printf("\n\nTime %i ms\n",inst_end-inst_begin);
  printf("Frame shown %i, decompressed %i\n",inst_shown,inst_decomp);
  if (inst_shown)
    printf("Dropped %i (%i%%), late %i (%i%%), very late %i (%i%%)\n",
	   inst_dropped, inst_dropped*100/inst_shown,
	   inst_late, inst_late*100/inst_shown,
	   inst_very_late, inst_very_late*100/inst_shown);
  printf("Forced seek %i\n",inst_seek);
  printf("End at frame %i\n",frame_shown+1);
  /*SDL_Quit();*/
  exit(0);
}


void help() 
{ 
  char* options[] = {"--debug","--noaudio","--novideo","--mono","--fullscreen",
		     "--scale (float)","--rgba32","--rgb24","--yuv420",NULL};
  char** a;
  printf("oqtplayer [options] filename\n");
  printf("options:\n");
  for (a=options;*a;a++) printf("         %s\n",*a);
  exit(1);
}

int main(int argc, char** argv)
{
  char* filename = NULL;
  int nb_audio, nb_video;
  int i;
  int mono=0;
  int can_scale = 1;
  double scale = 0;
  colormodel wanted = none;

  n_audio = n_video = 0;

  /* parse args */

  for (argc--,argv++;argc && *argv;argc--,argv++) {
    if (!strcmp(*argv,"--novideo")) n_video=-1;
    else if (!strcmp(*argv,"--noaudio")) n_audio=-1;
    else if (!strcmp(*argv,"--mono")) mono=1;
    else if (!strcmp(*argv,"--yuv420")) wanted=yuv420p;
    else if (!strcmp(*argv,"--rgba32")) wanted=rgba32;
    else if (!strcmp(*argv,"--rgb24")) wanted=rgb24;
    else if (!strcmp(*argv,"--scale")) 
      if (argc>1 && argv[1]) { scale = atof(argv[1]); argv++; argc--; } else help();
    else if (!strcmp(*argv,"--fullscreen")) fullscreen=1;
    else if (!strcmp(*argv,"--debug")) debug=1;
    else if (!strcmp(*argv,"-?") || !strcmp(*argv,"-h") || !strcmp(*argv,"--help") || **argv=='-') help();
    else filename=*argv;
  }

  if (!filename) help();

  /* open QT file and get infos */

  qtfile = quicktime_open(filename, 1, 0);
  if (!qtfile) { 
    fprintf(stderr, "Unable to open file '%s'\n", filename); 
    exit(1); 
  }

  nb_audio = quicktime_audio_tracks(qtfile);
  nb_video = quicktime_video_tracks(qtfile);

  printf("\n");

  if (n_video>=0 && n_video<nb_video &&
      quicktime_supported_video(qtfile, n_video)) {
    video_width = quicktime_video_width(qtfile, n_video);
    video_height = quicktime_video_height(qtfile, n_video);
    video_rate = quicktime_frame_rate(qtfile, n_video);
    video_length = quicktime_video_length(qtfile, n_video);
    printf("Video track %i/%i, %ix%i at %i fps, length %i s.\n",
	   n_video, nb_video-1, video_width, video_height, 
	   video_rate, video_length/video_rate);

    cmodel = none;
    if ((wanted==rgb24 || wanted==none) && quicktime_reads_cmodel(qtfile, BC_RGB888, n_video)) cmodel = rgb24;
    if ((wanted==rgba32 || wanted==none) && quicktime_reads_cmodel(qtfile, BC_RGBA8888, n_video)) cmodel = rgba32;
    if ((wanted==yuv420p || wanted==none) && quicktime_reads_cmodel(qtfile, BC_YUV420P, n_video)) cmodel = yuv420p;
    
    switch (cmodel) {
    case none:
      printf("No colormodel supported\n");
      exit(1);
      break;
    
    case rgba32:
      printf("Output in RGBA32 (no scaling)\n");
      can_scale = 0;
      break;

    case rgb24:
      printf("Output in RGB24 (no scaling)\n");
      can_scale = 0;
      break;

    case yuv420p:
      printf("Output in YUV420\n");
      break;
    }

  }
  else {
    printf("No such video track supported.\n");
    n_video = -1;
    fullscreen = 0;
  }

  if (n_audio>=0 && n_audio<nb_audio &&
      quicktime_supported_audio(qtfile, n_audio)) {
    audio_rate = quicktime_sample_rate(qtfile, n_audio);
    audio_length = quicktime_audio_length(qtfile, n_audio);
    nb_channel = quicktime_track_channels(qtfile, n_audio);
    printf("Audio track %i/%i at %i sps, %i channel(s), length %i s.\n",
	   n_audio,nb_audio-1,audio_rate, nb_channel, audio_length/audio_rate);
  }
  else {
    printf("No such audio track supported.\n");
    n_audio = -1;
  }

  if (n_audio>=0 && n_video<0) {
    video_rate = 25;
    video_length = (video_rate*audio_length)/audio_rate;
  }

  if (n_audio<0 && n_video<0) {
    printf("No audio or video supported.\n");
    return 0;
  }

  /* init SDL & buffers */


  if (SDL_Init(SDL_INIT_AUDIO|SDL_INIT_VIDEO|SDL_INIT_TIMER)<0) {
    fprintf(stderr,"Unable to initialize SDL: %s\n",SDL_GetError());
    exit(1);
  }

  if (can_scale && scale) {
    scaled_width = (double)video_width*scale;
    scaled_height = (double)video_height*scale;
  } 
  else {
    /* cannot scale or no scale forced => *1 */
    scaled_width = video_width;
    scaled_height = video_height;
  }    


  if (fullscreen) {
    SDL_Rect** r;
    /* find the screen with lower resolution that is bigger than the video */
    r = SDL_ListModes(NULL,SDL_FULLSCREEN|SDL_ANYFORMAT);
    screen_width=0;
    screen_height=0;
    for (;*r;r++)
      if ((*r)->w>=scaled_width && (*r)->h>=scaled_height+20 &&
	  ((*r)->w<screen_width && (*r)->h<screen_height || !screen_width)) {
	screen_width = (*r)->w;
	screen_height = (*r)->h;
      }
    if (!screen_width) {
      fprintf(stderr,"Unable to find a fullscreen allowing at least %ix%i\n",scaled_width,scaled_height+20);
      exit(1);
    }
    printf("Using fullscreen mode %ix%i\n",screen_width,screen_height);
    screen = SDL_SetVideoMode(screen_width, screen_height, cmodel==rgb24 ? 24 : 32, SDL_ANYFORMAT|SDL_FULLSCREEN|SDL_SWSURFACE|SDL_HWSURFACE);

    /* we can scale & no scale forced => scale to fullscreen  */
    if (can_scale && !scale) {
      scaled_width = screen_width;
      scaled_height = video_height*scaled_width/video_width;
      if (scaled_height>screen_height-20) {
	scaled_height = screen_height-20;
	scaled_width = video_width*scaled_height/video_height;
      }
    }

    /* center */
    pos_x = (screen_width-scaled_width)/2;
    pos_y = (screen_height-scaled_height-20)/2;
  }
  else
    screen = SDL_SetVideoMode(scaled_width, scaled_height+20, cmodel==rgb24 ? 24: 32, SDL_ANYFORMAT|SDL_SWSURFACE|SDL_HWSURFACE);

  if (!screen)
    {
      fprintf(stderr,"Unable to set video mode: %s\n",SDL_GetError());
      SDL_Quit();
      exit(1);
    } 
  
  if (n_audio>=0) {
    if (mono) nb_channel = 1;
    in.freq = audio_rate;
    in.format = AUDIO_S16SYS;
    in.channels = nb_channel;
    in.samples = SAMPLES;
    in.callback = audio_tick;
    if (SDL_OpenAudio (&in,NULL)==-1)
    {
      fprintf(stderr,"Unable to open audio.\n");
      SDL_Quit();
      exit(1);
    } 
    
  }
  
  SDL_WM_SetCaption("OQTPlayer", NULL);
  
  if (n_video>=0)
    for (i=0;i<NFRAMES;i++){

      switch (cmodel) {
      case rgba32:
#if SDL_BYTEORDER==SDL_LIL_ENDIAN
	frames[i].surf =
	  SDL_CreateRGBSurface(SDL_SWSURFACE, 
			       video_width, video_height, 32,
			       0x0000ff, 0x00ff00, 0xff0000, 0);
#else
        frames[i].surf = 
	  SDL_CreateRGBSurface(SDL_SWSURFACE, 
			       video_width, video_height, 32,
			       0xff000000, 0x00ff0000, 0x0000ff00, 0);
#endif
	if (!frames[i].surf) sdlerror();
	break;

      case rgb24:
#if SDL_BYTEORDER==SDL_LIL_ENDIAN
	frames[i].surf =
	  SDL_CreateRGBSurface(SDL_SWSURFACE, 
			       video_width, video_height, 24,
			       0x0000ff, 0x00ff00, 0xff0000, 0);
#else
        frames[i].surf = 
	  SDL_CreateRGBSurface(SDL_SWSURFACE, 
			       video_width, video_height, 24,
			       0xff0000, 0x00ff00, 0x0000ff, 0);
#endif
	if (!frames[i].surf) sdlerror();
	break;

      case yuv420p:
	frames[i].yuv = SDL_CreateYUVOverlay(video_width, video_height, 
					     SDL_YV12_OVERLAY, screen);
	if (!frames[i].yuv) sdlerror();
	break;
      }
    }
  
  if (n_audio>=0)
    for (i=0;i<NCHUNKS;i++){
      chunks[i].stream = (__s16*) malloc(SAMPLES*2*nb_channel);
      if (!chunks[i].stream) sdlerror();
    }
  
  color_black = SDL_MapRGB (screen->format,0,0,0);
  color_white = SDL_MapRGB (screen->format,196,196,196);
  
  /* create threads & timers */
  
  audio_compute_mutex = SDL_CreateMutex();
  audio_next_cond = SDL_CreateCond();
  audio_computed_cond = SDL_CreateCond();
  video_compute_mutex = SDL_CreateMutex();
  video_next_cond = SDL_CreateCond();
  video_computed_cond = SDL_CreateCond();

  if (n_video>=0) {
    video_decompress_thread = SDL_CreateThread(video_decompress_start, NULL);
    if (!video_decompress_thread) sdlerror();
  }
  if (n_audio>=0) {
    audio_decompress_thread = SDL_CreateThread(audio_decompress_start, NULL);
    if (!audio_decompress_thread) sdlerror();
  }

  inst_begin = SDL_GetTicks();

  /* SDL event loop */

  delay = 1000.0/((double)video_rate);

  bufferize();
  play(0);

  for (;;) {
    SDL_Event event;
    
    SDL_WaitEvent(&event);
    
    switch (event.type) {

    case SDL_KEYDOWN:
      switch( event.key.keysym.sym ) {

      case SDLK_p:
	play(0);
	break;

      case SDLK_s:
	stop();
	break;

      case SDLK_ESCAPE:
	quit();
	break;
      }
      break;
      
    case SDL_MOUSEBUTTONUP:
      if (event.button.button==SDL_BUTTON_LEFT) play(delay*video_rate/2.);
      else play(0);
      break;

    case SDL_MOUSEBUTTONDOWN:
      stop();
      if (event.button.button==SDL_BUTTON_LEFT &&
	  event.motion.y>=pos_y+scaled_height &&
	  event.motion.x>=pos_x &&
	  event.motion.x<=pos_x+scaled_width)
	seek((double)(event.motion.x-pos_x)*(double)video_length/
	     (double)scaled_width,
	     1);
      break;

    case SDL_QUIT:
      quit();
    }
  }

  quit();
  return 0;
}
