#include <iostream>

#include "paman.hh"
#include "SinkWindow.hh"

#define GLADE_NAME "sinkWindow"

SinkWindow::SinkWindow(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade) :
    Gtk::Window(cobject),
    nameLabel(NULL),
    descriptionLabel(NULL),
    indexLabel(NULL),
    sampleTypeLabel(NULL),
    latencyLabel(NULL),
    ownerModuleLabel(NULL),
    monitorSourceLabel(NULL),
    volumeLabel(NULL),
    closeButton(NULL),
    toMonitorSourceButton(NULL),
    toOwnerModuleButton(NULL),
    volumeResetButton(NULL),
    volumeMuteButton(NULL),
    volumeMeterButton(NULL),
    volumeScale(NULL) {

    refGlade->get_widget("nameLabel", nameLabel);
    refGlade->get_widget("descriptionLabel", descriptionLabel);
    refGlade->get_widget("indexLabel", indexLabel);
    refGlade->get_widget("sampleTypeLabel", sampleTypeLabel);
    refGlade->get_widget("latencyLabel", latencyLabel);
    refGlade->get_widget("ownerModuleLabel", ownerModuleLabel);
    refGlade->get_widget("monitorSourceLabel", monitorSourceLabel);
    refGlade->get_widget("closeButton", closeButton);
    refGlade->get_widget("toMonitorSourceButton", toMonitorSourceButton);
    refGlade->get_widget("toOwnerModuleButton", toOwnerModuleButton);
    refGlade->get_widget("volumeLabel", volumeLabel);
    refGlade->get_widget("volumeScale", volumeScale);
    refGlade->get_widget("volumeResetButton", volumeResetButton);
    refGlade->get_widget("volumeMuteButton", volumeMuteButton);
    refGlade->get_widget("volumeMeterButton", volumeMeterButton);

    closeButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onCloseButton));
    toMonitorSourceButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onToMonitorSourceButton));
    toOwnerModuleButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onToOwnerModuleButton));
    volumeScale->signal_value_changed().connect(sigc::mem_fun(*this, &SinkWindow::onVolumeScaleValueChanged));
    volumeResetButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onVolumeResetButton));
    volumeMuteButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onVolumeMuteButton));
    volumeMeterButton->signal_clicked().connect(sigc::mem_fun(*this, &SinkWindow::onVolumeMeterButton));
}

SinkWindow* SinkWindow::create() {
    SinkWindow *w = NULL;
    Glib::RefPtr<Gnome::Glade::Xml> refXml = Gnome::Glade::Xml::create(GLADE_FILE, GLADE_NAME);
    refXml->get_widget_derived(GLADE_NAME, w);
    return w;
}

void SinkWindow::updateInfo(const SinkInfo &i) {
    char t[64], ss[PA_SAMPLE_SPEC_SNPRINT_MAX];
    double percent, db;

    nameLabel->set_text(i.name);
    descriptionLabel->set_text(i.description);
    snprintf(t, sizeof(t), "#%u", i.index);
    indexLabel->set_text(t);
    pa_sample_spec_snprint(ss, sizeof(ss), &i.sample_spec);
    sampleTypeLabel->set_text(ss);
    snprintf(t, sizeof(t), "#%u", i.owner_module);
    ownerModuleLabel->set_text(t);

    snprintf(t, sizeof(t), "%0.0f usec", (double) i.latency);
    latencyLabel->set_text(t);

    SourceInfo *source = serverInfoManager->getSourceInfo(i.monitor_source);
    monitorSourceLabel->set_text(source->name);

    percent = pa_volume_to_user(i.volume) * 100;
    db = pa_volume_to_dB(i.volume);
    volumeScale->set_value(percent);
    if (db != PA_DECIBEL_MININFTY)
        snprintf(t, sizeof(t), "%0.0f%% (%0.2fdB)", percent, db);
    else
        snprintf(t, sizeof(t), "%0.0f%% (-&#8734;dB)", percent);
    volumeLabel->set_markup(t);
    
    set_title("Sink: "+i.name);
    
    monitor_source = i.monitor_source;
    owner_module = i.owner_module;
    index = i.index;
    monitor_source_name = i.monitor_source_name;

    toOwnerModuleButton->set_sensitive(owner_module != (uint32_t) -1);
}

void SinkWindow::onCloseButton() {
    hide();
}

void SinkWindow::onToMonitorSourceButton() {
    serverInfoManager->showSourceWindow(monitor_source);
}

void SinkWindow::onToOwnerModuleButton() {
    if (owner_module != (uint32_t) -1)
        serverInfoManager->showModuleWindow(owner_module);
}

void SinkWindow::onVolumeScaleValueChanged() {
    serverInfoManager->setSinkVolume(index, pa_volume_from_user(volumeScale->get_value()/100));
}

void SinkWindow::onVolumeResetButton() {
    serverInfoManager->setSinkVolume(index, PA_VOLUME_NORM);
}

void SinkWindow::onVolumeMuteButton() {
    serverInfoManager->setSinkVolume(index, PA_VOLUME_MUTED);
}

bool SinkWindow::on_delete_event(GdkEventAny*) {
    hide();
    return false;
}

void SinkWindow::onVolumeMeterButton() {
    serverInfoManager->runVolumeMeter(monitor_source_name);
}
