/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * pbs_dsh - a distribute task program using the Task Management API
 *
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <sys/signal.h>
#include "tm.h"

/*
 * a bit of code to map a tm_ error number to the symbol
 */

struct tm_errcode {
	int	trc_code;
	char   *trc_name;
} tm_errcode[] = {
	{ TM_ESYSTEM, "TM_ESYSTEM" },
	{ TM_ENOEVENT, "TM_ENOEVENT" },
	{ TM_ENOTCONNECTED, "TM_ENOTCONNECTED" },
	{ TM_EUNKNOWNCMD, "TM_EUNKNOWNCMD" },
	{ TM_ENOTIMPLEMENTED, "TM_ENOTIMPLEMENTED" },
	{ TM_EBADENVIRONMENT, "TM_EBADENVIRONMENT" },
	{ TM_ENOTFOUND, "TM_ENOTFOUND" },
	{ 0,		"?" }
};

int	   *ev;
tm_event_t *events_spawn;
tm_event_t *events_obit;
int	    numnodes;
tm_task_id *tid;
int	    verbose = 0;
sigset_t	allsigs;
char		*id;

char *get_ecname(int rc)
{
	struct tm_errcode *p;

	for (p=&tm_errcode[0]; p->trc_code; ++p) {
		if (p->trc_code == rc)
			break;
	}
	return (p->trc_name);
}

int	fire_phasers = 0;

void
bailout(sig)
	int	sig;
{
	fire_phasers = sig;
}
	
/*
 * wait_for_task - wait for all spawned tasks to 
 *	a. have the spawn acknowledged, and
 *	b. the task to terminate and return the obit with the exit status
 */

void wait_for_task(nspawned)
	int	*nspawned;	/* number of tasks spawned */
{
	int	    c;
	tm_event_t  eventpolled;
	int	    nobits = 0;
	int	    rc;
	int	    tm_errno;

	while ( *nspawned || nobits ) {
		if (verbose) {
			printf("pbsdsh: waiting on %d spawned and %d obits\n",
				*nspawned, nobits);
		}

		if (fire_phasers) {
			tm_event_t	event;

			for (c=0; c<numnodes; c++) {
				if (*(tid+c) == TM_NULL_TASK)
					continue;
				printf("pbsdsh: killing task %lu signal %d\n",
					*(tid+c), fire_phasers);
				(void)tm_kill(*(tid+c), fire_phasers, &event);
			}
			tm_finalize();
			exit(1);
		}
				

		sigprocmask(SIG_UNBLOCK, &allsigs, NULL);
		rc = tm_poll(TM_NULL_EVENT, &eventpolled, 1, &tm_errno);
		sigprocmask(SIG_BLOCK, &allsigs, NULL);

		if (rc != TM_SUCCESS) {
			fprintf(stderr,"%s: Event poll failed, error %s\n",
					id, get_ecname(rc));
			exit(2);
		}

		for (c = 0; c < numnodes; ++c) {
			if (eventpolled == *(events_spawn + c)) {
				/* spawn event returned - register obit */
				(*nspawned)--;
				if (tm_errno) {
				    fprintf(stderr, "error %d on spawn\n", tm_errno);
				    continue;
				}
				rc = tm_obit(*(tid+c), ev+c, events_obit+c);
				if (rc == TM_SUCCESS) {
				    if (*(events_obit+c) == TM_NULL_EVENT) {
					if (verbose) {
					    fprintf(stderr,"task already dead\n");
					}
				    } else if (*(events_obit+c) == TM_ERROR_EVENT) {
					if (verbose) {
					    fprintf(stderr, "Error on Obit return\n");
					}
				    } else {
					nobits++;
				    }
				} else if (verbose) {
				    fprintf(stderr, "%s: failed to register for task termination notice, task %d\n", id, c);
				}


			} else if (eventpolled == *(events_obit + c)) {
				/* obit event, task exited */
				nobits--;
				*(tid+c) = TM_NULL_TASK;
				if (verbose || *(ev+c) != 0) {
					printf("%s: task %d exit status %d\n",
						id, c, *(ev+c));
				}
			}
		}
	}
}

main(argc, argv, envp)
	int   argc;
	char *argv[];
	char *envp[];
{
	int c;
	int err = 0;
	int ncopies = -1;
	int onenode = -1;
	int rc;
	struct tm_roots rootrot;
	int	 nspawned = 0;
	int	 nobits = 0;
	tm_event_t eventin = TM_NULL_EVENT;
	tm_node_id *nodelist;
	int tm_errno;
	int start;
	int stop;
	int sync = 0;
	struct	sigaction	act;

	extern int   optind;
	extern char *optarg;
	extern int pbs_tcp_interrupt;

	while ((c = getopt(argc, argv, "c:n:sv")) != EOF) {
	    switch(c) {
		case 'c':
		    ncopies = atoi(optarg);
		    if (ncopies < 0) {
			err = 1;
		    }
		    break;
		case 'n':
		    onenode = atoi(optarg);
		    if (onenode < 0) {
			err = 1;
		    }
		    break;
		case 's':
		    sync = 1;		/* force synchronous spawns */
		    break;
		case 'v':
		    verbose = 1;	/* turn on verbose output */
		    break;
		default:
		    err = 1;
		    break;
	    }
	}
	if (err || (onenode >= 0 && ncopies >= 0)) {
		fprintf(stderr, "Usage: %s [-c copies][-s]-[v] program [args]...]\n", argv[0]);
		fprintf(stderr, "       %s [-n nodenumber][-s]-[v] program [args]...\n", argv[0]);
		fprintf(stderr, "Where -c copies =  run  copy of \"args\" on the first \"copies\" nodes,\n");
		fprintf(stderr, "      -n nodenumber = run a copy of \"args\" on the \"nodenumber\"-th node,\n");

		fprintf(stderr, "      -s = forces synchronous execution,\n");
		fprintf(stderr, "      -v = forces verbose output.\n");

		exit(1);
	}

	id = argv[0];
	if (getenv("PBS_ENVIRONMENT") == 0) {
		fprintf(stderr, "%s: not executing under PBS\n", id);
		return 1;
	}

/*
 *	Set up interface to the Task Manager 
 */
	if ((rc = tm_init(0, &rootrot)) != TM_SUCCESS) {
		fprintf(stderr, "%s: tm_init failed, rc = %s (%d)\n", id,
						get_ecname(rc), rc);
		return 1;
	}

	sigemptyset(&allsigs);
	sigaddset(&allsigs, SIGHUP);
	sigaddset(&allsigs, SIGINT);
	sigaddset(&allsigs, SIGTERM);

	act.sa_mask = allsigs;
	act.sa_flags = 0;
	/*
	** We want to abort system calls and call a function.
	*/
#ifdef	SA_INTERRUPT
	act.sa_flags |= SA_INTERRUPT;
#endif
	act.sa_handler = bailout;
	sigaction(SIGHUP, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGTERM, &act, NULL);
	pbs_tcp_interrupt = 1;

#ifdef DEBUG
	if (rootrot.tm_parent == TM_NULL_TASK) {
		printf("%s: I am the mother of all tasks\n", id);
	} else {
		printf("%s: I am but a child in the scheme of things\n", id);
	}
#endif /* DEBUG */

	if ((rc = tm_nodeinfo(&nodelist, &numnodes)) != TM_SUCCESS) {
		fprintf(stderr, "%s: tm_nodeinfo failed, rc = %s (%d)\n", id,
						get_ecname(rc), rc);
		return 1;
	}

	/* malloc space for various arrays based on number of nodes/tasks */

	tid = (tm_task_id *)calloc(numnodes, sizeof (tm_task_id));
	if (tid == NULL) {
		fprintf(stderr, "%s: malloc of task ids failed\n", id);
		return 1;
	}
	events_spawn = (tm_event_t *)calloc(numnodes, sizeof (tm_event_t));
	events_obit  = (tm_event_t *)calloc(numnodes, sizeof (tm_event_t));
	ev = (int *)calloc(numnodes, sizeof (int));
	for (c = 0; c < numnodes; c++ ) {
		*(tid + c)          = TM_NULL_TASK;
		*(events_spawn + c) = TM_NULL_EVENT;
		*(events_obit  + c) = TM_NULL_EVENT;
		*(ev + c)	    = 0;
	}


	/* Now spawn the program to where it goes */

	if (onenode >= 0) {

		/* Spawning one copy onto logical node "onenode" */

		start = onenode;
		stop  = onenode + 1;

	} else if (ncopies >= 0) {
		/* Spawn a copy of the program to the first "ncopies" nodes */

		start = 0;
		stop  = ncopies;
	} else {
		/* Spawn a copy on all nodes */

		start = 0;
		stop  = numnodes;
	}

	sigprocmask(SIG_BLOCK, &allsigs, NULL);
	for (c = start; c < stop; ++c) {
		if ((rc = tm_spawn(argc-optind,
			     argv+optind,
			     NULL,
			     *(nodelist + c),
			     tid + c,
			     events_spawn + c)) != TM_SUCCESS)  {
			fprintf(stderr, "%s: spawn failed on node %d err %s\n",
				id, c, get_ecname(rc));
		} else {
			if (verbose)
				printf("%s: spawned task %d\n", id, c);
			++nspawned;
			if (sync)
				wait_for_task(&nspawned); /* one at a time */
		}

	}

	if (sync == 0)
		wait_for_task(&nspawned);	/* wait for all to finish */

		
/*
 *	Terminate interface with Task Manager
 */
	tm_finalize();

	return 0;
}
