/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
**	This program exists to give a way to mark nodes
**	Down, Offline, or Free in PBS.
**
**	usage: pbsnodes [-s server][-{c|l|o|r}] node node ...
**
**	where the node(s) are the names given in the node
**	description file.
**
**	pbsnodes			clear "DOWN" from all nodes so marked
** 
**	pbsnodes node1 node2		set nodes node1, node2 "DOWN"
**					unmark "DOWN" from any other node
**
**	pbsnodes -a			list all nodes 
**
**	pbsnodes -l			list all nodes marked in any way
**
**	pbsnodes -o node1 node2		mark nodes node1, node2 as OFF_LINE
**					even if currently in use.
**
**	pbsnodes -r node1 node2 	clear OFF_LINE from listed nodes
**
**	pbsnodes -c node1 node2		clear OFF_LINE or DOWN from listed nodes
*/
#include	<pbs_config.h>   /* the master config generated by configure */

#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>

#include	"portability.h"
#include	"pbs_ifl.h"

#define	DOWN	0
#define	LIST	1
#define	CLEAR	2
#define	OFFLINE	3
#define	RESET	4
#define ALL	5


int quiet = 0;

static char ident[] = "@(#) $RCSfile: pbsnodes.c,v $ $Revision: 2.2 $";

/*
 * cmp_node_name - compare two node names, allow the second to match the
 *	first if the same upto a dot ('.') in the second; i.e.
 *	"foo" == "foo.bar"
 *
 *	return 0 if match, 1 if not
 */

static int cmp_node_name(char *n1, char *n2)
{
	while ((*n1 != '\0') && (*n2 != '\0')) {
		if (*n1 != *n2)
			break;
		n1++;
		n2++;
	}
	if (*n1 == *n2) 
		return 0;
	else if ((*n1 == '.') && (*n2 == '\0'))
		return 0;
	else
		return 1;
}

static void prt_node_attr(struct batch_status *pbs)
{
	struct attrl *pat;

	for (pat = pbs->attribs; pat; pat = pat->next) {
		printf("     %s = %s\n", pat->name, pat->value);
	}
}

static char *get_nstate(struct batch_status *pbs)
{
	struct attrl *pat;

	for (pat = pbs->attribs; pat; pat = pat->next) {
		if (strcmp(pat->name, ATTR_NODE_state) == 0)
			return pat->value;
	}
	return "";
}

/*
 * is_down - returns indication if node is marked down or not
 *
 *	returns 1 if node is down and 0 if not down
 */

static int is_down(struct batch_status *pbs)
{
	struct attrl *pat;

	if (strstr(get_nstate(pbs), ND_down) != NULL)
		return 1;
	else
		return 0;
}

static int is_offline(struct batch_status *pbs)
{
	struct attrl *pat;

	if (strstr(get_nstate(pbs), ND_offline) != NULL)
		return 1;
	else
		return 0;
}

static int marknode(int con, char *name, 
		    char *state1, enum batch_op op1,
 		    char *state2, enum batch_op op2)
{
	char	       *errmsg;
	struct attropl  new[2];
	int		rc;

	new[0].name     = ATTR_NODE_state;
	new[0].resource = NULL;
	new[0].value    = state1;
	new[0].op       = op1;

	if (state2 == NULL) {
		new[0].next     = NULL;
	} else {
		new[0].next     = &new[1];
		new[1].next     = NULL;
		new[1].name     = ATTR_NODE_state;
		new[1].resource = NULL;
		new[1].value    = state2;
		new[1].op	= op2;
	}

	rc = pbs_manager(con, MGR_CMD_SET, MGR_OBJ_NODE, name, new, NULL);
	if (rc && !quiet) {
		fprintf(stderr, "Error marking node %s - ", name);
		if ((errmsg = pbs_geterrmsg(con)) != NULL)
			fprintf(stderr, "%s\n", errmsg);
		else
			fprintf(stderr, "error: %d\n",pbs_errno);
	}
	return (rc);
}


main(argc, argv)
    int		argc;
    char	*argv[];
{
	struct batch_status *bstatus = NULL;
	int	con;
	char	*def_server;
	int	 errflg = 0;
	char	*errmsg;
	int	i;
	extern	char	*optarg;
	extern	int	 optind;
	char	       **pa;
	struct batch_status *pbstat;
	int	flag = DOWN;


	/* get default server, may be changed by -s option */

	def_server = pbs_default();
	if (def_server == NULL)
		def_server = "";

	while ((i = getopt(argc, argv, "acloqrs:")) != EOF)
	switch (i) {

	case 'a':
		flag = ALL;
		break;

	case 'c':
		flag = CLEAR;
		break;

	case 'l':
		flag = LIST;
		break;

	case 'o':
		flag = OFFLINE;
		break;

	case 'q':
		quiet = 1;
		break;

	case 'r':
		flag = RESET;
		break;

	case 's':
		def_server = optarg;
		break;

	case '?':
	default:
		errflg = 1;
		break;
	}

	if (errflg ||
	    (flag == LIST && optind != argc)) {
		if (!quiet)
			fprintf(stderr,
				"usage:\t%s [-{c|o|r}][-s server] node node ...\n\t%s -{l|a} [-s server]\n",
				argv[0], argv[0]);
		exit(1);
	}

	con= cnt2server(def_server);
	if (con<= 0) {
	    if (!quiet)
		fprintf(stderr, "%s: cannot connect to server %s, error=%d\n",
				argv[0], def_server, pbs_errno);
	    exit(1);
	}

	/* if flag is ALL, DOWN or LIST, get status of all nodes */

	if ((flag == ALL) || (flag == DOWN) || (flag == LIST)) {
	    bstatus = pbs_statnode(con, "", NULL, NULL);
	    if (bstatus == NULL) {
		if ( pbs_errno ) {
		    if (!quiet) {
			if ((errmsg = pbs_geterrmsg(con)) != NULL)
			    fprintf(stderr, "%s: %s\n", argv[0], errmsg);
			else
			    fprintf(stderr, "%s: Error %d\n",argv[0],pbs_errno);
		    }
		    exit(1);
		} else {
		    if (!quiet)
			fprintf(stderr, "%s: No nodes found\n", argv[0]);
		    exit(0);
		}
	    }
	}



	switch (flag) {

	    case DOWN:

		/*
		 * loop through the list of nodes returned above:
		 *   if node is up and is in argv list, mark it down;
		 *   if node is down and not in argv list, mark it up;
		 * for all changed nodes, send in request to server
		 */

		for (pbstat = bstatus; pbstat; pbstat = pbstat->next) {
			for (pa = argv+optind; *pa; pa++) {
			    if (cmp_node_name(*pa, pbstat->name) == 0) {
				if (is_down(pbstat) == 0) {
				    marknode(con, pbstat->name,
					     ND_down, INCR, NULL, INCR);
				    break;
				}
			    }
			}
			if (*pa == NULL) {
			
			    /* node not in list, if down now, set up */
			    if (is_down(pbstat) == 1)
				marknode(con, pbstat->name, 
					 ND_down, DECR, NULL, DECR);
			}
		}
		break;

	    case CLEAR:

		/* clear DOWN and OFF_LINE from specified nodes		*/

		for (pa = argv+optind; *pa; pa++) {
			marknode(con, *pa, ND_offline, DECR, ND_down, DECR);	
		}

		break;

	    case RESET:

		/* clear OFF_LINE from specified nodes			*/

		for (pa = argv+optind; *pa; pa++) {
			marknode(con, *pa, ND_offline, DECR, NULL, DECR);	
		}
		break;

	    case OFFLINE:

		/* set OFF_LINE on specified nodes			*/
		for (pa = argv+optind; *pa; pa++) {
			marknode(con, *pa, ND_offline, SET, NULL, INCR);	
		}
		break;

	    case ALL:
		for (pbstat = bstatus; pbstat; pbstat = pbstat->next) {
			printf("%s\n", pbstat->name);
			prt_node_attr(pbstat);
			putchar('\n');
		}
		break;

	    case LIST:

		/* list any node that is DOWN or OFF_LINE		*/

		for (pbstat = bstatus; pbstat; pbstat = pbstat->next) {
			if (is_down(pbstat) || is_offline(pbstat)) {
				printf("%-20.20s %s\n", pbstat->name,
						    get_nstate(pbstat));
			}
		}

		break;

	}
	return 0;
}
