/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* 
 *
 * qalter - (PBS) alter batch job
 *
 * Authors:
 *      Terry Heidelberg
 *      Livermore Computing
 *
 *      Bruce Kelly
 *      National Energy Research Supercomputer Center
 *
 *      Lawrence Livermore National Laboratory
 *      University of California
 */

#include "cmds.h"
#include <pbs_config.h>   /* the master config generated by configure */

static char ident[] = "@(#) $RCSfile: qalter.c,v $ $Revision: 2.1 $";

main(argc, argv, envp) /* qalter */
int argc;
char **argv;
char **envp;
{
    int c;
    int errflg=0;
    int any_failed=0;
    char *pc;
    int i, colon, count;
    int u_cnt, o_cnt, s_cnt, n_cnt;
    struct timeval *tp;
    struct timezone *tzp;
    struct attrl *attrib = NULL;
    char *keyword;
    char *pdepend;
    char *valuewd;
    time_t after;
    char a_value[80];

    char job_id[PBS_MAXCLTJOBID];

    char job_id_out[PBS_MAXCLTJOBID];
    char server_out[MAXSERVERNAME];
    char rmt_server[MAXSERVERNAME];
    char path_out[MAXPATHLEN+1];

#define GETOPT_ARGS "a:A:c:e:h:j:k:l:m:M:N:o:p:r:S:u:W:"
                 
    while ((c = getopt (argc, argv, GETOPT_ARGS )) != EOF)
        switch (c) {
        case 'a':
            if ( (after = cvtdate(optarg)) < 0 ) {
                fprintf(stderr, "qalter: illegal -a value\n");
                errflg++;
		break;
            }
            sprintf(a_value, "%d", after);
            set_attr(&attrib, ATTR_a, a_value);
            break;
        case 'A':
            set_attr(&attrib, ATTR_A, optarg);
            break;
        case 'c':
	    while ( isspace((int)*optarg) ) optarg++;
	    pc = optarg;
            if ( strlen(pc) == 1 ) {
                if ( *pc != 'n' && *pc != 's' && *pc != 'c' ) {
                    fprintf(stderr, "qalter: illegal -c value\n");
                    errflg++;
		    break;
                }
            } else {
                if ( strncmp(pc, "c=", 2) != 0 ) {
                    fprintf(stderr, "qalter: illegal -c value\n");
                    errflg++;
		    break;
                }
                pc += 2;
                if ( *pc == '\0' ) {
                    fprintf(stderr, "qalter: illegal -c value\n");
                    errflg++;
		    break;
                }
                while ( isdigit(*pc) ) pc++;
                if ( *pc != '\0' ) {
                    fprintf(stderr, "qalter: illegal -c value\n");
                    errflg++;
		    break;
                }
            }
            set_attr(&attrib, ATTR_c, optarg);
            break;
        case 'e':
            if ( prepare_path(optarg, path_out) == 0 ) {
                set_attr(&attrib, ATTR_e, path_out);
            } else {
                fprintf(stderr, "qalter: illegal -e value\n");
                errflg++;
            }
	    break;
        case 'h':
	    while ( isspace((int)*optarg) ) optarg++;
            if ( strlen(optarg) == 0 ) {
                fprintf(stderr, "qalter: illegal -h value\n");
                errflg++;
		break;
            }
	    pc = optarg;
	    u_cnt = o_cnt = s_cnt = n_cnt = 0;
            while ( *pc) {
	        if ( *pc == 'u' )
		    u_cnt++;
                else if ( *pc == 'o' )
		    o_cnt++;
                else if ( *pc == 's' )
		    s_cnt++;
                else if ( *pc == 'n' )
		    n_cnt++;
		else {
		    fprintf(stderr, "qalter: illegal -h value\n");
		    errflg++;
		    break;
		}
                pc++;
            }
            if ( n_cnt && (u_cnt + o_cnt + s_cnt) ) {
                fprintf(stderr, "qalter: illegal -h value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_h, optarg);
            break;
        case 'j':
	    if ( strcmp(optarg, "oe") != 0 &&
		 strcmp(optarg, "eo") != 0 &&
		 strcmp(optarg,  "n") != 0 ) {
		fprintf(stderr, "qalter: illegal -j value\n");
		errflg++;
		break;
	    }
            set_attr(&attrib, ATTR_j, optarg);
            break;
        case 'k':
            if ( strcmp(optarg,  "o") != 0 &&
                 strcmp(optarg,  "e") != 0 &&
                 strcmp(optarg, "oe") != 0 &&
                 strcmp(optarg, "eo") != 0 &&
                 strcmp(optarg,  "n") != 0 ) {
                fprintf(stderr, "qalter: illegal -k value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_k, optarg);
            break;
        case 'l':
            if ( set_resources(&attrib, optarg, TRUE) ) {
		fprintf(stderr, "qalter: illegal -l value\n");
		errflg++;
	    }
            break;
        case 'm':
	    while ( isspace((int)*optarg) ) optarg++;
            if ( strlen(optarg) == 0 ) {
                fprintf(stderr, "qalter: illegal -m value\n");
                errflg++;
		break;
            }
            if ( strcmp(optarg, "n") != 0 ) {
                pc = optarg;
                while (*pc) {
                    if ( *pc != 'a' && *pc != 'b' && *pc != 'e' ) {
                        fprintf(stderr, "qalter: illegal -m value\n");
                        errflg++;
			break;
                    }
                    pc++;
                }
            }
            set_attr(&attrib, ATTR_m, optarg);
            break;
        case 'M':
            if ( parse_at_list(optarg, FALSE, FALSE) ) {
                fprintf(stderr, "qalter: illegal -M value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_M, optarg);
            break;
        case 'N':
	    if (check_job_name(optarg, 1) == 0) {
		set_attr(&attrib, ATTR_N, optarg);
	    } else {
		fprintf(stderr, "qalter: illegal -N value\n");
		errflg++;
	    }
            break;
        case 'o':
            if ( prepare_path(optarg, path_out) == 0 ) {
                set_attr(&attrib, ATTR_o, path_out);
            } else {
                fprintf(stderr, "qalter: illegal -o value\n");
                errflg++;
            }
	    break;
        case 'p':
	    while ( isspace((int)*optarg) ) optarg++;
            pc = optarg;
            if ( *pc == '-' || *pc == '+' ) pc++;
            if ( strlen(pc) == 0 ) {
                fprintf(stderr, "qalter: illegal -p value\n");
                errflg++;
		break;
            }
            while ( *pc != '\0' ) {
                if ( ! isdigit(*pc) ) {
                    fprintf(stderr, "qalter: illegal -p value\n");
                    errflg++;
		    break;
                }
                pc++;
            }
            i = atoi(optarg);
            if ( i < -1024 || i > 1023 ) {
                fprintf(stderr, "qalter: illegal -p value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_p, optarg);
            break;
        case 'r':
            if ( strlen(optarg) != 1 ) {
                fprintf(stderr, "qalter: illegal -r value\n");
                errflg++;
		break;
            }
            if ( *optarg != 'y' && *optarg != 'n' ) {
                fprintf(stderr, "qalter: illegal -r value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_r, optarg);
            break;
        case 'S':
            if ( parse_at_list(optarg, TRUE, TRUE) ) {
                fprintf(stderr, "qalter: illegal -S value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_S, optarg);
            break;
        case 'u':
            if ( parse_at_list(optarg, TRUE, FALSE) ) {
                fprintf(stderr, "qalter: illegal -u value\n");
                errflg++;
		break;
            }
            set_attr(&attrib, ATTR_u, optarg);
            break;
	case 'W':
            while ( isspace((int)*optarg) ) optarg++;
            if ( strlen(optarg) == 0 ) {
                fprintf(stderr, "qalter: illegal -W value\n");
                errflg++;
		break;
            }
	    i = parse_equal_string(optarg, &keyword, &valuewd);
	    while (i == 1) {
		if (strcmp(keyword, ATTR_depend) == 0) {
		    pdepend = malloc(PBS_DEPEND_LEN);
                    if ( parse_depend_list(valuewd, pdepend, PBS_DEPEND_LEN) ) {
                        fprintf(stderr, "qalter: illegal -W value\n");
                        errflg++;
			break;
                    }
		    valuewd = pdepend;
		} else if (strcmp(keyword, ATTR_stagein) == 0) {
                    if ( parse_stage_list(valuewd) ) {
                        fprintf(stderr, "qalter: illegal -W value\n");
                        errflg++;
			break;
                    }
		} else if (strcmp(keyword, ATTR_stageout) == 0) {
                    if ( parse_stage_list(valuewd) ) {
                        fprintf(stderr, "qalter: illegal -W value\n");
                        errflg++;
			break;
                    }
		} else if (strcmp(keyword, ATTR_g) == 0) {
                    if ( parse_at_list(valuewd, TRUE, FALSE) ) {
                        fprintf(stderr, "qalter: illegal -W value\n");
                        errflg++;
			break;
                    }
		}
		set_attr(&attrib, keyword, valuewd);
		i = parse_equal_string((char *)0, &keyword, &valuewd);
	    }
	    if (i == -1) {
                fprintf(stderr, "qalter: illegal -W value\n");
                errflg++;
	    }
	    break;
        case '?':
        default :
            errflg++;
            break;
        }

    if (errflg || optind == argc) {
        static char usage[]="usage: qalter \
[-a date_time] [-A account_string] [-c interval] [-e path] \n\
[-h hold_list] [-j y|n] [-k keep] [-l resource_list] [-m mail_options] \n\
[-M user_list] [-N jobname] [-o path] [-p priority] [-r y|n] [-S path] \n\
[-u user_list] [-W dependency_list] job_identifier...\n";
        fprintf(stderr, usage);
        exit (2);
    }

    for ( ; optind < argc; optind++) {
        int connect;
        int stat=0;
        char *errmsg;
	int located = FALSE;

        strcpy(job_id, argv[optind]);
        if ( get_server(job_id, job_id_out, server_out) ) {
            fprintf(stderr, "qalter: illegally formed job identifier: %s\n", job_id);
            any_failed = 1;
            continue;
        }
cnt:
        connect = cnt2server(server_out);
        if ( connect <= 0 ) {
            fprintf(stderr, "qalter: cannot connect to server %s (errno=%d)\n",
                    pbs_server, pbs_errno);
            any_failed = pbs_errno;
            continue;
        }

        stat = pbs_alterjob(connect, job_id_out, attrib, NULL);
        if ( stat && (pbs_errno != PBSE_UNKJOBID) ) {
	    prt_job_err("qalter", connect, job_id_out);
            any_failed = pbs_errno;
        } else if ( stat && (pbs_errno == PBSE_UNKJOBID) && !located ) {
	    located = TRUE;
	    if ( locate_job(job_id_out, server_out, rmt_server) ) {
	        pbs_disconnect(connect);
		strcpy(server_out, rmt_server);
		goto cnt;
	    }
	    prt_job_err("qalter", connect, job_id_out);
	    any_failed = pbs_errno;
	}

        pbs_disconnect(connect);
    }
    exit(any_failed);
}
