/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * qenable
 *  The qenable command directs that a destination should no longer accept
 *  batch jobs.
 *
 * Synopsis:
 *  qenable destination ...
 *
 * Arguments:
 *  destination ...
 *      A list of destinations.  A destination has one of the following
 *      three forms:
 *          queue
 *          @server
 *          queue@server
 *      If queue is specified, the request is to enable the queue at
 *      the default server.  If @server is given, the request is to
 *      enable the default queue at the server.  If queue@server is
 *      used, the request is to enable the named queue at the named
 *      server.
 *
 * Written by:
 *  Bruce Kelly
 *  National Energy Research Supercomputer Center
 *  Livermore, CA
 *  May, 1993
 */

#include "cmds.h"
#include <pbs_config.h>   /* the master config generated by configure */

static char ident[] = "@(#) $RCSfile: qenable.c,v $ $Revision: 2.1 $";

int exitstatus = 0; /* Exit Status */
static void execute ();


main ( argc, argv )
int argc;
char **argv;
{
/*
 *  This routine sends a Manage request to the batch server specified by
 * the destination.  The ENABLED queue attribute is set to {True}.  If the
 * batch request is accepted, the server will accept Queue Job requests for
 * the specified queue.
 */

    int dest;		/* Index into the destination array (argv) */
    char *queue;	/* Queue name part of destination */
    char *server;	/* Server name part of destination */
    
    if ( argc == 1 ) {
        fprintf(stderr, "Usage: qenable [queue][@server] ...\n");
        exit(1);
    }
    
    for ( dest=1; dest<argc; dest++ )
        if ( parse_destination_id(argv[dest], &queue, &server) == 0 )
		execute(queue, server);
        else {
            fprintf(stderr, "qenable: illegally formed destination: %s\n",
                    argv[dest]);
            exitstatus = 1;
        }
    exit (exitstatus);
}


/*
 * void execute( char *queue, char *server )
 *
 * queue    The name of the queue to disable.
 * server   The name of the server that manages the queue.
 *
 * Returns:
 *  None
 *
 * File Variables:
 *  exitstatus  Set to two if an error occurs.
 */
static void
execute( queue, server )
char *queue;
char *server;
{
    int ct;         /* Connection to the server */
    int merr;       /* Error return from pbs_manager */
    char *errmsg;   /* Error message from pbs_manager */
                    /* The disable request */
    static struct attropl attr = {NULL, "enabled", NULL, "TRUE", SET};
    
    if ( (ct = cnt2server(server)) > 0 ) {
        merr = pbs_manager(ct,MGR_CMD_SET,MGR_OBJ_QUEUE,queue,&attr,NULL);
        if ( merr != 0 ) {
            errmsg = pbs_geterrmsg(ct);
            if ( errmsg != NULL ) {
                fprintf(stderr, "qenable: %s ", errmsg);
            } else {
                fprintf(stderr, "qenable: Error (%d) enabling queue ", pbs_errno);
            }
            if ( notNULL(queue) )
                fprintf(stderr, "%s", queue);
            if ( notNULL(server) )
                fprintf(stderr, "@%s", server);
            fprintf(stderr, "\n");
            exitstatus = 2;
        }
        pbs_disconnect(ct);
    } else {
        fprintf(stderr, "qenable: could not connect to server %s (%d)\n", server, pbs_errno);
        exitstatus = 2;
    }
}
