/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * qrun
 *  The qrun command forces a batch job to run.
 *
 * Synopsis:
 *  qrun [-H host] job_identifier ...
 *
 * Arguments:
 *  host
 *      The host to run the job at.
 *  job_identifier ...
 *      A list of job_identifiers.  A job_identifier has the following form:
 *          sequence_number[.server_name][@server]
 *
 * Written by:
 *  Bruce Kelly
 *  National Energy Research Supercomputer Center
 *  Livermore, CA
 *  May, 1993
 */

#include "cmds.h"
#include <pbs_config.h>   /* the master config generated by configure */

static char ident[] = "@(#) $RCSfile: qrun.c,v $ $Revision: 2.1 $";

int exitstatus = 0; /* Exit Status */
static void execute ();



main ( argc, argv )
int argc;
char **argv;
{
/*
 *  This routine sends a Run Job request to the batch server.  If the
 * batch request is accepted, the server will have started the execution
 * of the job.
 */

    int id;                         /* Index into the job_id array (argv) */
    char job[PBS_MAXCLTJOBID];      /* Job Id */
    char server[MAXSERVERNAME];	    /* Server name */
    char *location = NULL;          /* Where to run the job */
    
    static char opts[] = "H:";     /* See man getopt */
    static char *usage = "Usage: qrun [-H host] job_id ...\n";
    int s;
    int errflg = 0;
    
    /* Command line options */
    while ((s = getopt(argc, argv, opts)) != EOF)
        switch (s) {
        case 'H':
	    if ( strlen(optarg) == 0 ) {
		fprintf(stderr, "qrun: illegal -H value\n");
		errflg++;
		break;
	    }
            location = optarg;
            break;
        case '?':
        default:
            errflg++;
            break;
        }

    if ( errflg || (optind >= argc) ) {
        fprintf(stderr, usage);
        exit(1);
    }
    
    for ( ; optind < argc; optind++ ) {
        if ( get_server(argv[optind], job, server) ) {
            fprintf(stderr, "qrun: illegally formed job identifier: %s\n", argv[optind]);
	    exitstatus = 1;
	    continue;
	}
        execute(job, server, location);
    }
    exit (exitstatus);
}


/*
 * void execute( char *job, char *server, char *location )
 *
 * job      The fully qualified job id.
 * server   The name of the server that manages the job.
 * location The name of the server to run the job.
 *
 * Returns:
 *  None
 *
 * File Variables:
 *  exitstatus  Set to two if an error occurs.
 */
static void
execute( job, server, location )
char *job;
char *server;
char *location;
{
    int ct;         /* Connection to the server */
    int err;        /* Error return from pbs_run */
    char *errmsg;   /* Error message from pbs_run */
    int located = FALSE;
    char rmt_server[MAXSERVERNAME];

cnt:    
    if ( (ct = cnt2server(server)) > 0 ) {
        err = pbs_runjob(ct, job, location, NULL);
        if ( err && (pbs_errno != PBSE_UNKJOBID) ) {
	    prt_job_err("qrun", ct, job);
            exitstatus = 2;
        } else if ( err && (pbs_errno == PBSE_UNKJOBID) && !located ) {
	    located = TRUE;
	    if ( locate_job(job, server, rmt_server) ) {
	        pbs_disconnect(ct);
		strcpy(server, rmt_server);
		goto cnt;
	    }
	    prt_job_err("qrun", ct, job);
            exitstatus = 2;
	}
        pbs_disconnect(ct);
    } else {
        fprintf(stderr, "qrun: could not connect to server %s (%d)\n", server, pbs_errno);
        exitstatus = 2;
    }
}
