/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* 
 *
 * qsub - (PBS) submit batch job
 *
 * Authors:
 *      Terry Heidelberg
 *      Livermore Computing
 *
 *      Bruce Kelly
 *      National Energy Research Supercomputer Center
 *
 *      Lawrence Livermore National Laboratory
 *      University of California
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <termios.h>

#ifdef sun
#include <sys/stream.h>
#endif /* sun */

#if defined(HAVE_SYS_IOCTL_H)
#include <sys/ioctl.h>
#endif	/* HAVE_SYS_IOCTL_H */

#if !defined(sgi) && !defined(_AIX) && !defined(linux)
#include <sys/tty.h>
#endif

#if defined(FD_SET_IN_SYS_SELECT_H)
#  include <sys/select.h>
#endif


#include "cmds.h"

static char ident[] = "@(#) $RCSfile: qsub.c,v $ $Revision: 2.1 $";

#define MAX_QSUB_PREFIX_LEN 32

static char PBS_DPREFIX_DEFAULT[] = "#PBS";

char *
set_dir_prefix(prefix, diropt)
char *prefix;
int diropt;
{
    char *s;

    if ( notNULL(prefix) )
        return(prefix);
    else if ( diropt == TRUE )
	return("");
    else if ( ( s = getenv("PBS_DPREFIX") ) != NULL )
        return(s);
    else
        return(PBS_DPREFIX_DEFAULT);
}

int
isexecutable(s)
char *s;
{
    char *c;

    c = s;
    if ((*c == ':') || ((*c == '#') && (*(c+1) == '!'))) return FALSE;
    while ( isspace(*c) ) c++;
    if ( notNULL(c) ) return ( *c != '#' );
    return FALSE;
}

char *
ispbsdir(s, prefix)
char *s;
char *prefix;
{
    char *it;
    int l;

    it = s;
    while (isspace(*it)) it++;
    l = strlen(prefix);
    if ( l > 0 && strncmp(it, prefix, l) == 0 ) 
        return(it+l);
    else
        return((char *)NULL);
}

int
get_script(file, script, prefix)
FILE *file;
char *script;
char *prefix;
{
    char s[MAX_LINE_LEN+1];
    char *sopt;
    int exec = FALSE;
    char *cont;
    char tmp_name[L_tmpnam+1];
    FILE *TMP_FILE;
    char *in;
    
    tmpnam(tmp_name);
    if ( (TMP_FILE = fopen(tmp_name, "w+")) == NULL ) {
	fprintf(stderr, "qsub: could not create copy of script %s\n", tmp_name);
        return(4);
    }

    while ( (in = fgets(s, MAX_LINE_LEN, file)) != NULL ) {
        if ( !exec && ((sopt = ispbsdir(s, prefix)) != NULL)) {
	    while ( (*(cont = in + strlen(in) - 2) == '\\') &&
		    (*(cont+1) == '\n') ) { 
		/* next line is continuation of this line */
		*cont = '\0';	/* clear newline from our copy */
		if ( fputs(in, TMP_FILE) < 0 ) {
		    fprintf(stderr, "qsub: error writing copy of script, %s\n", tmp_name);
                    fclose(TMP_FILE);
                    return (3);
		}
		in = cont;
		if ((in = fgets(in, MAX_LINE_LEN-(in - s), file)) == NULL) {
		    fprintf(stderr, "qsub: unexpected end-of-file or read error in script\n");
		    fclose(TMP_FILE);
		    return (6);
		}
	    }
	    if (do_dir(sopt) != 0)
		return (-1);
        } else if ( !exec && isexecutable(s) ) {
            exec = TRUE;
        }
	if ( fputs(in, TMP_FILE) < 0 ) {
	    fprintf(stderr, "qsub: error writing copy of script, %s\n", tmp_name);
            fclose(TMP_FILE);
            return (3);
	}
    }
    fclose(TMP_FILE);
    if ( ferror(file) ) {
	fprintf(stderr, "qsub: error reading script file\n");
        return(5);
    } else {
        strcpy(script, tmp_name);
    }
    return(0);
}

void
make_argv(argc, argv, line)
int *argc;
char *argv[];
char *line;
{
    char *l, *b, *c;
    char buffer[4096];
    int len;
    char quote;

    *argc = 0;
    argv[(*argc)++] = "qsub";
    l = line;
    b = buffer;
    while ( isspace(*l) ) l++;
    c = l;
    while ( *c != '\0' ) {
	if ((*c == '"') || (*c == '\'')) {
	    quote = *c;
	    c++;
	    while ((*c != quote) && *c)
	    	*b++ = *c++;
	    if ( *c == '\0' ) {
	    	fprintf(stderr, "qsub: unmatched %c\n", *c);
	    	exit(1);
	    }
	    c++;
	} else if ( *c == '\\' ) {
	    c++;
	    *b++ = *c++;
	} else if ( isspace(*c) ) {
            len = c - l;
            if ( argv[*argc] != NULL ) free(argv[*argc]);
            argv[*argc] = (char *) malloc(len + 1);
            if ( argv[*argc] == NULL ) {
                fprintf(stderr, "qsub: out of memory\n");
                exit(2);
            }
	    *b = '\0';
            strcpy(argv[(*argc)++], buffer);
            while ( isspace(*c) ) c++;
            l = c;
	    b = buffer;
	} else
	    *b++ = *c++;
    }
    if ( c != l ) {
        len = c - l;
        if ( argv[*argc] != NULL ) free(argv[*argc]);
        argv[*argc] = (char *) malloc(len + 1);
        if ( argv[*argc] == NULL ) {
            fprintf(stderr, "qsub: out of memory\n");
            exit(2);
        }
        *b = '\0';
        strcpy(argv[(*argc)++], buffer);
    }
}

int
do_dir(opts)
char *opts;
{
    static int opt_pass = 1;
    int argc;
#define MAX_ARGV_LEN 128
    static char *vect[MAX_ARGV_LEN+1];

    if ( opt_pass == 1 ) {
        argc = 0;
        while ( argc < MAX_ARGV_LEN+1 ) vect[argc++] = NULL;
    }
    make_argv(&argc, vect, opts);
    return process_opts(argc, vect, opt_pass++);
}



/* globals */

int inter_sock;
struct termios oldtio;
struct winsize wsz;

struct attrl *attrib = NULL;
char *new_jobname;                  /* return from submit request */
char dir_prefix[MAX_QSUB_PREFIX_LEN+1];
char path_out[MAXPATHLEN+1];
char destination[PBS_MAXDEST];
static char server_out[PBS_MAXSERVERNAME+PBS_MAXPORTNUM+2];

/* state booleans for protecting already-set options */
int a_opt = FALSE;
int c_opt = FALSE;
int e_opt = FALSE;
int h_opt = FALSE;
int j_opt = FALSE;
int k_opt = FALSE;
int l_opt = FALSE;
int m_opt = FALSE;
int o_opt = FALSE;
int p_opt = FALSE;
int q_opt = FALSE;
int r_opt = FALSE;
int u_opt = FALSE;
int v_opt = FALSE;
int z_opt = FALSE;
int A_opt = FALSE;
int C_opt = FALSE;
int M_opt = FALSE;
int N_opt = FALSE;
int S_opt = FALSE;
int V_opt = FALSE;
int Depend_opt    = FALSE;
int Interact_opt  = FALSE;
int Stagein_opt   = FALSE;
int Stageout_opt  = FALSE;
int Grouplist_opt = FALSE;
char *v_value = NULL;


char *copy_env_value(dest, pv, quote_flg)
	char *dest;	/* destination  */
	char *pv;	/* value string */
	int   quote_flg; /* non-zero then assume single word (quoting on) */
{
	int   go = 1;
	int   q_ch = 0;

	while (*dest)
		++dest;

	while (go && *pv) {
		switch (*pv) {
		    case '"':
		    case '\'':
			if (q_ch) {	/* local quoting is in progress */
				if (q_ch == (int)*pv) {
					q_ch = 0;	/* end quote */
				} else {
					*dest++ = '\\';	/* escape quote */
					*dest++ = *pv;
				}
			} else if (quote_flg) {	  /* global quoting is on */
				*dest++ = '\\';	  /* escape quote */
				*dest++ = *pv;
			} else {
				q_ch = (int)*pv;  /* turn local quoting on */
			}
			break;
				
		    case '\\':
			*dest++ = '\\';		/* escape back-slash */
			*dest++ = *pv;
			break;

		    case ',':
			if (q_ch || quote_flg) {
				*dest++ = '\\';
				*dest++ = *pv;
			} else {
				go = 0;		/* end of value string */
			}
			break;

		    default:
			*dest++ = *pv;
			break;
		}
		pv++;
	}
	*dest = '\0';
	if (q_ch)
		return ((char *)0);	/* error-unterminated quote */
	else
		return (pv);
}

int
set_job_env(envp)
char **envp;
{
    char **evp;
    char *job_env;
    char *s, *c, *env, l;
    char host[PBS_MAXHOSTNAME+1];
    struct in_addr  ina;
    int len;
    int rc;
    char *getcwd();

    /* Calculate how big to make the variable string. */
    len = 0;
    if ( v_opt ) {
        len += strlen(v_value);
    }
    if ( V_opt ) {
        evp = envp;
        while ( notNULL(*evp) ) {
            len += strlen(*evp);
            evp++;
        }
    }
    env = getenv("HOME");
    if ( env != NULL ) len += strlen(env);
    env = getenv("LANG");
    if ( env != NULL ) len += strlen(env);
    env = getenv("LOGNAME");
    if ( env != NULL ) len += strlen(env);
    env = getenv("PATH");
    if ( env != NULL ) len += strlen(env);
    env = getenv("MAIL");
    if ( env != NULL ) len += strlen(env);
    env = getenv("SHELL");
    if ( env != NULL ) len += strlen(env);
    env = getenv("TZ");
    if ( env != NULL ) len += strlen(env);
    len += PBS_MAXHOSTNAME;
    len += MAXPATHLEN;
    len += len;     /* Double it for all the commas, etc. */
    if ( (job_env = (char *) malloc(len)) == NULL ) return FALSE;
    *job_env = '\0';
    
    /* Send the required variables with the job. */
    c = getenv("HOME");
    strcat(job_env, "PBS_O_HOME=");
    if ( c != NULL )
        strcat(job_env, c);
    else
        strcat(job_env, "/");
    c = getenv("LANG");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_LANG=");
        strcat(job_env, c);
    }
    c = getenv("LOGNAME");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_LOGNAME=");
        strcat(job_env, c);
    }
    c = getenv("PATH");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_PATH=");
        strcat(job_env, c);
    }
    c = getenv("MAIL");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_MAIL=");
        strcat(job_env, c);
    }
    c = getenv("SHELL");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_SHELL=");
        strcat(job_env, c);
    }
    c = getenv("TZ");
    if ( c != NULL ) {
        strcat(job_env, ",PBS_O_TZ=");
        strcat(job_env, c);
    }
    if ((rc = gethostname(host, PBS_MAXHOSTNAME+1)) == 0 ) {
        if ( (rc = get_fullhostname(host, host, PBS_MAXHOSTNAME)) == 0 ) {
                strcat(job_env, ",PBS_O_HOST=");
                strcat(job_env, host);
        }
    }
    if (rc != 0) {
	fprintf(stderr, "qsub: cannot get full local host name\n");
	exit(3);
    }
    /* get current working directory, use $PWD if available, it is more	*/
    /* NFS automounter "friendly".  But must double check that is right */
    s = job_env + strlen(job_env);
    strcat(job_env, ",PBS_O_WORKDIR=");
    c = getenv("PWD");
    if ( c != NULL ) {
	struct stat statbuf;
	dev_t	dev;
	ino_t	ino;

	if (stat(c, &statbuf) < 0) {
		c = NULL;	/* cannot stat, cannot trust it */
	} else {
		dev = statbuf.st_dev;
		ino = statbuf.st_ino;
		if (stat(".", &statbuf) < 0) {	/* compare against "." */
		    perror("qsub: cannot stat current directory: ");
		    exit(3);
		}
		if ((dev == statbuf.st_dev) && (ino == statbuf.st_ino)) {
		    strcat(job_env, c);
		} else {
		   c = NULL;
		}
	}
    }
    if (c == NULL) {	/* fall back to using the cwd */
	c = job_env + strlen(job_env);
	if ( getcwd(c, MAXPATHLEN+1) == NULL) *s = '\0';
    }
	
    /* Send these variables with the job. */
    /* POSIX requirement: If a variable is given without a value, supply the
                          value from the environment. */
    /* MY requirement:    There can be no white space in -v value. */
    if ( v_opt ) {
        c = v_value;
state1:         /* Initial state comes here */
        switch (*c) {
        case ',':
        case '=':
            return FALSE;
        case '\0':
            goto final;
        }
        s = c;
state2:         /* Variable name */
        switch (*c) {
        case ',':
        case '\0':
            goto state3;
        case '=':
            goto state4;
        default:
            c++;
            goto state2;
        }
state3:         /* No value - get it from qsub environment */
        l = *c;
        *c = '\0';
        env = getenv(s);
        if ( env == NULL ) return FALSE;
        strcat(job_env, ",");
        strcat(job_env, s);
        strcat(job_env, "=");
	if (copy_env_value(job_env, env, 1) == (char *)0) return FALSE;
        if ( l == ',' ) c++;
        goto state1;
state4:         /* Value specified */
        *c++ = '\0';;
	(void)strcat(job_env, ",");
	(void)strcat(job_env, s);
	(void)strcat(job_env, "=");
	if ((c = copy_env_value(job_env, c, 0)) == (char *)0) return FALSE;
	goto state1;
    }

final:

    if ( V_opt ) {      /* Send every environment variable with the job. */
        evp = envp;
        while ( notNULL(*evp) ) {
	    s = *evp;
	    while ((*s != '=') && *s)
		++s;
	    *s = '\0';
            strcat(job_env, ",");
            strcat(job_env, *evp);
	    strcat(job_env, "=");
	    (void)copy_env_value(job_env, s+1, 1);
            evp++;
        }
    }

    set_attr(&attrib, ATTR_v, job_env);
    free(job_env);

    return TRUE;
}


/*
 * The following bunch of functions support the "Interactive Job"
 * capability of PBS.   
 */

/*
 * interactive_port - get a socket to listen to for "interactive" job
 *	When the "interactive" job is run, its standard in, out, and error
 *	will be connected to this socket.
 */

char *
interactive_port()
{
    int  namelen;
    static char portstring[8];
    struct sockaddr_in myaddr;
    unsigned short port;

    if ((isatty(0) == 0) || (isatty(1) == 0)) {
	fprintf(stderr, "qsub:\tstandard input and output must be a terminal for \n\tinteractive job submission\n");
	exit(1);
    }
    inter_sock = socket(AF_INET, SOCK_STREAM, 0);
    if (inter_sock < 0) {
	perror("qsub: unable to obtain socket");
	exit(1);
    }
    myaddr.sin_family = AF_INET;
    myaddr.sin_addr.s_addr = INADDR_ANY;
    myaddr.sin_port = 0;
    if (bind(inter_sock, (struct sockaddr *)&myaddr, sizeof(myaddr)) < 0) {
	perror("qsub: unable to bind to socket");
	exit(1);
    }

    /* get port number assigned */

    namelen = sizeof(myaddr);
    if (getsockname(inter_sock, (struct sockaddr *)&myaddr, &namelen) < 0) {
	perror("qsub: unable to get port number");
	exit(1);
    }
    port = ntohs(myaddr.sin_port);
    (void)sprintf(portstring, "%u", (unsigned int)port);
    if (listen(inter_sock, 1) < 0) {
	perror("qsub: listen on interactive socket");
	exit(1);
    }
    
    return (portstring);
}

/*
 * settermraw - set terminal into "raw" mode
 */

void
settermraw(ptio)
    struct termios *ptio;
{
    struct termios tio;

    tio = *ptio;

    tio.c_lflag &= ~(ICANON|ISIG|ECHO|ECHOE|ECHOK);
    tio.c_iflag &= ~(IGNBRK|INLCR|ICRNL|IXON|IXOFF);
    tio.c_oflag = 0;
    tio.c_oflag |= (OPOST); /* TAB3 */
    tio.c_cc[VMIN] = 1;
    tio.c_cc[VTIME] = 0;

#if defined(TABDLY) && defined(TAB3)
    if ((tio.c_oflag & TABDLY) == TAB3)
	tio.c_oflag &= ~TABDLY;
#endif
    tio.c_cc[VKILL]  = -1;
    tio.c_cc[VERASE] = -1;

    if (tcsetattr(0, TCSANOW, &tio) < 0)
	perror("qsub: set terminal mode");
}

/*
 * stopme - suspend process on ~^Z or ~^Y
 *	on suspend, reset terminal to normal "cooked" mode;
 *	when resumed, again set terminal to raw.
 */

void
stopme(p)
    pid_t p;	/* pid of 0 (process group) or just myself (writer) */
{
    (void)tcsetattr(0, TCSANOW, &oldtio); /* reset terminal */
    kill(p, SIGTSTP);
    (void)settermraw(&oldtio);            /* back to raw when we resume */
}

/*
 * Interactive Reader process: reads from the remote socket,
 *      and writes that out to the stdout
 */

int
reader(s)
    int s;	/* socket */
{
    char buf[4096];
    int  c;
    char *p;
    int  wc;

    /* read from the socket, and write to stdout */
    while (1) {
	c = read(s, buf, sizeof(buf));
	if (c > 0) {
	    p = buf;
	    while (c) {
		if ((wc = write(1, p, c)) < 0) {
		    if (errno == EINTR) {
			continue;
		    } else {
			perror("qsub: write error");
			return (-1);
		    }
		}
		c -= wc;
		p += wc;
	    }
	} else if (c == 0) {
	    return (0);		/* EOF - all done */
	} else {
	    if (errno == EINTR)
		continue;
	    else {
		perror("qsub: read error");
		return (-1);
	    }
	}
    }
}
	
		    
/*
 * Writer process: reads from stdin, and writes
 * data out to the rem socket
 */

void
writer(s)
    int s;	/* socket */
{
    char c;
    int i;
    int newline = 1;
    int process = 0;
    char tilda = '~';
    int wi;

    /* read from stdin, and write to the socket */

    while (1) {
	i = read(0, &c, 1);
	if (i > 0) {		/* read data */
	    if (newline) {
		if (c == tilda) {	/* maybe escape character */

		    /* read next character to check */

		    while ((i = read(0, &c, 1)) != 1) {
			if ((i == -1) && (errno == EINTR))
			    continue;
			else
			    break;
		    }
		    if (i != 1)
			break;
		    if (c == '.')	/* termination character */
			break;
 		    else if (c == oldtio.c_cc[VSUSP]) {
			stopme(0);	/* ^Z suspend all */
			continue;
#ifdef VDSUSP
		    } else if (c == oldtio.c_cc[VDSUSP]) {
			stopme(getpid());
			continue;
#endif	/* VDSUSP */
		    } else {	/* not escape, write out tilda */
			while ((wi = write(s, &tilda, 1)) != 1) {
			    if ((wi == -1) && (errno == EINTR))
				continue;
			    else
				break;
			}
			if (wi != 1)
			    break;
		    }
		}
		newline = 0;   /* no longer at start of line */
	    } else {
		/* reset to newline if \n \r kill or interrupt */
		newline = (c == '\n') ||
			  (c == oldtio.c_cc[VKILL]) ||
			  (c == oldtio.c_cc[VINTR]) ||
			  (c == '\r') ;
	    }
	    while ((wi = write(s, &c, 1)) != 1) {   /* write out character */
		if ((wi == -1) && (errno == EINTR))
		    continue;
		else
		    break;
	    }
	    if (wi != 1)
		break;

	} else if (i == 0) {	/* EOF */
	    break;
	} else if (i < 0) {	/* error */
	    if (errno == EINTR)
		continue;
	    else {
		perror("qsub: read error");
		return;
	    }
	}
    }
    return;
}
		
/*
 * getwinsize - get the current window size
 */
int
getwinsize(pwsz)
    struct winsize *pwsz;
{
    if (ioctl(0, TIOCGWINSZ, &wsz) < 0) {
	perror("qsub: unable to get window size");
	return (-1);
    }
    return (0);
}

/*
 * send_winsize = send the current tty's window size
 */

void
send_winsize(sock)
    int sock;
{
    char  buf[PBS_TERM_BUF_SZ];

    (void)sprintf(buf, "WINSIZE %hu,%hu,%hu,%hu", wsz.ws_row, wsz.ws_col,
			wsz.ws_xpixel, wsz.ws_ypixel);
    (void)write(sock, buf, PBS_TERM_BUF_SZ);
    return;
}

/*
 * send_term - send the current TERM type and certain control characters
 */

void
send_term(sock)
    int sock;
{
    char  buf[PBS_TERM_BUF_SZ];
    char *term;
    char  cc_array[PBS_TERM_CCA];

    (void)strcpy(buf, "TERM=");
    term = getenv("TERM");
    if (term == (char *)0)
	(void)strcat(buf, "unknown");
    else
	(void)strncat(buf, term, PBS_TERM_BUF_SZ-5);
    (void)write(sock, buf, PBS_TERM_BUF_SZ);

    cc_array[0] = oldtio.c_cc[VINTR];
    cc_array[1] = oldtio.c_cc[VQUIT];
    cc_array[2] = oldtio.c_cc[VERASE];
    cc_array[3] = oldtio.c_cc[VKILL];
    cc_array[4] = oldtio.c_cc[VEOF];
    cc_array[5] = oldtio.c_cc[VSUSP];
    write(sock, cc_array, PBS_TERM_CCA);
}


/*
 * catchchild = signal handler for Death of Child
 */    
 
void
catchchild(sig)
    int sig;
{
    int status;
    int pid;

    while (1) {
    	pid = waitpid(-1, &status, WNOHANG|WUNTRACED);
    	if (pid == 0)
	    return;
    	if ( (pid > 0) && (WIFSTOPPED(status) == 0) )
	    break;
	if ((pid == -1) && (errno != EINTR)) {
	    perror("qsub: bad status in catchchild: ");
	    return;
        }
    }
   
    /* reset terminal to cooked mode */

    (void)tcsetattr(0, TCSANOW, &oldtio);
    exit(0);
}

void no_suspend(sig)
	int sig;
{
	printf("Sorry, you cannot suspend qsub until the job is started\n");
	fflush(stdout);
}

void
bailout()
{
	int	c;

	(void)shutdown(inter_sock, 2);
	(void)close(inter_sock);
	printf("Job %s is being deleted\n", new_jobname);	
	c = cnt2server(server_out);
	if ( c <= 0 ) {
		fprintf(stderr,
			"qsub: cannot connect to server %s (errno=%d)\n",
	        	pbs_server, pbs_errno);
		exit(1);
	}
	(void)pbs_deljob(c, new_jobname, (char *)0);
	pbs_disconnect(c);
	exit(0);
}

void
toolong(sig)
    int	sig;
{
	printf("Timeout -- deleting job\n");
	bailout();
}

void
catchint(sig)
    int sig;
{
    int c;

    printf("Do you wish to terminate the job and exit (y|[n])? ");
    fflush(stdout);
    while (1) {
	alarm(60);	/* give a minute to think about it */
	c = getchar();

	if ((c == 'n') || (c == 'N') || (c == '\n'))
	    break;
	else if ((c == 'y') || (c == 'Y')) {
	    bailout();
	} else {
	    printf("yes or no please\n");
	    while ((c != '\n') && (c != EOF))
		c = getchar();
	}
    }
    alarm(0);		/* reset alarm */
    while ((c != '\n') && (c != EOF))
	c = getchar();
    return;
}

/*
 * interactive - set up for interactive communication with job
 */

void
interactive()
{
    int  amt;
    char cur_server[PBS_MAXSERVERNAME+PBS_MAXPORTNUM+2];
    int  fromlen;
    char momjobid[PBS_MAXSVRJOBID+1];
    int  news;
    int  nsel;
    char *pc;
    fd_set selset;
    struct sigaction act;
    struct sockaddr_in from;
    struct timeval timeout;
    struct winsize wsz;
    int child;
    
    /* Catch SIGINT and SIGTERM, and */
    /* setup to catch Death of child */

    sigemptyset(&act.sa_mask);
    act.sa_handler = catchint;
    act.sa_flags   = 0;
    if ( (sigaction(SIGINT, &act, (struct sigaction *)0) < 0) ||
         (sigaction(SIGTERM, &act, (struct sigaction *)0) < 0) ) {
		perror("unable to catch signals");
		exit(1);
    }
    act.sa_handler = toolong;
    if ( (sigaction(SIGALRM, &act, (struct sigaction *)0) < 0) ) {
	perror("cannot catch alarm");
	exit(2);
    }

    /* save the old terminal setting */

    if (tcgetattr(0, &oldtio) < 0) {
	perror("qsub: unable to get terminal settings");
	exit (1);
    }

    /* Get the current window size, to be sent to MOM later */

    if (getwinsize(&wsz)) {
	wsz.ws_row = 20;	/* unable to get actual values	*/
	wsz.ws_col = 80;	/* set defaults			*/
	wsz.ws_xpixel = 0;
	wsz.ws_ypixel = 0;
    }

    printf("qsub: waiting for job %s to start\n", new_jobname);

    /* Accept connection on socket set up earlier */

    nsel = 0;
    while (nsel == 0) {
	FD_ZERO(&selset);
	FD_SET(inter_sock, &selset);
        timeout.tv_usec = 0;
        timeout.tv_sec  = 30;
	nsel = select(FD_SETSIZE, &selset, (fd_set *)0, (fd_set *)0, &timeout);
	if (nsel > 0) {
		break;
	} else if (nsel == -1) {
	    if (errno == EINTR)
		nsel = 0;
	    else {
		perror("qsub: select failed");
		exit(1);
	    }
	}
	/* connect to server, status job to see if still there */

        if ( ! locate_job(new_jobname, server_out, cur_server) ) {
		fprintf(stderr,"qsub: job %s apparently deleted\n",new_jobname);
		exit(1);
	}

    }

    /* apparently someone is attempting to connect to us */

    fromlen = sizeof(from);
    if ((news = accept(inter_sock, (struct sockaddr *)&from, &fromlen)) < 0) {
	perror("qsub: accept error");
	exit (1);
    }

    /* When MOM connects, she will send the job id for us to verify */


    amt = PBS_MAXSVRJOBID+1;
    pc = momjobid;
    while (amt > 0) {
	fromlen = read(news, pc, amt);
	if (fromlen <= 0)
	    break;
	pc += fromlen;
	if (*(pc-1) == '\0')
	    break;
	amt -= fromlen;
    }
    if (strncmp(momjobid, new_jobname, PBS_MAXSVRJOBID) != 0) {
	fprintf(stderr, "qsub: invalid job name from execution server\n");
	shutdown(news, 2);
	exit (1);
    }
    
    /*
     * got the right job, send:
     *		terminal type as "TERM=xxxx"
     *		window size as   "WINSIZE=r,c,x,y"
     */
    send_term(news);
    send_winsize(news);

    printf("qsub: job %s ready\n\n", new_jobname);

    /* set SIGINT, SIGTERM processing to default */

    act.sa_handler = SIG_DFL;
    if ( (sigaction(SIGINT, &act, (struct sigaction *)0) < 0)  ||
         (sigaction(SIGTERM, &act, (struct sigaction *)0) < 0) ||
         (sigaction(SIGALRM, &act, (struct sigaction *)0) < 0) ||
         (sigaction(SIGTSTP, &act, (struct sigaction *)0) < 0) ) {
		perror("unable to reset signals");
		exit(1);
    }

    child = fork();
    if (child == 0) {
	/*
	 * child process - start the reader function
	 *		   set terminal into raw mode
	 */

	settermraw(&oldtio);

	(void)reader(news);

	/* reset terminal */
	tcsetattr(0, TCSANOW, &oldtio);
	printf("\nqsub: job %s completed\n", new_jobname);
	exit (0);

    } else if (child > 0) {
	/*
	 * parent - start the writer function
	 */

    	act.sa_handler = catchchild;
	if (sigaction(SIGCHLD, &act, (struct sigaction *)0) < 0)
	    exit(1);

	writer(news);

	/* all done - make sure reader child is gone and reset terminal */

	kill(child, SIGTERM);
	tcsetattr(0, TCSANOW, &oldtio);
	exit (0);
    } else {
	perror ("qsub: unable to fork");
	exit (1);
    }
}


int
process_opts(argc, argv, pass)
int argc;
char **argv;
int pass;
{
    int i;
    int c;
    int errflg = 0;
    int passet;
    time_t after;
    char a_value[80];
    char *keyword;
    char *valuewd;
    char *pc;
    char *user, *host;
    char *pdepend;

#if !defined(PBS_NO_POSIX_VIOLATION)
#define GETOPT_ARGS "a:A:c:C:e:hIj:k:l:m:M:N:o:p:q:r:S:u:v:VW:z"
#else
#define GETOPT_ARGS "a:A:c:C:e:hj:k:l:m:M:N:o:p:q:r:S:u:v:VW:z"
#endif	/* PBS_NO_POSIX_VIOLATION */

#define MAX_RES_LIST_LEN 64
    char *res_out[MAX_RES_LIST_LEN+1];

/* The following macro, together the value of passet (pass + 1) is used	*/
/* to enforce the following rules: 1. option on the command line take	*/
/* precedence over those in script directives.   2. With in the command	*/
/* line or within the script, the last occurance of an option takes	*/
/* precedence over the earlier occurance.				*/

#define if_cmd_line(x) if( (pass==0)||(x!=1) )

    passet = pass + 1;
    if ( pass > 0 ) {
#ifdef linux
        optind = 0;  /* prime getopt's starting point */
#else
        optind = 1;  /* prime getopt's starting point */
#endif
   }	
    while ((c = getopt(argc, argv, GETOPT_ARGS )) != EOF) {
        switch (c) {
        case 'a':
            if_cmd_line(a_opt) {
                a_opt = passet;
                if ( (after = cvtdate(optarg)) < 0 ) {
                    fprintf(stderr, "qsub: illegal -a value\n");
                    errflg++;
		    break;
                }
		sprintf(a_value, "%d", after);
                set_attr(&attrib, ATTR_a, a_value);
            }
            break;
        case 'A':
            if_cmd_line(A_opt) {
                A_opt = passet;
                set_attr(&attrib, ATTR_A, optarg);
            }
            break;
        case 'c':
            if_cmd_line(c_opt) {
                c_opt = passet;
		while ( isspace((int)*optarg) ) optarg++;
                if ( strlen(optarg) == 0 ) {
                    fprintf(stderr, "qsub: illegal -c value\n");
                    errflg++;
		    break;
                }
		pc = optarg;
		if ( strlen(optarg) == 1 ) {
		    if ( *pc != 'n' && *pc != 's' && *pc != 'c' ) {
		        fprintf(stderr, "qsub: illegal -c value\n");
			errflg++;
			break;
		    }
		} else {
		    if ( strncmp(optarg, "c=", 2) != 0 ) {
                        fprintf(stderr, "qsub: illegal -c value\n");
                        errflg++;
			break;
                    }
		    pc += 2;
		    if ( *pc == '\0' ) {
			fprintf(stderr, "qsub: illegal -c value\n");
			errflg++;
			break;
		    }
		    while ( isdigit(*pc) ) pc++;
		    if ( *pc != '\0' ) {
                        fprintf(stderr, "qsub: illegal -c value\n");
                        errflg++;
			break;
                    }
		}
                set_attr(&attrib, ATTR_c, optarg);
            }
            break;
        case 'C':
            if_cmd_line(C_opt) {
                C_opt = passet;
                strcpy(dir_prefix, optarg);
            }
            break;
        case 'e':
            if_cmd_line(e_opt) {
                e_opt = passet;
                if ( prepare_path(optarg, path_out) == 0 ) {
                    set_attr(&attrib, ATTR_e, path_out);
                } else {
                    fprintf(stderr, "qsub: illegal -e value\n");
                    errflg++;
                }
            }
            break;
        case 'h':
            if_cmd_line(h_opt) {
                h_opt = passet;
                set_attr(&attrib, ATTR_h, "u");
            }
            break;
#if !defined(PBS_NO_POSIX_VIOLATION)
	case 'I':
	    if_cmd_line(Interact_opt) {
		Interact_opt = passet;
		set_attr(&attrib, ATTR_inter, interactive_port());
	    }
	    break;
#endif	/* PBS_NO_POSIX_VIOLATION */
        case 'j':
            if_cmd_line(j_opt) {
                j_opt = passet;
		if ( strcmp(optarg, "oe") != 0 &&
		     strcmp(optarg, "eo") != 0 &&
		     strcmp(optarg,  "n") != 0 ) {
		    fprintf(stderr, "qsub: illegal -j value\n");
		    errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_j, optarg);
            }
            break;
        case 'k':
            if_cmd_line(k_opt) {
                k_opt = passet;
		if ( strcmp(optarg,  "o") != 0 &&
		     strcmp(optarg,  "e") != 0 &&
		     strcmp(optarg, "oe") != 0 &&
		     strcmp(optarg, "eo") != 0 &&
		     strcmp(optarg,  "n") != 0 ) {
		    fprintf(stderr, "qsub: illegal -k value\n");
		    errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_k, optarg);
            }
            break;
        case 'l':
            l_opt = passet;
            if ( set_resources(&attrib, optarg, (pass==0)) ) {
		fprintf(stderr, "qsub: illegal -l value\n");
		errflg++;
	    }
            break;
        case 'm':
            if_cmd_line(m_opt) {
                m_opt = passet;
		while ( isspace((int)*optarg) ) optarg++;
		if ( strlen(optarg) == 0 ) {
		    fprintf(stderr, "qsub: illegal -m value\n");
		    errflg++;
		    break;
		}
		if ( strcmp(optarg, "n") != 0 ) {
		    pc = optarg;
		    while (*pc) {
			if ( *pc != 'a' && *pc != 'b' && *pc != 'e' ) {
		            fprintf(stderr, "qsub: illegal -m value\n");
		            errflg++;
			    break;
			}
			pc++;
		    }
		}
                set_attr(&attrib, ATTR_m, optarg);
            }
            break;
        case 'M':
            if_cmd_line(M_opt) {
                M_opt = passet;
		if ( parse_at_list(optarg, FALSE, FALSE) ) {
	            fprintf(stderr, "qsub: illegal -M value\n");
	            errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_M, optarg);
            }
            break;
        case 'N':
            if_cmd_line(N_opt) {
                N_opt = passet;
		if (check_job_name(optarg, 1) == 0) {
                    set_attr(&attrib, ATTR_N, optarg);
		} else {
		    fprintf(stderr, "qsub: illegal -N value\n");
		    errflg++;
		}
            }
            break;
        case 'o':
            if_cmd_line(o_opt) {
                o_opt = passet;
                if ( prepare_path(optarg, path_out) == 0 ) {
                    set_attr(&attrib, ATTR_o, path_out);
                } else {
                    fprintf(stderr, "qsub: illegal -o value\n");
                    errflg++;
                }
            }
            break;
        case 'p':
            if_cmd_line(p_opt) {
                p_opt = passet;
		while ( isspace((int)*optarg) ) optarg++;
		pc = optarg;
		if ( *pc == '-' || *pc == '+' ) pc++;
		if ( strlen(pc) == 0 ) {
		    fprintf(stderr, "qsub: illegal -p value\n");
		    errflg++;
		    break;
		}
		while ( *pc != '\0' ) {
		    if ( ! isdigit(*pc) ) {
			fprintf(stderr, "qsub: illegal -p value\n");
			errflg++;
			break;
		    }
		    pc++;
		}
		i = atoi(optarg);
		if ( i < -1024 || i > 1023 ) {
		    fprintf(stderr, "qsub: illegal -p value\n");
		    errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_p, optarg);
            }
            break;
        case 'q':
            if_cmd_line(q_opt) {
                q_opt = passet;
                strcpy(destination, optarg);
            }
            break;
        case 'r':
            if_cmd_line(r_opt) {
                r_opt = passet;
		if ( strlen(optarg) != 1 ) {
		    fprintf(stderr, "qsub: illegal -r value\n");
		    errflg++;
		    break;
		}
		if ( *optarg != 'y' && *optarg != 'n' ) {
		    fprintf(stderr, "qsub: illegal -r value\n");
		    errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_r, optarg);
            }
            break;
        case 'S':
            if_cmd_line(S_opt) {
                S_opt = passet;
		if ( parse_at_list(optarg, TRUE, TRUE) ) {
	            fprintf(stderr, "qsub: illegal -S value\n");
	            errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_S, optarg);
            }
            break;
        case 'u':
            if_cmd_line(u_opt) {
                u_opt = passet;
		if ( parse_at_list(optarg, TRUE, FALSE) ) {
	            fprintf(stderr, "qsub: illegal -u value\n");
	            errflg++;
		    break;
		}
                set_attr(&attrib, ATTR_u, optarg);
            }
            break;
        case 'v':
            if_cmd_line(v_opt) {
                v_opt = passet;
		if ( v_value != (char *)0 ) free(v_value);
		v_value = (char *) malloc(strlen(optarg)+1);
		if ( v_value == (char *)0 ) {
		    fprintf(stderr, "qsub: out of memory\n");
		    errflg++;
		    break;
		}
                strcpy(v_value, optarg);
            }
            break;
        case 'V':
            if_cmd_line(V_opt) {
                V_opt = passet;
            }
            break;
        case 'W':
	    while ( isspace((int)*optarg) ) optarg++;
	    if ( strlen(optarg) == 0 ) {
		fprintf(stderr, "qsub: illegal -W value\n");
		errflg++;
		break;
	    }
	    i = parse_equal_string(optarg, &keyword, &valuewd);
	    while (i == 1) {
		if (strcmp(keyword, ATTR_depend) == 0) {
            	    if_cmd_line(Depend_opt) {
                	Depend_opt = passet;
			pdepend = malloc(PBS_DEPEND_LEN);
			if (parse_depend_list(valuewd,pdepend,PBS_DEPEND_LEN)) {
			    fprintf(stderr, "qsub: illegal -W value\n");
			    errflg++;
			    break;
			}
                	set_attr(&attrib, ATTR_depend, pdepend);
		    }
		} else if (strcmp(keyword, ATTR_stagein) == 0) {
            	    if_cmd_line(Stagein_opt) {
                	Stagein_opt = passet;
			if ( parse_stage_list(valuewd) ) {
			    fprintf(stderr, "qsub: illegal -W value\n");
			    errflg++;
			    break;
			}
                	set_attr(&attrib, ATTR_stagein, valuewd);
		    }
		} else if (strcmp(keyword, ATTR_stageout) == 0) {
            	    if_cmd_line(Stageout_opt) {
                	Stageout_opt = passet;
                        if ( parse_stage_list(valuewd) ) {
                            fprintf(stderr, "qsub: illegal -W value\n");
                            errflg++;
			    break;
                        }
                	set_attr(&attrib, ATTR_stageout, valuewd);
		    }
		} else if (strcmp(keyword, ATTR_g) == 0) {
		    if_cmd_line(Grouplist_opt) {
			Grouplist_opt = passet;
                	if ( parse_at_list(valuewd, TRUE, FALSE) ) {
                	    fprintf(stderr, "qsub: illegal -W value\n");
                	    errflg++;
			    break;
                	}
			set_attr(&attrib, ATTR_g, valuewd);
		    }
		} else if (strcmp(keyword, ATTR_inter) == 0) {
		    if_cmd_line(Interact_opt) {
			Interact_opt = passet;
			if ( strcmp(valuewd, "true") != 0 ) {
			    fprintf(stderr, "qsub: illegal -W value\n");
			    errflg++;
			    break;
			}
			set_attr(&attrib, ATTR_inter, interactive_port());
		    }
		} else {
			set_attr(&attrib, keyword, valuewd);
		}
		i = parse_equal_string((char *)0, &keyword, &valuewd);
            }
	    if (i == -1) {
		fprintf(stderr, "qsub: illegal -W value\n");
		errflg++;
	    }
            break;
        case 'z':
            if_cmd_line(z_opt) z_opt = passet;
            break;
        case '?':
        default :
            errflg++;
        }
    }
    if ( !errflg && pass ) {
        errflg = (optind != argc);
	if ( errflg ) {
	    fprintf(stderr, "qsub: directive error: ");
	    for (optind=1; optind<argc; optind++)
	    	fprintf(stderr, "%s ", argv[optind]);
	    fprintf(stderr, "\n");
	}
    }
    return (errflg);
}


/*
 * set_opt_defaults - if not already set, set certain job attributes to
 *	their default value
 */
void set_opt_defaults()
{
    if (c_opt == FALSE)
	set_attr(&attrib, ATTR_c, CHECKPOINT_UNSPECIFIED);
    if (h_opt == FALSE)
	set_attr(&attrib, ATTR_h, NO_HOLD);
    if (j_opt == FALSE)
	set_attr(&attrib, ATTR_j, NO_JOIN);
    if (k_opt == FALSE)
	set_attr(&attrib, ATTR_k, NO_KEEP);
    if (m_opt == FALSE)
	set_attr(&attrib, ATTR_m, MAIL_AT_ABORT);
    if (p_opt == FALSE)
	set_attr(&attrib, ATTR_p, "0");
    if (r_opt == FALSE)
	set_attr(&attrib, ATTR_r, "TRUE");
}

main(argc, argv, envp)   /* qsub */
int argc;
char **argv;
char **envp;
{
    int errflg;                         /* option error */
    static char script[MAXPATHLEN+1] = "";   /* name of script file */
    char script_tmp[MAXPATHLEN+1] = "";      /* name of script file copy */
    char  basename[16];			/* base name of script for job name*/
    char *bnp;
    FILE *f;                            /* FILE pointer to the script */
    char *q_n_out;                      /* queue part of destination */
    char *s_n_out;                      /* server part of destination */
                                        /* server:port to send request to */
    int connect;                        /* return from pbs_connect */
    char *errmsg;                       /* return from pbs_geterrmsg */
    struct stat statbuf;
    struct sigaction act;

    errflg = process_opts(argc, argv, 0);  /* get cmd-line options */

    if (errflg || ((optind+1) < argc)) {
        static char usage[]=
"usage: qsub [-a date_time] [-A account_string] [-c interval]\n\
[-C directive_prefix] [-e path] [-h ] [-I] [-j y|n] [-k keep] \n\
[-l resource_list] [-m mail_options] [-M user_list] [-N jobname] [-o path]\n\
[-p priority] [-r y|n] [-S path] [-u user_list] [-W otherattributes=value...]\n\
[-v variable_list] [-V ] [-z] [script]\n";
        fprintf(stderr, usage);
        exit (2);
    }

    if ( optind < argc ) strcpy(script, argv[optind]);

    /* if script is empty, get standard input */
    if ( (strcmp(script,"") == 0) || (strcmp(script, "-") == 0) ) {
        if ( ! N_opt ) set_attr(&attrib, ATTR_N, "STDIN");
	if (Interact_opt == FALSE) {
            if ( (errflg=get_script(stdin, script_tmp, set_dir_prefix(dir_prefix, C_opt))) > 0) {
                unlink(script_tmp);
                exit(1);
            } else if ( errflg < 0 ) {
                exit(1);
	    }
        }
    } else {  /* non-empty script, read it for directives */
	if (stat(script, &statbuf) < 0) {
	    perror("qsub: script file:");
	    exit(1);
	}
	if ( ! S_ISREG(statbuf.st_mode) ) {
	    fprintf(stderr, "qsub: script not a file\n");
	    exit(1);
	}
        if ( (f = fopen(script, "r")) != NULL) {
            if ( ! N_opt ) {
		if (bnp = strrchr(script, (int)'/'))
			bnp++;
		else
			bnp = script;
		(void)strncpy(basename, bnp, 15);
		basename[15] = '\0';
		if (check_job_name(basename, 0) == 0) {
			set_attr(&attrib, ATTR_N, basename);
		} else {
			fprintf(stderr, "qsub: cannot form a valid job name from the script name\n");
			exit(1);
		}
	    }
            if ( (errflg=get_script(f, script_tmp, set_dir_prefix(dir_prefix, C_opt))) > 0 ) {
                unlink(script_tmp);
                exit(1);
            } else if ( errflg < 0 ) {
                exit(1);
            }
        } else {
            perror("qsub: opening script file:");
            exit(8);
        }
    }
    set_opt_defaults();		/* set option default values */
    server_out[0] = '\0';
    if ( parse_destination_id(destination, &q_n_out, &s_n_out) ) {
        fprintf(stderr,"qsub: illegally formed destination: %s\n", destination);
        unlink(script_tmp);
        exit(2);
    } else {
        if ( notNULL(s_n_out) ) {
            strcpy(server_out, s_n_out);
        }
    }

    /* Connect to the server */
    connect = cnt2server(server_out);
    if ( connect <= 0 ) {
        fprintf(stderr, "qsub: cannot connect to server %s (errno=%d)\n",
                pbs_server, pbs_errno);
        unlink(script_tmp);
        exit(pbs_errno);
    }

    /* Get required environment variables to be sent to the server. */
    if ( ! set_job_env(envp) ) {
        fprintf(stderr, "qsub: cannot send environment with the job\n");
        exit(3);
    }

    /* disallow ^Z which hangs up MOM starting an interactive job */

    sigemptyset(&act.sa_mask);
    act.sa_handler = no_suspend;
    act.sa_flags = 0;
    if (sigaction(SIGTSTP, &act, (struct sigaction *)0) < 0) {
	perror("unable to catch signals");
	exit(1);
    }
    
    /* Send submit request to the server. */

    pbs_errno = 0;
    new_jobname = pbs_submit(connect, (struct attropl *)attrib, script_tmp, destination, NULL);
    if ( new_jobname == NULL ) {
        errmsg = pbs_geterrmsg(connect);
        if ( errmsg != NULL ) {
            fprintf(stderr, "qsub: %s\n", errmsg);
        } else
            fprintf(stderr, "qsub: Error (%d) submitting job\n", pbs_errno);
        unlink(script_tmp);
        exit(pbs_errno);
    } else {
        if ( ! z_opt && ! Interact_opt)
		printf("%s\n", new_jobname);
    }

    /* Disconnet from the server. */
    pbs_disconnect(connect);

    unlink(script_tmp);

    /* is this an interactive job ??? */

    if (Interact_opt == 1)
	interactive();
    exit(0);
}
