#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <assert.h>
#include <netdb.h>
#include <pwd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "libpbs.h"
#include "dis.h"
#include "server_limits.h"
#include "net_connect.h"
#include "credential.h"

/*
 * pbs_iff - authenticates the user to the PBS server.
 *
 *	Usage: call via pbs_connect() with
 *		pbs_iff [-t] hostname port [parent_connection_port]
 *
 *	The parent_connection_port is required unless -t (for test) is given.
 */
/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

static char ident[] = "@(#) $RCSfile: iff2.c,v $ $Revision: 2.2 $";

int pbs_errno;
struct connect_handle connection[NCONNECTS];

main(argc, argv, envp)
	int	argc;
	char   *argv[];
	char   *envp[];
{
	int		 amt;
	int		 auth_type = PBS_credentialtype_none;
	int		 err = 0;
	pbs_net_t	 hostaddr;
	int		 i;
	char		*out;
	uid_t		 myrealuid;
	unsigned int	 parentport;
	int		 parentsock = -1;
	struct passwd   *pwent;
	int	 	servport = -1;
	int 		 sock;
	struct sockaddr_in sockname;
	int		 socknamelen;
	int		 testmode = 0;
	int		 rc;
	struct batch_reply   *reply;
	char *parse_servername A_((char *, short *));
	extern int   optind;
	extern char *optarg;

	/* Need to unset LOCALDOMAIN if set, want local host name */

	for (i=0; envp[i]; ++i) {
		if (!strncmp(envp[i], "LOCALDOMAIN=", 12)) {
			envp[i] = "";
			break;
		}
	}

	while ((i = getopt(argc, argv, "t")) != EOF) {
	    switch (i) {
		case 't':
			testmode = 1;
			break;
		default:
			err = 1;
	    }
	}

	if ( (err == 1) ||
	     (testmode &&  (argc - optind) != 2) || 
	     (!testmode && (argc - optind) != 3) ) {
		fprintf(stderr, "Usage: %s [-t] host port [parent_port]\n", 
								argv[0]);
		return (1);
	}

	if ( !testmode && isatty(fileno(stdout)) )   {
		fprintf(stderr,"pbs_iff: output is a tty & not test mode\n");
		return (1);
	}

	/* first, make sure we have a valid server (host), and ports */

	if ((hostaddr = get_hostaddr(argv[optind])) == (pbs_net_t)0) {
		fprintf(stderr, "pbs_iff: unknown host %s\n", argv[optind]);
		return (1);
	}
	if ((servport = atoi(argv[++optind])) <= 0)
		return (1);

	for (i=0; i<10; i++) {
		sock = client_to_svr(hostaddr, (unsigned int)servport, 1);
		if (sock != PBS_NET_RC_RETRY)
			break;
		sleep(1);
	}
	if (sock < 0) {
		fprintf(stderr, "pbs_iff: cannot connect to host\n");
		return (4);
	}

	connection[1].ch_inuse = 1;
	connection[1].ch_errno = 0;
	connection[1].ch_socket = sock;
	connection[1].ch_errtxt = (char *)0;
	DIS_tcp_setup(sock);

	if (testmode == 0) {
		if ((parentsock = atoi(argv[++optind])) < 0)
			return (1);
	} else {
		/* for test mode, use my own port rather than the parents */
		parentsock = sock;
	}

	/* next, get the real user name */

	myrealuid = getuid();
	pwent = getpwuid(myrealuid);
	if (pwent == (struct passwd *)0)
		return (3);

	/* now get the parent's client-side port */

	socknamelen = sizeof (sockname);
	if (getsockname(parentsock,(struct sockaddr *)&sockname,&socknamelen)<0)
		return (3);
	parentport = ntohs(sockname.sin_port);


	/* send authentication information */

	if ( (rc=encode_DIS_ReqHdr(sock,PBS_BATCH_AuthenUser,pwent->pw_name)) ||
	     (rc=diswui(sock, parentport)) ||
	     (rc = encode_DIS_ReqExtend(sock,(char *)0)) ) {
		return (2);
	}
	if (DIS_tcp_wflush(sock)) {
                return (2);
        }

	/* read back the response */

	reply = PBSD_rdrpy(1);
	if (reply == NULL) {
		fprintf(stderr, "pbs_iff: error returned: %d\n", pbs_errno);
		return (1);
	}

	if (reply->brp_code != 0) {
		fprintf(stderr,"pbs_iff: error returned: %d\n",reply->brp_code);
		if (reply->brp_choice == BATCH_REPLY_CHOICE_Text)
		    fprintf(stderr, "pbs_iff: %s\n",
			    reply->brp_un.brp_txt.brp_str);
		return (1);
	}

	/* send back "type none" credential */

	(void)write(fileno(stdout), &auth_type, sizeof (int));
	(void)fclose(stdout);
	return (0);
}
