/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <ctype.h>
#include <memory.h>
#ifndef NDEBUG
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_fn_arst.c,v $ $Revision: 2.1 $";

/*
 * This file contains general function for attributes of type 
 *	array of (pointers to) strings
 *
 * Each set has functions for:
 *	Decoding the value string to the machine representation.
 *	Encoding the internal representation of the attribute to external
 *	Setting the value by =, + or - operators.
 *	Comparing a (decoded) value with the attribute value.
 *	Freeing the space malloc-ed to the attribute value.
 *
 * Some or all of the functions for an attribute type may be shared with
 * other attribute types.
 * 
 * The prototypes are declared in "attribute.h"
 *
 * ----------------------------------------------------------------------------
 * Attribute functions for attributes with value type "array of strings".
 *
 * The "encoded" or external form of the value is a string with the orginial
 * strings separated by commas (or new-lines) and terminated by a null.
 * Any embedded commas or black-slashes must be escaped by a prefixed back-
 * slash.
 *
 * The "decoded" form is a set of strings pointed to by an array_strings struct
 * ----------------------------------------------------------------------------
 */

static int decode_arst_direct(patr, val)
	struct attribute *patr;
	char   		 *val;
{
	unsigned long		 bksize;
	int			 j;
	int			 ns;
	char			*pbuf;
	char			*pc;
	char			*pstr;
	size_t			 ssize;
	struct array_strings	*stp;

	/*
	 * determine number of sub strings, each sub string is terminated
	 * by a non-escaped comma or a new-line, the whole string is terminated
	 * by a null
	 */

	ns = 1;
	ssize = strlen(val) + 1;
	for (pc = val; *pc; pc++) {
		if (*pc == '\\') {
			if (*++pc == '\0')
				break;
		} else if ((*pc == ',') || (*pc == '\n')) {
			++ns;
		}
	}
	pc--;
	if ((*pc == '\n') || ((*pc == ',') && (*(pc-1) != '\\'))) {
		ns--;		/* strip any trailing null string */
		*pc = '\0';
		ssize--;
	}
			
	pbuf  = malloc((unsigned)ssize);
	if (pbuf == (char *)0)
		return (PBSE_SYSTEM);
	bksize = (ns-1) * sizeof (char *) + sizeof (struct array_strings);
	stp = (struct array_strings *)malloc(bksize);
	if (!stp)
		return (PBSE_SYSTEM);
	stp->as_npointers = ns;		/* number of slots (sub strings) */
					/* for the strings themselves */
	stp->as_buf     = pbuf;
	stp->as_next    = pbuf;
	stp->as_bufsize = (unsigned int)ssize;

	/* now copy in substrings and set pointers */
	pc = pbuf;
	j  = 0;
	pstr = parse_comma_string(val);
	while ((pstr != (char *)0) && (j < ns)) {
		stp->as_string[j] = pc;
		while (*pstr) {
			if (*pstr == '\\')
				if (*++pstr == '\0')
					break;
			*pc++ = *pstr++;
		}	
		*pc++ = '\0';
		j++;
		pstr = parse_comma_string((char *)0);
	}
	stp->as_usedptr = j;
	stp->as_next = pc;

	patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	patr->at_val.at_arst = stp;
	return (0);
}

/*
 * decode_arst - decode a comma string into an attr of type ATR_TYPE_ARST
 *
 *	Returns: 0 if ok,
 *		>0 error number1 if error,
 *		*patr members set
 */

int decode_arst(patr, name, rescn, val)
	struct attribute *patr;
	char *name;		/* attribute name */
	char *rescn;		/* resource name, unused here */
	char *val;		/* attribute value */
{
	int	  rc;
	attribute temp;

	if ((val == (char *)0) || (strlen(val) == 0)) {
	    free_arst(patr);
	    patr->at_flags &= ATR_VFLAG_MODIFY;	/* _SET cleared in free_arst */
	    return (0);
	}

	if ((patr->at_flags & ATR_VFLAG_SET) && (patr->at_val.at_arst)) {
		/* already have values, decode new into temp	*/
		/* then use set(incr) to add new to existing	*/

		
		temp.at_val.at_arst = 0;
		if ((rc = decode_arst_direct(&temp, val)) != 0)
			return (rc);
		rc = set_arst(patr, &temp, INCR);
		free_arst(&temp);
		return (rc);

	} else {
		/* decode directly into real attribute */
		return ( decode_arst_direct(patr, val) );
	}
}

/*
 * encode_arst - encode attr of type ATR_TYPE_ARST into attrlist entry
 *
 * Mode ATR_ENCODE_CLIENT - encode strings into single super string
 *			    separated by ','
 *
 * Mode ATR_ENCODE_SVR    - treated as above
 *
 * Mode ATR_ENCODE_MOM    - treated as above
 *
 * Mode ATR_ENCODE_SAVE - encode strings into single super string
 *			  separated by '\n'
 *
 *	Returns: >0 if ok, entry created and linked into list
 *		 =0 no value to encode, entry not created
 *		 -1 if error
 */
/*ARGSUSED*/

int encode_arst(attr, phead, atname, rsname, mode)
	attribute     *attr;	/* ptr to attribute to encode */
	list_head     *phead;	/* ptr to head of attrlist list */
	char	      *atname;	/* attribute name */
	char	      *rsname;	/* resource name or null */
	int	       mode;	/* encode mode */
{
	char	 *end;
	int	  i;
	int	  j;
	svrattrl *pal;
	char	 *pc;
	char     *pfrom;
	char	  separator;


	if (!attr)
		return (-2);
	if (((attr->at_flags & ATR_VFLAG_SET) == 0) || !attr->at_val.at_arst ||
	     !attr->at_val.at_arst->as_usedptr) 
		return (0);	/* no values */


	i = (int)(attr->at_val.at_arst->as_next - attr->at_val.at_arst->as_buf);
	if (mode == ATR_ENCODE_SAVE) {
		separator = '\n';	/* new-line for encode_acl  */
		/* allow one extra byte for final new-line */
		j = i + 1;
	} else {
		separator = ',';	/* normally a comma is the separator */
		j = i;
	}

	/* how many back-slashes are required */

	for (pc = attr->at_val.at_arst->as_buf; pc < attr->at_val.at_arst->as_next; ++pc) {
		if ((*pc == '"') || (*pc == '\'') || (*pc == ',') ||
		    (*pc == '\\'))
			++j;
	}
	pal = attrlist_create(atname, rsname, j);
	if (pal == (svrattrl *)0)
		return (-1);

	pal->al_flags = attr->at_flags;

	pc    = pal->al_value;
	pfrom = attr->at_val.at_arst->as_buf;

	/* replace nulls between sub-strings with separater characters */
	/* escape any embedded special character */

	end = attr->at_val.at_arst->as_next;
	while (pfrom < end) {
		if ((*pfrom == '"') || (*pfrom == '\'') || (*pfrom == ',') ||
		    (*pfrom == '\\')) {
			*pc++ = '\\';
			*pc   = *pfrom;
		} else if (*pfrom == '\0') {
			*pc = separator;
		} else {
			*pc = *pfrom;
		}
		pc++;
		pfrom++;
	}

	/* convert the last null to separator only if going to new-lines */
	
	if (mode == ATR_ENCODE_SAVE)
		*pc = '\0';	/* insure string terminator */
	else
		*(pc-1) = '\0';
	append_link(phead, &pal->al_link, pal);

	return (1);
}

/* 
 * set_arst - set value of attribute of type ATR_TYPE_ARST to another
 *
 *	A=B --> set of strings in A replaced by set of strings in B
 *	A+B --> set of strings in B appended to set of strings in A
 *	A-B --> any string in B found is A is removed from A
 *
 *	Returns: 0 if ok
 *		>0 if error
 */

int set_arst(attr, new, op)
	struct attribute *attr;
	struct attribute *new;
	enum batch_op op;
{
	int	 i;
	int	 j;
	unsigned long nsize;
	unsigned long need;
	long	 offset;
	char	*pc;
	long	 used;
	struct array_strings *newpas;
	struct array_strings *pas;
	struct array_strings *xpasx;
	void free_arst A_((attribute *));

	assert(attr && new && (new->at_flags & ATR_VFLAG_SET));

	pas = attr->at_val.at_arst;
	xpasx = new->at_val.at_arst;
	if ( !xpasx )
		return (PBSE_INTERNAL);

	if ( !pas ) {

		/* not array_strings control structure, make one */

		j = xpasx->as_npointers;
		if (j < 1)
			return (PBSE_INTERNAL);
		need = sizeof(struct array_strings) + (j-1) * sizeof (char *);
		pas=(struct array_strings *)malloc(need);
		if ( !pas )
			return (PBSE_SYSTEM);
		pas->as_npointers = j;
		pas->as_usedptr = 0;
		pas->as_bufsize = 0;
		pas->as_buf     = (char *)0;
		pas->as_next   = (char *)0;
		attr->at_val.at_arst = pas;
	}
	if ((op == INCR) && !pas->as_buf)
		op = SET;	/* no current strings, change op to SET */

	/*
	 * At this point we know we have a array_strings struct initialized
	 */

	switch (op) {
	    case SET:

		/*
		 * Replace old array of strings with new array, this is
		 * simply done by deleting old strings and appending the
		 * new (to the null set).
		 */

		for (i=0; i< pas->as_usedptr; i++) 
			pas->as_string[i] = (char *)0;	/* clear all pointers */
		pas->as_usedptr = 0;
		pas->as_next   = pas->as_buf;

		if (new->at_val.at_arst == (struct array_strings *)0)
			break;	/* none to set */

		nsize = xpasx->as_next - xpasx->as_buf; /* space needed */
		if ( nsize > pas->as_bufsize) {		 /* new wont fit */
		    if (pas->as_buf)
			free(pas->as_buf);
		    nsize += nsize / 2;			/* alloc extra space */
		    if ( !(pas->as_buf = malloc(nsize)) ) {
			pas->as_bufsize = 0;
			return (PBSE_SYSTEM);
		    }
		    pas->as_bufsize = nsize;

		} else {			/* str does fit, clear buf */
		    (void)memset(pas->as_buf, 0, pas->as_bufsize);
		}

		pas->as_next = pas->as_buf;

		/* No break, "SET" falls into "INCR" to add strings */

	    case INCR:

		nsize = xpasx->as_next - xpasx->as_buf;   /* space needed */
		used = pas->as_next - pas->as_buf;

		if (nsize > (pas->as_bufsize - used)) {
		    need = pas->as_bufsize + 2 * nsize;  /* alloc new buf */
		    if (pas->as_buf)
			pc = realloc(pas->as_buf, need);
		    else
			pc = malloc(need);
		    if (pc == (char *)0)
			return (PBSE_SYSTEM);
		    offset = pc - pas->as_buf;
		    pas->as_buf   = pc;
		    pas->as_next = pc + used;
		    pas->as_bufsize = need;

		    for (j=0; j<pas->as_usedptr; j++)	/* adjust points */
			pas->as_string[j] += offset;
		}

		j = pas->as_usedptr + xpasx->as_usedptr;
		if (j > pas->as_npointers) {	/* need more pointers */
			j = 3 * j / 2;		/* allocate extra     */
			need = sizeof(struct array_strings) + (j-1) * sizeof(char *);
			newpas=(struct array_strings *)realloc((char *)pas,need);
			if (newpas == (struct array_strings *)0)
				return (PBSE_SYSTEM);
			newpas->as_npointers = j;
			pas = newpas;
			attr->at_val.at_arst = pas;
		}

		/* now append new strings */

		for (i=0; i<xpasx->as_usedptr; i++) {
			(void)strcpy(pas->as_next, xpasx->as_string[i]);
			pas->as_string[pas->as_usedptr++] = pas->as_next;
			pas->as_next += strlen(pas->as_next) + 1;
		}
		break;
		    
	    case DECR:	/* decrement (remove) string from array */
		for (j=0; j<xpasx->as_usedptr; j++) {
		    for (i=0; i<pas->as_usedptr; i++) {
			if (!strcmp(pas->as_string[i], xpasx->as_string[j])) {
				/* compact buffer */
				nsize = strlen(pas->as_string[i]) + 1;
				pc = pas->as_string[i] + nsize;
				need = pas->as_next - pc;
				(void)memcpy(pas->as_string[i], pc, (int)need);
				pas->as_next -= nsize;
				/* compact pointers */
				for ( ++i; i < pas->as_npointers; i++)
					pas->as_string[i-1] = pas->as_string[i] - nsize;
				pas->as_string[i-1] = (char *)0;
				pas->as_usedptr--;
				break;
			}
		    }
		}
		break;

	    default:	return (PBSE_INTERNAL);
	}
	attr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return (0);
}

/*
 * comp_arst - compare two attributes of type ATR_TYPE_ARST
 *
 *	Returns: 0 if the set of strings in the two attributes match
 *		+1 otherwise
 */

int comp_arst(attr, with)
	struct attribute *attr;
	struct attribute *with;
{
	int	i;
	int	j;
	struct array_strings *apa;
	struct array_strings *bpb;

	if ( !attr || !with || !attr->at_val.at_arst || !with->at_val.at_arst)
		return (1);
	if ((attr->at_type != ATR_TYPE_ARST) ||
	    (with->at_type != ATR_TYPE_ARST))
		return (1);
	apa = attr->at_val.at_arst;
	bpb = with->at_val.at_arst;

	for (j=0; j<bpb->as_usedptr; j++ ) {
		for (i=0; i<apa->as_usedptr; i++) {
			if (strcmp(bpb->as_string[j], apa->as_string[i]))
				return (1);
		}
	}
	
	return (0);	/* all match */
}

void free_arst(attr)
	struct attribute *attr;
{
	if ((attr->at_flags & ATR_VFLAG_SET) && (attr->at_val.at_arst)) {
		if (attr->at_val.at_arst->as_buf)
			(void)free(attr->at_val.at_arst->as_buf);
		(void)free((char *)attr->at_val.at_arst);
	}
	attr->at_val.at_arst = (struct array_strings *)0;
	attr->at_flags &= ~ATR_VFLAG_SET;
}


/*
 * arst_string - see if a string occurs as a prefix in an arst attribute entry
 *	Search each entry in the value of an arst attribute for a sub-string
 *	that begins with the passed string 
 * 
 * Return: pointer to string if so, null if not
 */

char *arst_string(str, pattr)
	char	   *str;
	attribute  *pattr;
{
	int    i;
	size_t len;
	struct array_strings *parst;

	if ((pattr->at_type != ATR_TYPE_ARST) || !(pattr->at_flags & ATR_VFLAG_SET))
		return ((char *)0);

	len = strlen(str);
	parst = pattr->at_val.at_arst;
	for (i = 0; i < parst->as_usedptr; i++) {
		if (strncmp(str, parst->as_string[i], len) == 0)
			return (parst->as_string[i]);
	}
	return ((char *)0);
}
	
