/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <ctype.h>
#include <memory.h>
#ifndef NDEBUG
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <pbs_ifl.h>
#include "log.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_fn_resc.c,v $ $Revision: 2.1 $";

/*
 * This file contains functions for manipulating attributes of type
 *	resource
 *
 * A "resource" is similiar to an attribute but with two levels of
 * names.  The first name is the attribute name, e.g. "resource-list",
 * the second name is the resource name, e.g. "mem".
 *
 * Each resource_def has functions for:
 *	Decoding the value string to the internal representation.
 *	Encoding the internal attribute to external form
 *	Setting the value by =, + or - operators.
 *	Comparing a (decoded) value with the attribute value.
 *	freeing the resource value space (if extra memory is allocated)
 *
 * Some or all of the functions for an resource type may be shared with
 * other resource types or even attributes.
 * 
 * The prototypes are declared in "attribute.h", also see resource.h
 *
 * ----------------------------------------------------------------------------
 * Attribute functions for attributes with value type resource
 * ----------------------------------------------------------------------------
 */

/* Global Variables */

int resc_access_perm;

/* External Global Items */

int comp_resc_gt;	/* count of resources compared > */
int comp_resc_eq;	/* count of resources compared = */
int comp_resc_lt;	/* count of resources compared < */
int comp_resc_nc;	/* count of resources not compared  */

extern resource_def svr_resc_def[];
extern int	    svr_resc_size;


/*
 * decode_resc - decode a "attribute name/resource name/value" triplet into
 *	         a resource type attribute
 *
 *	Returns: 0 if ok,
 *		>0 error number if error,
 *		*patr members set
 */

int decode_resc(patr, name, rescn, val)
	struct attribute *patr;		/* Modified on Return */
	char *name;			/* attribute name */
	char *rescn;			/* resource name - is used here */
	char *val;			/* resource value */
{
	resource	*prsc;
	resource_def	*prdef;
	int		 rc = 0;
	int		 rv;

	if (patr == (attribute *)0)
		return (PBSE_INTERNAL);
	if (rescn == (char *)0)
		return (PBSE_UNKRESC);
	if ( !(patr->at_flags & ATR_VFLAG_SET))
		CLEAR_HEAD(patr->at_val.at_list);
	

	prdef = find_resc_def(svr_resc_def, rescn, svr_resc_size);
	if (prdef == (resource_def *)0) {
		/*
		 * didn't find resource with matching name, use unknown;
		 * but return PBSE_UNKRESC incase caller dosn`t wish to
		 * accept unknown resources
 		 */
		rc = PBSE_UNKRESC;
		prdef = svr_resc_def + (svr_resc_size-1);
	}

	prsc = find_resc_entry(patr, prdef);
	if (prsc == (resource *)0) 	/* no current resource entry, add it */
		if ((prsc = add_resource_entry(patr,prdef)) == (resource *)0) {
			return (PBSE_SYSTEM);
		}

	/* note special use of ATR_DFLAG_ACCESS, see server/attr_recov() */

	if (((prsc->rs_defin->rs_flags & resc_access_perm) ==0) &&
	    (resc_access_perm != ATR_DFLAG_ACCESS))
		return (PBSE_ATTRRO);

	patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	rv = prdef->rs_decode(&prsc->rs_value, name, rescn, val);
	if (rv)
		return (rv);
	else
		return (rc);
}

/*
 * encode_resc - encode attr of type ATR_TYPE_RESR into attr_extern form
 *
 * Here we are a little different from the typical attribute.  Most have a
 * single value to be encoded.  But resource attribute may have a whole bunch.
 * First get the name of the parent attribute (typically "resource-list").
 * Then for each resource in the list, call the individual resource decode
 * routine with "aname" set to the parent attribute name.
 *
 * If mode is either ATR_ENCODE_SAVE or ATR_ENCODE_SVR, then any resource 
 * currently set to the default value is not encoded.   This allows it to be
 * reset if the default changes or it is moved.
 *
 * If the mode is ATR_ENCODE_CLIENT or ATR_ENCODE_MOM, the client permission
 * passed in the global variable resc_access_perm is checked against each
 * definition.  This allows a resource by resource access setting, not just
 * on the attribute.
 *
 *	Returns: >0 if ok
 *		 =0 if no value to encode, no entries added to list
 *		 <0 if some resource entry had an encode error.
 */

int encode_resc(attr, phead, atname, rsname, mode)
	attribute	*attr;	  /* ptr to attribute to encode */
	list_head	*phead;	  /* head of attrlist list */
	char		*atname;   /* attribute name */
	char		*rsname;   /* resource name, null on call */
	int		 mode;	  /* encode mode */
{
	int	    dflt;
	resource   *prsc;
	int	    rc;
	int	    grandtotal = 0;
	int	    perm;

	if ( !attr )
		return (-1);
	if ( !(attr->at_flags & ATR_VFLAG_SET))
		return (0);	/* no resources at all */

	/* ok now do each separate resource */

	prsc = (resource *)GET_NEXT(attr->at_val.at_list);
	while (prsc != (resource *)0) {

	    /*
	     * encode if sending to client or MOM with permission
	     * encode if saving and not default value
	     * encode if sending to server and not default and have permission
	     */

	    perm = prsc->rs_defin->rs_flags & resc_access_perm ;
	    dflt = prsc->rs_value.at_flags & ATR_VFLAG_DEFLT;
	    if ( ((mode == ATR_ENCODE_CLIENT) && perm)  ||
		 ((mode == ATR_ENCODE_MOM) && perm)     ||
		 ((mode == ATR_ENCODE_SAVE) && (dflt == 0))  ||
		 ((mode == ATR_ENCODE_SVR)  && (dflt == 0) && perm) ) {
		
		    rsname = prsc->rs_defin->rs_name;
		    rc = prsc->rs_defin->rs_encode(&prsc->rs_value, phead,
						   atname, rsname, mode);
		    if (rc < 0)
			return (rc);
		    grandtotal += rc;
	    }
	    prsc = (resource *)GET_NEXT(prsc->rs_link);
	}
	return (grandtotal);
}
	


/* 
 * set_resc - set value of attribute of type ATR_TYPE_RESR to another
 *
 *	For each resource in the list headed by the "new" attribute,
 *	the correspondingly name resource in the list headed by "old"
 *	is modified.  
 *
 *	The mapping of the operations incr and decr depend on the type
 *	of each individual resource.  
 *	Returns: 0 if ok
 *		>0 if error
 */

int set_resc(old, new, op)
	struct attribute *old;
	struct attribute *new;
	enum batch_op op;
{
	enum batch_op local_op;
	resource *newresc;
	resource *oldresc;
	int	  rc;

	assert(old && new);

	newresc = (resource *)GET_NEXT(new->at_val.at_list);
	while (newresc != (resource *)0) {

		local_op = op;
	
		/* search for old that has same definition as new */

		oldresc = find_resc_entry(old, newresc->rs_defin);
		if (oldresc == (resource *)0) {
			/* add new resource to list */
			oldresc = add_resource_entry(old,newresc->rs_defin);
			if (oldresc == (resource *)0) {
				log_err(-1,"set_resc","Unable to malloc space");
				return (PBSE_SYSTEM);
			}
		}

		/*
		 * unlike other attributes, resources can be "unset"
		 * if new is "set" to a value, the old one is set to that
		 * value; if the new resource is unset (no value), then the
		 * old resource is unset by freeing it.
		 */

		if (newresc->rs_value.at_flags & ATR_VFLAG_SET) {

		    /* call resource type dependent  set routine */

			if ((rc = oldresc->rs_defin->rs_set(&oldresc->rs_value,
			     &newresc->rs_value, local_op)) != 0)
				return (rc);
		} else {
			oldresc->rs_defin->rs_free(&oldresc->rs_value);
		}

		newresc = (resource *)GET_NEXT(newresc->rs_link);
	}
	old->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return (0);
}

/*
 * comp_resc - compare two attributes of type ATR_TYPE_RESR
 *
 *	DANGER Will Robinson, DANGER
 *
 *	As you can see from the returns, this is different from the
 *	at_comp model...  PLEASE read the Internal Design Spec
 *
 *	Returns: 0 if compare successful:
 *		   sets comp_resc_gt to count of "greater than" compares
 *				attr > with
 *		   sets comp_resc_eq to count of "equal to" compares
 *				attr == with
 *		   sets comp_resc_lt to count of "less than" compares
 *				attr < with
 *		-1 if error
 */

int comp_resc(attr, with)
	struct attribute *attr;
	struct attribute *with;
{
	resource *atresc;
	resource *wiresc;
	int rc;

	comp_resc_gt = 0;
	comp_resc_eq = 0;
	comp_resc_lt = 0;
	comp_resc_nc = 0;

	if ((attr == (attribute *)0) || (with == (attribute *)0))
		return (-1);

	wiresc = (resource *)GET_NEXT(with->at_val.at_list);
	while (wiresc != (resource *)0) {
	    if ((wiresc->rs_value.at_flags & ATR_VFLAG_SET) &&
	        ((wiresc->rs_value.at_flags & ATR_VFLAG_DEFLT) == 0)) {
		atresc = find_resc_entry(attr, wiresc->rs_defin);
		if (atresc != (resource *)0) {
			if (atresc->rs_value.at_flags & ATR_VFLAG_SET) {
			    if ((rc=atresc->rs_defin->rs_comp(&atresc->rs_value, 				      &wiresc->rs_value)) > 0)
					comp_resc_gt++;
			    else if (rc < 0)
					comp_resc_lt++;
			    else
					comp_resc_eq++;
			}
		} else {	
			comp_resc_nc++;
		}
	    }
	    wiresc = (resource *)GET_NEXT(wiresc->rs_link);
	}
	return (0);
}
	
/*
 * free_resc - free space associated with attribute value
 *
 *	For each entry in the resource list, the entry is delinked,
 *	the resource entry value space freed (by calling the resource 
 *	free routine), and then the resource structure is freed.
 */

void free_resc(pattr)
	attribute *pattr;
{
	resource *next;
	resource *pr;

	pr = (resource *)GET_NEXT(pattr->at_val.at_list);
	while (pr != (resource *)0) {
		next = (resource *)GET_NEXT(pr->rs_link);
		delete_link(&pr->rs_link);
		pr->rs_defin->rs_free(&pr->rs_value);
		(void)free(pr);
		pr = next;
	}
	CLEAR_HEAD(pattr->at_val.at_list);
	pattr->at_flags &= ~ATR_VFLAG_SET;
}

/*
 * find_resc_def - find the resource_def structure for a resource with
 *	a given name
 *
 *	Returns: pointer to the structure or NULL
 */

resource_def *find_resc_def (rscdf, name, limit)
	resource_def *rscdf;	/* address of array of resource_def structs */
	char         *name;	/* name of resource */
	int	      limit;	/* number of members in resource_def array */
{
	while (limit--) {
		if (strcmp(rscdf->rs_name, name) == 0)
			return (rscdf);
		rscdf++;
	}
	return ((resource_def *)0);
}

/*
 * find_resc_entry - find a resource (value) entry in a list headed in an
 * an attribute that points to the specified resource_def structure 
 *
 *	Returns: pointer to struct resource  or NULL
 */

resource *find_resc_entry(pattr, rscdf)
	attribute	*pattr;
	resource_def	*rscdf;
{
	resource *pr;

	pr = (resource *)GET_NEXT(pattr->at_val.at_list);
	while (pr != (resource *)0) {
		if (pr->rs_defin == rscdf)
			break;
		pr = (resource *)GET_NEXT(pr->rs_link);
	}
	return (pr);
}

/* 
 * add_resource_entry - add and "unset" entry for a resource type to a
 *	list headed in an attribute.  Just for later displaying, the
 *	resource list is maintained in an alphabetic order.
 *	The parent attribute is marked with ATR_VFLAG_SET and ATR_VFLAG_MODIFY
 *
 *	Returns: pointer to the newly added entry or NULL if unable
 *		 to add it (malloc failed).  If the resource already
 *		 exists (it shouldn't) then that one is returned.
 */

resource *add_resource_entry (pattr, prdef)
	attribute	*pattr;
	resource_def	*prdef;
{
	int 		 i;
	resource	*new;
	resource	*pr;

	pr = (resource *)GET_NEXT(pattr->at_val.at_list);
	while (pr != (resource *)0) {
		i = strcmp(pr->rs_defin->rs_name, prdef->rs_name);
		if (i == 0)	/* found an matching entry */
			return (pr);
		else if (i > 0)
			break;
		pr = (resource *)GET_NEXT(pr->rs_link);
	}
	new = (resource *)malloc(sizeof (resource));
	if (new == (resource *)0) {
		log_err(-1, "add_resource_entry", "unable to malloc space");
		return ((resource *)0);
	}
	CLEAR_LINK(new->rs_link);
	new->rs_defin = prdef;
	new->rs_value.at_type = prdef->rs_type;
	new->rs_value.at_flags = 0;
	prdef->rs_free(&new->rs_value);

	if (pr != (resource *)0) {
		insert_link(&pr->rs_link, &new->rs_link, new,LINK_INSET_BEFORE);
	} else {
		append_link(&pattr->at_val.at_list, &new->rs_link, new);
	}
	pattr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return (new);
}

/*
 * action_resc - the at_action for the resource_list attribute
 *	For each resource in the list, if it has its own action routine,
 *	call it.
 */

int action_resc(pattr, pobject, actmode)
	attribute *pattr;
	void *pobject;
	int actmode;
{
	resource *pr;

	pr = (resource *)GET_NEXT(pattr->at_val.at_list);
	while (pr) {
		if ( (pr->rs_value.at_flags & ATR_VFLAG_MODIFY) &&
		     (pr->rs_defin->rs_action) ) 
			    pr->rs_defin->rs_action(pr, pattr, actmode);

		pr->rs_value.at_flags &= ~ATR_VFLAG_MODIFY;
		pr = (resource *)GET_NEXT(pr->rs_link);
	}
	return (0);
}
