/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_func.c,v $ $Revision: 2.1 $";

/*
 * This file contains general functions for manipulating attributes.
 * Included are:
 *	clear_attr()
 *	find_attr()
 *	free_null()
 *	attrlist_alloc()
 *	attrlist_create()
 *	parse_equal_string()
 *	parse_comma_string()
 *	count_substrings()
 *
 * The prototypes are declared in "attr_func.h"
 */

/*
 * clear_attr - clear an attribute value structure and clear ATR_VFLAG_SET
 */

void clear_attr(pattr, pdef)
	attribute	*pattr;
	attribute_def	*pdef;
{
#ifndef NDEBUG
	if (pdef == 0) {
	    (void)fprintf(stderr, "Assertion failed, bad pdef in clear_attr\n");
	    abort();
	}
#endif
	(void)memset((char *)pattr, 0, sizeof (struct attribute));
	pattr->at_type  = pdef->at_type;
	if ((pattr->at_type == ATR_TYPE_RESC) ||
	    (pattr->at_type == ATR_TYPE_LIST))
		CLEAR_HEAD(pattr->at_val.at_list);
}

/*
 * str_nc_cmp - string no-case comparision
 */

static int str_nc_cmp(s1, s2)
	char *s1;
	char *s2;
{
	do {
		if (tolower((int)*s1) != tolower((int)*s2))
			return (-1);
	} while (*s1++ && *s2++);
	return (0);
}

/*
 * find_attr - find attribute definition by name
 *
 *	Searches array of attribute definition strutures to find one
 *	whose name matches the requested name.
 *
 *	Returns: >= 0 index into definition struture array 
 *		   -1 if didn't find matching name
 */

int find_attr(attr_def, name, limit)
	struct attribute_def *attr_def;	/* ptr to attribute definitions */
	char *name;			/* attribute name to find	*/
	int limit;			/* limit on size of def array	*/
{
	int index;

	if (attr_def) {
		for (index = 0; index < limit; index++) {
			if (!str_nc_cmp(attr_def->at_name, name))
				return (index);
			attr_def++;
		}
	}
	return (-1);
}

/*
 * free_null - A free routine for attributes which do not
 *	have malloc-ed space ( boolean, char, long ).
 */
/*ARGSUSED*/
void free_null(attr)
	struct attribute *attr;
{
	/*
	 * a bit of a kludge here, clear out the value (for unset) 
	 * the at_size is the larger of the union that does not have
	 * additional allocated space, so we use it here
	 */
	attr->at_val.at_size.atsv_num   = 0;
	attr->at_val.at_size.atsv_shift = 0;
	attr->at_flags &= ~ATR_VFLAG_SET;
}

/*
 * comp_null - A do nothing, except return 0, attribute comparison
 *	       function.
 */

int comp_null(attr, with)
	struct attribute *attr, *with;
{
   return 0;
}


/*
 * attrlist_alloc - allocate space for an svrattrl structure entry
 *
 *	The space required for the entry is calculated and allocated.
 *	The total size and three string lengths are set in the entry,
 *	but no values are placed in it.
 *
 * Return: ptr to entry or NULL if error
 */

svrattrl *attrlist_alloc(szname, szresc, szval)
	int szname;
	int szresc;
	int szval;
{
	register size_t tsize;
	svrattrl *pal;

	tsize = sizeof(svrattrl) + szname + szresc + szval;
	pal = (svrattrl *)malloc(tsize);
	if (pal == (svrattrl *)0) 
		return ((svrattrl *)0);
#ifdef DEBUG
	memset(pal, 0, sizeof(svrattrl));
#endif

	CLEAR_LINK(pal->al_link);	/* clear link */
	pal->al_atopl.next = 0;
	pal->al_tsize = tsize;		/* set various string sizes */
	pal->al_nameln = szname;
	pal->al_rescln = szresc;
	pal->al_valln  = szval;
	pal->al_flags  = 0;
	pal->al_op     = SET;
	pal->al_name = (char *)pal + sizeof (svrattrl);
	if (szresc)
		pal->al_resc = pal->al_name + szname;
	else
		pal->al_resc = (char *)0;
	pal->al_value = pal->al_name + szname + szresc;
	return (pal);
}

/* 
 * attrlist_create - create an svrattrl structure entry
 *
 * The space required for the entry is calculated and allocated.
 * The attribute and resource name is copied into the entry.
 * Note, the value string should be inserted by the caller after this returns.
 *
 * Return: ptr to entry or NULL if error
 */

svrattrl *attrlist_create(aname, rname, vsize)
	char  *aname;	/* attribute name */
	char  *rname;	/* resource name if needed or null */
	int    vsize;	/* size of resource value         */
{
	svrattrl *pal;
	size_t	     asz;
	size_t	     rsz;

	asz = strlen(aname) + 1;     /* attribute name,allow for null term */

	if (rname == (char *)0)      /* resource name only if type resource */
		rsz = 0;
	else
		rsz = strlen(rname) + 1;

	pal = attrlist_alloc(asz, rsz, vsize);
	strcpy(pal->al_name, aname);	/* copy name right after struct */
	if (rsz)
		strcpy(pal->al_resc, rname);

	return (pal);
}

/*
 * free_attrlist - free the space allocated to a list of svrattrl
 *	structures
 */

void free_attrlist(pattrlisthead)
	list_head *pattrlisthead;
{
	svrattrl *pal;
	svrattrl *nxpal;

	pal = (svrattrl *)GET_NEXT(*pattrlisthead);
	while (pal != (svrattrl *)0) {
		nxpal = (struct svrattrl *)GET_NEXT(pal->al_link);
		delete_link(&pal->al_link);
		(void)free(pal);
		pal = nxpal;
	}
}

#if 0		/* This code is not used, but is too good to delete */
/*
 * parse_equal_string - parse a string of the form:
 *		name1 = value1[, value2 ...][, name2 = value3 [, value4 ...]]
 *	into <name1> <value1[, value2 ...>
 *	     <name2> <value3 [, value4 ...>
 *
 *	On the first call,
 *		*name will point to "name1"
 *		*value will point to "value1 ..." upto but not
 *			including the comma before "name2".
 *	On a second call, with start = (char *)0,
 *		*name will point to "name2"
 *		*value will point t0 "value3 ..."
 * Arguments:
 *	start is the start of the string to parse.  If called again with
 *	start  being a null pointer, it will resume parsing where it stoped
 * 	on the prior call.
 *
 * Returns:
 *	integer function return = 1 if  name and value are found,
 *				  0 if nothing (more) is parsed (null input)
 *				 -1 if a syntax error was detected.
 *	*name is set to point to the name string
 *	*value is set to point to the value string
 *	each string is null terminated.
 */

int parse_equal_string(start, name, value)
	char  *start;
	char **name;
	char **value;
{
	static char *pc;	/* where prior call left off */
	char        *backup;
	int	     quoting = 0;

	if (start != (char *)0)
		pc = start;

	if (*pc == '\0') {
		*name = (char *)0;
		return (0);	/* already at end, return no strings */
	}

	/* strip leading spaces */

	while (isspace((int)*pc) && *pc)
		pc++;

	if (*pc == '\0') {
		*name = (char *)0;	/* null name */
		return (0);
	} else if ((*pc == '=') || (*pc == ','))
		return (-1);	/* no name, return error */

	*name = pc;

	/* have found start of name, look for end of it */

	while ( !isspace((int)*pc) && (*pc != '=') && *pc)
		pc++;

	/* now look for =, while stripping blanks between end of name and = */

	while ( isspace((int)*pc) && *pc)
		*pc++ = '\0';
	if (*pc != '=')
		return (-1);	/* should have found a = as first non blank */
	*pc++ = '\0';

	/* that follows is the value string, skip leading white space */

	while ( isspace((int)*pc) && *pc)
		pc++;

	/* is the value string to be quoted ? */

	if ((*pc == '"') || (*pc == '\''))
		quoting = (int)*pc++;
	*value = pc;

	/*
	 * now go to first equal sign, or if quoted, the first equal sign
	 * after the close quote 
	 */

	if (quoting) {
		while ( (*pc != (char)quoting) && *pc)	/* look for matching */
			pc++;
		if (*pc)
			*pc = ' ';	/* change close quote to space */
		else 
			return (-1);
	}
	while ((*pc != '=') && *pc)
		pc++;

	if (*pc == '\0')
		return (1);	/* no equal, just end of line, stop here */
	
	/* back up to the first comma found prior to the equal sign */

	while (*--pc != ',')
		if (pc <= *value)	/* gone back too far, no comma, error */
			return (-1);
	backup = pc++;
	*backup = '\0';			/* null the comma */

	/* strip off any trailing white space */

	while (isspace((int)*--backup))
		*backup = '\0';
	return (1);
}
#endif

/*
 * parse_comma_string() - parse a string of the form:
 *		value1 [, value2 ...]
 *
 *	On the first call, start is non null, a pointer to the first value
 *	element upto a comma, new-line, or end of string is returned.
 *      Commas escaped by a back-slash '\' are ignored.
 *
 *	On any following calls with start set to a null pointer (char *)0,
 *	the next value element is returned...
 *
 *	A null pointer is returned when there are no (more) value elements.
 */

char *parse_comma_string(start)
	char *start;
{
	static char *pc;	/* if start is null, restart from here */

	char	    *back;
	char	    *rv;

	if (start != (char *)0)
		pc = start;
	
	if (*pc == '\0')
		return ((char *)0);	/* already at end, no strings */

	/* skip over leading white space */

	while ((*pc != '\n') && isspace((int)*pc) && *pc)
		pc++;

	rv = pc;		/* the start point which will be returned */

	/* go find comma or end of line */

	while (*pc) {
		if (*pc == '\\') {
			if (*++pc == 0)
				break;
		} else if ((*pc == ',') || (*pc == '\n'))
			break;
		++pc;
	}
	back = pc;
	while (isspace((int)*--back))	/* strip trailing spaces */
			*back = '\0';

	if (*pc)
		*pc++ = '\0';	/* if not end, terminate this and adv past */
		
	return (rv);
}




/*
 * count_substrings
 * counts number of substrings in a comma separated string
 * see parse_comma_string
 */
int	count_substrings( val, pcnt )
        char	*val;           /*comma separated string of substrings*/
	int	*pcnt;		/*where to return the value*/

{
	int	rc = 0;
        int	ns;
	char   *pc;

	if ( val == (char *)0 )
	     return  (PBSE_INTERNAL);
        /*
         * determine number of substrings, each sub string is terminated
         * by a non-escaped comma or a new-line, the whole string is terminated
         * by a null
         */

        ns = 1;
        for (pc = val; *pc; pc++) {
                if (*pc == '\\') {
                        if (*++pc == '\0')
                                break;
                } else if ((*pc == ',') || (*pc == '\n')) {
                        ++ns;
                }
        }
        pc--;
        if ((*pc == '\n') || ((*pc == ',') && (*(pc-1) != '\\'))) {
                ns--;           /* strip any trailing null string */
                *pc = '\0';
        }

	*pcnt = ns;
	return rc;
}



/*
 * attrl_fixlink - fix up the next pointer within the attropl substructure
 *	within a svrattrl list.
 */

void attrl_fixlink(phead)
	list_head *phead;	/* pointer to head of svrattrl list */
{
	svrattrl *pal;
	svrattrl *pnxt;

	pal = (svrattrl *)GET_NEXT(*phead);
	while (pal) {
		pnxt = (svrattrl *)GET_NEXT(pal->al_link);
		if (pal->al_flags & ATR_VFLAG_DEFLT) {
			pal->al_atopl.op = DFLT;
		} else {
			pal->al_atopl.op = SET;
		}
		if (pnxt) 
			pal->al_atopl.next = &pnxt->al_atopl;
		else
			pal->al_atopl.next = (struct attropl *)0;
		pal = pnxt;
	}
}
