/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <stddef.h>
#include <string.h>

#include "Long.h"
#include "Long_.h"

#undef strTouL

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

static unsigned x_val;
static const char ident[] = "@(#) $RCSfile: strTouL.c,v $ $Revision: 2.1 $";
static const char letters[] = "abcdefghijklmnopqrstuvwxyz";
static char table[UCHAR_MAX + 1];

/*
 * strTouL - returns the unsigned Long value representing the string whose
 * 	     first character is *nptr, when interpreted as an integer in base,
 *	     base.
 *
 * 	If base is zero, the base of the integer is determined by the way the
 * 	string starts.  The string is interpreted as decimal if the first
 * 	character after leading white space and an optional sign is a digit
 * 	between 1 and 9, inclusive.  The string is interpreted as octal if the
 * 	first character after leading white space and an optional sign is the
 * 	digit "0" and the next character is not an "X" (either upper or lower
 * 	case).  The string is interpreted as hexidecimal if the first character
 * 	after leading white space and an optional sign is the digit "0",
 * 	followed by an "X" (either upper or lower case).
 *
 * 	If base is greater than 1 and less than the number of characters in the
 *	Long_dig array, it represents the base in which the number will be
 *	interpreted.  Characters for digits beyond 9 are represented by the
 *	letters of the alphabet, either upper case or lower case.
 */

u_Long strTouL(nptr, endptr, base)
    const char		*nptr;
    char		**endptr;
    int			base;
{
	unsigned	digit;
	u_Long		limit, value;
	enum {
	    unknown1,
	    unknown2,
	    hex1,
	    hex2,
	    hex3,
	    known,
	    working,
	    overflow
	}		state;

	if (table[(unsigned char)'1'] != 1) {
		int	i;		/* Initialize conversion table */

		(void)memset(table, CHAR_MAX, sizeof(table));
		for (i = (int)strlen(Long_dig) - 1; i >= 0; i--)
			table[(unsigned char)Long_dig[i]] = i;
		for (i = (int)strlen(letters) - 1; i >= 0; i--)
			table[(unsigned char)letters[i]] = i + 10;
		x_val = table[(unsigned char)'x'];
	}
	if (nptr == NULL) {
		if (endptr != NULL)
		        *endptr = (char *)nptr;
	        return (0);
	}
	if (base < 0 || base == 1 || (size_t)base > strlen(Long_dig)) {
	        errno = EDOM;
		if (endptr != NULL)
		        *endptr = (char *)nptr;
	        return (0);
	}
	switch (base) {
	    case 0:
		state = unknown1;
		break;
	    case 16:
		state = hex1;
		break;
	    default:
		state = known;
	}
	while (isspace(*nptr++));
	Long_neg = FALSE;
	switch (*--nptr) {
	    case '-':
		Long_neg = TRUE;
	    case '+':
		nptr++;
	}
        value = 0;
	while ((digit = table[(unsigned char)*nptr++]) != CHAR_MAX) {
		switch (state) {
		    case unknown1:
			if (digit >= 10)
				goto done;
			if (digit == 0) {
				state = unknown2;
				break;
			}
			base = 10;
			state = working;
			limit = UlONG_MAX / 10;
			value = digit;
			break;
		    case unknown2:
			if (digit >= 8) {
				if (digit != x_val)
					goto done;
				base = 16;
				state = hex3;
				break;
			}
			base = 8;
			state = working;
			limit = UlONG_MAX / 8;
			value = digit;
			break;
		    case hex1:
			if (digit >= base)
			        goto done;
			if (digit == 0) {
				state = hex2;
				break;
			}
			state = working;
			limit = UlONG_MAX / 16;
			value = digit;
			break;
		    case hex2:
			if (digit == x_val) {
				state = hex3;
				break;
			}
		    case hex3:
		    case known:
			if (digit >= base)
				goto done;
			state = working;
			limit = UlONG_MAX / base;
			value = digit;
			break;
		    case working:
			if (digit >= base)
			        goto done;
			if (value < limit) {
				value = value * base + digit;
				break;
			}
			if (value > limit ||
			    UlONG_MAX - (value *= base) < digit) {
				state = overflow;
				value = UlONG_MAX;
				errno = ERANGE;
				break;
			}
			value += digit;
			break;
		    case overflow:
			if (digit >= base)
			        goto done;
		}
	}
    done:
	if (endptr != NULL) {
		if (state == hex3)
		        nptr--;
	        *endptr = (char *)--nptr;
	}
        if (Long_neg)
                return (-value);
        return (value);
}
