/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <stdlib.h>

static char ident[] = "@(#) $RCSfile: parse_equal.c,v $ $Revision: 2.1 $";

/*
 * parse_equal_string - parse a string of the form:
 *		name1 = value1[, value2 ...][, name2 = value3 [, value4 ...]]
 *	into <name1> <value1[, value2 ...>
 *	     <name2> <value3 [, value4 ...>
 *
 *	On the first call,
 *		*name will point to "name1"
 *		*value will point to "value1 ..." upto but not
 *			including the comma before "name2".
 *	On a second call, with start = (char *)0,
 *		*name will point to "name2"
 *		*value will point t0 "value3 ..."
 * Arguments:
 *	start is the start of the string to parse.  If called again with
 *	start  being a null pointer, it will resume parsing where it stoped
 * 	on the prior call.
 *
 * Returns:
 *	integer function return = 1 if  name and value are found,
 *				  0 if nothing (more) is parsed (null input)
 *				 -1 if a syntax error was detected.
 *	*name is set to point to the name string
 *	*value is set to point to the value string
 *	each string is null terminated.
 */

int parse_equal_string(start, name, value)
	char  *start;
	char **name;
	char **value;
{
	static char *pc;	/* where prior call left off */
	char        *backup;
	int	     quoting = 0;

	if (start != (char *)0)
		pc = start;

	if (*pc == '\0') {
		*name = (char *)0;
		return (0);	/* already at end, return no strings */
	}

	/* strip leading spaces */

	while (isspace((int)*pc) && *pc)
		pc++;

	if (*pc == '\0') {
		*name = (char *)0;	/* null name */
		return (0);
	} else if ((*pc == '=') || (*pc == ','))
		return (-1);	/* no name, return error */

	*name = pc;

	/* have found start of name, look for end of it */

	while ( !isspace((int)*pc) && (*pc != '=') && *pc)
		pc++;

	/* now look for =, while stripping blanks between end of name and = */

	while ( isspace((int)*pc) && *pc)
		*pc++ = '\0';
	if (*pc != '=')
		return (-1);	/* should have found a = as first non blank */
	*pc++ = '\0';

	/* that follows is the value string, skip leading white space */

	while ( isspace((int)*pc) && *pc)
		pc++;

	/* is the value string to be quoted ? */

	if ((*pc == '"') || (*pc == '\''))
		quoting = (int)*pc++;
	*value = pc;

	/*
	 * now go to first equal sign, or if quoted, the first equal sign
	 * after the close quote 
	 */

	if (quoting) {
		while ( (*pc != (char)quoting) && *pc)	/* look for matching */
			pc++;
		if (*pc)
			*pc = ' ';	/* change close quote to space */
		else 
			return (-1);
	}
	while ((*pc != '=') && *pc)
		pc++;

	if (*pc == '\0') {
		while (isspace((int)*--pc));
		if ( *pc == ',' )	/* trailing comma is a no no */
			return (-1);
		pc++;
		return (1);	/* no equal, just end of line, stop here */
	}
	
	/* back up to the first comma found prior to the equal sign */

	while (*--pc != ',')
		if (pc <= *value)	/* gone back too far, no comma, error */
			return (-1);
	backup = pc++;
	*backup = '\0';			/* null the comma */

	/* strip off any trailing white space */

	while (isspace((int)*--backup))
		*backup = '\0';
	return (1);
}
