/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * Synopsis:
 * 	float disrf(int stream, int *retval)
 *
 *	Gets a Data-is-Strings floating point number from <stream> and converts
 *	it into a float and returns it.  The number from <stream> consists of
 *	two consecutive signed integers.  The first is the coefficient, with its
 *	implied decimal point at the low-order end.  The second is the exponent
 *	as a power of 10.
 *
 *	*<retval> gets DIS_SUCCESS if everything works well.  It gets an error
 *	code otherwise.  In case of an error, the <stream> character pointer is
 *	reset, making it possible to retry with some other conversion strategy.
 *
 *	By fiat of the author, neither loss of significance nor underflow are
 *	errors.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <math.h>
#include <stddef.h>

#include "dis.h"
#include "dis_.h"
#undef disrf

static unsigned ndigs;
static unsigned nskips;
static double dval;

static int disrd_(stream, count)
    int			stream;
    unsigned		count;
    {
	int		c;
	int		negate;
	unsigned	unum;
	char		*cp;

	if (dis_umaxd == 0)
	        disiui_();
	switch (c = (*dis_getc)(stream)) {
	    case '-':
	    case '+':
		negate = c == '-';
		nskips = count > FLT_DIG ? count - FLT_DIG : 0;
		count -= nskips;
		ndigs = count;
		dval = 0.0;
		do {
			if ((c = (*dis_getc)(stream)) < '0' || c > '9') {
				if (c < 0)
				        return (DIS_EOD);
				return (DIS_NONDIGIT);
			}
			dval = dval * 10.0 + (double)(c - '0');
		} while (--count);
		if ((count = nskips) > 0) {
			count--;
			switch ((*dis_getc)(stream)) {
			    case '5':
				if(count == 0)
				        break;
			    case '6':
			    case '7':
			    case '8':
			    case '9':
			        dval += 1.0;
			    case '0':
			    case '1':
			    case '2':
			    case '3':
			    case '4':
				if (count > 0 &&
				   (*disr_skip)(stream, (size_t)count) < 0)
				        return (DIS_EOD);
				break;
			    default:
				return (DIS_NONDIGIT);
			}
		}
		dval = negate ? -dval : dval;
		return (DIS_SUCCESS);
	    case '0':
		return (DIS_LEADZRO);
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
		unum = c - '0';
		if (count > 1) {
			if ((*dis_gets)(stream, dis_buffer + 1, count - 1) !=
								count - 1)
			        return (DIS_EOD);
			cp = dis_buffer;
			if (count >= dis_umaxd) {
				if (count > dis_umaxd)
				        break;
				*cp = c;
				if (memcmp(dis_buffer, dis_umax, dis_umaxd) > 0)
				        break;
			}
			while (--count) {
				if ((c = *++cp) < '0' || c > '9')
				        return (DIS_NONDIGIT);
				unum = unum * 10 + (unsigned)(c - '0');
			}
		}
		return (disrd_(stream, unum));
	    case -1:
		return (DIS_EOD);
	    case -2:
		return (DIS_EOF);
	    default:
		return (DIS_NONDIGIT);
	}
	dval = HUGE_VAL;
	return (DIS_OVERFLOW);
}

float disrf(stream, retval)
    int			stream;
    int			*retval;
    {
	int		expon;
	unsigned	uexpon;
	int		locret;
	int		negate;

	assert(retval != NULL);
	assert(stream >= 0);
	assert(dis_getc != NULL);
	assert(dis_gets != NULL);
	assert(disr_skip != NULL);
	assert(disr_commit != NULL);

	dval = 0.0;
	if ((locret = disrd_(stream, 1)) == DIS_SUCCESS) {
		locret = disrsi_(stream, &negate, &uexpon, 1);
		if (locret == DIS_SUCCESS) {
			expon = negate ? nskips - uexpon : nskips + uexpon;
			if (expon + (int)ndigs > FLT_MAX_10_EXP) {
				if (expon + (int)ndigs > FLT_MAX_10_EXP + 1) {
					dval = dval < 0.0 ?
					    -HUGE_VAL : HUGE_VAL;
					locret = DIS_OVERFLOW;
				} else {
				        dval *= disp10d_(expon - 1);
					if (dval > FLT_MAX / 10.0) {
						dval = dval < 0.0 ?
						    -HUGE_VAL : HUGE_VAL;
						locret = DIS_OVERFLOW;
					} else
					        dval *= 10.0;
				}
			} else {
				if (expon < DBL_MIN_10_EXP) {
					dval *= disp10d_(expon + (int)ndigs);
					dval /= disp10d_((int)ndigs);
				} else
				        dval *= disp10d_(expon);
			}
		}
	}
	if ((*disr_commit)(stream, locret == DIS_SUCCESS) < 0)
	        locret = DIS_NOCOMMIT;
	*retval = locret;
	return (dval);
}
